﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/ae.h>
#include <nn/spsm/spsm_Api.h>
#include <nn/spsm/spsm_Debug.h>

#include <nnt/gtest/gtest.h>

namespace
{
    void PrintForTeamCity(const char* key, int64_t value)
    {
        NN_LOG("##teamcity[buildStatisticValue key='%s' value='%lld']\n", key, value);
    }
}

extern "C" void nninitStartup()
{
}

extern "C" void nndiagStartup()
{
}

TEST( PowerState, SleepWakePerformanceCheck )
{
    nn::spsm::Initialize();
    nn::spsm::ResetEventLog();

    nn::os::SystemEventType event;
    nn::ae::InitializeNotificationMessageEvent(&event);

    // SA としての最初の通知を受け取る
    while (nn::ae::Message_ChangeIntoForeground != nn::ae::WaitForNotificationMessage(&event)) {}

    NN_LOG("[SleepWakePerformanceChecker] Wait Sleep\n");

    // Power ボタン短押し, またはバッテリ電圧低下を検知したらスリープを開始する
    while ( NN_STATIC_CONDITION(true) )
    {
        auto message = nn::ae::WaitForNotificationMessage( &event );
        if ( message == nn::ae::Message_DetectShortPressingPowerButton
             || message == nn::ae::Message_SleepRequiredByLowBattery )
        {
            break;
        }
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ae::StartSleepSequence());

    // スリープからの復帰通知を待つ
    while (nn::ae::Message_FinishedSleepSequence != nn::ae::WaitForNotificationMessage(&event)) {}

    // 計測データを出力
    auto timeStampAppResume = nn::os::GetSystemTick();

    nn::spsm::SleepWakeSequenceAnalyzedData data;
    nn::spsm::AnalyzeLogForLastSleepWakeSequence(&data);
    nn::spsm::Finalize();

    PrintForTeamCity("SleepTriggerToFullAwakeExit",
        nn::os::ConvertToTimeSpan(nn::os::Tick(data.timeStampFullAwakeExit) - nn::os::Tick(data.timeStampSleepTrigger)).GetMilliSeconds());
    PrintForTeamCity("SleepTriggerToMinimumAwakeDone",
        nn::os::ConvertToTimeSpan(nn::os::Tick(data.timeStampMinimumAwakeDownwardDone) - nn::os::Tick(data.timeStampSleepTrigger)).GetMilliSeconds());
    PrintForTeamCity("SleepTriggerToSleepReadyDone",
        nn::os::ConvertToTimeSpan(nn::os::Tick(data.timeStampSleepReadyDone) - nn::os::Tick(data.timeStampSleepTrigger)).GetMilliSeconds());
    PrintForTeamCity("SleepTriggerToEssentialServicesSleepReadyDone",
        nn::os::ConvertToTimeSpan(nn::os::Tick(data.timeStampEssentialServicesSleepReadyDone) - nn::os::Tick(data.timeStampSleepTrigger)).GetMilliSeconds());
    PrintForTeamCity("SleepTriggerToSc7Entry",
        nn::os::ConvertToTimeSpan(nn::os::Tick(data.timeStampSc7Entry) - nn::os::Tick(data.timeStampSleepTrigger)).GetMilliSeconds());
    PrintForTeamCity("SleepSc7ExitToEssentialServicesAwakeDone",
        nn::os::ConvertToTimeSpan(nn::os::Tick(data.timeStampEssentialServicesAwakeDone) - nn::os::Tick(data.timeStampSc7Exit)).GetMilliSeconds());
    PrintForTeamCity("SleepSc7ExitToMinimumAwakeDone",
        nn::os::ConvertToTimeSpan(nn::os::Tick(data.timeStampMinimumAwakeUpwardDone) - nn::os::Tick(data.timeStampSc7Exit)).GetMilliSeconds());
    PrintForTeamCity("SleepSc7ExitToFullAwakeDone",
        nn::os::ConvertToTimeSpan(nn::os::Tick(data.timeStampFullAwakeDone) - nn::os::Tick(data.timeStampSc7Exit)).GetMilliSeconds());
    PrintForTeamCity("SleepSc7ExitToAppResume",
        nn::os::ConvertToTimeSpan(timeStampAppResume - nn::os::Tick(data.timeStampSc7Exit)).GetMilliSeconds());

    NN_LOG("[SleepWakePerformanceChecker] Success Tests\n");
}

void SystemAppletMenuMain(nn::ae::SystemAppletParameters* param)
{
    NN_UNUSED(param);

    int    argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    auto result = RUN_ALL_TESTS();
    NN_UNUSED(result);

    // SA が終了すると FATAL するので無限ループしておく
    for (;;)
    {
        nn::os::SleepThread( nn::TimeSpan::FromDays(1) );
    }
}

extern "C" void nnMain()
{
    nn::ae::InvokeSystemAppletMain(nn::ae::AppletId_SystemAppletMenu, SystemAppletMenuMain);
}
