﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/spsm/spsm_Api.h>
#include <nn/spsm/spsm_Debug.h>

#include <nnt/gtest/gtest.h>

namespace
{
    void PrintForTeamCity(const char* key, int64_t value)
    {
        NN_LOG("##teamcity[buildStatisticValue key='%s' value='%lld']\n", key, value);
    }
}

/*
    How to use:
      1. Enable performance check in psc and spsm.
      2. Rebuild FW.
      3. Initialize target.
      4. Run sleep / wake sequence manually.
      5. Run this test.
*/

TEST( PowerState, DumpLastSleepWakeSequencePerformance )
{
    nn::spsm::Initialize();

    nn::spsm::SleepWakeSequenceAnalyzedData data;
    nn::spsm::AnalyzeLogForLastSleepWakeSequence(&data);

    nn::spsm::SleepWakeSequencePerformanceData performance;
    nn::spsm::AnalyzePerformanceLogForLastSleepWakeSequence(&performance);

    nn::spsm::ResetEventLog();

    nn::spsm::Finalize();

    PrintForTeamCity("SleepTriggerToFullAwakeExit",
        nn::os::ConvertToTimeSpan(nn::os::Tick(data.timeStampFullAwakeExit) - nn::os::Tick(data.timeStampSleepTrigger)).GetMilliSeconds());
    PrintForTeamCity("SleepTriggerToMinimumAwakeDone",
        nn::os::ConvertToTimeSpan(nn::os::Tick(data.timeStampMinimumAwakeDownwardDone) - nn::os::Tick(data.timeStampSleepTrigger)).GetMilliSeconds());
    PrintForTeamCity("SleepTriggerToSleepReadyDone",
        nn::os::ConvertToTimeSpan(nn::os::Tick(data.timeStampSleepReadyDone) - nn::os::Tick(data.timeStampSleepTrigger)).GetMilliSeconds());
    PrintForTeamCity("SleepTriggerToEssentialServicesSleepReadyDone",
        nn::os::ConvertToTimeSpan(nn::os::Tick(data.timeStampEssentialServicesSleepReadyDone) - nn::os::Tick(data.timeStampSleepTrigger)).GetMilliSeconds());
    PrintForTeamCity("SleepTriggerToSc7Entry",
        nn::os::ConvertToTimeSpan(nn::os::Tick(data.timeStampSc7Entry) - nn::os::Tick(data.timeStampSleepTrigger)).GetMilliSeconds());
    PrintForTeamCity("SleepSc7ExitToEssentialServicesAwakeDone",
        nn::os::ConvertToTimeSpan(nn::os::Tick(data.timeStampEssentialServicesAwakeDone) - nn::os::Tick(data.timeStampSc7Exit)).GetMilliSeconds());
    PrintForTeamCity("SleepSc7ExitToMinimumAwakeDone",
        nn::os::ConvertToTimeSpan(nn::os::Tick(data.timeStampMinimumAwakeUpwardDone) - nn::os::Tick(data.timeStampSc7Exit)).GetMilliSeconds());
    PrintForTeamCity("SleepSc7ExitToFullAwakeDone",
        nn::os::ConvertToTimeSpan(nn::os::Tick(data.timeStampFullAwakeDone) - nn::os::Tick(data.timeStampSc7Exit)).GetMilliSeconds());

    NN_LOG("[DumpLastSleepWakeSequencePerformance] Dump %d transition logs\n", performance.validDataCount);

    for (int i = 0; i < performance.validDataCount; i++)
    {
        nn::psc::util::PrintModuleInformation(
            performance.data[i].transitionInfo,
            performance.data[i].moduleCount,
            performance.data[i].moduleInfoList,
            performance.data[i].dependencyCount,
            performance.data[i].dependencyInfoList
        );
    }

    NN_LOG("[DumpLastSleepWakeSequencePerformance] Finish\n");
}
