﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Assert.h>

#include <nn/os.h>

#include <nn/os/os_Mutex.h>
#include <nn/psm/psm.h>
#include <nn/psm/psm_System.h>
#include <nn/psm/psm_SystemProcess.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_IntrusiveList.h>

namespace {

bool g_EnoughPowerChargeEmulation = false;
bool g_FastBatteryCharging = true;

nn::psm::ChargerType g_ChargerType = nn::psm::ChargerType_EnoughPower;
bool g_PowerSupplyEnough;
nn::psm::BatteryVoltageState g_BatteryVoltageState;

struct StaticMutex
{
    ::nn::os::MutexType mutex;

    void lock() NN_NOEXCEPT
    {
        ::nn::os::LockMutex(&mutex);
    }

    void unlock() NN_NOEXCEPT
    {
        ::nn::os::UnlockMutex(&mutex);
    }
};

StaticMutex g_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

class SessionContext : public nn::util::IntrusiveListBaseNode<SessionContext>
{
public:
    SessionContext() NN_NOEXCEPT
    : m_StateChangeEvent()
    , m_ChargerTypeChangeEventEnabled(false)
    , m_PowerSupplyChangeEventEnabled(false)
    , m_BatteryVoltageStateChangeEventEnabled(false)
    {
        nn::os::CreateSystemEvent(&m_StateChangeEvent, nn::os::EventClearMode_ManualClear, true);
    }

    ~SessionContext() NN_NOEXCEPT
    {
        nn::os::DestroySystemEvent(&m_StateChangeEvent);
    }

public:
    void SetChargerTypeChangeEventEnabled(bool enabled) NN_NOEXCEPT
    {
        m_ChargerTypeChangeEventEnabled = enabled;
    }

    void SetPowerSupplyChangeEventEnabled(bool enabled) NN_NOEXCEPT
    {
        m_PowerSupplyChangeEventEnabled = enabled;
    }

    void SetBatteryVoltageStateChangeEventEnabled(bool enabled) NN_NOEXCEPT
    {
        m_BatteryVoltageStateChangeEventEnabled = enabled;
    }

    void SignalChargerTypeChangeEvent() NN_NOEXCEPT
    {
        if ( m_ChargerTypeChangeEventEnabled )
        {
            nn::os::SignalSystemEvent(&m_StateChangeEvent);
        }
    }

    void SignalPowerSupplyChangeEvent() NN_NOEXCEPT
    {
        if ( m_PowerSupplyChangeEventEnabled )
        {
            nn::os::SignalSystemEvent(&m_StateChangeEvent);
        }
    }

    void SignalBatteryVoltageStateChangeEvent() NN_NOEXCEPT
    {
        if ( m_BatteryVoltageStateChangeEventEnabled )
        {
            nn::os::SignalSystemEvent(&m_StateChangeEvent);
        }
    }

    void GetEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
    {
        nn::os::NativeHandle handle = nn::os::GetReadableHandleOfSystemEvent(&m_StateChangeEvent);
        nn::os::AttachReadableHandleToSystemEvent(pEvent, handle, false, nn::os::EventClearMode_ManualClear);
    }

private:
    nn::os::SystemEventType m_StateChangeEvent;
    bool m_ChargerTypeChangeEventEnabled;
    bool m_PowerSupplyChangeEventEnabled;
    bool m_BatteryVoltageStateChangeEventEnabled;
};

nn::util::IntrusiveList<SessionContext, nn::util::IntrusiveListBaseNodeTraits<SessionContext>> g_SessionContextList;

const size_t MaxSessionContexts = 30;

SessionContext* CreateSessionContext() NN_NOEXCEPT
{
    NN_ASSERT(g_SessionContextList.size() < MaxSessionContexts)

    SessionContext* pSessionContext = new SessionContext();

    g_SessionContextList.push_front(*pSessionContext);

    return pSessionContext;
}

void DestroySessionContext(SessionContext* pSessionContext) NN_NOEXCEPT
{
    for ( auto itr = g_SessionContextList.begin(); itr != g_SessionContextList.end(); itr++ )
    {
        if ( &(*itr) == pSessionContext )
        {
            g_SessionContextList.erase(itr);
            delete &(*itr);

            return;
        }
    }

    NN_ASSERT(false);
}

void SetAndSignalChargerType(nn::psm::ChargerType chargerType) NN_NOEXCEPT
{
    bool change = (g_ChargerType != chargerType);
    g_ChargerType = chargerType;
    if ( change )
    {
        for ( auto itr = g_SessionContextList.begin(); itr != g_SessionContextList.end(); itr++ )
        {
            (*itr).SignalChargerTypeChangeEvent();
        }
    }

    bool enough = (chargerType == nn::psm::ChargerType_EnoughPower);
    change = (g_PowerSupplyEnough != enough);
    g_PowerSupplyEnough = enough;
    if ( change )
    {
        for ( auto itr = g_SessionContextList.begin(); itr != g_SessionContextList.end(); itr++ )
        {
            (*itr).SignalPowerSupplyChangeEvent();
        }
    }
}

void SetAndSignalBatteryVoltageState(nn::psm::BatteryVoltageState batteryVoltageState) NN_NOEXCEPT
{
    bool change = (g_BatteryVoltageState != batteryVoltageState);
    g_BatteryVoltageState = batteryVoltageState;
    if ( change )
    {
        for ( auto itr = g_SessionContextList.begin(); itr != g_SessionContextList.end(); itr++ )
        {
            (*itr).SignalBatteryVoltageStateChangeEvent();
        }
    }
}

} // namespace

// APM サーバライブラリ側インタフェース

namespace nn { namespace psm {

void Initialize() NN_NOEXCEPT
{
}

void Finalize() NN_NOEXCEPT
{
}

void OpenSession(Session* pOutSession) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT_NOT_NULL(pOutSession);

    // セッション構造体への参照を格納する場所が無いので reinterpret して IPsmSession に押し込む。
    pOutSession->_pHandle = reinterpret_cast<IPsmSession*>(CreateSessionContext());

    NN_ASSERT_NOT_NULL(pOutSession->_pHandle);
}

void CloseSession(Session* pSession) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT_NOT_NULL(pSession);
    SessionContext* pSessionContext = reinterpret_cast<SessionContext*>(pSession->_pHandle);
    NN_ASSERT_NOT_NULL(pSessionContext);

    DestroySessionContext(pSessionContext);
}

void BindStateChangeEvent(nn::os::SystemEventType* pEvent, Session* pSession) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT_NOT_NULL(pSession);
    SessionContext* pSessionContext = reinterpret_cast<SessionContext*>(pSession->_pHandle);
    NN_ASSERT_NOT_NULL(pSessionContext);

    NN_ASSERT_NOT_NULL(pEvent);
    NN_ASSERT(pSession->_pEvent == nullptr);

    pSessionContext->GetEvent(pEvent);
    pSession->_pEvent = pEvent;
}

void UnbindStateChangeEvent(Session* pSession) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT_NOT_NULL(pSession);
    NN_ASSERT_NOT_NULL(pSession->_pEvent);

    nn::os::DestroySystemEvent(pSession->_pEvent);
    pSession->_pEvent = nullptr;
}

void SetChargerTypeChangeEventEnabled(Session* pSession, bool isEnabled) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT_NOT_NULL(pSession);
    SessionContext* pSessionContext = reinterpret_cast<SessionContext*>(pSession->_pHandle);
    NN_ASSERT_NOT_NULL(pSessionContext);

    pSessionContext->SetChargerTypeChangeEventEnabled(isEnabled);
}

void SetPowerSupplyChangeEventEnabled(Session* pSession, bool isEnabled) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT_NOT_NULL(pSession);
    SessionContext* pSessionContext = reinterpret_cast<SessionContext*>(pSession->_pHandle);
    NN_ASSERT_NOT_NULL(pSessionContext);

    pSessionContext->SetPowerSupplyChangeEventEnabled(isEnabled);
}

void SetBatteryVoltageStateChangeEventEnabled(Session* pSession, bool isEnabled) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT_NOT_NULL(pSession);
    SessionContext* pSessionContext = reinterpret_cast<SessionContext*>(pSession->_pHandle);
    NN_ASSERT_NOT_NULL(pSessionContext);

    pSessionContext->SetBatteryVoltageStateChangeEventEnabled(isEnabled);
}

void EnableEnoughPowerChargeEmulation() NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    g_EnoughPowerChargeEmulation = true;
}

void DisableEnoughPowerChargeEmulation() NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    g_EnoughPowerChargeEmulation = false;
}

void EnableFastBatteryCharging() NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    g_FastBatteryCharging = true;
}

void DisableFastBatteryCharging() NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    g_FastBatteryCharging = false;
}

BatteryVoltageState GetBatteryVoltageState() NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    return g_BatteryVoltageState;
}

bool IsEnoughPowerSupplied() NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    return (g_EnoughPowerChargeEmulation) ? true : (g_ChargerType == ChargerType_EnoughPower);
}

}} // namespace nn::psm

namespace nnt { namespace psm {

void SetChargerType(nn::psm::ChargerType chargerType) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    SetAndSignalChargerType(chargerType);
}

void SetBatteryVoltageState(nn::psm::BatteryVoltageState batteryVoltageState) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    SetAndSignalBatteryVoltageState(batteryVoltageState);
}

}} // namespace nnt::psm
