﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>

#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/apm/apm.h>
#include <nn/apm/apm_System.h>
#include <nn/gpio/gpio.h>
#include <nn/os.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>

namespace nnt { namespace apm {

namespace {

class ApmEnvironment : public ::testing::Environment
{
public:
    virtual void SetUp() override
    {
        nn::apm::Initialize();
        nn::apm::InitializeForSystem();
    }

    virtual void TearDown() override
    {
        nn::apm::Finalize();
        nn::apm::FinalizeForSystem();
    }
};

void ForceThrottling() NN_NOEXCEPT
{
    // Vbus のドロップに必要な時間。
    const int64_t VbusDropDurationMilliSeconds = 5000;

    nn::gpio::Initialize();

    nn::gpio::GpioPadSession session;
    nn::gpio::OpenSession(&session, nn::gpio::GpioPadName_OtgFet2ForSdev);

    nn::gpio::Direction direction = nn::gpio::GetDirection(&session);
    nn::gpio::SetDirection(&session, nn::gpio::Direction_Output);

    nn::gpio::GpioValue value = nn::gpio::GetValue(&session);
    nn::gpio::SetValue(&session, nn::gpio::GpioValue_High);

    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(VbusDropDurationMilliSeconds));

    nn::gpio::SetDirection(&session, direction);
    nn::gpio::SetValue(&session, value);

    nn::gpio::CloseSession(&session);
    nn::gpio::Finalize();
}

} // namespace

TEST(GetThrottlingStateTest, CheckCurrentThrottlingState)
{
    // スロットリングが有効になる性能にする。
    nn::apm::SetPerformanceConfiguration(nn::apm::PerformanceMode_Boost, nn::apm::PerformanceConfiguration_Cpu1020MhzGpu768Mhz);

    nn::apm::ThrottlingState throttlingState;

    nn::apm::GetThrottlingState(&throttlingState);

    EXPECT_EQ(0,     throttlingState.durationNanoSeconds);
    EXPECT_EQ(0x40,  throttlingState.throttlingRateCpuDividend);
    EXPECT_EQ(0x100, throttlingState.throttlingRateCpuDivisor);
    EXPECT_EQ(1,     throttlingState.throttlingRateGpuDividend);
    EXPECT_EQ(4,     throttlingState.throttlingRateGpuDivisor);

    EXPECT_EQ(0x00000001, throttlingState.throttleStatus & 0x00001001);
    EXPECT_EQ(0x00000000, throttlingState.cpuPSkipStatus);
    EXPECT_EQ(0x00000000, throttlingState.gpuPSkipStatus);

    if ( nn::apm::GetPerformanceMode() == nn::apm::PerformanceMode_Normal )
    {
        EXPECT_FALSE(throttlingState.enabled);
    }
    else
    {
        EXPECT_TRUE(throttlingState.enabled);
    }

    EXPECT_FALSE(throttlingState.invoked);
}

TEST(GetThrottlingStateTest, CheckForceThrottling)
{
    // スロットリングが有効になる性能にする。
    nn::apm::SetPerformanceConfiguration(nn::apm::PerformanceMode_Boost, nn::apm::PerformanceConfiguration_Cpu1020MhzGpu768Mhz);

    // 今は余裕を見て 1000 ms としておく。
    const int64_t ThrottlingDurationLimitNanoSeconds = 1000000000LL;
    nn::apm::ThrottlingState throttlingState;

    nn::apm::ClearLastThrottlingState();
    nn::apm::GetLastThrottlingState(&throttlingState);

    EXPECT_EQ(0, throttlingState.durationNanoSeconds);
    EXPECT_EQ(0, throttlingState.throttlingRateCpuDividend);
    EXPECT_EQ(0, throttlingState.throttlingRateCpuDivisor);
    EXPECT_EQ(0, throttlingState.throttlingRateGpuDividend);
    EXPECT_EQ(0, throttlingState.throttlingRateGpuDivisor);

    EXPECT_EQ(0x00000000, throttlingState.throttleStatus & 0x00001001);
    EXPECT_EQ(0x00000000, throttlingState.cpuPSkipStatus);
    EXPECT_EQ(0x00000000, throttlingState.gpuPSkipStatus);

    EXPECT_FALSE(throttlingState.enabled);
    EXPECT_FALSE(throttlingState.invoked);

    if ( nn::apm::GetPerformanceMode() == nn::apm::PerformanceMode_Boost )
    {
        ForceThrottling();

        nn::apm::GetLastThrottlingState(&throttlingState);

        NN_LOG("Throttling duration is %d micro seconds.\n", throttlingState.durationNanoSeconds / 1000);
        EXPECT_GT(ThrottlingDurationLimitNanoSeconds, throttlingState.durationNanoSeconds);
        EXPECT_EQ(0x40,  throttlingState.throttlingRateCpuDividend);
        EXPECT_EQ(0x100, throttlingState.throttlingRateCpuDivisor);
        EXPECT_EQ(1,     throttlingState.throttlingRateGpuDividend);
        EXPECT_EQ(4,     throttlingState.throttlingRateGpuDivisor);

        EXPECT_EQ(0x00000001, throttlingState.throttleStatus & 0x00001001);
        EXPECT_EQ(0x0003fff1, throttlingState.cpuPSkipStatus);
        EXPECT_EQ(0x00300001, throttlingState.gpuPSkipStatus);

        EXPECT_TRUE(throttlingState.enabled);
        EXPECT_TRUE(throttlingState.invoked);

        int count = 0;
        const int countMax = 50;
        while ( nn::apm::GetPerformanceMode() != nn::apm::PerformanceMode_Boost && count < countMax )
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100LL));
            count++;
        }

        EXPECT_EQ(nn::apm::PerformanceMode_Boost, nn::apm::GetPerformanceMode());
    }

    nn::apm::ClearLastThrottlingState();
}

}} // namespace nnt::apm

extern "C" void nninitStartup()
{
    ::testing::AddGlobalTestEnvironment(new nnt::apm::ApmEnvironment);
}
