﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/account.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiPrivate.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/ns/ns_UserResourceManagementApi.h>
#include <nn/oe.h>
#include <nn/os.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/pdm/detail/pdm_AccountPlayEventBuffer.h>
#include <nn/pdm/detail/pdm_Config.h>
#include <nn/pdm/detail/pdm_PlayEventBuffer.h>
#include <nn/pdm/detail/pdm_Util.h>
#include <nn/pdm/pdm_NotifyEventApi.h>
#include <nn/pdm/pdm_NotifyEventApiForDebug.h>
#include <nn/pdm/pdm_QueryApiForDebug.h>
#include <nn/pdm/pdm_QueryApiForSystem.h>
#include <nn/pdm/pdm_SystemTypes.h>
#include <nn/time.h>
#include <nn/util/util_LockGuard.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include "testPdm_Common.h"

using namespace nn;
using namespace nn::pdm;

namespace t = nnt::pdm;

namespace {

    account::Uid TestUsers[] = {account::InvalidUid, account::InvalidUid};


#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    //!------------------------------------------------------------------------------------------------------
    os::SdkRecursiveMutex g_AccountLibraryMutex;
    static const TimeSpan LatencyForAccountOpenCloseNotification = TimeSpan::FromMilliSeconds(50);

    //!------------------------------------------------------------------------------------------------------
    Result DeleteUserPrivate(const account::Uid& user) NN_NOEXCEPT
    {
        const TimeSpan deleteTimeout = TimeSpan::FromMilliSeconds(10000);
        {
            NN_UTIL_LOCK_GUARD(g_AccountLibraryMutex);
            account::Finalize();
            account::InitializeForAdministrator();
            NN_UTIL_SCOPE_EXIT
            {
                account::Finalize();
                account::Initialize();
            };

            nn::ns::ProgressMonitorForDeleteUserSaveDataAll progress;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ns::DeleteUserSaveDataAll(&progress, user));
            nn::os::SystemEvent deletedEvent;
            progress.GetSystemEvent(&deletedEvent);
            NN_ABORT_UNLESS(deletedEvent.TimedWait(deleteTimeout));
            NN_ABORT_UNLESS_RESULT_SUCCESS(progress.GetResult());
            NN_RESULT_DO(account::DeleteUser(user));
        }
        os::SleepThread(TimeSpan::FromMilliSeconds(50));
        NN_RESULT_SUCCESS;
    }

    //!------------------------------------------------------------------------------------------------------
    void RegisterNewUsers(const int newRegisterCount)
    {
        NN_SDK_REQUIRES(0 <= newRegisterCount && newRegisterCount <= account::UserCountMax);

        // 古いユーザーを削除。
        NN_LOG("[testPdm] Cleanup old users, And new users registration.\n");

        int nUsers;
        account::Uid users[account::UserCountMax];
        NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&nUsers, users, NN_ARRAY_SIZE(users)));
        for (int i = 0; i < nUsers; ++i)
        {
            const auto& uid = users[i];
            NN_LOG("[testPdm] Deleted old user[0x%llx-0x%llx].\n", uid._data[0], uid._data[1]);
            DeleteUserPrivate(uid);
        }

        if (newRegisterCount > 0)
        {
            NN_UTIL_LOCK_GUARD(g_AccountLibraryMutex);
            account::Finalize();
            account::InitializeForAdministrator();
            NN_UTIL_SCOPE_EXIT
            {
                account::Finalize();
                account::Initialize();
            };

            account::Uid newUsers[account::UserCountMax];
            for (int i = 0; i < newRegisterCount; ++i)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(account::BeginUserRegistration(&newUsers[i]));
                NN_ABORT_UNLESS_RESULT_SUCCESS(account::CompleteUserRegistrationForcibly(newUsers[i]));
                NN_LOG("[testPdm] Registered new user[0x%llx-0x%llx].\n", newUsers[i]._data[0], newUsers[i]._data[1]);
            }
        }
    }

    //!------------------------------------------------------------------------------------------------------
    Result OpenUserPrivate(account::UserHandle* pOutHandle, const account::Uid& user) NN_NOEXCEPT
    {
        g_AccountLibraryMutex.Lock();
        auto result = account::OpenUser(pOutHandle, user);
        if (result.IsFailure())
        {
            g_AccountLibraryMutex.Unlock();
            return result;
        }
        os::SleepThread(LatencyForAccountOpenCloseNotification);
        NN_RESULT_SUCCESS;
    }

    //!------------------------------------------------------------------------------------------------------
    void CloseUserPrivate(const account::UserHandle& handle) NN_NOEXCEPT
    {
        account::CloseUser(handle);
        g_AccountLibraryMutex.Unlock();
        os::SleepThread(LatencyForAccountOpenCloseNotification);
    }
#endif

}

class NotifyTest : public testing::Test
{
protected:
    virtual void SetUp()
    {
        pdm::NotifyClearAllEvent();
    }
    virtual void TearDown()
    {
    }

    static void SetUpTestCase()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(time::Initialize());
        pdm::InitializeForNotification();
        pdm::InitializeForQuery();

#if defined( NN_BUILD_CONFIG_OS_WIN )

        account::InitializeForAdministrator();

        int userCount;
        account::Uid uids[account::UserCountMax];
        NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&userCount, uids, NN_ARRAY_SIZE(uids)));
        for( int i = 0; i < userCount; i++ )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(account::DeleteUser(uids[i]));
        }

        for (int i = 0; i < NN_ARRAY_SIZE(TestUsers); ++i)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(account::BeginUserRegistration(&TestUsers[i]));
            NN_ABORT_UNLESS_RESULT_SUCCESS(account::CompleteUserRegistrationForcibly(TestUsers[i]));
        }
        pdm::detail::AccountPlayEventProvider::GetInstance().Initialize(TestUsers, NN_ARRAY_SIZE(TestUsers), pdm::detail::MigrationState::Done);

#elif defined(NN_BUILD_CONFIG_OS_HORIZON)

        oe::Initialize();
        account::Initialize();

#endif

    }

    static void TearDownTestCase()
    {

#if defined( NN_BUILD_CONFIG_OS_WIN )
        for (int i = 0; i < NN_ARRAY_SIZE(TestUsers); ++i)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(account::DeleteUser(TestUsers[i]));
        }
        pdm::detail::AccountPlayEventProvider::GetInstance().UpdateUserRegistry(nullptr, 0);
#endif

        pdm::FinalizeForQuery();
        pdm::FinalizeForNotification();
        time::Finalize();
    }
};

// イベントシーケンスの妥当性をチェックしている ASSERT に引っかかるので通常は動かさない。
/*
TEST_F(NotifyTest, AppletEvent)
{
    ncm::ProgramId programId1 = { 0x0000000011111111LL };
    uint32_t version = 1;

    // Type が 0 から増加するように定義（チェックをループで済ますため）
    pdm::NotifyAppletEvent(nn::pdm::AppletEventType::Launch, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);
    pdm::NotifyAppletEvent(nn::pdm::AppletEventType::Exit, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);
    pdm::NotifyAppletEvent(nn::pdm::AppletEventType::InFocus, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);
    pdm::NotifyAppletEvent(nn::pdm::AppletEventType::OutOfFocus, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);
    pdm::NotifyAppletEvent(nn::pdm::AppletEventType::Background, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);
    pdm::NotifyAppletEvent(nn::pdm::AppletEventType::Terminate, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);
    pdm::NotifyAppletEvent(nn::pdm::AppletEventType::AbnormalExit, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);

#if defined( NN_BUILD_CONFIG_OS_WIN )
    pdm::detail::PlayEventBuffer::GetInstance().Flush();
    // Windows 環境では PlayEventBuffer を触れるので直接確認。
    ASSERT_EQ(7, pdm::detail::PlayEventBuffer::GetInstance().GetCount());
    pdm::PlayEvent readEvent[7];
    pdm::detail::PlayEventBuffer::GetInstance().BeginRead();
    NN_UTIL_SCOPE_EXIT{ pdm::detail::PlayEventBuffer::GetInstance().EndRead(); };
    auto count = pdm::detail::PlayEventBuffer::GetInstance().Read(readEvent, 0, 7);
    ASSERT_EQ(7, count);
    for( int i = 0; i < 7; i++ )
    {
        EXPECT_EQ(pdm::PlayEventCategory::Applet, readEvent[i].eventCategory);
        EXPECT_EQ(static_cast<pdm::AppletEventType>(i), readEvent[i].appletEventData.eventType);
        EXPECT_EQ(programId1.value, pdm::detail::GetApplicationId(readEvent[i]).value);
        EXPECT_EQ(version, readEvent[i].appletEventData.version);
        EXPECT_EQ(applet::AppletId_Application, readEvent[i].appletEventData.appletId);
        EXPECT_EQ(ncm::StorageId::Card, readEvent[i].appletEventData.storageId);
        EXPECT_EQ(ns::PlayLogPolicy::All, readEvent[i].appletEventData.playLogPolicy);
    }
#endif
}
*/

TEST_F(NotifyTest, AppletEventSimpleSequence)
{
    ncm::ProgramId programId1 = { 0x0000000011111111LL };
    ncm::ProgramId programId2 = { 0x0000000022222222LL };
    uint32_t version = 1;

    // Notify によるイベントの登録。
    pdm::NotifyPowerStateChangeEvent(nn::pdm::PowerStateChangeEventType::On);

    pdm::NotifyAppletEvent(nn::pdm::AppletEventType::Launch, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);
    pdm::NotifyAppletEvent(nn::pdm::AppletEventType::OutOfFocus, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);
    pdm::NotifyAppletEvent(nn::pdm::AppletEventType::InFocus, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);
    pdm::NotifyAppletEvent(nn::pdm::AppletEventType::Exit, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);

    pdm::NotifyPowerStateChangeEvent(nn::pdm::PowerStateChangeEventType::Sleep);
    pdm::NotifyPowerStateChangeEvent(nn::pdm::PowerStateChangeEventType::Awake);

    pdm::NotifyAppletEvent(nn::pdm::AppletEventType::Launch, programId2, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);
    pdm::NotifyAppletEvent(nn::pdm::AppletEventType::OutOfFocus, programId2, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);
    pdm::NotifyAppletEvent(nn::pdm::AppletEventType::InFocus, programId2, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);
    pdm::NotifyAppletEvent(nn::pdm::AppletEventType::Exit, programId2, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);

    pdm::NotifyPowerStateChangeEvent(nn::pdm::PowerStateChangeEventType::On);

    // QueryApplicationEvent で登録内容の確認。
    const int EventQueryNum = 50;
    pdm::ApplicationEvent appEvent[EventQueryNum];
    int count = pdm::QueryApplicationEvent(appEvent, EventQueryNum, 0);
    ASSERT_EQ(8, count); // AppletEvent のみの数。

    for( int i = 0; i < 4; i++ )
    {
        // 0 は On なので最初のイベントのインデックスは 1。
        EXPECT_EQ(i + 1, appEvent[i].eventTime.eventIndex);
        EXPECT_EQ(programId1.value, appEvent[i].applicationId.value);
    }

    for( int i = 4; i < 8; i++ )
    {
        EXPECT_EQ(i + 3, appEvent[i].eventTime.eventIndex);
        EXPECT_EQ(programId2.value, appEvent[i].applicationId.value);
    }

    for( int i = 0; i < 2; i++ )
    {
        EXPECT_EQ(pdm::AppletEventType::Launch, appEvent[i * 4 + 0].eventType);
        EXPECT_EQ(pdm::AppletEventType::OutOfFocus, appEvent[i * 4 + 1].eventType);
        EXPECT_EQ(pdm::AppletEventType::InFocus, appEvent[i * 4 + 2].eventType);
        EXPECT_EQ(pdm::AppletEventType::Exit, appEvent[i * 4 + 3].eventType);
    }

    // QueryPlayStatistics で登録内容の確認。
    const int StatisticsQueryNum = 5;
    pdm::PlayStatistics appStatistics[StatisticsQueryNum];
    count = pdm::QueryPlayStatistics(appStatistics, StatisticsQueryNum);
    EXPECT_EQ(2, count);
    EXPECT_EQ(1, appStatistics[0].totalPlayCount);
    EXPECT_EQ(1, appStatistics[0].firstEventTime.eventIndex);
    EXPECT_EQ(4, appStatistics[0].latestEventTime.eventIndex);
    EXPECT_EQ(1, appStatistics[1].totalPlayCount);
    EXPECT_EQ(7, appStatistics[1].firstEventTime.eventIndex);
    EXPECT_EQ(10, appStatistics[1].latestEventTime.eventIndex);
}

TEST_F(NotifyTest, Power)
{
    // Type が 0 から増加するように定義（チェックをループで済ますため）
    pdm::NotifyPowerStateChangeEvent(pdm::PowerStateChangeEventType::On);
    pdm::NotifyPowerStateChangeEvent(pdm::PowerStateChangeEventType::Off);
    pdm::NotifyPowerStateChangeEvent(pdm::PowerStateChangeEventType::Sleep);
    pdm::NotifyPowerStateChangeEvent(pdm::PowerStateChangeEventType::Awake);
    pdm::NotifyPowerStateChangeEvent(pdm::PowerStateChangeEventType::BackgroundServicesAwake);
    pdm::NotifyPowerStateChangeEvent(pdm::PowerStateChangeEventType::Terminate);

#if defined( NN_BUILD_CONFIG_OS_WIN )
    // Windows 環境では PlayEventBuffer を触れるので直接確認。
    pdm::detail::PlayEventBuffer::GetInstance().Flush();
    ASSERT_EQ(6, pdm::detail::PlayEventBuffer::GetInstance().GetCount());
    pdm::PlayEvent readEvent[6];
    pdm::detail::PlayEventBuffer::GetInstance().BeginRead();
    NN_UTIL_SCOPE_EXIT{ pdm::detail::PlayEventBuffer::GetInstance().EndRead(); };
    auto count = pdm::detail::PlayEventBuffer::GetInstance().Read(readEvent, 0, 6);
    ASSERT_EQ(6, count);
    for( int i = 0; i < 6; i++ )
    {
        EXPECT_EQ(pdm::PlayEventCategory::PowerStateChange, readEvent[i].eventCategory);
        EXPECT_EQ(static_cast<pdm::PowerStateChangeEventType>(i), readEvent[i].powerStateChangeEventData.eventType);
    }
#endif
}

TEST_F(NotifyTest, OperationMode)
{
    // Type が 0 から増加するように定義（チェックをループで済ますため）
    pdm::NotifyOperationModeChangeEvent(oe::OperationMode_Handheld);
    pdm::NotifyOperationModeChangeEvent(oe::OperationMode_Console);

#if defined( NN_BUILD_CONFIG_OS_WIN )
    // Windows 環境では PlayEventBuffer を触れるので直接確認。
    pdm::detail::PlayEventBuffer::GetInstance().Flush();
    ASSERT_EQ(2, pdm::detail::PlayEventBuffer::GetInstance().GetCount());
    pdm::PlayEvent readEvent[2];
    pdm::detail::PlayEventBuffer::GetInstance().BeginRead();
    NN_UTIL_SCOPE_EXIT{ pdm::detail::PlayEventBuffer::GetInstance().EndRead(); };
    auto count = pdm::detail::PlayEventBuffer::GetInstance().Read(readEvent, 0, 2);
    ASSERT_EQ(2, count);
    for( int i = 0; i < 2; i++ )
    {
        EXPECT_EQ(pdm::PlayEventCategory::OperationModeChange, readEvent[i].eventCategory);
        EXPECT_EQ(static_cast<pdm::OperationMode>(i), readEvent[i].operationModeEventData.operationMode);
    }
#endif
}

TEST_F(NotifyTest, NotifyEventForDebug)
{
    const nn::ncm::ApplicationId TestApplicationId1 = nn::ncm::ApplicationId{ 0x1111111111111111ull };
    const uint32_t  TestVersion = 0;
    nn::account::Uid user1;
    user1._data[0] = 1u;
    user1._data[1] = 2u;
    const pdm::PlayEvent playEvent[] = {
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, user1, 0, 0, 0),
        t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, user1, 60, 60, 60),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Background, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 60, 60, 60),
        t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, TestApplicationId1, TestVersion, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 60, 60, 60),
    };
    const auto count = sizeof(playEvent) / sizeof(PlayEvent);
    pdm::NotifyEventForDebug(playEvent, count);

    pdm::PlayEvent queriedPlayEvent[count];
    pdm::QueryPlayEvent(queriedPlayEvent, count, 0);

    for( size_t i = 0; i < count; i++ )
    {
        EXPECT_EQ(playEvent[i], queriedPlayEvent[i]);
    }
}


#if defined( NN_BUILD_CONFIG_OS_HORIZON )
TEST_F(NotifyTest, AccountOpenClose)
{
    int count;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserCount(&count));
    if( count == 0 )
    {
        NN_LOG("No account is registered. Skip this test.\n");
        SUCCEED();
        return;
    }

    int actualLength;
    account::Uid uid;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&actualLength, &uid, 1));

    ncm::ProgramId programId1 = { 0x0000000011111111LL };
    uint32_t version = 1;

    const int StatisticsQueryNum = 5;
    pdm::PlayStatistics playStatistics[StatisticsQueryNum];

    // 最も単純な正常系シーケンスの最小単位を Notify + Query してテスト。
    {
        // 起動 → InFocus → オープン → クローズ → Background → 終了
        pdm::NotifyAppletEvent(nn::pdm::AppletEventType::Launch, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);
        pdm::NotifyAppletEvent(nn::pdm::AppletEventType::InFocus, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);

        account::UserHandle userHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&userHandle, uid));
         // Open と Close に間がないと、イベントが記録されないことがあるので Sleep で調整。
        os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        nn::account::CloseUser(userHandle);

        // TORIAEZU : 記録されるイベント順が Open -> Close -> Exit となるように Sleep で調整。
        //            Notify の呼び出しが早いと pdm 側のアカウント処理スレッドよりも先に Exit がきてテスト結果が崩れる。
        //            それはそれで正常系なものの、QueryPlayStatistics でイベントのインデックスを確認しているこのテストでは都合が悪い。
        os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        pdm::NotifyAppletEvent(nn::pdm::AppletEventType::Background, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);
        pdm::NotifyAppletEvent(nn::pdm::AppletEventType::Exit, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);

        // QueryPlayStatistics で登録内容の確認。
        // 注釈) アカウントDB のイベント単位は「アカウントOpen/Close期間」及び「Applet<InFocus> 期間」。
        count = pdm::QueryPlayStatistics(playStatistics, StatisticsQueryNum, uid);  // アカウントDB経由の PlayStatistics.
        ASSERT_EQ(1, count);
        EXPECT_EQ(programId1.value, playStatistics[0].applicationId.value);
        EXPECT_EQ(1, playStatistics[0].totalPlayCount);
        EXPECT_EQ(1, playStatistics[0].firstEventTime.eventIndex);  // 記録されるイベントは InFocus 単位であり、InOpen → InFocus の順に記録される事からインデクスは 1。
        EXPECT_EQ(1, playStatistics[0].latestEventTime.eventIndex); // １プレイ１オープンの場合、記録される InFocus レコードは１つなので、最後のインデクスも 1。

        pdm::NotifyClearAllEvent();
    }
}
#endif

TEST_F(NotifyTest, SuspendResume)
{
    int count;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserCount(&count));
    if (count == 0)
    {
        NN_LOG("No account is registered. Skip this test.\n");
        SUCCEED();
        return;
    }

#if defined( NN_BUILD_CONFIG_OS_HORIZON )

    int actualLength;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&actualLength, TestUsers, NN_ARRAY_SIZE(TestUsers)));
    ASSERT_TRUE(actualLength > 0);

#endif

    const account::Uid& uid = TestUsers[0];
    account::Uid fakeUser = {{0xffff0000ffff0000ull, 0xffff0000ffff0000ull}};
    EXPECT_TRUE(pdm::ResultServiceNotAvailable::Includes(pdm::SuspendUserAccountEventService(fakeUser)));
    EXPECT_TRUE(pdm::ResultServiceNotAvailable::Includes(pdm::ResumeUserAccountEventService(fakeUser)));

    NNT_EXPECT_RESULT_SUCCESS(pdm::SuspendUserAccountEventService(uid));
    NNT_EXPECT_RESULT_SUCCESS(pdm::SuspendUserAccountEventService(uid));
    NNT_EXPECT_RESULT_SUCCESS(pdm::SuspendUserAccountEventService(uid));

    NNT_EXPECT_RESULT_SUCCESS(pdm::ResumeUserAccountEventService(uid));
    NNT_EXPECT_RESULT_SUCCESS(pdm::ResumeUserAccountEventService(uid));
    NNT_EXPECT_RESULT_SUCCESS(pdm::ResumeUserAccountEventService(uid));

    {
        ncm::ProgramId programId1 = {0x0000000011111111LL};
        uint32_t version = 1;

#if defined( NN_BUILD_CONFIG_OS_WIN )

        const pdm::PlayEvent playEvent[] = {
            t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Launch, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
            t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::InFocus, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 0, 0, 0),
            t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Open, uid, 0, 0, 0),
            t::MakeUserAccountEventInMinutes(pdm::UserAccountEventType::Close, uid, 60, 60, 60),
            t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Background, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 60, 60, 60),
            t::MakeAppletEventInMinutes(nn::pdm::AppletEventType::Exit, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All, 60, 60, 60),
        };
        pdm::NotifyEventForDebug(playEvent, sizeof(playEvent) / sizeof(PlayEvent));

#elif defined( NN_BUILD_CONFIG_OS_HORIZON )

        pdm::NotifyAppletEvent(nn::pdm::AppletEventType::Launch, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);
        pdm::NotifyAppletEvent(nn::pdm::AppletEventType::InFocus, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);

        account::UserHandle userHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&userHandle, uid));
        // Open と Close に間がないと、イベントが記録されないことがあるので Sleep で調整。
        os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        nn::account::CloseUser(userHandle);

        // TORIAEZU : 記録されるイベント順が Open -> Close -> Exit となるように Sleep で調整。
        //            Notify の呼び出しが早いと pdm 側のアカウント処理スレッドよりも先に Exit がきてテスト結果が崩れる。
        //            それはそれで正常系なものの、QueryPlayStatistics でイベントのインデックスを確認しているこのテストでは都合が悪い。
        os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

        pdm::NotifyAppletEvent(nn::pdm::AppletEventType::Background, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);
        pdm::NotifyAppletEvent(nn::pdm::AppletEventType::Exit, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);

#endif

        // QueryPlayStatistics で登録内容の確認。
        // 注釈) アカウントDB のイベント単位は「アカウントOpen/Close期間」及び「Applet<InFocus> 期間」。
        pdm::PlayStatistics playStatistics[5];
        count = pdm::QueryPlayStatistics(playStatistics, NN_ARRAY_SIZE(playStatistics), uid);  // アカウントDB経由の PlayStatistics.
        ASSERT_EQ(1, count);
        EXPECT_EQ(programId1.value, playStatistics[0].applicationId.value);
        EXPECT_EQ(1, playStatistics[0].totalPlayCount);
        EXPECT_EQ(1, playStatistics[0].firstEventTime.eventIndex);  // 記録されるイベントは InFocus 単位であり、InOpen → InFocus の順に記録される事からインデクスは 1。
        EXPECT_EQ(1, playStatistics[0].latestEventTime.eventIndex); // １プレイ１オープンの場合、記録される InFocus レコードは１つなので、最後のインデクスも 1。
    }
}

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
// ユーザーを動的に追加して、いきなり Query → イベント送信。
TEST_F(NotifyTest, DelayedAddUserAndSendPlayRecord)
{
    RegisterNewUsers(1);

    int nUsers;
    account::Uid users[account::UserCountMax];
    NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&nUsers, users, NN_ARRAY_SIZE(users)));
    if (0 >= nUsers)
    {
        NN_LOG("[testPdm] Could not create the new user.\n");
        FAIL();
        return;
    }

    // マウントが行われてない状態でのクエリ
    const auto& uid = users[0];
    account::UserHandle userHandle;
    pdm::PlayStatistics playStatistics[5];
    {
        std::memset(playStatistics, 0, sizeof(playStatistics));
        int count = pdm::QueryPlayStatistics(playStatistics, NN_ARRAY_SIZE(playStatistics), uid);  // アカウントDB経由の PlayStatistics.
        EXPECT_EQ(0, count);
    }

    uint32_t version = 1;
    ncm::ProgramId programId1 = {0x0000000011111111LL};

    pdm::NotifyAppletEvent(nn::pdm::AppletEventType::Launch, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);
    pdm::NotifyAppletEvent(nn::pdm::AppletEventType::InFocus, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);

    NN_ABORT_UNLESS_RESULT_SUCCESS(OpenUserPrivate(&userHandle, uid));
    CloseUserPrivate(userHandle);

    pdm::NotifyAppletEvent(nn::pdm::AppletEventType::Background, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);
    pdm::NotifyAppletEvent(nn::pdm::AppletEventType::Exit, programId1, version, nn::applet::AppletId_Application, nn::ncm::StorageId::Card, nn::ns::PlayLogPolicy::All);

    // QueryPlayStatistics で登録内容の確認。
    // 注釈) アカウントDB のイベント単位は「アカウントOpen/Close期間」及び「Applet<InFocus> 期間」。
    std::memset(playStatistics, 0, sizeof(playStatistics));
    int count = pdm::QueryPlayStatistics(playStatistics, NN_ARRAY_SIZE(playStatistics), uid);  // アカウントDB経由の PlayStatistics.
    ASSERT_EQ(1, count);
    EXPECT_EQ(programId1.value, playStatistics[0].applicationId.value);
    EXPECT_EQ(1, playStatistics[0].totalPlayCount);
    EXPECT_EQ(1, playStatistics[0].firstEventTime.eventIndex);  // 記録されるイベントは InFocus 単位であり、InOpen → InFocus の順に記録される事からインデクスは 1。
    EXPECT_EQ(1, playStatistics[0].latestEventTime.eventIndex); // １プレイ１オープンの場合、記録される InFocus レコードは１つなので、最後のインデクスも 1。
}
#endif
