﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/account.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiPrivate.h>
#include <nn/account/account_Result.h>
#include <nn/oe.h>
#include <nn/os.h>
#include <nn/pdm/detail/pdm_AccountPlayEventBuffer.h>
#include <nn/pdm/detail/pdm_Config.h>
#include <nn/pdm/detail/pdm_PlayEventBuffer.h>
#include <nn/pdm/detail/pdm_PlayEventFactory.h>
#include <nn/pdm/detail/pdm_QueryServiceHelper.h>
#include <nn/pdm/pdm_NotifyEventApi.h>
#include <nn/pdm/pdm_QueryApi.h>
#include <nn/pdm/pdm_QueryApiForDebug.h>
#include <nn/pdm/pdm_QueryApiForSystem.h>
#include <nn/time.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

using namespace nn;

namespace {

//!------------------------------------------------------------------------------------------------------
// 初期登録ユーザー数 ( 0 ～ 8 )
const int InitialRegisteredUserCount = 5;
NN_STATIC_ASSERT(0 <= InitialRegisteredUserCount && InitialRegisteredUserCount <= account::UserCountMax);

//!------------------------------------------------------------------------------------------------------
class TestContext
{
public:

    static TestContext& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(TestContext, s_Instance);
        return s_Instance;
    }

    void Initialize() NN_NOEXCEPT
    {
#if defined( NN_BUILD_CONFIG_OS_WIN )

        account::InitializeForAdministrator();  // ユーザー登録したいので管理者権限で初期化。

        int userCount;
        account::Uid uids[account::UserCountMax];
        NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&userCount, uids, NN_ARRAY_SIZE(uids)));
        for( int i = 0; i < userCount; i++ )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(account::DeleteUser(uids[i]));
        }

        for (int i = 0; i < InitialRegisteredUserCount; ++i)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(account::BeginUserRegistration(&m_TestUsers[i]));
            NN_ABORT_UNLESS_RESULT_SUCCESS(account::CompleteUserRegistrationForcibly(m_TestUsers[i]));
        }
        account::Finalize();    // ユーザー登録終了したので管理者権限を終了。
        account::Initialize();  // UserOpen/UserClose 用に一般権限で初期化。

        pdm::InitializeForQuery();
        pdm::InitializeForNotification();

        pdm::detail::AccountPlayEventProvider::GetInstance().Initialize(m_TestUsers, InitialRegisteredUserCount, pdm::detail::MigrationState::None);

#else

        account::Initialize();  // UserOpen/UserClose 用に一般権限で初期化。
        pdm::InitializeForQuery();
        pdm::InitializeForNotification();

#endif
    }

    void Finalize() NN_NOEXCEPT
    {

#if defined( NN_BUILD_CONFIG_OS_WIN )

        pdm::detail::AccountPlayEventProvider::GetInstance().UpdateUserRegistry(nullptr, 0);
        pdm::FinalizeForNotification();
        pdm::FinalizeForQuery();

        account::Finalize();    // UserOpen/UserClose 用の一般権限を終了。
        account::InitializeForAdministrator();  // ユーザー削除したいので管理者権限で初期化。
        for (int i = 0; i < InitialRegisteredUserCount; ++i)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(account::DeleteUser(m_TestUsers[i]));
        }
        account::Finalize();    // ユーザー削除終了したので管理者権限を終了。

#else

        pdm::FinalizeForNotification();
        pdm::FinalizeForQuery();

#endif
    }

    const account::Uid& GetUser(int index) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(0 <= index && index < account::UserCountMax);
        return m_TestUsers[index];
    }

private:
    account::Uid m_TestUsers[account::UserCountMax];
};

//------------------------------------------------------------------------------------------------------
template<typename TEntryData, typename Predicate, int EntryCapacity>
TEntryData* FindEntryFrom(TEntryData(&entries)[EntryCapacity], Predicate onMatch) NN_NOEXCEPT
{
    for (int i = 0; i < EntryCapacity; ++i)
    {
        if (onMatch(entries[i]))
        {
            return &entries[i];
        }
    }
    return nullptr;
}

//!------------------------------------------------------------------------------------------------------
//! @brief      account::OpenUser/CloseUser の呼び出しレイテンシ。
//!
//! @details    ユーザー状態遷移イベントが pdm に通知される前に次の状態遷移要求を発生させないようにするための目安。
//!             pdm のログ出力量やユーザーアカウントイベント監視スレッドの処理負荷などに依存するため。
//!             ちなみに前述の「pdmに通知される前に次の状態遷移要求を発生させる」場合は 5ms などで良い。
//!             発生するケース例としては以下。
//!                 - GetUserStateChangeNotifier 経由でのシステムイベント1回目( ユーザーA Open )が発生。
//!                 - pdm の受信スレッドが遅延して、1回目のイベント検知できないまま、システムイベント2回目( ユーザーA Close )が発生。
//!                 - 1回目の変動を見逃してしまい、2回目のイベント発生後に、account::ListOpenUsers で確認。
//!                 - 結果、ユーザーA の Open状態 を取り逃す。
//!
//!             上記のような事象に対しての検証は本テストには含まない。
//!             これは同検証が「アカウントシステムが管理したアカウント状態遷移とクライアントプログラム側が認識した状態遷移の整合性」という主題になるため。
static const TimeSpan LatencyForAccountOpenCloseNotification = TimeSpan::FromMilliSeconds(30);

//------------------------------------------------------------------------------------------------------
// プリセットプログラムID
static const ncm::ProgramId         HomeAppletId{0x0100000000002064};
static const ncm::ApplicationId     SelfApplicationId{0x0005000C10000000};

//------------------------------------------------------------------------------------------------------
Result OpenUserPrivate(account::UserHandle* pOutHandle, const account::Uid& user) NN_NOEXCEPT
{
    NN_RESULT_DO(account::OpenUser(pOutHandle, user));
    os::SleepThread(LatencyForAccountOpenCloseNotification);
    NN_RESULT_SUCCESS;
}

//------------------------------------------------------------------------------------------------------
void CloseUserPrivate(const account::UserHandle& handle) NN_NOEXCEPT
{
    account::CloseUser(handle);
    os::SleepThread(LatencyForAccountOpenCloseNotification);
}

//------------------------------------------------------------------------------------------------------
struct EventContext
{
    enum class Type : uint8_t
    {
        Applet,
        UserOpen,
        UserClose,
        PowerChange,
    };

    struct Applet
    {
        ncm::ProgramId          programId;
        pdm::AppletEventType    appletEvent;
        applet::AppletId        appletId;
        ncm::StorageId          storageId;
    };
    struct User
    {
        uint32_t                userIndex;
    };
    struct PowerChange
    {
        pdm::PowerStateChangeEventType powerType;
    };

    Type            type;
    union
    {
        Applet      applet;
        User        user;
        PowerChange power;
    };

    inline static EventContext MakeSelfApplicationEvent(const pdm::AppletEventType& type_) NN_NOEXCEPT
    {
        EventContext result{EventContext::Type::Applet};
        result.applet = {SelfApplicationId, type_, applet::AppletId_Application, ncm::StorageId::Card};
        return result;
    }
    inline static EventContext MakeHomeAppletEvent(const pdm::AppletEventType& type_) NN_NOEXCEPT
    {
        EventContext result{EventContext::Type::Applet};
        result.applet = {HomeAppletId, type_, applet::AppletId_SystemAppletMenu, ncm::StorageId::BuildInSystem};
        return result;
    }
    inline static EventContext MakeUserOpen(uint32_t userIndex) NN_NOEXCEPT
    {
        EventContext result{EventContext::Type::UserOpen};
        result.user.userIndex = userIndex;
        return result;
    }
    inline static EventContext MakeUserClose(uint32_t userIndex) NN_NOEXCEPT
    {
        EventContext result{EventContext::Type::UserClose};
        result.user.userIndex = userIndex;
        return result;
    }
    inline static EventContext MakePowerChange(const pdm::PowerStateChangeEventType& type) NN_NOEXCEPT
    {
        EventContext result{EventContext::Type::PowerChange};
        result.power.powerType = type;
        return result;
    }

    static void ChangeUserIndex(EventContext* const pEntries, const uint32_t entryCount, const uint32_t userIndex) NN_NOEXCEPT
    {
        for (uint32_t i = 0; i < entryCount; ++i)
        {
            auto& entry = pEntries[i];
            if (EventContext::Type::UserOpen == entry.type || EventContext::Type::UserClose == entry.type)
            {
                entry.user.userIndex = userIndex;
            }
        }
    }
};

//------------------------------------------------------------------------------------------------------
class OpenHandle
{
public:
    explicit OpenHandle() NN_NOEXCEPT : m_Used(false)
    {
        m_Handle = {{0llu, 0llu}, 0llu};
    }

    nn::Result Open(const account::Uid& uid) NN_NOEXCEPT
    {
        m_Used = true;
        return OpenUserPrivate(&m_Handle, uid);
    }

    void Close() NN_NOEXCEPT
    {
        if (m_Used)
        {
            m_Used = false;
            CloseUserPrivate(m_Handle);
        }
    }

private:
    account::UserHandle m_Handle;
    bool                m_Used;
};

//------------------------------------------------------------------------------------------------------
struct EventNotifierReport
{
    struct OpenCounter
    {
        account::Uid    uid;
        uint32_t        count;
        bool            opened;

        OpenCounter() NN_NOEXCEPT : uid(account::InvalidUid), count(0), opened(false) {}

        NN_FORCEINLINE void Reset() NN_NOEXCEPT
        {
            uid = account::InvalidUid;
            count = 0;
            opened = false;
        }

        NN_FORCEINLINE void Open(const account::Uid& userId) NN_NOEXCEPT
        {
            uid = userId;
            opened = true;
        }

        NN_FORCEINLINE bool Close() NN_NOEXCEPT
        {
            if (opened)
            {
                opened = false;
                ++count;
                return true;
            }
            return false;
        }
    };

    struct FocusCounter
    {
        uint32_t        count;
        ncm::ProgramId  required[4];

        FocusCounter() NN_NOEXCEPT
        {
            Reset();
        }

        void Reset() NN_NOEXCEPT
        {
            count = 0;
            FindEntryFrom(required, [](ncm::ProgramId& entry) -> bool {
                entry = ncm::ProgramId::GetInvalidId();
                return false;
            });
        }

        void Begin(const ncm::ProgramId& id) NN_NOEXCEPT
        {
            if (nullptr == FindEntryFrom(required, [&id](const ncm::ProgramId& entry) -> bool {
                return (id == entry);
            }))
            {
                auto pEntry = FindEntryFrom(required, [](const ncm::ProgramId& entry) -> bool {
                    return (entry == ncm::ProgramId::GetInvalidId());
                });
                if (nullptr != pEntry)
                {
                    *pEntry = id;
                }
            }
        }

        void End(const ncm::ProgramId& id, bool record) NN_NOEXCEPT
        {
            FindEntryFrom(required, [&](ncm::ProgramId& entry) -> bool {
                if (id == entry)
                {
                    if (record)
                    {
                        ++count;
                    }
                    entry = ncm::ProgramId::GetInvalidId();
                    return true;
                }
                return false;
            });
        }

        void CountUpForActiveAll() NN_NOEXCEPT
        {
            FindEntryFrom(required, [&](ncm::ProgramId& entry) -> bool {
                count += (entry != ncm::ProgramId::GetInvalidId()) ? 1 : 0;
                return false;
            });

        }
    };


    struct User
    {
        OpenCounter     inOpen;
        FocusCounter    inFocus;

        NN_FORCEINLINE void Reset() NN_NOEXCEPT
        {
            inOpen.Reset();
            inFocus.Reset();
        }

        NN_FORCEINLINE void Open(const account::Uid& uid) NN_NOEXCEPT
        {
            inOpen.Open(uid);
        }

        NN_FORCEINLINE bool Close() NN_NOEXCEPT
        {
            return inOpen.Close();
        }

        NN_FORCEINLINE void FocusIn(const ncm::ProgramId& id) NN_NOEXCEPT
        {
            inFocus.Begin(id);
        }

        NN_FORCEINLINE void FocusOut(const ncm::ProgramId& id, bool record) NN_NOEXCEPT
        {
            inFocus.End(id, record);
        }

        NN_FORCEINLINE void FocusOutWithClose() NN_NOEXCEPT
        {
            inFocus.CountUpForActiveAll();
        }

        NN_FORCEINLINE bool IsOpened() const NN_NOEXCEPT
        {
            return inOpen.opened;
        }
    };

    explicit EventNotifierReport() NN_NOEXCEPT
    {
        FindEntryFrom(users, [](User& entry) -> bool {
            entry.Reset();
            return false;
        });
    }

    void Open(int index, const account::Uid& uid) NN_NOEXCEPT
    {
        users[index].Open(uid);
    }

    void Close(int index) NN_NOEXCEPT
    {
        auto& user = users[index];
        if (user.Close())
        {
            user.FocusOutWithClose();
        }
    }

    void FocusIn(const ncm::ProgramId& id) NN_NOEXCEPT
    {
        FindEntryFrom(users, [&id](User& entry) -> bool {
            entry.FocusIn(id);
            return false;
        });
    }

    void FocusOut(const ncm::ProgramId& id) NN_NOEXCEPT
    {
        FindEntryFrom(users, [&id](User& entry) -> bool {
            entry.FocusOut(id, entry.IsOpened());
            return false;
        });
    }

    void PowerOff() NN_NOEXCEPT
    {
        for (int i = 0; i < NN_ARRAY_SIZE(users); ++i)
        {
            Close(i);
        }
    }

    User    users[account::UserCountMax];
};

//------------------------------------------------------------------------------------------------------
class EventNotifier
{
public:
    explicit EventNotifier(account::Uid* pUsers, EventNotifierReport* pReport) NN_NOEXCEPT
        : m_pUsers(pUsers), m_pReport(pReport) {}

    ~EventNotifier() NN_NOEXCEPT
    {
        for (int i = 0; i < account::UserCountMax; ++i)
        {
            m_UserHandle[i].Close();
        }
    }

    void Notify(const EventContext& context) NN_NOEXCEPT
    {
        switch (context.type)
        {
        case EventContext::Type::Applet:
            {
                const auto& applet = context.applet;
                pdm::NotifyAppletEvent(applet.appletEvent, applet.programId, 0, applet.appletId, applet.storageId, ns::PlayLogPolicy::All);
                if (pdm::AppletEventType::Launch == applet.appletEvent)
                {
                    // Launch は無視。( テスト用の Report::Open 記録には applet 情報を記録しないので無視で良い。)
                }
                else if (pdm::AppletEventType::InFocus == applet.appletEvent)
                {
                    m_pReport->FocusIn(applet.programId);
                }
                else
                {
                    m_pReport->FocusOut(applet.programId);
                }
            }
            break;
        case EventContext::Type::UserOpen:
            {
                const auto index = context.user.userIndex;
                const auto& user = m_pUsers[index];
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_UserHandle[index].Open(user));

#if defined( NN_BUILD_CONFIG_OS_WIN )
                pdm::detail::PlayEventBuffer::GetInstance().Add(pdm::detail::MakeUserAccountEvent(pdm::UserAccountEventType::Open, user));
#endif

                m_pReport->Open(index, m_pUsers[index]);
            }
            break;
        case EventContext::Type::UserClose:
            {
                const auto index = context.user.userIndex;
                m_UserHandle[index].Close();

#if defined( NN_BUILD_CONFIG_OS_WIN )
                pdm::detail::PlayEventBuffer::GetInstance().Add(pdm::detail::MakeUserAccountEvent(pdm::UserAccountEventType::Close, m_pUsers[index]));
#endif

                m_pReport->Close(index);
            }
            break;
        case EventContext::Type::PowerChange:
            pdm::NotifyPowerStateChangeEvent(context.power.powerType);
            m_pReport->PowerOff();
            break;
        default:
            break;
        }
    }

private:
    account::Uid*           m_pUsers;
    EventNotifierReport*    m_pReport;
    OpenHandle              m_UserHandle[account::UserCountMax];
};

//------------------------------------------------------------------------------------------------------
// プリセットイベント
EventContext g_PresetEventContext[] =
{
    // boot in today.
    EventContext::MakePowerChange(pdm::PowerStateChangeEventType::On),

    // 1st play in today.
    EventContext::MakeSelfApplicationEvent(pdm::AppletEventType::Launch),
    EventContext::MakeSelfApplicationEvent(pdm::AppletEventType::InFocus),
    EventContext::MakeUserOpen(0),
    EventContext::MakeUserClose(0),
    EventContext::MakeSelfApplicationEvent(pdm::AppletEventType::Background),
    EventContext::MakeSelfApplicationEvent(pdm::AppletEventType::Exit),

    // 2nd play in today.
    EventContext::MakeSelfApplicationEvent(pdm::AppletEventType::Launch),
    EventContext::MakeSelfApplicationEvent(pdm::AppletEventType::InFocus),
    EventContext::MakeUserOpen(0),
    EventContext::MakeSelfApplicationEvent(pdm::AppletEventType::Background),

    EventContext::MakeHomeAppletEvent(pdm::AppletEventType::InFocus),
    EventContext::MakeHomeAppletEvent(pdm::AppletEventType::Background),

    EventContext::MakeSelfApplicationEvent(pdm::AppletEventType::InFocus),
    EventContext::MakeUserClose(0),
    EventContext::MakeSelfApplicationEvent(pdm::AppletEventType::Background),
    EventContext::MakeSelfApplicationEvent(pdm::AppletEventType::Exit),

    // 3rd play in today.
    EventContext::MakeSelfApplicationEvent(pdm::AppletEventType::Launch),
    EventContext::MakeSelfApplicationEvent(pdm::AppletEventType::InFocus),
    EventContext::MakeUserOpen(0),
    EventContext::MakeSelfApplicationEvent(pdm::AppletEventType::Background),

    EventContext::MakeHomeAppletEvent(pdm::AppletEventType::InFocus),
    EventContext::MakeHomeAppletEvent(pdm::AppletEventType::Background),

    EventContext::MakeSelfApplicationEvent(pdm::AppletEventType::InFocus),
    EventContext::MakeUserClose(0),
    EventContext::MakeSelfApplicationEvent(pdm::AppletEventType::Background),
    EventContext::MakeSelfApplicationEvent(pdm::AppletEventType::Exit),

    // finish in today.
    EventContext::MakePowerChange(pdm::PowerStateChangeEventType::Off),
};

//------------------------------------------------------------------------------------------------------
EventNotifierReport g_Report;

//------------------------------------------------------------------------------------------------------
// プリセット PlayEvent の書き込み件数設定定数です。
// デバイスDB想定最大件数(16MiB, 約30万件)を書き込む場合は pdm::detail::PlayEventCountMax を指定します。
//const uint32_t TotalNotifyCount = pdm::detail::PlayEventCountMax;
const uint32_t TotalNotifyCount = 30000u;

}

//!------------------------------------------------------------------------------------------------------
class ConvertTest : public testing::Test
{
protected:
    virtual void SetUp() {}
    virtual void TearDown() {}

    static void SetUpTestCase()
    {

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

        oe::Initialize();

#endif

        NN_ABORT_UNLESS_RESULT_SUCCESS(time::Initialize());
        TestContext::GetInstance().Initialize();
    }

    static void TearDownTestCase()
    {
        TestContext::GetInstance().Finalize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(time::Finalize());
    }
};

//!------------------------------------------------------------------------------------------------------
//! @brief      TotalNotifyCount で宣言された件数の プリセット PlayEvent を書き込みます。
//!             デバイスDB想定最大件数( 16MiB, 約30万件 )を書き込む場合は、TotalNotifyCount に pdm::detail::PlayEventCountMax を指定します。
//!
//! @details    実機での変換処理の確認手法について。
//!             現時点では再起動しただけでは変換は発生しません。
//!             起動時マイグレーションが実施できるように、意図的に pdm ライブラリソースを変更する必要があります。
//!
//!             このテストフィクスチャ(TEST_F)が初回実施である場合、Launch → InFocus までのイベントは通知されていますが、追加送信します。
//!------------------------------------------------------------------------------------------------------
TEST_F(ConvertTest, PrepareFullEvent)
{
    int nUsers;
    account::Uid users[account::UserCountMax];
    NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&nUsers, users, NN_ARRAY_SIZE(users)));
    if(0 >= nUsers)
    {
        NN_LOG("No account is registered. Skip this test.\n");
        return;
    }

    // アカウントDBへのインポートを拒否するためにサスペンドします。
    for (int i = 0; i < nUsers; ++i)
    {
        pdm::SuspendUserAccountEventService(users[i]);
    }
    pdm::NotifyClearAllEvent();

    EventNotifier eventNotifier(users, &g_Report);
    int changeUserIndex = 0;

    // 既存イベント数の取得( デバッグ用API )
    int startIndex;
    int latestIndex;
    const auto presetCount = NN_ARRAY_SIZE(g_PresetEventContext);
    const auto existEventCount = pdm::GetAvailablePlayEventRange(&startIndex, &latestIndex);
    const uint32_t AppendableCount = static_cast<uint32_t>(TotalNotifyCount - existEventCount);
    for (uint32_t i = 0; i < AppendableCount; ++i)
    {
        const auto presetIndex = i % presetCount;
        if (0 == presetIndex)
        {
            NN_LOG("Change user index: %d, in event index: %lu\n", changeUserIndex, i);
            EventContext::ChangeUserIndex(g_PresetEventContext, presetCount, changeUserIndex);
            changeUserIndex = (changeUserIndex + 1) % nUsers;
        }
        eventNotifier.Notify(g_PresetEventContext[presetIndex]);
    }

    // 終了内容のフラッシュ及びConditionリセットのために PowerOff を送ります。
    pdm::NotifyPowerStateChangeEvent(pdm::PowerStateChangeEventType::Off);
    g_Report.PowerOff();  // Open継続のまま、直前 PowerOff に対しての Close ケア。

    // アカウントDBへのインポートを許可するためレジュームします。
    for (int i = 0; i < nUsers; ++i)
    {
        pdm::ResumeUserAccountEventService(users[i]);
    }

    NN_LOG("Finish the ConvertTest.PrepareFullEvent, Send event at[ %lu ] ( within last PowerOff ).\n", AppendableCount + 1);
}

#if defined( NN_BUILD_CONFIG_OS_WIN )

namespace nn { namespace pdm { namespace detail {

    uint32_t QueryInFocusCount(AccountPlayEventBuffer* pBuffer) NN_NOEXCEPT
    {
        Bit8 pTemporary[64 * 1024];
        AccountPlayEventBuffer::ScopedReadLock lock;
        NN_ABORT_UNLESS_RESULT_SUCCESS(pBuffer->AcquireRead(&lock));
        const uint32_t readableCount = NN_ARRAY_SIZE(pTemporary) / sizeof(AccountPlayEvent);
        const uint32_t startIndex = pBuffer->GetStartIndex();
        auto pEventBuffer = reinterpret_cast<AccountPlayEvent*>(pTemporary);
        uint32_t totalReadCount = 0;
        uint32_t resultCount = 0;

        while (NN_STATIC_CONDITION(true))
        {
            const auto readCount = pBuffer->Read(pEventBuffer, startIndex + totalReadCount, readableCount);
            if (0 == readCount)
            {
                // 最後まで読み込み、これ以上読み込むものがなくなった。
                break;
            }
            for (uint32_t i = 0; i < readCount; ++i)
            {
                const auto& nowEvent = pEventBuffer[i];
                switch (nowEvent.category)
                {
                case AccountPlayEvent::Category::InFocus:
                    {
                        ++resultCount;
                    }
                    break;
                case AccountPlayEvent::Category::InOpen:
                    {
                    }
                    break;
                default:
                    break;
                }
            }
            totalReadCount += readCount;
        }
        return resultCount;
    }

    void MigrateAccountPlayEventDatabase(AccountPlayEventProvider* pProvider, LockableMemoryBuffer* const pLockableBuffer) NN_NOEXCEPT;
}}}

//------------------------------------------------------------------------------------------------------
// PrepareFullEvent で追加したイベントの変換処理、及び変換後の整合性テストです。
// Generic 環境のみでの実施です。
TEST_F(ConvertTest, VerifyMigrationConversion)
{
    int nUsers;
    account::Uid users[account::UserCountMax];
    NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&nUsers, users, NN_ARRAY_SIZE(users)));
    if (0 >= nUsers)
    {
        NN_LOG("No account is registered. Skip this test.\n");
        return;
    }

    // アカウントDBマイグレーション変換処理部
    const uint32_t MigrationTemporaryByteCapacity = 64 * 1024;
    pdm::detail::LockableMemoryBuiltIn<MigrationTemporaryByteCapacity> migrationTemporary;
    MigrateAccountPlayEventDatabase(&pdm::detail::AccountPlayEventProvider::GetInstance(), &migrationTemporary);

    pdm::PlayStatistics pPlayStatistics[5];
    for (int userIndex = 0; userIndex < nUsers; ++userIndex)
    {
        const account::Uid& uid = users[userIndex];
        const auto& report = g_Report.users[userIndex];

        // 照合 ( InOpen 記録 )
        const auto& inOpen = report.inOpen;
        NN_LOG("[testPdm_Convert] UserAccount[%ld]{0x%llx, 0x%llx} == Expect{0x%llx, 0x%llx}(inOpen: %lu)\n",
            userIndex, uid._data[0], uid._data[1],
            inOpen.uid._data[0],
            inOpen.uid._data[1],
            inOpen.count
        );

        std::memset(pPlayStatistics, 0, sizeof(pPlayStatistics));
        auto count = pdm::QueryPlayStatistics(pPlayStatistics, NN_ARRAY_SIZE(pPlayStatistics), uid);
        const auto& st = pPlayStatistics[0];

        NN_LOG("[testPdm_Convert] TotalPlayCount( %lu ), TotalPlayTime( %lu minutes ), first[ index=%lu, userClock=%lu ], latest[ index=%lu, userClock=%lu ].\n",
            st.totalPlayCount,
            st.totalPlayTime,
            st.firstEventTime.eventIndex,
            st.firstEventTime.userClockTime,
            st.latestEventTime.eventIndex,
            st.latestEventTime.userClockTime
        );

        // 期待するアプリケーションは Self 1個のはず。( Home は applet::AppletId_SystemAppletMenu なので )
        // 送信したイベント数とユーザー数に応じて、特定ユーザーは記録がない可能性があるので、`期待 inOpen 記録数` で期待アプリ数を決定。
        const int expectCount = (inOpen.count > 0) ? 1 : 0;
        EXPECT_EQ(count, expectCount);
        if (expectCount > 0)
        {
            // プレイ時間は、イベント送信が分( minutes )以上かからないケースがあるので、この評価では対象外。
            EXPECT_EQ(SelfApplicationId.value, st.applicationId.value);
            EXPECT_EQ(inOpen.uid._data[0], uid._data[0]);
            EXPECT_EQ(inOpen.uid._data[1], uid._data[1]);
            EXPECT_EQ(inOpen.count, st.totalPlayCount);
            EXPECT_LE(0u, st.firstEventTime.eventIndex);
            EXPECT_LE(0u, st.latestEventTime.eventIndex);
            EXPECT_LT(0u, st.firstEventTime.userClockTime);
            EXPECT_LT(0u, st.latestEventTime.userClockTime);
        }

        // 照合( InFocus 記録 )
        const auto& inFocus = report.inFocus;
        auto& dbAccount = pdm::detail::AccountPlayEventProvider::GetInstance();
        auto capacity = pdm::detail::AccountPlayEventProvider::BufferIndexCountMax;
        for (decltype(capacity) bufferIndex = 0; bufferIndex < capacity; ++bufferIndex)
        {
            auto& buffer = dbAccount.GetBuffer(bufferIndex);
            if (0 == buffer.Find(&uid, 1))
            {
                const auto inFocusCount = QueryInFocusCount(&const_cast<pdm::detail::AccountPlayEventBuffer&>(buffer));
                EXPECT_EQ(inFocusCount, inFocus.count);
            }
        }
    }
}

#endif
