﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/pcv/pcv.h>
#include <nn/spl/spl_Api.h>
#include <nn/spl/spl_Types.h>
#include <nn/nn_Common.h>
#include <nnt/nntest.h>

// 開発者に提供する使用可能なクロックレートの具体的な値を確認するテスト

namespace nnt { namespace pcv {
    struct ClockRatesInfo
    {
        nn::pcv::Module name;
        int numRates;
        int rates[nn::pcv::MaxNumClockRates];
    };
}}

class GetPossibleClockRatesTest : public ::testing::TestWithParam<nnt::pcv::ClockRatesInfo>{};

const nnt::pcv::ClockRatesInfo clockRatesInfo[] =
    {
        {
            nn::pcv::Module_Cpu, 16,
            {
                 204 * 1000 * 1000,
                 306 * 1000 * 1000,
                 408 * 1000 * 1000,
                 510 * 1000 * 1000,
                 612 * 1000 * 1000,
                 714 * 1000 * 1000,
                 816 * 1000 * 1000,
                 918 * 1000 * 1000,
                1020 * 1000 * 1000,
                1122 * 1000 * 1000,
                1224 * 1000 * 1000,
                1326 * 1000 * 1000,
                1428 * 1000 * 1000,
                1581 * 1000 * 1000,
                1683 * 1000 * 1000,
                1785 * 1000 * 1000,
            },
        },
        {
            nn::pcv::Module_Gpu, 16,
            {
                 76800 * 1000,
                153600 * 1000,
                230400 * 1000,
                307200 * 1000,
                384000 * 1000,
                460800 * 1000,
                537600 * 1000,
                614400 * 1000,
                691200 * 1000,
                768000 * 1000,
                844800 * 1000,
                921600 * 1000,
                998400 * 1000,
                1075200 * 1000,
                1152000 * 1000,
                1228800 * 1000,
            },
        },
    };

const int GpuNormalNumRates = 12;

INSTANTIATE_TEST_CASE_P(GetPossibleClockRates, GetPossibleClockRatesTest, testing::ValuesIn(clockRatesInfo));

TEST_P(GetPossibleClockRatesTest, ClockRatesInfo)
{
    // Pcv library should already be initialized
    ASSERT_TRUE(nn::pcv::IsInitialized());
    nn::pcv::Module module = GetParam().name;

    nn::pcv::ClockRatesListType listType = nn::pcv::ClockRatesListType_Invalid;
    nn::pcv::ClockHz rates[nn::pcv::MaxNumClockRates];
    memset(rates, 0, sizeof(rates) / sizeof(rates[0]));

    int numRates = 0;

    nn::Result queryResult = nn::pcv::GetPossibleClockRates(
        &listType,
        rates,
        &numRates,
        module,
        nn::pcv::MaxNumClockRates);

    EXPECT_TRUE(queryResult.IsSuccess());
    EXPECT_EQ(nn::pcv::ClockRatesListType_Discrete, listType);

    int expectedNumRates = GetParam().numRates;

    if ( module == nn::pcv::Module_Gpu )
    {
        nn::spl::Initialize();
        auto higherRatesPossible = (nn::spl::GetHardwareType() == nn::spl::HardwareType_Iowa)
            && (nn::spl::GetRegulator() == nn::spl::Regulator_Max77812PhaseConfiguration31);
        nn::spl::Finalize();

        if ( !higherRatesPossible )
        {
            expectedNumRates = GpuNormalNumRates;
        }
    }

    EXPECT_EQ(expectedNumRates, numRates);

    for (int i = 0; i < expectedNumRates; i++)
    {
        EXPECT_EQ(GetParam().rates[i], rates[i]);
    }
}
