﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/pctl/pctl_Api.h>
#include <nn/pctl/pctl_ApiSystem.h>
#include <nn/pctl/pctl_ApiForAuthentication.h>
#include <nn/pctl/pctl_ApiStereoVision.h>

class SettingsTest : public ::testing::Test
{
protected:
    virtual void SetUp() NN_OVERRIDE
    {
        // テストケース開始時に毎回初期化を行う
        nn::pctl::DeleteSettings();
    }

    virtual void TearDown() NN_OVERRIDE
    {
        // テストケース終了時にも(ごみが残らないように)毎回初期化を行う
        nn::pctl::DeleteSettings();
    }
};

// SafetyLevelの設定テスト
TEST_F(SettingsTest, TestSetStereoVisionRestriction)
{
    nn::pctl::SetPinCode("0123");
    EXPECT_TRUE(nn::pctl::IsRestrictionEnabled());

    // 初期状態から nn::pctl::IsRestrictionEnabled() == true になっている場合は可能なはず
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmStereoVisionRestrictionConfigurable());

    nn::pctl::SetStereoVisionRestriction(true);
    EXPECT_TRUE(nn::pctl::GetStereoVisionRestriction());
    nn::pctl::SetStereoVisionRestriction(false);
    EXPECT_FALSE(nn::pctl::GetStereoVisionRestriction());
}

// SafetyLevelの設定テスト
TEST_F(SettingsTest, TestConfirmStereoVisionPermission)
{
    nn::pctl::SetPinCode("0123");
    EXPECT_TRUE(nn::pctl::IsRestrictionEnabled());

    // 初期状態から nn::pctl::IsRestrictionEnabled() == true になっている場合は可能なはず
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmStereoVisionRestrictionConfigurable());

    // 1. 設定値が false の場合
    nn::pctl::SetStereoVisionRestriction(false);
    EXPECT_FALSE(nn::pctl::GetStereoVisionRestriction());
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmStereoVisionPermission());
    nn::pctl::ResetConfirmedStereoVisionPermission();
    EXPECT_FALSE(nn::pctl::GetStereoVisionRestriction());
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmStereoVisionPermission());

    // 2. 設定値が true の場合
    nn::pctl::SetStereoVisionRestriction(true);
    EXPECT_TRUE(nn::pctl::GetStereoVisionRestriction());
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultStereoVisionRestricted, nn::pctl::ConfirmStereoVisionPermission());
    nn::pctl::ResetConfirmedStereoVisionPermission();
    EXPECT_TRUE(nn::pctl::GetStereoVisionRestriction());
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultStereoVisionRestricted, nn::pctl::ConfirmStereoVisionPermission());

    // 3. 設定値が true、かつ一時解除されている場合
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::UnlockRestrictionTemporarily("0123"));
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmStereoVisionPermission());
    nn::pctl::RevertRestrictionTemporaryUnlocked();

    // 4. そもそも制限が無効の場合
    nn::pctl::SetPinCode(nullptr);
    EXPECT_FALSE(nn::pctl::IsRestrictionEnabled());
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmStereoVisionPermission());
}

TEST_F(SettingsTest, TestResetConfirmedStereoVisionPermission)
{
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    nn::pctl::SetPinCode("0123");
    EXPECT_TRUE(nn::pctl::IsRestrictionEnabled());

    // 立体視機能が制限
    nn::pctl::SetStereoVisionRestriction(true);
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultStereoVisionRestricted, nn::pctl::ConfirmStereoVisionPermission());
    // 立体視機能が未使用なので，スリープ復帰後のチェックは成功
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));
    // 一時解除で制限チェックパス
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::UnlockRestrictionTemporarily("0123"));
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmStereoVisionPermission());
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));
    // 一時解除の無効化により，スリープ復帰後のチェックに失敗
    nn::pctl::RevertRestrictionTemporaryUnlocked();
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultStereoVisionRestricted, nn::pctl::ConfirmResumeApplicationPermission(appId, prop));
    // 立体視機能許可状態をリセット
    nn::pctl::ResetConfirmedStereoVisionPermission();
    // スリープ復帰後のチェックに成功
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));
    // 実際に立体視機能を使用すると失敗
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultStereoVisionRestricted, nn::pctl::ConfirmStereoVisionPermission());
}

TEST_F(SettingsTest, TestIsStereoVisionPermitted)
{
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    nn::pctl::SetPinCode("0123");
    EXPECT_TRUE(nn::pctl::IsRestrictionEnabled());

    // 立体視機能が制限
    nn::pctl::SetStereoVisionRestriction(true);
    ASSERT_FALSE(nn::pctl::IsStereoVisionPermitted());
    // 立体視機能が未使用なので，スリープ復帰後のチェックは成功
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));
    // 一時解除で制限チェックパス
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::UnlockRestrictionTemporarily("0123"));
    ASSERT_TRUE(nn::pctl::IsStereoVisionPermitted());
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));
    // 一時解除の無効化後でも，立体視機能使用許可状態に遷移していないので、スリープ復帰後のチェックは成功
    nn::pctl::RevertRestrictionTemporaryUnlocked();
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));
}
