﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/pctl/pctl_Api.h>
#include <nn/pctl/pctl_ApiSystem.h>
#include <nn/pctl/pctl_ApiForAuthentication.h>
#include <nn/pctl/pctl_ApiForDebug.h>

TEST(ExemptApplicationList, Test_Empty)
{
    nn::pctl::ClearExemptApplicationListForDebug();

    nn::pctl::ExemptApplicationInfo appInfo;
    EXPECT_EQ(0, nn::pctl::GetExemptApplicationListCountForDebug());
    EXPECT_EQ(0, nn::pctl::GetExemptApplicationListForDebug(&appInfo, 0, 1));
}

// 制限対象外リストへの追加
TEST(ExemptApplicationList, Test_AddOne_Exempted)
{
    nn::pctl::ClearExemptApplicationListForDebug();

    const nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId
    nn::pctl::SafetyLevelSettings customSettings;

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = false;
    customSettings.freeCommunicationRestriction = true; // 既定値

    nn::pctl::ExemptApplicationInfo appInfo;
    nn::pctl::SetPinCode("0123");
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToExemptApplicationListForDebug(appId);

    EXPECT_EQ(1, nn::pctl::GetExemptApplicationListCountForDebug());
    EXPECT_EQ(1, nn::pctl::GetExemptApplicationListForDebug(&appInfo, 0, 1));
    EXPECT_EQ(appId, appInfo.applicationId);
    EXPECT_EQ(false, appInfo.isExempted);
}

// 制限対象外=true へ更新
TEST(ExemptApplicationList, Test_UpdateOne_Exempted)
{
    nn::pctl::ClearExemptApplicationListForDebug();

    const nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId
    nn::pctl::SafetyLevelSettings customSettings;

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = false;
    customSettings.freeCommunicationRestriction = false; // 既定値

    nn::pctl::ExemptApplicationInfo appInfo;
    nn::pctl::SetPinCode("0123");
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToExemptApplicationListForDebug(appId);

    appInfo.applicationId = appId;
    appInfo.isExempted = true;
    nn::pctl::UpdateExemptApplicationListForDebug(&appInfo, 1);

    std::memset(&appInfo, 0, sizeof(appInfo));
    EXPECT_EQ(1, nn::pctl::GetExemptApplicationListCountForDebug());
    EXPECT_EQ(1, nn::pctl::GetExemptApplicationListForDebug(&appInfo, 0, 1));
    EXPECT_EQ(appId, appInfo.applicationId);
    EXPECT_EQ(true, appInfo.isExempted);
}

// 制限対象外=false へ更新
TEST(ExemptApplicationList, Test_UpdateOne_NotExempted)
{
    nn::pctl::ClearExemptApplicationListForDebug();

    const nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId
    nn::pctl::SafetyLevelSettings customSettings;

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = false;
    customSettings.freeCommunicationRestriction = true; // 既定値

    nn::pctl::ExemptApplicationInfo appInfo;
    nn::pctl::SetPinCode("0123");
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToExemptApplicationListForDebug(appId);

    appInfo.applicationId = appId;
    appInfo.isExempted = false;
    nn::pctl::UpdateExemptApplicationListForDebug(&appInfo, 1);

    std::memset(&appInfo, 0, sizeof(appInfo));
    EXPECT_EQ(1, nn::pctl::GetExemptApplicationListCountForDebug());
    EXPECT_EQ(1, nn::pctl::GetExemptApplicationListForDebug(&appInfo, 0, 1));
    EXPECT_EQ(appId, appInfo.applicationId);
    EXPECT_EQ(false, appInfo.isExempted);
}

// ペアコン設定削除によって規定値（=制限対象外ではない）に戻る
TEST(ExemptApplicationList, Test_DeleteSettings_ResetToNotExempted)
{
    nn::pctl::ClearExemptApplicationListForDebug();

    const nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId
    nn::pctl::SafetyLevelSettings customSettings;

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = false;
    customSettings.freeCommunicationRestriction = true; // 既定値

    nn::pctl::ExemptApplicationInfo appInfo;
    nn::pctl::SetPinCode("0123");
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToExemptApplicationListForDebug(appId);

    appInfo.applicationId = appId;
    appInfo.isExempted = true;
    nn::pctl::UpdateExemptApplicationListForDebug(&appInfo, 1);

    nn::pctl::DeleteSettings();

    std::memset(&appInfo, 0, sizeof(appInfo));
    EXPECT_EQ(1, nn::pctl::GetExemptApplicationListCountForDebug());
    EXPECT_EQ(1, nn::pctl::GetExemptApplicationListForDebug(&appInfo, 0, 1));
    EXPECT_EQ(appId, appInfo.applicationId);
    EXPECT_EQ(false, appInfo.isExempted);
}
