﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/http.h>
#include <nn/http/http_JsonResponse.h>
#include <nn/socket.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_Base64.h>

#include "testPctl_TestUtil.h"


namespace
{
    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

    inline bool IsReservedCharacter(char c) NN_NOEXCEPT
    {
        // https://tools.ietf.org/html/rfc3986#section-2.2
        static const char Reserved[] = {
        0x20, // スペース : 未定義
        0x21, // !
        0x22, // " : 未定義
        0x23, // #
        0x24, // $
        0x25, // % : 未定義
        0x26, // &
        0x27, // '
        0x28, // (
        0x29, // )
        0x2A, // *
        0x2B, // +
        0x2C, // ,
        // 0x2D, // -
        // 0x2E, // .
        0x2F, // /
        0x3A, // :
        0x3B, // ;
        0x3C, // < : 未定義
        0x3D, // =
        0x3E, // > : 未定義
        0x3F, // ?
        0x40, // @
        0x5B, // [
        0x5C, // \ : 未定義
        0x5D, // ]
        0x5E, // ^ : 未定義
        // 0x5F, // _
        0x60, // ` : 未定義
        0x7B, // { : 未定義
        0x7C, // | : 未定義
        0x7D, // } : 未定義
        // 0x7E, // ~
        };
        size_t i = 0u;
        while (i < sizeof(Reserved) && Reserved[i] != c)
        {
            ++ i;
        }
        return i != sizeof(Reserved);
    }

    inline bool IsUnreservedCharacter(char c) NN_NOEXCEPT
    {
        // https://tools.ietf.org/html/rfc3986#section-2.2
        return false
            || ('A' <= c && c <= 'Z')
            || ('a' <= c && c <= 'z')
            || ('0' <= c && c <= '9')
            || c == '-'
            || c == '_'
            || c == '.'
            || c == '~';
    }

    inline size_t PutCharacterWithUrlEncoding(char* str, size_t strSize, char c) NN_NOEXCEPT
    {
        // https://tools.ietf.org/html/rfc3986#section-2.2
        NN_SDK_ASSERT(strSize >= 3);
        NN_UNUSED(strSize);
        auto p = str;
        if (c == ' ')
        {
            // +
            *(p++) = '+';
        }
        else if (IsUnreservedCharacter(c))
        {
            // c
            *(p++) = c;
        }
        else
        {
            NN_ABORT_UNLESS(IsReservedCharacter(c));
            auto u = (c >> 4);
            NN_ABORT_UNLESS(u <= 7);
            auto l = (c & 0x0F);
            NN_SDK_ASSERT(l <= 15);

            // %xx
            *(p++) = '%';
            *(p++) = static_cast<char>('0' + u);
            *(p++) = static_cast<char>((l >= 10 ? 'A' + l - 10 : '0' + l));
        }
        return static_cast<int>(p - str);
    }
}

namespace nn
{
namespace pctl
{
namespace test
{

int UrlEncode(char *dst, size_t dstSize, const char* src, size_t srcSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(strnlen(src, srcSize) < srcSize); // 終端
    NN_ABORT_UNLESS(dstSize >= 3 * (srcSize - 1) + 1); // 上限
    size_t offset = 0;
    for (size_t i = 0u; i < srcSize; ++ i)
    {
        auto c = src[i];
        if (c == '\0')
        {
            *(dst + offset) = '\0';
            return static_cast<int>(offset);
        }

        auto l = PutCharacterWithUrlEncoding(dst + offset, dstSize - offset, c);
        NN_SDK_ASSERT(l < dstSize - offset);
        offset += l;
    }
    NN_ABORT("ABORT: Unreachable\n");
}

nn::Result GetPairingInfo(PairingInfo* pOutPairingInfo, const char* pEmail, const char* pPassword) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutPairingInfo);
    NN_SDK_ASSERT_NOT_NULL(pEmail);
    NN_SDK_ASSERT_NOT_NULL(pPassword);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfigWithMemory));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ssl::Initialize());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::http::Initialize());

    NN_UTIL_SCOPE_EXIT
    {
        nn::http::Finalize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ssl::Finalize());
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Finalize());
    };

    {
        nn::http::ConnectionBroker connectionBroker;
        NN_RESULT_DO(connectionBroker.Initialize());
        connectionBroker.SetEnableClientCert(true);
        connectionBroker.SetVerbose(true);

        nn::http::Request request(connectionBroker);
        request.SetMethodType(nn::http::MethodType_Post);
        request.SetVerbose(true);

        char JsonBuffer[2048];

        {
            nn::http::JsonResponse jsonResponse(JsonBuffer, sizeof(JsonBuffer));
            const char url[] = "https://hikarie.nntt.mng.nintendo.net/api/v1/nas/authorize";
            request.SetUrlPointer(url);

            char encodedEmail[256];
            char encodedPassword[256];
            UrlEncode(encodedEmail, sizeof(encodedEmail), pEmail, nn::util::Strnlen(pEmail, sizeof(encodedEmail)) + 1);
            UrlEncode(encodedPassword, sizeof(encodedPassword), pPassword, nn::util::Strnlen(pPassword, sizeof(encodedPassword)) + 1);

            char postData[1024];
            auto ret = nn::util::SNPrintf(postData, sizeof(postData),
                "client_id=f99340b4766de193&"
                "redirect_uri=http://localhost/callback&"
                "response_type=token&"
                "scope=user user.mii moonUser:administration moonDevice:create moonParentalControlSetting moonParentalControlSetting:update moonParentalControlSettingState moonOwnedDevice:administration moonPairingState moonSmartDevice:administration moonDailySummary moonMonthlySummary&"
                "state=aaaaa&"
                "email=%s&"
                "password=%s",
                encodedEmail, encodedPassword);
            NN_ABORT_UNLESS(ret < sizeof(postData));
            request.SetPostDataPointer(postData, nn::util::Strnlen(postData, sizeof(postData)));

            NN_RESULT_DO(request.GetResponse(&jsonResponse));
            auto statusCode = jsonResponse.GetStatusCode();
            NN_LOG("statusCode: [%d]\n", static_cast<int>(statusCode));
            NN_RESULT_THROW_UNLESS(200 <= statusCode && statusCode < 300, jsonResponse.GetStatusCodeAsResult());

            const nn::http::JsonResponse::ParseRule parseRules[] = {
                { "access_token", nullptr, nn::http::JsonResponse::ValueType_String, pOutPairingInfo->accessToken, sizeof(pOutPairingInfo->accessToken) },
            {}
            };

            NN_RESULT_DO(jsonResponse.ReadAndParseWithRule(parseRules));
            NN_LOG("access_token: [%s]\n", pOutPairingInfo->accessToken);

            NN_ABORT_UNLESS(0 < nn::util::Strnlen(pOutPairingInfo->accessToken, NN_ARRAY_SIZE(pOutPairingInfo->accessToken)));
            jsonResponse.Dispose();
        }

        {
            nn::http::JsonResponse jsonResponse(JsonBuffer, sizeof(JsonBuffer));
            const char url[] = "https://hikarie.nntt.mng.nintendo.net/api/v1/jwt/decode";
            request.SetUrlPointer(url);

            char postData[1024];
            auto ret = nn::util::SNPrintf(postData, sizeof(postData),
                "token=%s",
                pOutPairingInfo->accessToken);
            NN_ABORT_UNLESS(ret < sizeof(postData));
            request.SetPostDataPointer(postData, nn::util::Strnlen(postData, sizeof(postData)));

            NN_RESULT_DO(request.GetResponse(&jsonResponse));
            auto statusCode = jsonResponse.GetStatusCode();
            NN_LOG("statusCode: [%d]\n", static_cast<int>(statusCode));
            NN_RESULT_THROW_UNLESS(200 <= statusCode && statusCode < 300, jsonResponse.GetStatusCodeAsResult());

            const nn::http::JsonResponse::ParseRule parseRules[] = {
                { "sub", nullptr, nn::http::JsonResponse::ValueType_String, pOutPairingInfo->nintendoAccountId, sizeof(pOutPairingInfo->nintendoAccountId) },
            {}
            };

            NN_RESULT_DO(jsonResponse.ReadAndParseWithRule(parseRules));
            NN_LOG("NintendoAccountId: [%s]\n", pOutPairingInfo->nintendoAccountId);

            NN_ABORT_UNLESS(0 < nn::util::Strnlen(pOutPairingInfo->accessToken, NN_ARRAY_SIZE(pOutPairingInfo->accessToken)));
            jsonResponse.Dispose();
        }

        {
            nn::http::JsonResponse jsonResponse(JsonBuffer, sizeof(JsonBuffer));
            const char url[] = "https://api-%.pctl.srv.nintendo.net/moon/v1/devices";
            request.SetUrlPointer(url);

            request.AddHeaderFormat("X-Moon-App-Internal-Version: 99999");
            request.AddHeaderFormat("X-Moon-Os: IOS");
            request.AddHeaderFormat("Content-Type: application/json");
            request.AddHeaderFormat("Authorization:Bearer %s", pOutPairingInfo->accessToken);

            NN_RESULT_DO(request.GetResponse(&jsonResponse));
            auto statusCode = jsonResponse.GetStatusCode();
            NN_RESULT_THROW_UNLESS(200 <= statusCode && statusCode < 300, jsonResponse.GetStatusCodeAsResult());

            const nn::http::JsonResponse::ParseRule parseRules[] = {
                { "code", nullptr, nn::http::JsonResponse::ValueType_String, pOutPairingInfo->pairingCode, sizeof(pOutPairingInfo->pairingCode) },
                { "id",   nullptr, nn::http::JsonResponse::ValueType_String, pOutPairingInfo->deviceId, sizeof(pOutPairingInfo->deviceId) },
            {}
            };

            NN_RESULT_DO(jsonResponse.ReadAndParseWithRule(parseRules));
            NN_LOG("deviceId: %s\n", pOutPairingInfo->deviceId);
            NN_LOG("pairingCode: %s\n", pOutPairingInfo->pairingCode);
            NN_ABORT_UNLESS(0 < nn::util::Strnlen(pOutPairingInfo->deviceId, sizeof(pOutPairingInfo->deviceId)));
            NN_ABORT_UNLESS(0 < nn::util::Strnlen(pOutPairingInfo->pairingCode, sizeof(pOutPairingInfo->pairingCode)));

            jsonResponse.Dispose();
        }
    }
    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

nn::Result ForceUnlinkPairing(const PairingInfo* pairingInfo) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfigWithMemory));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ssl::Initialize());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::http::Initialize());

    NN_UTIL_SCOPE_EXIT
    {
        nn::http::Finalize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ssl::Finalize());
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Finalize());
    };

    NN_SDK_ASSERT_NOT_NULL(pairingInfo);
    {
        nn::http::ConnectionBroker connectionBroker;
        NN_RESULT_DO(connectionBroker.Initialize());
        connectionBroker.SetEnableClientCert(true);
        connectionBroker.SetVerbose(true);

        nn::http::Request request(connectionBroker);
        request.SetMethodType(nn::http::MethodType_Delete);
        request.SetVerbose(true);

        nn::http::Response response;
        {
            char url[1024];
            auto ret = nn::util::SNPrintf(url, sizeof(url),
                "https://api-%%.pctl.srv.nintendo.net/moon/v1/users/%s/devices/%s",
                pairingInfo->nintendoAccountId,
                pairingInfo->deviceId
            );
            NN_ABORT_UNLESS(ret < sizeof(url));
            NN_LOG("url for unlink=%s\n", url);
            request.SetUrlPointer(url);

            request.AddHeaderFormat("Authorization:Bearer %s", pairingInfo->accessToken);
            request.AddHeaderFormat("X-Moon-App-Internal-Version: 99999");
            request.AddHeaderFormat("X-Moon-Os: IOS");

            NN_RESULT_DO(request.GetResponse(&response));
            auto statusCode = response.GetStatusCode();
            NN_LOG("@ForceUnlinkPairing statusCode=%d\n", statusCode);
            NN_RESULT_THROW_UNLESS(200 <= statusCode && statusCode < 300, response.GetStatusCodeAsResult());
        }
    }
    NN_RESULT_SUCCESS;
}

}
}
}
