﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/os.h>
#include <nn/gfx.h>
#include <nn/init/init_Malloc.h>

#include <nnt/gtest/gtest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>

#include <nv/nv_MemoryManagement.h>

extern "C" int NvOsDrvOpen(const char *pathname, int flags);

namespace {
    const size_t GlobalHeapSize = 32 * 1024 * 1024;
    const size_t GlobalMemoryBlockSize = 16 * 1024 * 1024;
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;

    nn::gfx::Device g_Device;

    void InitializeDevice()
    {
        nn::gfx::Device::InfoType info;
        info.SetDefault();
        info.SetApiVersion(nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion);
        g_Device.Initialize(info);
    }

    void FinalizeDevice()
    {
        g_Device.Finalize();
    }

    void* Allocate(size_t size, size_t alignment, void*)
    {
        return std::aligned_alloc(alignment, size);
    }

    void Free(void* addr, void*)
    {
        std::free(addr);
    }

    void* Reallocate(void* addr, size_t newSize, void*)
    {
        return std::realloc(addr, newSize);
    }
}

extern "C" void nninitStartup()
{
    nn::Result result;

    /* set heap size */
    result = nn::os::SetMemoryHeapSize(GlobalHeapSize);
    if (!result.IsSuccess()) {
        NN_LOG("Failed SetMemoryHeapSize\n");
        return;
    }
    NN_LOG("SetMemoryHeapSize 0x%x OK\n", GlobalHeapSize);

    uintptr_t address;
    result = nn::os::AllocateMemoryBlock(&address, GlobalMemoryBlockSize);
    NN_ASSERT(result.IsSuccess());

    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), GlobalMemoryBlockSize);
}

extern "C" void nnMain()
{
    // グラフィックスシステムのためのメモリ周りの初期化
    nv::SetGraphicsAllocator(Allocate, Free, Reallocate, NULL);
    nv::SetGraphicsDevtoolsAllocator(Allocate, Free, Reallocate, NULL);
    nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);

    // GPU の電源を入れることを目的としてグラフィックスシステムを初期化
    nn::gfx::Initialize();
    InitializeDevice();

    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    int result = RUN_ALL_TESTS();
    ::nnt::Exit(result);

    // グラフィックスシステムの終了
    FinalizeDevice();
    nn::gfx::Finalize();
}
