﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdlib>
#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nnt.h>

#include <nn/pcie/pcie.h>

nn::pcie::ClassDriverHandle g_ClassDriverHandle = nn::pcie::InvalidClassDriverHandle;
nn::os::SystemEventType g_RegistrationEvent;
nn::os::SystemEventType g_PmEvent;
nn::os::SystemEventType g_IrqEvent;
nn::pcie::FunctionState* g_pFuncUnderTest = NULL;
nn::pcie::FunctionState g_FuncStates[nn::pcie::MaxEnumeratedDeviceFunctions];
uint8_t NN_ALIGNAS(nn::pcie::MinimumDmaAddressAlignment) g_BigDmaBuffer[0x100000];

void PrintFunctionState(nn::pcie::FunctionState& functionState)
{
    NN_LOG("functionHandle   = 0x%p\n",   reinterpret_cast<void*>(functionState.functionHandle));
    NN_LOG("  busNum           = %d\n",     functionState.busNum);
    NN_LOG("  deviceNum        = %d\n",     functionState.deviceNum);
    NN_LOG("  funcNum          = %d\n",     functionState.funcNum);
    NN_LOG("  vendorId         = 0x%x\n",   functionState.vendorId);
    NN_LOG("  deviceId         = 0x%x\n",   functionState.deviceId);
    NN_LOG("  headerType       = 0x%x\n",   functionState.headerType);
    NN_LOG("  classCode        = 0x%x\n",   functionState.classCode);
    NN_LOG("  revision         = 0x%x\n",   functionState.revision);
    NN_LOG("  speed            = %d\n",     functionState.speed);
    NN_LOG("  maxNumMsiVectors = %d\n",     functionState.maxNumMsiVectors);
    NN_LOG("  pmState          = %d\n",     functionState.pmState);
    NN_LOG("  isAcquired       = %d\n",     functionState.isAcquired);
}

void PrintBars()
{
    if(g_pFuncUnderTest != NULL)
    {
        for(uint8_t bar=0; bar < nn::pcie::StandardConfigMaxBaseAddresses; bar++)
        {
            nn::pcie::BarProfile barProfile;
            if(nn::pcie::GetBarProfile(&barProfile, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle, bar).IsSuccess())
            {
                volatile uint8_t* pBar = NULL;
                NN_LOG("BAR %d:\n", bar);
                NN_LOG("  size   = 0x%x\n", barProfile.size);
                NN_LOG("  flags  = 0x%x\n", barProfile.flags);
                if(nn::pcie::GetMappedBar(&pBar, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle, bar).IsSuccess())
                {
                    NN_LOG("  pBar  = %p\n", pBar);
                }
            }
        }

    }

}

TEST( PcieServer_SleepWake, Setup)
{
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::Initialize());

    // accept any device for this test
    nn::pcie::ClassDriverConfig cfg = {0};
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::RegisterClassDriver(&g_ClassDriverHandle, &g_RegistrationEvent, &cfg,
                                                            nn::os::EventClearMode_ManualClear));

    do
    {
        // Wait for initial availability
        NN_LOG("Waiting for device function availability...");
        nn::os::WaitSystemEvent(&g_RegistrationEvent);
        NN_LOG("AVAILABLE.\n");
        nn::os::ClearSystemEvent(&g_RegistrationEvent);

        // Query to see what is there
        int32_t numReturned = 0;
        NNT_EXPECT_RESULT_SUCCESS(nn::pcie::QueryFunctions(g_FuncStates, &numReturned, g_ClassDriverHandle,
                                                           sizeof(g_FuncStates)));
        for(int32_t i = 0; i < numReturned; i++)
        {
            nn::pcie::FunctionState *pF = g_FuncStates + i;

            // Don't take the host bridge
           // if(pF->busNum != 2) continue;

            // Acquire first available
            if(!pF->isAcquired)
            {
                g_pFuncUnderTest = pF;
                NN_LOG("Trying to acquire functionHandle 0x%llx...",g_pFuncUnderTest->functionHandle);
                NNT_EXPECT_RESULT_SUCCESS(nn::pcie::AcquireFunction(&g_PmEvent, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                                    nn::os::EventClearMode_ManualClear));
                PrintFunctionState(*pF);
                PrintBars();
                NN_LOG("DONE.\n");
                break;
            }
        }
    }while(g_pFuncUnderTest == NULL);
}

// This test case handles continuous transitions, avoiding re-enumeration each time.
TEST( PcieServer_SleepWake, D0_D3Hot_Transitions)
{
    const int Iterations = 3;
    int iterations = 0;
    nn::pcie::FunctionState funcState;
    nn::pcie::BusAddress busAddr = 0;

    // Do some iterations, requires power button pushes to cycle
    do
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::pcie::SetResetUponResumeEnable(g_ClassDriverHandle, g_pFuncUnderTest->functionHandle, false));
        NNT_EXPECT_RESULT_SUCCESS(nn::pcie::MapDma(&busAddr, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                   nn::pcie::DmaDirection_BiDirectional,
                                                   g_BigDmaBuffer, nn::pcie::MinimumDmaAddressAlignment));
        NNT_EXPECT_RESULT_SUCCESS(nn::pcie::AcquireIrq(&g_IrqEvent, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                       nn::os::EventClearMode_ManualClear, nn::pcie::IrqType_Msi));
        // Go to sleep
        do
        {
            NN_LOG("Waiting for PM event - Transition to sleep...\n");
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
        }while(!nn::os::TryWaitSystemEvent(&g_PmEvent));
        nn::os::ClearSystemEvent(&g_PmEvent);

        // Check for PmState_D3Hot
        NNT_EXPECT_RESULT_SUCCESS(nn::pcie::GetFunctionState(&funcState, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle));
        if(funcState.pmState == nn::pcie::PmState_D3Hot)
        {
            NN_LOG("\n\n\n!!!!!!!!!!!!!!!!!!!\nAs expected, PmState_D3Hot.\n!!!!!!!!!!!!!!!!!!!\n\n");
            PrintFunctionState(funcState);
        }
        else
        {
            GTEST_FAIL() << "Unexpected function state.\n";
        }

        // Wakeup
        do
        {
            NN_LOG("Waiting for PM event - Transition to wake...\n");
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
        }while(!nn::os::TryWaitSystemEvent(&g_PmEvent));
        nn::os::ClearSystemEvent(&g_PmEvent);

        // Check for PmState_D0
        NNT_EXPECT_RESULT_SUCCESS(nn::pcie::GetFunctionState(&funcState, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle));
        if(funcState.pmState == nn::pcie::PmState_D0)
        {
            NN_LOG("\n\n\n!!!!!!!!!!!!!!!!!!!\nAs expected, PmState_D0.\n!!!!!!!!!!!!!!!!!!!\n\n");
            PrintFunctionState(funcState);
            PrintBars();
            NNT_EXPECT_RESULT_FAILURE(nn::pcie::ResultAlreadyAcquired,
                                      nn::pcie::AcquireIrq(&g_IrqEvent, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                           nn::os::EventClearMode_ManualClear, nn::pcie::IrqType_Msi));

            // We already did this above during D3Hot
            if(iterations < Iterations)
            {
                NNT_EXPECT_RESULT_SUCCESS(nn::pcie::ReleaseIrq(&g_IrqEvent, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle));
            }

            NNT_EXPECT_RESULT_SUCCESS(nn::pcie::UnmapDma(g_ClassDriverHandle, g_pFuncUnderTest->functionHandle, g_BigDmaBuffer));
        }
        else
        {
            GTEST_FAIL() << "Unexpected function state.\n";
        }
    }while(iterations++ < Iterations);

}


// This test case handles continuous transitions, avoiding re-enumeration each time.
TEST( PcieServer_SleepWake, Back_To_D3Cold)
{
    nn::pcie::FunctionState funcState = {0};


    // Go to sleep
    do
    {
        NN_LOG("Waiting for PM event - Transition to sleep...\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
    }while(!nn::os::TryWaitSystemEvent(&g_PmEvent));
    nn::os::ClearSystemEvent(&g_PmEvent);

    // This function will disappear since SetResetUponResumeEnable() was not called
    do
    {
        NN_LOG("Waiting registration event of newly enumerated function...\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
    }while(!nn::os::TryWaitSystemEvent(&g_RegistrationEvent));
    nn::os::ClearSystemEvent(&g_RegistrationEvent);

    // Should be in PmState_D3Off now
    NNT_EXPECT_RESULT_FAILURE(nn::pcie::ResultInvalidFunctionHandle,
                              nn::pcie::GetFunctionState(&funcState, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle));
}



TEST( PcieServer_SleepWake, Teardown)
{
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::Finalize());
}
