﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

// MemoryResourceObserver
#include <nn/nn_Allocator.h>
#include <limits>
#include <nn/os/os_Mutex.h>
#include <algorithm>
#include <utility>
#include <mutex>
#include <nn/sf/sf_DefaultAllocationPolicy.h>

#include <nn/ovln/ovln_CommonTypes.h>
#include <nn/ovln/ovln_ServiceTypes.h>

#define NNT_OVLN_ASSERT_RESULT_SUCCESS(result) ASSERT_TRUE((result).IsSuccess())
#define NNT_OVLN_ASSERT_RESULT_FAILED(expectedResult, result) ASSERT_TRUE(expectedResult::Includes(result))

namespace nnt { namespace ovln {

class ObservedMemoryResource
    : public nn::MemoryResource
{
public:

    template <typename Int>
    struct MinMax
    {
        Int min;
        Int max;

        MinMax() NN_NOEXCEPT
        {
            Reset();
        }

        void Reset() NN_NOEXCEPT
        {
            this->min = std::numeric_limits<Int>::max();
            this->max = std::numeric_limits<Int>::min();
        }

        void Update(Int value) NN_NOEXCEPT
        {
            this->min = std::min(this->min, value);
            this->max = std::max(this->max, value);
        }
    };

    struct AllocationInfo
    {
        int allocateCount;
        int deallocateCount;
        int currentAllocateCountMax;
        MinMax<size_t> allocatedSizeMinMax;
        MinMax<size_t> allocatedAlignmentMinMax;
        size_t currentAllocatedSize;
        size_t currentAllocatedSizeMax;

        AllocationInfo() NN_NOEXCEPT
        {
            Reset();
        }

        void Reset() NN_NOEXCEPT
        {
            this->allocateCount = 0;
            this->deallocateCount = 0;
            this->currentAllocateCountMax = 0;
            this->allocatedSizeMinMax.Reset();
            this->allocatedAlignmentMinMax.Reset();
            this->currentAllocatedSize = 0;
            this->currentAllocatedSizeMax = 0;
        }

        void OnAllocated(std::size_t size, std::size_t alignment) NN_NOEXCEPT
        {
            ++this->allocateCount;
            this->currentAllocateCountMax = std::max(currentAllocateCountMax, allocateCount - deallocateCount);
            this->allocatedSizeMinMax.Update(size);
            this->allocatedAlignmentMinMax.Update(alignment);
            this->currentAllocatedSize += size;
            this->currentAllocatedSizeMax = std::max(currentAllocatedSizeMax, currentAllocatedSize);
        }

        void OnDeallocated(std::size_t size, std::size_t alignment) NN_NOEXCEPT
        {
            NN_UNUSED(alignment);
            ++this->deallocateCount;
            this->currentAllocatedSize -= size;
        }
    };

    explicit ObservedMemoryResource(nn::MemoryResource* pMemoryResource = nn::sf::GetNewDeleteMemoryResource()) NN_NOEXCEPT
        : m_Mutex(false)
        , m_pMemoryResource(pMemoryResource)
        , m_pAllocationInfo(nullptr)
        , m_AllocationEnabled(true)
    {
    }

    void SetAllocationInfo(AllocationInfo* pAllocationInfo) NN_NOEXCEPT
    {
        this->m_pAllocationInfo = pAllocationInfo;
    }

    AllocationInfo* GetAllocationInfo() const NN_NOEXCEPT
    {
        return m_pAllocationInfo;
    }

    void SetAllocationEnabled(bool enabled) NN_NOEXCEPT
    {
        this->m_AllocationEnabled = enabled;
    }

private:

    mutable nn::os::Mutex m_Mutex;
    nn::MemoryResource* m_pMemoryResource;
    AllocationInfo* m_pAllocationInfo;
    bool m_AllocationEnabled;

    virtual void* do_allocate(std::size_t size, std::size_t alignment) NN_NOEXCEPT
    {
        if (!m_AllocationEnabled)
        {
            return nullptr;
        }
        auto p = m_pMemoryResource->allocate(size, alignment);
        if (!p)
        {
            return nullptr;
        }
        if (m_pAllocationInfo)
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            m_pAllocationInfo->OnAllocated(size, alignment);
        }
        return p;
    }

    virtual void do_deallocate(void* p, std::size_t size, std::size_t alignment) NN_NOEXCEPT
    {
        m_pMemoryResource->deallocate(p, size, alignment);
        if (m_pAllocationInfo)
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            m_pAllocationInfo->OnDeallocated(size, alignment);
        }
    }

    virtual bool do_is_equal(const MemoryResource& other) const NN_NOEXCEPT
    {
        return m_pMemoryResource->is_equal(other);
    }

};

const nn::ovln::SourceName SourceName1 = nn::ovln::MakeSourceName("1");
const nn::ovln::SourceName SourceName2 = nn::ovln::MakeSourceName("2");
const nn::ovln::SourceName SourceName3 = nn::ovln::MakeSourceName("3");
const nn::ovln::SourceName SourceName4 = nn::ovln::MakeSourceName("4");

inline nn::ovln::QueueAttribute MakeQueueAttribute(uint32_t queueLength) NN_NOEXCEPT
{
    nn::ovln::QueueAttribute ret = {};
    ret.queueLength = queueLength;
    return ret;
}

inline nn::ovln::SendOption MakeSendOption(nn::ovln::EnqueuePosition enqueuePosition, nn::ovln::OverflowOption overflowOption) NN_NOEXCEPT
{
    nn::ovln::SendOption ret = {};
    ret.enqueuePosition = static_cast<decltype(ret.enqueuePosition)>(enqueuePosition);
    ret.overflowOption = static_cast<decltype(ret.overflowOption)>(overflowOption);
    return ret;
}

const nn::ovln::SendOption SendOptionBackError = MakeSendOption(nn::ovln::EnqueuePosition_Back, nn::ovln::OverflowOption_Error);
const nn::ovln::SendOption SendOptionFrontError = MakeSendOption(nn::ovln::EnqueuePosition_Front, nn::ovln::OverflowOption_Error);
const nn::ovln::SendOption SendOptionFrontRemoveFront = MakeSendOption(nn::ovln::EnqueuePosition_Front, nn::ovln::OverflowOption_RemoveFront);
const nn::ovln::SendOption SendOptionFrontRemoveBack = MakeSendOption(nn::ovln::EnqueuePosition_Front, nn::ovln::OverflowOption_RemoveBack);

}}
