﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"

#include <cstring>
#include <cstdarg>

#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/util/util_FormatString.h>

#include <nn/os.h>
#include <nnt/nnt_Argument.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/os/os_InterruptEvent.h>
#include <nn/os/os_SystemEvent.h>

#include <nnc/os/os_SystemEvent.h>
#include <nnc/os/os_MemoryFence.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/teamcity/testTeamcity_Logger.h>

extern "C" void nntosTestTypeInfoDumpForC(void);

extern "C" bool nntosThreadTestForC(void);
extern "C" bool nntosEventTestForC(void);
extern "C" bool nntosSemaphoreTestForC(void);
extern "C" bool nntosTimerEventTestForC(void);
extern "C" bool nntosInterruptEventTestForC(void);
extern "C" bool nntosMutexTestForC(void);
extern "C" bool nntosConditionVariableTestForC(void);
extern "C" bool nntosMultipleWaitTestForC(void);
extern "C" bool nntosSystemEventTestForC(void);
extern "C" bool nntosMessageQueueTestForC(void);
extern "C" bool nntosThreadLocalStorageTestForC(void);
extern "C" bool nntosTickTestForC(void);
extern "C" bool nntosMemoryFenceTestForC(void);
extern "C" void testMultipleDefinition(void);

namespace nnt { namespace os {

TEST(LinkageC, TypeInfoDump)
{
    NN_LOG("\n");
    NN_LOG("C++:            ThreadType: size=0x%04x align=0x%02x\n", sizeof(nn::os::ThreadType), NN_ALIGNOF(nn::os::ThreadType));
    NN_LOG("C++:         SemaphoreType: size=0x%04x align=0x%02x\n", sizeof(nn::os::SemaphoreType), NN_ALIGNOF(nn::os::SemaphoreType));
    NN_LOG("C++:      MessageQueueType: size=0x%04x align=0x%02x\n", sizeof(nn::os::MessageQueueType), NN_ALIGNOF(nn::os::MessageQueueType));
    NN_LOG("C++:             EventType: size=0x%04x align=0x%02x\n", sizeof(nn::os::EventType), NN_ALIGNOF(nn::os::EventType));
    NN_LOG("C++:        TimerEventType: size=0x%04x align=0x%02x\n", sizeof(nn::os::TimerEventType), NN_ALIGNOF(nn::os::TimerEventType));
    NN_LOG("C++:    InterruptEventType: size=0x%04x align=0x%02x\n", sizeof(nn::os::InterruptEventType), NN_ALIGNOF(nn::os::InterruptEventType));
    NN_LOG("C++:       SystemEventType: size=0x%04x align=0x%02x\n", sizeof(nn::os::SystemEventType), NN_ALIGNOF(nn::os::SystemEventType));
    NN_LOG("C++:         MultiWaitType: size=0x%04x align=0x%02x\n", sizeof(nn::os::MultiWaitType), NN_ALIGNOF(nn::os::MultiWaitType));
    NN_LOG("C++:   MultiWaitHolderType: size=0x%04x align=0x%02x\n", sizeof(nn::os::MultiWaitHolderType), NN_ALIGNOF(nn::os::MultiWaitHolderType));
    NN_LOG("C++:             MutexType: size=0x%04x align=0x%02x\n", sizeof(nn::os::MutexType), NN_ALIGNOF(nn::os::MutexType));
    NN_LOG("C++: ConditionVariableType: size=0x%04x align=0x%02x\n", sizeof(nn::os::ConditionVariableType), NN_ALIGNOF(nn::os::ConditionVariableType));
    NN_LOG("C++:           BarrierType: size=0x%04x align=0x%02x\n", sizeof(nn::os::BarrierType), NN_ALIGNOF(nn::os::BarrierType));
    NN_LOG("\n");

    nntosTestTypeInfoDumpForC();
}

//-----------------------------------------------------------------------------
//  C 言語 API の動作テスト
//-----------------------------------------------------------------------------

TEST(LinkageC_Thread, ThreadApiTestForC)
{
    bool result = nntosThreadTestForC();
    result ? SUCCEED() : ADD_FAILURE();
}

TEST(LinkageC_Event, EventApiTestForC)
{
    bool result = nntosEventTestForC();
    result ? SUCCEED() : ADD_FAILURE();
}

TEST(LinkageC_Semaphore, SemaphoreApiTestForC)
{
    bool result = nntosSemaphoreTestForC();
    result ? SUCCEED() : ADD_FAILURE();
}

TEST(LinkageC_TimerEvent, TimerEventApiTestForC)
{
    bool result = nntosTimerEventTestForC();
    result ? SUCCEED() : ADD_FAILURE();
}

// Horizon OS でのみテストする
// Win32 では予め Event を作ったり名前を生成したりする必要がありテストコストが高いため
// （その割に誰も使わない想定なので…。）
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
TEST(LinkageC_InterruptEvent, InterruptEventApiTestForC)
{
    bool result = nntosInterruptEventTestForC();
    result ? SUCCEED() : ADD_FAILURE();
}
#endif

TEST(LinkageC_Mutex, MutexApiTestForC)
{
    bool result = nntosMutexTestForC();
    result ? SUCCEED() : ADD_FAILURE();
}

TEST(LinkageC_ConditionVariable, ConditionVariableApiTestForC)
{
    bool result = nntosConditionVariableTestForC();
    result ? SUCCEED() : ADD_FAILURE();
}

TEST(LinkageC_MultipleWait, MultipleWaitApiTestForC)
{
    bool result = nntosMultipleWaitTestForC();
    result ? SUCCEED() : ADD_FAILURE();
}

TEST(LinkageC_SystemEvent, SystemEventApiTestForC)
{
    bool result = nntosSystemEventTestForC();
    result ? SUCCEED() : ADD_FAILURE();
}

TEST(LinkageC_MesasgeQueue, MessageQueueApiTestForC)
{
    bool result = nntosMessageQueueTestForC();
    result ? SUCCEED() : ADD_FAILURE();
}

TEST(LinkageC_ThreadLocalStorage, ThreadLocalStorageApiTestForC)
{
    bool result = nntosThreadLocalStorageTestForC();
    result ? SUCCEED() : ADD_FAILURE();
}

TEST(LinkageC_Tick, TickApiTestForC)
{
    bool result = nntosTickTestForC();
    result ? SUCCEED() : ADD_FAILURE();
}

TEST(LinkageC_Tick, TickMemoryFenceTestForC)
{
    bool result = nntosMemoryFenceTestForC();
    result ? SUCCEED() : ADD_FAILURE();
}

//-----------------------------------------------------------------------------

TEST(LinkageC_SystemEvent, SystemEventTypeTestForC)
{
    nnosSystemEventType systemEventForC;

    nn::os::SystemEventType* systemEvent = reinterpret_cast<nn::os::SystemEventType*>(&systemEventForC);

    // SystemEvent を nn::os::Event として初期化
    NN_LOG(NN_TEXT("nnosSystemEvent を nn::os::Event として初期化\n"));
    auto result = nn::os::CreateSystemEvent(systemEvent, nn::os::EventClearMode_AutoClear, false);
    EXPECT_TRUE( result.IsSuccess() );

    NN_LOG(NN_TEXT("nnosSystemEvent を ClearSystemEvent() でクリア\n"));
    nn::os::ClearSystemEvent(systemEvent);

    NN_LOG(NN_TEXT("nnosSystemEvent を SignalSystemEvent() でシグナル化\n"));
    nn::os::SignalSystemEvent(systemEvent);

    NN_LOG(NN_TEXT("nnosSystemEvent を WaitSystemEvent() で待機\n"));
    nn::os::WaitSystemEvent(systemEvent);

    NN_LOG(NN_TEXT("nnosSystemEvent を TryWaitSystemEvent() で false を確認\n"));
    bool ret = nn::os::TryWaitSystemEvent(systemEvent);
    EXPECT_FALSE( ret );

    NN_LOG(NN_TEXT("nnosSystemEvent を TimedWaitSystemEvent() で false を確認\n"));
    ret = nn::os::TimedWaitSystemEvent(systemEvent, nn::TimeSpan::FromMilliSeconds(1));
    EXPECT_FALSE( ret );

    NN_LOG(NN_TEXT("nnosSystemEvent を破棄\n"));
    nn::os::DestroySystemEvent(systemEvent);

    // SystemEvent を nn::os::InterProcessEvent として初期化
    NN_LOG(NN_TEXT("nnosSystemEvent を nn::os::InterProcessEvent として初期化\n"));
    result = nn::os::CreateSystemEvent(systemEvent, nn::os::EventClearMode_AutoClear, true);
    EXPECT_TRUE( result.IsSuccess() );

    NN_LOG(NN_TEXT("nnosSystemEvent を ClearSystemEvent() でクリア\n"));
    nn::os::ClearSystemEvent(systemEvent);

    NN_LOG(NN_TEXT("nnosSystemEvent を SignalSystemEvent() でシグナル化\n"));
    nn::os::SignalSystemEvent(systemEvent);

    NN_LOG(NN_TEXT("nnosSystemEvent を WaitSystemEvent() で待機\n"));
    nn::os::WaitSystemEvent(systemEvent);

    NN_LOG(NN_TEXT("nnosSystemEvent を TryWaitSystemEvent() で false を確認\n"));
    ret = nn::os::TryWaitSystemEvent(systemEvent);
    EXPECT_FALSE( ret );

    NN_LOG(NN_TEXT("nnosSystemEvent を TimedWaitSystemEvent() で false を確認\n"));
    ret = nn::os::TimedWaitSystemEvent(systemEvent, nn::TimeSpan::FromMilliSeconds(1));
    EXPECT_FALSE( ret );

    NN_LOG(NN_TEXT("nnosSystemEvent をデタッチ\n"));
    auto readableHandle = nn::os::DetachReadableHandleOfSystemEvent(systemEvent);
    auto writableHandle = nn::os::DetachWritableHandleOfSystemEvent(systemEvent);

    NN_LOG(NN_TEXT("nnosSystemEvent を破棄\n"));
    nn::os::DestroySystemEvent(systemEvent);

    NN_LOG(NN_TEXT("nnosSystemEvent をアタッチで初期化\n"));
    nn::os::AttachSystemEvent(systemEvent, readableHandle, true, writableHandle, true, nn::os::EventClearMode_ManualClear);

    NN_LOG(NN_TEXT("nnosSystemEvent を ClearSystemEvent() でクリア\n"));
    nn::os::ClearSystemEvent(systemEvent);

    NN_LOG(NN_TEXT("nnosSystemEvent を SignalSystemEvent() でシグナル化\n"));
    nn::os::SignalSystemEvent(systemEvent);

    NN_LOG(NN_TEXT("nnosSystemEvent を WaitSystemEvent() で待機\n"));
    nn::os::WaitSystemEvent(systemEvent);

    NN_LOG(NN_TEXT("nnosSystemEvent を TryWaitSystemEvent() で true を確認\n"));
    ret = nn::os::TryWaitSystemEvent(systemEvent);
    EXPECT_TRUE( ret );

    NN_LOG(NN_TEXT("nnosSystemEvent を TimedWaitSystemEvent() で true を確認\n"));
    ret = nn::os::TimedWaitSystemEvent(systemEvent, nn::TimeSpan::FromMilliSeconds(1));
    EXPECT_TRUE( ret );

    NN_LOG(NN_TEXT("nnosSystemEvent を破棄\n"));
    nn::os::DestroySystemEvent(systemEvent);

    SUCCEED();
}

//-----------------------------------------------------------------------------

TEST(LinkageC_ListTest, CallFunction)
{
    testMultipleDefinition();
}

//-----------------------------------------------------------------------------

extern "C" void nnMain()
{
    // コマンドライン引数を取得
    int    argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    NN_LOG("TEST: argc=%d\n", argc);
    NN_LOG("TEST: argv=0x%p\n", argv);

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);

    // TeamCity の表示を適切にするため、イベントリスナの登録を一旦すべて解除し、
    // ServiceMessageLogger -> デフォルトのイベントリスナ の順で登録し直す。
    ::testing::TestEventListeners& listeners = ::testing::UnitTest::GetInstance()->listeners();
    ::testing::TestEventListener* defaultResultPrinter = listeners.Release(listeners.default_result_printer());
    listeners.Append(new nnt::teamcity::ServiceMessageLogger());
    listeners.Append(defaultResultPrinter);

    int result = RUN_ALL_TESTS();

    // 終了コードを設定
    nnt::Exit( result );
}

//-----------------------------------------------------------------------------

}}  // namespace nnt::os
