﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_UserExceptionHandler.h>

#include <nnt/nntest.h>
#include "../Common/test_SetjmpWithFpuContext.h"
#include "../Common/test_Helper.h"


namespace nnt { namespace os { namespace userExceptionHandler {

enum ExpectAddress
{
    ExpectAddress_InThreadStackRegion,
    ExpectAddress_InHandlerStackRegion,
    ExpectAddress_InGlobalObject,
};

const size_t              g_ExceptionHandlerStackSize = 0x8000;
NN_ALIGNAS(16) char       g_ExceptionHandlerStack[g_ExceptionHandlerStackSize];
nn::os::UserExceptionInfo g_UserExceptionInfo;

//---------------------------------------------------------------------------

nntosJmpbufWithFpuContext   g_JumpBuf;

const uintptr_t g_HandlerStackTop    = reinterpret_cast<const uintptr_t>(g_ExceptionHandlerStack);
const uintptr_t g_HandlerStackBottom = reinterpret_cast<const uintptr_t>(g_ExceptionHandlerStack + g_ExceptionHandlerStackSize);

uintptr_t       g_ThreadStackBegin;
uintptr_t       g_ThreadStackEnd;

ExpectAddress   g_ExpectUserExceptionStack;
ExpectAddress   g_ExpectUserExceptionInfo;

//---------------------------------------------------------------------------
// ユーザ例外ハンドラ
//  nn::os::UserExceptionInfo の内容をダンプして、
//  ハンドラスタックの位置と、UserExceptionInfo の位置が正しいかチェック

void UserExceptionHandlerFunction(nn::os::UserExceptionInfo* info)
{
#if defined(NN_BUILD_CONFIG_OS_WIN32)

    NN_UNUSED( info );

#elif defined(NN_BUILD_CONFIG_OS_HORIZON)

    uintptr_t sp;
    asm volatile("mov %0, sp" : "=r"(sp) : : "memory");

    // ユーザ例外ハンドラ起動情報のログ出力
    NN_LOG("\n");
    NN_LOG("Address of current stack = 0x%p\n", sp);
    NN_LOG("Address of info          = 0x%p\n", info);

    // UserExceptionInfo のログ出力
    NN_LOG("\n");
    NN_LOG("ExceptionType   = 0x%08x\n", info->exceptionType);

#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A) || \
    defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
    for (int i=0; i<13; ++i)
    {
        NN_LOG("r%02d             = 0x%08x\n", i, info->detail.r[i]);
    }
    NN_LOG("r13(sp)         = 0x%08x\n", info->detail.sp);
    NN_LOG("r14(lr)         = 0x%08x\n", info->detail.lr);
    NN_LOG("r15(pc)         = 0x%08x\n", info->detail.pc);
#elif defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)
    for (int i=0; i<30; ++i)
    {
        NN_LOG("r%02d             = 0x%p\n", i, info->detail.r[i]);
    }
    NN_LOG("r30(lr)         = 0x%p\n", info->detail.lr);
    NN_LOG("sp              = 0x%p\n", info->detail.sp);
    NN_LOG("pc              = 0x%p\n", info->detail.pc);
#endif
    NN_LOG("\n");

#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A)
    NN_LOG("cpsr            = 0x%08x\n", info->detail.cpsr);
    NN_LOG("fsr             = 0x%08x\n", info->detail.fsr);
    NN_LOG("far             = 0x%08x\n", info->detail.far);
    NN_LOG("fpexc           = 0x%08x\n", info->detail.fpexc);
    NN_LOG("fpinst          = 0x%08x\n", info->detail.fpinst);
    NN_LOG("fpinst2         = 0x%08x\n", info->detail.fpinst2);
#elif defined(NN_OS_CPU_ARM_AARCH32_ARMV8A) || \
      defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)
    NN_LOG("pstate          = 0x%08x\n", info->detail.pstate);
    NN_LOG("afsr0           = 0x%08x\n", info->detail.afsr0);
    NN_LOG("afsr1           = 0x%08x\n", info->detail.afsr1);
    NN_LOG("esr             = 0x%08x\n", info->detail.esr);
    NN_LOG("far             = 0x%p\n",   info->detail.far);
#endif
    NN_LOG("\n");

#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A) || \
    defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
#if NN_BUILD_CONFIG_FPU_NUM_DOUBLE_REGISTERS == 32
    for (int i=0; i<32; i += 2)
#else
    for (int i=0; i<16; i += 2)
#endif
    {
        NN_LOG("D%02d:D%02d         = 0x%016llx:%016llx\n", i + 1, i, info->detail.D[i + 1], info->detail.D[i]);
    }
#elif defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)
    for (int i=0; i<32; ++i)
    {
        NN_LOG("V%02d             = 0x%016llx%016llx\n", i, static_cast<nn::Bit64>(info->detail.V[i] >> 64), static_cast<nn::Bit64>(info->detail.V[i]));
    }
#endif
    NN_LOG("\n");

    // ユーザ例外ハンドラのスタック位置が正しいか
    NN_LOG(NN_TEXT("ユーザ例外ハンドラの sp（0x%p）が\n"), sp);
    switch (g_ExpectUserExceptionStack)
    {
    case ExpectAddress_InThreadStackRegion:
            NN_LOG(NN_TEXT("スレッドスタック（0x%p - 0x%p）の中にあるか"), g_ThreadStackBegin, g_ThreadStackEnd);
            ASSERT_TRUE( (sp >= g_ThreadStackBegin) &&
                         (sp <  g_ThreadStackEnd) );
            NN_LOG(" ... OK\n");
            break;

    case ExpectAddress_InHandlerStackRegion:
            NN_LOG(NN_TEXT("ハンドラスタック（0x%p - 0x%p）の中にあるか"), g_HandlerStackTop, g_HandlerStackBottom);
            ASSERT_TRUE( (sp >= g_HandlerStackTop) &&
                         (sp <  g_HandlerStackBottom) );
            NN_LOG(" ... OK\n");
            break;

    case ExpectAddress_InGlobalObject:
            // スタックに対して、この指定は NG
            FAIL();
            break;

    default:
            break;
    }

    // 例外情報の格納位置が正しいか
    uintptr_t infoAddr = reinterpret_cast<uintptr_t>(info);
    NN_LOG(NN_TEXT("ユーザ例外ハンドラの UserExceptionInfo のアドレス（0x%p）が\n"), infoAddr);
    switch (g_ExpectUserExceptionInfo)
    {
    case ExpectAddress_InThreadStackRegion:
            NN_LOG(NN_TEXT("スレッドスタック（0x%p - 0x%p）の中にあるか"), g_ThreadStackBegin, g_ThreadStackEnd);
            ASSERT_TRUE( (infoAddr >= g_ThreadStackBegin) &&
                         (infoAddr <  g_ThreadStackEnd) );
            NN_LOG(" ... OK\n");
            break;

    case ExpectAddress_InHandlerStackRegion:
            NN_LOG(NN_TEXT("ハンドラスタック（0x%p - 0x%p）の中にあるか"), g_HandlerStackTop, g_HandlerStackBottom);
            ASSERT_TRUE( (infoAddr >= g_HandlerStackTop) &&
                         (infoAddr <  g_HandlerStackBottom) );
            NN_LOG(" ... OK\n");
            break;

    case ExpectAddress_InGlobalObject:
            NN_LOG(NN_TEXT("専用のグローバル領域（0x%p）を指しているか"), &g_UserExceptionInfo);
            ASSERT_TRUE( info == &g_UserExceptionInfo );
            NN_LOG(" ... OK\n");
            break;

    default:
            break;
    }

    NN_LOG("\n");

    // 問題なければ、メモリアクセス違反を起こす寸前へ戻る
    nntosLongjmpWithFpuContext( &g_JumpBuf );
#endif
}

//---------------------------------------------------------------------------
//  自スレッドのスタック領域の先頭と末尾を計算（メインスレッドに限る）
//  stackSize は yml の設定値と合わせておくこと。
//
void CalcMyStackRegion()
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    uintptr_t sp;
    asm volatile("mov %0, sp" : "=r"(sp) : : "memory");

    const size_t    stackSize = 0x1000;

    g_ThreadStackBegin  = sp & ~(stackSize - 1);
    g_ThreadStackEnd    = g_ThreadStackBegin + stackSize;
#endif
}

//---------------------------------------------------------------------------
// メモリアクセス例外を意図的に発生させて、ユーザ例外ハンドラを起動させる。
//
bool AccessViolation()
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)

    CalcMyStackRegion();

    volatile int* accessViolation = reinterpret_cast<volatile int*>(AccessViolation);

    NN_LOG(NN_TEXT("ユーザ例外を引き起こすアドレス           = 0x%p\n"), accessViolation);

    if (nntosSetjmpWithFpuContext( &g_JumpBuf ) == 0)
    {
#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A) || \
    defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
        //                                       Dn[31:0]        Dn[63:32]
        asm volatile("vmov d0,  %0, %1" : : "r"(0x00000000), "r"(0x00FFFF00) :);
        asm volatile("vmov d1,  %0, %1" : : "r"(0x01000001), "r"(0x01FFFF01) :);
        asm volatile("vmov d2,  %0, %1" : : "r"(0x02000002), "r"(0x02FFFF02) :);
        asm volatile("vmov d3,  %0, %1" : : "r"(0x03000003), "r"(0x03FFFF03) :);
        asm volatile("vmov d4,  %0, %1" : : "r"(0x04000004), "r"(0x04FFFF04) :);
        asm volatile("vmov d5,  %0, %1" : : "r"(0x05000005), "r"(0x05FFFF05) :);
        asm volatile("vmov d6,  %0, %1" : : "r"(0x06000006), "r"(0x06FFFF06) :);
        asm volatile("vmov d7,  %0, %1" : : "r"(0x07000007), "r"(0x07FFFF07) :);
        asm volatile("vmov d8,  %0, %1" : : "r"(0x08000008), "r"(0x08FFFF08) :);
        asm volatile("vmov d9,  %0, %1" : : "r"(0x09000009), "r"(0x09FFFF09) :);
        asm volatile("vmov d10, %0, %1" : : "r"(0x0a00000a), "r"(0x0aFFFF0a) :);
        asm volatile("vmov d11, %0, %1" : : "r"(0x0b00000b), "r"(0x0bFFFF0b) :);
        asm volatile("vmov d12, %0, %1" : : "r"(0x0c00000c), "r"(0x0cFFFF0c) :);
        asm volatile("vmov d13, %0, %1" : : "r"(0x0d00000d), "r"(0x0dFFFF0d) :);
        asm volatile("vmov d14, %0, %1" : : "r"(0x0e00000e), "r"(0x0eFFFF0e) :);
        asm volatile("vmov d15, %0, %1" : : "r"(0x0f00000f), "r"(0x0fFFFF0f) :);
#if NN_BUILD_CONFIG_FPU_NUM_DOUBLE_REGISTERS == 32
        asm volatile("vmov d16, %0, %1" : : "r"(0x10000010), "r"(0x10FFFF10) :);
        asm volatile("vmov d17, %0, %1" : : "r"(0x11000011), "r"(0x11FFFF11) :);
        asm volatile("vmov d18, %0, %1" : : "r"(0x12000012), "r"(0x12FFFF12) :);
        asm volatile("vmov d19, %0, %1" : : "r"(0x13000013), "r"(0x13FFFF13) :);
        asm volatile("vmov d20, %0, %1" : : "r"(0x14000014), "r"(0x14FFFF14) :);
        asm volatile("vmov d21, %0, %1" : : "r"(0x15000015), "r"(0x15FFFF15) :);
        asm volatile("vmov d22, %0, %1" : : "r"(0x16000016), "r"(0x16FFFF16) :);
        asm volatile("vmov d23, %0, %1" : : "r"(0x17000017), "r"(0x17FFFF17) :);
        asm volatile("vmov d24, %0, %1" : : "r"(0x18000018), "r"(0x18FFFF18) :);
        asm volatile("vmov d25, %0, %1" : : "r"(0x19000019), "r"(0x19FFFF19) :);
        asm volatile("vmov d26, %0, %1" : : "r"(0x1a00001a), "r"(0x1aFFFF1a) :);
        asm volatile("vmov d27, %0, %1" : : "r"(0x1b00001b), "r"(0x1bFFFF1b) :);
        asm volatile("vmov d28, %0, %1" : : "r"(0x1c00001c), "r"(0x1cFFFF1c) :);
        asm volatile("vmov d29, %0, %1" : : "r"(0x1d00001d), "r"(0x1dFFFF1d) :);
        asm volatile("vmov d30, %0, %1" : : "r"(0x1e00001e), "r"(0x1eFFFF1e) :);
        asm volatile("vmov d31, %0, %1" : : "r"(0x1f00001f), "r"(0x1fFFFF1f) :);
#endif
#elif defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)
        //                                       <--- 64 bit --->
        asm volatile("ins v0.D[0],  %0" :: "r"(0x0000000000000000ul) : ); // 31:0
        asm volatile("ins v0.D[1],  %0" :: "r"(0xffffffffffffffff) : ); // 63:32
        asm volatile("ins v1.D[0],  %0" :: "r"(0x0101010101010101) : ); // 31:0
        asm volatile("ins v1.D[1],  %0" :: "r"(0xfefefefefefefefe) : ); // 63:32
        asm volatile("ins v2.D[0],  %0" :: "r"(0x0202020202020202) : ); // 31:0
        asm volatile("ins v2.D[1],  %0" :: "r"(0xfdfdfdfdfdfdfdfd) : ); // 63:32
        asm volatile("ins v3.D[0],  %0" :: "r"(0x0303030303030303) : ); // 31:0
        asm volatile("ins v3.D[1],  %0" :: "r"(0xfcfcfcfcfcfcfcfc) : ); // 63:32
        asm volatile("ins v4.D[0],  %0" :: "r"(0x0404040404040404) : ); // 31:0
        asm volatile("ins v4.D[1],  %0" :: "r"(0xfbfbfbfbfbfbfbfb) : ); // 63:32
        asm volatile("ins v5.D[0],  %0" :: "r"(0x0505050505050505) : ); // 31:0
        asm volatile("ins v5.D[1],  %0" :: "r"(0xfafafafafafafafa) : ); // 63:32
        asm volatile("ins v6.D[0],  %0" :: "r"(0x0606060606060606) : ); // 31:0
        asm volatile("ins v6.D[1],  %0" :: "r"(0xf9f9f9f9f9f9f9f9) : ); // 63:32
        asm volatile("ins v7.D[0],  %0" :: "r"(0x0707070707070707) : ); // 31:0
        asm volatile("ins v7.D[1],  %0" :: "r"(0xf8f8f8f8f8f8f8f8) : ); // 63:32
        asm volatile("ins v8.D[0],  %0" :: "r"(0x0808080808080808) : ); // 31:0
        asm volatile("ins v8.D[1],  %0" :: "r"(0xf7f7f7f7f7f7f7f7) : ); // 63:32
        asm volatile("ins v9.D[0],  %0" :: "r"(0x0909090909090909) : ); // 31:0
        asm volatile("ins v9.D[1],  %0" :: "r"(0xf6f6f6f6f6f6f6f6) : ); // 63:32
        asm volatile("ins v10.D[0], %0" :: "r"(0x0a0a0a0a0a0a0a0a) : ); // 31:0
        asm volatile("ins v10.D[1], %0" :: "r"(0xf5f5f5f5f5f5f5f5) : ); // 63:32
        asm volatile("ins v11.D[0], %0" :: "r"(0x0b0b0b0b0b0b0b0b) : ); // 31:0
        asm volatile("ins v11.D[1], %0" :: "r"(0xf4f4f4f4f4f4f4f4) : ); // 63:32
        asm volatile("ins v12.D[0], %0" :: "r"(0x0c0c0c0c0c0c0c0c) : ); // 31:0
        asm volatile("ins v12.D[1], %0" :: "r"(0xf3f3f3f3f3f3f3f3) : ); // 63:32
        asm volatile("ins v13.D[0], %0" :: "r"(0x0d0d0d0d0d0d0d0d) : ); // 31:0
        asm volatile("ins v13.D[1], %0" :: "r"(0xf2f2f2f2f2f2f2f2) : ); // 63:32
        asm volatile("ins v14.D[0], %0" :: "r"(0x0e0e0e0e0e0e0e0e) : ); // 31:0
        asm volatile("ins v14.D[1], %0" :: "r"(0xf1f1f1f1f1f1f1f1) : ); // 63:32
        asm volatile("ins v15.D[0], %0" :: "r"(0x0f0f0f0f0f0f0f0f) : ); // 31:0
        asm volatile("ins v15.D[1], %0" :: "r"(0xf0f0f0f0f0f0f0f0) : ); // 63:32

        asm volatile("ins v16.D[0], %0" :: "r"(0x1010101010101010) : ); // 31:0
        asm volatile("ins v16.D[1], %0" :: "r"(0xefefefefefefefef) : ); // 63:32
        asm volatile("ins v17.D[0], %0" :: "r"(0x1111111111111111) : ); // 31:0
        asm volatile("ins v17.D[1], %0" :: "r"(0xeeeeeeeeeeeeeeee) : ); // 63:32
        asm volatile("ins v18.D[0], %0" :: "r"(0x1212121212121212) : ); // 31:0
        asm volatile("ins v18.D[1], %0" :: "r"(0xedededededededed) : ); // 63:32
        asm volatile("ins v19.D[0], %0" :: "r"(0x1313131313131313) : ); // 31:0
        asm volatile("ins v19.D[1], %0" :: "r"(0xecececececececec) : ); // 63:32
        asm volatile("ins v20.D[0], %0" :: "r"(0x1414141414141414) : ); // 31:0
        asm volatile("ins v20.D[1], %0" :: "r"(0xebebebebebebebeb) : ); // 63:32
        asm volatile("ins v21.D[0], %0" :: "r"(0x1515151515151515) : ); // 31:0
        asm volatile("ins v21.D[1], %0" :: "r"(0xeaeaeaeaeaeaeaea) : ); // 63:32
        asm volatile("ins v22.D[0], %0" :: "r"(0x1616161616161616) : ); // 31:0
        asm volatile("ins v22.D[1], %0" :: "r"(0xe9e9e9e9e9e9e9e9) : ); // 63:32
        asm volatile("ins v23.D[0], %0" :: "r"(0x1717171717171717) : ); // 31:0
        asm volatile("ins v23.D[1], %0" :: "r"(0xe8e8e8e8e8e8e8e8) : ); // 63:32
        asm volatile("ins v24.D[0], %0" :: "r"(0x1818181818181818) : ); // 31:0
        asm volatile("ins v24.D[1], %0" :: "r"(0xe7e7e7e7e7e7e7e7) : ); // 63:32
        asm volatile("ins v25.D[0], %0" :: "r"(0x1919191919191919) : ); // 31:0
        asm volatile("ins v25.D[1], %0" :: "r"(0xe6e6e6e6e6e6e6e6) : ); // 63:32
        asm volatile("ins v26.D[0], %0" :: "r"(0x1a1a1a1a1a1a1a1a) : ); // 31:0
        asm volatile("ins v26.D[1], %0" :: "r"(0xe5e5e5e5e5e5e5e5) : ); // 63:32
        asm volatile("ins v27.D[0], %0" :: "r"(0x1b1b1b1b1b1b1b1b) : ); // 31:0
        asm volatile("ins v27.D[1], %0" :: "r"(0xe4e4e4e4e4e4e4e4) : ); // 63:32
        asm volatile("ins v28.D[0], %0" :: "r"(0x1c1c1c1c1c1c1c1c) : ); // 31:0
        asm volatile("ins v28.D[1], %0" :: "r"(0xe3e3e3e3e3e3e3e3) : ); // 63:32
        asm volatile("ins v29.D[0], %0" :: "r"(0x1d1d1d1d1d1d1d1d) : ); // 31:0
        asm volatile("ins v29.D[1], %0" :: "r"(0xe2e2e2e2e2e2e2e2) : ); // 63:32
        asm volatile("ins v30.D[0], %0" :: "r"(0x1e1e1e1e1e1e1e1e) : ); // 31:0
        asm volatile("ins v30.D[1], %0" :: "r"(0xe1e1e1e1e1e1e1e1) : ); // 63:32
        asm volatile("ins v31.D[0], %0" :: "r"(0x1f1f1f1f1f1f1f1f) : ); // 31:0
        asm volatile("ins v31.D[1], %0" :: "r"(0xe0e0e0e0e0e0e0e0) : ); // 63:32
#endif

        // メモリアクセス違反を起こす
        *accessViolation = *accessViolation;

        // メモリアクセス違反が起こらなければテスト失敗
        ADD_FAILURE();
    }
    else
    {
        // longjmp() で復帰できればテスト成功
        SUCCEED();
    }
#endif

    return true;
}   // NOLINT(readability/fn_size)

//---------------------------------------------------------------------------
//  ユーザ例外ハンドラの動作テスト
//  stackBottom と pExceptionInfo の場所を色々変えてテストする（全 2x3 通り）
//
//  stackBottom は以下の２種類
//      1) アプリで用意する例外ハンドラ専用スタック
//      2) nn::os::HandlerStackUsesThreadStack によるスレッドスタック
//
//  pExceptionInfo は以下の３種類
//      A) アプリで用意する例外ハンドラ専用のオブジェクト領域
//      B) nn::os::UserExceptionInfoUsesHandlerStack によるハンドラスタック
//      C) nn::os::UserExceptionInfoUsesThreadStack  によるスレッドスタック
//---------------------------------------------------------------------------

TEST(UserExceptionHandlerTest, test_UserExceptionHandlerPrologue)
{
    NN_LOG(NN_TEXT("ユーザ例外ハンドラ専用のスタック領域先頭 = 0x%p\n"), g_HandlerStackTop);
    NN_LOG(NN_TEXT("ユーザ例外ハンドラ専用のスタック領域末尾 = 0x%p\n"), g_HandlerStackBottom);
    NN_LOG(NN_TEXT("ユーザ例外ハンドラ専用の例外情報格納領域 = 0x%p\n"), &g_UserExceptionInfo);

    // ユーザ例外ハンドラの API が呼べることを確認する
    nn::os::EnableUserExceptionHandlerOnDebugging(true);
    nn::os::EnableUserExceptionHandlerOnDebugging(false);

    SUCCEED();
}

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

TEST(UserExceptionHandlerTest, test_UserExceptionHandler_1_A)
{
    // 1-A) 例外スタックは専用、例外情報も専用領域
    NN_LOG(NN_TEXT("ユーザ例外ハンドラの設定とハンドリング(1-A)\n"));

    nn::os::SetUserExceptionHandler(UserExceptionHandlerFunction,
                                    g_ExceptionHandlerStack,
                                    sizeof(g_ExceptionHandlerStack),
                                    &g_UserExceptionInfo);

    // 例外処理中の UserExpectInfo 領域とスタック領域の期待値を設定
    g_ExpectUserExceptionStack = ExpectAddress_InHandlerStackRegion;
    g_ExpectUserExceptionInfo  = ExpectAddress_InGlobalObject;

    AccessViolation();

    RestoreUserExceptionHandler();
}

TEST(UserExceptionHandlerTest, test_UserExceptionHandler_1_B)
{
    // 1-B) 例外スタックは専用、例外情報はハンドラスタック
    NN_LOG(NN_TEXT("ユーザ例外ハンドラの設定とハンドリング(1-B)\n"));

    nn::os::SetUserExceptionHandler(UserExceptionHandlerFunction,
                                    g_ExceptionHandlerStack,
                                    sizeof(g_ExceptionHandlerStack),
                                    nn::os::UserExceptionInfoUsesHandlerStack);

    // 例外処理中の UserExpectInfo 領域とスタック領域の期待値を設定
    g_ExpectUserExceptionStack = ExpectAddress_InHandlerStackRegion;
    g_ExpectUserExceptionInfo  = ExpectAddress_InHandlerStackRegion;

    AccessViolation();

    RestoreUserExceptionHandler();
}

TEST(UserExceptionHandlerTest, test_UserExceptionHandler_1_C)
{
    // 1-C) 例外スタックは専用、例外情報はスレッドスタック
    NN_LOG(NN_TEXT("ユーザ例外ハンドラの設定とハンドリング(1-C)\n"));

    nn::os::SetUserExceptionHandler(UserExceptionHandlerFunction,
                                    g_ExceptionHandlerStack,
                                    sizeof(g_ExceptionHandlerStack),
                                    nn::os::UserExceptionInfoUsesThreadStack);

    // 例外処理中の UserExpectInfo 領域とスタック領域の期待値を設定
    g_ExpectUserExceptionStack = ExpectAddress_InHandlerStackRegion;
    g_ExpectUserExceptionInfo  = ExpectAddress_InThreadStackRegion;

    AccessViolation();

    RestoreUserExceptionHandler();
}


TEST(UserExceptionHandlerTest, test_UserExceptionHandler_2_A)
{
    // 2-A) 例外スタックはスレッドスタック、例外情報も専用領域
    NN_LOG(NN_TEXT("ユーザ例外ハンドラの設定とハンドリング(2-A)\n"));

    nn::os::SetUserExceptionHandler(UserExceptionHandlerFunction,
                                    nn::os::HandlerStackUsesThreadStack,
                                    0,
                                    &g_UserExceptionInfo);

    // 例外処理中の UserExpectInfo 領域とスタック領域の期待値を設定
    g_ExpectUserExceptionStack = ExpectAddress_InThreadStackRegion;
    g_ExpectUserExceptionInfo  = ExpectAddress_InGlobalObject;

    AccessViolation();

    RestoreUserExceptionHandler();
}

TEST(UserExceptionHandlerTest, test_UserExceptionHandler_2_B)
{
    // 2-B) 例外スタックはスレッドスタック、例外情報はハンドラスタック
    NN_LOG(NN_TEXT("ユーザ例外ハンドラの設定とハンドリング(2-B)\n"));

    nn::os::SetUserExceptionHandler(UserExceptionHandlerFunction,
                                    nn::os::HandlerStackUsesThreadStack,
                                    0,
                                    nn::os::UserExceptionInfoUsesHandlerStack);

    // 例外処理中の UserExpectInfo 領域とスタック領域の期待値を設定
    //  ※）Info はハンドラスタックを使う設定になっているが、
    //      ハンドラスタック自身がスレッドスタックを使う設定になっているため、
    //      Info の場所はスレッドスタック上にあることが期待値となる。
    g_ExpectUserExceptionStack = ExpectAddress_InThreadStackRegion;
    g_ExpectUserExceptionInfo  = ExpectAddress_InThreadStackRegion;

    AccessViolation();

    RestoreUserExceptionHandler();
}

TEST(UserExceptionHandlerTest, test_UserExceptionHandler_2_C)
{
    // 2-C) 例外スタックはスレッドスタック、例外情報はスレッドスタック
    NN_LOG(NN_TEXT("ユーザ例外ハンドラの設定とハンドリング(2-C)\n"));

    nn::os::SetUserExceptionHandler(UserExceptionHandlerFunction,
                                    nn::os::HandlerStackUsesThreadStack,
                                    0,
                                    nn::os::UserExceptionInfoUsesThreadStack);

    // 例外処理中の UserExpectInfo 領域とスタック領域の期待値を設定
    g_ExpectUserExceptionStack = ExpectAddress_InThreadStackRegion;
    g_ExpectUserExceptionInfo  = ExpectAddress_InThreadStackRegion;

    AccessViolation();

    RestoreUserExceptionHandler();
}

// ユーザー例外ハンドラでスタックオーバーフローする場合のテスト
// 実機環境で Death test ができないので無効にしておく
#if 0
NN_OS_ALIGNAS_THREAD_STACK  char    g_ThreadStack[4096];

void UserExceptionHandlerForStackOverflowTest(nn::os::UserExceptionInfo* info)
{
    NN_UNUSED(info);

    volatile char dummy[4080];

    for (int i=0; i<4080; i++)
    {
        NN_LOG("%d ", dummy[i]);
    }
}

void testFunc_TestStackOverflowAtExceptionHandler(void* argument)
{
    NN_UNUSED(argument);

    AccessViolation();
}

TEST(UserExceptionHandlerTest, test_StackOverflow)
{
    nn::os::ThreadType thread;

    // StackOverflow) 例外スタックはスレッドスタック、例外情報はスレッドスタック
    NN_LOG(NN_TEXT("スタックオーバーフロー状態で例外ハンドラを起動\n"));

    nn::os::SetUserExceptionHandler(UserExceptionHandlerForStackOverflowTest,
                                    nn::os::HandlerStackUsesThreadStack,
                                    0,
                                    &g_UserExceptionInfo);

    // 例外処理中の UserExpectInfo 領域とスタック領域の期待値を設定
    g_ExpectUserExceptionStack = ExpectAddress_InThreadStackRegion;
    g_ExpectUserExceptionInfo  = ExpectAddress_InThreadStackRegion;

    auto result = nn::os::CreateThread( &thread, testFunc_TestStackOverflowAtExceptionHandler, NULL, g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority);
    ASSERT_TRUE( result.IsSuccess() );

    nn::os::StartThread( &thread );
    nn::os::WaitThread( &thread );
    nn::os::DestroyThread( &thread );

    nn::os::SetUserExceptionHandler(NULL, NULL, 0, NULL);
}
#endif

#endif // defined(NN_BUILD_CONFIG_OS_HORIZON)

//---------------------------------------------------------------------------

}}} // namespace nnt::os::userExceptionHandler

