﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_UserExceptionHandler.h>
#include <nn/os/os_DefaultUserExceptionHandler.h>
#include <nn/applet/applet.h>
#include <nn/diag/diag_Break.h>

#include <nnt/nntest.h>
#include "../Common/test_SetjmpWithFpuContext.h"
#include "../Common/test_Helper.h"

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/svc/svc_Base.h>

namespace nn { namespace os { namespace detail {
//-----------------------------------------------------------------------------
//  デフォルトユーザ例外ハンドラのヘッダ部分のログ出力を変更し、
//  テストのスクリプトで FAILED 扱いにならないようにする。
//-----------------------------------------------------------------------------

void PrintExceptionHeader(nn::os::UserExceptionInfo* info) NN_NOEXCEPT
{
    NN_UNUSED(info);

#if defined(NNT_OS_SYSTEM_PROGRAM_TEST)
    os::ProcessId processId = {};
    nn::svc::GetProcessId(&processId.value, svc::PSEUDO_HANDLE_CURRENT_PROCESS);
    NN_LOG("DefaultUserExceptionHandler: pid=%lld\n", processId);
#else
    NN_LOG("DefaultUserExceptionHandler: ARUID=%lld\n", nn::applet::GetAppletResourceUserId());
#endif
}

}}} // namespace nn::os::detail

//-----------------------------------------------------------------------------

namespace nnt { namespace os { namespace userExceptionHandler {

namespace {

NN_ALIGNAS(4096) char g_MemoryBlockForPermissionException[0x1000];
nntosJmpbufWithFpuContext   g_JumpBuf;

}

//-----------------------------------------------------------------------------
// デフォルトのユーザー例外ハンドラが呼ばれるかどうかをチェック
//
void TestDefaultUserExceptionHandler(nn::os::UserExceptionInfo *info) NN_NOEXCEPT
{
    NN_LOG(NN_TEXT("ユーザー例外ハンドラの先頭\n"));

    NN_LOG(NN_TEXT("nn::os::DefaultUserExceptionHandler() を呼ぶ\n"));
    nn::os::DefaultUserExceptionHandler(info);

    NN_LOG(NN_TEXT("テストのために復帰\n"));

    // メモリアクセス違反を起こす寸前へ戻る
    nntosLongjmpWithFpuContext( &g_JumpBuf );
}

TEST(DefaultUserExceptionHandlerTest, CheckDefaultUserExceptionHandler)
{
    nn::os::SetUserExceptionHandler(TestDefaultUserExceptionHandler, nn::os::HandlerStackUsesThreadStack, 0, nn::os::UserExceptionInfoUsesThreadStack);

    if (nntosSetjmpWithFpuContext( &g_JumpBuf ) == 0)
    {
        NN_LOG(NN_TEXT("Null ポインタにアクセスし、例外を発生させる\n"));
        volatile int* nullPtr = NULL;
        *nullPtr = *nullPtr;
    }
    NN_LOG("OK.\n");

    RestoreUserExceptionHandler();
}

//-----------------------------------------------------------------------------
// 許可されていないシステムコールを呼び出した場合
//
TEST(DefaultUserExceptionHandlerTest, CallInvalidSVC)
{
    nn::os::SetUserExceptionHandler(TestDefaultUserExceptionHandler, nn::os::HandlerStackUsesThreadStack, 0, nn::os::UserExceptionInfoUsesThreadStack);

    if (nntosSetjmpWithFpuContext( &g_JumpBuf ) == 0)
    {
        NN_LOG(NN_TEXT("存在しない SVC を発行し、例外を発生させる\n"));

#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A)
        asm volatile ("svc #0");
#elif defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
        asm volatile ("svc #0");
#elif defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)
        asm volatile ("svc #0");
#endif
    }
    NN_LOG("OK.\n");

    RestoreUserExceptionHandler();
}

//-----------------------------------------------------------------------------
// NN_DIAG_BREAK() による BREAK 命令の埋め込み
//
TEST(DefaultUserExceptionHandlerTest, EmplaceNnDiagBreak)
{
    nn::os::SetUserExceptionHandler(TestDefaultUserExceptionHandler, nn::os::HandlerStackUsesThreadStack, 0, nn::os::UserExceptionInfoUsesThreadStack);

    if (nntosSetjmpWithFpuContext( &g_JumpBuf ) == 0)
    {
        NN_LOG(NN_TEXT("NN_DIAG_BREAK() マクロによる BREAK 命令を配置し、例外を発生させる\n"));
        NN_DIAG_BREAK();
    }
    NN_LOG("OK.\n");

    RestoreUserExceptionHandler();
}

//-----------------------------------------------------------------------------
// NN_DIAG_BREAK() 以外の未定義命令例外の発生
//
TEST(DefaultUserExceptionHandlerTest, UndefinedInstruction)
{
    nn::os::SetUserExceptionHandler(TestDefaultUserExceptionHandler, nn::os::HandlerStackUsesThreadStack, 0, nn::os::UserExceptionInfoUsesThreadStack);

    if (nntosSetjmpWithFpuContext( &g_JumpBuf ) == 0)
    {
        NN_LOG(NN_TEXT("未定義命令例外を発生させる\n"));
        asm volatile(" .word 0xE7F0DEF0");
    }
    NN_LOG("OK.\n");

    RestoreUserExceptionHandler();
}

//-----------------------------------------------------------------------------
// 各種データアボート例外の発生
//
TEST(DefaultUserExceptionHandlerTest, DataAbort)
{
    nn::os::SetUserExceptionHandler(TestDefaultUserExceptionHandler, nn::os::HandlerStackUsesThreadStack, 0, nn::os::UserExceptionInfoUsesThreadStack);

    for (int testNum=0; testNum<3; ++testNum)
    {
        if (nntosSetjmpWithFpuContext( &g_JumpBuf ) == 0)
        {
            switch (testNum)
            {
            case 0:
                NN_LOG(NN_TEXT("未マップ空間へのメモリアクセス\n"));
                *reinterpret_cast<volatile int *>(0x4) = 0;
                break;

            case 1:
                NN_LOG(NN_TEXT("パーミション違反のメモリアクセス（ReadOnly 領域への Write）\n"));
                nn::os::SetMemoryPermission(reinterpret_cast<uintptr_t>(g_MemoryBlockForPermissionException), sizeof(g_MemoryBlockForPermissionException), nn::os::MemoryPermission_ReadOnly);
                g_MemoryBlockForPermissionException[0] = 0;
                break;

            case 2:
                NN_LOG(NN_TEXT("パーミション違反のメモリアクセス（EX 領域への Write）\n"));
                *reinterpret_cast<volatile int*>(nntosSetjmpWithFpuContext) = 0;
                break;

            default: NN_UNEXPECTED_DEFAULT;
            }
        }
        NN_LOG("OK.\n");
    }

    RestoreUserExceptionHandler();
}

//-----------------------------------------------------------------------------
// 不正なメモリ領域への命令アクセス例外の発生
//
TEST(DefaultUserExceptionHandlerTest, InvalidInstructionAccess)
{
    nn::os::SetUserExceptionHandler(TestDefaultUserExceptionHandler, nn::os::HandlerStackUsesThreadStack, 0, nn::os::UserExceptionInfoUsesThreadStack);

    if (nntosSetjmpWithFpuContext( &g_JumpBuf ) == 0)
    {
        NN_LOG(NN_TEXT("命令アクセス例外を発生させる（ReadOnly 領域へのジャンプ）\n"));
        reinterpret_cast<void (*)()>(g_MemoryBlockForPermissionException)();
    }
    NN_LOG("OK.\n");

    RestoreUserExceptionHandler();
}

}}} // namespace nnt::os::userExceptionHandler

#endif
