﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  TimerEvent 関連機能のテスト
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include <nn/nn_Common.h>
#include <nn/nn_SdkText.h>
#include <nn/os.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"
#include "test_TimerEventHelper.h"

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

#define EVENT_AUTO_CLEAR    nn::os::EventClearMode_AutoClear
#define EVENT_MANUAL_CLEAR  nn::os::EventClearMode_ManualClear

namespace nnt { namespace os { namespace timerEvent {

//---------------------------------------------------------------------------
// InitializeTimerEvent 直後の初期状態チェック

TEST(InitializeTimerEvent, test_initiallySignaled1)
{
    // InitializeTimerEvent 初期状態チェック
    doTest1(false,                      // TimerEvent は Not Signaled
            EVENT_AUTO_CLEAR,           // TimerEvent は自動クリアモード
            THR_NOTUSE,                 // Thread1 も未使用
            true,                       // Thread1 での API 返値
            false,                      // TimerEvent は Not Signaled
            THR_STATE_NONE);            // Thread1 は終了
}

TEST(InitializeTimerEvent, test_initiallySignaled2)
{
    // InitializeTimerEvent 初期状態チェック
    doTest1(false,                      // TimerEvent は Not Signaled
            EVENT_MANUAL_CLEAR,         // TimerEvent は手動クリアモード
            THR_NOTUSE,                 // Thread1 も未使用
            true,                       // Thread1 での API 返値
            false,                      // TimerEvent は Not Signaled
            THR_STATE_NONE);            // Thread1 は終了
}

TEST(InitializeTimerEvent, test_initiallyNotSignaled1)
{
    // InitializeTimerEvent 初期状態チェック
    doTest1(true,                       // TimerEvent は Signaled
            EVENT_AUTO_CLEAR,           // TimerEvent は自動クリアモード
            THR_NOTUSE,                 // Thread1 も未使用
            true,                       // Thread1 での API 返値
            true,                       // TimerEvent は Signaled
            THR_STATE_NONE);            // Thread1 は終了
}

TEST(InitializeTimerEvent, test_initiallyNotSignaled2)
{
    // InitializeTimerEvent 初期状態チェック
    doTest1(true,                       // TimerEvent は Signaled
            EVENT_MANUAL_CLEAR,         // TimerEvent は手動クリアモード
            THR_NOTUSE,                 // Thread1 も未使用
            true,                       // Thread1 での API 返値
            true,                       // TimerEvent は Signaled
            THR_STATE_NONE);            // Thread1 は終了
}

//---------------------------------------------------------------------------
// 単独の SignalTimerEvent() テスト

TEST(SignalTimerEvent, test_NotSignalToSignal)
{
    // SignalTimerEvent 非シグナルからシグナルへ
    doTest1(false,                      // TimerEvent は Not Signaled
            EVENT_AUTO_CLEAR,           // TimerEvent は自動クリアモード
            THR_CALL_SIGNAL,            // Thread1 は SignalTimerEvent()
            true,                       // Thread1 での API 返値
            true,                       // TimerEvent は Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(SignalTimerEvent, test_SignalToSignal)
{
    // SignalTimerEvent シグナルからシグナルへ
    doTest1(true,                       // TimerEvent は Signaled
            EVENT_AUTO_CLEAR,           // TimerEvent は自動クリアモード
            THR_CALL_SIGNAL,            // Thread1 は SignalTimerEvent()
            true,                       // Thread1 での API 返値
            true,                       // TimerEvent は Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

//---------------------------------------------------------------------------

// 単独の ClearTimerEvent() テスト

TEST(ClearTimerEvent, test_NotSignalToNotSignal1)
{
    // SignalTimerEvent 非シグナルから非シグナルへ
    doTest1(false,                      // TimerEvent は Not Signaled
            EVENT_AUTO_CLEAR,           // TimerEvent は自動クリアモード
            THR_CALL_CLEAR,             // Thread1 は ClearTimerEvent()
            true,                       // Thread1 での API 返値
            false,                      // TimerEvent は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(ClearTimerEvent, test_NotSignalToNotSignal2)
{
    // SignalTimerEvent 非シグナルから非シグナルへ
    doTest1(false,                      // TimerEvent は Not Signaled
            EVENT_MANUAL_CLEAR,         // TimerEvent は手動クリアモード
            THR_CALL_CLEAR,             // Thread1 は ClearTimerEvent()
            true,                       // Thread1 での API 返値
            false,                      // TimerEvent は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(ClearTimerEvent, test_SignalToNotSignal1)
{
    // SignalTimerEvent シグナルから非シグナルへ
    doTest1(true,                       // TimerEvent は Signaled
            EVENT_AUTO_CLEAR,           // TimerEvent は自動クリアモード
            THR_CALL_CLEAR,             // Thread1 は ClearTimerEvent()
            true,                       // Thread1 での API 返値
            false,                      // TimerEvent は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(ClearTimerEvent, test_SignalToNotSignal2)
{
    // SignalTimerEvent シグナルから非シグナルへ
    doTest1(true,                       // TimerEvent は Signaled
            EVENT_MANUAL_CLEAR,         // TimerEvent は手動クリアモード
            THR_CALL_CLEAR,             // Thread1 は ClearTimerEvent()
            true,                       // Thread1 での API 返値
            false,                      // TimerEvent は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

//---------------------------------------------------------------------------
// 単独の WaitTimerEvent() AutoClear テスト

TEST(WaitTimerEventAutoClear, test_WaitTimerEventAutoClear1)
{
    // AUTO_CLEAR/Signaled に対して WaitTimerEvent 発行
    doTest1(true,                       // TimerEvent は Signaled
            EVENT_AUTO_CLEAR,           // TimerEvent は自動クリアモード
            THR_CALL_WAIT,              // Thread1 は WaitTimerEvent()
            true,                       // Thread1 での API 返値
            false,                      // TimerEvent は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(WaitTimerEventAutoClear, test_WaitTimerEventAutoClear2)
{
    // AUTO_CLEAR/NotSignaled に対して WaitTimerEvent 発行
    doTest1(false,                      // TimerEvent は Not Signaled
            EVENT_AUTO_CLEAR,           // TimerEvent は自動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitTimerEvent()
            true,                       // Thread1 での API 返値
            false,                      // TimerEvent は Not Signaled
            THR_STATE_WAITING);         // Thread1 はシグナル待ち
}

TEST(WaitTimerEventAutoClear, test_WaitTimerEventAutoClear3)
{
    // AUTO_CLEAR/Signaled に対して WaitTimerEvent 発行×２スレッド
    doTest2(true,                       // TimerEvent は Signaled
            EVENT_AUTO_CLEAR,           // TimerEvent は自動クリアモード
            THR_CALL_WAIT,              // Thread1 は WaitTimerEvent()
            THR_WAIT_WAIT,              // Thread1 は WaitTimerEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            false,                      // TimerEvent は Not Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_WAITING);         // Thread2 はシグナル待ち
}

TEST(WaitTimerEventAutoClear, test_WaitTimerEventAutoClear4)
{
    // AUTO_CLEAR/NotSignaled に対して WaitTimerEvent 発行×２スレッド
    doTest2(false,                      // TimerEvent は Not Signaled
            EVENT_AUTO_CLEAR,           // TimerEvent は自動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitTimerEvent()
            THR_WAIT_WAIT,              // Thread1 は WaitTimerEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            false,                      // TimerEvent は Not Signaled
            THR_STATE_WAITING,          // Thread1 はシグナル待ち
            THR_STATE_WAITING);         // Thread2 はシグナル待ち
}

//---------------------------------------------------------------------------
// 単独の WaitTimerEvent() ManualClear テスト

TEST(WaitTimerEventManualClear, test_WaitTimerEventManualClear1)
{
    // MANUAL_CLEAR/Signaled に対して WaitTimerEvent 発行
    doTest1(true,                       // TimerEvent は Signaled
            EVENT_MANUAL_CLEAR,         // TimerEvent は手動クリアモード
            THR_CALL_WAIT,              // Thread1 は WaitTimerEvent()
            true,                       // Thread1 での API 返値
            true,                       // TimerEvent は Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(WaitTimerEventManualClear, test_WaitTimerEventManualClear2)
{
    // MANUAL_CLEAR/NotSignaled に対して WaitTimerEvent 発行
    doTest1(false,                      // TimerEvent は Not Signaled
            EVENT_MANUAL_CLEAR,         // TimerEvent は手動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitTimerEvent()
            true,                       // Thread1 での API 返値
            false,                      // TimerEvent は Not Signaled
            THR_STATE_WAITING);         // Thread1 はシグナル待ち
}

TEST(WaitTimerEventManualClear, test_WaitTimerEventManualClear3)
{
    // MANUAL_CLEAR/Signaled に対して WaitTimerEvent 発行×２スレッド
    doTest2(true,                       // TimerEvent は Signaled
            EVENT_MANUAL_CLEAR,         // TimerEvent は手動クリアモード
            THR_CALL_WAIT,              // Thread1 は WaitTimerEvent()
            THR_CALL_WAIT,              // Thread1 は WaitTimerEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            true,                       // TimerEvent は Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は終了
}

TEST(WaitTimerEventManualClear, test_WaitTimerEventManualClear4)
{
    // MANUAL_CLEAR/NotSignaled に対して WaitTimerEvent 発行×２スレッド
    doTest2(false,                      // TimerEvent は Not Signaled
            EVENT_MANUAL_CLEAR,         // TimerEvent は手動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitTimerEvent()
            THR_WAIT_WAIT,              // Thread1 は WaitTimerEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            false,                      // TimerEvent は Not Signaled
            THR_STATE_WAITING,          // Thread1 はシグナル待ち
            THR_STATE_WAITING);         // Thread2 はシグナル待ち
}

//---------------------------------------------------------------------------
// 単独の TryWaitTimerEvent() AutoClear テスト

TEST(TryWaitTimerEventAutoClear, test_TryWaitTimerEventAutoClear1)
{
    // AUTO_CLEAR/Signaled に対して TryWaitTimerEvent 発行
    doTest1(true,                       // TimerEvent は Signaled
            EVENT_AUTO_CLEAR,           // TimerEvent は自動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitTimerEvent()
            true,                       // Thread1 での API 返値
            false,                      // TimerEvent は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TryWaitTimerEventAutoClear, test_TryWaitTimerEventAutoClear2)
{
    // AUTO_CLEAR/NotSignaled に対して TryWaitTimerEvent 発行
    doTest1(false,                      // TimerEvent は Not Signaled
            EVENT_AUTO_CLEAR,           // TimerEvent は自動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitTimerEvent()
            false,                      // Thread1 での API 返値
            false,                      // TimerEvent は Not Signaled
            THR_STATE_EXITED);          // Thread1 はシグナル待ち
}

TEST(TryWaitTimerEventAutoClear, test_TryWaitTimerEventAutoClear3)
{
    // AUTO_CLEAR/Signaled に対して TryWaitTimerEvent 発行×２スレッド
    doTest2(true,                       // TimerEvent は Signaled
            EVENT_AUTO_CLEAR,           // TimerEvent は自動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitTimerEvent()
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitTimerEvent()
            true,                       // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            false,                      // TimerEvent は Not Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 はシグナル待ち
}

TEST(TryWaitTimerEventAutoClear, test_TryWaitTimerEventAutoClear4)
{
    // AUTO_CLEAR/NotSignaled に対して TryWaitTimerEvent 発行×２スレッド
    doTest2(false,                      // TimerEvent は Not Signaled
            EVENT_AUTO_CLEAR,           // TimerEvent は自動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitTimerEvent()
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitTimerEvent()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            false,                      // TimerEvent は Not Signaled
            THR_STATE_EXITED,           // Thread1 はシグナル待ち
            THR_STATE_EXITED);          // Thread2 はシグナル待ち
}

//---------------------------------------------------------------------------
// 単独の TryWaitTimerEvent() ManualClear テスト

TEST(TryWaitTimerEventManualClear, test_TryWaitTimerEventManualClear1)
{
    // MANUAL_CLEAR/Signaled に対して TryWaitTimerEvent 発行
    doTest1(true,                       // TimerEvent は Signaled
            EVENT_MANUAL_CLEAR,         // TimerEvent は手動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitTimerEvent()
            true,                       // Thread1 での API 返値
            true,                       // TimerEvent は Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TryWaitTimerEventManualClear, test_TryWaitTimerEventManualClear2)
{
    // MANUAL_CLEAR/NotSignaled に対して TryWaitTimerEvent 発行
    doTest1(false,                      // TimerEvent は Not Signaled
            EVENT_MANUAL_CLEAR,         // TimerEvent は手動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitTimerEvent()
            false,                      // Thread1 での API 返値
            false,                      // TimerEvent は Not Signaled
            THR_STATE_EXITED);          // Thread1 はシグナル待ち
}

TEST(TryWaitTimerEventManualClear, test_TryWaitTimerEventManualClear3)
{
    // MANUAL_CLEAR/Signaled に対して TryWaitTimerEvent 発行×２スレッド
    doTest2(true,                       // TimerEvent は Signaled
            EVENT_MANUAL_CLEAR,         // TimerEvent は手動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitTimerEvent()
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitTimerEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            true,                       // TimerEvent は Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は終了
}

TEST(TryWaitTimerEventManualClear, test_TryWaitTimerEventManualClear4)
{
    // MANUAL_CLEAR/NotSignaled に対して TryWaitTimerEvent 発行×２スレッド
    doTest2(false,                      // TimerEvent は Not Signaled
            EVENT_MANUAL_CLEAR,         // TimerEvent は手動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitTimerEvent()
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitTimerEvent()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            false,                      // TimerEvent は Not Signaled
            THR_STATE_EXITED,           // Thread1 はシグナル待ち
            THR_STATE_EXITED);          // Thread2 はシグナル待ち
}

//---------------------------------------------------------------------------
// AUTO_CLEAR/Wait 待機中に Signal する

TEST(CombinedTimerEventTestAutoClear, test_WaitAndSignalAutoClear)
{
    // Thread1 で WaitTimerEvent()、Thread2 で SignalTimerEvent()
    doTest2(false,                      // TimerEvent は Not Signaled
            EVENT_AUTO_CLEAR,           // TimerEvent は自動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitTimerEvent()
            THR_CALL_SIGNAL,            // Thread2 は SignalTimerEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            false,                      // TimerEvent は Not Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

//---------------------------------------------------------------------------
// MANUAL_CLEAR/Wait 待機中に Signal する

TEST(CombinedTimerEventTestManualClear, test_WaitAndSignalManualClear)
{
    // Thread1 で WaitTimerEvent()、Thread2 で SignalTimerEvent()
    doTest2(false,                      // TimerEvent は Not Signaled
            EVENT_MANUAL_CLEAR,         // TimerEvent は手動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitTimerEvent()
            THR_CALL_SIGNAL,            // Thread2 は SignalTimerEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            true,                       // TimerEvent は Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

//---------------------------------------------------------------------------
//  TimerEvent(AutoClear) のワンショットタイマー機能のテスト
//  StartOneShotTimerEvent() によるタイマーの基本動作と、
//  タイマー動作中の Start, Stop, Signal, Clear 操作のテスト。
//
TEST(StartOneShotTimerEvent, test_StartOneShotTimerEventTestAutoClear)
{
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan  time1msec   = nn::TimeSpan::FromMilliSeconds(1);
        const nn::TimeSpan  time50msec  = nn::TimeSpan::FromMilliSeconds(50);
        const nn::TimeSpan  time100msec = nn::TimeSpan::FromMilliSeconds(100);

        SEQ_SET(3000);
        NNT_OS_LOG(NN_TEXT("TimerEvent（AutoClear）のワンショットタイマーテスト\n"));
        nn::os::TimerEventType event;
        nn::os::InitializeTimerEvent(&event, nn::os::EventClearMode_AutoClear);

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event の初期状態が false"));
        CheckBool( false == nn::os::TryWaitTimerEvent(&event) );

        // StartOneShot( 1msec ) のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(1msec) でシグナル化する"));
        nn::os::StartOneShotTimerEvent(&event, time1msec);
        nn::os::WaitTimerEvent(&event);
        CheckBool( true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ワンショット待機後の event が false"));
        CheckBool( false == nn::os::TryWaitTimerEvent(&event) );

        // StartOneShot( 100msec ) 中に Signal するテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(100msec) 中に Signal でシグナル化可能か"));
        nn::os::StartOneShotTimerEvent(&event, time100msec);
        nn::os::SignalTimerEvent(&event);
        CheckBool( true == nn::os::TryWaitTimerEvent(&event) ); // クリアされる
        nn::os::WaitTimerEvent(&event); // StartOneShotTimeEvent に対する Wait

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ワンショット待機後の event が false"));
        CheckBool( false == nn::os::TryWaitTimerEvent(&event) );

        // StartOneShot( 100msec ) 中に Clear するテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(100msec) 中に Clear しても影響がないか"));
        nn::os::StartOneShotTimerEvent(&event, time100msec);
        nn::os::ClearTimerEvent(&event);
        CheckBool( false == nn::os::TryWaitTimerEvent(&event) );
        nn::os::WaitTimerEvent(&event); // StartOneShotTimeEvent に対する Wait

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ワンショット待機後の event が false"));
        CheckBool( false == nn::os::TryWaitTimerEvent(&event) );

        // StartOneShot( 100msec ) 中に StartOneShot( 100msec ) するテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(100msec) 中に StartOneShot(100msec) 可能か"));
        nn::os::StartOneShotTimerEvent(&event, time100msec);
        EXPECT_FALSE( nn::os::TryWaitTimerEvent(&event) );
        nn::os::StartOneShotTimerEvent(&event, time100msec);
        nn::os::WaitTimerEvent(&event); // 後発の StartOneShot に対する Wait
        CheckBool( true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ワンショット待機後の event が false"));
        CheckBool( false == nn::os::TryWaitTimerEvent(&event) );

        // StartOneShot( 100msec ) 中に StartPeriodic() するテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(100msec) 中に StartPeriodic(50msec,50msec) 可能か"));
        nn::os::StartOneShotTimerEvent(&event, time100msec);
        EXPECT_FALSE( nn::os::TryWaitTimerEvent(&event) );
        nn::os::StartPeriodicTimerEvent(&event, time50msec, time50msec);
        nn::os::WaitTimerEvent(&event);     // Periodic の１発目
        nn::os::WaitTimerEvent(&event);     // Periodic の２発目
        nn::os::WaitTimerEvent(&event);     // Periodic の３発目
        nn::os::StopTimerEvent(&event);     // タイマー停止
        nn::os::ClearTimerEvent(&event);    // クリア
        CheckBool( true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ワンショット待機後の event が false"));
        CheckBool( false == nn::os::TryWaitTimerEvent(&event) );

        // StartOneShot( 50msec ) 中に Stop() するテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(50msec) 中に StopOneShot() 可能か"));
        nn::os::StartOneShotTimerEvent(&event, time50msec);
        EXPECT_FALSE( nn::os::TryWaitTimerEvent(&event) );
        nn::os::StopTimerEvent(&event);
        nn::os::SleepThread(time100msec);
        CheckBool( false == nn::os::TryWaitTimerEvent(&event) );

        // Finalize
        nn::os::FinalizeTimerEvent(&event);
    }

    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  TimerEvent(ManualClear) のワンショットタイマー機能のテスト
//  StartOneShotTimerEvent() によるタイマーの基本動作と、
//  タイマー動作中の Start, Stop, Signal, Clear 操作のテスト。
//
TEST(StartOneShotTimerEvent, test_StartOneShotTimerEventTestManualClear)
{
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan  time1msec   = nn::TimeSpan::FromMilliSeconds(1);
        const nn::TimeSpan  time50msec  = nn::TimeSpan::FromMilliSeconds(50);
        const nn::TimeSpan  time100msec = nn::TimeSpan::FromMilliSeconds(100);

        SEQ_SET(3100);
        NNT_OS_LOG(NN_TEXT("TimerEvent（ManualClear）のワンショットタイマーテスト\n"));
        nn::os::TimerEventType event;
        nn::os::InitializeTimerEvent(&event, nn::os::EventClearMode_ManualClear);

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event の初期状態が false"));
        CheckBool( false == nn::os::TryWaitTimerEvent(&event) );

        // StartOneShot( 1msec ) のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(1msec) でシグナル化する"));
        nn::os::StartOneShotTimerEvent(&event, time1msec);
        nn::os::WaitTimerEvent(&event);
        CheckBool( true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ワンショット待機後の event が true"));
        CheckBool( true == nn::os::TryWaitTimerEvent(&event) );
        nn::os::ClearTimerEvent(&event);

        // StartOneShot( 50msec ) 中に Signal するテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(50msec) 中に Signal でシグナル化可能か"));
        nn::os::StartOneShotTimerEvent(&event, time50msec);
        nn::os::SignalTimerEvent(&event);
        CheckBool( true == nn::os::TryWaitTimerEvent(&event) );
        nn::os::SleepThread(time100msec);
        nn::os::WaitTimerEvent(&event); // StartOneShotTimeEvent に対する Wait

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ワンショット待機後の event が true"));
        CheckBool( true == nn::os::TryWaitTimerEvent(&event) );
        nn::os::ClearTimerEvent(&event);

        // StartOneShot( 100msec ) 中に Clear するテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(100msec) 中に Clear しても影響がないか"));
        nn::os::StartOneShotTimerEvent(&event, time100msec);
        nn::os::ClearTimerEvent(&event);
        CheckBool( false == nn::os::TryWaitTimerEvent(&event) );
        nn::os::WaitTimerEvent(&event); // StartOneShotTimeEvent に対する Wait

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ワンショット待機後の event が true"));
        CheckBool( true == nn::os::TryWaitTimerEvent(&event) );
        nn::os::ClearTimerEvent(&event);

        // StartOneShot( 100msec ) 中に StartOneShot( 100msec ) するテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(100msec) 中に StartOneShot(100msec) 可能か"));
        nn::os::StartOneShotTimerEvent(&event, time100msec);
        EXPECT_FALSE( nn::os::TryWaitTimerEvent(&event) );
        nn::os::StartOneShotTimerEvent(&event, time100msec);
        nn::os::WaitTimerEvent(&event); // 後発の StartOneShot に対する Wait
        CheckBool( true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ワンショット待機後の event が true"));
        CheckBool( true == nn::os::TryWaitTimerEvent(&event) );
        nn::os::ClearTimerEvent(&event);

        // StartOneShot( 100msec ) 中に StartPeriodic() するテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(100msec) 中に StartPeriodic(50msec,50msec) 可能か"));
        nn::os::StartOneShotTimerEvent(&event, time100msec);
        EXPECT_FALSE( nn::os::TryWaitTimerEvent(&event) );
        nn::os::StartPeriodicTimerEvent(&event, time50msec, time50msec);
        for (int i=0; i<4; ++i)
        {
            nn::os::WaitTimerEvent(&event);
            nn::os::ClearTimerEvent(&event);
            EXPECT_FALSE( nn::os::TryWaitTimerEvent(&event) );
        }
        nn::os::StopTimerEvent(&event);     // タイマー停止
        nn::os::ClearTimerEvent(&event);    // クリア
        CheckBool( true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ワンショット待機後の event が false"));
        CheckBool( false == nn::os::TryWaitTimerEvent(&event) );

        // StartOneShot( 50msec ) 中に Stop() するテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(50msec) 中に StopOneShot() 可能か"));
        nn::os::StartOneShotTimerEvent(&event, time50msec);
        EXPECT_FALSE( nn::os::TryWaitTimerEvent(&event) );
        nn::os::StopTimerEvent(&event);
        nn::os::SleepThread(time100msec);
        CheckBool( false == nn::os::TryWaitTimerEvent(&event) );

        // Finalize
        nn::os::FinalizeTimerEvent(&event);
    }

    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  TimerEvent(AutoClear) の周期タイマー機能のテスト
//  StartPeriodicTimerEvent() によるタイマーの基本動作と、
//  タイマー動作中の Start, Stop, Signal, Clear 操作のテスト。
//
TEST(StartPeriodicTimerEvent, test_StartPeriodicTimerEventTestAutoClear)
{
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan  time1msec   = nn::TimeSpan::FromMilliSeconds(1);
        const nn::TimeSpan  time10msec  = nn::TimeSpan::FromMilliSeconds(10);
        const nn::TimeSpan  time20msec  = nn::TimeSpan::FromMilliSeconds(20);
        const nn::TimeSpan  time50msec  = nn::TimeSpan::FromMilliSeconds(50);
        const nn::TimeSpan  time100msec = nn::TimeSpan::FromMilliSeconds(100);
        const nn::TimeSpan  time1sec    = nn::TimeSpan::FromSeconds(1);
        const nn::TimeSpan  time10sec   = nn::TimeSpan::FromSeconds(10);

        SEQ_SET(3200);
        NNT_OS_LOG(NN_TEXT("TimerEvent（AutoClear）の周期タイマーテスト\n"));
        nn::os::TimerEventType event;
        nn::os::InitializeTimerEvent(&event, nn::os::EventClearMode_AutoClear);

        SEQ_CHECK(3201);
        NNT_OS_LOG(NN_TEXT("event の初期状態が false"));
        CheckBool( false == nn::os::TryWaitTimerEvent(&event) );

        // StartPeriodic( 10msec, 20msec ) のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartPeriodic(10msec, 20msec) でシグナル化する\n"));
        nn::os::Tick startTick = nn::os::GetSystemTick();
        nn::os::StartPeriodicTimerEvent(&event, time10msec, time20msec);
        for (int i=0; i<5; ++i)
        {
            nn::os::WaitTimerEvent(&event);
            EXPECT_FALSE( nn::os::TryWaitTimerEvent(&event) );
            nn::os::Tick currTick = nn::os::GetSystemTick();
            SEQ_NONE();
            NNT_OS_LOG("Signaled interval = %6d (usec)\n", (currTick - startTick).ToTimeSpan().GetMicroSeconds());
            startTick = currTick;
        }
        nn::os::StopTimerEvent(&event);     // タイマー停止
        nn::os::ClearTimerEvent(&event);    // クリア

        // StartPeriodic( 10msec, 20msec ) 中に Signal/Clear するテスト
        SEQ_CHECK(3202);
        NNT_OS_LOG(NN_TEXT("StartPeriodic(10msec, 20msec) 中に Signal/Clear 可能か\n"));
        startTick = nn::os::GetSystemTick();
        nn::os::StartPeriodicTimerEvent(&event, time10msec, time20msec);
        for (int i=0; i<5; ++i)
        {
            nn::os::SignalTimerEvent(&event);
            EXPECT_TRUE( nn::os::TryWaitTimerEvent(&event) );
            nn::os::ClearTimerEvent(&event);
            EXPECT_FALSE( nn::os::TryWaitTimerEvent(&event) );

            nn::os::WaitTimerEvent(&event);
            EXPECT_FALSE( nn::os::TryWaitTimerEvent(&event) );

            nn::os::Tick currTick = nn::os::GetSystemTick();
            SEQ_NONE();
            NNT_OS_LOG("Signaled interval = %6d (usec)\n", (currTick - startTick).ToTimeSpan().GetMicroSeconds());
            startTick = currTick;
        }
        nn::os::StopTimerEvent(&event);     // タイマー停止
        nn::os::ClearTimerEvent(&event);    // クリア

        // StartPeriodic(10sec, 1sec) 中に StartPeriodic(20ms, 10ms) するテスト
        SEQ_CHECK(3203);
        NNT_OS_LOG(NN_TEXT("StartPeriodic(10s,1s) 中に StartPeriodic(20ms,10ms) 可能か"));
        nn::os::StartPeriodicTimerEvent(&event, time10sec, time1sec);
        EXPECT_FALSE( nn::os::TryWaitTimerEvent(&event) );

        startTick = nn::os::GetSystemTick();
        nn::os::StartPeriodicTimerEvent(&event, time20msec, time10msec);
        CheckBool( true );

        for (int i=0; i<5; ++i)
        {
            nn::os::WaitTimerEvent(&event);
            EXPECT_FALSE( nn::os::TryWaitTimerEvent(&event) );
            nn::os::Tick currTick = nn::os::GetSystemTick();
            SEQ_NONE();
            NNT_OS_LOG("Signaled interval = %6d (usec)\n", (currTick - startTick).ToTimeSpan().GetMicroSeconds());
            startTick = currTick;
        }
        nn::os::StopTimerEvent(&event);     // タイマー停止
        nn::os::ClearTimerEvent(&event);    // クリア

        // StartPeriodic( 100msec, 1msec ) 中に StartOneShot(50msec) するテスト
        SEQ_CHECK(3204);
        NNT_OS_LOG(NN_TEXT("StartPeriodic(100ms, 1ms) 中に StartOneShot(50msec) 可能か"));
        nn::os::StartPeriodicTimerEvent(&event, time100msec, time1msec);
        EXPECT_FALSE( nn::os::TryWaitTimerEvent(&event) );
        nn::os::StartOneShotTimerEvent(&event, time50msec);
        nn::os::WaitTimerEvent(&event);     // OneShot の１発目
        CheckBool( true );

        nn::os::SleepThread( time100msec );
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ワンショット後 100msec 経っても false か"));
        CheckBool( false == nn::os::TryWaitTimerEvent(&event) );

        nn::os::StopTimerEvent(&event);     // タイマー停止
        nn::os::ClearTimerEvent(&event);    // クリア

        // StartPeriodic( 50msec, 10msec ) 中に Stop() するテスト
        SEQ_CHECK(3205);
        NNT_OS_LOG(NN_TEXT("StartPeriodic(50msec,10msec) 中に StopPeriodic() 可能か"));
        nn::os::StartPeriodicTimerEvent(&event, time50msec, time10msec);
        EXPECT_FALSE( nn::os::TryWaitTimerEvent(&event) );
        nn::os::StopTimerEvent(&event);
        nn::os::SleepThread(time100msec);
        CheckBool( false == nn::os::TryWaitTimerEvent(&event) );

        // Finalize
        nn::os::FinalizeTimerEvent(&event);
    }

    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  TimerEvent(ManualClear) の周期タイマー機能のテスト
//  StartPeriodicTimerEvent() によるタイマーの基本動作と、
//  タイマー動作中の Start, Stop, Signal, Clear 操作のテスト。
//
TEST(StartPeriodicTimerEvent, test_StartPeriodicTimerEventTestManualClear)
{
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan  time1msec   = nn::TimeSpan::FromMilliSeconds(1);
        const nn::TimeSpan  time10msec  = nn::TimeSpan::FromMilliSeconds(10);
        const nn::TimeSpan  time20msec  = nn::TimeSpan::FromMilliSeconds(20);
        const nn::TimeSpan  time50msec  = nn::TimeSpan::FromMilliSeconds(50);
        const nn::TimeSpan  time100msec = nn::TimeSpan::FromMilliSeconds(100);
        const nn::TimeSpan  time1sec    = nn::TimeSpan::FromSeconds(1);
        const nn::TimeSpan  time10sec   = nn::TimeSpan::FromSeconds(10);

        SEQ_SET(3300);
        NNT_OS_LOG(NN_TEXT("TimerEvent（ManualClear）の周期タイマーテスト\n"));
        nn::os::TimerEventType event;
        nn::os::InitializeTimerEvent(&event, nn::os::EventClearMode_ManualClear);

        SEQ_CHECK(3301);
        NNT_OS_LOG(NN_TEXT("event の初期状態が false"));
        CheckBool( false == nn::os::TryWaitTimerEvent(&event) );

        // StartPeriodic( 10msec, 20msec ) のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartPeriodic(10msec, 20msec) でシグナル化する\n"));
        nn::os::Tick startTick = nn::os::GetSystemTick();
        nn::os::StartPeriodicTimerEvent(&event, time10msec, time20msec);
        for (int i=0; i<5; ++i)
        {
            nn::os::WaitTimerEvent(&event);
            EXPECT_TRUE( nn::os::TryWaitTimerEvent(&event) );
            // あえて TimerEvent をクリアしない

            nn::os::Tick currTick = nn::os::GetSystemTick();
            SEQ_NONE();
            NNT_OS_LOG("Signaled interval = %6d (usec)\n", (currTick - startTick).ToTimeSpan().GetMicroSeconds());
            startTick = currTick;
        }
        nn::os::StopTimerEvent(&event);     // タイマー停止
        nn::os::ClearTimerEvent(&event);    // クリア

        // StartPeriodic( 10msec, 20msec ) 中に Signal/Clear するテスト
        SEQ_CHECK(3302);
        NNT_OS_LOG(NN_TEXT("StartPeriodic(10msec, 20msec) 中に Signal/Clear 可能か\n"));
        startTick = nn::os::GetSystemTick();
        nn::os::StartPeriodicTimerEvent(&event, time10msec, time20msec);
        for (int i=0; i<5; ++i)
        {
            nn::os::SignalTimerEvent(&event);
            EXPECT_TRUE( nn::os::TryWaitTimerEvent(&event) );
            nn::os::ClearTimerEvent(&event);
            EXPECT_FALSE( nn::os::TryWaitTimerEvent(&event) );

            nn::os::WaitTimerEvent(&event);
            EXPECT_TRUE( nn::os::TryWaitTimerEvent(&event) );

            nn::os::Tick currTick = nn::os::GetSystemTick();
            SEQ_NONE();
            NNT_OS_LOG("Signaled interval = %6d (usec)\n", (currTick - startTick).ToTimeSpan().GetMicroSeconds());
            startTick = currTick;
        }
        nn::os::StopTimerEvent(&event);     // タイマー停止
        nn::os::ClearTimerEvent(&event);    // クリア

        // StartPeriodic(10sec, 1sec) 中に StartPeriodic(20ms, 10ms) するテスト
        SEQ_CHECK(3303);
        NNT_OS_LOG(NN_TEXT("StartPeriodic(10s,1s) 中に StartPeriodic(20ms,10ms) 可能か"));
        nn::os::StartPeriodicTimerEvent(&event, time10sec, time1sec);
        EXPECT_FALSE( nn::os::TryWaitTimerEvent(&event) );

        startTick = nn::os::GetSystemTick();
        nn::os::StartPeriodicTimerEvent(&event, time20msec, time10msec);
        CheckBool( true );

        for (int i=0; i<5; ++i)
        {
            nn::os::WaitTimerEvent(&event);
            EXPECT_TRUE( nn::os::TryWaitTimerEvent(&event) );
            // あえて TimerEvent をクリアしない

            nn::os::Tick currTick = nn::os::GetSystemTick();
            SEQ_NONE();
            NNT_OS_LOG("Signaled interval = %6d (usec)\n", (currTick - startTick).ToTimeSpan().GetMicroSeconds());
            startTick = currTick;
        }
        nn::os::StopTimerEvent(&event);     // タイマー停止
        nn::os::ClearTimerEvent(&event);    // クリア

        // StartPeriodic( 100msec, 1msec ) 中に StartOneShot(50msec) するテスト
        SEQ_CHECK(3304);
        NNT_OS_LOG(NN_TEXT("StartPeriodic(100ms, 1ms) 中に StartOneShot(50msec) 可能か"));
        nn::os::StartPeriodicTimerEvent(&event, time100msec, time1msec);
        EXPECT_FALSE( nn::os::TryWaitTimerEvent(&event) );
        nn::os::StartOneShotTimerEvent(&event, time50msec);
        nn::os::WaitTimerEvent(&event);     // OneShot の１発目
        CheckBool( true );

        nn::os::SleepThread( time100msec );
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ワンショット後 100msec 経っても true のままか"));
        CheckBool( true == nn::os::TryWaitTimerEvent(&event) );

        nn::os::StopTimerEvent(&event);     // タイマー停止
        nn::os::ClearTimerEvent(&event);    // クリア

        // StartPeriodic( 50msec, 10msec ) 中に Stop() するテスト
        SEQ_CHECK(3305);
        NNT_OS_LOG(NN_TEXT("StartPeriodic(50msec,10msec) 中に StopPeriodic() 可能か"));
        nn::os::StartPeriodicTimerEvent(&event, time50msec, time10msec);
        EXPECT_FALSE( nn::os::TryWaitTimerEvent(&event) );
        nn::os::StopTimerEvent(&event);
        nn::os::SleepThread(time100msec);
        CheckBool( false == nn::os::TryWaitTimerEvent(&event) );

        nn::os::StopTimerEvent(&event);     // タイマー停止
        nn::os::ClearTimerEvent(&event);    // クリア

        // Finalize
        nn::os::FinalizeTimerEvent(&event);
    }

    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  TimerEvent(AutoClear) のタイマー待機のテスト
//  WaitTimerEvent() 中に他スレッドからタイマー動作を変更
//  タイマー待機中の Start, Stop, Signal, Clear 操作のテスト。
//
NN_ALIGNAS(4096) char   g_Stack[8192];
nn::os::BarrierType     g_Barrier;

void test_WaitTimerEventTestHelperThread(void* arg)
{
    nn::os::TimerEventType* event = static_cast<nn::os::TimerEventType*>(arg);

    const nn::TimeSpan  time10msec  = nn::TimeSpan::FromMilliSeconds(10);
    const nn::TimeSpan  time20msec  = nn::TimeSpan::FromMilliSeconds(20);
    const nn::TimeSpan  time50msec  = nn::TimeSpan::FromMilliSeconds(50);
    const nn::TimeSpan  time100msec = nn::TimeSpan::FromMilliSeconds(100);

    // step1: 20msec 後に StartOneShot( 50msec ) を発行
    nn::os::SleepThread( time20msec );
    nn::os::StartOneShotTimerEvent(event, time50msec);

    nn::os::AwaitBarrier(&g_Barrier);

    // step2: 20msec 後に StartOneShot( 50msec, 10msec ) を発行
    nn::os::SleepThread( time20msec );
    nn::os::StartPeriodicTimerEvent(event, time50msec, time10msec);

    nn::os::AwaitBarrier(&g_Barrier);

    // step3: StartPeriodic(IN64_MAX, 1nsec) で待機しているスレッドを起こす
    nn::os::SleepThread( time100msec );
    nn::os::SignalTimerEvent(event);

    nn::os::AwaitBarrier(&g_Barrier);

    // step4: StartPeriodic(0, IN64_MAX) で待機しているスレッドを起こす
    nn::os::SleepThread( time100msec );
    nn::os::SignalTimerEvent(event);

    nn::os::AwaitBarrier(&g_Barrier);

    // step5: StartOneShot(IN64_MAX) で待機しているスレッドを起こす
    nn::os::SleepThread( time100msec );
    nn::os::SignalTimerEvent(event);

    nn::os::AwaitBarrier(&g_Barrier);
}

TEST(WaitTimerEvent, test_WaitTimerEventTestAutoClear)
{
    CLEAR_GOOGLE_TEST();

    nn::os::InitializeBarrier(&g_Barrier, 2);

    {
        const nn::TimeSpan  time1nsec   = nn::TimeSpan::FromNanoSeconds(1);
        const nn::TimeSpan  timeMax     = nn::TimeSpan::FromNanoSeconds(INT64_MAX);

        SEQ_SET(3400);
        NNT_OS_LOG(NN_TEXT("TimerEvent（AutoClear）の周期タイマーテスト\n"));
        nn::os::TimerEventType event;
        nn::os::InitializeTimerEvent(&event, nn::os::EventClearMode_AutoClear);

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("子スレッドを生成する"));
        nn::os::ThreadType  thread;
        int pri = nn::os::GetThreadPriority(nn::os::GetCurrentThread());
        nn::os::CreateThread(&thread, test_WaitTimerEventTestHelperThread, &event, g_Stack, sizeof(g_Stack), pri - 1);
        nn::os::StartThread(&thread);
        CheckBool(true);

        // step1: WaitTimerEvent() 中に他スレッドから StartOneShot()
        SEQ_CHECK(3401);
        NNT_OS_LOG(NN_TEXT("WaitTimerEvent() 中に他スレッドから StartOneShot()"));

        nn::os::Tick startTick = nn::os::GetSystemTick();
        nn::os::WaitTimerEvent(&event);
        CheckBool( false == nn::os::TryWaitTimerEvent(&event) );    // 自動クリア

        nn::os::Tick currTick = nn::os::GetSystemTick();
        SEQ_NONE();
        NNT_OS_LOG("Signaled interval = %6d (usec)\n", (currTick - startTick).ToTimeSpan().GetMicroSeconds());
        startTick = currTick;


        nn::os::AwaitBarrier(&g_Barrier);

        // step2: WaitTimerEvent() 中に他スレッドから StartPeriodic()
        SEQ_CHECK(3402);
        NNT_OS_LOG(NN_TEXT("WaitTimerEvent() 中に他スレッドから StartPeriodic()"));

        startTick = nn::os::GetSystemTick();
        nn::os::WaitTimerEvent(&event);
        CheckBool( false == nn::os::TryWaitTimerEvent(&event) );    // 自動クリア

        currTick = nn::os::GetSystemTick();
        SEQ_NONE();
        NNT_OS_LOG("Signaled interval = %6d (usec)\n", (currTick - startTick).ToTimeSpan().GetMicroSeconds());
        startTick = currTick;

        for (int i=0; i<5; ++i)
        {
            nn::os::WaitTimerEvent(&event);
            EXPECT_FALSE( nn::os::TryWaitTimerEvent(&event) );

            currTick = nn::os::GetSystemTick();
            SEQ_NONE();
            NNT_OS_LOG("Signaled interval = %6d (usec)\n", (currTick - startTick).ToTimeSpan().GetMicroSeconds());
            startTick = currTick;
        }

        nn::os::AwaitBarrier(&g_Barrier);

        // step3: StartPeriodic(IN64_MAX, 1nsec) で待機し、子スレッドから起床
        SEQ_CHECK(3403);
        NNT_OS_LOG(NN_TEXT("StartPeriodic(INT64_MAX, 1nsec) で待機し、子スレッドから起床"));
        nn::os::StartPeriodicTimerEvent( &event, timeMax, time1nsec );
        nn::os::WaitTimerEvent(&event);
        CheckBool( true );

        nn::os::AwaitBarrier(&g_Barrier);

        // step4: StartPeriodic(0, IN64_MAX) で待機し、子スレッドから起床
        SEQ_CHECK(3404);
        NNT_OS_LOG(NN_TEXT("StartPeriodic(0, INT64_MAX) で待機し、子スレッドから起床"));
        nn::os::StartPeriodicTimerEvent( &event, 0, timeMax );
        nn::os::WaitTimerEvent(&event);
        CheckBool( true );

        nn::os::AwaitBarrier(&g_Barrier);

        // step5: StartOneShot(IN64_MAX) で待機し、子スレッドから起床
        SEQ_CHECK(3405);
        NNT_OS_LOG(NN_TEXT("StartOneShot(INT64_MAX) で待機し、子スレッドから起床"));
        nn::os::StartOneShotTimerEvent( &event, timeMax );
        nn::os::WaitTimerEvent(&event);
        CheckBool( true );

        nn::os::AwaitBarrier(&g_Barrier);

        // step6: StartOneShot(0) で待機するもののすぐに返ってくる
        SEQ_CHECK(3406);
        NNT_OS_LOG(NN_TEXT("StartOneShot(0) で待機するもののすぐに返ってくる"));
        nn::os::StartOneShotTimerEvent( &event, 0 );
        nn::os::WaitTimerEvent(&event);
        CheckBool( true );

        // Finalize
        nn::os::WaitThread(&thread);
        nn::os::DestroyThread(&thread);

        nn::os::FinalizeTimerEvent(&event);
    }

    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  TimerEvent(AutoClear) のワンショットタイマー機能の多重待ちテスト
//  StartOneShotTimerEvent() によるタイマーの基本動作と、
//  タイマー動作中の Start, Stop, Signal, Clear 操作のテスト。
//
TEST(StartOneShotTimerEventMultiWait, test_StartOneShotTimerEventTestAutoClear)
{
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan  time1msec   = nn::TimeSpan::FromMilliSeconds(1);
        const nn::TimeSpan  time50msec  = nn::TimeSpan::FromMilliSeconds(50);
        const nn::TimeSpan  time100msec = nn::TimeSpan::FromMilliSeconds(100);

        SEQ_SET(3500);
        NNT_OS_LOG(NN_TEXT("TimerEvent（AutoClear）のワンショットタイマー多重待ちテスト\n"));
        nn::os::TimerEventType event;
        nn::os::InitializeTimerEvent(&event, nn::os::EventClearMode_AutoClear);

        // 多重待ちオブジェクトの準備 ----------
        nn::os::MultiWaitType       multiWait;
        nn::os::MultiWaitHolderType multiWaitHolder;

        nn::os::InitializeMultiWait(&multiWait);
        nn::os::InitializeMultiWaitHolder(&multiWaitHolder, &event);
        nn::os::LinkMultiWaitHolder(&multiWait, &multiWaitHolder);

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event の初期状態が false"));
        CheckBool( NULL == nn::os::TryWaitAny(&multiWait) );

        // StartOneShot( 1msec ) のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(1msec) でシグナル化する"));
        nn::os::StartOneShotTimerEvent(&event, time1msec);
        nn::os::WaitAny(&multiWait);
        CheckBool( true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ワンショット多重待ち後の event は true のまま"));
        CheckBool( &multiWaitHolder == nn::os::TryWaitAny(&multiWait) );
        nn::os::ClearTimerEvent(&event);

        // StartOneShot( 100msec ) 中に Signal するテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(100msec) 中に Signal でシグナル化可能か"));
        nn::os::StartOneShotTimerEvent(&event, time100msec);
        nn::os::SignalTimerEvent(&event);
        CheckBool( &multiWaitHolder == nn::os::TryWaitAny(&multiWait) );
        nn::os::ClearTimerEvent(&event);
        nn::os::WaitAny(&multiWait); // StartOneShotTimeEvent に対する WaitAny

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ワンショット多重待ち後の event が true のまま"));
        CheckBool( &multiWaitHolder == nn::os::TryWaitAny(&multiWait) );
        nn::os::ClearTimerEvent(&event);

        // StartOneShot( 100msec ) 中に Clear するテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(100msec) 中に Clear しても影響がないか"));
        nn::os::StartOneShotTimerEvent(&event, time100msec);
        nn::os::ClearTimerEvent(&event);
        CheckBool( NULL == nn::os::TryWaitAny(&multiWait) );
        nn::os::WaitAny(&multiWait); // StartOneShotTimeEvent に対する WaitAny

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ワンショット多重待ち後の event が true のまま"));
        CheckBool( &multiWaitHolder == nn::os::TryWaitAny(&multiWait) );
        nn::os::ClearTimerEvent(&event);

        // StartOneShot( 100msec ) 中に StartOneShot( 100msec ) するテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(100msec) 中に StartOneShot(100msec) 可能か"));

        nn::os::StartOneShotTimerEvent(&event, time100msec);
        EXPECT_TRUE( NULL == nn::os::TryWaitAny(&multiWait) );

        nn::os::StartOneShotTimerEvent(&event, time100msec);
        // 後発の StartOneShot に対する WaitAny
        CheckBool( &multiWaitHolder == nn::os::WaitAny(&multiWait) );
        nn::os::ClearTimerEvent(&event);

        // StartOneShot( 100msec ) 中に StartPeriodic() するテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(100msec) 中に StartPeriodic(50msec,50msec) 可能か"));
        nn::os::StartOneShotTimerEvent(&event, time100msec);
        EXPECT_TRUE( NULL == nn::os::TryWaitAny(&multiWait) );
        nn::os::StartPeriodicTimerEvent(&event, time50msec, time50msec);

        // Periodic の１発目
        EXPECT_TRUE( &multiWaitHolder == nn::os::WaitAny(&multiWait) );
        nn::os::ClearTimerEvent(&event);

        // Periodic の２発目
        EXPECT_TRUE( &multiWaitHolder == nn::os::WaitAny(&multiWait) );
        nn::os::ClearTimerEvent(&event);

        // Periodic の３発目
        EXPECT_TRUE( &multiWaitHolder == nn::os::WaitAny(&multiWait) );
        nn::os::StopTimerEvent(&event);     // タイマー停止
        nn::os::ClearTimerEvent(&event);    // クリア
        CheckBool( true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ワンショット待機後の event が false"));
        CheckBool( NULL == nn::os::TryWaitAny(&multiWait) );

        // StartOneShot( 50msec ) 中に Stop() するテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(50msec) 中に StopOneShot() 可能か"));
        nn::os::StartOneShotTimerEvent(&event, time50msec);
        EXPECT_TRUE( NULL == nn::os::TryWaitAny(&multiWait) );
        nn::os::StopTimerEvent(&event);
        nn::os::SleepThread(time100msec);
        CheckBool( NULL == nn::os::TryWaitAny(&multiWait) );

        // Finalize
        nn::os::UnlinkMultiWaitHolder(&multiWaitHolder);
        nn::os::FinalizeMultiWaitHolder(&multiWaitHolder);
        nn::os::FinalizeMultiWait(&multiWait);
        nn::os::FinalizeTimerEvent(&event);
    }

    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  TimerEvent(ManualClear) のワンショットタイマー機能の多重待ちテスト
//  StartOneShotTimerEvent() によるタイマーの基本動作と、
//  タイマー動作中の Start, Stop, Signal, Clear 操作のテスト。
//
TEST(StartOneShotTimerEventMultiWait, test_StartOneShotTimerEventTestManualClear)
{
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan  time1msec   = nn::TimeSpan::FromMilliSeconds(1);
        const nn::TimeSpan  time50msec  = nn::TimeSpan::FromMilliSeconds(50);
        const nn::TimeSpan  time100msec = nn::TimeSpan::FromMilliSeconds(100);

        SEQ_SET(3600);
        NNT_OS_LOG(NN_TEXT("TimerEvent（ManualClear）のワンショットタイマー多重待ちテスト\n"));
        nn::os::TimerEventType event;
        nn::os::InitializeTimerEvent(&event, nn::os::EventClearMode_ManualClear);

        // 多重待ちオブジェクトの準備 ----------
        nn::os::MultiWaitType       multiWait;
        nn::os::MultiWaitHolderType multiWaitHolder;

        nn::os::InitializeMultiWait(&multiWait);
        nn::os::InitializeMultiWaitHolder(&multiWaitHolder, &event);
        nn::os::LinkMultiWaitHolder(&multiWait, &multiWaitHolder);

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event の初期状態が false"));
        CheckBool( NULL == nn::os::TryWaitAny(&multiWait) );

        // StartOneShot( 1msec ) のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(1msec) でシグナル化する"));
        nn::os::StartOneShotTimerEvent(&event, time1msec);
        CheckBool( &multiWaitHolder == nn::os::WaitAny(&multiWait) );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ワンショット待機後の event が true"));
        CheckBool( &multiWaitHolder == nn::os::TryWaitAny(&multiWait) );
        nn::os::ClearTimerEvent(&event);

        // StartOneShot( 50msec ) 中に Signal するテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(50msec) 中に Signal でシグナル化可能か"));
        nn::os::StartOneShotTimerEvent(&event, time50msec);
        nn::os::SignalTimerEvent(&event);
        CheckBool( &multiWaitHolder == nn::os::TryWaitAny(&multiWait) );
        nn::os::ClearTimerEvent(&event);

        nn::os::SleepThread(time100msec);
        // StartOneShotTimeEvent に対する Wait
        EXPECT_TRUE( &multiWaitHolder == nn::os::WaitAny(&multiWait) );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ワンショット待機後の event が true"));
        CheckBool( &multiWaitHolder == nn::os::TryWaitAny(&multiWait) );
        nn::os::ClearTimerEvent(&event);

        // StartOneShot( 100msec ) 中に Clear するテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(100msec) 中に Clear しても影響がないか"));
        nn::os::StartOneShotTimerEvent(&event, time100msec);
        nn::os::ClearTimerEvent(&event);
        CheckBool( NULL == nn::os::TryWaitAny(&multiWait) );
        // StartOneShotTimeEvent に対する Wait
        EXPECT_TRUE( &multiWaitHolder == nn::os::WaitAny(&multiWait) );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ワンショット待機後の event が true"));
        CheckBool( &multiWaitHolder == nn::os::TryWaitAny(&multiWait) );
        nn::os::ClearTimerEvent(&event);

        // StartOneShot( 100msec ) 中に StartOneShot( 100msec ) するテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(100msec) 中に StartOneShot(100msec) 可能か"));
        nn::os::StartOneShotTimerEvent(&event, time100msec);
        EXPECT_TRUE( NULL == nn::os::TryWaitAny(&multiWait) );

        nn::os::StartOneShotTimerEvent(&event, time100msec);
        // 後発の StartOneShot に対する Wait
        CheckBool( &multiWaitHolder == nn::os::WaitAny(&multiWait) );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ワンショット待機後の event が true"));
        CheckBool( &multiWaitHolder == nn::os::TryWaitAny(&multiWait) );
        nn::os::ClearTimerEvent(&event);

        // StartOneShot( 100msec ) 中に StartPeriodic() するテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(100msec) 中に StartPeriodic(50msec,50msec) 可能か"));
        nn::os::StartOneShotTimerEvent(&event, time100msec);
        EXPECT_TRUE( NULL == nn::os::TryWaitAny(&multiWait) );

        nn::os::StartPeriodicTimerEvent(&event, time50msec, time50msec);
        for (int i=0; i<4; ++i)
        {
            EXPECT_TRUE( &multiWaitHolder == nn::os::WaitAny(&multiWait) );
            nn::os::ClearTimerEvent(&event);
        }
        nn::os::StopTimerEvent(&event);     // タイマー停止
        nn::os::ClearTimerEvent(&event);    // クリア
        CheckBool( true );

        // StartOneShot( 50msec ) 中に Stop() するテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartOneShot(50msec) 中に StopOneShot() 可能か"));
        nn::os::StartOneShotTimerEvent(&event, time50msec);
        EXPECT_TRUE( NULL == nn::os::TryWaitAny(&multiWait) );

        nn::os::StopTimerEvent(&event);
        nn::os::SleepThread(time100msec);
        CheckBool( NULL == nn::os::TryWaitAny(&multiWait) );

        // Finalize
        nn::os::UnlinkMultiWaitHolder(&multiWaitHolder);
        nn::os::FinalizeMultiWaitHolder(&multiWaitHolder);
        nn::os::FinalizeMultiWait(&multiWait);
        nn::os::FinalizeTimerEvent(&event);
    }

    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  TimerEvent(AutoClear) の周期タイマー機能の多重待ちテスト
//  StartPeriodicTimerEvent() によるタイマーの基本動作と、
//  タイマー動作中の Start, Stop, Signal, Clear 操作のテスト。
//
TEST(StartPeriodicTimerEventMultiWait, test_StartPeriodicTimerEventTestAutoClear)
{
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan  time1msec   = nn::TimeSpan::FromMilliSeconds(1);
        const nn::TimeSpan  time10msec  = nn::TimeSpan::FromMilliSeconds(10);
        const nn::TimeSpan  time20msec  = nn::TimeSpan::FromMilliSeconds(20);
        const nn::TimeSpan  time50msec  = nn::TimeSpan::FromMilliSeconds(50);
        const nn::TimeSpan  time100msec = nn::TimeSpan::FromMilliSeconds(100);
        const nn::TimeSpan  time1sec    = nn::TimeSpan::FromSeconds(1);
        const nn::TimeSpan  time10sec   = nn::TimeSpan::FromSeconds(10);

        SEQ_SET(3700);
        NNT_OS_LOG(NN_TEXT("TimerEvent（AutoClear）の周期タイマー多重待ちテスト\n"));
        nn::os::TimerEventType event;
        nn::os::InitializeTimerEvent(&event, nn::os::EventClearMode_AutoClear);

        // 多重待ちオブジェクトの準備 ----------
        nn::os::MultiWaitType       multiWait;
        nn::os::MultiWaitHolderType multiWaitHolder;

        nn::os::InitializeMultiWait(&multiWait);
        nn::os::InitializeMultiWaitHolder(&multiWaitHolder, &event);
        nn::os::LinkMultiWaitHolder(&multiWait, &multiWaitHolder);

        SEQ_CHECK(3701);
        NNT_OS_LOG(NN_TEXT("event の初期状態が false"));
        CheckBool( NULL == nn::os::TryWaitAny(&multiWait) );

        // StartPeriodic( 50msec, 20msec ) のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartPeriodic(50msec, 20msec) でシグナル化する\n"));
        nn::os::Tick startTick = nn::os::GetSystemTick();
        nn::os::StartPeriodicTimerEvent(&event, time50msec, time20msec);
        for (int i=0; i<5; ++i)
        {
            EXPECT_TRUE( &multiWaitHolder == nn::os::WaitAny(&multiWait) );
            // 多重待ちなのでクリアされていないことを確認
            EXPECT_TRUE( &multiWaitHolder == nn::os::TryWaitAny(&multiWait) );
            nn::os::ClearTimerEvent(&event);

            nn::os::Tick currTick = nn::os::GetSystemTick();
            SEQ_NONE();
            NNT_OS_LOG("Signaled interval = %6d (usec)\n", (currTick - startTick).ToTimeSpan().GetMicroSeconds());
            startTick = currTick;
        }
        nn::os::StopTimerEvent(&event);     // タイマー停止
        nn::os::ClearTimerEvent(&event);    // クリア

        // StartPeriodic( 10msec, 20msec ) 中に Signal/Clear するテスト
        SEQ_CHECK(3702);
        NNT_OS_LOG(NN_TEXT("StartPeriodic(10msec, 20msec) 中に Signal/Clear 可能か\n"));
        startTick = nn::os::GetSystemTick();
        nn::os::StartPeriodicTimerEvent(&event, time10msec, time20msec);
        for (int i=0; i<5; ++i)
        {
            nn::os::SignalTimerEvent(&event);
            EXPECT_TRUE( &multiWaitHolder == nn::os::TryWaitAny(&multiWait) );
            nn::os::ClearTimerEvent(&event);
            EXPECT_TRUE( NULL == nn::os::TryWaitAny(&multiWait) );

            EXPECT_TRUE( &multiWaitHolder == nn::os::WaitAny(&multiWait) );
            // 多重待ちなのでクリアされていないことを確認
            EXPECT_TRUE( &multiWaitHolder == nn::os::TryWaitAny(&multiWait) );
            nn::os::ClearTimerEvent(&event);

            nn::os::Tick currTick = nn::os::GetSystemTick();
            SEQ_NONE();
            NNT_OS_LOG("Signaled interval = %6d (usec)\n", (currTick - startTick).ToTimeSpan().GetMicroSeconds());
            startTick = currTick;
        }
        nn::os::StopTimerEvent(&event);     // タイマー停止
        nn::os::ClearTimerEvent(&event);    // クリア

        // StartPeriodic(10sec, 1sec) 中に StartPeriodic(20ms, 10ms) するテスト
        SEQ_CHECK(3703);
        NNT_OS_LOG(NN_TEXT("StartPeriodic(10s,1s) 中に StartPeriodic(20ms,10ms) 可能か"));
        nn::os::StartPeriodicTimerEvent(&event, time10sec, time1sec);
        EXPECT_TRUE( NULL == nn::os::TryWaitAny(&multiWait) );

        startTick = nn::os::GetSystemTick();
        nn::os::StartPeriodicTimerEvent(&event, time20msec, time10msec);
        CheckBool( true );

        for (int i=0; i<5; ++i)
        {
            EXPECT_TRUE( &multiWaitHolder == nn::os::WaitAny(&multiWait) );
            EXPECT_TRUE( &multiWaitHolder == nn::os::TryWaitAny(&multiWait) );
            nn::os::ClearTimerEvent(&event);

            nn::os::Tick currTick = nn::os::GetSystemTick();
            SEQ_NONE();
            NNT_OS_LOG("Signaled interval = %6d (usec)\n", (currTick - startTick).ToTimeSpan().GetMicroSeconds());
            startTick = currTick;
        }
        nn::os::StopTimerEvent(&event);     // タイマー停止
        nn::os::ClearTimerEvent(&event);    // クリア

        // StartPeriodic( 100msec, 1msec ) 中に StartOneShot(50msec) するテスト
        SEQ_CHECK(3704);
        NNT_OS_LOG(NN_TEXT("StartPeriodic(100ms, 1ms) 中に StartOneShot(50msec) 可能か"));
        nn::os::StartPeriodicTimerEvent(&event, time100msec, time1msec);
        EXPECT_TRUE( NULL == nn::os::TryWaitAny(&multiWait) );

        nn::os::StartOneShotTimerEvent(&event, time50msec);
        // OneShot の１発目
        CheckBool( &multiWaitHolder == nn::os::WaitAny(&multiWait) );
        nn::os::ClearTimerEvent(&event);

        nn::os::SleepThread( time100msec );
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ワンショット後 100msec 経っても false か"));
        CheckBool( NULL == nn::os::TryWaitAny(&multiWait) );

        nn::os::StopTimerEvent(&event);     // タイマー停止
        nn::os::ClearTimerEvent(&event);    // クリア

        // StartPeriodic( 50msec, 10msec ) 中に Stop() するテスト
        SEQ_CHECK(3705);
        NNT_OS_LOG(NN_TEXT("StartPeriodic(50msec,10msec) 中に StopPeriodic() 可能か"));
        nn::os::StartPeriodicTimerEvent(&event, time50msec, time10msec);
        EXPECT_TRUE( NULL == nn::os::TryWaitAny(&multiWait) );

        nn::os::StopTimerEvent(&event);
        nn::os::SleepThread(time100msec);
        CheckBool( NULL == nn::os::TryWaitAny(&multiWait) );

        // Finalize
        nn::os::UnlinkMultiWaitHolder(&multiWaitHolder);
        nn::os::FinalizeMultiWaitHolder(&multiWaitHolder);
        nn::os::FinalizeMultiWait(&multiWait);
        nn::os::FinalizeTimerEvent(&event);
    }

    JUDGE_GOOGLE_TEST();
}   // NOLINT(readability/fn_size)

//---------------------------------------------------------------------------
//  TimerEvent(ManualClear) の周期タイマー機能の多重待ちテスト
//  StartPeriodicTimerEvent() によるタイマーの基本動作と、
//  タイマー動作中の Start, Stop, Signal, Clear 操作のテスト。
//
TEST(StartPeriodicTimerEventMultiWait, test_StartPeriodicTimerEventTestManualClear)
{
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan  time1msec   = nn::TimeSpan::FromMilliSeconds(1);
        const nn::TimeSpan  time10msec  = nn::TimeSpan::FromMilliSeconds(10);
        const nn::TimeSpan  time20msec  = nn::TimeSpan::FromMilliSeconds(20);
        const nn::TimeSpan  time50msec  = nn::TimeSpan::FromMilliSeconds(50);
        const nn::TimeSpan  time100msec = nn::TimeSpan::FromMilliSeconds(100);
        const nn::TimeSpan  time1sec    = nn::TimeSpan::FromSeconds(1);
        const nn::TimeSpan  time10sec   = nn::TimeSpan::FromSeconds(10);

        SEQ_SET(3800);
        NNT_OS_LOG(NN_TEXT("TimerEvent（ManualClear）の周期タイマー多重待ちテスト\n"));
        nn::os::TimerEventType event;
        nn::os::InitializeTimerEvent(&event, nn::os::EventClearMode_ManualClear);

        // 多重待ちオブジェクトの準備 ----------
        nn::os::MultiWaitType       multiWait;
        nn::os::MultiWaitHolderType multiWaitHolder;

        nn::os::InitializeMultiWait(&multiWait);
        nn::os::InitializeMultiWaitHolder(&multiWaitHolder, &event);
        nn::os::LinkMultiWaitHolder(&multiWait, &multiWaitHolder);

        SEQ_CHECK(3801);
        NNT_OS_LOG(NN_TEXT("event の初期状態が false"));
        CheckBool( NULL == nn::os::TryWaitAny(&multiWait) );

        // StartPeriodic( 50msec, 20msec ) のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("StartPeriodic(50msec, 20msec) でシグナル化する\n"));
        nn::os::Tick startTick = nn::os::GetSystemTick();
        nn::os::StartPeriodicTimerEvent(&event, time50msec, time20msec);
        for (int i=0; i<5; ++i)
        {
            EXPECT_TRUE( &multiWaitHolder == nn::os::WaitAny(&multiWait) );
            EXPECT_TRUE( &multiWaitHolder == nn::os::TryWaitAny(&multiWait) );
            // あえて TimerEvent をクリアしない

            nn::os::Tick currTick = nn::os::GetSystemTick();
            SEQ_NONE();
            NNT_OS_LOG("Signaled interval = %6d (usec)\n", (currTick - startTick).ToTimeSpan().GetMicroSeconds());
            startTick = currTick;
        }
        nn::os::StopTimerEvent(&event);     // タイマー停止
        nn::os::ClearTimerEvent(&event);    // クリア

        // StartPeriodic( 50msec, 20msec ) 中に Signal/Clear するテスト
        SEQ_CHECK(3802);
        NNT_OS_LOG(NN_TEXT("StartPeriodic(50msec, 20msec) 中に Signal/Clear 可能か\n"));
        startTick = nn::os::GetSystemTick();
        nn::os::StartPeriodicTimerEvent(&event, time50msec, time20msec);
        for (int i=0; i<5; ++i)
        {
            nn::os::SignalTimerEvent(&event);
            EXPECT_TRUE( &multiWaitHolder == nn::os::TryWaitAny(&multiWait) );
            nn::os::ClearTimerEvent(&event);
            EXPECT_TRUE( NULL == nn::os::TryWaitAny(&multiWait) );

            EXPECT_TRUE( &multiWaitHolder == nn::os::WaitAny(&multiWait) );
            EXPECT_TRUE( &multiWaitHolder == nn::os::TryWaitAny(&multiWait) );
            nn::os::ClearTimerEvent(&event);

            nn::os::Tick currTick = nn::os::GetSystemTick();
            SEQ_NONE();
            NNT_OS_LOG("Signaled interval = %6d (usec)\n", (currTick - startTick).ToTimeSpan().GetMicroSeconds());
            startTick = currTick;
        }
        nn::os::StopTimerEvent(&event);     // タイマー停止
        nn::os::ClearTimerEvent(&event);    // クリア

        // StartPeriodic(10sec, 1sec) 中に StartPeriodic(20ms, 10ms) するテスト
        SEQ_CHECK(3803);
        NNT_OS_LOG(NN_TEXT("StartPeriodic(10s,1s) 中に StartPeriodic(20ms,10ms) 可能か"));
        nn::os::StartPeriodicTimerEvent(&event, time10sec, time1sec);
        EXPECT_TRUE( NULL == nn::os::TryWaitAny(&multiWait) );

        startTick = nn::os::GetSystemTick();
        nn::os::StartPeriodicTimerEvent(&event, time20msec, time10msec);
        CheckBool( true );

        for (int i=0; i<5; ++i)
        {
            EXPECT_TRUE( &multiWaitHolder == nn::os::WaitAny(&multiWait) );
            EXPECT_TRUE( &multiWaitHolder == nn::os::TryWaitAny(&multiWait) );
            // あえて TimerEvent をクリアしない

            nn::os::Tick currTick = nn::os::GetSystemTick();
            SEQ_NONE();
            NNT_OS_LOG("Signaled interval = %6d (usec)\n", (currTick - startTick).ToTimeSpan().GetMicroSeconds());
            startTick = currTick;
        }
        nn::os::StopTimerEvent(&event);     // タイマー停止
        nn::os::ClearTimerEvent(&event);    // クリア

        // StartPeriodic( 100msec, 1msec ) 中に StartOneShot(50msec) するテスト
        SEQ_CHECK(3804);
        NNT_OS_LOG(NN_TEXT("StartPeriodic(100ms, 1ms) 中に StartOneShot(50msec) 可能か"));
        nn::os::StartPeriodicTimerEvent(&event, time100msec, time1msec);
        EXPECT_TRUE( NULL == nn::os::TryWaitAny(&multiWait) );

        nn::os::StartOneShotTimerEvent(&event, time50msec);
        CheckBool( &multiWaitHolder == nn::os::WaitAny(&multiWait) );

        nn::os::SleepThread( time100msec );
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ワンショット後 100msec 経っても true のままか"));
        CheckBool( &multiWaitHolder == nn::os::TryWaitAny(&multiWait) );

        nn::os::StopTimerEvent(&event);     // タイマー停止
        nn::os::ClearTimerEvent(&event);    // クリア

        // StartPeriodic( 50msec, 10msec ) 中に Stop() するテスト
        SEQ_CHECK(3805);
        NNT_OS_LOG(NN_TEXT("StartPeriodic(50msec,10msec) 中に StopPeriodic() 可能か"));
        nn::os::StartPeriodicTimerEvent(&event, time50msec, time10msec);
        EXPECT_TRUE( NULL == nn::os::TryWaitAny(&multiWait) );

        nn::os::StopTimerEvent(&event);
        nn::os::SleepThread(time100msec);
        CheckBool( NULL == nn::os::TryWaitAny(&multiWait) );

        nn::os::StopTimerEvent(&event);     // タイマー停止
        nn::os::ClearTimerEvent(&event);    // クリア

        // Finalize
        nn::os::UnlinkMultiWaitHolder(&multiWaitHolder);
        nn::os::FinalizeMultiWaitHolder(&multiWaitHolder);
        nn::os::FinalizeMultiWait(&multiWait);
        nn::os::FinalizeTimerEvent(&event);
    }

    JUDGE_GOOGLE_TEST();
}   // NOLINT(readability/fn_size)

//---------------------------------------------------------------------------
//  TimerEvent(AutoClear) のタイマー待機の多重待ちテスト
//  WaitAny() 中に他スレッドからタイマー動作を変更
//  タイマー待機中の Start, Stop, Signal, Clear 操作のテスト。
//
//  単独待ちテストで使用した g_Stack, g_Barrier を使用する。
//
void test_WaitAnyTimerEventTestHelperThread(void* arg)
{
    nn::os::TimerEventType* event = static_cast<nn::os::TimerEventType*>(arg);

    const nn::TimeSpan  time10msec  = nn::TimeSpan::FromMilliSeconds(10);
    const nn::TimeSpan  time20msec  = nn::TimeSpan::FromMilliSeconds(20);
    const nn::TimeSpan  time50msec  = nn::TimeSpan::FromMilliSeconds(50);

    // step1: 20msec 後に StartOneShot( 50msec ) を発行
    nn::os::SleepThread( time20msec );
    nn::os::StartOneShotTimerEvent(event, time50msec);

    nn::os::AwaitBarrier(&g_Barrier);

    // step2: 20msec 後に StartOneShot( 50msec, 10msec ) を発行
    nn::os::SleepThread( time20msec );
    nn::os::StartPeriodicTimerEvent(event, time50msec, time10msec);

    nn::os::AwaitBarrier(&g_Barrier);
}

TEST(WaitAnyTimerEventMultiWait, test_WaitAnyTimerEventTestAutoClear)
{
    CLEAR_GOOGLE_TEST();

    nn::os::InitializeBarrier(&g_Barrier, 2);

    {
        SEQ_SET(3900);
        NNT_OS_LOG(NN_TEXT("TimerEvent（AutoClear）の周期タイマー多重待ちテスト\n"));
        nn::os::TimerEventType event;
        nn::os::InitializeTimerEvent(&event, nn::os::EventClearMode_AutoClear);

        // 多重待ちオブジェクトの準備 ----------
        nn::os::MultiWaitType       multiWait;
        nn::os::MultiWaitHolderType multiWaitHolder;

        nn::os::InitializeMultiWait(&multiWait);
        nn::os::InitializeMultiWaitHolder(&multiWaitHolder, &event);
        nn::os::LinkMultiWaitHolder(&multiWait, &multiWaitHolder);

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("子スレッドを生成する"));
        nn::os::ThreadType  thread;
        int pri = nn::os::GetThreadPriority(nn::os::GetCurrentThread());
        nn::os::CreateThread(&thread, test_WaitAnyTimerEventTestHelperThread, &event, g_Stack, sizeof(g_Stack), pri + 1);
        nn::os::StartThread(&thread);
        CheckBool(true);

        // step1: WaitAny() 中に他スレッドから StartOneShot()
        SEQ_CHECK(3901);
        NNT_OS_LOG(NN_TEXT("WaitAny() 中に他スレッドから StartOneShot()"));

        nn::os::Tick startTick = nn::os::GetSystemTick();
        EXPECT_TRUE( &multiWaitHolder == nn::os::WaitAny(&multiWait) );
        CheckBool( &multiWaitHolder == nn::os::TryWaitAny(&multiWait) );
        nn::os::ClearTimerEvent(&event);

        nn::os::Tick currTick = nn::os::GetSystemTick();
        SEQ_NONE();
        NNT_OS_LOG("Signaled interval = %6d (usec)\n", (currTick - startTick).ToTimeSpan().GetMicroSeconds());
        startTick = currTick;

        nn::os::AwaitBarrier(&g_Barrier);

        // step2: WaitAny() 中に他スレッドから StartPeriodic()
        SEQ_CHECK(3902);
        NNT_OS_LOG(NN_TEXT("WaitAny() 中に他スレッドから StartPeriodic()"));

        startTick = nn::os::GetSystemTick();
        EXPECT_TRUE( &multiWaitHolder == nn::os::WaitAny(&multiWait) );
        CheckBool( &multiWaitHolder == nn::os::TryWaitAny(&multiWait) );
        nn::os::ClearTimerEvent(&event);

        currTick = nn::os::GetSystemTick();
        SEQ_NONE();
        NNT_OS_LOG("Signaled interval = %6d (usec)\n", (currTick - startTick).ToTimeSpan().GetMicroSeconds());
        startTick = currTick;

        for (int i=0; i<5; ++i)
        {
            EXPECT_TRUE( &multiWaitHolder == nn::os::WaitAny(&multiWait) );
            EXPECT_TRUE( &multiWaitHolder == nn::os::TryWaitAny(&multiWait) );
            nn::os::ClearTimerEvent(&event);

            currTick = nn::os::GetSystemTick();
            SEQ_NONE();
            NNT_OS_LOG("Signaled interval = %6d (usec)\n", (currTick - startTick).ToTimeSpan().GetMicroSeconds());
            startTick = currTick;
        }

        nn::os::AwaitBarrier(&g_Barrier);

        // Finalize
        nn::os::WaitThread(&thread);
        nn::os::DestroyThread(&thread);

        nn::os::UnlinkMultiWaitHolder(&multiWaitHolder);
        nn::os::FinalizeMultiWaitHolder(&multiWaitHolder);
        nn::os::FinalizeMultiWait(&multiWait);
        nn::os::FinalizeTimerEvent(&event);
    }

    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  TimerEvent(AutoClear) のタイマー待機の多重待ちテスト
//  3 つの TimerEvent を多重待ちし、それぞれ異なる設定値のタイマー動作を
//  正しく多重待ちできるかをテスト。
//
TEST(WaitAnyTimerEventMultiWait, test_WaitAnyThreeTimerEventsTestAutoClear)
{
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan  time10sec  = nn::TimeSpan::FromSeconds(10);
        const nn::TimeSpan  time50msec = nn::TimeSpan::FromMilliSeconds(50);

        SEQ_SET(4000);
        NNT_OS_LOG(NN_TEXT("TimerEvent（AutoClear）の周期タイマー多重待ちテスト\n"));

        // TimerEvent および多重待ちオブジェクトの準備
        const int eventNum = 3;

        nn::os::MultiWaitType       multiWait;
        nn::os::MultiWaitHolderType multiWaitHolder[eventNum];
        nn::os::TimerEventType      timerEvent[eventNum];

        nn::os::InitializeMultiWait(&multiWait);
        for (uintptr_t i = 0; i < eventNum; ++i)
        {
            nn::os::InitializeTimerEvent(&timerEvent[i], nn::os::EventClearMode_AutoClear);
            nn::os::InitializeMultiWaitHolder(&multiWaitHolder[i], &timerEvent[i]);
            nn::os::SetMultiWaitHolderUserData(&multiWaitHolder[i], i);
            nn::os::LinkMultiWaitHolder(&multiWait, &multiWaitHolder[i]);
        }

        // 位相をずらして eventNum 個の TimerEvent を StartPeriodic する
        SEQ_CHECK(4001);
        NNT_OS_LOG(NN_TEXT("%d 個の TimerEvent を位相をずらして StartPeriodic()"), eventNum);

        nn::os::Tick startTick = nn::os::GetSystemTick();
        for (int i = 0; i < eventNum; ++i)
        {
            nn::os::StartPeriodicTimerEvent( &timerEvent[i],
                            nn::TimeSpan::FromMilliSeconds( 100 + 100 * i ),
                            nn::TimeSpan::FromMilliSeconds( 100 * eventNum ) );
        }
        CheckBool( true );

        // 多重待ちを行なう
        for (int i = 0; i < eventNum * 5; ++i)
        {
            auto holder = nn::os::WaitAny(&multiWait);
            auto index  = nn::os::GetMultiWaitHolderUserData(holder);
            if (!nn::os::TryWaitTimerEvent(&timerEvent[index]))
            {
                continue;
            }

            nn::os::Tick currTick = nn::os::GetSystemTick();
            SEQ_NONE();
            NNT_OS_LOG("Signaled[%d] interval = %6lld (usec)\n", index, (currTick - startTick).ToTimeSpan().GetMicroSeconds());
            startTick = currTick;
        }

        // StartOneShot(10sec) を行ない、100msec でタイムアウトできるか
        for (int i = 0; i < eventNum; ++i)
        {
            nn::os::StartOneShotTimerEvent( &timerEvent[i], time10sec);
            nn::os::ClearTimerEvent( &timerEvent[i] );
        }
        SEQ_CHECK(4002);
        NNT_OS_LOG(NN_TEXT("%d 個の StartOneShot(10sec) を行ない、50msec でタイムアウト"), eventNum);

        startTick = nn::os::GetSystemTick();
        CheckBool( NULL == nn::os::TimedWaitAny(&multiWait, time50msec) );
        nn::os::Tick currTick  = nn::os::GetSystemTick();

        SEQ_NONE();
        NNT_OS_LOG("timedout: %7lld (usec)\n", (currTick - startTick).ToTimeSpan().GetMicroSeconds());
        NN_UNUSED(currTick);

        // Finalize
        for (int i = 0; i < eventNum; ++i)
        {
            nn::os::UnlinkMultiWaitHolder(&multiWaitHolder[i]);
            nn::os::FinalizeMultiWaitHolder(&multiWaitHolder[i]);
            nn::os::FinalizeTimerEvent(&timerEvent[i]);
        }
        nn::os::FinalizeMultiWait(&multiWait);
    }

    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  TimerEvent(AutoClear) のタイマー動作の境界値テスト
//  StartOneShotTimerEvent() と StartPeriodicTimerEvent() に対して、
//  TimeSpan 引数の境界条件をテストする。
//
void test_StartTimerEventTestDeathTest1()
{
    SEQ_SET(4100);
    NNT_OS_LOG(NN_TEXT("StartOneShot(-1) の境界値 Death テスト\n"));

    nn::os::TimerEventType event;
    nn::os::InitializeTimerEvent(&event, nn::os::EventClearMode_AutoClear);

    nn::os::StartOneShotTimerEvent( &event, nn::TimeSpan::FromNanoSeconds(-1) );

    nn::os::FinalizeTimerEvent(&event);
}

void test_StartTimerEventTestDeathTest2()
{
    SEQ_SET(4110);
    NNT_OS_LOG(NN_TEXT("StartOneShot(INT64_MIN) の境界値 Death テスト\n"));

    nn::os::TimerEventType event;
    nn::os::InitializeTimerEvent(&event, nn::os::EventClearMode_AutoClear);

    nn::os::StartOneShotTimerEvent( &event, nn::TimeSpan::FromNanoSeconds(INT64_MIN) );

    nn::os::FinalizeTimerEvent(&event);
}

void test_StartTimerEventTestDeathTest3()
{
    SEQ_SET(4120);
    NNT_OS_LOG(NN_TEXT("StartPeriodic(-1, 10msec) の境界値 Death テスト\n"));

    nn::os::TimerEventType event;
    nn::os::InitializeTimerEvent(&event, nn::os::EventClearMode_AutoClear);

    nn::os::StartPeriodicTimerEvent( &event,
                                     nn::TimeSpan::FromNanoSeconds(-1),
                                     nn::TimeSpan::FromMilliSeconds(10) );

    nn::os::FinalizeTimerEvent(&event);
}

void test_StartTimerEventTestDeathTest4()
{
    SEQ_SET(4130);
    NNT_OS_LOG(NN_TEXT("StartPeriodic(INT64_MIN, 10msec) の境界値 Death テスト\n"));

    nn::os::TimerEventType event;
    nn::os::InitializeTimerEvent(&event, nn::os::EventClearMode_AutoClear);

    nn::os::StartPeriodicTimerEvent( &event,
                                     nn::TimeSpan::FromNanoSeconds(INT64_MIN),
                                     nn::TimeSpan::FromMilliSeconds(10) );

    nn::os::FinalizeTimerEvent(&event);
}

void test_StartTimerEventTestDeathTest5()
{
    SEQ_SET(4130);
    NNT_OS_LOG(NN_TEXT("StartPeriodic(10msec, 0) の境界値 Death テスト\n"));

    nn::os::TimerEventType event;
    nn::os::InitializeTimerEvent(&event, nn::os::EventClearMode_AutoClear);

    nn::os::StartPeriodicTimerEvent( &event,
                                     nn::TimeSpan::FromMilliSeconds(10),
                                     nn::TimeSpan::FromNanoSeconds(0));

    nn::os::FinalizeTimerEvent(&event);
}

void test_StartTimerEventTestDeathTest6()
{
    SEQ_SET(4130);
    NNT_OS_LOG(NN_TEXT("StartPeriodic(10msec, INT64_MIN) の境界値 Death テスト\n"));

    nn::os::TimerEventType event;
    nn::os::InitializeTimerEvent(&event, nn::os::EventClearMode_AutoClear);

    nn::os::StartPeriodicTimerEvent( &event,
                                     nn::TimeSpan::FromMilliSeconds(10),
                                     nn::TimeSpan::FromNanoSeconds(INT64_MIN));

    nn::os::FinalizeTimerEvent(&event);
}

TEST(StartTimerEventDeathTest, test_StartTimerEventTestDeathTest)
{
    CLEAR_GOOGLE_TEST();

    {
        EXPECT_DEATH_IF_SUPPORTED( test_StartTimerEventTestDeathTest1(), "");
        EXPECT_DEATH_IF_SUPPORTED( test_StartTimerEventTestDeathTest2(), "");
        EXPECT_DEATH_IF_SUPPORTED( test_StartTimerEventTestDeathTest3(), "");
        EXPECT_DEATH_IF_SUPPORTED( test_StartTimerEventTestDeathTest4(), "");
        EXPECT_DEATH_IF_SUPPORTED( test_StartTimerEventTestDeathTest5(), "");
        EXPECT_DEATH_IF_SUPPORTED( test_StartTimerEventTestDeathTest6(), "");
    }

    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
//  TimerEvent クラスのテストその１
//  ここでは、TimerEvent クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  test_TimerEventClassTestAutoClear ではブロックしない AutoClear イベントの
//  テストを実施する。
//
TEST(TimerEventClass, test_TimerEventClassTestAutoClear)
{
    CLEAR_GOOGLE_TEST();

    {
        SEQ_SET(5000);
        NNT_OS_LOG(NN_TEXT("event インスタンスの生成（AutoClear）\n"));
        nn::os::TimerEvent event(nn::os::EventClearMode_AutoClear);

        // ここからがテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event.TryWait() が false を返す"));
        CheckBool( event.TryWait() == false );

        // Wait のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event.Signal() に成功する\n"));
        event.Signal();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event.Wait() に成功する\n"));
        event.Wait();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event.TryWait() が false を返す"));
        CheckBool( event.TryWait() == false );

        // TryWait のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event.Signal() に成功する\n"));
        event.Signal();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event.TryWait() が true を返す"));
        CheckBool( event.TryWait() == true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event.TryWait() が false を返す"));
        CheckBool( event.TryWait() == false );
    }

    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  TimerEvent クラスのテストその２
//  ここでは、TimerEvent クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  test_TimerEventClassTestManualClear ではブロックしない ManualClear イベントの
//  テストを実施する。
//
TEST(TimerEventClass, test_TimerEventClassTestManualClear)
{
    CLEAR_GOOGLE_TEST();

    {
        SEQ_SET(5100);
        NNT_OS_LOG(NN_TEXT("event インスタンスの生成（ManualClear）\n"));
        nn::os::TimerEvent event(nn::os::EventClearMode_ManualClear);

        // ここからがテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event.TryWait() が false を返す"));
        CheckBool( event.TryWait() == false );

        // Wait のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event.Signal() に成功する\n"));
        event.Signal();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event.Wait() に成功する\n"));
        event.Wait();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event.TryWait() が true を返す"));
        CheckBool( event.TryWait() == true );

        // TryWait のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event.Signal() に成功する\n"));
        event.Signal();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event.TryWait() が true を返す"));
        CheckBool( event.TryWait() == true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event.TryWait() が true を返す"));
        CheckBool( event.TryWait() == true );

        // Clear の確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event.Clear() に成功する\n"));
        event.Clear();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event.TryWait() が false を返す"));
        CheckBool( event.TryWait() == false );
    }

    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  TimerEvent クラスのテストその３
//  ここでは、TimerEvent クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  test_TimerEventClassTestBlock ではブロック系のテストを２スレッド使って
//  実施する。
//
NN_ALIGNAS( 4096 ) char g_StackOfThread[ 8192 ];

void TimerEventClassTestThread(void* arg)
{
    auto&              event    = *static_cast<nn::os::TimerEvent*>(arg);
    const nn::TimeSpan interval = nn::TimeSpan::FromMilliSeconds(100);

    for (int i = 0; i < 2; ++i)
    {
        nn::os::SleepThread( interval );

        SEQ_CHECK(5202 + i * 3);
        NNT_OS_LOG(NN_TEXT("sub:  event.Signal() でシグナル化\n"));
        event.Signal();
    }
}

TEST(TimerEventClass, test_TimerEventClassTestBlock)
{
    CLEAR_GOOGLE_TEST();

    {
        SEQ_SET(5200);
        NNT_OS_LOG(NN_TEXT("main: TimerEvent インスタンスの生成"));
        nn::os::TimerEvent event(nn::os::EventClearMode_AutoClear);

        // テスト用の子スレッドを作成
        nn::os::ThreadType  thread;
        {
            auto pri    = nn::os::GetThreadPriority(nn::os::GetCurrentThread());
            auto result = nn::os::CreateThread( &thread,
                                                &TimerEventClassTestThread,
                                                &event,
                                                g_StackOfThread,
                                                sizeof(g_StackOfThread),
                                                pri + 1);
            CheckBool( result.IsSuccess() );
        }

        // 子スレッドを起動
        nn::os::StartThread( &thread );

        // Step1: Wait で待機して子スレッドから起床
        SEQ_CHECK(5201);
        NNT_OS_LOG(NN_TEXT("main: evnet.Wait() で待機\n"));
        event.Wait();
        SEQ_CHECK(5203);
        NNT_OS_LOG(NN_TEXT("main: -> 起床\n"));

        // 子スレッドの終了待ち
        nn::os::WaitThread( &thread );
        nn::os::DestroyThread( &thread );
    }

    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  TimerEvent クラスのテストその４
//  ここでは、TimerEvent クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  test_TimerEventClassTestStartTimer ではタイマー動作に関するテストを実施する。
//
TEST(TimerEventClass, test_TimerEventClassTestStartTimer)
{
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan  time100msec = nn::TimeSpan::FromMilliSeconds(100);
        const nn::TimeSpan  time80msec  = nn::TimeSpan::FromMilliSeconds(80);
        const nn::TimeSpan  time50msec  = nn::TimeSpan::FromMilliSeconds(50);

        SEQ_SET(5300);
        NNT_OS_LOG(NN_TEXT("TimerEvent インスタンスの生成（AutoClear）\n"));
        nn::os::TimerEvent event(nn::os::EventClearMode_AutoClear);

        // StartOneShot のテスト
        SEQ_CHECK(5301);
        NNT_OS_LOG(NN_TEXT("TimerEvent::StartOneShot(100msec) 直後は非シグナル状態"));
        event.StartOneShot(time100msec);
        CheckBool( false == event.TryWait() );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("TimerEvent::Wait() に成功し、直後は非シグナル状態"));
        event.Wait();
        CheckBool( false == event.TryWait() );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("TimerEvent::TryWait() が false を返す"));
        CheckBool( event.TryWait() == false );

        // StartPeriodic のテスト
        SEQ_CHECK(5302);
        NNT_OS_LOG(NN_TEXT("TimerEvent::StartPeriodic(80msec, 50msec) 直後は非シグナル状態"));
        event.StartPeriodic(time80msec, time50msec);
        CheckBool( false == event.TryWait() );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("TimerEvent::Wait() に成功し、直後は非シグナル状態"));
        event.Wait();
        CheckBool( false == event.TryWait() );

        for (int i = 0; i < 5; ++i)
        {
            SEQ_NONE();
            NNT_OS_LOG(NN_TEXT("TimerEvent::Wait() に成功し、直後は非シグナル状態"));
            event.Wait();
            CheckBool( false == event.TryWait() );
        }

        // Stop のテスト
        SEQ_CHECK(5303);
        NNT_OS_LOG(NN_TEXT("TimerEvent::Stop() 直後は非シグナル状態"));
        event.Stop();
        CheckBool( false == event.TryWait() );
    }

    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
//  TimerEvent クラスの型変換関数のテスト
//  ここでは、TimerEvent クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  ここでは、GetBase() および operator TimerEventType&() の動作テストを行なう。
//
TEST(TimerEventClass, test_TimerEventClassTestTypeExchange)
{
    CLEAR_GOOGLE_TEST();

    {
        SEQ_SET(5400);
        NNT_OS_LOG(NN_TEXT("event インスタンスの生成（AutoClear）\n"));
        nn::os::TimerEvent event(nn::os::EventClearMode_AutoClear);

        // ここからがテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event.GetBase() で TimerEventType オブジェクトを取得"));
        nn::os::TimerEventType*  eventType = event.GetBase();
        CheckBool( eventType != NULL );

        // 初期シグナル状態が false であることを確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event.TryWait() で非シグナル状態であることを確認"));
        CheckBool( event.TryWait() == false );

        // TimerEventType にてセットし、TryWait で true を確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("eventType を使ってシグナル状態にセット"));
        nn::os::SignalTimerEvent(eventType);
        CheckBool( event.TryWait() == true );

        // TimerEventType にてクリアし、TryWait で false を確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("eventType を使って非シグナル状態にクリア"));
        nn::os::ClearTimerEvent(eventType);
        CheckBool( event.TryWait() == false );

        // operator TimerEventType&() の確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("operator TimerEventType&() を使ってオブジェクトの参照を取得"));
        nn::os::TimerEventType& eventTypeRefer = event;
        CheckBool( &eventTypeRefer == eventType );

        // operator const TimerEventType&() の確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("operator const TimerEventType&() を使ってオブジェクトの参照を取得"));
        const nn::os::TimerEventType& eventTypeReferConst = event;
        CheckBool( &eventTypeReferConst == eventType );
    }

    JUDGE_GOOGLE_TEST();
}


}}} // namespace nnt::os::timerEvent

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    int result;

    nn::os::ChangeThreadPriority( nn::os::GetCurrentThread(),
                                  nn::os::DefaultThreadPriority );

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of TimerEvent APIs\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of TimerEvent APIs\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}
