﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"


#include <nn/nn_Common.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/nn_Result.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"
#if !defined(NN_BUILD_CONFIG_OS_WIN32)
    #include <nn/svc/svc_ThreadLocalRegion.h>
    #include <nn/os/os_ThreadApi.h>
#endif
#include <climits>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

namespace nnt { namespace os { namespace tlv {

#if !defined(NN_BUILD_CONFIG_OS_WIN32)

static bool TestResult = true;

NN_ALIGNAS( 4096 ) char g_StackOfThread1[ 3 * 4096 ];
NN_ALIGNAS( 4096 ) char g_StackOfThread2[ 3 * 4096 ];
NN_ALIGNAS( 4096 ) char g_StackOfThread3[ 3 * 4096 ];

enum TestSyncPoint
{
    NNT_POINT_THREAD1_START             = 1,
    NNT_POINT_THREAD2_START,
    NNT_POINT_THREAD3_START,
    NNT_POINT_THREAD1_AFTER_SET_TLS,
    NNT_POINT_THREAD2_AFTER_SET_TLS,
    NNT_POINT_THREAD3_AFTER_SET_TLS,
    NNT_POINT_THREAD1_AFTER_GET_TLS,
    NNT_POINT_THREAD2_AFTER_GET_TLS,
    NNT_POINT_THREAD3_AFTER_GET_TLS,
};

//---------------------------------------------------------------------------

nn::os::ThreadType  g_Thread1;
nn::os::ThreadType  g_Thread2;
nn::os::ThreadType  g_Thread3;

const int   priThread1  = 8;
const int   priThread2  = 16;
const int   priThread3  = 24;

int                     GameMinus;
thread_local short      GamePlus;
thread_local int        GamePlus100 = 100;

static void GameUpdatePlusMinus()
{
    GameMinus -= 5;;
    GamePlus += 5;
    GamePlus100 += 5;
}


//---------------------------------------------------------------------------
//  TLS テスト用 Thread 関数
//---------------------------------------------------------------------------

void    ThreadFunc1(void* exinf)
{
    NN_UNUSED( exinf );
    int i;

    // タイミング合わせ
    SEQ_SYNC_WAIT( NNT_POINT_THREAD1_START );

    SEQ_CHECK(201);
    NNT_OS_LOG("Thread1:\n");

    {
        SEQ_CHECK(202);
        for (i = 0; i < 5; i++)
        {
            // normal global data
            GameMinus--;
            // thread local storage data
            GamePlus++;
            GamePlus100++;
            GameUpdatePlusMinus();
        }
    }

    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD2_START );
    SEQ_SYNC_WAIT( NNT_POINT_THREAD1_AFTER_SET_TLS );

    {
        SEQ_SET(500);
        if ((GamePlus != 30) || (GamePlus100 != 130))
        {
            NNT_OS_LOG("Thread1: GameMinus=%d.\n",
                GameMinus);
            NNT_OS_LOG("Thread1: GamePlus=%d.\n",
                GamePlus);
            NNT_OS_LOG("Thread1: GamePlus100=%d.\n",
                GamePlus100);
            TestResult = false;
        } else {
            NNT_OS_LOG("Thread1: Successful!\n");
        }
    }

    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD2_AFTER_SET_TLS );
    SEQ_SYNC_WAIT( NNT_POINT_THREAD1_AFTER_GET_TLS );

    SEQ_SET(590);
    NNT_OS_LOG("Just ExitThread in %s\n", __FUNCTION__);

    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD2_AFTER_GET_TLS );
}

void    ThreadFunc2(void* exinf)
{
    NN_UNUSED( exinf );
    int i;

    // タイミング合わせ
    SEQ_SYNC_WAIT( NNT_POINT_THREAD2_START );

    SEQ_CHECK(301);
    NNT_OS_LOG("Thread2:\n");

    {
        SEQ_CHECK(302);
        for (i = 0; i < 5; i++)
        {
            // normal global data
            GameMinus--;
            // thread local storage data
            GamePlus++;
            GamePlus100++;
            GameUpdatePlusMinus();
        }
    }

    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD3_START );
    SEQ_SYNC_WAIT( NNT_POINT_THREAD2_AFTER_SET_TLS );

    {
        SEQ_SET(510);
        if ((GamePlus != 30) || (GamePlus100 != 130))
        {
            NNT_OS_LOG("Thread2: GameMinus=%d.\n",
                GameMinus);
            NNT_OS_LOG("Thread2: GamePlus=%d.\n",
                GamePlus);
            NNT_OS_LOG("Thread2: GamePlus100=%d.\n",
                GamePlus100);
            TestResult = false;
        } else {
            NNT_OS_LOG("Thread2: Successful!\n");
        }
    }

    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD3_AFTER_SET_TLS );
    SEQ_SYNC_WAIT( NNT_POINT_THREAD2_AFTER_GET_TLS );

    SEQ_CHECK(593);
    NNT_OS_LOG("Just ExitThread in %s\n", __FUNCTION__);

    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD3_AFTER_GET_TLS );
}

void    ThreadFunc3(void* exinf)
{
    NN_UNUSED( exinf );
    int i;

    // タイミング合わせ
    SEQ_SYNC_WAIT( NNT_POINT_THREAD3_START );

    SEQ_CHECK(401);
    NNT_OS_LOG("Thread3:\n");

    {
        SEQ_CHECK(402);
        for (i = 0; i < 5; i++)
        {
            // normal global data
            GameMinus--;
            // thread local storage data
            GamePlus++;
            GamePlus100++;
            GameUpdatePlusMinus();
        }
    }

    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD1_AFTER_SET_TLS );
    SEQ_SYNC_WAIT( NNT_POINT_THREAD3_AFTER_SET_TLS );

    {
        SEQ_SET(520);
        if ((GamePlus != 30) || (GamePlus100 != 130))
        {
            NNT_OS_LOG("Thread3: GameMinus=%d.\n",
                GameMinus);
            NNT_OS_LOG("Thread3: GamePlus=%d.\n",
                GamePlus);
            NNT_OS_LOG("Thread3: GamePlus100=%d.\n",
                GamePlus100);
            TestResult = false;
        } else {
            NNT_OS_LOG("Thread3: Successful!\n");
        }
    }

    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD1_AFTER_GET_TLS );
    SEQ_SYNC_WAIT( NNT_POINT_THREAD3_AFTER_GET_TLS );

    SEQ_CHECK(598);
    NNT_OS_LOG("Just ExitThread in %s\n", __FUNCTION__);
}

//---------------------------------------------------------------------------
// スレッドの作成（ここで作成したスレッドを次の TEST で使う）
TEST(TestPreparationTlv, test_CreateThread)
{
    nn::Result  result;

    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // スレッド１の生成
    SEQ_SET(100);
    NNT_OS_LOG("Create Thread1:");
    result = nn::os::CreateThread(
                     &g_Thread1,                        // Thread オブジェクト
                     &ThreadFunc1,                      // エントリ関数
                     reinterpret_cast<void*>(0x5555),   // 引数
                     g_StackOfThread1,                  // スタック
                     sizeof(g_StackOfThread1),          // スタックサイズ
                     priThread1);                       // 優先度

    CheckBool( result.IsSuccess() );


    // スレッド２の生成
    SEQ_CHECK(101);
    NNT_OS_LOG("Create Thread2:");
    result = nn::os::CreateThread(
                     &g_Thread2,                        // Thread オブジェクト
                     &ThreadFunc2,                      // エントリ関数
                     reinterpret_cast<void*>(0xAAAA),   // 引数
                     g_StackOfThread2,                  // スタック
                     sizeof(g_StackOfThread2),          // スタックサイズ
                     priThread2);                       // 優先度

    CheckBool( result.IsSuccess() );


    // スレッド３の生成
    SEQ_CHECK(102);
    NNT_OS_LOG("Create Thread3:");
    result = nn::os::CreateThread(
                     &g_Thread3,                        // Thread オブジェクト
                     &ThreadFunc3,                      // エントリ関数
                     reinterpret_cast<void*>(0xBBBB),   // 引数
                     g_StackOfThread3,                  // スタック
                     sizeof(g_StackOfThread3),          // スタックサイズ
                     priThread3);                       // 優先度

    CheckBool( result.IsSuccess() );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// 各スレッドの動作開始
TEST(TestUsingTlv, test_StartThread)
{
    nn::Result  result;

    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // シーケンス制御の初期化
    SEQ_SYNC_INIT();

    // スレッド１の動作開始 ---------------
    SEQ_SET(200);
    NNT_OS_LOG("Start Thread1:");
    nn::os::StartThread( &g_Thread1 );
    CheckBool( true );

    // スレッド２の動作開始 ---------------
    NNT_OS_LOG("Start Thread2:");
    nn::os::StartThread( &g_Thread2 );
    CheckBool( true );

    // スレッド３の動作開始 ---------------
    NNT_OS_LOG("Start Thread3:");
    nn::os::StartThread( &g_Thread3 );
    CheckBool( true );

    // テスト開始
    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD1_START );

    // スレッドの終了待ち
    nn::os::WaitThread( &g_Thread1 );
    nn::os::WaitThread( &g_Thread2 );
    nn::os::WaitThread( &g_Thread3 );

    CheckBool( TestResult );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// スレッドの破棄
TEST(TestPostProcessTlv, test_DestroyThread)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // スレッド１の破棄
    SEQ_SET(600);
    NNT_OS_LOG("Destroy Thread1:");
    nn::os::DestroyThread( &g_Thread1 );
    CheckBool( true );

    // スレッド２の破棄
    SEQ_CHECK(601);
    NNT_OS_LOG("Destroy Thread2:");
    nn::os::DestroyThread( &g_Thread2 );
    CheckBool( true );

    // スレッド３の破棄
    SEQ_CHECK(602);
    NNT_OS_LOG("Destroy Thread3:");
    nn::os::DestroyThread( &g_Thread3 );
    CheckBool( true );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

#else

TEST(TestPreparationTlv, test_Win32Unsupported)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // スレッド１の生成
    SEQ_SET(100);

    CheckBool( true );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

#endif // #if !defined(NN_BUILD_CONFIG_OS_WIN32)

}}} // namespace nnt::os::tlv

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    int result;

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of ThreadLocalVariables\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of ThreadLocalVariables\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}

