﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Thread 関連機能のテスト
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/util/util_BitUtil.h>
#include <nn/os.h>
#include <nn/os/os_SdkThread.h>
#include <nn/os/os_SdkThreadInfoApi.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/os/os_SharedMemory.h>
#include <nn/os/os_Thread.private.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"
#include "../Common/test_MemoryAccess.h"
#include "../Thread/test_ThreadHelper.h"

#include <atomic>
#include <mutex>
#include <thread>
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#include <future>
#endif
#include <cstring>
#if defined(NN_BUILD_CONFIG_TOOLCHAIN_CLANG)
#include <cfenv>
#endif

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

namespace nnt { namespace os { namespace thread {

//---------------------------------------------------------------------------
// 作成可能なスレッド数の最大数
const int CreatableThreadMax = 64;

//---------------------------------------------------------------------------
// この NN_OS_ALIGNAS_THREAD_STACK マクロによって正しく 4KB アライメント
// されているか否かは CreateThread() が成功することで分かる。
//
NN_OS_ALIGNAS_THREAD_STACK  char    g_ThreadStack[16384];

//---------------------------------------------------------------------------
// 自スレッドに単独の ChangeThreadPriority() 発行

TEST(ChangeThreadPriority1, test_ChangeThreadPriority15)
{
    // 自スレッドに単独の ChangeThreadPriority(-12) 実行
    doTest1(24,                     // Thread1 の初期優先度
            THR_CALL_CHG_PRI_SYS_HIGH,// Thread1 は ChangeThreadPriority(-12) 発行
            true,                   // Thread1 での API 返値
            nn::os::HighestSystemThreadPriority,
                                    // Thread1 の優先度期待値（変更される）
            THR_STATE_EXITED);      // Thread1 は終了
}

TEST(ChangeThreadPriority1, test_ChangeThreadPriority16)
{
    // 自スレッドに単独の ChangeThreadPriority(35) 実行
    doTest1(24,                     // Thread1 の初期優先度
            THR_CALL_CHG_PRI_SYS_LOW,// Thread1 は ChangeThreadPriority(35) 発行
            true,                   // Thread1 での API 返値
            nn::os::LowestSystemThreadPriority,
                                    // Thread1 の優先度期待値（変更される）
            THR_STATE_EXITED);      // Thread1 は終了
}


//---------------------------------------------------------------------------
// 他スレッドに単独の ChangeThreadPriority() 発行

TEST(ChangeThreadPriority2, test_ChangeThreadPriority25)
{
    // 他スレッドに単独の ChangeThreadPriority(-12) 実行
    doTest2(24,                     // Thread1 の初期優先度
            THR_WAIT_SLEEP,         // Thread1 は SleepThread() 発行
            THR_CALL_CHG_PRI_SYS_HIGH,// Thread2 は ChangeThreadPriority(-12) 発行
            false,                  // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            nn::os::HighestSystemThreadPriority,
                                    // Thread1 の優先度期待値（変更される）
            THR_STATE_SLEEP,        // Thread1 は待ち→終了
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(ChangeThreadPriority2, test_ChangeThreadPriority26)
{
    // 他スレッドに単独の ChangeThreadPriority(35) 実行
    doTest2(24,                     // Thread1 の初期優先度
            THR_WAIT_SLEEP,         // Thread1 は SleepThread() 発行
            THR_CALL_CHG_PRI_SYS_LOW,// Thread2 は ChangeThreadPriority(35) 発行
            false,                  // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            nn::os::LowestSystemThreadPriority,
                                    // Thread1 の優先度期待値（変更される）
            THR_STATE_SLEEP,        // Thread1 は待ち→終了
            THR_STATE_EXITED);      // Thread2 は終了
}


//---------------------------------------------------------------------------
// CreateThread() でスレッド優先度の上限と下限を設定し、
// 生成されたスレッドでの GetThreadPriority() の結果を確認する。
void
testFunc_GetThreadPriorityThread(void* argument)
{
    int*  p = static_cast<int*>( argument );

    *p = nn::os::GetThreadPriority( nn::os::GetCurrentThread() );

    // スレッドは終了
}

TEST(OtherApis, test_CreateThreadWithLimitThreadPriority)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // まず HighestSystemThreadPriority を指定して CreateThread する。
    int priority = nn::os::HighestSystemThreadPriority;
    {
        nn::os::ThreadType  thread;
        NNT_OS_LOG("CreateThread( priority=%d ):", priority);

        int value = 9999;
        nn::Result  result = nn::os::CreateThread( &thread, testFunc_GetThreadPriorityThread, &value, g_ThreadStack, sizeof(g_ThreadStack), priority);
        CheckBool( result.IsSuccess() );

        nn::os::StartThread( &thread );
        nn::os::WaitThread( &thread );
        nn::os::DestroyThread( &thread );

        // value に代入されている GetThreadPriority() の値を精査
        NNT_OS_LOG("GetThreadPriority() = %d", value);
        CheckBool( value == priority );
    }

    // 次に LowestSystemThreadPriority を指定して CreateThread する。
    priority = nn::os::LowestSystemThreadPriority;
    {
        nn::os::ThreadType  thread;
        NNT_OS_LOG("CreateThread( priority=%d ):", priority);

        int value = 9999;
        nn::Result  result = nn::os::CreateThread( &thread, testFunc_GetThreadPriorityThread, &value, g_ThreadStack, sizeof(g_ThreadStack), priority);
        CheckBool( result.IsSuccess() );

        nn::os::StartThread( &thread );
        nn::os::WaitThread( &thread );
        nn::os::DestroyThread( &thread );

        // value に代入されている GetThreadPriority() の値を精査
        NNT_OS_LOG("GetThreadPriority() = %d", value);
        CheckBool( value == priority );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
// 単独の GetCurrentCoreNumber() 発行

// メインスレッドに対して GetCurrentCoreNumber() を呼出して、
// とりあえず値が返ってこれば OK とする。
TEST(OtherApis, test_GetCurrentCoreNumber)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // 値の精査までは行なわない。
    NNT_OS_LOG("GetCurrentCoreNumber() = %d", nn::os::GetCurrentCoreNumber());
    CheckBool( true );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


// CreateThread でスレッドを生成する際に processorNumber を指定せず、
// GetCurrentCoreNumber() で 1 が返ってくるかのテスト（実機でのみ値をテスト）。
void
testFunc_GetCurrentCoreNumberThread(void* argument)
{
    volatile int* pValue = reinterpret_cast<volatile int*>( argument );

    *pValue = nn::os::GetCurrentCoreNumber();

    // スレッドは終了
}

TEST(OtherApis, test_GetCurrentCoreNumberDefaultValue)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // value には volatile を付けておく。理由は以下の通り。
    // value の値は子スレッド側で書替えられ、後続の WaitThread() が完了した後で
    // 本関数から参照する必要があるが、コード生成上、CreateThread() が完了した
    // 段階で value を参照して CPU レジスタに乗せてしまう可能性がある。
    // この、コンパイラの最適化を抑止するため、volatile を付けておく。
    volatile int        value = -1;  // 無効な値
    nn::os::ThreadType  thread;
    NNT_OS_LOG("CreateThread( processorNumber=default ):");

    // まず CreateThread する。
    // 本テストではプロセッサ番号は指定せず、デフォルト値に従う。
    // const_cast<> は volatile を外すために必要。
    nn::Result  result = nn::os::CreateThread( &thread, testFunc_GetCurrentCoreNumberThread, const_cast<int*>(&value), g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority);
    CheckBool( result.IsSuccess() );

    nn::os::StartThread( &thread );
    nn::os::WaitThread( &thread );
    nn::os::DestroyThread( &thread );

    // value に代入されている GetCurrentCoreNumber() の値を精査
    NNT_OS_LOG("GetCurrentCoreNumber() = %d", value);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    // 実ターゲット環境ではコア番号 1 で動くことを期待
    CheckBool( value == 1 );
#endif

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
// CreateThread でスレッドを生成する際に processorNumber に 0 を指定し、
// GetCurrentCoreNumber() で同じコア番号が返ってくるかのテスト。
// プロセッサ番号 0 に大きな意味はなく、また実際にどのプロセッサで動作
// しているかはテストの範疇ではない。
TEST(OtherApis, test_GetCurrentCoreNumberSpecifyProcessorNumber)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // value には volatile を付けておく。
    volatile int        value = -1;  // 無効な値
    nn::os::ThreadType  thread;
    NNT_OS_LOG("CreateThread( processorNumber=0 ):");

    // まず CreateThread する。プロセッサ番号は 0 を指定する。
    nn::Result  result = nn::os::CreateThread( &thread, testFunc_GetCurrentCoreNumberThread, const_cast<int*>(&value), g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority, 0);
    CheckBool( result.IsSuccess() );

    nn::os::StartThread( &thread );
    nn::os::WaitThread( &thread );
    nn::os::DestroyThread( &thread );

    // value に代入されている GetCurrentCoreNumber() の値を精査
    NNT_OS_LOG("GetCurrentCoreNumber() = %d", value);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    CheckBool( value == 0 );
#endif

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


//-----------------------------------------------------------------------------
// WaitThread の排他漏れをチェックするテスト（SIGLO-10339）
//

const size_t StackSize = 32768;

struct SystemAnalysisInfo
{
    int coreNums;
    int creatableThreadNums;
};

void ThreadFunction2ForWaitThreadOnMultiCore(void *arg)
{
    NN_UNUSED(arg);
}

void ThreadFunction1ForWaitThreadOnMultiCore(void *arg)
{
    auto systemInfo = reinterpret_cast<SystemAnalysisInfo*>(arg);
    int  coreNums            = systemInfo->coreNums;
    int  creatableThreadNums = systemInfo->creatableThreadNums;
    auto coreIndex = nn::os::GetCurrentCoreNumber();
    NN_UNUSED(coreIndex);

    uintptr_t stackBlock;
    int threadNums = creatableThreadNums / coreNums;
    const size_t stackBlockSize = nn::util::align_up(StackSize * threadNums, nn::os::MemoryBlockUnitSize);
    auto result = nn::os::AllocateMemoryBlock(&stackBlock, stackBlockSize);
    EXPECT_TRUE( result.IsSuccess() );

    for (int k=0; k<3; ++k)
    {
        NNT_OS_LOG("passed: core=%d loop=%d\n", coreIndex, k);

        for (int j=0; j<100; ++j)
        {
            NN_SDK_ASSERT(threadNums < CreatableThreadMax);
            nn::os::ThreadType threads[CreatableThreadMax];
            for (int i = 0; i < threadNums; ++i)
            {
                uintptr_t stackTop = stackBlock + (i * StackSize);
retry_loop:
                auto ret = nn::os::CreateThread(
                                    &threads[i],
                                    ThreadFunction2ForWaitThreadOnMultiCore,
                                    NULL,
                                    reinterpret_cast<void*>(stackTop),
                                    StackSize,
                                    nn::os::DefaultThreadPriority,
                                    (i % coreNums));

                if (!ret.IsSuccess())
                {
                    if (ret <= nn::os::ResultOutOfResource())
                    {
                        NNT_OS_LOG("=== Detect nn::os::ResultOutOfResource on core=%d. Retry.\n", coreIndex);
                        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1000) );

                        goto retry_loop;
                    }
                    else
                    {
                        NN_ABORT_UNLESS_RESULT_SUCCESS(ret);
                    }
                }
            }

            for(int i = 0; i < threadNums; ++i)
            {
                nn::os::StartThread(&threads[i]);
            }
            for(int i = 0; i < threadNums; ++i)
            {
                nn::os::WaitThread(&threads[i]);
            }
            for(int i = 0; i < threadNums; ++i)
            {
                nn::os::DestroyThread(&threads[i]);
            }
        }
    }
    nn::os::FreeMemoryBlock(stackBlock, nn::os::MemoryBlockUnitSize);
}

nn::os::ThreadType g_CoreThreads[ CreatableThreadMax ];
NN_OS_ALIGNAS_THREAD_STACK char g_CountThreadStack[CreatableThreadMax][4096];

int GetCreatableThreadCount()
{
    int creatableThreadNums = CreatableThreadMax;
    for (int i=0; i<CreatableThreadMax; ++i)
    {
        auto result = nn::os::CreateThread( &g_CoreThreads[i], ThreadFunction2ForWaitThreadOnMultiCore, NULL, g_CountThreadStack[i], 4096, nn::os::DefaultThreadPriority);
        if (!result.IsSuccess())
        {
            creatableThreadNums = i;
            break;
        }
    }

    for (int i=0; i<creatableThreadNums; ++i)
    {
        nn::os::DestroyThread( &g_CoreThreads[i] );
    }

    return creatableThreadNums;
}

TEST(WaitThreadOnMultiCore, test_WaitThreadOnMultiCore)
{
    NNT_OS_LOG("Begin WaitThread() on multi-core test.\n");

    // 搭載コア数を算出（最下位の 1 のビット数をカウントしてコア数とする）
    nn::Bit64 coreMask = nn::os::GetThreadAvailableCoreMask();
              coreMask = nn::util::maskt1<nn::Bit64>(coreMask);
    int coreNums = nn::util::cntt0<nn::Bit64>(~coreMask);
    NNT_OS_LOG("Core nums = %d\n", coreNums);

    // 作成可能なスレッド数をカウント
    int creatableThreadNums = GetCreatableThreadCount() - coreNums;
    NNT_OS_LOG("Creatable thread nums = %d\n", creatableThreadNums);

    SystemAnalysisInfo  systemInfo;
    systemInfo.coreNums             = coreNums;
    systemInfo.creatableThreadNums  = creatableThreadNums;

    // ヒープからスタック用領域を確保
    auto result = nn::os::SetMemoryHeapSize(nn::os::MemoryBlockUnitSize * 2 * (coreNums + 1));
    EXPECT_TRUE( result.IsSuccess() );

    uintptr_t stackBlock;
    result = nn::os::AllocateMemoryBlock(&stackBlock, nn::os::MemoryBlockUnitSize);
    EXPECT_TRUE( result.IsSuccess() );

    // テスト用のスレッドを生成
    for (int i = 0; i < coreNums; ++i)
    {
        auto ret = nn::os::CreateThread(
                        &g_CoreThreads[i],
                        ThreadFunction1ForWaitThreadOnMultiCore,
                        &systemInfo,
                        reinterpret_cast<void*>(stackBlock + (i * StackSize)),
                        StackSize,
                        nn::os::DefaultThreadPriority,
                        i);
        EXPECT_TRUE( ret.IsSuccess() );
    }

    for(int i = 0; i < coreNums; ++i)
    {
        nn::os::StartThread(&g_CoreThreads[i]);
    }

    for(int i = 0; i < coreNums; ++i)
    {
        nn::os::WaitThread(&g_CoreThreads[i]);
    }

    for(int i = 0; i < coreNums; ++i)
    {
        nn::os::DestroyThread(&g_CoreThreads[i]);
    }

    // ヒープを返却し、0 に戻しておく
    nn::os::FreeMemoryBlock(stackBlock, nn::os::MemoryBlockUnitSize);
    nn::os::SetMemoryHeapSize(0);

    NNT_OS_LOG("End WaitThread() on multi-core test.\n");
}


}}} // namespace nnt::os::thread

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of Thread APIs\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    int result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of Thread APIs\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}
