﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Thread 関連機能のテスト
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/util/util_BitUtil.h>
#include <nn/os.h>
#include <nn/os/os_SdkThread.h>
#include <nn/os/os_SdkThreadInfoApi.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/os/os_SharedMemory.h>
#include <nn/os/os_Thread.private.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"
#include "../Common/test_MemoryAccess.h"
#include "test_ThreadHelper.h"

#include <atomic>
#include <mutex>
#include <thread>
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#include <future>
#endif
#include <cstring>
#if defined(NN_BUILD_CONFIG_TOOLCHAIN_CLANG)
#include <cfenv>
#endif

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

namespace nnt { namespace os { namespace thread {

//---------------------------------------------------------------------------
// メインスレッドを除いて作成可能なスレッド数
const int NumManyThread = 32 - 1;

//---------------------------------------------------------------------------
// この NN_OS_ALIGNAS_THREAD_STACK マクロによって正しく 4KB アライメント
// されているか否かは CreateThread() が成功することで分かる。
//
NN_OS_ALIGNAS_THREAD_STACK  char    g_ThreadStack[16384];

//---------------------------------------------------------------------------
// 単独の WaitThread( thread1 ) 発行

#if 0
TEST(WaitThread, test_WaitThread1)
{
    // 未起動の Thread1 に対して WaitThread() 実行
    doTest2(24,                         // Thread1 の初期優先度
            THR_NOTUSE,                 // Thread1 は未使用
            THR_CALL_WAIT,              // Thread2 は WaitThread() 発行
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            24,                         // Thread1 の優先度期待値
            THR_STATE_NONE,             // Thread1 は未使用
            THR_STATE_EXITED);          // Thread2 は終了
}
#endif

#if 0
TEST(WaitThread, test_WaitThread2)
{
    // 自スレッドに対して WaitThread() 実行（OS ライブラリ内部で停止）
    doTest1(24,                         // Thread1の初期優先度
            THR_CALL_WAIT,              // Thread1 は WaitThread() 発行
            true,                       // Thread1 での API 返値
            24,                         // Thread1 の優先度期待値
            THR_STATE_EXITED);          // Thread1 は終了
}
#endif


//---------------------------------------------------------------------------
// 単独の SleepThread() 発行

TEST(SleepThread, test_SleepThread1)
{
    // 単独の SleepThread() 実行
    doTest1(24,                         // Thread1 の初期優先度
            THR_WAIT_SLEEP,             // Thread1 は SleepThread() 発行
            false,                      // Thread1 での API 返値
            24,                         // Thread1 の優先度期待値
            THR_STATE_SLEEP);           // Thread1 は終了
}


//---------------------------------------------------------------------------
// 複合の Sleep & WaitThread( thread1 ) 発行

TEST(CombinedSleepAndWait, test_SleepAndWait)
{
    // Thread1 で Sleep() し、Exit するまで Thread2 で WaitThread() 実行
    doTest2(24,                         // Thread1 の初期優先度
            THR_WAIT_SLEEP,             // Thread1 は SleepThread() 発行
            THR_CALL_WAIT,              // Thread2 は WaitThread() 発行
            false,                      // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            24,                         // Thread1 の優先度期待値
            THR_STATE_SLEEP,            // Thread1 は待ち→終了
            THR_STATE_SLEEP);           // Thread2 は終了
}

//---------------------------------------------------------------------------
// 単独の SetThreadName() 発行

TEST(SetThreadName, test_SetThreadName)
{
    // 単独の SetThreadName() 実行
    doTest1(24,                         // Thread1 の初期優先度
            THR_CALL_SET_NAME,          // Thread1 は SetThreadName() 発行
            true,                       // Thread1 での API 返値
            24,                         // Thread1 の優先度期待値
            THR_STATE_EXITED);          // Thread1 は終了
}


//---------------------------------------------------------------------------
// 単独の GetThreadName() 発行

TEST(GetThreadName, test_GetThreadName)
{
    // 単独の GetThreadName() 実行
    doTest1(24,                         // Thread1 の初期優先度
            THR_CALL_GET_NAME_DFLT,     // Thread1 は GetThreadName() 発行
            true,                       // Thread1 での API 返値
            24,                         // Thread1 の優先度期待値
            THR_STATE_EXITED);          // Thread1 は終了
}

//---------------------------------------------------------------------------
// 複合の SetThreadName() & GetThreadName() 発行

TEST(SetAndGetThreadName, test_SetAndGetThreadName)
{
    // SetThreadName() したスレッド名を GetThreadName() で取得、検証
    doTest2(24,                         // Thread1 の初期優先度
            THR_CALL_SET_NAME,          // Thread1 は SetThreadName() 発行
            THR_CALL_GET_NAME_SET,      // Thread2 は GetThreadName() 発行
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            24,                         // Thread1 の優先度期待値
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は終了
}

//---------------------------------------------------------------------------
// 自スレッドに単独の ChangeThreadPriority() 発行

#if 0
TEST(ChangeThreadPriority1, test_ChangeThreadPriority11)
{
    // 自スレッドに単独の ChangeThreadPriority(-1) 実行
    doTest1(24,                     // Thread1 の初期優先度
            THR_CALL_CHG_PRI_UDR,   // Thread1 は ChangeThreadPriority(-1) 発行
            false,                  // Thread1 での API 返値
            24,                     // Thread1 の優先度期待値（変更されない）
            THR_STATE_EXITED);      // Thread1 は終了
}
#endif

TEST(ChangeThreadPriority1, test_ChangeThreadPriority12)
{
    // 自スレッドに単独の ChangeThreadPriority(0) 実行
    doTest1(24,                     // Thread1 の初期優先度
            THR_CALL_CHG_PRI_HIGH,  // Thread1 は ChangeThreadPriority(0) 発行
            true,                   // Thread1 での API 返値
            nn::os::HighestThreadPriority,
                                    // Thread1 の優先度期待値（変更される）
            THR_STATE_EXITED);      // Thread1 は終了
}

TEST(ChangeThreadPriority1, test_ChangeThreadPriority13)
{
    // 自スレッドに単独の ChangeThreadPriority(31) 実行
    doTest1(24,                     // Thread1 の初期優先度
            THR_CALL_CHG_PRI_LOW,   // Thread1 は ChangeThreadPriority(31) 発行
            true,                   // Thread1 での API 返値
            nn::os::LowestThreadPriority,
                                    // Thread1 の優先度期待値（変更される）
            THR_STATE_EXITED);      // Thread1 は終了
}

#if 0
TEST(ChangeThreadPriority1, test_ChangeThreadPriority14)
{
    // 自スレッドに単独の ChangeThreadPriority(32) 実行
    doTest1(24,                     // Thread1 の初期優先度
            THR_CALL_CHG_PRI_OVR,   // Thread1 は ChangeThreadPriority(32) 発行
            false,                  // Thread1 での API 返値
            24,                     // Thread1 の優先度期待値（変更されない）
            THR_STATE_EXITED);      // Thread1 は終了
}
#endif

//---------------------------------------------------------------------------
// 他スレッドに単独の ChangeThreadPriority() 発行

#if 0
TEST(ChangeThreadPriority2, test_ChangeThreadPriority21)
{
    // 他スレッドに単独の ChangeThreadPriority(-1) 実行
    doTest2(24,                     // Thread1 の初期優先度
            THR_WAIT_SLEEP,         // Thread1 は SleepThread() 発行
            THR_CALL_CHG_PRI_UDR,   // Thread2 は ChangeThreadPriority(-1) 発行
            false,                  // Thread1 での API 返値
            false,                  // Thread2 での API 返値
            24,                     // Thread1 の優先度期待値（変更されない）
            THR_STATE_SLEEP,        // Thread1 は待ち→終了
            THR_STATE_EXITED);      // Thread2 は終了
}
#endif

TEST(ChangeThreadPriority2, test_ChangeThreadPriority22)
{
    // 他スレッドに単独の ChangeThreadPriority(0) 実行
    doTest2(24,                     // Thread1 の初期優先度
            THR_WAIT_SLEEP,         // Thread1 は SleepThread() 発行
            THR_CALL_CHG_PRI_HIGH,  // Thread2 は ChangeThreadPriority(0) 発行
            false,                  // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            nn::os::HighestThreadPriority,
                                    // Thread1 の優先度期待値（変更される）
            THR_STATE_SLEEP,        // Thread1 は待ち→終了
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(ChangeThreadPriority2, test_ChangeThreadPriority23)
{
    // 他スレッドに単独の ChangeThreadPriority(31) 実行
    doTest2(24,                     // Thread1 の初期優先度
            THR_WAIT_SLEEP,         // Thread1 は SleepThread() 発行
            THR_CALL_CHG_PRI_LOW,   // Thread2 は ChangeThreadPriority(31) 発行
            false,                  // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            nn::os::LowestThreadPriority,
                                    // Thread1 の優先度期待値（変更される）
            THR_STATE_SLEEP,        // Thread1 は待ち→終了
            THR_STATE_EXITED);      // Thread2 は終了
}

#if 0
TEST(ChangeThreadPriority2, test_ChangeThreadPriority24)
{
    // 他スレッドに単独の ChangeThreadPriority(32) 実行
    doTest2(24,                     // Thread1 の初期優先度
            THR_WAIT_SLEEP,         // Thread1 は SleepThread() 発行
            THR_CALL_CHG_PRI_OVR,   // Thread2 は ChangeThreadPriority(32) 発行
            false,                  // Thread1 での API 返値
            false,                  // Thread2 での API 返値
            24,                     // Thread1 の優先度期待値（変更されない）
            THR_STATE_SLEEP,        // Thread1 は待ち→終了
            THR_STATE_EXITED);      // Thread2 は終了
}
#endif

//---------------------------------------------------------------------------
// 単独の GetThreadCurrentPriority() 発行

#if 0
TEST(GetThreadCurrentPriority, test_GetThreadCurrentPriority1)
{
    // 他スレッドに単独の GetThreadCurrentPriority(-1) 実行
    doTest2(24,                     // Thread1 の初期優先度
            THR_CALL_CHG_PRI_UDR,   // Thread1 は ChangeThreadPriority(-1) 発行
            THR_CALL_GET_PRI,       // Thread2 は GetThreadCurrentPriority() 発行
            false,                  // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            24,                     // Thread1 の優先度期待値（変更されない）
            THR_STATE_WAITING2,     // Thread1 は待ち→終了
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(GetThreadCurrentPriority, test_GetThreadCurrentPriority2)
{
    // 他スレッドに単独の GetThreadCurrentPriority(0) 実行
    doTest2(24,                     // Thread1 の初期優先度
            THR_CALL_CHG_PRI_MIN,   // Thread1 は ChangeThreadPriority(0) 発行
            THR_CALL_GET_PRI,       // Thread2 は GetThreadCurrentPriority() 発行
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            0,                      // Thread1 の優先度期待値（変更される）
            THR_STATE_WAITING2,     // Thread1 は待ち→終了
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(GetThreadCurrentPriority, test_GetThreadCurrentPriority3)
{
    // 他スレッドに単独の GetThreadCurrentPriority(31) 実行
    doTest2(24,                     // Thread1 の初期優先度
            THR_CALL_CHG_PRI_MAX,   // Thread1 は ChangeThreadPriority(31) 発行
            THR_CALL_GET_PRI,       // Thread2 は GetThreadCurrentPriority() 発行
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            31,                     // Thread1 の優先度期待値（変更される）
            THR_STATE_WAITING2,     // Thread1 は待ち→終了
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(GetThreadCurrentPriority, test_GetThreadCurrentPriority4)
{
    // 他スレッドに単独の GetThreadCurrentPriority(32) 実行
    doTest2(24,                     // Thread1 の初期優先度
            THR_CALL_CHG_PRI_OVR,   // Thread1 は ChangeThreadPriority(32) 発行
            THR_CALL_GET_PRI,       // Thread2 は GetThreadCurrentPriority() 発行
            false,                  // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            24,                     // Thread1 の優先度期待値（変更されない）
            THR_STATE_WAITING2,     // Thread1 は待ち→終了
            THR_STATE_EXITED);      // Thread2 は終了
}
#endif



//---------------------------------------------------------------------------
// 単独の YieldThread() 発行

TEST(YieldThreadPriority, test_YieldThread1)
{
    // Thread1 で単独の YieldThread() 実行
    doTest1(24,                     // Thread1 の初期優先度
            THR_CALL_YLD,           // Thread1 は YieldThread(32) 発行
            true,                   // Thread1 での API 返値
            24,                     // Thread1 の優先度期待値（変更されない）
            THR_STATE_EXITED);      // Thread1 は終了
}

TEST(GetThreadCurrentPriority, test_YieldThread2)
{
    // Thread2 で単独の YieldThread() 実行
    doTest2(24,                     // Thread1 の初期優先度
            THR_NOTUSE,             // Thread1 未使用
            THR_CALL_YLD,           // Thread2 は YieldThread() 発行
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            24,                     // Thread1 の優先度期待値（変更されない）
            THR_STATE_NONE,         // Thread1 未使用
            THR_STATE_EXITED);      // Thread2 は終了
}

//---------------------------------------------------------------------------
// GetThreadId() の呼出し、および、パフォーマンステスト
//
void CheckPerformanceOfGetThreadId(nn::os::ThreadType* thread)
{
    // 計測前にダミーで 1 回呼んでおく（キャッシュに載せる）
    auto threadId = nn::os::GetThreadId(thread);
    EXPECT_TRUE( threadId != 0ull );

    // 1000 回呼んでその合計時間を計測
    auto startTick = nn::os::GetSystemTick();
    for (int i=0; i<100; ++i)
    {
        nn::os::GetThreadId(thread);
        nn::os::GetThreadId(thread);
        nn::os::GetThreadId(thread);
        nn::os::GetThreadId(thread);
        nn::os::GetThreadId(thread);    // 5

        nn::os::GetThreadId(thread);
        nn::os::GetThreadId(thread);
        nn::os::GetThreadId(thread);
        nn::os::GetThreadId(thread);
        nn::os::GetThreadId(thread);    // 10
    }
    auto endTick = nn::os::GetSystemTick();
    NN_LOG("nn::os::GetThreadId(TID=%lld) x 1000: elapse = %d(usec)\n", threadId, (endTick - startTick).ToTimeSpan().GetMicroSeconds());

    // 最初に取得した時と同じ値かのチェック
    EXPECT_TRUE( threadId == nn::os::GetThreadId(thread) );
}

void
testFunc_GetThreadId(void* argument)
{
    NN_UNUSED(argument);
    CheckPerformanceOfGetThreadId( nn::os::GetCurrentThread() );
}

TEST(GetThreadid, test_GetThreadId)
{
    CheckPerformanceOfGetThreadId( nn::os::GetCurrentThread() );

    {
        nn::os::ThreadType thread;

        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &thread, testFunc_GetThreadId, nullptr, g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority) );

        // StartThread 前に ThreadId を確認
        CheckPerformanceOfGetThreadId( &thread );
        nn::os::StartThread( &thread );

        // WaitThread 前に ThreadId を確認
        CheckPerformanceOfGetThreadId( &thread );
        nn::os::WaitThread( &thread );

        // DestroyThread 前に ThreadId を確認
        CheckPerformanceOfGetThreadId( &thread );
        nn::os::DestroyThread( &thread );
    }
}


//---------------------------------------------------------------------------
// CreateThread() でスレッド優先度の上限と下限を設定し、
// 生成されたスレッドでの GetThreadPriority() の結果を確認する。
void
testFunc_GetThreadPriorityThread(void* argument)
{
    int*  p = static_cast<int*>( argument );

    *p = nn::os::GetThreadPriority( nn::os::GetCurrentThread() );

    // スレッドは終了
}

TEST(OtherApis, test_CreateThreadWithLimitThreadPriority)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // まず HighestThreadPriority を指定して CreateThread する。
    int priority = nn::os::HighestThreadPriority;
    {
        nn::os::ThreadType  thread;
        NNT_OS_LOG("CreateThread( priority=%d ):", priority);

        int value = 9999;
        nn::Result  result = nn::os::CreateThread( &thread, testFunc_GetThreadPriorityThread, &value, g_ThreadStack, sizeof(g_ThreadStack), priority);
        CheckBool( result.IsSuccess() );

        nn::os::StartThread( &thread );
        nn::os::WaitThread( &thread );
        nn::os::DestroyThread( &thread );

        // value に代入されている GetThreadPriority() の値を精査
        NNT_OS_LOG("GetThreadPriority() = %d", value);
        CheckBool( value == priority );
    }

    // 次に LowestThreadPriority を指定して CreateThread する。
    priority = nn::os::LowestThreadPriority;
    {
        nn::os::ThreadType  thread;
        NNT_OS_LOG("CreateThread( priority=%d ):", priority);

        int value = 9999;
        nn::Result  result = nn::os::CreateThread( &thread, testFunc_GetThreadPriorityThread, &value, g_ThreadStack, sizeof(g_ThreadStack), priority);
        CheckBool( result.IsSuccess() );

        nn::os::StartThread( &thread );
        nn::os::WaitThread( &thread );
        nn::os::DestroyThread( &thread );

        // value に代入されている GetThreadPriority() の値を精査
        NNT_OS_LOG("GetThreadPriority() = %d", value);
        CheckBool( value == priority );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
// 単独の GetCurrentCoreNumber() 発行

// メインスレッドに対して GetCurrentCoreNumber() を呼出して、
// とりあえず値が返ってこれば OK とする。
TEST(OtherApis, test_GetCurrentCoreNumber)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // 値の精査までは行なわない。
    NNT_OS_LOG("GetCurrentCoreNumber() = %d", nn::os::GetCurrentCoreNumber());
    CheckBool( true );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


// CreateThread でスレッドを生成する際に processorNumber を指定せず、
// GetCurrentCoreNumber() で 1 が返ってくるかのテスト（実機でのみ値をテスト）。
void
testFunc_GetCurrentCoreNumberThread(void* argument)
{
    volatile int* pValue = reinterpret_cast<volatile int*>( argument );

    *pValue = nn::os::GetCurrentCoreNumber();

    // スレッドは終了
}

TEST(OtherApis, test_GetCurrentCoreNumberDefaultValue)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // value には volatile を付けておく。理由は以下の通り。
    // value の値は子スレッド側で書替えられ、後続の WaitThread() が完了した後で
    // 本関数から参照する必要があるが、コード生成上、CreateThread() が完了した
    // 段階で value を参照して CPU レジスタに乗せてしまう可能性がある。
    // この、コンパイラの最適化を抑止するため、volatile を付けておく。
    volatile int        value = -1;  // 無効な値
    nn::os::ThreadType  thread;
    NNT_OS_LOG("CreateThread( processorNumber=default ):");

    // まず CreateThread する。
    // 本テストではプロセッサ番号は指定せず、デフォルト値に従う。
    // const_cast<> は volatile を外すために必要。
    nn::Result  result = nn::os::CreateThread( &thread, testFunc_GetCurrentCoreNumberThread, const_cast<int*>(&value), g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority);
    CheckBool( result.IsSuccess() );

    nn::os::StartThread( &thread );
    nn::os::WaitThread( &thread );
    nn::os::DestroyThread( &thread );

    // value に代入されている GetCurrentCoreNumber() の値を精査
    NNT_OS_LOG("GetCurrentCoreNumber() = %d", value);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    // 実ターゲット環境ではコア番号 0 で動くことを期待
    CheckBool( value == 0 );
#endif

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
// CreateThread でスレッドを生成する際に processorNumber に 0 を指定し、
// GetCurrentCoreNumber() で同じコア番号が返ってくるかのテスト。
// プロセッサ番号 0 に大きな意味はなく、また実際にどのプロセッサで動作
// しているかはテストの範疇ではない。
TEST(OtherApis, test_GetCurrentCoreNumberSpecifyProcessorNumber)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // value には volatile を付けておく。
    volatile int        value = -1;  // 無効な値
    nn::os::ThreadType  thread;
    NNT_OS_LOG("CreateThread( processorNumber=0 ):");

    // まず CreateThread する。プロセッサ番号は 0 を指定する。
    nn::Result  result = nn::os::CreateThread( &thread, testFunc_GetCurrentCoreNumberThread, const_cast<int*>(&value), g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority, 0);
    CheckBool( result.IsSuccess() );

    nn::os::StartThread( &thread );
    nn::os::WaitThread( &thread );
    nn::os::DestroyThread( &thread );

    // value に代入されている GetCurrentCoreNumber() の値を精査
    NNT_OS_LOG("GetCurrentCoreNumber() = %d", value);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    CheckBool( value == 0 );
#endif

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
// CreateThread でスレッドを生成する際に processorNumber に存在しないコア番号を指定し、
// 正しくアサートに引っかかるかテスト
TEST(OtherApisDeathTest, test_InvalidProcessorNumberAtCreateThread)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // value には volatile を付けておく。
    volatile int        value = -1;  // 無効な値
    int                 invalidCoreNumber = 63;  // 存在しない（はずの）コア番号
    nn::os::ThreadType  thread;
    NNT_OS_LOG("CreateThread( processorNumber=63 ):");

    // プロセッサ番号に 63 を指定して CreateThread
    EXPECT_DEATH_IF_SUPPORTED(nn::os::CreateThread( &thread, testFunc_GetCurrentCoreNumberThread, const_cast<int*>(&value), g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority, invalidCoreNumber), "");

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
// CreateThread 直後の名前変更が正しく行なえるか。

void
testFunc_SetThreadNameJustAfterCreateThread(void* argument)
{
    // 何もしない
    NN_UNUSED( argument );
}

TEST(OtherApis, test_SetThreadNameJustAfterCreateThread)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // まず CreateThread して、可能な限りすぐに SetThreadName() する
    nn::os::ThreadType  thread;
    nn::Result  result = nn::os::CreateThread( &thread, testFunc_SetThreadNameJustAfterCreateThread, NULL, g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority);

    const char *setName = "TestThreadName";
    nn::os::SetThreadName( &thread, setName );

    // スレッドが生成できているか
    NNT_OS_LOG("CreateThread(): ");
    CheckBool( result.IsSuccess() );

    // 少し間を空ける（10msec）
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 10 ) );

    // スレッド名を確認する
    const char* name = nn::os::GetThreadNamePointer( &thread );
    NNT_OS_LOG("SetThreadName(): \x22%s\x22\n", setName);
    NNT_OS_LOG("GetThreadName(): \x22%s\x22 ", name);
    CheckBool( strcmp(setName, name) == 0 );

    // スレッドを開始し、終了するのを待って破棄する
    nn::os::StartThread( &thread );
    nn::os::WaitThread( &thread );
    nn::os::DestroyThread( &thread );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}



//---------------------------------------------------------------------------
// SleepThread(1msec) の具体的な待機時間を計測する。
//
TEST(SleepThreadInterval, test_MeasureSleepThread1msec)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    nn::TimeSpan    span = nn::TimeSpan::FromMilliSeconds(1);

    for (int i = 0; i < 10; ++i)
    {
        nn::os::SleepThread( span );
        nn::os::Tick    startTick = nn::os::GetSystemTick();
        nn::os::SleepThread( span );
        nn::os::Tick    endTick   = nn::os::GetSystemTick();

        int64_t interval = nn::os::ConvertToTimeSpan( endTick - startTick ).GetMicroSeconds();
        NNT_OS_LOG("(-----): SleepThread(1msec) Interval= %d usec", interval);
        CheckBool( interval >= span.GetMicroSeconds() );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// SleepThread(1usec) の具体的な待機時間を計測する。
//
TEST(SleepThreadInterval, test_MeasureSleepThread1usec)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    nn::TimeSpan    span = nn::TimeSpan::FromMicroSeconds(1);

    for (int i = 0; i < 10; ++i)
    {
        nn::os::SleepThread( span );
        nn::os::Tick    startTick = nn::os::GetSystemTick();
        nn::os::SleepThread( span );
        nn::os::Tick    endTick   = nn::os::GetSystemTick();

        int64_t interval = nn::os::ConvertToTimeSpan( endTick - startTick ).GetMicroSeconds();
        NNT_OS_LOG("(-----): SleepThread(1usec) Interval= %d usec", interval);
        CheckBool( interval >= span.GetMicroSeconds() );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// SleepThread(1nsec) の具体的な待機時間を計測する。
//
TEST(SleepThreadInterval, test_MeasureSleepThread1nsec)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    nn::TimeSpan    span = nn::TimeSpan::FromNanoSeconds(1);

    for (int i = 0; i < 10; ++i)
    {
        nn::os::SleepThread( span );
        nn::os::Tick    startTick = nn::os::GetSystemTick();
        nn::os::SleepThread( span );
        nn::os::Tick    endTick   = nn::os::GetSystemTick();

        int64_t interval = nn::os::ConvertToTimeSpan( endTick - startTick ).GetMicroSeconds();
        NNT_OS_LOG("(-----): SleepThread(1nsec) Interval= %d usec", interval);
        CheckBool( interval >= span.GetMicroSeconds() );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

// 負のタイムアウト値を許容することになったときのためにテストを残しておく
#if 0
//---------------------------------------------------------------------------
// SleepThread(-1nsec) の具体的な待機時間を計測する。
//
TEST(SleepThreadInterval, test_MeasureSleepThreadMinus1nsec)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    nn::TimeSpan    span = nn::TimeSpan::FromNanoSeconds(-1);

    for (int i = 0; i < 10; ++i)
    {
        nn::os::SleepThread( span );
        nn::os::Tick    startTick = nn::os::GetSystemTick();
        nn::os::SleepThread( span );
        nn::os::Tick    endTick   = nn::os::GetSystemTick();

        int64_t interval = nn::os::ConvertToTimeSpan( endTick - startTick ).GetMicroSeconds();
        NNT_OS_LOG("(-----): SleepThread(-1nsec) Interval= %d usec", interval);
        CheckBool( interval >= 0 );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

#else
// 負のタイムアウト値を許容しない場合の DEATH TEST

TEST(SleepThreadDeathTest, test_SleepThreadMinus1nsecDeathTest)
{
    EXPECT_DEATH_IF_SUPPORTED( nn::os::SleepThread( nn::TimeSpan::FromNanoSeconds(-1) ), "");
}

#endif

//---------------------------------------------------------------------------
// CreateThread() 後にスタックガードが正しく構築されているか
//

inline uintptr_t PageRoundUp(uintptr_t address)
{
    return (address + nn::os::MemoryPageSize) & ~(nn::os::MemoryPageSize - 1);
}

void TestMemoryAccess(uintptr_t address, size_t size, nnt::os::detail::MemAttribute attr, const char* str)
{
    NN_UNUSED(str);

    uintptr_t   regionTop    = address;
    uintptr_t   regionBottom = address + size;

    for (auto adrs = regionTop; adrs<regionBottom; adrs += nn::os::MemoryPageSize)
    {
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("adrs=0x%08x のメモリが %s"), adrs, str);
#if defined(NN_BUILD_CONFIG_OS_WIN32) || !defined(NNT_OS_SYSTEM_PROGRAM_TEST)
        NN_UNUSED(attr);
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
        bool result = MemoryAccessTest(adrs, attr);
        CheckBool( result == true );
#endif /* defined(NN_BUILD_CONFIG_OS_HORIZON) */
    }
}

void testFunc_ThreadStackGuard(void* arg)
{
    NN_UNUSED(arg);

    size_t      stackSize   = sizeof(g_ThreadStack);
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    uintptr_t   stackBottom = reinterpret_cast<uintptr_t>( &stackSize );
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    uintptr_t   stackBottom = PageRoundUp( reinterpret_cast<uintptr_t>(__builtin_frame_address(0)) );
#endif
    uintptr_t   stackTop    = stackBottom - stackSize;

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("スタック領域がエイリアスされたものになっているか？\n"));
    SEQ_NONE();
    NNT_OS_LOG("originalStack=0x%08x aliasedStack=0x%08x", g_ThreadStack, stackTop);
    CheckBool( reinterpret_cast<uintptr_t>(g_ThreadStack) != stackTop );

    TestMemoryAccess(stackTop, stackSize,
                     detail::MemAttr_Stack_ReadWrite,
                     NN_TEXT("Stack かつ ReadWrite 可能か？"));

    TestMemoryAccess(stackTop - nn::os::MemoryPageSize, nn::os::MemoryPageSize,
                     detail::MemAttr_NonAccess,
                     NN_TEXT("前方ガードとしてアクセス不可か？"));

    TestMemoryAccess(stackTop + stackSize, nn::os::MemoryPageSize,
                     detail::MemAttr_NonAccess,
                     NN_TEXT("後方ガードとしてアクセス不可か？"));
}

TEST(ThreadStackGuard, test_StackGuardOfThreadStack)
{
    uintptr_t origStackTop  = reinterpret_cast<uintptr_t>( g_ThreadStack );
    size_t    origStackSize = sizeof(g_ThreadStack);

    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("エイリアス前のスタック領域のメモリ属性チェック\n"));
    TestMemoryAccess(origStackTop, origStackSize,
                     detail::MemAttr_Normal_ReadWrite,
                     NN_TEXT("Normal かつ ReadWrite 可能か？"));

    // CreateThread でスレッドを生成する（同時にスタックがエイリアスされる）
    nn::os::ThreadType  thread;
    auto result = nn::os::CreateThread( &thread, testFunc_ThreadStackGuard, NULL, g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority);
    EXPECT_TRUE( result.IsSuccess() );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("スレッドを生成しました\n"));

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("エイリアスされたスタック領域のメモリ属性チェック\n"));
    TestMemoryAccess(origStackTop, origStackSize,
                     detail::MemAttr_Normal_NonAccess_Locked,
                     NN_TEXT("Normal かつ Locked かつ アクセス不可 か？"));

    nn::os::StartThread( &thread );
    nn::os::WaitThread( &thread );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("スレッドが終了しました\n"));

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("CreateThread 時に渡したスタック領域のメモリ属性チェック\n"));
    TestMemoryAccess(origStackTop, origStackSize,
                     detail::MemAttr_Normal_ReadWrite,
                     NN_TEXT("Normal かつ ReadWrite に戻されたか？"));

    nn::os::DestroyThread( &thread );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("スレッドを削除しました\n"));

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

#if defined(NN_BUILD_CONFIG_OS_WIN32)
//---------------------------------------------------------------------------
// std::thread で使用するテスト用スレッド関数
//
//  スレッドの中で、OS 機能を使用する場合と使用しない場合の２つを用意する。
//  いずれの場合でも、正しくスレッドの後始末が行なわれる必要がある。
//
void StdThreadTestFunctionNotUsingOsApi()
{
    NNT_OS_LOG(NN_TEXT("sub:  std::thread で生成されたスレッドが起動しました。\n"));
    NNT_OS_LOG(NN_TEXT("sub:  std::thread で生成されたスレッドが終了します。\n"));
}

void StdThreadTestFunctionUsingOsApi()
{
    NNT_OS_LOG(NN_TEXT("sub:  std::thread で生成されたスレッドが起動しました。\n"));
    NNT_OS_LOG("sub:  thread= 0x%p\n", nn::os::GetCurrentThread());
    NNT_OS_LOG(NN_TEXT("sub:  std::thread で生成されたスレッドが終了します。\n"));
}

//---------------------------------------------------------------------------
// std::thread でスレッドを作成し、std::thread::join() する。
//
TEST(StdThreadTest, test_StdThreadAndJoin)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        NNT_OS_LOG(NN_TEXT("main: std::thread で２つのスレッドを生成します。\n"));

        std::thread stdThread1( StdThreadTestFunctionNotUsingOsApi );
        std::thread stdThread2( StdThreadTestFunctionUsingOsApi );

        NNT_OS_LOG(NN_TEXT("main: std::thread で生成した２つのスレッドに join します。\n"));

        stdThread1.join();
        stdThread2.join();

        NNT_OS_LOG(NN_TEXT("main: std::thread のテストを終了します。\n"));
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// std::thread でスレッドを作成し、std::thread::detach() する。
//
TEST(StdThreadTest, test_StdThreadAndDetach)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        NNT_OS_LOG(NN_TEXT("main: std::thread で２つのスレッドを生成します。\n"));

        std::thread stdThread1( StdThreadTestFunctionNotUsingOsApi );
        std::thread stdThread2( StdThreadTestFunctionUsingOsApi );

        NNT_OS_LOG(NN_TEXT("main: std::thread で生成した２つのスレッドを detach します。\n"));

        stdThread1.detach();
        stdThread2.detach();

        NNT_OS_LOG(NN_TEXT("main: std::thread のテストを終了します。\n"));
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// std::thread で使用するテスト用スレッド関数
//
//  スレッドの中で、OS 機能を使用する場合と使用しない場合の２つを用意する。
//  いずれの場合でも、正しくスレッドの後始末が行なわれる必要がある。
//
void StdFutureTestFunctionNotUsingOsApi(const char* str)
{
    NN_UNUSED(str);

    NNT_OS_LOG(NN_TEXT("sub:  std::%s で生成されたスレッドが起動しました。\n"), str);
    NNT_OS_LOG(NN_TEXT("sub:  std::%s で生成されたスレッドが終了します。\n"), str);
}

void StdFutureTestFunctionUsingOsApi(const char* str)
{
    NN_UNUSED(str);

    NNT_OS_LOG(NN_TEXT("sub:  std::%s で生成されたスレッドが起動しました。\n"), str);
    NNT_OS_LOG("sub:  thread= 0x%p\n", nn::os::GetCurrentThread());
    NNT_OS_LOG(NN_TEXT("sub:  std::%s で生成されたスレッドが終了します。\n"), str);
}

//---------------------------------------------------------------------------
// std::async でスレッドを立て、完了を待つ。
//
TEST(StdFutureTest, test_StdAsync)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        NNT_OS_LOG(NN_TEXT("main: std::async で非同期処理を生成します。\n"));

        auto f1 = std::async(std::launch::async, StdFutureTestFunctionNotUsingOsApi, "async");
        auto f2 = std::async(std::launch::async, StdFutureTestFunctionUsingOsApi, "async");

        NNT_OS_LOG(NN_TEXT("main: std::async で生成した非同期処理の結果を取得します。\n"));

        f1.get();
        f2.get();

        NNT_OS_LOG(NN_TEXT("main: std::async のテストを終了します。\n"));
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}
#endif /* defined(NN_BUILD_CONFIG_OS_WIN32) */

//---------------------------------------------------------------------------
// スレッド名変更のテスト
//  メインスレッド および 非メインスレッド でスレッド名の変更／取得／初期化。
//  SetThreadName, SetThreadNamePointer, GetThreadNamePointer のテスト。
//
void TestOfChangeThreadName(int type)
{
    const char* differentName = "DifferentName";
    const char* longName      = "LongThreadNameForTestOfSetThreadNamePointer";

    const char* prefix = (type == 0) ? "main: " : "sub:  ";
    const char* defaultName = (type == 0) ? "MainThread" : "Thread_0x";

    size_t defaultNameCheckSize = std::strlen(defaultName);


    // GetThreadNamePointer()
    SEQ_NONE();
    NN_UNUSED(prefix);
    NNT_OS_LOG(prefix);
    NNT_OS_LOG(NN_TEXT("スレッドのデフォルトスレッド名を取得する\n"));

    const char* name = nn::os::GetThreadNamePointer( nn::os::GetCurrentThread() );
    SEQ_NONE();
    NNT_OS_LOG(prefix);
    NNT_OS_LOG(NN_TEXT("スレッド名= \x22%s\x22"), name);
    CheckBool(std::memcmp(defaultName, name, defaultNameCheckSize) == 0);

    // SetThreadName( 別名 )
    SEQ_NONE();
    NNT_OS_LOG(prefix);
    NNT_OS_LOG(NN_TEXT("SetThreadName() でスレッド名を一旦別名に変更する\n"));
    nn::os::SetThreadName( nn::os::GetCurrentThread(), differentName );

    name = nn::os::GetThreadNamePointer( nn::os::GetCurrentThread() );
    SEQ_NONE();
    NNT_OS_LOG(prefix);
    NNT_OS_LOG(NN_TEXT("スレッド名= \x22%s\x22"), name);
    CheckBool(std::strcmp(differentName, name) == 0);

    // SetThreadName( NULL )
    SEQ_NONE();
    NNT_OS_LOG(prefix);
    NNT_OS_LOG(NN_TEXT("SetThreadName(NULL) でスレッド名をデフォルトに戻す\n"));
    nn::os::SetThreadName( nn::os::GetCurrentThread(), NULL );

    name = nn::os::GetThreadNamePointer( nn::os::GetCurrentThread() );
    SEQ_NONE();
    NNT_OS_LOG(prefix);
    NNT_OS_LOG(NN_TEXT("スレッド名= \x22%s\x22"), name);
    CheckBool(std::memcmp(defaultName, name, defaultNameCheckSize) == 0);

    // SetThreadNamePointer( 長めのスレッド名 )
    SEQ_NONE();
    NNT_OS_LOG(prefix);
    NNT_OS_LOG(NN_TEXT("SetThreadNamePointer() でスレッド名のポインタを設定する\n"));
    nn::os::SetThreadNamePointer( nn::os::GetCurrentThread(), longName );

    name = nn::os::GetThreadNamePointer( nn::os::GetCurrentThread() );
    SEQ_NONE();
    NNT_OS_LOG(prefix);
    NNT_OS_LOG(NN_TEXT("スレッド名= \x22%s\x22"), name);
    CheckBool(std::strcmp(longName, name) == 0);

    // SetThreadNamePointer( NULL )
    SEQ_NONE();
    NNT_OS_LOG(prefix);
    NNT_OS_LOG(NN_TEXT("SetThreadNamePointer() でスレッド名をデフォルトに戻す\n"));
    nn::os::SetThreadNamePointer( nn::os::GetCurrentThread(), NULL );

    name = nn::os::GetThreadNamePointer( nn::os::GetCurrentThread() );
    SEQ_NONE();
    NNT_OS_LOG(prefix);
    NNT_OS_LOG(NN_TEXT("スレッド名= \x22%s\x22"), name);
    CheckBool(std::memcmp(defaultName, name, defaultNameCheckSize) == 0);
}

void SubThreadOfChangeThreadNameTest(void* arg)
{
    NN_UNUSED(arg);

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("サブスレッドでのスレッド名変更テスト\n"));
    TestOfChangeThreadName(1);
}

TEST(ChangeThreadName, test_ChangeThreadNameTest)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // メインスレッドでのスレッド名変更テスト
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("メインスレッドでのスレッド名変更テスト\n"));
    TestOfChangeThreadName(0);

    // 非メインスレッドでのスレッド名変更テスト
    nn::os::ThreadType  thread;
    nn::Result result = nn::os::CreateThread( &thread, SubThreadOfChangeThreadNameTest, NULL, g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority);
    if (!result.IsSuccess())
    {
        FAIL();
    }

    nn::os::StartThread( &thread );
    nn::os::WaitThread( &thread );
    nn::os::DestroyThread( &thread );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

void SubThreadForCheckingManyThreads(void* arg)
{
    NN_UNUSED(arg);
}


//-----------------------------------------------------------------------------
// WaitThread の排他漏れをチェックするテスト（SIGLO-10339）
//
// 計 32 個のスレッドを 3 コア上で並列動作させる。

const size_t StackSize = 32768;

void ThreadFunction2ForWaitThreadOnMultiCore(void *arg)
{
    NN_UNUSED(arg);
}
void ThreadFunction1ForWaitThreadOnMultiCore(void *arg)
{
    int coreNums = *reinterpret_cast<int*>(arg);
    auto coreIndex = nn::os::GetCurrentCoreNumber();
    NN_UNUSED(coreIndex);

    uintptr_t stackBlock;
    int threadNums = (NumManyThread - coreNums - 1) / coreNums;
    size_t stackBlockSize = nn::util::align_up(StackSize * threadNums, nn::os::MemoryBlockUnitSize);
    auto result = nn::os::AllocateMemoryBlock(&stackBlock, stackBlockSize);
    EXPECT_TRUE( result.IsSuccess() );

    for (int k=0; k<3; ++k)
    {
        NNT_OS_LOG("passed: core=%d loop=%d\n", coreIndex, k);

        for (int j=0; j<100; ++j)
        {
            NN_SDK_ASSERT(threadNums < 32);
            nn::os::ThreadType threads[32];
            for (int i = 0; i < threadNums; ++i)
            {
                uintptr_t stackTop = stackBlock + (i * StackSize);
                auto pri = nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread()) - 1;
                auto ret = nn::os::CreateThread(
                                    &threads[i],
                                    ThreadFunction2ForWaitThreadOnMultiCore,
                                    NULL,
                                    reinterpret_cast<void*>(stackTop),
                                    StackSize,
                                    pri,
                                    (i % coreNums));

                // アプリであれば最大 32 スレッド生成は保証されているので、
                // 上記 CreateThread() は基本成功すべき。
                NN_ABORT_UNLESS_RESULT_SUCCESS(ret);
            }

            for(int i = 0; i < threadNums; ++i)
            {
                nn::os::StartThread(&threads[i]);
            }
            for(int i = 0; i < threadNums; ++i)
            {
                nn::os::WaitThread(&threads[i]);
            }
            for(int i = 0; i < threadNums; ++i)
            {
                nn::os::DestroyThread(&threads[i]);
            }
        }
    }
    nn::os::FreeMemoryBlock(stackBlock, nn::os::MemoryBlockUnitSize);
}

nn::os::ThreadType g_CoreThreads[ 64 ];

TEST(WaitThreadOnMultiCore, test_WaitThreadOnMultiCore)
{
    NNT_OS_LOG("Begin WaitThread() on multi-core test.\n");

    // 搭載コア数を算出（最下位の 1 のビット数をカウントしてコア数とする）
    nn::Bit64 coreMask = nn::os::GetThreadAvailableCoreMask();
              coreMask = nn::util::maskt1<nn::Bit64>(coreMask);
    int       coreNums = nn::util::cntt0<nn::Bit64>(~coreMask);
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    if (coreNums > 3)
    {
        coreNums = 3;
    }
#endif

    NNT_OS_LOG("Core nums = %d\n", coreNums);

    // ヒープからスタック用領域を確保
    auto result = nn::os::SetMemoryHeapSize(nn::os::MemoryBlockUnitSize * (coreNums + 1));
    EXPECT_TRUE( result.IsSuccess() );

    uintptr_t stackBlock;
    result = nn::os::AllocateMemoryBlock(&stackBlock, nn::os::MemoryBlockUnitSize);
    EXPECT_TRUE( result.IsSuccess() );

    // テスト用のスレッドを生成
    for (int i = 0; i < coreNums; ++i)
    {
        auto pri = nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread()) - 1;
        auto ret = nn::os::CreateThread(
                        &g_CoreThreads[i],
                        ThreadFunction1ForWaitThreadOnMultiCore,
                        &coreNums,
                        reinterpret_cast<void*>(stackBlock + (i * StackSize)),
                        StackSize,
                        pri,
                        i);
        EXPECT_TRUE( ret.IsSuccess() );
    }

    for(int i = 0; i < coreNums; ++i)
    {
        nn::os::StartThread(&g_CoreThreads[i]);
    }

    for(int i = 0; i < coreNums; ++i)
    {
        nn::os::WaitThread(&g_CoreThreads[i]);
    }

    for(int i = 0; i < coreNums; ++i)
    {
        nn::os::DestroyThread(&g_CoreThreads[i]);
    }

    // ヒープを返却し、0 に戻しておく
    nn::os::FreeMemoryBlock(stackBlock, nn::os::MemoryBlockUnitSize);
    nn::os::SetMemoryHeapSize(0);

    NNT_OS_LOG("End WaitThread() on multi-core test.\n");
}


//-----------------------------------------------------------------------------
// メインスレッド含めて 32 個のスレッドが作成可能なことを確認
nn::os::ThreadType          g_ThreadPool[NumManyThread];
NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStackPool[NumManyThread][8192];

TEST(CreateManyThreads, test_CreateManyThreads)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    NNT_OS_LOG(NN_TEXT("%d 個のスレッドが作成可能かどうか確認\n"), NumManyThread);

    for (int i=0; i<NumManyThread; ++i)
    {
        nn::Result result = nn::os::CreateThread( &g_ThreadPool[i], SubThreadForCheckingManyThreads, NULL, g_ThreadStackPool[i], sizeof(g_ThreadStackPool[i]), nn::os::DefaultThreadPriority);
        EXPECT_TRUE(result.IsSuccess());
    }
    for (int i=0; i<NumManyThread; ++i)
    {
        nn::os::DestroyThread( &g_ThreadPool[i] );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

TEST(CoreMaskTest, BasicTest)
{
    int       newIdealCore = 0;
    nn::Bit64 newAffinity  = 0x3;

    int       originalIdealCore;
    nn::Bit64 originalAffinity;

    nn::Bit64 availableCoreMask = nn::os::GetThreadAvailableCoreMask();
    NNT_OS_LOG("available core mask = 0x%x\n", availableCoreMask);
    NN_UNUSED(availableCoreMask);

    auto thread = nn::os::GetCurrentThread();
    nn::os::GetThreadCoreMask(&originalIdealCore, &originalAffinity, thread);

    NNT_OS_LOG("current core number   = %d\n", nn::os::GetCurrentCoreNumber());
    NNT_OS_LOG("current ideal core id = %d\n", originalIdealCore);
    NNT_OS_LOG("current core affinity = 0x%x\n", originalAffinity);

    // 設定を変更
    nn::os::SetThreadCoreMask(thread, newIdealCore, newAffinity);

    NNT_OS_LOG("--- Ideal core and affinity changed. ---\n");

    // 変更した設定を取得
    int       idealCore;
    nn::Bit64 affinity;
    nn::os::GetThreadCoreMask(&idealCore, &affinity, thread);

    NNT_OS_LOG("new core number   = %d\n", nn::os::GetCurrentCoreNumber());
    NNT_OS_LOG("new ideal core id = %d\n", idealCore);
    NNT_OS_LOG("new core affinity = 0x%x\n", affinity);

    EXPECT_EQ(newIdealCore, idealCore);
    EXPECT_EQ(newAffinity, affinity);

    // 他のテストに影響がないように設定を元に戻す
    nn::os::SetThreadCoreMask(thread, originalIdealCore, originalAffinity);
}

TEST(CoreMaskTest, IgnoreNullArgument)
{
    int       originalIdealCore;
    nn::Bit64 originalAffinity;
    int       idealCore;
    nn::Bit64 affinity;

    auto thread = nn::os::GetCurrentThread();
    nn::os::GetThreadCoreMask(&originalIdealCore, &originalAffinity, thread);

    // nullptr を指定した項目の情報は受け取らない
    nn::os::GetThreadCoreMask(nullptr, &affinity, thread);
    nn::os::GetThreadCoreMask(&idealCore, nullptr, thread);

    EXPECT_EQ(originalIdealCore, idealCore);
    EXPECT_EQ(originalAffinity, affinity);
}

TEST(CoreMaskTest, SpecialIdealCoreValue)
{
    int       originalIdealCore;
    nn::Bit64 originalAffinity;
    nn::Bit64 newAffinity = 0x3;

    auto thread = nn::os::GetCurrentThread();
    nn::os::GetThreadCoreMask(&originalIdealCore, &originalAffinity, thread);

    // 優先コアを変更しない
    NNT_OS_LOG("Test IdealCoreNoUpdate: ");
    nn::os::SetThreadCoreMask(thread, nn::os::IdealCoreNoUpdate, newAffinity);

    // 変更した設定を取得
    int       idealCore;
    nn::Bit64 affinity;
    nn::os::GetThreadCoreMask(&idealCore, &affinity, thread);

    EXPECT_EQ(originalIdealCore, idealCore);
    EXPECT_EQ(newAffinity, affinity);
    NNT_OS_LOG("Ok.\n");

    // 優先コアを指定しない
    NNT_OS_LOG("Test IdealCoreDontCare: ");
    nn::os::SetThreadCoreMask(thread, nn::os::IdealCoreDontCare, newAffinity);

    // 変更した設定を取得
    nn::os::GetThreadCoreMask(&idealCore, &affinity, thread);

    EXPECT_EQ(nn::os::IdealCoreDontCare, idealCore);
    EXPECT_EQ(newAffinity, affinity);
    NNT_OS_LOG("Ok.\n");


    // デフォルトの優先コアのみで動作
    NNT_OS_LOG("Test IdealCoreUseDefaultValue: ");
    nn::os::SetThreadCoreMask(thread, nn::os::IdealCoreUseDefaultValue, 0xffff);

    // 変更した設定を取得
    nn::os::GetThreadCoreMask(&idealCore, &affinity, thread);

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    EXPECT_EQ(0,     idealCore);  // Win32 では最も番号の小さいコア
    EXPECT_EQ(0x01U, affinity);
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    EXPECT_EQ(0,     idealCore);  // desc/nmeta で指定されたデフォルトコア
    EXPECT_EQ(0x01U, affinity);
#endif
    NNT_OS_LOG("Ok.\n");

    // 他のテストに影響がないように設定を元に戻す
    nn::os::SetThreadCoreMask(thread, originalIdealCore, originalAffinity);
}

TEST(CoreMaskDeathTest, InvalidCoreAffinityMask)
{
    int       deathIdealCore = 0;
    nn::Bit64 deathAffinity  = 0xffff;

    auto thread = nn::os::GetCurrentThread();
    EXPECT_DEATH_IF_SUPPORTED(nn::os::SetThreadCoreMask(thread, deathIdealCore, deathAffinity), "");
}

TEST(CoreMaskDeathTest, InvalidIdealCore)
{
    int       deathIdealCore = 0;
    nn::Bit64 deathAffinity  = 0x2;

    auto thread = nn::os::GetCurrentThread();
    EXPECT_DEATH_IF_SUPPORTED(nn::os::SetThreadCoreMask(thread, deathIdealCore, deathAffinity), "");
}

TEST(CoreMaskDeathTest, InvalidCombination)
{
    int       originalIdealCore;
    nn::Bit64 originalAffinity;
    nn::Bit64 deathAffinity;

    auto thread = nn::os::GetCurrentThread();
    nn::os::GetThreadCoreMask(&originalIdealCore, &originalAffinity, thread);

    // 指定可能なコアマスクから優先コアのみのビットを落とす
    deathAffinity = nn::os::GetThreadAvailableCoreMask() & ~(1ULL << originalIdealCore);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    // Win32 では優先コアの概念がないため死なない
    EXPECT_DEATH_IF_SUPPORTED(nn::os::SetThreadCoreMask(thread, nn::os::IdealCoreNoUpdate, deathAffinity), "");
#endif
}

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
struct FpuEnvType
{
    #if defined(NN_BUILD_CONFIG_CPU_ARM)
        nn::Bit32   _fpscr;
    #elif defined(NN_BUILD_CONFIG_CPU_ARM64)
        nn::Bit32   _fpcr;
        nn::Bit32   _fpsr;
    #endif
};

inline void SetFpuRegs(FpuEnvType* pFpuRegs)
{
    #if defined(NN_BUILD_CONFIG_CPU_ARM)
    asm volatile (" vmsr fpscr, %0" : : "r"(pFpuRegs->_fpscr) : "memory");
    #elif defined(NN_BUILD_CONFIG_CPU_ARM64)
    asm volatile (" msr fpcr, %0" : : "r"(static_cast<nn::Bit64>(pFpuRegs->_fpcr)) : "memory");
    asm volatile (" msr fpsr, %0" : : "r"(static_cast<nn::Bit64>(pFpuRegs->_fpsr)) : "memory");
    #endif
}

inline void GetFpuRegs(FpuEnvType* pFpuRegs)
{
    #if defined(NN_BUILD_CONFIG_CPU_ARM)
    nn::Bit32 value32;
    asm volatile (" vmrs %0, fpscr" : "=r"(value32) : : "memory");
    pFpuRegs->_fpscr = value32;
    #elif defined(NN_BUILD_CONFIG_CPU_ARM64)
    nn::Bit64 value64;
    asm volatile (" mrs %0, fpcr" : "=r"(value64) : : "memory");
    pFpuRegs->_fpcr = value64 & 0xffffffff;
    asm volatile (" mrs %0, fpsr" : "=r"(value64) : : "memory");
    pFpuRegs->_fpsr = value64 & 0xffffffff;
    #endif
}

//---------------------------------------------------------------------------
//  SIGLO-12409
//  FENV の動作モード（丸めモードなどの設定値）が
//  親スレッドから子スレッドに継承されていることを確認するテスト。
//
//  SIGLO-84095
//  テスト内容を改善。
//  - FpcrRegisterInheritanceTest テストでは、スレッド生成時に fpcr が
//    正しく継承されるかどうかを直接レジスタを見て検査する。
//  - FpuEnvType テストでは、std::feset*(), std::feget() の動作テストを行なう。
//
void CheckThreadOfFpcrRegisterInheritance(void* arg)
{
    FpuEnvType* parentFpuRegs = reinterpret_cast<FpuEnvType*>(arg);

    FpuEnvType childFpuRegs;
    GetFpuRegs( &childFpuRegs );

    auto result = std::memcmp(parentFpuRegs, &childFpuRegs, sizeof(FpuEnvType));
    EXPECT_TRUE( result == 0 );
}

TEST(FpcrRegisterInheritance, FpcrRegisterInheritanceTest)
{
    FpuEnvType savedFpuRegs;
    GetFpuRegs( &savedFpuRegs );

    // 親スレッドの FPCR をデフォルトと異なる値へ変更（FPCR.RMode を変更）
    FpuEnvType parentFpuRegs;
    GetFpuRegs( &parentFpuRegs );
    #if defined(NN_BUILD_CONFIG_CPU_ARM)
    parentFpuRegs._fpscr ^= 0x07c00000;
    #elif defined(NN_BUILD_CONFIG_CPU_ARM64)
    parentFpuRegs._fpcr  ^= 0x07c00000;
    #endif
    SetFpuRegs( &parentFpuRegs );

    nn::os::ThreadType  thread;
    auto result = nn::os::CreateThread(&thread, CheckThreadOfFpcrRegisterInheritance, &parentFpuRegs, g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority);
    EXPECT_TRUE( result.IsSuccess() );

    nn::os::StartThread( &thread );
    nn::os::WaitThread( &thread );
    nn::os::DestroyThread( &thread );

    // 親スレッドの FPCR を元へ戻しておく
    SetFpuRegs( &savedFpuRegs );
}

#if defined(NN_BUILD_CONFIG_TOOLCHAIN_CLANG)
//  以下の動作確認を行う
//  - std::fesetround()
//  - std::fegetround()
//  - std::fesetenv()
//  - std::fegetenv()
//
TEST(LibcRuntimeTest, FpuEnvTest)
{
    FpuEnvType savedFpuRegs = {};
    GetFpuRegs( &savedFpuRegs );

    {
        // std::fegetround() の動作チェック
        FpuEnvType fenv = {};
        GetFpuRegs( &fenv );
        auto value = static_cast<nn::Bit32>( std::fegetround() );

#if defined(NN_BUILD_CONFIG_CPU_ARM)
        EXPECT_EQ(fenv._fpscr & 0x01c00000, value);
#elif defined(NN_BUILD_CONFIG_CPU_ARM64)
        EXPECT_EQ(fenv._fpcr  & 0x01c00000, value);
#endif

        // FZ と Rmode を変更
        value ^= 0x01c00000;
        std::fesetround( value );

        // std::fesetround() が正しく動作したのかをチェック
        GetFpuRegs( &fenv );
        value = static_cast<nn::Bit32>( std::fegetround() );

#if defined(NN_BUILD_CONFIG_CPU_ARM)
        EXPECT_EQ(fenv._fpscr & 0x01c00000, value);
#elif defined(NN_BUILD_CONFIG_CPU_ARM64)
        EXPECT_EQ(fenv._fpcr  & 0x01c00000, value);
#endif
    }

    {
        // std::fegetenv() の動作チェック
        FpuEnvType fenv = {};
        GetFpuRegs( &fenv );

        std::fenv_t fenvValue;
        std::fegetenv( &fenvValue );

#if defined(NN_BUILD_CONFIG_CPU_ARM)
        EXPECT_EQ(fenv._fpscr, fenvValue.__cw);
#elif defined(NN_BUILD_CONFIG_CPU_ARM64)
        EXPECT_EQ(fenv._fpcr,  fenvValue.__fpcr);
#endif

        // AHP, DN, FZ, Rmode を変更
#if defined(NN_BUILD_CONFIG_CPU_ARM)
        fenvValue.__cw ^= 0x07c00000;
#elif defined(NN_BUILD_CONFIG_CPU_ARM64)
        fenvValue.__fpcr  ^= 0x07c00000;
#endif
        std::fesetenv( &fenvValue );

        // std::fesetenv() が正しく動作したのかをチェック
        GetFpuRegs( &fenv );
        std::fegetenv( &fenvValue );

#if defined(NN_BUILD_CONFIG_CPU_ARM)
        EXPECT_EQ(fenv._fpscr, fenvValue.__cw);
#elif defined(NN_BUILD_CONFIG_CPU_ARM64)
        EXPECT_EQ(fenv._fpcr,  fenvValue.__fpcr);
#endif
    }

    // 元に戻しておく
    SetFpuRegs( &savedFpuRegs );
}
#endif  // defined(NN_BUILD_CONFIG_TOOLCHAIN_CLANG)
#endif  // defined(NN_BUILD_CONFIG_OS_HORIZON)

//---------------------------------------------------------------------------
//  SIGLO-20336 スレッド開始前に nn::os::SetThreadCoreMask() を呼んでもすぐにはコアマスクが変わらない
//
volatile bool g_CoreMaskTestDoneFlag = false;

void CheckCoreMaskFunc(void* arg)
{
    int mainThreadCoreNumber = *static_cast<int*>(arg);
    int currectCore = nn::os::GetCurrentCoreNumber();
    ASSERT_TRUE(currectCore != mainThreadCoreNumber);

    // メインスレッドのビジーループを抜けさせる
    g_CoreMaskTestDoneFlag = true;
}

TEST(CoreMaskTest, SetCoreMaskBeforeStartThread)
{
    int mainThreadCoreNumber = nn::os::GetCurrentCoreNumber();

    // サブスレッドを生成する
    // 最初の時点ではメインスレッドと同一コアで動くように設定
    nn::os::ThreadType  thread;
    auto result = nn::os::CreateThread(&thread, CheckCoreMaskFunc, reinterpret_cast<void*>(&mainThreadCoreNumber), g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority + 1, mainThreadCoreNumber);
    EXPECT_TRUE( result.IsSuccess() );

    nn::Bit64 affinity = nn::os::GetThreadAvailableCoreMask();
    affinity = affinity & ~(1 << mainThreadCoreNumber);

    // StartThread の前に CoreMask を変更
    // メインスレッドとは別のコアで動くように設定
    nn::os::SetThreadCoreMask(&thread, nn::os::IdealCoreDontCare, affinity);

    // サブスレッドを StartThread
    // ここで動き出した時点で SetThreadCoreMask の設定が反映されれば
    // サブスレッドはメインスレッドと別のコアで動き始めるはず
    nn::os::StartThread(&thread);

    // ビジーループする。
    // もしサブスレッドのアフィニティが StartThread 時点で変更できていなければ、
    // メインスレッドの方が優先度が高いので以下のビジーループが回り続ける。
    // サブスレッドには処理が回ってこないためループを抜けられない。
    nn::os::Tick start = nn::os::GetSystemTick();
    while (g_CoreMaskTestDoneFlag == false)
    {
        nn::os::Tick tick = nn::os::GetSystemTick();
        if (nn::os::ConvertToTimeSpan(tick - start).GetSeconds() >= 5)
        {
            // 5 秒待っても抜けなければエラー
            EXPECT_TRUE(false);
            break;
        }
    }

    nn::os::WaitThread(&thread);
    nn::os::DestroyThread(&thread);
}

TEST(CoreMaskTest, DestroyThreadBeforeStartThread)
{
    // サブスレッドを生成する
    nn::os::ThreadType  thread;
    auto result = nn::os::CreateThread(&thread, CheckCoreMaskFunc, nullptr, g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority);
    EXPECT_TRUE( result.IsSuccess() );

    // 動作しないまま Destroy
    // 問題なければ OK
    nn::os::DestroyThread(&thread);
}

TEST(TlsForSdk, SizeCheck)
{
    NNT_OS_LOG("original: sizeof(ThreadType)=%d\n", sizeof(nn::os::ThreadType));
}

//---------------------------------------------------------------------------
//  SIGLO-18633 スレッドスタック領域からスレッドスタックを確保した際のアボートメッセージを改善する
//
//  異常なメモリをスタックメモリとして指定して、アボートを起こすテスト (普段は無効にしておく)
//
#if 0
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
void testDoNothingThreadFunction( void* arg )
{
    NN_UNUSED( arg );
}

    // 自分のスレッドスタックを使ってスレッドを作成し、アボートを起こすテスト
    #if 0
    void testInvokeOwnStackAbortThreadFunction( void* arg )
    {
        NN_UNUSED( arg );

        static const size_t STACK_SIZE = 8192;
        static const size_t ALIGN_SIZE = 4096;

        uint8_t arrayForStack[STACK_SIZE];
        uintptr_t* p = reinterpret_cast<uintptr_t*>(arrayForStack);

        // 4KB アラインになるまでインクリメント
        while ( reinterpret_cast<uint64_t>(p) % ALIGN_SIZE != 0 )
        {
            p++;
        }

        nn::os::ThreadType thread;

        // ここでアボートするはず
        nn::Result result = nn::os::CreateThread( &thread, testDoNothingThreadFunction, NULL, p, STACK_SIZE, nn::os::DefaultThreadPriority);
        CheckBool( result.IsSuccess() );

        nn::os::StartThread( &thread );
        nn::os::WaitThread( &thread );
        nn::os::DestroyThread( &thread );
    }

    TEST(ThreadOwnStackAbortTest, test_ThreadOwnStackAbort)
    {
        nn::os::ThreadType thread;
        nn::Result result = nn::os::CreateThread( &thread, testInvokeStackAbortThreadFunction, NULL, g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority);
        CheckBool( result.IsSuccess() );

        nn::os::StartThread( &thread );
        nn::os::WaitThread( &thread );
        nn::os::DestroyThread( &thread );
    }
    #endif

    // [TransferMemory (ReadOnly) + 有効なメモリ]をスタックメモリ指定してアボートを起こすテスト
    #if 0
    TEST(ThreadTransferMemoryStackAbortTest1, test_ThreadTransferMemoryStackAbort1)
    {
        nn::os::TransferMemoryType transferMemory;
        {
            auto result = nn::os::CreateTransferMemory( &transferMemory, g_ThreadStack, sizeof(g_ThreadStack) / 2, nn::os::MemoryPermission_ReadOnly );
            CheckBool( result.IsSuccess() );
        }

        nn::os::ThreadType thread;
        {
            auto result = nn::os::CreateThread( &thread, testDoNothingThreadFunction, NULL, g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority);
            CheckBool( result.IsSuccess() );
        }

        nn::os::StartThread( &thread );
        nn::os::WaitThread( &thread );
        nn::os::DestroyThread( &thread );
    }
    #endif

    // [有効なメモリ + TransferMemory (ReadOnly)]をスタックメモリ指定してアボートを起こすテスト
    #if 0
    TEST(ThreadTransferMemoryStackAbortTest2, test_ThreadTransferMemoryStackAbort2)
    {
        const size_t TransferMemoryOffset = 8192;
        const size_t TransferMemorySize = 8192;

        nn::os::TransferMemoryType transferMemory;
        {
            auto result = nn::os::CreateTransferMemory( &transferMemory,
                g_ThreadStack + TransferMemoryOffset,
                TransferMemorySize, nn::os::MemoryPermission_ReadOnly );
            CheckBool( result.IsSuccess() );
        }

        nn::os::ThreadType thread;
        {
            auto result = nn::os::CreateThread( &thread, testDoNothingThreadFunction, NULL, g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority);
            CheckBool( result.IsSuccess() );
        }

        nn::os::StartThread( &thread );
        nn::os::WaitThread( &thread );
        nn::os::DestroyThread( &thread );
    }
    #endif

    // [TransferMemory (ReadOnly) + 有効なメモリ + TransferMemory (ReadOnly)]をスタックメモリ指定してアボートを起こすテスト
    #if 0
    TEST(ThreadTransferMemoryStackAbortTest3, test_ThreadTransferMemoryStackAbort3)
    {
        const size_t TransferMemoryOffset = 4096;
        const size_t TransferMemorySize = 4096;

        nn::os::TransferMemoryType transferMemory1;
        {
            auto result = nn::os::CreateTransferMemory( &transferMemory1,
                g_ThreadStack + TransferMemoryOffset,
                TransferMemorySize, nn::os::MemoryPermission_ReadOnly );
            CheckBool( result.IsSuccess() );
        }

        nn::os::TransferMemoryType transferMemory2;
        {
            auto result = nn::os::CreateTransferMemory( &transferMemory2,
                g_ThreadStack + sizeof(g_ThreadStack) - TransferMemoryOffset,
                TransferMemorySize, nn::os::MemoryPermission_ReadOnly );
            CheckBool( result.IsSuccess() );
        }

        nn::os::ThreadType thread;
        {
            auto result = nn::os::CreateThread( &thread, testDoNothingThreadFunction, NULL, g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority);
            CheckBool( result.IsSuccess() );
        }

        nn::os::StartThread( &thread );
        nn::os::WaitThread( &thread );
        nn::os::DestroyThread( &thread );
    }
    #endif

    // SharedMemory をスタックメモリ指定してアボートを起こすテスト
    #if 0
    TEST(ThreadSharedMemoryStackAbortTest, test_ThreadSharedMemoryStackAbort)
    {
        const size_t SharedMemorySize = 16384;

        nn::os::SharedMemoryType sharedMemory;
        {
            auto result = nn::os::CreateSharedMemory( &sharedMemory, SharedMemorySize, nn::os::MemoryPermission_ReadWrite, nn::os::MemoryPermission_ReadOnly );
            CheckBool( result.IsSuccess() );
            void* address = nn::os::MapSharedMemory( &sharedMemory, nn::os::MemoryPermission_ReadWrite );
            CheckBool( address != NULL );
        }

        nn::os::ThreadType thread;
        {
            auto result = nn::os::CreateThread( &thread, testDoNothingThreadFunction, NULL,
                nn::os::GetSharedMemoryAddress( &sharedMemory),
                nn::os::GetSharedMemorySize( &sharedMemory ),
                nn::os::DefaultThreadPriority );
            CheckBool( result.IsSuccess() );
        }

        nn::os::StartThread( &thread );
        nn::os::WaitThread( &thread );
        nn::os::DestroyThread( &thread );
    }
    #endif
#endif
#endif


#if defined(NN_BUILD_CONFIG_OS_HORIZON) && defined(NN_BUILD_CONFIG_TOOLCHAIN_CLANG)
//---------------------------------------------------------------------------
//  SuspendThread(), ResumeThread() に関するテスト
//
nn::os::Barrier g_BarrierForSuspendThreadTest(2);
nn::os::Event   g_FinishEventForSuspendThreadTest(nn::os::EventClearMode_ManualClear);
std::atomic<bool>   g_SuspendDisableFlag(false);
char            g_MemcpyBuffer[16384];

void CreatedThreadForSuspendThreadTest()
{
}


void SubThreadForSuspendThreadTest(void* arg)
{
    int subThreadCoreNumber  = *reinterpret_cast<int*>(arg);

    // Affinity を現在コアだけにする
    nn::Bit64 affinity = 0x1lu << subThreadCoreNumber;
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), subThreadCoreNumber, affinity);

    // メインスレッドとの同期待ち
    g_FinishEventForSuspendThreadTest.Clear();
    g_BarrierForSuspendThreadTest.Await();

    // サブスレッドの本体処理（メインスレッドから頻繁に Suspend/Resume される）
    auto startTick = nn::os::GetSystemTick();

#if defined(NN_BUILD_CONFIG_OS_WIN)
    for (int i=0; i<10000; ++i)
    {
        for (int j=0; j<1000; ++j)
        {
            // Windows の場合、std::thread で VC++ ランタイム内部の処理が走り、
            // 変なタイミングで SuspendThread するとデッドロックを引き起こす。
            // そのため、Unity モデルと同じようにフラグを使って、
            // SuspendThread 禁止区間を設ける。
            g_SuspendDisableFlag.store(true, std::memory_order_release);

//          これだとうまく動かない。原因は不明。
//            std::thread stdThread(CreatedThreadForSuspendThreadTest);
//            stdThread.join();
            nn::os::Mutex mutex(false);
            {
                // 無駄に時間を消費するコードを入れておく
                std::lock_guard<nn::os::Mutex> lock(mutex);
                std::memcpy(g_MemcpyBuffer, g_MemcpyBuffer, sizeof(g_MemcpyBuffer));
            }

            g_SuspendDisableFlag.store(false, std::memory_order_release);
        }
    }
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    for (int i=0; i<10; ++i)
    {
        for (int j=0; j<1000; ++j)
        {
            g_SuspendDisableFlag.store(true, std::memory_order_release);

            // 開発機でも SDK-API 呼出中はフラグを操作して
            // Suspend 禁止区間を設ける。
            std::thread stdThread(CreatedThreadForSuspendThreadTest);
            stdThread.join();

            g_SuspendDisableFlag.store(false, std::memory_order_release);
        }
    }
#endif

    auto endTick = nn::os::GetSystemTick();

    // 一連の処理を終えたら終了シグナルを出して、終了の同期待ち
    g_FinishEventForSuspendThreadTest.Signal();
    g_BarrierForSuspendThreadTest.Await();

    // SuspendThread() する側が完全にシーケンスを終了したらログを出力
    NNT_OS_LOG("Test time span = %lld (usec)\n", (endTick - startTick).ToTimeSpan().GetMicroSeconds());
    NN_UNUSED(startTick);
    NN_UNUSED(endTick);
}

TEST(SuspendThreadTest, SuspendThreadTest)
{
    int mainThreadCoreNumber = nn::os::GetCurrentCoreNumber();
    int subThreadCoreNumber  = mainThreadCoreNumber == 0 ? 1 : 0;

    // サブスレッドを生成する（メインスレッドとは別コアで動作させる）
    nn::os::ThreadType  thread;
    auto result = nn::os::CreateThread(&thread, SubThreadForSuspendThreadTest, &subThreadCoreNumber, g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority, subThreadCoreNumber);
    EXPECT_TRUE( result.IsSuccess() );
    nn::os::StartThread(&thread);

    // Affinity を現在コアだけにする
    nn::Bit64 affinity = 0x1lu << mainThreadCoreNumber;
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), mainThreadCoreNumber, affinity);

    // サブスレッドとの同期待ち
    g_BarrierForSuspendThreadTest.Await();

    auto suspendCount = nn::os::GetThreadSuspendCount( &thread );
    EXPECT_TRUE( suspendCount == 0 );

    int suspendedCount = 0;
    int contextOutRate = 0;
    // サブスレッドに対して、ひたすら Suspend/Resume
    while ( !g_FinishEventForSuspendThreadTest.TryWait() )
    {
        // １ネスト分の SuspendThread() を行なう
        EXPECT_TRUE( nn::os::SuspendThread( &thread ) == 0);
        if (g_SuspendDisableFlag.load(std::memory_order_acquire) == true)
        {
            // 対象スレッドが Suspend 禁止区間だった場合は、
            // Resume して一定時間後に再度 Suspend を試みる。
            EXPECT_TRUE( nn::os::ResumeThread( &thread ) == 1);
            nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(1) );
            continue;
        }

        ++suspendedCount;
        EXPECT_TRUE( nn::os::GetThreadSuspendCount( &thread ) == 1);

        nn::os::ThreadContextInfo context;
        nn::os::GetThreadContext(&context, &thread);

        EXPECT_TRUE( nn::os::ResumeThread( &thread ) == 1);
        EXPECT_TRUE( nn::os::GetThreadSuspendCount( &thread ) == 0);
        nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(10) );

        // ログ出力はどのスレッドも SuspendThread() していない状態で行なう
        ++contextOutRate;
        if (contextOutRate >= 100)
        {
            // TargetManager のログ出力が間に合わなくて不安定になるので、
            // ログ出力頻度を下げる。
            contextOutRate = 0;
#if defined(NN_BUILD_CONFIG_OS_HORIZON) && defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)
            NNT_OS_LOG("context: pc=0x%p sp=0x%p\n", context.pc, context.sp);
#elif defined(NN_BUILD_CONFIG_OS_HORIZON) && defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
            NNT_OS_LOG("context: pc=0x%016llx r13=0x%016llx\n", context.pc, context.r[13]);
#elif defined(NN_BUILD_CONFIG_OS_HORIZON) && defined(NN_OS_CPU_ARM_AARCH32_ARMV7A)
            NNT_OS_LOG("context: pc=0x%p sp=0x%p\n", context.pc, context.sp);
#elif defined(NN_BUILD_CONFIG_OS_WIN) && defined(NN_BUILD_CONFIG_ADDRESS_64)
            NNT_OS_LOG("context: Rip=0x%p Rsp=0x%p\n", context.Rip, context.Rsp);
#elif defined(NN_BUILD_CONFIG_OS_WIN) && defined(NN_BUILD_CONFIG_ADDRESS_32)
            NNT_OS_LOG("context: Eip=0x%08x Esp=0x%08p\n", context.Eip, context.Esp);
#endif
        }

        // ３ネスト分の SuspendThread() を行なう
        EXPECT_TRUE( nn::os::SuspendThread( &thread ) == 0);
        if (g_SuspendDisableFlag.load(std::memory_order_acquire) == true)
        {
            // 対象スレッドが Suspend 禁止区間だった場合は、
            // Resume して一定時間後に再度 Suspend を試みる。
            EXPECT_TRUE( nn::os::ResumeThread( &thread ) == 1);
            nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(1) );
            continue;
        }

        EXPECT_TRUE( nn::os::SuspendThread( &thread ) == 1);
        EXPECT_TRUE( nn::os::SuspendThread( &thread ) == 2);
        EXPECT_TRUE( nn::os::GetThreadSuspendCount( &thread ) == 3);

        nn::os::GetThreadContext(&context, &thread);

        EXPECT_TRUE( nn::os::ResumeThread( &thread ) == 3);
        EXPECT_TRUE( nn::os::ResumeThread( &thread ) == 2);
        EXPECT_TRUE( nn::os::ResumeThread( &thread ) == 1);
        EXPECT_TRUE( nn::os::GetThreadSuspendCount( &thread ) == 0);
        nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(10) );
    }

    g_BarrierForSuspendThreadTest.Await();
    nn::os::WaitThread(&thread);
    nn::os::DestroyThread(&thread);

    NN_LOG("SuspendThreadTest: suspended count = %d\n", suspendedCount);
    EXPECT_TRUE( suspendedCount > 0 );

}   // NOLINT(readability/fn_size)
#endif /* defined(NN_BUILD_CONFIG_OS_HORIZON) && defined(NN_BUILD_CONFIG_TOOLCHAIN_CLANG) */


//---------------------------------------------------------------------------
//  Test for CPU Profiler
//---------------------------------------------------------------------------

volatile bool g_ProfilerWorkerTestDoneFlag = false;
nn::os::ThreadType g_ProfilerWorkerThread;

//
// Worker thread
//
void ProfilerWorkerThread(void* arg)
{
    nn::os::SdkLastThreadContext  context;
    nn::os::ThreadType*           thread;
    nn::Bit32                     flag;

    NN_UNUSED(arg);

    for (int i=0; i<1000; ++i)
    {
        //
        // Get the last thread information
        //
        auto result = nn::os::GetLastThreadInfo(&thread, &context, &flag);
        if (result.IsSuccess())
        {
            //
            // Get stack information
            //
            uintptr_t stackTop;
            size_t    stackSize;
            nn::os::GetThreadStackInfo(nullptr, nullptr, thread);       // test
            nn::os::GetThreadStackInfo(&stackTop, nullptr, thread);     // test
            nn::os::GetThreadStackInfo(nullptr, &stackSize, thread);    // test
            nn::os::GetThreadStackInfo(&stackTop, &stackSize, thread);

            //
            // Get thread name
            //
            const char* name = GetThreadNamePointer(thread);
            if (!name)
            {
                name = "<Invalid>";
            }
            if (thread == &g_ProfilerWorkerThread)
            {
                name = "<Worker Thread>";
            }
            NNT_OS_LOG("[%s]: pc=0x%zx 0x%zx (stack 0x%zx:0x%zx) fp=0x%zx sp=0x%zx lr=0x%zx flag=0x%x (%d)\n", name, context.pc, thread, stackTop, stackSize, context.fp, context.sp, context.lr, flag, nn::os::GetCurrentCoreNumber());
        }
        else
        {
            if (result <= nn::os::ResultNoThread())
            {
                // Basically idle
                //NNT_OS_LOG("No thread.\n");
            }
            else if (result <= nn::os::ResultUnknownThread())
            {
                // Thread of other processes were working
                NNT_OS_LOG("Unknown thread.\n");
            }
            else
            {
                // Should not reach here
                NNT_OS_LOG("result=0x%x\n", result.GetInnerValueForDebug());
            }
        }
        nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(500) );
    }
    g_ProfilerWorkerTestDoneFlag = true;
}

TEST(ThreadInfoTest, GetLastThreadInfo)
{
    // Show thread information
    NNT_OS_LOG("MAIN THREAD=0x%p (%d)\n", nn::os::GetCurrentThread(), nn::os::GetCurrentCoreNumber());
    NNT_OS_LOG("SUB THREAD=0x%p\n", &g_ProfilerWorkerThread);

    //
    // Create profiler worker thread (1 core only, so far)
    //
    auto result = nn::os::CreateThread(&g_ProfilerWorkerThread, ProfilerWorkerThread, nullptr, g_ThreadStack, sizeof(g_ThreadStack), nn::os::HighestThreadPriority);
    EXPECT_TRUE( result.IsSuccess() );
    nn::os::StartThread(&g_ProfilerWorkerThread);

    //
    // Dummy task
    //
    int counter = 0;
    while (!g_ProfilerWorkerTestDoneFlag) // Exits, if profiler worker thread works 1000 times
    {
        ++counter;
        nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(10) );
        //nn::os::GetCurrentCoreNumber();
    }

    NNT_OS_LOG("counter: %d\n", counter);

    // Finalize profiler worker thread
    nn::os::WaitThread(&g_ProfilerWorkerThread);
    nn::os::DestroyThread(&g_ProfilerWorkerThread);

    // Should return NULL for destroyed thread
    uintptr_t stackTop;
    size_t    stackSize;
    nn::os::GetThreadStackInfo(&stackTop, &stackSize, &g_ProfilerWorkerThread);
    EXPECT_EQ(stackTop, 0);
    EXPECT_EQ(stackSize, 0);
}

}}} // namespace nnt::os::thread

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    int result;

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of Thread APIs\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of Thread APIs\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}
