﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Event としての SystemEvent 関連機能のテスト
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include <nn/nn_Common.h>
#include <nn/nn_SdkText.h>
#include <nn/os.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"
#include "test_SystemEventAsEventHelper.h"
#include "test_DelayedSystemEvent.h"

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

#define EVENT_AUTO_CLEAR    nn::os::EventClearMode_AutoClear
#define EVENT_MANUAL_CLEAR  nn::os::EventClearMode_ManualClear

namespace nnt { namespace os { namespace systemEventAsEvent {

//---------------------------------------------------------------------------
// CreateSystemEvent 直後の初期状態チェック

TEST(CreateSystemEvent, test_initiallySignaled1)
{
    // CreateSystemEvent 初期状態チェック
    doTest1(EVENT_AUTO_CLEAR,           // SystemEvent は自動クリアモード
            THR_NOTUSE,                 // Thread1 も未使用
            true,                       // Thread1 での API 返値
            false,                      // SystemEvent は Not Signaled
            THR_STATE_NONE);            // Thread1 は終了
}

TEST(CreateSystemEvent, test_initiallySignaled2)
{
    // CreateSystemEvent 初期状態チェック
    doTest1(EVENT_MANUAL_CLEAR,         // SystemEvent は手動クリアモード
            THR_NOTUSE,                 // Thread1 も未使用
            true,                       // Thread1 での API 返値
            false,                      // SystemEvent は Not Signaled
            THR_STATE_NONE);            // Thread1 は終了
}

//---------------------------------------------------------------------------
// 単独の SignalSystemEvent() テスト

TEST(SignalSystemEvent, test_NotSignalToSignal)
{
    // SignalSystemEvent 非シグナルからシグナルへ
    doTest1(EVENT_AUTO_CLEAR,           // SystemEvent は自動クリアモード
            THR_CALL_SIGNAL,            // Thread1 は SignalSystemEvent()
            true,                       // Thread1 での API 返値
            true,                       // SystemEvent は Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

//---------------------------------------------------------------------------

// 単独の ClearSystemEvent() テスト

TEST(ClearSystemEvent, test_NotSignalToNotSignal1)
{
    // SignalSystemEvent 非シグナルからシグナルへ
    doTest1(EVENT_AUTO_CLEAR,           // SystemEvent は自動クリアモード
            THR_CALL_CLEAR,             // Thread1 は ClearSystemEvent()
            true,                       // Thread1 での API 返値
            false,                      // SystemEvent は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(ClearSystemEvent, test_NotSignalToNotSignal2)
{
    // SignalSystemEvent 非シグナルからシグナルへ
    doTest1(EVENT_MANUAL_CLEAR,         // SystemEvent は手動クリアモード
            THR_CALL_CLEAR,             // Thread1 は ClearSystemEvent()
            true,                       // Thread1 での API 返値
            false,                      // SystemEvent は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

//---------------------------------------------------------------------------
// 単独の WaitSystemEvent() AutoClear テスト

TEST(WaitSystemEventAutoClear, test_WaitSystemEventAutoClear2)
{
    // AUTO_CLEAR/NotSignaled のイベントに対して WaitSystemEvent 発行
    doTest1(EVENT_AUTO_CLEAR,           // SystemEvent は自動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitSystemEvent()
            true,                       // Thread1 での API 返値
            false,                      // SystemEvent は Not Signaled
            THR_STATE_WAITING);         // Thread1 はシグナル待ち
}

TEST(WaitSystemEventAutoClear, test_WaitSystemEventAutoClear4)
{
    // AUTO_CLEAR/NotSignaled のイベントに対して WaitSystemEvent 発行×２スレッド
    doTest2(EVENT_AUTO_CLEAR,           // SystemEvent は自動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitSystemEvent()
            THR_WAIT_WAIT,              // Thread1 は WaitSystemEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            false,                      // SystemEvent は Not Signaled
            THR_STATE_WAITING,          // Thread1 はシグナル待ち
            THR_STATE_WAITING);         // Thread2 はシグナル待ち
}

//---------------------------------------------------------------------------
// 単独の WaitSystemEvent() ManualClear テスト

TEST(WaitSystemEventManualClear, test_WaitSystemEventManualClear2)
{
    // MANUAL_CLEAR/NotSignaled のイベントに対して WaitSystemEvent 発行
    doTest1(EVENT_MANUAL_CLEAR,         // SystemEvent は手動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitSystemEvent()
            true,                       // Thread1 での API 返値
            false,                      // SystemEvent は Not Signaled
            THR_STATE_WAITING);         // Thread1 はシグナル待ち
}

TEST(WaitSystemEventManualClear, test_WaitSystemEventManualClear4)
{
    // MANUAL_CLEAR/NotSignaled のイベントに対して WaitSystemEvent 発行×２スレッド
    doTest2(EVENT_MANUAL_CLEAR,         // SystemEvent は手動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitSystemEvent()
            THR_WAIT_WAIT,              // Thread1 は WaitSystemEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            false,                      // SystemEvent は Not Signaled
            THR_STATE_WAITING,          // Thread1 はシグナル待ち
            THR_STATE_WAITING);         // Thread2 はシグナル待ち
}

//---------------------------------------------------------------------------
// 単独の TryWaitSystemEvent() AutoClear テスト

TEST(TryWaitSystemEventAutoClear, test_TryWaitSystemEventAutoClear2)
{
    // AUTO_CLEAR/NotSignaled のイベントに対して TryWaitSystemEvent 発行
    doTest1(EVENT_AUTO_CLEAR,           // SystemEvent は自動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitSystemEvent()
            false,                      // Thread1 での API 返値
            false,                      // SystemEvent は Not Signaled
            THR_STATE_EXITED);          // Thread1 はシグナル待ち
}

TEST(TryWaitSystemEventAutoClear, test_TryWaitSystemEventAutoClear4)
{
    // AUTO_CLEAR/NotSignaled のイベントに対して TryWaitSystemEvent 発行×２スレッド
    doTest2(EVENT_AUTO_CLEAR,           // SystemEvent は自動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitSystemEvent()
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitSystemEvent()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            false,                      // SystemEvent は Not Signaled
            THR_STATE_EXITED,           // Thread1 はシグナル待ち
            THR_STATE_EXITED);          // Thread2 はシグナル待ち
}

//---------------------------------------------------------------------------
// 単独の TryWaitSystemEvent() ManualClear テスト

TEST(TryWaitSystemEventManualClear, test_TryWaitSystemEventManualClear2)
{
    // MANUAL_CLEAR/NotSignaled のイベントに対して TryWaitSystemEvent 発行
    doTest1(EVENT_MANUAL_CLEAR,         // SystemEvent は手動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitSystemEvent()
            false,                      // Thread1 での API 返値
            false,                      // SystemEvent は Not Signaled
            THR_STATE_EXITED);          // Thread1 はシグナル待ち
}

TEST(TryWaitSystemEventManualClear, test_TryWaitSystemEventManualClear4)
{
    // MANUAL_CLEAR/NotSignaled のイベントに対して TryWaitSystemEvent 発行×２スレッド
    doTest2(EVENT_MANUAL_CLEAR,         // SystemEvent は手動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitSystemEvent()
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitSystemEvent()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            false,                      // SystemEvent は Not Signaled
            THR_STATE_EXITED,           // Thread1 はシグナル待ち
            THR_STATE_EXITED);          // Thread2 はシグナル待ち
}

//---------------------------------------------------------------------------
// 単独の TimedWaitSystemEvent() AutoClear テスト

TEST(TimedWaitSystemEventAutoClear, test_TimedWaitSystemEventAutoClear2)
{
    // AUTO_CLEAR/NotSignaled のイベントに対して TimedWaitSystemEvent 発行
    doTest1(EVENT_AUTO_CLEAR,           // SystemEvent は自動クリアモード
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitSystemEvent()
            false,                      // Thread1 での API 返値
            false,                      // SystemEvent は Not Signaled
            THR_STATE_TIMEDOUT);        // Thread1 はタイムアウト
}

TEST(TimedWaitSystemEventAutoClear, test_TimedWaitSystemEventAutoClear4)
{
    // AUTO_CLEAR/NotSignaled のイベントに対して TimedWaitSystemEvent 発行×２スレッド
    doTest2(EVENT_AUTO_CLEAR,           // SystemEvent は自動クリアモード
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitSystemEvent()
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitSystemEvent()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            false,                      // SystemEvent は Not Signaled
            THR_STATE_TIMEDOUT,         // Thread1 はタイムアウト
            THR_STATE_TIMEDOUT);        // Thread2 はタイムアウト
}

//---------------------------------------------------------------------------
// 単独の TimedWaitSystemEvent() ManualClear テスト

TEST(TimedWaitSystemEventManualClear, test_TimedWaitSystemEventManualClear2)
{
    // MANUAL_CLEAR/NotSignaled のイベントに対して TimedWaitSystemEvent 発行
    doTest1(EVENT_MANUAL_CLEAR,         // SystemEvent は手動クリアモード
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitSystemEvent()
            false,                      // Thread1 での API 返値
            false,                      // SystemEvent は Not Signaled
            THR_STATE_TIMEDOUT);        // Thread1 はタイムアウト
}

TEST(TimedWaitSystemEventManualClear, test_TimedWaitSystemEventManualClear4)
{
    // MANUAL_CLEAR/NotSignaled のイベントに対して TimedWaitSystemEvent 発行×２スレッド
    doTest2(EVENT_MANUAL_CLEAR,         // SystemEvent は手動クリアモード
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitSystemEvent()
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitSystemEvent()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            false,                      // SystemEvent は Not Signaled
            THR_STATE_TIMEDOUT,         // Thread1 はタイムアウト
            THR_STATE_TIMEDOUT);        // Thread2 はタイムアウト
}

//---------------------------------------------------------------------------
// AUTO_CLEAR/Wait 待機中に Signal する（TimedWait 含む）

TEST(CombinedSystemEventTestAutoClear, test_WaitAndSignalAutoClear)
{
    // Thread1 で WaitSystemEvent()、Thread2 で SignalSystemEvent()
    doTest2(EVENT_AUTO_CLEAR,           // SystemEvent は自動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitSystemEvent()
            THR_CALL_SIGNAL,            // Thread2 は SignalSystemEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            false,                      // SystemEvent は Not Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

TEST(CombinedSystemEventTestAutoClear, test_TimdeWaitAndSignalAutoClear)
{
    // Thread1 で WaitSystemEvent()、Thread2 で SignalSystemEvent()
    doTest2(EVENT_AUTO_CLEAR,           // SystemEvent は自動クリアモード
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitSystemEvent()
            THR_CALL_SIGNAL,            // Thread2 は SignalSystemEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            false,                      // SystemEvent は Not Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

//---------------------------------------------------------------------------
// MANUAL_CLEAR/Wait 待機中に Signal する（TimedWait 含む）

TEST(CombinedSystemEventTestManualClear, test_WaitAndSignalManualClear)
{
    // Thread1 で WaitSystemEvent()、Thread2 で SignalSystemEvent()
    doTest2(EVENT_MANUAL_CLEAR,         // SystemEvent は手動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitSystemEvent()
            THR_CALL_SIGNAL,            // Thread2 は SignalSystemEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            true,                       // SystemEvent は Not Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

TEST(CombinedSystemEventTestManualClear, test_TimdeWaitAndSignalManualClear)
{
    // Thread1 で WaitSystemEvent()、Thread2 で SignalSystemEvent()
    doTest2(EVENT_MANUAL_CLEAR,         // SystemEvent は手動クリアモード
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitSystemEvent()
            THR_CALL_SIGNAL,            // Thread2 は SignalSystemEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            true,                       // SystemEvent は Not Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}


//---------------------------------------------------------------------------
// 多重待ちによる Event としての SystemEvent 受理（AutoReset）
//
//  システムイベントを生成し、WaitSystemEvent の動作を確認する。
//  DelayedSystemEventServer 機能を使い、
//  システムイベントに対して Signal してもらう。
//
TEST(SystemEventAsEventWaitAny, test_WaitAny_SystemEventAsEventAutoReset)
{
    nn::os::SystemEventType         systemEvent;
    nn::os::MultiWaitType           multiWait;
    nn::os::MultiWaitHolderType     holder;

    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();
    NNT_OS_LOG(NN_TEXT("<< AutoReset の SystemEvent に対する WaitAny のテスト >>\n"));
    NNT_OS_LOG(NN_TEXT("<< 事前に server に通知して 200msec 後に Signal してもらう。 >>\n"));

    // システムイベントの初期化
    nn::os::CreateSystemEvent( &systemEvent, EVENT_AUTO_CLEAR, false );
    nn::os::InitializeMultiWait( &multiWait );
    nn::os::InitializeMultiWaitHolder( &holder, &systemEvent );
    nn::os::LinkMultiWaitHolder( &multiWait, &holder );

    {   // サーバーを起動
        detail::DelayedSystemEventServer  server( &systemEvent );

        // 初期化直後が非シグナル状態化をチェック
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("CreateSystemEvent() 直後は非シグナル状態か"));
        bool ret = TryWaitSystemEvent( &systemEvent );
        CheckBool( ret == false );

        // 通知後にすぐに多重待ち
        server.RequireDelayedSignal( nn::TimeSpan::FromMilliSeconds(200) );
        nn::os::WaitAny( &multiWait );

#if 0   // TORIEAEZU:
        //  SIGLO-3875 の対応に合わせて修正する。

        // sytemEvent が正しく AUTO_CLEAR されているかチェックする。
        // TryWaitSystemEvent() で false なら期待通りクリアされている。
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("WaitAny() 後に自動クリアされているか"));
        ret = TryWaitSystemEvent( &systemEvent );
        CheckBool( ret == false );
#endif
    }

    // システムイベントの削除
    nn::os::UnlinkMultiWaitHolder( &holder );
    nn::os::FinalizeMultiWaitHolder( &holder );
    nn::os::DestroySystemEvent( &systemEvent );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
// 多重待ちによる Event としての SystemEvent 受理（ManualReset）
//
//  システムイベントを生成し、WaitSystemEvent の動作を確認する。
//  DelayedSystemEventServer 機能を使い、
//  システムイベントに対して Signal してもらう。
//
TEST(SystemEventAsEventWaitAny, test_WaitAny_SystemEventAsEventManualReset)
{
    nn::os::SystemEventType         systemEvent;
    nn::os::MultiWaitType           multiWait;
    nn::os::MultiWaitHolderType     holder;

    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();
    NNT_OS_LOG(NN_TEXT("<< ManualReset の SystemEvent に対する WaitAny のテスト >>\n"));
    NNT_OS_LOG(NN_TEXT("<< 事前に server に通知して 200msec 後に Signal してもらう。 >>\n"));

    // システムイベントの初期化
    nn::os::CreateSystemEvent( &systemEvent, EVENT_MANUAL_CLEAR, false );
    nn::os::InitializeMultiWait( &multiWait );
    nn::os::InitializeMultiWaitHolder( &holder, &systemEvent );
    nn::os::LinkMultiWaitHolder( &multiWait, &holder );

    {   // サーバーを起動
        detail::DelayedSystemEventServer  server( &systemEvent );

        // 初期化直後が非シグナル状態化をチェック
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("CreateSystemEvent() 直後は非シグナル状態か"));
        bool ret = TryWaitSystemEvent( &systemEvent );
        CheckBool( ret == false );

        // 通知後にすぐに多重待ち
        server.RequireDelayedSignal( nn::TimeSpan::FromMilliSeconds(200) );
        nn::os::WaitAny( &multiWait );

        // sytemEvent が正しく AUTO_CLEAR されているかチェックする。
        // TryWaitSystemEvent() で false なら期待通りクリアされている。
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("WaitAny() 後にシグナル状態が維持されているか"));
        ret = TryWaitSystemEvent( &systemEvent );
        CheckBool( ret == true );
    }

    // システムイベントの削除
    nn::os::UnlinkMultiWaitHolder( &holder );
    nn::os::FinalizeMultiWaitHolder( &holder );
    nn::os::DestroySystemEvent( &systemEvent );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
// 多重待ちによる Event としての TryWaitSystemEvent 動作テスト
//
//  システムイベントを生成し、TryWaitSystemEvent の動作を確認する。
//  DelayedSystemEventServer 機能を使い、
//  システムイベントに対して Signal してもらう。
//  シグナル化されたシステムイベントに対して TryWait して true を確認した後、
//  非シグナル状態のシステムイベントに対しても TryWait して false を確認する。
//  なお、本テストは ClearSystenEvent() のテストにもなっている。
//
TEST(SystemEventAsEventWaitAny, test_TryWaitAny_SystemEventAsEvent)
{
    nn::os::SystemEventType         systemEvent;
    nn::os::MultiWaitType           multiWait;
    nn::os::MultiWaitHolderType     holder;

    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();
    NNT_OS_LOG(NN_TEXT("<< ManualReset の SystemEvent に対する TryWaitAny のテスト >>\n"));
    NNT_OS_LOG(NN_TEXT("<< 事前に server に通知して 200msec 後に Signal してもらう。 >>\n"));

    // システムイベントの初期化
    nn::os::CreateSystemEvent( &systemEvent, EVENT_MANUAL_CLEAR, false );
    nn::os::InitializeMultiWait( &multiWait );
    nn::os::InitializeMultiWaitHolder( &holder, &systemEvent );
    nn::os::LinkMultiWaitHolder( &multiWait, &holder );

    {   // サーバーを起動
        detail::DelayedSystemEventServer  server( &systemEvent );

        // 通知後にすぐに TryWait を発行
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("非シグナル状態中に TryWaitAny() でポーリングする。\n"));
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("返値が false から true になったのを確認してループを抜ける。"));

        server.RequireDelayedSignal( nn::TimeSpan::FromMilliSeconds(200) );
        int count = 0;
        for (;;)
        {
            nn::os::MultiWaitHolderType* ret = nn::os::TryWaitAny( &multiWait );
            if (ret != NULL)
            {
                break;
            }
            ++count;
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
        }
        CheckBool( count > 0);

        // その後すぐに TryWait を発行
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("シグナル状態中に TryWaitAny() で true を確認"));

        nn::os::MultiWaitHolderType* ret = nn::os::TryWaitAny( &multiWait );
        CheckBool( ret == &holder );

        // 一旦クリアする
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ClearSystemEvent() を発行\n"));
        nn::os::ClearSystemEvent( &systemEvent );

        // すぐに TryWait を発行
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("非シグナル状態中に TryWaitAny() で false を確認"));
        ret = nn::os::TryWaitAny( &multiWait );
        CheckBool( ret == NULL );
    }

    // システムイベントの削除
    nn::os::UnlinkMultiWaitHolder( &holder );
    nn::os::FinalizeMultiWaitHolder( &holder );
    nn::os::DestroySystemEvent( &systemEvent );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
// 多重待ちによる Event としての TimedWaitSystemEvent 動作テスト
//
//  プロセス間同期イベントを生成し、TimedWaitInterProcessEvent の動作を
//  確認する。Pipe 機能を使い、プロセス間同期イベントの WritableHandle を
//  サーバに渡し、200msec 後に SignalInterProcessEvent() してもらう。
//
//  タイムアウト 1msec で、うまくタイムアウトできることの確認。
//  タイムアウト 2sec  で、それまでにシグナルが受理されることの確認。
//
TEST(SystemEventAsEventWaitAny, test_TimedWaitAny_SystemEventAsEvent)
{
    nn::os::SystemEventType         systemEvent;
    nn::os::MultiWaitType           multiWait;
    nn::os::MultiWaitHolderType     holder;

    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();
    NNT_OS_LOG(NN_TEXT("<< まず最初に、単純にタイムアウトが成立するかを確認 >>\n"));
    NNT_OS_LOG(NN_TEXT("<< その後、タイムアウト前に起床されるかを確認 >>\n"));

    // システムイベントの初期化
    nn::os::CreateSystemEvent( &systemEvent, EVENT_MANUAL_CLEAR, false );
    nn::os::InitializeMultiWait( &multiWait );
    nn::os::InitializeMultiWaitHolder( &holder, &systemEvent );
    nn::os::LinkMultiWaitHolder( &multiWait, &holder );

    // 単純に TimedWaitAny でタイムアウトが成立するのを確認する
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("TimedWaitAny ですぐにタイムアウトが成立するか"));
    auto* ret = nn::os::TimedWaitAny( &multiWait, nn::TimeSpan::FromMilliSeconds(1) );
    CheckBool( ret == NULL );

    {   // サーバーを起動
        detail::DelayedSystemEventServer  server( &systemEvent );

        // 通知後にすぐに TimedWait でタイムアウト前に起床されるのを確認する
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("TimedWaitAny でタイムアウト前に起床されるか"));
        server.RequireDelayedSignal( nn::TimeSpan::FromMilliSeconds(200) );
        ret = nn::os::TimedWaitAny( &multiWait, nn::TimeSpan::FromMilliSeconds(2000) );
        CheckBool( ret == &holder );

        // シグナル状態が true かを確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("シグナル状態中に TryWaitAny() で true を確認"));

        ret = nn::os::TryWaitAny( &multiWait );
        CheckBool( ret == &holder );

        // 一旦クリアする
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ClearSystemEvent() を発行\n"));
        nn::os::ClearSystemEvent( &systemEvent );

        // シグナル状態が false かを確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("非シグナル状態中に TryWaitAny() で false を確認"));
        ret = nn::os::TryWaitAny( &multiWait );
        CheckBool( ret == NULL );
    }

    // システムイベントの削除
    nn::os::UnlinkMultiWaitHolder( &holder );
    nn::os::FinalizeMultiWaitHolder( &holder );
    nn::os::DestroySystemEvent( &systemEvent );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  SystemEvent クラスの Event としてのテストその１
//  ここでは、SystemEvent クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  test_SystemEventClassTestAutoClear ではブロックしないテストを実施
//
//  このテストは Event で行なっているものと内容的に等価である。
//
TEST(SystemEventClass, test_SystemEventClassTestAutoClear)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan  timeout = nn::TimeSpan::FromMicroSeconds(1);

        SEQ_SET(4000);
        NNT_OS_LOG(NN_TEXT("main: systemEvent インスタンスの生成（AutoClear）\n"));
        nn::os::SystemEvent systemEvent(nn::os::EventClearMode_AutoClear, false);

        // ここからがテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: systemEvent.TryWait() が false を返す"));
        CheckBool( systemEvent.TryWait() == false );

        // Wait のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: systemEvent.Signal() に成功する\n"));
        systemEvent.Signal();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: systemEvent.Wait() に成功する\n"));
        systemEvent.Wait();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: systemEvent.TryWait() が false を返す"));
        CheckBool( systemEvent.TryWait() == false );

        // TryWait のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: systemEvent.Signal() に成功する\n"));
        systemEvent.Signal();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: systemEvent.TryWait() が true を返す"));
        CheckBool( systemEvent.TryWait() == true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: systemEvent.TryWait() が false を返す"));
        CheckBool( systemEvent.TryWait() == false );

        // TimedWait のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: systemEvent.Signal() に成功する\n"));
        systemEvent.Signal();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: systemEvent.TimedWait() が true を返す"));
        CheckBool( systemEvent.TimedWait(timeout) == true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: systemEvent.TimedWait(0) が false を返す"));
        CheckBool( systemEvent.TimedWait(nn::TimeSpan(0)) == false );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  SystemEvent クラスの Event としてのテストその２
//  ここでは、SystemEvent クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  test_SystemEventClassTestManualClear ではブロックしないテストを実施
//
//  このテストは Event で行なっているものと内容的に等価である。
//
TEST(SystemEventClass, test_SystemEventClassTestManualClear)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan  timeout = nn::TimeSpan::FromMicroSeconds(1);

        SEQ_SET(4100);
        NNT_OS_LOG(NN_TEXT("main: systemEvent インスタンスの生成（ManualClear）\n"));
        nn::os::SystemEvent systemEvent(nn::os::EventClearMode_ManualClear, false);

        // ここからがテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: systemEvent.TryWait() が false を返す"));
        CheckBool( systemEvent.TryWait() == false );

        // Wait のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: systemEvent.Signal() に成功する\n"));
        systemEvent.Signal();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: systemEvent.Wait() に成功する\n"));
        systemEvent.Wait();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: systemEvent.TryWait() が true を返す"));
        CheckBool( systemEvent.TryWait() == true );

        // TryWait のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: systemEvent.Signal() に成功する\n"));
        systemEvent.Signal();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: systemEvent.TryWait() が true を返す"));
        CheckBool( systemEvent.TryWait() == true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: systemEvent.TryWait() が true を返す"));
        CheckBool( systemEvent.TryWait() == true );

        // TimedWait のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: systemEvent.Signal() に成功する\n"));
        systemEvent.Signal();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: systemEvent.TimedWait() が true を返す"));
        CheckBool( systemEvent.TimedWait(timeout) == true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: systemEvent.TimedWait(0) が true を返す"));
        CheckBool( systemEvent.TimedWait(nn::TimeSpan(0)) == true );

        // Clear の確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: systemEvent.Clear() に成功する\n"));
        systemEvent.Clear();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: systemEvent.TryWait() が false を返す"));
        CheckBool( systemEvent.TryWait() == false );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  SystemEvent クラスの Event としてのテスト
//  ここでは、SystemEvent クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  test_SystemEventClassTestBlock ではブロック系テストを実施。
//
//  このテストは Event で行なっているものと内容的に等価である。
//
NN_ALIGNAS( 4096 ) char g_StackOfThread[ 8192 ];

void SystemEventClassTestThread(void* arg)
{
    auto&        systemEvent    = *static_cast<nn::os::SystemEvent*>(arg);
    const nn::TimeSpan interval = nn::TimeSpan::FromMilliSeconds(100);

    for (int i = 0; i < 2; ++i)
    {
        nn::os::SleepThread( interval );

        SEQ_CHECK(4202 + i * 3);
        NNT_OS_LOG(NN_TEXT("sub:  systemEvent.Signal() でシグナル化\n"));
        systemEvent.Signal();
    }
}

TEST(SystemEventClass, test_SystemEventClassTestBlock)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan  timeout = nn::TimeSpan::FromSeconds(1);

        SEQ_SET(4200);
        NNT_OS_LOG(NN_TEXT("main: SystemEvent インスタンスの生成"));
        nn::os::SystemEvent systemEvent(nn::os::EventClearMode_AutoClear, false);

        // テスト用の子スレッドを作成
        nn::os::ThreadType  thread;
        {
            auto pri    = nn::os::GetThreadPriority(nn::os::GetCurrentThread());
            auto result = nn::os::CreateThread( &thread,
                                                &SystemEventClassTestThread,
                                                &systemEvent,
                                                g_StackOfThread,
                                                sizeof(g_StackOfThread),
                                                pri + 1);
            CheckBool( result.IsSuccess() );
        }

        // 子スレッドを起動
        nn::os::StartThread( &thread );

        // Step1: Wait で待機して子スレッドから起床
        SEQ_CHECK(4201);
        NNT_OS_LOG(NN_TEXT("main: evnet.Wait() で待機\n"));
        systemEvent.Wait();
        SEQ_CHECK(4203);
        NNT_OS_LOG(NN_TEXT("main: -> 起床\n"));

        // Step2: TimedWait で待機し、子スレッドからシグナルしてもらう
        SEQ_CHECK(4204);
        NNT_OS_LOG(NN_TEXT("main: systemEvent.TimedWait() で待機\n"));
        systemEvent.TimedWait(timeout);
        SEQ_CHECK(4206);
        NNT_OS_LOG(NN_TEXT("main: -> 起床\n"));

        // 子スレッドの終了待ち
        nn::os::WaitThread( &thread );
        nn::os::DestroyThread( &thread );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  SystemEvent クラスの型変換関数のテスト
//  ここでは、SystemEvent クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  ここでは、GetBase() および operator SystemEventType&() の動作テストを行なう。
//
TEST(SystemEventClass, test_SystemEventClassTestTypeExchange)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        SEQ_SET(4300);
        NNT_OS_LOG(NN_TEXT("systemEvent インスタンスの生成（AutoClear）\n"));
        nn::os::SystemEvent systemEvent(nn::os::EventClearMode_AutoClear, false);

        // ここからがテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("systemEvent.GetBase() で SystemEventType オブジェクトを取得"));
        nn::os::SystemEventType*  systemEventType = systemEvent.GetBase();
        CheckBool( systemEventType != NULL );

        // 初期シグナル状態が false であることを確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("systemEvent.TryWait() で非シグナル状態であることを確認"));
        CheckBool( systemEvent.TryWait() == false );

        // SystemEventType にてセットし、TryWait で true を確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("systemEventType を使ってシグナル状態にセット"));
        nn::os::SignalSystemEvent(systemEventType);
        CheckBool( systemEvent.TryWait() == true );

        // SystemEventType にてクリアし、TryWait で false を確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("systemEventType を使って非シグナル状態にクリア"));
        nn::os::ClearSystemEvent(systemEventType);
        CheckBool( systemEvent.TryWait() == false );

        // operator SystemEventType&() の確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("operator SystemEventType&() を使ってオブジェクトの参照を取得"));
        nn::os::SystemEventType& systemEventTypeRefer = systemEvent;
        CheckBool( &systemEventTypeRefer == systemEventType );

        // operator const SystemEventType&() の確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("operator const SystemEventType&() を使ってオブジェクトの参照を取得"));
        const nn::os::SystemEventType& systemEventTypeReferConst = systemEvent;
        CheckBool( &systemEventTypeReferConst == systemEventType );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

}}} // namespace nnt::os::systemEventAsEvent

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    int result;

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of SystemEvent APIs\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of SystemEvent APIs\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}
