﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Semaphore 関連機能のテスト
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

// windows.h で min/max が定義されるのを抑制する
#ifndef NOMINMAX
#define NOMINMAX
#endif

#include <limits>

#include <nn/nn_Common.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Macro.h>
#include <nn/os.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"
#include "test_SemaphoreHelper.h"

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

namespace nnt { namespace os { namespace semaphore {

//---------------------------------------------------------------------------
// 単独の AcquireSemaphore() テスト

TEST(AcquireSemaphore, test_AcquireSemaphore1)
{
    // AcquireSemaphore 境界条件１
    doTest1(1,                          // Semaphore 初期値は 1
            1,                          // Semaphore 最大値は 1
            THR_CALL_ACQ,               // Thread1 は AcquireSemaphore()
            true,                       // Thread1 での API 返値
            0,                          // Semaphore 期待値は 0
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(AcquireSemaphore, test_AcquireSemaphore2)
{
    // AcquireSemaphore 境界条件２（待ち）
    doTest1(0,                          // Semaphore 初期値は 0
            1,                          // Semaphore 最大値は 1
            THR_WAIT_ACQ,               // Thread1 は AcquireSemaphore()
            true,                       // Thread1 での API 返値
            0,                          // Semaphore 期待値は 0
            THR_STATE_WAITING);         // Thread1 は待ち状態
}

TEST(AcquireSemaphore, test_AcquireSemaphore3)
{
    // AcquireSemaphore 待ち(num=max-1、Thread1 は通過、Thread2 は待機)
    doTest2(1,                          // Semaphore 初期値は 1
            1,                          // Semaphore 最大値は 1
            THR_CALL_ACQ,               // Thread1 は AcquireSemaphore()
            THR_WAIT_ACQ,               // Thread2 は AcquireSemaphore()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            0,                          // Semaphore 期待値は 0
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_WAITING);         // Thread2 は待ち状態
}

TEST(AcquireSemaphore, test_AcquireSemaphore4)
{
    // AcquireSemaphore 待ち(num=max、スレッド２つ)
    doTest2(0,                          // Semaphore 初期値は 0
            1,                          // Semaphore 最大値は 1
            THR_WAIT_ACQ,               // Thread1 は AcquireSemaphore()
            THR_WAIT_ACQ,               // Thread2 は AcquireSemaphore()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            0,                          // Semaphore 期待値は 0
            THR_STATE_WAITING,          // Thread1 は待ち状態
            THR_STATE_WAITING);         // Thread2 は待ち状態
}

//---------------------------------------------------------------------------
// 単独の TryAcquireSemaphore() テスト

TEST(TryAcquireSemaphore, test_TryAcquireSemaphore1)
{
    // TryAcquireSemaphore 境界条件１
    doTest1(1,                          // Semaphore 初期値は 1
            1,                          // Semaphore 最大値は 1
            THR_CALL_TRY_ACQ,           // Thread1 は TryAcquireSemaphore()
            true,                       // Thread1 での API 返値
            0,                          // Semaphore 期待値は 0
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TryAcquireSemaphore, test_TryAcquireSemaphore2)
{
    // TryAcquireSemaphore 境界条件２（待ち）
    doTest1(0,                          // Semaphore 初期値は 0
            1,                          // Semaphore 最大値は 1
            THR_CALL_TRY_ACQ,           // Thread1 は TryAcquireSemaphore()
            false,                      // Thread1 での API 返値
            0,                          // Semaphore 期待値は 0
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TryAcquireSemaphore, test_TryAcquireSemaphore3)
{
    // TryAcquireSemaphore 待ち(num=max-1、Thread1 は通過、Thread2 は待機)
    doTest2(1,                          // Semaphore 初期値は 1
            1,                          // Semaphore 最大値は 1
            THR_CALL_TRY_ACQ,           // Thread1 は TryAcquireSemaphore()
            THR_CALL_TRY_ACQ,           // Thread2 は TryAcquireSemaphore()
            true,                       // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            0,                          // Semaphore 期待値は 0
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は終了
}

TEST(TryAcquireSemaphore, test_TryAcquireSemaphore4)
{
    // TryAcquireSemaphore 待ち(num=max、スレッド２つ)
    doTest2(0,                          // Semaphore 初期値は 0
            1,                          // Semaphore 最大値は 1
            THR_CALL_TRY_ACQ,           // Thread1 は TryAcquireSemaphore()
            THR_CALL_TRY_ACQ,           // Thread2 は TryAcquireSemaphore()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            0,                          // Semaphore 期待値は 0
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は終了
}


//---------------------------------------------------------------------------
// 単独の TimedAcquireSemaphore() テスト

TEST(TimedAcquireSemaphore, test_TimedAcquireSemaphore1)
{
    // TimedAcquireSemaphore 境界条件１
    doTest1(1,                          // Semaphore 初期値は 1
            1,                          // Semaphore 最大値は 1
            THR_CALL_TIMED_ACQ,         // Thread1 は TimedAcquireSemaphore()
            true,                       // Thread1 での API 返値
            0,                          // Semaphore 期待値は 0
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TimedAcquireSemaphore, test_TimedAcquireSemaphore2)
{
    // TimedAcquireSemaphore 境界条件２（待ち）
    doTest1(0,                          // Semaphore 初期値は 0
            1,                          // Semaphore 最大値は 1
            THR_WAIT_TIMED_ACQ,         // Thread1 は TimedAcquireSemaphore()
            false,                      // Thread1 での API 返値
            0,                          // Semaphore 期待値は 0
            THR_STATE_TIMEDOUT);        // Thread1 はタイムアウト
}

TEST(TimedAcquireSemaphore, test_TimedAcquireSemaphore3)
{
    // TimedAcquireSemaphore 待ち(num=max-1、Thread1 は通過、Thread2 は待機)
    doTest2(1,                          // Semaphore 初期値は 1
            1,                          // Semaphore 最大値は 1
            THR_CALL_TIMED_ACQ,         // Thread1 は TimedAcquireSemaphore()
            THR_WAIT_TIMED_ACQ,         // Thread2 は TimedAcquireSemaphore()
            true,                       // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            0,                          // Semaphore 期待値は 0
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_TIMEDOUT);        // Thread2 はタイムアウト
}

TEST(TimedAcquireSemaphore, test_TimedAcquireSemaphore4)
{
    // TimedAcquireSemaphore 待ち(num=max、スレッド２つ)
    doTest2(0,                          // Semaphore 初期値は 0
            1,                          // Semaphore 最大値は 1
            THR_WAIT_TIMED_ACQ,         // Thread1 は TimedAcquireSemaphore()
            THR_WAIT_TIMED_ACQ,         // Thread2 は TimedAcquireSemaphore()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            0,                          // Semaphore 期待値は 0
            THR_STATE_TIMEDOUT,         // Thread1 はタイムアウト
            THR_STATE_TIMEDOUT);        // Thread2 はタイムアウト
}


// 負のタイムアウト値を許容することになったときのためにテストを残しておく
#if 0
//---------------------------------------------------------------------------
// 単独の TimedAcquireSemaphore() テスト（負のタイムアウト値）

TEST(TimedAcquireSemaphoreWithMinusTimeout, test_TimedAcquireSemaphoreWithMinusTimeout1)
{
    // TimedAcquireSemaphore 境界条件１
    doTest1(1,                          // Semaphore 初期値は 1
            1,                          // Semaphore 最大値は 1
            THR_CALL_TIMED_ACQ_MINUS,   // Thread1 は負のタイムアウト値で TimedAcquireSemaphore()
            true,                       // Thread1 での API 返値
            0,                          // Semaphore 期待値は 0
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TimedAcquireSemaphoreWithMinusTimeout, test_TimedAcquireSemaphoreWithMinusTimeout2)
{
    // TimedAcquireSemaphore 境界条件２（待ち）
    doTest1(0,                          // Semaphore 初期値は 0
            1,                          // Semaphore 最大値は 1
            THR_WAIT_TIMED_ACQ_MINUS,   // Thread1 は負のタイムアウト値で TimedAcquireSemaphore()
            false,                      // Thread1 での API 返値
            0,                          // Semaphore 期待値は 0
            THR_STATE_TIMEDOUT);        // Thread1 はタイムアウト
}

#else
// 負のタイムアウト値を許容しない場合の DEATH TEST

TEST(TimedAcquireSemaphoreWithMinusTimeoutDeathTest, test_TimedAcquireSemaphoreWithMinusTimeoutDeathTest)
{
    // TimedAcquireSemaphore 境界条件１
    EXPECT_DEATH_IF_SUPPORTED(
        doTest1(1,                          // Semaphore 初期値は 1
                1,                          // Semaphore 最大値は 1
                THR_CALL_TIMED_ACQ_MINUS,   // Thread1 は負のタイムアウト値で TimedAcquireSemaphore()
                true,                       // Thread1 での API 返値
                0,                          // Semaphore 期待値は 0
                THR_STATE_EXITED)           // Thread1 は終了
    , "");
}


#endif


//---------------------------------------------------------------------------
// 単独の ReleaseSemaphore() テスト

TEST(ReleaseSemaphore, test_ReleaseSemaphore1)
{
    // ReleaseSemaphore 境界条件１（count=max-1）
    doTest1(0,                          // Semaphore 初期値は 0
            1,                          // Semaphore 最大値は 1
            THR_CALL_REL,               // Thread1 は ReleaseSemaphore()
            true,                       // Thread1 での API 返値
            1,                          // Semaphore 期待値は 1
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(ReleaseSemaphore, test_ReleaseSemaphore2)
{
    // ReleaseSemaphore 境界条件２（count=max）
    EXPECT_DEATH_IF_SUPPORTED(
        doTest1(1,                      // Semaphore 初期値は 1
                1,                      // Semaphore 最大値は 1
                THR_CALL_REL,           // Thread1 は ReleaseSemaphore()
                false,                  // Thread1 での API 返値
                1,                      // Semaphore 期待値は 1
                THR_STATE_EXITED)       // Thread1 は終了
    , "");
}

TEST(ReleaseSemaphore, test_ReleaseSemaphore3)
{
    // ReleaseSemaphore 待ち(num=max-1、Thread1 は通過、Thread2 は待機)
    EXPECT_DEATH_IF_SUPPORTED(
        doTest2(0,                      // Semaphore 初期値は 0
                1,                      // Semaphore 最大値は 1
                THR_CALL_REL,           // Thread1 は ReleaseSemaphore()
                THR_CALL_REL,           // Thread2 は ReleaseSemaphore()
                true,                   // Thread1 での API 返値
                false,                  // Thread2 での API 返値
                1,                      // Semaphore 期待値は 1
                THR_STATE_EXITED,       // Thread1 は終了
                THR_STATE_EXITED)       // Thread2 は終了
    , "");
}

TEST(ReleaseSemaphore, test_ReleaseSemaphore4)
{
    // ReleaseSemaphore 待ち(num=max、スレッド２つ)
    doTest2(0,                          // Semaphore 初期値は 0
            2,                          // Semaphore 最大値は 2
            THR_CALL_REL,               // Thread1 は ReleaseSemaphore()
            THR_CALL_REL,               // Thread2 は ReleaseSemaphore()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            2,                          // Semaphore 期待値は 2
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は終了
}


//---------------------------------------------------------------------------
// 単独の ReleaseSemaphore( count ) テスト

TEST(ReleaseSemaphoreCounts, test_ReleaseSemaphoreCounts1)
{
    // ReleaseSemaphore 待ちなし
    doTest1(0,                          // Semaphore 初期値は 0
            3,                          // Semaphore 最大値は 3
            THR_CALL_REL_NUM3,          // Thread1 は ReleaseSemaphore(3)
            true,                       // Thread1 での API 返値
            3,                          // Semaphore 期待値は 1
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(ReleaseSemaphoreCounts, test_ReleaseSemaphoreCounts4)
{
    // ReleaseSemaphore 待ちなし
    doTest2(0,                          // Semaphore 初期値は 0
            6,                          // Semaphore 最大値は 6
            THR_CALL_REL_NUM3,          // Thread1 は ReleaseSemaphore()
            THR_CALL_REL_NUM3,          // Thread2 は ReleaseSemaphore()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            6,                          // Semaphore 期待値は 6
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は終了
}

//---------------------------------------------------------------------------
// Acquire/TimedAcquire 待機中に Release する

TEST(CombinedSemaphoreTest, test_AcquireAndRelease)
{
    // Thread1 で Acquire()、Thread2 で Release()
    doTest2(0,                          // Semaphore 初期値は 0
            1,                          // Semaphore 最大値は 1
            THR_WAIT_ACQ,               // Thread1 は AcquireSemaphore()
            THR_CALL_REL,               // Thread2 は ReleaseSemaphore()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            0,                          // Semaphore 期待値は 0
            THR_STATE_EXITED,           // Thread1 は待ち解除して終了
            THR_STATE_EXITED);          // Thread2 は終了
}

TEST(CombinedSemaphoreTest, test_TimedAcquireAndRelease)
{
    // Thread1 で TimedAcquire()、Thread2 で Release()
    doTest2(0,                          // Semaphore 初期値は 0
            1,                          // Semaphore 最大値は 1
            THR_WAIT_TIMED_ACQ,         // Thread1 は TimedAcquireSemaphore()
            THR_CALL_REL,               // Thread2 は ReleaseSemaphore()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            0,                          // Semaphore 期待値は 0
            THR_STATE_EXITED,           // Thread1 は待ち解除して終了
            THR_STATE_EXITED);          // Thread2 は終了
}

TEST(CombinedSemaphoreTest, test_AcquireAndReleaseCounts)
{
    // Thread1 で Acquire()、Thread2 で Release()
    doTest2(0,                          // Semaphore 初期値は 0
            3,                          // Semaphore 最大値は 3
            THR_WAIT_ACQ,               // Thread1 は AcquireSemaphore()
            THR_CALL_REL_NUM3,          // Thread2 は ReleaseSemaphore()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            2,                          // Semaphore 期待値は 2
            THR_STATE_EXITED,           // Thread1 は待ち解除して終了
            THR_STATE_EXITED);          // Thread2 は終了
}

TEST(CombinedSemaphoreTest, test_TimedAcquireAndReleaseCounts)
{
    // Thread1 で TimedAcquire()、Thread2 で Release()
    doTest2(0,                          // Semaphore 初期値は 0
            3,                          // Semaphore 最大値は 3
            THR_WAIT_TIMED_ACQ,         // Thread1 は TimedAcquireSemaphore()
            THR_CALL_REL_NUM3,          // Thread2 は ReleaseSemaphore()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            2,                          // Semaphore 期待値は 2
            THR_STATE_EXITED,           // Thread1 は待ち解除して終了
            THR_STATE_EXITED);          // Thread2 は終了
}

//---------------------------------------------------------------------------
//  Semaphore クラスのテスト
//  ここでは、Semaphore クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//
NN_ALIGNAS( 4096 ) char g_StackOfThread[ 8192 ];

void SemaphoreClassTestThread(void* arg)
{
    auto& semaphore = *static_cast<nn::os::Semaphore*>(arg);

    SEQ_CHECK(3002);
    NNT_OS_LOG(NN_TEXT("sub : 子スレッドの起動\n"));

    SEQ_CHECK(3003);
    NNT_OS_LOG(NN_TEXT("sub : semaphore を 5 つ返却\n"));
    semaphore.Release(5);

    while ( semaphore.GetCurrentCount() != 0 )
    {
        nn::os::YieldThread();
    }

    // 10msec 待ってから再度 Release() する
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
    semaphore.Release();

    SEQ_CHECK(3004);
    NNT_OS_LOG(NN_TEXT("sub : 子スレッドの終了\n"));
}


TEST(SemaphoreClass, test_SemaphoreClassTest)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        SEQ_SET(3000);
        NNT_OS_LOG(NN_TEXT("main: Semaphore インスタンスの生成（count=0, max=5）\n"));
        nn::os::Semaphore   semaphore(0, 5);

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: Semaphore カウンタは 0 か"));
        CheckBool( semaphore.GetCurrentCount() == 0 );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: Semaphore::Release() を発行\n"));
        semaphore.Release();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: Semaphore カウンタは 1 か"));
        CheckBool( semaphore.GetCurrentCount() == 1 );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: Semaphore::Release(3) を発行\n"));
        semaphore.Release(3);

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: Semaphore カウンタは 4 か"));
        CheckBool( semaphore.GetCurrentCount() == 4 );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: Semaphore::Release(1) を発行\n"));
        semaphore.Release(1);

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: Semaphore カウンタは 5 か"));
        CheckBool( semaphore.GetCurrentCount() == 5 );

        SEQ_NONE();
        NNT_OS_LOG("main: Semaphore::Acquire()");
        semaphore.Acquire();
        CheckBool( semaphore.GetCurrentCount() == 4 );

        SEQ_NONE();
        NNT_OS_LOG("main: Semaphore::TryAcquire()");
        bool result = semaphore.TryAcquire();
        EXPECT_TRUE( result );
        CheckBool( semaphore.GetCurrentCount() == 3 );

        SEQ_NONE();
        NNT_OS_LOG("main: Semaphore::TimedAcquire(0nsec)");
        result = semaphore.TimedAcquire( nn::TimeSpan::FromNanoSeconds(0) );
        EXPECT_TRUE( result );
        CheckBool( semaphore.GetCurrentCount() == 2 );

        SEQ_NONE();
        NNT_OS_LOG("main: Semaphore::TimedAcquire(1nsec)");
        result = semaphore.TimedAcquire( nn::TimeSpan::FromNanoSeconds(1) );
        EXPECT_TRUE( result );
        CheckBool( semaphore.GetCurrentCount() == 1 );

        SEQ_NONE();
        NNT_OS_LOG("main: Semaphore::TimedAcquire(MAXnsec)");
        result = semaphore.TimedAcquire( nn::TimeSpan::FromNanoSeconds( std::numeric_limits<int64_t>::max() ) );
        EXPECT_TRUE( result );
        CheckBool( semaphore.GetCurrentCount() == 0 );

        SEQ_NONE();
        NNT_OS_LOG("main: Semaphore::TimedAcquire(0nsec)");
        result = semaphore.TimedAcquire( nn::TimeSpan::FromNanoSeconds(0) );
        EXPECT_FALSE( result );
        CheckBool( semaphore.GetCurrentCount() == 0 );

        SEQ_NONE();
        NNT_OS_LOG("main: Semaphore::TimedAcquire(1nsec)");
        result = semaphore.TimedAcquire( nn::TimeSpan::FromNanoSeconds(1) );
        EXPECT_FALSE( result );
        CheckBool( semaphore.GetCurrentCount() == 0 );

        SEQ_NONE();
        NNT_OS_LOG("main: Semaphore::TryAcquire()");
        result = semaphore.TryAcquire();
        EXPECT_FALSE( result );
        CheckBool( semaphore.GetCurrentCount() == 0 );

        // 子スレッドの生成
        SEQ_CHECK(3001);
        NNT_OS_LOG(NN_TEXT("main: 子スレッドを生成"));

        nn::os::ThreadType  thread;
        {
            int pri = nn::os::GetThreadPriority(nn::os::GetCurrentThread());
            auto ret = nn::os::CreateThread( &thread,
                                             &SemaphoreClassTestThread,
                                             &semaphore,
                                             g_StackOfThread,
                                             sizeof(g_StackOfThread),
                                             pri);
            CheckBool( ret.IsSuccess() );
        }

        // 子スレッドを起動
        nn::os::StartThread( &thread );

        // 子スレッドでセマフォが返却されるのを待つ
        for (int i=0; i<5; i++)
        {
            semaphore.Acquire();
            NNT_OS_LOG("(-----): main: Semaphore::Acquire() -> count=%d\n",
                                            semaphore.GetCurrentCount());
        }

        // 最後にタイムアウト解除の確認
        result = semaphore.TimedAcquire( nn::TimeSpan::FromNanoSeconds( std::numeric_limits<int64_t>::max() ) );
        NNT_OS_LOG("(-----): main: Semaphore::TimedAcquire(MAXnsec)");
        EXPECT_TRUE( result );
        CheckBool( semaphore.GetCurrentCount() == 0 );

        // 子スレッドの終了待ち ＆ 削除
        nn::os::WaitThread( &thread );
        nn::os::DestroyThread( &thread );

        // セマフォカウンタの最終確認
        SEQ_CHECK(3005);
        NNT_OS_LOG(NN_TEXT("main: Semaphore カウンタは 0 か"));
        CheckBool( semaphore.GetCurrentCount() == 0 );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  Semaphore クラスの型変換関数のテスト
//  ここでは、Semaphore クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  ここでは、GetBase() および operator SemaphoreType&() の動作テストを行なう。
//
TEST(SemaphoreClass, test_SemaphoreClassTestTypeExchange)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        SEQ_SET(3100);
        NNT_OS_LOG(NN_TEXT("semaphore インスタンスの生成\n"));
        nn::os::Semaphore semaphore(0, 1);

        // ここからがテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("semaphore.GetBase() で SemaphoreType オブジェクトを取得"));
        nn::os::SemaphoreType*  semaphoreType = semaphore.GetBase();
        CheckBool( semaphoreType != NULL );

        // 初期シグナル状態が false であることを確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("semaphore.GetCurrentCount() でカウンタが 0 であることを確認"));
        CheckBool( semaphore.GetCurrentCount() == 0 );

        // SemaphoreType にてセットし、TryWait で true を確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("semaphoreType を使ってセマフォを返却"));
        nn::os::ReleaseSemaphore(semaphoreType);
        CheckBool( semaphore.GetCurrentCount() == 1 );

        // SemaphoreType にてクリアし、TryWait で false を確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("semaphoreType を使ってセマフォを獲得"));
        nn::os::AcquireSemaphore(semaphoreType);
        CheckBool( semaphore.GetCurrentCount() == 0 );

        // operator SemaphoreType&() の確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("operator SemaphoreType&() を使ってオブジェクトの参照を取得"));
        nn::os::SemaphoreType& semaphoreTypeRefer = semaphore;
        CheckBool( &semaphoreTypeRefer == semaphoreType );

        // operator const SemaphoreType&() の確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("operator const SemaphoreType&() を使ってオブジェクトの参照を取得"));
        const nn::os::SemaphoreType& semaphoreTypeReferConst = semaphore;
        CheckBool( &semaphoreTypeReferConst == semaphoreType );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


}}} // namespace nnt::os::semaphore

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    int result;

    nn::os::ChangeThreadPriority(nn::os::GetCurrentThread(),
                                 nn::os::DefaultThreadPriority);

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of Semaphore APIs\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of Semaphore APIs\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}
