﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  SdkMutex 関連機能のテスト
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#include <nn/nn_Windows.h>
#include <thread>
#endif
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"
#include "test_SdkMutexHelper.h"

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

namespace nnt { namespace os { namespace sdkmutex {

//---------------------------------------------------------------------------
// 単独の LockSdkMutex() テスト

TEST(LockSdkMutex, test_LockSdkMutex1)
{
    // LockMutex 再帰許可
    doTest1(true,                       // Mutex 再帰許可は true
            false,                      // Mutex 初期ロック状態は false
            THR_CALL_LOCK,              // Thread1 は LockMutex()
            true,                       // Thread1 での API 返値
            1,                          // Mutex ネストカウンタは 1
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(LockSdkMutex, test_LockSdkMutex2)
{
    // LockMutex 再帰禁止
    doTest1(false,                      // Mutex 再帰許可は false
            false,                      // Mutex 初期ロック状態は false
            THR_CALL_LOCK,              // Thread1 は LockMutex()
            true,                       // Thread1 での API 返値
            1,                          // Mutex ネストカウンタは 1
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(LockSdkMutex, test_LockSdkMutex3)
{
    // LockMutex 再帰許可（連続２回ロック）
    doTest1(true,                       // Mutex 再帰許可は true
            true,                       // Mutex 初期ロック状態は true
            THR_CALL_LOCK,              // Thread1 は LockMutex()
            true,                       // Thread1 での API 返値
            2,                          // Mutex ネストカウンタは 2
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(LockSdkMutex, test_LockSdkMutex4)
{
    // LockMutex 再帰禁止（連続２回ロック）
    EXPECT_DEATH_IF_SUPPORTED(
        doTest1(false,                  // Mutex 再帰許可は false
                true,                   // Mutex 初期ロック状態は true
                THR_CALL_LOCK,          // Thread1 は LockMutex()
                false,                  // Thread1 での API 返値
                1,                      // Mutex ネストカウンタは 1
                THR_STATE_EXITED)       // Thread1 は終了
    , "");
}

TEST(LockSdkMutex, test_LockSdkMutex5)
{
    // LockMutex 再帰許可（２スレッドでロック）
    doTest2(true,                       // Mutex 再帰許可は true
            false,                      // Mutex 初期ロック状態は false
            THR_CALL_LOCK,              // Thread1 は LockMutex()
            THR_WAIT_LOCK,              // Thread2 は LockMutex()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            1,                          // Mutex ネストカウンタは 1
            THR_STATE_WAITING2,         // Thread1 は終了待ち
            THR_STATE_WAITING);         // Thread2 は待ち状態
}

TEST(LockSdkMutex, test_LockSdkMutex6)
{
    // LockMutex 再帰禁止（２スレッドでロック）
    doTest2(false,                      // Mutex 再帰許可は false
            false,                      // Mutex 初期ロック状態は false
            THR_CALL_LOCK,              // Thread1 は LockMutex()
            THR_WAIT_LOCK,              // Thread2 は LockMutex()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            1,                          // Mutex ネストカウンタは 1
            THR_STATE_WAITING2,         // Thread1 は終了
            THR_STATE_WAITING);         // Thread2 は待ち状態
}

//---------------------------------------------------------------------------
// 単独の TryLockMutex() テスト

TEST(TryLockSdkMutex, test_TryLockSdkMutex1)
{
    // TryLockMutex 再帰許可
    doTest1(true,                       // Mutex 再帰許可は true
            false,                      // Mutex 初期ロック状態は false
            THR_CALL_TRY_LOCK,          // Thread1 は TryLockMutex()
            true,                       // Thread1 での API 返値
            1,                          // Mutex ネストカウンタは 1
            THR_STATE_EXITED);          // Thread1 は待ち状態
}

TEST(TryLockSdkMutex, test_TryLockSdkMutex2)
{
    // TryLockMutex 再帰禁止
    doTest1(false,                      // Mutex 再帰許可は false
            false,                      // Mutex 初期ロック状態は false
            THR_CALL_TRY_LOCK,          // Thread1 は TryLockMutex()
            true,                       // Thread1 での API 返値
            1,                          // Mutex ネストカウンタは 1
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TryLockSdkMutex, test_TryLockSdkMutex3)
{
    // TryLockMutex 再帰許可（連続２回ロック）
    doTest1(true,                       // Mutex 再帰許可は true
            true,                       // Mutex 初期ロック状態は true
            THR_CALL_TRY_LOCK,          // Thread1 は TryLockMutex()
            true,                       // Thread1 での API 返値
            2,                          // Mutex ネストカウンタは 2
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TryLockSdkMutex, test_TryLockSdkMutex4)
{
    // TryLockMutex 再帰禁止（連続２回ロック）
    EXPECT_DEATH_IF_SUPPORTED(
        doTest1(false,                  // Mutex 再帰許可は false
                true,                   // Mutex 初期ロック状態は true
                THR_CALL_TRY_LOCK,      // Thread1 は TryLockMutex()
                false,                  // Thread1 での API 返値
                1,                      // Mutex ネストカウンタは 1
                THR_STATE_EXITED)       // Thread1 は終了
    , "");
}

TEST(TryLockSdkMutex, test_TryLockSdkMutex5)
{
    // TryLockMutex 再帰許可（２スレッドでロック）
    doTest2(true,                       // Mutex 再帰許可は true
            false,                      // Mutex 初期ロック状態は false
            THR_CALL_TRY_LOCK,          // Thread1 は TryLockMutex()
            THR_CALL_TRY_LOCK,          // Thread2 は TryLockMutex()
            true,                       // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            1,                          // Mutex ネストカウンタは 1
            THR_STATE_EXITED,           // Thread1 は終了待ち
            THR_STATE_EXITED);          // Thread2 は待ち状態
}

TEST(TryLockSdkMutex, test_TryLockSdkMutex6)
{
    // TryLockMutex 再帰禁止（２スレッドでロック）
    doTest2(false,                      // Mutex 再帰許可は false
            false,                      // Mutex 初期ロック状態は false
            THR_CALL_TRY_LOCK,          // Thread1 は TryLockMutex()
            THR_CALL_TRY_LOCK,          // Thread2 は TryLockMutex()
            true,                       // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            1,                          // Mutex ネストカウンタは 1
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は待ち状態
}


//---------------------------------------------------------------------------
// 単独の UnlockMutex() テスト

TEST(UnlockSdkMutex, test_UnlockSdkMutex1)
{
    // UnlockMutex 再帰許可
    EXPECT_DEATH_IF_SUPPORTED(
        doTest1(true,                   // Mutex 再帰許可は true
                false,                  // Mutex 初期ロック状態は false
                THR_CALL_UNLOCK,        // Thread1 は UnlockMutex()
                false,                  // Thread1 での API 返値
                0,                      // Mutex ネストカウンタは 0
                THR_STATE_EXITED)       // Thread1 は待ち状態
    , "");
}

TEST(UnlockSdkMutex, test_UnlockSdkMutex2)
{
    // UnlockMutex 再帰禁止
    EXPECT_DEATH_IF_SUPPORTED(
        doTest1(false,                  // Mutex 再帰許可は false
                false,                  // Mutex 初期ロック状態は false
                THR_CALL_UNLOCK,        // Thread1 は UnlockMutex()
                false,                  // Thread1 での API 返値
                0,                      // Mutex ネストカウンタは 0
                THR_STATE_EXITED)       // Thread1 は終了
    , "");
}

TEST(UnlockSdkMutex, test_UnlockSdkMutex3)
{
    // UnlockMutex 再帰許可
    doTest1(true,                       // Mutex 再帰許可は true
            true,                       // Mutex 初期ロック状態は true
            THR_CALL_UNLOCK,            // Thread1 は UnlockMutex()
            true,                       // Thread1 での API 返値
            0,                          // Mutex ネストカウンタは 0
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(UnlockSdkMutex, test_UnlockSdkMutex4)
{
    // UnlockMutex 再帰禁止
    doTest1(false,                      // Mutex 再帰許可は false
            true,                       // Mutex 初期ロック状態は true
            THR_CALL_UNLOCK,            // Thread1 は UnlockMutex()
            true,                       // Thread1 での API 返値
            0,                          // Mutex ネストカウンタは 0
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(UnlockSdkMutex, test_UnlockSdkMutex5)
{
    // UnlockMutex 再帰許可（２スレッドでロック）
    EXPECT_DEATH_IF_SUPPORTED(
        doTest2(true,                   // Mutex 再帰許可は true
                false,                  // Mutex 初期ロック状態は false
                THR_CALL_UNLOCK,        // Thread1 は UnlockMutex()
                THR_CALL_UNLOCK,        // Thread2 は UnlockMutex()
                false,                  // Thread1 での API 返値
                false,                  // Thread2 での API 返値
                0,                      // Mutex ネストカウンタは 0
                THR_STATE_EXITED,       // Thread1 は終了待ち
                THR_STATE_EXITED)       // Thread2 は待ち状態
    , "");
}

TEST(UnlockSdkMutex, test_UnlockSdkMutex6)
{
    // UnlockMutex 再帰禁止（２スレッドでロック）
    EXPECT_DEATH_IF_SUPPORTED(
        doTest2(true,                   // Mutex 再帰許可は true
                true,                   // Mutex 初期ロック状態は true
                THR_CALL_UNLOCK,        // Thread1 は UnlockMutex()
                THR_CALL_UNLOCK,        // Thread2 は UnlockMutex()
                true,                   // Thread1 での API 返値
                false,                  // Thread2 での API 返値
                0,                      // Mutex ネストカウンタは 0
                THR_STATE_EXITED,       // Thread1 は終了
                THR_STATE_EXITED)       // Thread2 は待ち状態
    , "");
}

//---------------------------------------------------------------------------
// Lock 獲得中に他スレッドから Unlock してもエラー

TEST(CombinedSdkMutexTest, test_LockAndUnlock1)
{
    // Thread1 で Lock()、Thread2 で Unlock()
    EXPECT_DEATH_IF_SUPPORTED(
        doTest2(true,                   // Mutex 再帰許可は true
                false,                  // Mutex 初期ロック状態は false
                THR_CALL_LOCK,          // Thread1 は LockMutex()
                THR_CALL_UNLOCK,        // Thread2 は UnlockMutex()
                true,                   // Thread1 での API 返値
                false,                  // Thread2 での API 返値
                1,                      // Mutex ネストカウンタは 1
                THR_STATE_WAITING2,     // Thread1 は待ち解除して終了
                THR_STATE_EXITED)       // Thread2 は終了
    , "");
}

TEST(CombinedSdkMutexTest, test_LockAndUnlock2)
{
    // Thread1 で Lock()、Thread2 で Unlock()
    EXPECT_DEATH_IF_SUPPORTED(
        doTest2(true,                   // Mutex 再帰許可は true
                true,                   // Mutex 初期ロック状態は true
                THR_CALL_LOCK,          // Thread1 は LockMutex()
                THR_CALL_UNLOCK,        // Thread2 は UnlockMutex()
                true,                   // Thread1 での API 返値
                false,                  // Thread2 での API 返値
                2,                      // Mutex ネストカウンタは 2
                THR_STATE_WAITING2,     // Thread1 は待ち解除して終了
                THR_STATE_EXITED)       // Thread2 は終了
    , "");
}

#if defined(NN_BUILD_CONFIG_OS_WIN32)
//---------------------------------------------------------------------------
// std::thread で使用するテスト用スレッド関数
//
void StdThreadTestFunction()
{
    NNT_OS_LOG(NN_TEXT("sub:  std::thread で生成されたスレッドが起動しました。\n"));

    nn::os::SdkMutex          mutex;
    nn::os::SdkRecursiveMutex recursiveMutex;

    NNT_OS_LOG(NN_TEXT("sub:  std::thread で生成されたスレッド内で mutex を操作"));
    mutex.Lock();
    mutex.Unlock();
    CheckBool( true );

    NNT_OS_LOG(NN_TEXT("sub:  std::thread で生成されたスレッド内で recursiveMutex を操作"));
    recursiveMutex.Lock();
    recursiveMutex.Lock();
    recursiveMutex.Lock();
    recursiveMutex.Lock();
    recursiveMutex.Unlock();
    recursiveMutex.Unlock();
    recursiveMutex.Unlock();
    recursiveMutex.Unlock();
    CheckBool( true );

    NNT_OS_LOG(NN_TEXT("sub:  std::thread で生成されたスレッドが終了します。\n"));
}

//---------------------------------------------------------------------------
// std::thread でスレッドを作成し、std::thread::join() する。
//
TEST(StdThreadTest, test_StdThreadAndJoin)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        NNT_OS_LOG(NN_TEXT("main: std::thread でスレッドを生成します。\n"));

        std::thread stdThread( StdThreadTestFunction );

        NNT_OS_LOG(NN_TEXT("main: std::thread で生成したスレッドに join します。\n"));

        stdThread.join();

        NNT_OS_LOG(NN_TEXT("main: std::thread のテストを終了します。\n"));
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// std::thread でスレッドを作成し、std::thread::detach() する。
//
TEST(StdThreadTest, test_StdThreadAndDetach)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        NNT_OS_LOG(NN_TEXT("main: std::thread でスレッドを生成します。\n"));

        std::thread stdThread( StdThreadTestFunction );

        NNT_OS_LOG(NN_TEXT("main: std::thread で生成したスレッドを detach します。\n"));

        stdThread.detach();

        NNT_OS_LOG(NN_TEXT("main: std::thread のテストを終了します。\n"));
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}
#endif /* defined(NN_BUILD_CONFIG_OS_WIN32) */


//---------------------------------------------------------------------------
// nn::os::SdkMutex クラスの基本動作テスト
//

NN_ALIGNAS(4096) char   g_ThreadStack[8192];
int g_MutexTestCounter;

void test_MutexClassTestThreadNonRecursive(void* arg)
{
    auto& mutex = *reinterpret_cast<nn::os::SdkMutex*>( arg );

    for (int i=0; i<50000; ++i)
    {
        {   // lock_guard を利用
            std::lock_guard<nn::os::SdkMutex> lock( mutex );

            ++g_MutexTestCounter;
            nn::os::YieldThread();
        }
        nn::os::YieldThread();

        {   // unique_lock を利用
            std::unique_lock<nn::os::SdkMutex> lock( mutex );

            ++g_MutexTestCounter;
            nn::os::YieldThread();
        }
        nn::os::YieldThread();
    }
}

TEST(SdkMutexClassTest, test_SdkMutexClassNonRecursive)
{
    CLEAR_GOOGLE_TEST();

    // ミューテックスの構築
    nn::os::SdkMutex mutex;
    g_MutexTestCounter = 0;

    // スレッドの作成
    nn::os::ThreadType  thread;
    nn::os::CreateThread( &thread, test_MutexClassTestThreadNonRecursive, &mutex, g_ThreadStack, sizeof(g_ThreadStack), nn::os::GetThreadPriority(nn::os::GetCurrentThread()));

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("Mutex クラスの NonRecursive テスト\n"));
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("親スレッドと子スレッドで同じグローバル変数をインクリメント\n"));

    // テスト本体（スレッド実行開始）
    int countTryLock = 0;
    nn::os::StartThread( &thread );
    for (int i=0; i<50000; ++i)
    {
        // メンバ変数の Lock() を利用
        {
            mutex.Lock();

            ++g_MutexTestCounter;
            nn::os::YieldThread();

            mutex.Unlock();
        }

        // メンバ変数の TryLock() を利用
        {
            while (!mutex.TryLock())
            {
                // TryLock が成功するまでループするだけ
                ++countTryLock;
                nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
            }

            ++g_MutexTestCounter;
            nn::os::YieldThread();

            mutex.Unlock();
        }
    }

    // スレッドの終了待ち
    nn::os::WaitThread( &thread );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("グローバル変数カウンタが期待値通りか"));
    CheckParam( g_MutexTestCounter, 200000 );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("TryLock の空回り数＝%d\n"), countTryLock);

    // スレッドの破棄
    nn::os::DestroyThread( &thread );

    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// nn::os::SdkRecursiveMutex クラスの基本動作テスト
//

void test_MutexClassTestThreadRecursive(void* arg)
{
    auto& mutex = *reinterpret_cast<nn::os::SdkRecursiveMutex*>( arg );

    for (int i=0; i<50000; ++i)
    {
        mutex.Lock();

        ++g_MutexTestCounter;
        nn::os::YieldThread();

        {
            while (!mutex.TryLock())
            {
                // 成功するまでループ
                nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
            }

            ++g_MutexTestCounter;
            nn::os::YieldThread();

            mutex.Unlock();
        }
        mutex.Unlock();
        nn::os::YieldThread();
    }
}

TEST(SdkMutexClassTest, test_SdkMutexClassRecursive)
{
    CLEAR_GOOGLE_TEST();

    // ミューテックスの構築
    nn::os::SdkRecursiveMutex mutex;
    g_MutexTestCounter = 0;

    // スレッドの作成
    nn::os::ThreadType  thread;
    nn::os::CreateThread( &thread, test_MutexClassTestThreadRecursive, &mutex, g_ThreadStack, sizeof(g_ThreadStack), nn::os::GetThreadPriority(nn::os::GetCurrentThread()));

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("Mutex クラスの Recursive テスト\n"));
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("親スレッドと子スレッドで同じグローバル変数をインクリメント\n"));

    // テスト本体（スレッド実行開始）
    int countTryLock = 0;
    nn::os::StartThread( &thread );
    for (int i=0; i<50000; ++i)
    {
        // メンバ変数の Lock() を利用
        {
            mutex.Lock();

            ++g_MutexTestCounter;
            nn::os::YieldThread();

            // メンバ変数の TryLock() を利用
            {
                while (!mutex.TryLock())
                {
                    // TryLock が成功するまでループするだけ
                    ++countTryLock;
                    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
                }

                ++g_MutexTestCounter;
                nn::os::YieldThread();

                mutex.Unlock();
            }
            mutex.Unlock();
        }
    }

    // スレッドの終了待ち
    nn::os::WaitThread( &thread );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("グローバル変数カウンタが期待値通りか"));
    CheckParam( g_MutexTestCounter, 200000 );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("TryLock の空回り数＝%d"), countTryLock);
    CheckBool( countTryLock == 0 );

    // スレッドの破棄
    nn::os::DestroyThread( &thread );

    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
// nn::os::SdkMutexType 構造体の静的初期化のテスト
//

nn::os::SdkMutexType          g_StaticMutex1 = NN_OS_SDK_MUTEX_INITIALIZER();
nn::os::SdkRecursiveMutexType g_StaticMutex2 = NN_OS_SDK_RECURSIVE_MUTEX_INITIALIZER();

void test_StaticMutexTestThread(void* arg)
{
    int& failedCountOfTryLock = *reinterpret_cast<int*>(arg);

    while (failedCountOfTryLock < 10)
    {
        g_StaticMutex1.Lock();
        nn::os::YieldThread();

        g_StaticMutex1.Unlock();
        nn::os::YieldThread();

        g_StaticMutex2.Lock();
        nn::os::YieldThread();

        g_StaticMutex2.Unlock();
        nn::os::YieldThread();
    }
}

TEST(SdkMutexTypeStaticInitializerTest, test_SdkMutexTypeStaticInitializer)
{
    CLEAR_GOOGLE_TEST();

    // スレッドの作成
    int failedCountOfTryLock = 0;
    int count = 0;
    nn::os::ThreadType  thread;
    nn::os::CreateThread( &thread, test_StaticMutexTestThread, &failedCountOfTryLock, g_ThreadStack, sizeof(g_ThreadStack), nn::os::GetThreadPriority(nn::os::GetCurrentThread()));
    nn::os::StartThread( &thread );

    // 静的初期化された SdkMutexType オブジェクトを利用するテスト
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("静的初期化された g_StaticMutex で保護されたカウンタの操作\n"));
    while (failedCountOfTryLock < 10)
    {
        bool ret1 = g_StaticMutex1.TryLock();
        if (!ret1)
        {
            ++failedCountOfTryLock;
            g_StaticMutex1.Lock();
        }
        nn::os::YieldThread();

        bool ret2 = g_StaticMutex2.TryLock();
        if (!ret2)
        {
            ++failedCountOfTryLock;
            g_StaticMutex2.Lock();
        }
        nn::os::YieldThread();

        ++count;

        g_StaticMutex2.Unlock();
        nn::os::YieldThread();
        g_StaticMutex1.Unlock();
    }

    nn::os::WaitThread( &thread );
    nn::os::DestroyThread( &thread );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("TryLockMutex() に 10 回失敗するまでのループ数： %d\n"), count);

    // 再帰ロックの確認
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("静的初期化された g_StaticMutex2 が再帰可能か"));
    g_StaticMutex2.Lock();
    g_StaticMutex2.Lock();
    g_StaticMutex2.Lock();
    g_StaticMutex2.Unlock();
    g_StaticMutex2.Unlock();
    g_StaticMutex2.Unlock();
    CheckBool( true );

    // エージングテスト用にもう一度初期化状態に戻しておく
    g_StaticMutex1.Initialize();
    g_StaticMutex2.Initialize();

    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// nn::os::SdkMutexType および Mutex クラスの IsMutexLockedByCurrentThread() テスト
//
nn::os::SdkRecursiveMutexType g_SdkRecursiveMutexForIsLockedTest = NN_OS_SDK_RECURSIVE_MUTEX_INITIALIZER();
nn::os::SdkMutexType          g_SdkMutexForIsLockedTest = NN_OS_SDK_MUTEX_INITIALIZER();
nn::os::Barrier               g_BarrierForIsLockedTest(2);

void SubThreadForMutexLockedByCurrentThreadTest(void* arg)
{
    NN_UNUSED(arg);

    // ロックされていないチェック
    CheckBool( !g_SdkRecursiveMutexForIsLockedTest.IsLockedByCurrentThread() );
    CheckBool( !g_SdkMutexForIsLockedTest.IsLockedByCurrentThread() );

    // ２つのミューテックスをロック
    g_SdkRecursiveMutexForIsLockedTest.Lock();
    g_SdkMutexForIsLockedTest.Lock();

    // ロックされているチェック
    CheckBool( g_SdkRecursiveMutexForIsLockedTest.IsLockedByCurrentThread() );
    CheckBool( g_SdkMutexForIsLockedTest.IsLockedByCurrentThread() );

    g_BarrierForIsLockedTest.Await();
    g_BarrierForIsLockedTest.Await();

    // ２つのミューテックスをアンロック
    g_SdkMutexForIsLockedTest.Unlock();
    g_SdkRecursiveMutexForIsLockedTest.Unlock();

    // ロックされていないチェック
    CheckBool( !g_SdkRecursiveMutexForIsLockedTest.IsLockedByCurrentThread() );
    CheckBool( !g_SdkMutexForIsLockedTest.IsLockedByCurrentThread() );
}

TEST(SdkMutexLockedByCurrentThread, test_IsSdkMutexLockedByCurrentThread)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // 子スレッドを生成
    nn::os::ThreadType  thread;
    nn::os::CreateThread( &thread, SubThreadForMutexLockedByCurrentThreadTest, NULL, g_ThreadStack, sizeof(g_ThreadStack), nn::os::GetThreadPriority(nn::os::GetCurrentThread()));
    nn::os::StartThread( &thread );

    g_BarrierForIsLockedTest.Await();

    // ここからが SdkMutex 所有権のテスト
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("SdkMutex の自スレッド所有チェック\n"));

    {
        nn::os::SdkMutex mutex;

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("初期化直後の os::SdkMutex の IsLockedByCurrentThread() は false"));
        CheckBool( !mutex.IsLockedByCurrentThread() );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ロック直後の os::SdkMutex の IsLockedByCurrentThread() は true"));
        mutex.Lock();
        CheckBool( mutex.IsLockedByCurrentThread() );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("アンロック直後の os::SdkMutex の IsLockedByCurrentThread() は false"));
        mutex.Unlock();
        CheckBool( !mutex.IsLockedByCurrentThread() );
    }

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("MutexType の他スレッド所有チェック\n"));

    {
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("他スレッドでロックされた os::SdkMutex の IsMutexLockedByCurrentThread() は false"));
        CheckBool( !g_SdkMutexForIsLockedTest.IsLockedByCurrentThread() );
    }

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("SdkRecursiveMutex の自スレッド所有チェック\n"));

    {
        nn::os::SdkRecursiveMutex mutex;

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("初期化直後の os::SdkRecursiveMutex の IsLockedByCurrentThread() は false"));
        CheckBool( !mutex.IsLockedByCurrentThread() );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ロック直後の os::SdkRecursiveMutex の IsLockedByCurrentThread() は true"));
        mutex.Lock();
        CheckBool( mutex.IsLockedByCurrentThread() );
        mutex.Lock();
        CheckBool( mutex.IsLockedByCurrentThread() );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("アンロック直後の os::SdkRecursiveMutex の IsLockedByCurrentThread() は false"));
        mutex.Unlock();
        CheckBool( mutex.IsLockedByCurrentThread() );
        mutex.Unlock();
        CheckBool( !mutex.IsLockedByCurrentThread() );
    }

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("SdkRecursiveMutex の他スレッド所有チェック\n"));

    {
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("他スレッドでロックされた os::SdkRecursiveMutex の IsLockedByCurrentThread() は false"));
        CheckBool( !g_SdkRecursiveMutexForIsLockedTest.IsLockedByCurrentThread() );
    }

    g_BarrierForIsLockedTest.Await();

    // 子スレッドを削除
    nn::os::WaitThread( &thread );
    nn::os::DestroyThread( &thread );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

// ロック無し Mutex の速度を計測
#define NNT_OS_MLAUL(m)     do { \
    (m).Lock(); \
    (m).Unlock(); \
    } while( NN_STATIC_CONDITION(0) )
TEST(SdkMutexWithoutLockSpeedTest, testSdkMutexWithoutLockSpeed)
{
    const int loopCount = 10;
    const int unitSize = 50000;
    const int showSize = 1000;

    int64_t sum = 0;
    int64_t max = 0;
    int64_t min = INT64_MAX;
    int64_t leg;
    int64_t start;
    nn::os::SdkMutex m;

    // 時間計測
    for (int l = 0; l < loopCount; ++l)
    {
        start = nn::os::GetSystemTick().GetInt64Value();
        for (int u = 0; u < unitSize; ++u)
        {
            NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m);
            NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m);
            NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m);
            NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m);
        }
        leg = (nn::os::GetSystemTick().GetInt64Value() - start);
        sum += leg;

        // 最悪値と最善値を記録
        if (leg > max)
        {
            max = leg;
        }
        if (leg < min)
        {
            min = leg;
        }
    }

    // 値を補正し、showSize あたりの回数にする
    sum -= max + min;
    sum = sum * showSize / (unitSize * 20 * (loopCount - 2));

    // 結果の表示
    nn::os::Tick tick( sum );
    NNT_OS_LOG("Mutex without Lock : time=%lldus (tick=%lld) per %d calls.\n",
        tick.ToTimeSpan().GetMicroSeconds(), tick.GetInt64Value(), showSize);
}

// アンロック状態における IsMutexLockedByCurrentThread の速度を計測
#define NNT_OS_IMLBCT(m)    ((m).IsLockedByCurrentThread())
TEST(IsSdkMutexLockedByCurrentThreadWithUnlockSpeedTest, testIsSdkMutexLockedByCurrentThreadWithUnlockSpeed)
{
    const int loopCount = 10000;

    double sum = 0;
    double max = 0;
    double min = 10000;
    double leg;
    int64_t start;
    nn::os::SdkMutex m;

    // 時間計測
    for (int l = 0; l < loopCount; l++)
    {
        start = nn::os::GetSystemTick().GetInt64Value();

        NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m);
        NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m);
        NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m);
        NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m);

        leg = (nn::os::GetSystemTick().GetInt64Value() - start) / 20.0;
        sum += leg;

        // 最悪値と最善値を記録
        if (leg > max)
        {
            max = leg;
        }
        if (leg < min)
        {
            min = leg;
        }
    }

    // 値を補正し、1000回あたりの回数にする
    sum -= max + min;
    sum = sum * 1000 / (loopCount - 2);

    // 結果の表示
    nn::os::Tick tick(static_cast<int64_t>(sum));
    NNT_OS_LOG("IsMutexLockedByCurrentThread with Unlock : time=%lldus (tick=%lld) per %d calls.\n",
        tick.ToTimeSpan().GetMicroSeconds(), tick.GetInt64Value(), 1000);
}

}}} // namespace nnt::os::sdkmutex

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    int result;

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of Mutex APIs\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of Mutex APIs\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}
