﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"

#include <mutex>
#include <shared_mutex>

#include <nn/os/os_Config.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

#include <nnt/nntest.h>

#include "../Common/test_Thread.h"

namespace nnt { namespace os { namespace readWriteLock {

//-----------------------------------------------------------------------------
//  Readers-Writer ロック機能のテスト
//
//  このソースでは、nn::os::ReaderWriterLock クラスの機能テストを行なう。
//  基本的には test_ReaderWriterLock.cpp と同じアルゴリズムのままである。
//-----------------------------------------------------------------------------

namespace {

    int g_Sequence;

    inline void CheckReaderWriterLockHeld(const nn::os::ReaderWriterLock& rwlock, bool readLocked, bool writeLocked, bool isOwner)
    {
        EXPECT_EQ( readLocked,  rwlock.IsReadLockHeld() );
        EXPECT_EQ( writeLocked, rwlock.IsWriteLockHeldByCurrentThread() );
        EXPECT_EQ( isOwner,     rwlock.IsLockOwner() );
    }

    nnt::os::Stack g_ThreadStack1;
    nnt::os::Stack g_ThreadStack2;

}   // namespace


//-----------------------------------------------------------------------------
// 基本的なコア API の呼出し
//-----------------------------------------------------------------------------

TEST(ReaderWriterLockClassTest, ReadLock)
{
    // 初期化
    nn::os::ReaderWriterLock   rwlock;
    CheckReaderWriterLockHeld( rwlock, false, false, false );

    // Read ロックの基本テスト
    rwlock.AcquireReadLock();
    CheckReaderWriterLockHeld( rwlock, true, false, false );
    {
        // 再帰ロック
        rwlock.AcquireReadLock();
        CheckReaderWriterLockHeld( rwlock, true, false, false );

        // Read ロックのポーリング
        auto ret = rwlock.TryAcquireReadLock();
        EXPECT_TRUE( ret == true );

        CheckReaderWriterLockHeld( rwlock, true, false, false );
        rwlock.ReleaseReadLock();

        // Write ロックのポーリング
        ret = rwlock.TryAcquireWriteLock();
        EXPECT_TRUE( ret == false );
        CheckReaderWriterLockHeld( rwlock, true, false, false );

        // 再帰ロック解除
        rwlock.ReleaseReadLock();
        CheckReaderWriterLockHeld( rwlock, true, false, false );
    }
    rwlock.ReleaseReadLock();
    CheckReaderWriterLockHeld( rwlock, false, false, false );

    // Write ロックの基本テスト
    rwlock.AcquireWriteLock();
    CheckReaderWriterLockHeld( rwlock, false, true, true );
    {
        // 再帰ロック
        rwlock.AcquireWriteLock();
        CheckReaderWriterLockHeld( rwlock, false, true, true );

        // Write ロックのポーリング
        auto ret = rwlock.TryAcquireWriteLock();
        EXPECT_TRUE( ret == true );

        CheckReaderWriterLockHeld( rwlock, false, true, true );
        rwlock.ReleaseWriteLock();

        // Write ロック中の Read ロック
        rwlock.AcquireReadLock();
        CheckReaderWriterLockHeld( rwlock, true, true, true );

        rwlock.AcquireReadLock();
        CheckReaderWriterLockHeld( rwlock, true, true, true );

        rwlock.ReleaseReadLock();
        rwlock.ReleaseReadLock();
        CheckReaderWriterLockHeld( rwlock, false, true, true );

        // Write ロック中の Read ロックのポーリング
        ret = rwlock.TryAcquireReadLock();
        EXPECT_TRUE( ret == true );
        CheckReaderWriterLockHeld( rwlock, true, true, true );

        ret = rwlock.TryAcquireReadLock();
        EXPECT_TRUE( ret == true );
        CheckReaderWriterLockHeld( rwlock, true, true, true );

        rwlock.ReleaseReadLock();
        rwlock.ReleaseReadLock();
        CheckReaderWriterLockHeld( rwlock, false, true, true );

        // 再帰ロック解除
        rwlock.ReleaseWriteLock();
        CheckReaderWriterLockHeld( rwlock, false, true, true );
    }
    rwlock.ReleaseWriteLock();
    CheckReaderWriterLockHeld( rwlock, false, false, false );

    // Write/Read ロックの混在テスト
    // WriteLock -> ReadLock -> WriteUnlock -> ReadUnlock

    // Write ロック
    rwlock.AcquireWriteLock();
    CheckReaderWriterLockHeld( rwlock, false, true, true );
    {
        // Read ロック
        rwlock.AcquireReadLock();
        CheckReaderWriterLockHeld( rwlock, true, true, true );
        rwlock.ReleaseReadLock();
        CheckReaderWriterLockHeld( rwlock, false, true, true );

        // Read ロックポーリング
        auto ret = rwlock.TryAcquireReadLock();
        EXPECT_TRUE( ret == true );
        CheckReaderWriterLockHeld( rwlock, true, true, true );

        // Write ロック解除
        rwlock.ReleaseWriteLock();
        CheckReaderWriterLockHeld( rwlock, true, false, true );
    }
    // Read ロック解除
    rwlock.ReleaseReadLock();
    CheckReaderWriterLockHeld( rwlock, false, false, false );
}


//-----------------------------------------------------------------------------
// ReadLock 中の ReadLock 取得／解除と WriteLock 待機のテスト
//
//  1) メインスレッド側で ReadLock 取得
//  2) サブスレッド側で WriteLock 取得待ち
//  3) メインスレッド側で ReadLock 解除
//  4) サブスレッド側で WriteLock 取得
//  5) サブスレッド側で WriteLock 解除
//-----------------------------------------------------------------------------

void ReaderWriterLockClassTest_SubThread(void* arg)
{
    // ここに来た時点で Read ロック済み
    auto& rwlock = *reinterpret_cast<nn::os::ReaderWriterLock*>(arg);

    {
        // Read ロックのポーリング
        auto ret = rwlock.TryAcquireReadLock();
        EXPECT_TRUE( ret == true );

        // Read ロック
        rwlock.AcquireReadLock();

        // Read ロック解除×２
        rwlock.ReleaseReadLock();
        rwlock.ReleaseReadLock();
    }

    // Write ロックのポーリング
    auto ret = rwlock.TryAcquireWriteLock();
    EXPECT_TRUE( ret == false );

    // Write ロック取得待ち → 一旦メインスレッドへ戻る
    EXPECT_TRUE( ++g_Sequence == 1 );
    rwlock.AcquireWriteLock();

    // Write ロック取得
    EXPECT_TRUE( ++g_Sequence == 3 );

    // Write ロック解除
    rwlock.ReleaseWriteLock();
}

TEST(ReaderWriterLockClassTest, WriteLockDuringReadLocked)
{
    // 初期化
    nn::os::ReaderWriterLock   rwlock;

    nnt::os::Thread thread(ReaderWriterLockClassTest_SubThread, &rwlock, g_ThreadStack1, 0);

    // 初期化後の状態チェック
    CheckReaderWriterLockHeld( rwlock, false, false, false );

    // Read ロック
    g_Sequence = 0;
    rwlock.AcquireReadLock();
    thread.Start();
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(30) );

    EXPECT_TRUE( ++g_Sequence == 2 );

    rwlock.ReleaseReadLock();
    thread.Wait();

    EXPECT_TRUE( ++g_Sequence == 4 );

    // 破棄直前の状態チェック
    CheckReaderWriterLockHeld( rwlock, false, false, false );
}


//-----------------------------------------------------------------------------
// WriteLock 中の ReadLock 待機と WriteLock 待機のテスト
//
//  1) メインスレッド側で WriteLock 取得
//  2) サブスレッド１側で ReadLock 取得待ち
//  3) サブスレッド２側で WriteLock 取得待ち
//  4) メインスレッド側で WriteLock 解除
//
//  5) サブスレッド２側で WriteLock 取得 ← WriteLock が先にロックを取れる
//  6) サブスレッド２側で WriteLock 解除
//
//  7) サブスレッド１側で ReadLock 取得  ← ReadLock は WriteLock 待ちが
//                                          いなくなったら取れる
//  8) サブスレッド１側で ReadLock 解除
//-----------------------------------------------------------------------------

void ReaderWriterLockClassTest_SubThread1(void* arg)
{
    // ここに来た時点で Read ロック済み
    auto& rwlock = *reinterpret_cast<nn::os::ReaderWriterLock*>(arg);

    // Read ロックのポーリング
    auto ret = rwlock.TryAcquireReadLock();
    EXPECT_TRUE( ret == false );

    // Read ロック取得待ち → 一旦メインスレッドへ戻る
    EXPECT_TRUE( ++g_Sequence == 1 );
    rwlock.AcquireReadLock();

    // Read ロック取得
    // SIGLO-66053 の優先度継承により、シーケンス順が変わるのでコメントアウト
    ++g_Sequence;
    //EXPECT_TRUE( g_Sequence == 5 );

    // Read ロック解除
    rwlock.ReleaseReadLock();
}

void ReaderWriterLockClassTest_SubThread2(void* arg)
{
    // ここに来た時点で Read ロック済み
    auto& rwlock = *reinterpret_cast<nn::os::ReaderWriterLock*>(arg);

    // Write ロックのポーリング
    auto ret = rwlock.TryAcquireWriteLock();
    EXPECT_TRUE( ret == false );

    // Write ロック取得待ち → 一旦メインスレッドへ戻る
    EXPECT_TRUE( ++g_Sequence == 2 );
    rwlock.AcquireWriteLock();

    // Write ロック取得
    // SIGLO-66053 の優先度継承により、シーケンス順が変わるのでコメントアウト
    ++g_Sequence;
    //EXPECT_TRUE( g_Sequence == 4 );

    // Write ロック解除
    rwlock.ReleaseWriteLock();
}

TEST(ReaderWriterLockClassTest, ReadLockAndWriteLockDuringWriteLocked)
{
    // 初期化
    nn::os::ReaderWriterLock   rwlock;

    nnt::os::Thread thread1(ReaderWriterLockClassTest_SubThread1, &rwlock, g_ThreadStack1, 0);
    nnt::os::Thread thread2(ReaderWriterLockClassTest_SubThread2, &rwlock, g_ThreadStack2, 0);

    // 初期化後の状態チェック
    CheckReaderWriterLockHeld( rwlock, false, false, false );

    // Write ロック
    g_Sequence = 0;
    rwlock.AcquireWriteLock();
    CheckReaderWriterLockHeld( rwlock, false, true, true );

    thread1.Start();
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(30) );

    thread2.Start();
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(30) );

    // Write ロック解除
    EXPECT_TRUE( ++g_Sequence == 3 );
    rwlock.ReleaseWriteLock();
    thread1.Wait();
    thread2.Wait();

    EXPECT_TRUE( ++g_Sequence == 6 );

    // 破棄直前の状態チェック
    CheckReaderWriterLockHeld( rwlock, false, false, false );
}

//-----------------------------------------------------------------------------
// WriteLock 中の ReadLock 待機と WriteLock 待機のテスト
// ロックオーナーを維持した ReadLock 状態からのロック解除
//
//  1) メインスレッド側で WriteLock 取得
//     メインスレッド側で ReadLock 取得
//     メインスレッド側で WriteLock 解除
//
//  2) サブスレッド１側で ReadLock 取得待ち
//  3) サブスレッド２側で WriteLock 取得待ち
//
//  4) メインスレッド側で ReadLock 解除
//
//  5) サブスレッド２側で WriteLock 取得 ← WriteLock が先にロックを取れる
//  6) サブスレッド２側で WriteLock 解除
//
//  7) サブスレッド１側で ReadLock 取得  ← ReadLock は WriteLock 待ちが
//                                          いなくなったら取れる
//  8) サブスレッド１側で ReadLock 解除
//-----------------------------------------------------------------------------

void ReaderWriterLockClassTest2_SubThread1(void* arg)
{
    // ここに来た時点で Read ロック済み
    auto& rwlock = *reinterpret_cast<nn::os::ReaderWriterLock*>(arg);

    // Read ロックのポーリング
    auto ret = rwlock.TryAcquireReadLock();
    EXPECT_TRUE( ret == false );

    // Read ロック取得待ち → 一旦メインスレッドへ戻る
    EXPECT_TRUE( ++g_Sequence == 1 );
    rwlock.AcquireReadLock();

    // Read ロック取得
    // SIGLO-66053 の優先度継承により、シーケンス順が変わるのでコメントアウト
    ++g_Sequence;
    //EXPECT_TRUE( g_Sequence == 5 );

    // Read ロック解除
    rwlock.ReleaseReadLock();
}

void ReaderWriterLockClassTest2_SubThread2(void* arg)
{
    // ここに来た時点で Read ロック済み
    auto& rwlock = *reinterpret_cast<nn::os::ReaderWriterLock*>(arg);

    // Write ロックのポーリング
    auto ret = rwlock.TryAcquireWriteLock();
    EXPECT_TRUE( ret == false );

    // Write ロック取得待ち → 一旦メインスレッドへ戻る
    EXPECT_TRUE( ++g_Sequence == 2 );
    rwlock.AcquireWriteLock();

    // Write ロック取得
    // SIGLO-66053 の優先度継承により、シーケンス順が変わるのでコメントアウト
    ++g_Sequence;
    //EXPECT_TRUE( g_Sequence == 4 );

    // Write ロック解除
    rwlock.ReleaseWriteLock();
}

TEST(ReaderWriterLockClassTest, ReadLockAndWriteLockDuringOwnedReadLocked)
{
    // 初期化
    nn::os::ReaderWriterLock   rwlock;

    nnt::os::Thread thread1(ReaderWriterLockClassTest2_SubThread1, &rwlock, g_ThreadStack1, 0);
    nnt::os::Thread thread2(ReaderWriterLockClassTest2_SubThread2, &rwlock, g_ThreadStack2, 0);

    // 初期化後の状態チェック
    CheckReaderWriterLockHeld( rwlock, false, false, false );

    // Write ロック -> Read ロック -> Write ロック解除
    g_Sequence = 0;
    rwlock.AcquireWriteLock();
    rwlock.AcquireReadLock();
    rwlock.ReleaseWriteLock();
    CheckReaderWriterLockHeld( rwlock, true, false, true );

    thread1.Start();
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(30) );

    thread2.Start();
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(30) );

    // Write ロック解除
    EXPECT_TRUE( ++g_Sequence == 3 );
    rwlock.ReleaseReadLock();
    thread1.Wait();
    thread2.Wait();

    EXPECT_TRUE( ++g_Sequence == 6 );

    // 破棄直前の状態チェック
    CheckReaderWriterLockHeld( rwlock, false, false, false );
}

//-----------------------------------------------------------------------------
// SIGLO-81435:
//  os::ReaderWriterLock クラスを std::lock_guard<>, std::unique_lock<>,
//  std::shared_lock<> のテンプレート引数として利用可能であることを確認する。
//
//-----------------------------------------------------------------------------

TEST(ReaderWriterLockClassTest, StdSharedMutexTemplate)
{
    nn::os::ReaderWriterLock rwlock;

    // WriterLock の利用
    {
        std::lock_guard<decltype(rwlock)> wlock(rwlock);
        CheckReaderWriterLockHeld(rwlock, false, true, true);
    }
    CheckReaderWriterLockHeld(rwlock, false, false, false);

    // ReaderLock の利用
    {
        std::shared_lock<decltype(rwlock)> rlock(rwlock);
        CheckReaderWriterLockHeld(rwlock, true, false, false);
    }
    CheckReaderWriterLockHeld(rwlock, false, false, false);

    // WriterLock 中の ReaderLock のネスト
    {
        std::unique_lock<decltype(rwlock)> wlock(rwlock);
        CheckReaderWriterLockHeld(rwlock, false, true, true);
        {
            std::shared_lock<decltype(rwlock)> rlock(rwlock);
            CheckReaderWriterLockHeld(rwlock, true, true, true);

            rlock.unlock();
            CheckReaderWriterLockHeld(rwlock, false, true, true);

            ASSERT_TRUE(rlock.try_lock());
            CheckReaderWriterLockHeld(rwlock, true, true, true);

            wlock.unlock();
            CheckReaderWriterLockHeld(rwlock, true, false, true);

            ASSERT_TRUE(wlock.try_lock());
            CheckReaderWriterLockHeld(rwlock, true, true, true);
        }
        CheckReaderWriterLockHeld(rwlock, false, true, true);
    }
    CheckReaderWriterLockHeld(rwlock, false, false, false);

#if 0
    // ReaderLock 中の WriterLock のネスト
    // ただし、現状の SIGLO では ReaderLock から WriterLock への
    // アップグレード（エスカレーション）には未対応なので無効化しておく。
    {
        std::shared_lock<decltype(rwlock)> rlock(rwlock);
        CheckReaderWriterLockHeld(rwlock, true, false, false);
        {
            std::unique_lock<decltype(rwlock)> wlock(rwlock);
            CheckReaderWriterLockHeld(rwlock, true, true, true);

            rlock.unlock();
            CheckReaderWriterLockHeld(rwlock, false, true, true);

            ASSERT_TRUE(rlock.try_lock());
            CheckReaderWriterLockHeld(rwlock, true, true, true);

            wlock.unlock();
            CheckReaderWriterLockHeld(rwlock, true, false, true);

            ASSERT_TRUE(wlock.try_lock());
            CheckReaderWriterLockHeld(rwlock, true, true, true);
        }
        CheckReaderWriterLockHeld(rwlock, true, false, true);
    }
    CheckReaderWriterLockHeld(rwlock, false, false, false);
#endif
}

}}} // namespace nnt::os::readWriteLock

