﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/os/os_MemoryAttribute.h>
#include "../Common/test_Helper.h"

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/svc/svc_Base.h>
#endif

namespace nnt { namespace os { namespace memoryAttribute {

// 32bit の広い範囲のアドレスに対し繰り返し Read / Write （インクリメント）する
nn::TimeSpan ReadWriteTestRange(uintptr_t address, size_t size, int countMax)
{
    auto startTime = nn::os::GetSystemTick();
    volatile uint32_t* ptr = reinterpret_cast<uint32_t*>(address);
    for (int loopCount=0; loopCount<countMax; ++loopCount)
    {
        size_t length = size / sizeof(uint32_t);
        for (size_t i=0; i<length; ++i)
        {
            ++ptr[i];  // read / write
        }
    }
    auto duration = nn::os::GetSystemTick() - startTime;
    return nn::os::ConvertToTimeSpan(duration);
}

// 32bit の単一アドレスに対し繰り返し Read / Write （インクリメント）する
nn::TimeSpan ReadWriteTestSingleAddress(uintptr_t address, int countMax)
{
    auto startTime = nn::os::GetSystemTick();
    volatile uint32_t* ptr = reinterpret_cast<uint32_t*>(address);
    for (int loopCount=0; loopCount<countMax; ++loopCount)
    {
        ++ptr[0];  // read / write
    }
    auto duration = nn::os::GetSystemTick() - startTime;
    return nn::os::ConvertToTimeSpan(duration);
}

// 指定された領域が全て Uncached 属性かチェックする
void CheckMemoryAttribute(uintptr_t beginAddress, int regionSize)
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    auto address = beginAddress;
    while (address < beginAddress + regionSize)
    {
        // メモリ情報を取得
        nn::svc::MemoryInfo  memoryInfo;
        nn::svc::PageInfo    pageInfo;
        auto result = nn::svc::QueryMemory(&memoryInfo, &pageInfo, address);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        EXPECT_TRUE(memoryInfo.attribute & nn::svc::MemoryAttribute_Uncached);

        address += (memoryInfo.baseAddress + memoryInfo.size) - address;
    }
#else
    NN_UNUSED(beginAddress);
    NN_UNUSED(regionSize);
#endif
}

TEST(MemoryAttribute, ChangeMemoryAttribute_LargeMemoryRangeTest)
{
    // AllInOne テストではこのファイルの nnMain() は実行されないので、
    // 個別テストの中で、テストに必要な量のヒープサイズを指定しておく。
    auto result = nn::os::SetMemoryHeapSize( 16 * 1024 * 1024 );
    ASSERT_TRUE(result.IsSuccess());

    // メモリ確保
    uintptr_t   address;
    result = nn::os::AllocateMemoryBlock( &address, nn::os::MemoryBlockUnitSize );
    ASSERT_TRUE(result.IsSuccess());

    // 確保したメモリの属性を変更（まずは uncached に）
    nn::os::SetMemoryAttribute(address, nn::os::MemoryBlockUnitSize, nn::os::MemoryAttribute_Uncached);
    auto uncachedTime = ReadWriteTestRange(address, nn::os::MemoryPageSize * 10, 100);
    NNT_OS_LOG("Read/Write time (Uncached) = %d us\n", uncachedTime.GetMicroSeconds());

    CheckMemoryAttribute(address, nn::os::MemoryBlockUnitSize);

    // 確保したメモリの属性を変更（つぎは normal に）
    nn::os::SetMemoryAttribute(address, nn::os::MemoryBlockUnitSize, nn::os::MemoryAttribute_Normal);
    auto normalTime = ReadWriteTestRange(address, nn::os::MemoryPageSize * 10, 100);
    NNT_OS_LOG("Read/Write time (Normal) = %d us\n", normalTime.GetMicroSeconds());

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    // 実機環境でのみ速度を比較
    // 実際に確認して大きな差があれば有効にする
    ASSERT_GT(uncachedTime.GetNanoSeconds(), normalTime.GetNanoSeconds());
#endif

    // メモリを返却
    nn::os::FreeMemoryBlock(address, nn::os::MemoryBlockUnitSize);

    // ヒープサイズを戻しておく
    nn::os::SetMemoryHeapSize( 0 );
}

TEST(MemoryAttribute, ChangeMemoryAttribute_OneAddressTest)
{
    // AllInOne テストではこのファイルの nnMain() は実行されないので、
    // 個別テストの中で、テストに必要な量のヒープサイズを指定しておく。
    auto result = nn::os::SetMemoryHeapSize( 16 * 1024 * 1024 );
    ASSERT_TRUE(result.IsSuccess());

    // メモリ確保
    uintptr_t   address;
    result = nn::os::AllocateMemoryBlock( &address, nn::os::MemoryBlockUnitSize );
    ASSERT_TRUE(result.IsSuccess());

    // 確保したメモリの属性を変更（まずは uncached に）
    nn::os::SetMemoryAttribute(address, nn::os::MemoryBlockUnitSize, nn::os::MemoryAttribute_Uncached);
    auto uncachedTime = ReadWriteTestSingleAddress(address, nn::os::MemoryPageSize * 10 * 100 / sizeof(uint32_t));
    NNT_OS_LOG("Read/Write time (Uncached) = %d us\n", uncachedTime.GetMicroSeconds());

    CheckMemoryAttribute(address, nn::os::MemoryBlockUnitSize);

    // 確保したメモリの属性を変更（つぎは normal に）
    nn::os::SetMemoryAttribute(address, nn::os::MemoryBlockUnitSize, nn::os::MemoryAttribute_Normal);
    auto normalTime = ReadWriteTestSingleAddress(address, nn::os::MemoryPageSize * 10 * 100 / sizeof(uint32_t));
    NNT_OS_LOG("Read/Write time (Normal) = %d us\n", normalTime.GetMicroSeconds());

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    // 実機環境でのみ速度を比較
    // 実際に確認して大きな差があれば有効にする
    ASSERT_GT(uncachedTime.GetNanoSeconds(), normalTime.GetNanoSeconds());
#endif

    // メモリを返却
    nn::os::FreeMemoryBlock(address, nn::os::MemoryBlockUnitSize);

    // ヒープサイズを戻しておく
    nn::os::SetMemoryHeapSize( 0 );
}

TEST(MemoryAttribute, ChangeMemoryAttribute_MultipleMemoryBlocks)
{
    // AllInOne テストではこのファイルの nnMain() は実行されないので、
    // 個別テストの中で、テストに必要な量のヒープサイズを指定しておく。
    auto result = nn::os::SetMemoryHeapSize( 16 * 1024 * 1024 );
    ASSERT_TRUE(result.IsSuccess());

    // メモリ確保
    uintptr_t   address;
    result = nn::os::AllocateMemoryBlock( &address, nn::os::MemoryBlockUnitSize );
    ASSERT_TRUE(result.IsSuccess());

    // 属性を変更してメモリ属性を一様でなくす
    nn::os::SetMemoryAttribute(address + nn::os::MemoryPageSize, nn::os::MemoryPageSize, nn::os::MemoryAttribute_Uncached);

    // 確保したメモリの属性を変更（まずは uncached に）
    nn::os::SetMemoryAttribute(address, nn::os::MemoryBlockUnitSize, nn::os::MemoryAttribute_Uncached);
    auto uncachedTime = ReadWriteTestRange(address, nn::os::MemoryPageSize * 10, 100);
    NNT_OS_LOG("Read/Write time (Uncached) = %d us\n", uncachedTime.GetMicroSeconds());

    CheckMemoryAttribute(address, nn::os::MemoryBlockUnitSize);

    // 確保したメモリの属性を変更（つぎは normal に）
    nn::os::SetMemoryAttribute(address, nn::os::MemoryBlockUnitSize, nn::os::MemoryAttribute_Normal);
    auto normalTime = ReadWriteTestRange(address, nn::os::MemoryPageSize * 10, 100);
    NNT_OS_LOG("Read/Write time (Normal) = %d us\n", normalTime.GetMicroSeconds());

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    // 実機環境でのみ速度を比較
    // 実際に確認して大きな差があれば有効にする
    ASSERT_GT(uncachedTime.GetNanoSeconds(), normalTime.GetNanoSeconds());
#endif

    // メモリを返却
    nn::os::FreeMemoryBlock(address, nn::os::MemoryBlockUnitSize);

    // ヒープサイズを戻しておく
    nn::os::SetMemoryHeapSize( 0 );
}

}}} // namespace nnt::os::memoryAttribute

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of MemoryHeap APIs\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    int result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of MemoryHeap APIs\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}
