﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Calibration 機能
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"
#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

//---------------------------------------------------------------------------

NN_ALIGNAS(4096)    char g_ThreadStack1[4096];
NN_ALIGNAS(4096)    char g_ThreadStack2[4096];

//---------------------------------------------------------------------------

Calibration  g_Calibration;

//---------------------------------------------------------------------------

namespace {

int64_t g_startTime;
int64_t g_endTime;
int64_t g_totalTime;

nn::os::BarrierType     g_barrier;
nn::os::SemaphoreType   g_semaphore1;
nn::os::SemaphoreType   g_semaphore2;

}

void    threadFunction1(void* arg)
{
    NN_UNUSED( arg );

    for (int i = 0; i < nntMeasureCount; ++i)
    {
        nn::os::AwaitBarrier( &g_barrier );

        g_startTime = nn::os::GetSystemTick().GetInt64Value();

        for (int j = 0; j < 1000; j++)
        {
            nn::os::AcquireSemaphore( &g_semaphore1 );
            nn::os::ReleaseSemaphore( &g_semaphore2 );
        }

        g_endTime = nn::os::GetSystemTick().GetInt64Value();

        g_totalTime += (g_endTime - g_startTime) / 2;
    }
}

void    threadFunction2(void* arg)
{
    NN_UNUSED( arg );

    for (int i = 0; i < nntMeasureCount; ++i)
    {
        nn::os::AwaitBarrier( &g_barrier );

        for (int j = 0; j < 1000; j++)
        {
            nn::os::ReleaseSemaphore( &g_semaphore1 );
            nn::os::AcquireSemaphore( &g_semaphore2 );
        }
    }
}


void    Calibration::Initialize()
NN_NOEXCEPT
{
    nn::Result          result;
    nn::os::ThreadType  thread1;
    nn::os::ThreadType  thread2;

    g_totalTime = 0;

    // スレッド１の生成
    result = nn::os::CreateThread( &thread1, threadFunction1, 0,
                    g_ThreadStack1, 4096, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), NN_TEXT("スレッド１の生成に失敗"));

    // スレッド２の生成
    result = nn::os::CreateThread( &thread2, threadFunction2, 0,
                    g_ThreadStack2, 4096, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), NN_TEXT("スレッド２の生成に失敗"));

    // バリアの初期化
    nn::os::InitializeBarrier( &g_barrier, 2 );

    // セマフォの初期化
    nn::os::InitializeSemaphore( &g_semaphore1, 0, 1 );
    nn::os::InitializeSemaphore( &g_semaphore2, 0, 1 );

    // キャリブレーション開始
    NNT_OS_LOG("Start calibration...\n");

    // スレッドの開始
    nn::os::StartThread( &thread1 );
    nn::os::StartThread( &thread2 );

    // スレッドの終了待ち
    nn::os::WaitThread( &thread1 );
    nn::os::WaitThread( &thread2 );

    // スレッドの削除
    nn::os::DestroyThread( &thread1 );
    nn::os::DestroyThread( &thread2 );

    // バリアとセマフォのファイナライズ
    nn::os::FinalizeSemaphore( &g_semaphore1 );
    nn::os::FinalizeSemaphore( &g_semaphore2 );
    nn::os::FinalizeBarrier( &g_barrier );

    // キャリブレーション終了
    NNT_OS_LOG("End   calibration... ");

    // キャリブレーション結果を格納
    auto calibratedTime = nn::os::ConvertToTimeSpan( nn::os::Tick(g_totalTime / nntMeasureCount) ).GetNanoSeconds();
    NN_LOG("calibrated timespan = %lld(nsec)\n", calibratedTime );

    // キャリブレーション結果が短すぎる場合は下限値に補正
    if (calibratedTime < GetMinimumCalibratedTime().GetNanoSeconds())
    {
        calibratedTime = GetMinimumCalibratedTime().GetNanoSeconds();
    }

#if defined(NN_BULD_CONFIG_OS_HORIZON)
    #if defined(NN_BUILD_CONFIG_HARDWARE_SMMA53)
        calibratedTime = calibratedTime / 10;
    #endif
#endif  // defined(NN_BULD_CONFIG_OS_HORIZON)

    m_calibratedBase = (calibratedTime + 999999) / 1000000 * 1000000;
}

