﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabaseManager.h>
#include <nn/olsc/srv/olsc_SystemEventManager.h>
#include <nn/olsc/srv/olsc_TransferTask.h>
#include <nn/olsc/srv/olsc_TransferTaskAgent.h>
#include <nn/olsc/srv/olsc_TransferTaskFactory.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_File.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time.h>

#include <vector>

namespace nnt { namespace olsc {

    class TransferTaskForTest : public nn::olsc::srv::TransferTaskBase
    {
        NN_DISALLOW_COPY(TransferTaskForTest);

    public:
        TransferTaskForTest(nn::olsc::srv::database::TransferTaskDatabaseManager& ttdm,
            const nn::olsc::srv::TransferTaskDetailInfo& detailInfo,
            std::vector<nn::olsc::TransferTaskId>& doneTaskList,
            int taskProcessTime,
            nn::Result doneResult,
            TransferTaskExecutionResource& executionResource) NN_NOEXCEPT
            : nn::olsc::srv::TransferTaskBase(executionResource), m_Ttdm(ttdm), m_DetailInfo(detailInfo),
                m_DoneTaskList(doneTaskList), m_TaskProcessTime(taskProcessTime), m_DoneResult(doneResult), m_Context({})
        {
            m_Context.id = detailInfo.id;
            m_Context.totalSize = 1;
            m_Context.currentSize = 0;
            UpdateContext([&](TransferTaskContext& toUpdate) {
                toUpdate = m_Context;
            });
        }
        virtual ~TransferTaskForTest() NN_NOEXCEPT
        {}

        virtual void Execute() NN_NOEXCEPT NN_OVERRIDE
        {
            nn::os::TimerEvent t(nn::os::EventClearMode_ManualClear);
            t.StartOneShot(nn::TimeSpan::FromSeconds(m_TaskProcessTime));

            nn::util::optional<nn::olsc::srv::SaveDataArchiveInfo> sda;
            sda.emplace();
            while (!t.TryWait())
            {
                if (this->IsCanceled())
                {
                    SetLatestSaveDataArchive(sda);
                    CompleteTask(nn::olsc::ResultCanceled());
                    return;
                }
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            }

            m_Context.currentSize = 1;
            UpdateContext([&](TransferTaskContext& toUpdate) {
                toUpdate = m_Context;
            });

            m_DoneTaskList.push_back(m_DetailInfo.id);
            SetLatestSaveDataArchive(sda);
            CompleteTask(m_DoneResult);

            return;
        }

    private:
        nn::olsc::srv::database::TransferTaskDatabaseManager& m_Ttdm;
        const nn::olsc::srv::TransferTaskDetailInfo& m_DetailInfo;
        std::vector<nn::olsc::TransferTaskId>& m_DoneTaskList;
        int m_TaskProcessTime;
        nn::Result m_DoneResult;
        TransferTaskContext m_Context;

        virtual bool IsRetryableResult(nn::Result result) const NN_NOEXCEPT NN_OVERRIDE
        {
            if (result <= nn::olsc::ResultCanceled())
            {
                return true;
            }
            return false;
        }
    };

    class TransferTaskFactoryForTest : public nn::olsc::srv::TransferTaskFactoryBase
    {
        NN_DISALLOW_COPY(TransferTaskFactoryForTest);

    public:
        using TaskResultArbiter = std::function<nn::Result(const nn::olsc::srv::TransferTaskDetailInfo& detailInfo)>;

        TransferTaskFactoryForTest(nn::olsc::srv::database::TransferTaskDatabaseManager& ttdm, std::vector<nn::olsc::TransferTaskId>& doneTaskList, int taskProcessTime, TaskResultArbiter taskResultArbiter) NN_NOEXCEPT
            : m_Created(false), m_Ttdm(ttdm), m_DoneTaskList(doneTaskList), m_TaskProcessTime(taskProcessTime), m_TaskResultArbiter(taskResultArbiter)
        {}
        virtual ~TransferTaskFactoryForTest() NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_Lock)> lock(m_Lock);
            NN_SDK_REQUIRES(!m_Created);
        }

        // virtual nn::olsc::srv::TransferTaskBase* Create(const nn::olsc::srv::TransferTaskDetailInfo& detailInfo, const nn::olsc::srv::SeriesInfo& lastSi, nn::olsc::srv::TransferTaskBase::TransferTaskExecutionResource& m_ExecutionResource) NN_NOEXCEPT NN_OVERRIDE
        virtual nn::olsc::srv::TransferTaskBase* Create(const nn::olsc::srv::TransferTaskDetailInfo& detailInfo, const nn::olsc::srv::SeriesInfo& lastSi, nn::olsc::srv::TransferTaskBase::TransferTaskExecutionResource& m_ExecutionResource, nn::olsc::srv::TransferTaskSuspendedInfo&& suspendedInfo) NN_NOEXCEPT NN_OVERRIDE
        {
            std::lock_guard<decltype(m_Lock)> lock(m_Lock);
            NN_SDK_REQUIRES(!m_Created);

            auto p = new (&m_Storage) TransferTaskForTest(m_Ttdm, detailInfo, m_DoneTaskList, m_TaskProcessTime, m_TaskResultArbiter(detailInfo), m_ExecutionResource);
            p->Initialize(detailInfo, lastSi);
            m_Created = true;
            return p;
        }
        virtual void Destroy(nn::olsc::srv::TransferTaskBase* task) NN_NOEXCEPT NN_OVERRIDE
        {
            std::lock_guard<decltype(m_Lock)> lock(m_Lock);
            NN_SDK_REQUIRES(m_Created);
            m_Created = false;
            task->~TransferTaskBase();
        }

    private:

        std::aligned_storage<sizeof(TransferTaskForTest)>::type m_Storage;
        nn::os::SdkMutex m_Lock;
        bool m_Created;
        nn::olsc::srv::database::TransferTaskDatabaseManager& m_Ttdm;
        std::vector<nn::olsc::TransferTaskId>& m_DoneTaskList;
        int m_TaskProcessTime;
        TaskResultArbiter m_TaskResultArbiter;
    };

    using CompleteCondition = std::function<bool()>;
    bool WaitTaskEvent(nn::os::SystemEventType* taskEvent, const CompleteCondition& condition, int timeout)
    {
        const int FinishedIndex = 1;
        const int TimeoutIndex = 0;
        nn::os::TimerEvent timeoutEvent(nn::os::EventClearMode::EventClearMode_ManualClear);
        timeoutEvent.StartOneShot(nn::TimeSpan::FromSeconds(timeout));
        while (NN_STATIC_CONDITION(true))
        {
            auto signaled = nn::os::WaitAny(timeoutEvent.GetBase(), taskEvent);
            if (signaled == FinishedIndex)
            {
                nn::os::ClearSystemEvent(taskEvent);
                if (condition())
                {
                    return true;
                }
            }
            else if (signaled == TimeoutIndex)
            {
                NN_LOG("Timeout\n");
                return false;
            }
        }
        NN_ABORT("Not come here.\n");
    }
}}

