﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/database/olsc_SeriesInfoDatabase.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_SaveData.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>

//#define NN_OLSC_TEST_SERIES_INFO_DATABASE_MAX_ENTRY

using namespace nn;

namespace {
    const int64_t   SaveDataSize                = 0x00200000;   // 2MB
    const int64_t   JournalSize                 = 0x00200000;   // 2MB
    const int       SaveDataFlag                = 0;
    const fs::SystemSaveDataId SystemSaveDataId = 0x8000000000004000;

    olsc::srv::util::MountInfo TestSaveInfo = {
        SystemSaveDataId,
        SaveDataSize,
        JournalSize,
        SaveDataFlag,
    };

    class OlscDataArrayTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            NNT_ASSERT_RESULT_SUCCESS(InitializeSaveData());
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            account::InitializeForSystemService();
            fs::SetEnabledAutoAbort(false);
            fs::DisableAutoSaveDataCreation();
        }

        static void TearDownTestCase()
        {
        }
    private:

        static Result InitializeSaveData()
        {
            NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, SystemSaveDataId, {}))
                NN_RESULT_CATCH(fs::ResultTargetNotFound)
                {
                }
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        }
    };

    class Stopwatch
    {
    public:
        Stopwatch(bool enable, const char* fmt, ...) : m_Enable(enable)
        {
            std::va_list vlist;
            va_start(vlist, fmt);
            util::VSNPrintf(m_Title, sizeof(m_Title), fmt, vlist);
            va_end(vlist);

            m_Begin = os::GetSystemTick();
        }
        ~Stopwatch()
        {
            if (m_Enable)
            {
                auto end = os::GetSystemTick();
                NN_LOG("%s: %lld ms\n", m_Title, (end - m_Begin).ToTimeSpan().GetMilliSeconds());
            }
        }
    private:
        char m_Title[128];
        bool m_Enable;
        os::Tick m_Begin;
    };

    olsc::srv::SeriesInfo CreateSeriesInfoByApplicationId(ApplicationId appId)
    {
        olsc::srv::SeriesInfo si;
        si.seriesId = appId.value;
        char hash[32];
        crypto::GenerateSha256Hash(hash, sizeof(hash), &appId, sizeof(appId));
        memcpy(&si.commitId, hash, sizeof(si.commitId));
        return si;
    }

    Result GetUid(account::Uid* out, int index)
    {
        account::Uid uids[account::UserCountMax];

        int listCount;
        NN_RESULT_DO(account::ListAllUsers(&listCount, uids, account::UserCountMax));
        NN_ABORT_UNLESS(index < listCount);

        *out = uids[index];

        NN_RESULT_SUCCESS;
    }
}

TEST_F(OlscDataArrayTest, Basic)
{
    account::Uid uid;
    NNT_EXPECT_RESULT_SUCCESS(GetUid(&uid, 0));

    olsc::srv::util::DefaultMountManager mountManager(TestSaveInfo, TestSaveInfo, TestSaveInfo);
    nn::olsc::srv::database::SeriesInfoDatabase db(uid, mountManager);

#if defined(NN_OLSC_TEST_SERIES_INFO_DATABASE_MAX_ENTRY)
    const int MaxAppCount = olsc::srv::MaxApplicationCount;
#else
    const int MaxAppCount = 512;
#endif


    // 書き
    for (int i = 0; i < MaxAppCount; ++i)
    {
        ApplicationId appId = { static_cast<Bit64>(i) };
        auto si = CreateSeriesInfoByApplicationId(appId);
        {
            Stopwatch s(i % 32 == 0, "Confirm not registered: %016llx\n", appId.value);
            EXPECT_FALSE(db.Find(appId));
        }
        {
            Stopwatch s(i % 32 == 0, "Update: %016llx\n", appId.value);
            NNT_EXPECT_RESULT_SUCCESS(db.Update(appId, si));
        }
    }

    // 読み・削除
    for (int i = 0; i < MaxAppCount; ++i)
    {
        ApplicationId appId = { static_cast<Bit64>(i) };
        auto expected = CreateSeriesInfoByApplicationId(appId);
        {
            Stopwatch s(i % 32 == 0, "Get: %016llx\n", appId.value);
            auto si = db.Find(appId);
            EXPECT_TRUE(si);
            EXPECT_EQ(*si, expected);
        }
        {
            Stopwatch s(i % 32 == 0, "Remove: %016llx\n", appId.value);
            NNT_EXPECT_RESULT_SUCCESS(db.Remove(appId));

        }
        {
            Stopwatch s(i % 32 == 0, "Confirm removed: %016llx\n", appId.value);
            EXPECT_FALSE(db.Find(appId));
        }
    }
}
