﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/olsc_TransferTaskTypes.h>
#include <nn/olsc/srv/database/olsc_ErrorHistoryDatabase.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_SaveData.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>

#include <random>
#include "testOlsc_Stopwatch.h"

//#define NN_TEST_OLSC_ERROR_HISTORY_DATABASE_MAX_ENTRY

using namespace nn;

namespace {
    const int64_t   SaveDataSize                = 0x00200000;   // 2MB
    const int64_t   JournalSize                 = 0x00200000;   // 2MB
    const int       SaveDataFlag                = 0;
    const fs::SystemSaveDataId SystemSaveDataId = 0x8000000000004000;

    olsc::srv::util::MountInfo TestSaveInfo = {
        SystemSaveDataId,
        SaveDataSize,
        JournalSize,
        SaveDataFlag,
    };

    class OlscErrorHistoryDatabaseTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            NNT_ASSERT_RESULT_SUCCESS(InitializeSaveData());
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(time::Initialize());
            account::InitializeForSystemService();
            fs::SetEnabledAutoAbort(false);
            fs::DisableAutoSaveDataCreation();
        }

        static void TearDownTestCase()
        {
        }
    private:

        static Result InitializeSaveData()
        {
            NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, SystemSaveDataId, {}))
                NN_RESULT_CATCH(fs::ResultTargetNotFound)
                {
                }
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        }
    };

    Result GetUid(account::Uid* out, int index)
    {
        account::Uid uids[account::UserCountMax];

        int listCount;
        NN_RESULT_DO(account::ListAllUsers(&listCount, uids, account::UserCountMax));
        NN_ABORT_UNLESS(index < listCount);

        *out = uids[index];

        NN_RESULT_SUCCESS;
    }

    struct Key
    {
        account::Uid uid;
        ApplicationId appId;
    };

    void CreateKeyList(Key out[], int maxOutCount, int maxAppCount, const account::Uid uid[], int uidCount) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(uidCount * maxAppCount <= maxOutCount);

        int outCount = 0;
        for (int i = 0; i < uidCount; ++i)
        {
            for (int j = 0; j < maxAppCount; ++j)
            {
                Key key;
                key.appId = { static_cast<Bit64>(j) };
                key.uid = uid[i];
                out[outCount] = key;
                outCount++;
            }
        }

        std::random_device seed_gen;
        std::mt19937 engine(seed_gen());
        std::shuffle(&out[0], &out[outCount], engine);
    }
}

using ErrorHistoryDatabase = olsc::srv::database::ErrorHistoryDatabase;

namespace nn { namespace olsc { namespace srv { namespace database {
    bool operator==(const ErrorHistoryDatabase::ErrorInfo& lhs, const ErrorHistoryDatabase::ErrorInfo& rhs)
    {
        return //lhs.registeredTime == rhs.registeredTime
            lhs.result.GetInnerValueForDebug() == rhs.result.GetInnerValueForDebug()
            && lhs.taskKind == rhs.taskKind;
    }
}}}}

TEST_F(OlscErrorHistoryDatabaseTest, ApplicationSetting)
{
    const int MaxUserCount = 2;
    account::Uid uid[MaxUserCount];
    for (int i = 0; i < MaxUserCount; ++i)
    {
        NNT_EXPECT_RESULT_SUCCESS(GetUid(&uid[i], i));
    }

    olsc::srv::util::DefaultMountManager mountManager(TestSaveInfo, TestSaveInfo, TestSaveInfo);
    ErrorHistoryDatabase db(mountManager);

    const ErrorHistoryDatabase::ErrorInfo DummyErrorInfo = {
        olsc::TransferTaskKind::Download,
        true,
        {},
        olsc::ResultNotImplemented()
    };

#if defined(NN_TEST_OLSC_ERROR_HISTORY_DATABASE_MAX_ENTRY)
    const int MaxAppCount = olsc::srv::MaxApplicationCount * account::UserCountMax;
#else
    const int MaxAppCount = olsc::srv::MaxApplicationCount / 2;
#endif

    const int MaxKeyCount = MaxUserCount * MaxAppCount;

    std::unique_ptr<Key[]> keyList(new Key[MaxKeyCount]);
    CreateKeyList(keyList.get(), MaxKeyCount, MaxAppCount, uid, MaxUserCount);

    // 書き・読み
    for (int i = 0; i < MaxKeyCount; ++i)
    {
        bool enableLog = (i % 64 == 0);
        if (enableLog)
        {
            NN_LOG("GetAndSet[%d]\n", i);
        }
        auto& key = keyList[i];
        {
            nnt::olsc::Stopwatch s(enableLog, "  Get(Empty)");
            EXPECT_FALSE(db.GetLastErrorInfo(key.uid, key.appId));
        }

        {
            nnt::olsc::Stopwatch s(enableLog, "  Set");
            db.SetLastError(key.uid, key.appId, DummyErrorInfo.taskKind, true, DummyErrorInfo.result);
        }

        {
            nnt::olsc::Stopwatch s(enableLog, "  Get");
            auto lastErrorInfo = db.GetLastErrorInfo(key.uid, key.appId);
            ASSERT_TRUE(lastErrorInfo);
            EXPECT_EQ(DummyErrorInfo, *lastErrorInfo);
        }
    }

    std::unique_ptr<Key[]> sortedKeyList(new Key[MaxKeyCount]);
    std::memcpy(sortedKeyList.get(), keyList.get(), MaxKeyCount * sizeof(Key));
    std::sort(&sortedKeyList[0], &sortedKeyList[MaxKeyCount], [](const Key& lhs, const Key& rhs) -> bool
    {
        if (lhs.appId.value < rhs.appId.value) {
            return true;
        }
        else if (lhs.appId.value > rhs.appId.value) {
            return false;
        }

        for (int i = 0; i < 2; ++i)
        {
            if (lhs.uid._data[i] < rhs.uid._data[i]) {
                return true;
            }
            else if (lhs.uid._data[i] > rhs.uid._data[i]) {
                return false;
            }
        }
        return false;
    });

    std::unique_ptr<olsc::TransferTaskErrorInfo[]> listBuffer(new olsc::TransferTaskErrorInfo[MaxKeyCount]);
    db.ListLastErrorInfoAsTransferTaskErrorInfo(listBuffer.get(), MaxKeyCount, 0);

    for (int i = 0; i < MaxKeyCount; ++i)
    {
        auto& ei = listBuffer[i];
        auto& key = sortedKeyList[i];

        EXPECT_EQ(key.appId.value, ei.applicationId.value);
        EXPECT_EQ(key.uid, ei.uid);
    }

    // 削除・読み
    for (int i = 0; i < MaxAppCount; ++i)
    {
        bool enableLog = (i % 64 == 0);
        if (enableLog)
        {
            NN_LOG("GetAndRemove[%d]\n", i);
        }
        auto& key = keyList[i];
        {
            nnt::olsc::Stopwatch s(enableLog, "  Remove");
            db.RemoveLastErrorInfo(key.uid, key.appId);
        }

        {
            nnt::olsc::Stopwatch s(enableLog, "  Get(Empty)");
            EXPECT_FALSE(db.GetLastErrorInfo(key.uid, key.appId));
        }
    }
}
