﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/database/olsc_Datastore.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>

#include "testOlsc_Stopwatch.h"

using namespace nn;

namespace {
    const int64_t   SaveDataSize                = 0x00200000;   // 2MB
    const int64_t   JournalSize                 = 0x00200000;   // 2MB
    const int       SaveDataFlag                = 0;
    const char*     MetaFilePath                = "test_db_meta";
    const char*     EntryFilePath               = "test_db_data";
    const fs::SystemSaveDataId SystemSaveDataId = 0x8000000000004000;

    olsc::srv::util::MountInfo TestSaveInfo = {
        SystemSaveDataId,
        SaveDataSize,
        JournalSize,
        SaveDataFlag
    };

    class OlscSimpleDatastoreTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            NNT_ASSERT_RESULT_SUCCESS(InitializeSaveData());
        }

        virtual void TearDown()
        {
            NNT_ASSERT_RESULT_SUCCESS(FinalizeSaveData());
        }

        static void SetUpTestCase()
        {
            fs::SetEnabledAutoAbort(false);
            fs::DisableAutoSaveDataCreation();
            NNT_ASSERT_RESULT_SUCCESS(time::Initialize());
        }

        static void TearDownTestCase()
        {
            NNT_ASSERT_RESULT_SUCCESS(time::Finalize());
        }
    private:

        static Result InitializeSaveData()
        {
            NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, SystemSaveDataId, {}))
                NN_RESULT_CATCH(fs::ResultTargetNotFound)
                {
                }
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        }

        static Result FinalizeSaveData()
        {
            NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, SystemSaveDataId, {}))
                NN_RESULT_CATCH(fs::ResultTargetNotFound)
                {
                }
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        }

    };

    const int MaxEntryCount = 2000;

    using SimpleDatastoreForTestBase = olsc::srv::database::Datastore<olsc::srv::util::DefaultMountManager, int, MaxEntryCount>;
    class SimpleDatastoreForTest : public SimpleDatastoreForTestBase
    {
    public:
        NN_IMPLICIT SimpleDatastoreForTest(olsc::srv::util::DefaultMountManager& mountManager) : SimpleDatastoreForTestBase(mountManager)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(InitializeUnsafe());
        }

    protected:
        virtual const char* GetMetadataFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE
        {
            return MetaFilePath;
        }

        virtual const char* GetEntryFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE
        {
            return EntryFilePath;
        }

        virtual olsc::srv::util::ReadMount AcquireReadMount(olsc::srv::util::DefaultMountManager& mounter) const NN_NOEXCEPT NN_OVERRIDE
        {
            return mounter.AcquireDeviceSaveForRead();
        }

        virtual olsc::srv::util::WriteMount AcquireWriteMount(olsc::srv::util::DefaultMountManager& mounter) const NN_NOEXCEPT NN_OVERRIDE
        {
            return mounter.AcquireDeviceSaveForWrite();
        }
    };

}


TEST_F(OlscSimpleDatastoreTest, Basic)
{
    nn::olsc::srv::util::DefaultMountManager mountManager(TestSaveInfo, TestSaveInfo, TestSaveInfo);

    SimpleDatastoreForTest db(mountManager);

    std::lock_guard<decltype(db)> lock(db);

    EXPECT_EQ(0, db.GetCount());

    {
        nnt::olsc::Stopwatch s(true, "Add entries");
        for (int i = 0; i < MaxEntryCount; ++i)
        {
//            EXPECT_TRUE(db.HasSpace());
            int index;
            EXPECT_TRUE(db.Put(&index, i));
            EXPECT_EQ(i, index);
        }
    }

//    EXPECT_FALSE(db.HasSpace());
    EXPECT_EQ(MaxEntryCount, db.GetCount());

    {
        nnt::olsc::Stopwatch s(true, "Get entries");
        for (int i = 0; i < MaxEntryCount; ++i)
        {
            int value;
            EXPECT_TRUE(db.Get(&value, i));
            EXPECT_EQ(i, value);
        }
    }

    {
        nnt::olsc::Stopwatch s(true, "Replace entries");
        for (int i = 0; i < MaxEntryCount; ++i)
        {
            int value = i + 1;
            EXPECT_TRUE(db.Replace(i, value));
        }
    }

    {
        nnt::olsc::Stopwatch s(true, "Get entries");
        for (int i = 0; i < MaxEntryCount; ++i)
        {
            int value;
            EXPECT_TRUE(db.Get(&value, i));
            EXPECT_EQ(i + 1, value);
        }
    }

#if 0
    {
        EXPECT_FALSE(db.HasSpace());
        db.Remove(0);
        EXPECT_TRUE(db.HasSpace());
    }
#endif
    {
        nnt::olsc::Stopwatch s(true, "Remove entries");
        for (int i = 0; i < MaxEntryCount; ++i)
        {
            db.Remove(i);
            int value;
            EXPECT_FALSE(db.Get(&value, i));
            EXPECT_FALSE(db.Replace(i, value));
        }
    }

    EXPECT_EQ(0, db.GetCount());
}
