﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/database/olsc_CachedReaderWriter.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_File.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>

#include "testOlsc_Stopwatch.h"

#include <random>

using namespace nn;

namespace {
    const int64_t   SaveDataSize                = 0x00200000;   // 2MB
    const int64_t   JournalSize                 = 0x00200000;   // 2MB
    const int       SaveDataFlag                = 0;
    const fs::SystemSaveDataId SystemSaveDataId = 0x8000000000004000;

    olsc::srv::util::MountInfo TestSaveInfo = {
        SystemSaveDataId,
        SaveDataSize,
        JournalSize,
        SaveDataFlag
    };

    class OlscCachedReaderWriterTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            NNT_ASSERT_RESULT_SUCCESS(InitializeSaveData());
        }

        virtual void TearDown()
        {
            NNT_ASSERT_RESULT_SUCCESS(FinalizeSaveData());
        }

        static void SetUpTestCase()
        {
            fs::SetEnabledAutoAbort(false);
            fs::DisableAutoSaveDataCreation();
            NNT_ASSERT_RESULT_SUCCESS(time::Initialize());
        }

        static void TearDownTestCase()
        {
            NNT_ASSERT_RESULT_SUCCESS(time::Finalize());
        }
    private:

        static Result InitializeSaveData()
        {
            NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, SystemSaveDataId, {}))
                NN_RESULT_CATCH(fs::ResultTargetNotFound)
                {
                }
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        }

        static Result FinalizeSaveData()
        {
            NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, SystemSaveDataId, {}))
                NN_RESULT_CATCH(fs::ResultTargetNotFound)
                {
                }
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        }

    };

    void SetupTestData(const char* path, int out[], int maxCount)
    {
        for (int i = 0; i < maxCount; ++i)
        {
            out[i] = i;
        }

        auto createResult = olsc::srv::util::CreateFile(path, sizeof(int) * maxCount);
        ASSERT_TRUE(createResult.IsSuccess() || createResult <= fs::ResultPathAlreadyExists());
        NNT_ASSERT_RESULT_SUCCESS(olsc::srv::util::WriteFile(path, out, sizeof(int) * maxCount));
    }

    const int MaxEntryCount = 64;
    const int CacheCount = MaxEntryCount / 8;

    using ReaderWriter = olsc::srv::database::CachedReaderWriter<int>;

    void MakeExpectedMovedData(int out[], int maxCount, int dataCount, int moveCount, int moveOffset)
    {
        for (int i = 0; i < maxCount; ++i)
        {
            out[i] = i;
        }

        std::memmove(&out[moveOffset + moveCount], &out[moveOffset], dataCount * sizeof(int));
    }

    void TestMove(int maxDataCount, int moveDataCount, int moveCount, int moveOffset)
    {
        NN_LOG("MaxDataCount: %d, MoveDataCount: %d, MoveCount: %d, MoveOffset: %d\n", maxDataCount, moveDataCount, moveCount, moveOffset);

        nn::olsc::srv::util::DefaultMountManager mountManager(TestSaveInfo, TestSaveInfo, TestSaveInfo);
        auto writeMount = mountManager.AcquireDeviceSaveForWrite();
        char path[64];
        nn::util::TSNPrintf(path, sizeof(path), "%s%s", writeMount.GetRootPath(), "data");

        std::unique_ptr<int[]> initialData(new int[maxDataCount]);
        SetupTestData(path, initialData.get(), maxDataCount);

        std::unique_ptr<int[]> expectedData(new int[maxDataCount]);
        MakeExpectedMovedData(expectedData.get(), maxDataCount, moveDataCount, moveCount, moveOffset);

        std::unique_ptr<int[]> cacheBuffer(new int[CacheCount]);
        ReaderWriter readerWriter(cacheBuffer.get(), CacheCount, maxDataCount);

        NNT_ASSERT_RESULT_SUCCESS(readerWriter.Move(path, moveOffset + moveCount, moveOffset, moveDataCount));

        std::unique_ptr<int[]> readBuffer(new int[maxDataCount]);
        int readCount;
        NNT_ASSERT_RESULT_SUCCESS(readerWriter.Read(&readCount, readBuffer.get(), path, 0, maxDataCount));
        if (moveCount == 0)
        {
            EXPECT_TRUE(std::memcmp(readBuffer.get(), expectedData.get(), maxDataCount * sizeof(int)) == 0);
        }
        else if (moveCount > 0)
        {
            EXPECT_TRUE(std::memcmp(readBuffer.get(), expectedData.get(), moveOffset * sizeof(int)) == 0);

            int compOffset = moveOffset + moveCount;
            int compCount = maxDataCount - compOffset;
            EXPECT_TRUE(std::memcmp(readBuffer.get() + compOffset, expectedData.get() + compOffset, compCount * sizeof(int)) == 0);
        }
        else if (moveCount < 0)
        {
            auto destOffset = moveOffset + moveCount + moveDataCount;
            EXPECT_TRUE(std::memcmp(readBuffer.get(), expectedData.get(), destOffset * sizeof(int)) == 0);

            auto compOffset = moveOffset + moveDataCount;
            auto compCount = maxDataCount - compOffset;
            EXPECT_TRUE(std::memcmp(readBuffer.get() + compOffset, expectedData.get() + compOffset, compCount * sizeof(int)) == 0);
        }
    }
}


TEST_F(OlscCachedReaderWriterTest, Read)
{
    nn::olsc::srv::util::DefaultMountManager mountManager(TestSaveInfo, TestSaveInfo, TestSaveInfo);
    auto writeMount = mountManager.AcquireDeviceSaveForWrite();
    char path[64];
    nn::util::TSNPrintf(path, sizeof(path), "%s%s", writeMount.GetRootPath(), "data");


    static std::array<int, CacheCount> cacheBuffer;
    static std::array<int, MaxEntryCount> data;
    SetupTestData(path, data.data(), static_cast<int>(data.size()));

    static std::array<int, MaxEntryCount> readBuffer;

    // キャッシュサイズ以内の読み
    {
        ReaderWriter readerWriter(cacheBuffer.data(), static_cast<int>(cacheBuffer.size()), MaxEntryCount);
        for (int i = 0; i < MaxEntryCount; i += CacheCount)
        {
            NN_LOG("Read cache size, i = %d\n", i);
            int readSize;
            NNT_ASSERT_RESULT_SUCCESS(readerWriter.Read(&readSize, readBuffer.data(), path, i, CacheCount));
            EXPECT_EQ(CacheCount, readSize);
            EXPECT_TRUE(std::memcmp(&readBuffer[0], &data[i], sizeof(int) * readSize) == 0);
        }
    }

    // キャッシュサイズを超える読み
    {
        ReaderWriter readerWriter(cacheBuffer.data(), static_cast<int>(cacheBuffer.size()), MaxEntryCount);
        const int ReadSize = CacheCount + 1;
        for (int i = 0; i < MaxEntryCount; i += ReadSize)
        {
            NN_LOG("Read large size, i = %d\n", i);
            int readCount;
            NNT_ASSERT_RESULT_SUCCESS(readerWriter.Read(&readCount, readBuffer.data(), path, i, ReadSize));
            NN_LOG("ReadCount: %d\n", readCount);
            EXPECT_TRUE(std::memcmp(&readBuffer[0], &data[i], sizeof(int) * readCount) == 0);
        }
    }
}

TEST_F(OlscCachedReaderWriterTest, Write)
{
    nn::olsc::srv::util::DefaultMountManager mountManager(TestSaveInfo, TestSaveInfo, TestSaveInfo);
    auto writeMount = mountManager.AcquireDeviceSaveForWrite();
    char path[64];
    nn::util::TSNPrintf(path, sizeof(path), "%s%s", writeMount.GetRootPath(), "data");


    static std::array<int, CacheCount> cacheBuffer;
    static std::array<int, MaxEntryCount> data;
    SetupTestData(path, data.data(), static_cast<int>(data.size()));

    static std::array<int, MaxEntryCount> readBuffer;

    // キャッシュに載ってる箇所への書き込み
    {
        ReaderWriter readerWriter(cacheBuffer.data(), static_cast<int>(cacheBuffer.size()), MaxEntryCount);
        int readCount;
        NNT_ASSERT_RESULT_SUCCESS(readerWriter.Read(&readCount, readBuffer.data(), path, 0, CacheCount));

        NNT_ASSERT_RESULT_SUCCESS(readerWriter.Write(path, &data[1], 0, 1));
        NNT_ASSERT_RESULT_SUCCESS(readerWriter.Read(&readCount, readBuffer.data(), path, 0, CacheCount));
        EXPECT_EQ(1, readBuffer[0]);
    }

    // キャッシュに載ってない箇所への書き込み
    {
        ReaderWriter readerWriter(cacheBuffer.data(), static_cast<int>(cacheBuffer.size()), MaxEntryCount);
        int readCount;
        NNT_ASSERT_RESULT_SUCCESS(readerWriter.Read(&readCount, readBuffer.data(), path, 0, CacheCount));

        NNT_ASSERT_RESULT_SUCCESS(readerWriter.Write(path, &data[1], CacheCount, 1));
        NNT_ASSERT_RESULT_SUCCESS(readerWriter.Read(&readCount, readBuffer.data(), path, CacheCount, CacheCount));
        EXPECT_EQ(1, readBuffer[0]);
    }
}

TEST_F(OlscCachedReaderWriterTest, MoveToBack)
{
    nn::olsc::srv::util::DefaultMountManager mountManager(TestSaveInfo, TestSaveInfo, TestSaveInfo);
    auto writeMount = mountManager.AcquireDeviceSaveForWrite();
    char path[64];
    nn::util::TSNPrintf(path, sizeof(path), "%s%s", writeMount.GetRootPath(), "data");


    static std::array<int, CacheCount> cacheBuffer;
    static std::array<int, MaxEntryCount> data;
    SetupTestData(path, data.data(), static_cast<int>(data.size()));

    static std::array<int, MaxEntryCount> readBuffer;

    // 後ろ方向への Move
    {
        ReaderWriter readerWriter(cacheBuffer.data(), static_cast<int>(cacheBuffer.size()), MaxEntryCount);
        NNT_ASSERT_RESULT_SUCCESS(readerWriter.Move(path, 1, 0, MaxEntryCount - 1));
        int readCount;
        NNT_ASSERT_RESULT_SUCCESS(readerWriter.Read(&readCount, readBuffer.data(), path, 0, MaxEntryCount));
        for (int i = 0; i < MaxEntryCount - 1; ++i)
        {
            EXPECT_EQ(i, readBuffer[i + 1]);
        }
    }

}

TEST_F(OlscCachedReaderWriterTest, MoveToFront)
{
    nn::olsc::srv::util::DefaultMountManager mountManager(TestSaveInfo, TestSaveInfo, TestSaveInfo);
    auto writeMount = mountManager.AcquireDeviceSaveForWrite();
    char path[64];
    nn::util::TSNPrintf(path, sizeof(path), "%s%s", writeMount.GetRootPath(), "data");


    static std::array<int, CacheCount> cacheBuffer;
    static std::array<int, MaxEntryCount> data;
    SetupTestData(path, data.data(), static_cast<int>(data.size()));

    static std::array<int, MaxEntryCount> readBuffer;

    // 前方向への Move
    {
        ReaderWriter readerWriter(cacheBuffer.data(), static_cast<int>(cacheBuffer.size()), MaxEntryCount);
        NNT_ASSERT_RESULT_SUCCESS(readerWriter.Move(path, 0, 1, MaxEntryCount - 1));
        int readCount;
        NNT_ASSERT_RESULT_SUCCESS(readerWriter.Read(&readCount, readBuffer.data(), path, 0, MaxEntryCount));
        for (int i = 0; i < MaxEntryCount - 1; ++i)
        {
            EXPECT_EQ(i + 1, readBuffer[i]);
        }
    }

}

TEST_F(OlscCachedReaderWriterTest, MoveVariation)
{
    int MoveDataCountList[] = { CacheCount - 1, CacheCount, CacheCount + 1 };
    int MoveCountList[] = {
        0, 1, -1,
        CacheCount, CacheCount - 1, CacheCount + 1,
        -CacheCount, -(CacheCount - 1), -(CacheCount + 1)
    };
    int MoveOffsetList[] = { CacheCount * 2 };

    for (auto moveDataCount : MoveDataCountList)
    {
        for (auto moveCount : MoveCountList)
        {
            for (auto moveOffset : MoveOffsetList)
            {
                TestMove(MaxEntryCount, moveDataCount, moveCount, moveOffset);
            }
        }
    }
}
