﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/nn_Windows.h>
#endif

#include <cstdlib>
#include <cstring>
#include <nn/os.h>
#include <nn/nn_ApplicationId.h>

#include <nn/fs/fs_SaveData.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Directory.h>

#include <nn/ns/ns_ApplicationVersionSystemApi.h>
#include <nn/ns/ns_InitializationApi.h>

#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_Utility.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/fsUtil/testFs_util_function.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabase.h>
#include <nn/olsc/srv/transfer/olsc_SaveDataArchiveUpload.h>
#include <nn/olsc/srv/transfer/olsc_SaveDataArchiveDownload.h>

#include <nn/account/account_Types.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_TypesForSystemServices.h>

#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/time/time_Api.h>
#include <nn/socket.h>
#include <nn/socket/socket_SystemConfig.h>

#include <nn/ncm/ncm_ContentMetaId.h>

#include <curl/curl.h>
#include "testOlsc_Util.h"

namespace nnt { namespace olsc {

namespace
{
    // Socket が要求するメモリプール
    const int SocketCount = 3;
    const int SocketConcurrencyCount = SocketCount;
    nn::socket::SystemConfigLightDefaultWithMemory<SocketCount, 0> g_SocketConfigWithMemory(SocketConcurrencyCount);
}

const nn::Bit64 BaseApplicationIdValue = 0x010000000000B94AULL;

const nn::fs::SystemSaveDataId SystemSaveDataIdForTest = 0x8000000000004000;
const int64_t                  SystemSaveDataSizeForTest = 4 * 1024 * 1024;
const int64_t                  SystemSaveJournalSizeForTest = 4 * 1024 * 1024;
const uint32_t                 SystemSaveDataFlagsForTest = 0;

// テストで使いまわす MountInfo テンプレ
const nn::olsc::srv::util::MountInfo MountInfoForTest =
{
    SystemSaveDataIdForTest,
    SystemSaveDataSizeForTest,
    SystemSaveJournalSizeForTest,
    SystemSaveDataFlagsForTest,
};

const nn::olsc::srv::util::MountInfo MountInfoForTestDeviceSave =
{
    SystemSaveDataIdForTest + 1,
    SystemSaveDataSizeForTest,
    SystemSaveJournalSizeForTest,
    SystemSaveDataFlagsForTest,
};

const nn::olsc::srv::util::MountInfo MountInfoForTestUserSettingSave =
{
    SystemSaveDataIdForTest + 2,
    SystemSaveDataSizeForTest,
    SystemSaveJournalSizeForTest,
    SystemSaveDataFlagsForTest,
};

const nn::olsc::srv::util::MountInfo MountInfoForTestUserSeriesInfoSave =
{
    SystemSaveDataIdForTest + 3,
    SystemSaveDataSizeForTest,
    SystemSaveJournalSizeForTest,
    SystemSaveDataFlagsForTest,
};

inline void DumpSda(const nn::olsc::srv::SaveDataArchiveInfo& sda)
{
    NN_LOG("Sda Id  : 0x%llx\n", sda.id);
    NN_LOG("app Id  : 0x%llx\n", sda.applicationId.value);
    NN_LOG("uid[0]  : 0x%llx\n", sda.userId._data[0]);
    NN_LOG("uid[1]  : 0x%llx\n", sda.userId._data[1]);
    NN_LOG("Did     : 0x%llx\n", sda.deviceId);
    NN_LOG("Size    : 0x%llx\n", sda.dataSize);
    NN_LOG("count   : 0x%llx\n", sda.numOfPartitions);
    NN_LOG("thumb   : 0x%ld\n",  sda.hasThumbnail);
    NN_LOG("savedAt : %lld\n",   sda.savedAt.value);
    NN_LOG("finiAt  : %lld\n",   sda.finishedAt.value);
    NN_LOG("\n");
}

inline void CreateUsers(nn::account::Uid *pUsers, int count) NN_NOEXCEPT
{
    for (auto i = 0; i < count; ++ i)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::BeginUserRegistration(pUsers + i));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::CompleteUserRegistration(pUsers[i]));
    }
}

inline void CleanupUsers() NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        nn::account::Uid uid;
        int listCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&listCount, &uid, 1));
        if(listCount < 1)
        {
            break;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::DeleteUser(uid));
    }
}

inline nn::account::Uid GetFirstUserId() NN_NOEXCEPT
{
    int userCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserCount(&userCount));
    NN_ABORT_UNLESS(userCount > 0);

    nn::account::Uid uid;
    int listCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&listCount, &uid, 1));
    return uid;
}

inline nn::account::Uid GetUserId(int index) NN_NOEXCEPT
{
    int userCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserCount(&userCount));
    NN_ABORT_UNLESS(userCount > index);
    NN_ABORT_UNLESS(userCount <= nn::account::UserCountMax);

    nn::account::Uid uid[nn::account::UserCountMax];
    int listCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&listCount, uid, userCount));
    return uid[index];
}

inline void Initialize() NN_NOEXCEPT
{
    nn::fssystem::InitializeAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();

    nn::fs::SetEnabledAutoAbort(false);
    nn::time::Initialize();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfigWithMemory));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::InitializeSystem());

    NN_ABORT_UNLESS(curl_global_init(CURL_GLOBAL_ALL) == CURLE_OK);

#if defined( NN_BUILD_CONFIG_OS_WIN )
    nn::account::InitializeForAdministrator();

    nnt::olsc::CleanupUsers();
    nn::account::Uid uid;
    nnt::olsc::CreateUsers(&uid, 1);
#else
    nn::account::InitializeForSystemService();
#endif

}

inline void Finalize() NN_NOEXCEPT
{
    curl_global_cleanup();
    nn::socket::Finalize();
    // nn::nifm::Finalize();
}

// 与えられた uid, ApplicationId でセーブデータの箱だけ作り、マウントする
inline nn::Result CreateAndMountSaveData(const char* mountName, nn::ncm::ApplicationId appId, nn::fs::UserId userId, size_t savedataSize, size_t saveDataJournalSize) NN_NOEXCEPT
{
    NN_RESULT_TRY(nn::fs::MountSaveData(mountName, appId, userId))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
    {
        NNT_OLSC_RESULT_DO(nn::fs::CreateSaveData(appId, userId, appId.value, savedataSize, saveDataJournalSize, 0));
        NNT_OLSC_RESULT_DO(nn::fs::MountSaveData(mountName, appId, userId));
    }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

inline nn::Result DeleteTestSaveData(const nn::account::Uid& uid, const nn::ApplicationId& appId, nn::fs::SaveDataSpaceId spaceId)
{
    nn::util::optional<nn::fs::SaveDataInfo> infoSrc;
    nnt::fs::util::FindSaveData(&infoSrc, spaceId, [&](nn::fs::SaveDataInfo i) {
        return i.saveDataUserId._data[0] == uid._data[0] && i.saveDataUserId._data[1] == uid._data[1] && i.applicationId.value == appId.value;
    });
    if(infoSrc)
    {
        NNT_OLSC_RESULT_DO(nn::fs::DeleteSaveData(spaceId, infoSrc->saveDataId));
    }

    NN_RESULT_SUCCESS;
}

// 与えられた uid, ApplicationId でインクリで埋められたセーブデータを作成する
inline nn::Result CreateTestSaveData(const std::string fileName, const nn::account::Uid& uid, const nn::ApplicationId& appId, size_t savedataSize, size_t saveDataJournalSize, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    nn::ncm::ApplicationId ncmAppId;
    ncmAppId.value = appId.value;
    nn::fs::UserId userId;
    userId._data[0] = uid._data[0];
    userId._data[1] = uid._data[1];

    const std::string MountName = "save";

    std::string filePath = MountName + ":/" + fileName;

    NNT_OLSC_RESULT_DO(CreateAndMountSaveData(MountName.c_str(), ncmAppId, userId, savedataSize, saveDataJournalSize));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(MountName.c_str());
    };

    nn::fs::DeleteFile(filePath.c_str());
    NNT_OLSC_RESULT_DO(nn::fs::CreateFile(filePath.c_str(), 256));

    nn::fs::FileHandle handle;
    NNT_OLSC_RESULT_DO(nn::fs::OpenFile(&handle, filePath.c_str(), nn::fs::OpenMode_Write));

    auto writeBuffer = reinterpret_cast<char*>(workBuffer);

    // インクリのデータで埋めたファイル作成
    for(size_t i = 0; i < 256; i++)
    {
        writeBuffer[i] = static_cast<char>(i);
    }

    NNT_OLSC_RESULT_DO(nn::fs::WriteFile(handle, 0, writeBuffer, 256, nn::fs::WriteOption()));
    NNT_OLSC_RESULT_DO(nn::fs::FlushFile(handle));
    nn::fs::CloseFile(handle);
    nn::fs::CommitSaveData(MountName.c_str());

    NN_RESULT_SUCCESS;
}

inline nn::Result DeleteAllSaveDataArchiveFromServer(nn::olsc::srv::NsaIdToken& nsaIdToken, nn::account::Uid& uid, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    auto curl = curl_easy_init();

    nn::olsc::srv::SaveDataArchiveInfo sda[64];
    int count = 0;
    NNT_OLSC_RESULT_DO(nn::olsc::srv::transfer::RequestSaveDataArchiveInfoList(&count, sda, 64, nsaIdToken, curl, workBuffer, workBufferSize, nullptr));
    for(int i = 0; i < count; i++)
    {
        (nn::olsc::srv::transfer::DeleteSaveDataArchive(sda[i].id, nsaIdToken, curl, workBuffer, workBufferSize, nullptr));
    }

    NN_RESULT_SUCCESS;
}

inline nn::Result CalculateSaveDataHash(nnt::fs::util::Hash* pOut, const nn::account::Uid& uid, const nn::ApplicationId& appId, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    nn::ncm::ApplicationId ncmAppId;
    ncmAppId.value = appId.value;
    nn::fs::UserId userId;
    userId._data[0] = uid._data[0];
    userId._data[1] = uid._data[1];

    const std::string MountName = "save";

    NNT_OLSC_RESULT_DO(nn::fs::MountSaveData(MountName.c_str(), ncmAppId, userId));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(MountName.c_str());
    };

    nnt::fs::util::ListDirectoryRecursive((MountName + ":/").c_str());
    NNT_OLSC_RESULT_DO(nnt::fs::util::CalculateDirectoryTreeHash(pOut, (MountName + ":/").c_str(), workBuffer, workBufferSize));
    nnt::fs::util::DumpBuffer(pOut, sizeof(nnt::fs::util::Hash));
    NN_RESULT_SUCCESS;
}

/*
* @brief List から pred に最初にマッチする要素を取得します。
*/
template <typename Element, typename Pred>
inline void FindElement(nn::util::optional<Element>* pOutValue, Element* list, size_t listCount, Pred pred) NN_NOEXCEPT
{
    for(int i = 0; i < static_cast<int>(listCount); i++)
    {
        if(pred(list[i]))
        {
            *pOutValue = list[i];
            return;
        }
    }
    *pOutValue = nn::util::nullopt;
    return;
}

}} // ~namespace nnt::olsc
