﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <memory>

#include <nn/nn_Common.h>
#include <nnt/nntest.h>
#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/applet/applet.h>
#include <nv/nv_MemoryManagement.h>
#include <nv/nv_ServiceName.h>

#include <nv_Scheduler.h>

namespace
{
    const size_t GraphicsSystemMemorySize = 1 * 1024 * 1024;

    void DumpOverrunData(nv::sched::Client::OverrunEventType& overrunData)
    {
        NN_LOG("Overrun happens.\n");
        NN_LOG("  RunlistId: 0x%08llx\n", overrunData.runlist_id);
        NN_LOG("  Timeslice: %llu\n", overrunData.timeslice);
        NN_LOG("  DebtTotal: %llu\n", overrunData.debt_total);
        NN_LOG("  DebtFromPowergating: %llu\n", overrunData.debt_from_powergating);
    }
}

TEST(TestGpuScheduler, Overrun)
{
    nn::Result result;

    nv::SetGraphicsServiceName("nvdrv:s");
    nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);

    nv::sched::Client client;
    client.Open();

    nn::os::SystemEventType overrunEvent;
    client.GetOverrunEventHandle(&overrunEvent);

    nv::sched::Client::OverrunEventType overrunData;

    client.EnableOverrunEvents(true);

    for(int i = 0; i < 60; i++)
    {
        if(nn::os::TimedWaitSystemEvent(&overrunEvent, nn::TimeSpan::FromSeconds(1)))
        {
            NN_ABORT_UNLESS(client.HasOverrunData(), "Failed to get overrun data.\n");

            result = client.GetOverrunData(&overrunData);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            DumpOverrunData(overrunData);
        }
        else
        {
            NN_LOG("Overrun does not happen.\n");
        }
    }

    client.EnableOverrunEvents(false);

    // cleanup remained data
    while (client.HasOverrunData())
    {
        result = client.GetOverrunData(&overrunData);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
}
