﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_Thread.h>

#include <nn/ntc/shim/ntc_Shim.h>
#include <nn/socket.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiNetworkProfile.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_TypesNetworkInterface.h>

#include <algorithm>
#include <atomic>

#ifdef GetCurrentTime
#undef GetCurrentTime
#endif

#include <nn/time.h>
#include <nn/time/time_ApiForSystem.h>
#include <nn/time/time_ApiForMenu.h>
#include <nn/time/time_StandardNetworkSystemClock.h>

// EnsureNetworkClockAvailability, AsyncEnsureNetworkClockAvailabilityTask deprecated warning 回避
NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS

namespace
{
    nn::nifm::NetworkConnection* GetNetworkConnection() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(nn::nifm::NetworkConnection*, s_pNetworkConnection, = nullptr);
        NN_FUNCTION_LOCAL_STATIC(std::aligned_storage<sizeof(nn::nifm::NetworkConnection)>::type, s_NetworkConnectionStorage);

        if(!s_pNetworkConnection)
        {
            s_pNetworkConnection = new (&s_NetworkConnectionStorage) nn::nifm::NetworkConnection();
        }

        return s_pNetworkConnection;
    }

    void DumpNetworkTime() NN_NOEXCEPT
    {
        bool doInit = ! nn::time::IsInitialized();
        if(doInit)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::time::Initialize());
        }

        nn::time::PosixTime current;
        auto result = nn::time::StandardNetworkSystemClock::GetCurrentTime(&current);
        if(result.IsFailure())
        {
            NN_SDK_LOG("StandardNetworkSystemClock is not available.\n");
            return;
        }

        nn::time::CalendarTime c;
        NNT_ASSERT_RESULT_SUCCESS(nn::time::ToCalendarTime(&c, nullptr, current));

        NN_SDK_LOG("NetworkClock: %04d/%02d/%02d %02d:%02d:%02d\n",
            c.year, c.month, c.day, c.hour, c.minute, c.second);

        if(doInit)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize());
        }
    }

    void DumpNetworkInterface() NN_NOEXCEPT
    {
        NN_SDK_LOG("Dump Network interface ---------------------------\n");
        const int bufferCount = 8;
        nn::nifm::NetworkInterfaceInfo networkInterfaceInfo[bufferCount];
        int outCount;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::nifm::EnumerateNetworkInterfaces(
                networkInterfaceInfo, &outCount, bufferCount,
                nn::nifm::EnumerateNetworkInterfacesFilter_Ieee80211 | nn::nifm::EnumerateNetworkInterfacesFilter_Ethernet | nn::nifm::EnumerateNetworkInterfacesFilter_Extended
            )
        );

        NN_SDK_LOG("networkInterfaceInfoList.count : %d\n", outCount);
        const int count = std::min(bufferCount, outCount);
        for (int i = 0; i<count; ++i)
        {
            const auto &info = networkInterfaceInfo[i];
            NN_SDK_LOG(
                "%s (%s)\n",
                info.type == nn::nifm::NetworkInterfaceType_Invalid ? "Invalid" : info.type == nn::nifm::NetworkInterfaceType_Ieee80211 ? "IEEE802.11" : "Ethernet",
                info.isAvailable ? "available" : "not available"
            );
            NN_SDK_LOG("    mac address:%02x-%02x-%02x-%02x-%02x-%02x\n", info.macAddress.data[0], info.macAddress.data[1], info.macAddress.data[2], info.macAddress.data[3], info.macAddress.data[4], info.macAddress.data[5]);
        }

        NN_SDK_LOG("--------------------------------------------------\n");
    }
}

TEST(NtcApiTest, InitializeOtherLib)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::time::Initialize());
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());
}

#if !defined(NN_BUILD_CONFIG_OS_WIN) // 同一プロセスで実行される自立補正につられて成功扱いになることがあるので windows ではテストしない
// nn::nifm::SubmitNetworkRequestAndWait() 実行前に行う
TEST(NtcApiTest, NetworkRequestNotAccepted)
{
    // 非同期 API
    {
        nn::ntc::shim::CorrectionNetworkClockAsyncTask task(nn::os::EventClearMode_AutoClear, nn::ntc::EnsureNetworkClockAvailabilityMode_Default);
        NNT_ASSERT_RESULT_FAILURE(nn::time::ResultNetworkRequestNotAccepted, task.StartTask());
    }
    {
        nn::ntc::shim::CorrectionNetworkClockAsyncTask task(nn::os::EventClearMode_AutoClear, nn::ntc::EnsureNetworkClockAvailabilityMode_ForcibleDownload);
        NNT_ASSERT_RESULT_FAILURE(nn::time::ResultNetworkRequestNotAccepted, task.StartTask());
    }

    // ブロック API
    NNT_ASSERT_RESULT_FAILURE(nn::time::ResultNetworkRequestNotAccepted, nn::ntc::shim::EnsureNetworkClockAvailability());
}
#endif

TEST(NtcApiTest, SubmitNetworkRequest)
{
    const int TryCountMax = 3;

    for(int count = 0 ; count < TryCountMax ; ++count)
    {
        const nn::os::Tick start = nn::os::GetSystemTick();
        GetNetworkConnection()->SubmitRequestAndWait();
        const nn::os::Tick end = nn::os::GetSystemTick();

        if(GetNetworkConnection()->IsAvailable())
        {
            NNT_ASSERT_RESULT_SUCCESS(GetNetworkConnection()->GetResult());
            break;
        }

        NN_SDK_LOG("[NtcApiTest.SubmitNetworkRequest] Try %d/%d : SubmitRequestAndWait failed. IsAvailable:%s IsRequestOnHold:%s SubmitRequestAndWait elapsed %lld [msec]\n",
            count + 1,
            TryCountMax,
            GetNetworkConnection()->IsAvailable() ? "true" : "false",
            GetNetworkConnection()->IsRequestOnHold() ? "true" : "false",
            (end - start).ToTimeSpan().GetMilliSeconds());

        const auto result = GetNetworkConnection()->GetResult();
        NN_SDK_LOG("[NtcApiTest.SubmitNetworkRequest] SubmitRequestAndWait Result(%08x, %03d-%04d)\n",
            result.GetInnerValueForDebug(),
            result.GetModule(), result.GetDescription());

        DumpNetworkInterface();

        if(count == TryCountMax - 1)
        {
            NNT_ASSERT_RESULT_SUCCESS(GetNetworkConnection()->GetResult());
        }
    }
}

template <nn::ntc::EnsureNetworkClockAvailabilityMode Mode>
void AsyncTaskTest()
{
    {
        // StartTask して逃げる
        nn::ntc::shim::CorrectionNetworkClockAsyncTask task(nn::os::EventClearMode_AutoClear, Mode);
        EXPECT_FALSE(task.IsProcessing());
        NNT_ASSERT_RESULT_SUCCESS(task.StartTask());
    }
    {
        // キャンセルして逃げる
        nn::ntc::shim::CorrectionNetworkClockAsyncTask task(nn::os::EventClearMode_AutoClear, Mode);
        NNT_ASSERT_RESULT_SUCCESS(task.StartTask());
        task.Cancel();
        task.GetFinishNotificationEvent().Wait();
        auto result = task.GetResult();
        EXPECT_TRUE(nn::time::ResultCanceled::Includes(result) || result.IsSuccess()); // タイミングで成功もあり得る
        EXPECT_FALSE(task.IsProcessing());
    }
    {
        // インスタンスだけ作って逃げる
        nn::ntc::shim::CorrectionNetworkClockAsyncTask task(nn::os::EventClearMode_AutoClear, Mode);
    }

    {
        nn::ntc::shim::CorrectionNetworkClockAsyncTask task1(nn::os::EventClearMode_ManualClear, Mode);
        NNT_ASSERT_RESULT_SUCCESS(task1.StartTask());

        nn::ntc::shim::CorrectionNetworkClockAsyncTask task2(nn::os::EventClearMode_AutoClear, Mode);
        NNT_ASSERT_RESULT_SUCCESS(task2.StartTask());

        task1.GetFinishNotificationEvent().Wait();
        task2.GetFinishNotificationEvent().Wait();

        EXPECT_TRUE(task1.GetFinishNotificationEvent().TryWait()); // ManualClear なので Wait してもシグナル状態のはず
        EXPECT_FALSE(task2.GetFinishNotificationEvent().TryWait()); // AutoClear なので Wait 後は非シグナル状態のはず

#if defined(NN_BUILD_CONFIG_OS_WIN)
        // Win版だと CI 環境がネット接続不能な可能性があるので、 Result は問わない
#else
        // NX だと利用可能なネットワーク接続設定を注入してからテストされることを想定
        NNT_EXPECT_RESULT_SUCCESS(task1.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(task2.GetResult());
#endif

        if(NN_STATIC_CONDITION(Mode == nn::ntc::EnsureNetworkClockAvailabilityMode_GetServerTime))
        {
            if(task1.GetResult().IsSuccess())
            {
                const nn::time::PosixTime serverTime = task1.GetServerPosixTime();
                const nn::time::CalendarTime c = nn::time::ToCalendarTimeInUtc(serverTime);
                NN_SDK_LOG("ServerTime : %04d/%02d/%02d %02d:%02d:%02d (UTC)\n",
                    c.year, c.month, c.day, c.hour, c.minute, c.second);
            }
        }
    }
}

TEST(NtcApiTest, AsyncTaskTest_Default)
{
    AsyncTaskTest<nn::ntc::EnsureNetworkClockAvailabilityMode_Default>();
}

TEST(NtcApiTest, AsyncTaskTest_ForcibleDownload)
{
    AsyncTaskTest<nn::ntc::EnsureNetworkClockAvailabilityMode_ForcibleDownload>();
}

TEST(NtcApiTest, AsyncTaskTest_GetServerTime)
{
    AsyncTaskTest<nn::ntc::EnsureNetworkClockAvailabilityMode_GetServerTime>();
}

TEST(NtcApiTest, BlockCorrection)
{
    auto result = nn::ntc::shim::EnsureNetworkClockAvailability();
    if(result.IsSuccess())
    {
        NN_SDK_LOG("nn::timecrt::shim::EnsureNetworkClockAvailability was succeeded\n");
    }
    else
    {
        NN_SDK_LOG("nn::timecrt::shim::EnsureNetworkClockAvailability failed. (%08x, %03d-%04d)\n",
                result.GetInnerValueForDebug(),
                result.GetModule(), result.GetDescription());
    }

#if defined(NN_BUILD_CONFIG_OS_WIN)
    // Win版だと CI 環境がネット接続不能な可能性があるので、 Result は問わない
#else
    // NX だと利用可能なネットワーク接続設定を注入してからテストされることを想定
    NNT_EXPECT_RESULT_SUCCESS(result);
#endif

    DumpNetworkTime();
}

TEST(NtcApiTest, BlockCorrectionCancel)
{
    static volatile std::atomic<bool> isThreadFinishRequired(false);
    static NN_OS_ALIGNAS_THREAD_STACK char stackBuffer[1024 * 16];
    nn::os::ThreadType thread;
    NNT_ASSERT_RESULT_SUCCESS( nn::os::CreateThread(&thread,
        [](void*) {
            while(!isThreadFinishRequired)
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
                nn::ntc::shim::CancelEnsuringNetworkClockAvailability();
            }
        },
        nullptr, stackBuffer, sizeof(stackBuffer), 15) );

    nn::os::StartThread(&thread); // 1 msec 間隔でキャンセル発動
    auto result = nn::ntc::shim::EnsureNetworkClockAvailability();
    EXPECT_TRUE(nn::time::ResultCanceled::Includes(result) || result.IsSuccess()); // タイミングで成功もあり得る

    // スレッド破棄
    isThreadFinishRequired = true;
    nn::os::DestroyThread(&thread);
}

// nn::time:: 経由で補正 API が叩けるかどうか
TEST(NtcApiTest, FromTimeLibrary1)
{
    auto result = nn::time::EnsureNetworkClockAvailability();
    if(result.IsSuccess())
    {
        NN_SDK_LOG("nn::time::shim::EnsureNetworkClockAvailability was succeeded\n");
    }
    else
    {
        NN_SDK_LOG("nn::time::shim::EnsureNetworkClockAvailability failed. (%08x, %03d-%04d)\n",
                result.GetInnerValueForDebug(),
                result.GetModule(), result.GetDescription());
    }

#if defined(NN_BUILD_CONFIG_OS_WIN)
    // Win版だと CI 環境がネット接続不能な可能性があるので、 Result は問わない
#else
    // NX だと利用可能なネットワーク接続設定を注入してからテストされることを想定
    NNT_EXPECT_RESULT_SUCCESS(result);
#endif

    DumpNetworkTime();
}

TEST(NtcApiTest, FromTimeLibrary2)
{
    {
        nn::time::AsyncEnsureNetworkClockAvailabilityTask task;
        task.Initialize(nn::os::EventClearMode_AutoClear);

        NNT_EXPECT_RESULT_FAILURE(nn::time::ResultNotStarted, task.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(task.StartTask());
        task.GetFinishEvent().Wait();
        NNT_EXPECT_RESULT_SUCCESS(task.GetResult());
    }

    {
        nn::time::AsyncEnsureNetworkClockAvailabilityTask task;
        task.Initialize(nn::os::EventClearMode_AutoClear);

        NNT_EXPECT_RESULT_FAILURE(nn::time::ResultNotStarted, task.GetResult());
        NNT_ASSERT_RESULT_SUCCESS(task.StartTask());
        task.Cancel();
        auto result = task.GetResult();
        EXPECT_TRUE(nn::time::ResultCanceled::Includes(result) || result.IsSuccess()); // タイミングで成功もあり得る
    }

    {
        nn::time::AsyncEnsureNetworkClockAvailabilityTask task;
        // 何もせず破棄
    }
}

TEST(NtcApiTest, FinalizeOtherLib)
{
    nn::nifm::CancelNetworkRequest();

    NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize());
}

NN_PRAGMA_POP_WARNINGS
