﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/nsd/detail/nsd_DetailApi.h>
#include <nn/time.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nsd/nsd_ResultPrivate.h>
#include <nn/nsd/nsd_Version.h>
#include <nn/nsd/detail/config/nsd_Config.h>

using namespace nn::nsd::detail;

TEST(NsdUnitTest_DetailApi, InsertEnvironmentIdentifier)
{
    nn::nsd::Fqdn fqdn;
    const nn::nsd::EnvironmentIdentifier Env = {"dd2"}; // デフォルトで選択されないものを適当に
    {
        nn::nsd::Fqdn in= {"nintendo-%.com"};
        NNT_EXPECT_RESULT_SUCCESS( InsertEnvironmentIdentifier(&fqdn, in, Env) );
        EXPECT_STREQ("nintendo-dd2.com", fqdn.value);
    }
    {
        nn::nsd::Fqdn in= {"nintendo.%.com"};
        NNT_EXPECT_RESULT_SUCCESS( InsertEnvironmentIdentifier(&fqdn, in, Env) );
        EXPECT_STREQ("nintendo.dd2.com", fqdn.value);
    }
    {
        nn::nsd::Fqdn in= {"%.nintendo.com"};
        NNT_EXPECT_RESULT_SUCCESS( InsertEnvironmentIdentifier(&fqdn, in, Env) );
        EXPECT_STREQ("dd2.nintendo.com", fqdn.value);
    }
    {
        nn::nsd::Fqdn in = {"%23456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345"};
        NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultFqdnTooLong, InsertEnvironmentIdentifier(&fqdn, in, Env) );
    }
    {
        nn::nsd::Fqdn in = {"%3456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345"};
        NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultFqdnTooLong, InsertEnvironmentIdentifier(&fqdn, in, Env) );
    }
    {
        nn::nsd::Fqdn in = {"%23456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345"};
        in.value[in.Size - 1] = 'a'; // 終端文字を消す
        NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultFqdnIsNotNullTerminated, InsertEnvironmentIdentifier(&fqdn, in, Env) );
    }
    {
        // ギリギリOK
        nn::nsd::Fqdn in = {"%456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345"};
        NNT_EXPECT_RESULT_SUCCESS(InsertEnvironmentIdentifier(&fqdn, in, Env) );
    }
}

TEST(NsdUnitTest_DetailApi, HasBadChar)
{
    // OK
    const nn::nsd::Fqdn OkFqdns[] =
    {
        {"nintendo.com"},
        {"nInTendo.com"},
        {"nintendo.co.jp"},
        {"nintendo-lp1.co.jp"},
        {"nintendo.lp1.co.jp"},
        {"nintendo.12345.debug.co.jp"},
    };
    for(auto& fqdn : OkFqdns)
    {
        EXPECT_FALSE( HasBadChar(fqdn) ) << fqdn.value;
    }

    // NG
    const nn::nsd::Fqdn BadFqdns[] =
    {
        {"nintendo.!.com"},
        {"nintendo.\".com"},
        {"nintendo.#.com"},
        {"nintendo.$.com"},
        {"nintendo.%.com"},
        {"nintendo.&.com"},
        {"nintendo.'.com"},
        {"nintendo.(.com"},
        {"nintendo.).com"},
        {"nintendo.~.com"},
        {"nintendo._.com"},
        {"nintendo.+.com"},
        {"nintendo.*.com"},
        {"nintendo.`.com"},
        {"nintendo.{.com"},
        {"nintendo.}.com"},
        {"nintendo.?.com"},
        {"nintendo.<.com"},
        {"nintendo.>.com"},
        {"nintendo./.com"},
        {"nintendo.\\.com"},
        {"nintendo.^.com"},
        {"nintendo.@.com"},
    };
    for(auto& fqdn : BadFqdns)
    {
        EXPECT_TRUE( HasBadChar(fqdn) ) << fqdn.value;
    }
}

TEST(NsdUnitTest_DetailApi, HandleServerError)
{
    EXPECT_TRUE( nn::nsd::ResultMissingParameterInRequest::Includes(HandleServerError("0000")) );
    EXPECT_TRUE( nn::nsd::ResultInvalidParameterInRequest::Includes(HandleServerError("0001")) );
    EXPECT_TRUE( nn::nsd::ResultPasscodeNotFound::Includes(HandleServerError("0002")) );
    EXPECT_TRUE( nn::nsd::ResultForbidden::Includes(HandleServerError("0003")) );
    EXPECT_TRUE( nn::nsd::ResultMethodNotAllow::Includes( HandleServerError("0004")) );
    EXPECT_TRUE( nn::nsd::ResultApiVersionNotFound::Includes( HandleServerError("0005")) );
    EXPECT_TRUE( nn::nsd::ResultInternalServerError::Includes( HandleServerError("0500")) );
    EXPECT_TRUE( nn::nsd::ResultUnderMaintenance::Includes( HandleServerError("0503")) );

    EXPECT_TRUE( nn::nsd::ResultUnknownServerError::Includes( HandleServerError("1000")) ); // unknown error code.
    EXPECT_TRUE( nn::nsd::ResultUnknownServerError::Includes( HandleServerError("9999")) ); // unknown error code.
}

TEST(NsdUnitTest_DetailApi, GetDefaultSettings)
{
    nn::nsd::SaveData* pSaveData = new nn::nsd::SaveData;
    ASSERT_NE(nullptr, pSaveData );
    NN_UTIL_SCOPE_EXIT
    {
        delete pSaveData;
    };

    {
        nn::nsd::DeviceId deviceId = {{"123456789"}};
        GetDefaultSettings(pSaveData, deviceId, nn::nsd::EnvironmentIdentifierOfNotProductDevice);
        EXPECT_EQ(deviceId, pSaveData->deviceId);
        EXPECT_EQ(nn::nsd::EnvironmentIdentifierOfNotProductDevice, pSaveData->environmentIdentifier);
    }
    {
        nn::nsd::DeviceId deviceId = {{"9876543210"}};
        GetDefaultSettings(pSaveData, deviceId, nn::nsd::EnvironmentIdentifierOfProductDevice);
        EXPECT_EQ(deviceId, pSaveData->deviceId);
        EXPECT_EQ(nn::nsd::EnvironmentIdentifierOfProductDevice, pSaveData->environmentIdentifier);
    }
}

TEST(NsdUnitTest_DetailApi, IsInitialSaveData)
{
    nn::nsd::SaveData* pSaveData = new nn::nsd::SaveData;
    ASSERT_NE(nullptr, pSaveData );
    NN_UTIL_SCOPE_EXIT
    {
        delete pSaveData;
    };

    std::memset(pSaveData, 0, sizeof(nn::nsd::SaveData));
    EXPECT_TRUE(IsInitialSaveData(*pSaveData));

    pSaveData->version = 1;
    EXPECT_FALSE(IsInitialSaveData(*pSaveData));

    std::memset(pSaveData, 0, sizeof(nn::nsd::SaveData));
    EXPECT_TRUE(IsInitialSaveData(*pSaveData));
    pSaveData->applicationSettings.fqdnEntries[0].src.value[0] = 'a';
    EXPECT_FALSE(IsInitialSaveData(*pSaveData));

    std::memset(pSaveData, 0, sizeof(nn::nsd::SaveData));
    EXPECT_TRUE(IsInitialSaveData(*pSaveData));
    pSaveData->applicationSettings.fqdnEntries[0].dst.value[0] = 'a';
    EXPECT_FALSE(IsInitialSaveData(*pSaveData));

    std::memset(pSaveData, 0, sizeof(nn::nsd::SaveData));
    EXPECT_TRUE(IsInitialSaveData(*pSaveData));
    pSaveData->applicationSettings.fqdnEntries[nn::nsd::ApplicationSettings::FqdnEntryCountMax - 1].src.value[0] = 'a';
    EXPECT_FALSE(IsInitialSaveData(*pSaveData));

    std::memset(pSaveData, 0, sizeof(nn::nsd::SaveData));
    EXPECT_TRUE(IsInitialSaveData(*pSaveData));
    pSaveData->applicationSettings.fqdnEntries[nn::nsd::ApplicationSettings::FqdnEntryCountMax - 1].dst.value[0] = 'a';
    EXPECT_FALSE(IsInitialSaveData(*pSaveData));

    std::memset(pSaveData, 0, sizeof(nn::nsd::SaveData));
    EXPECT_TRUE(IsInitialSaveData(*pSaveData));
    pSaveData->environmentIdentifier.value[0] = 'a';
    EXPECT_FALSE(IsInitialSaveData(*pSaveData));

    std::memset(pSaveData, 0, sizeof(nn::nsd::SaveData));
    EXPECT_TRUE(IsInitialSaveData(*pSaveData));
    pSaveData->backboneSettings.fqdnEntries[nn::nsd::BackboneSettings::FqdnEntryCountMax - 1].src.value[0] = 'a';
    EXPECT_FALSE(IsInitialSaveData(*pSaveData));

    std::memset(pSaveData, 0, sizeof(nn::nsd::SaveData));
    EXPECT_TRUE(IsInitialSaveData(*pSaveData));
    pSaveData->backboneSettings.fqdnEntries[nn::nsd::BackboneSettings::FqdnEntryCountMax - 1].dst.value[0] = 'a';
    EXPECT_FALSE(IsInitialSaveData(*pSaveData));

    std::memset(pSaveData, 0, sizeof(nn::nsd::SaveData));
    EXPECT_TRUE(IsInitialSaveData(*pSaveData));
    pSaveData->backboneSettings.nasServiceSettings.entries[0].nasServiceName.value[0] = 'a';
    EXPECT_FALSE(IsInitialSaveData(*pSaveData));

    std::memset(pSaveData, 0, sizeof(nn::nsd::SaveData));
    EXPECT_TRUE(IsInitialSaveData(*pSaveData));
    pSaveData->backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.clientId = 1;
    EXPECT_FALSE(IsInitialSaveData(*pSaveData));

    std::memset(pSaveData, 0, sizeof(nn::nsd::SaveData));
    EXPECT_TRUE(IsInitialSaveData(*pSaveData));
    pSaveData->backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.redirectUri.value[0] = 'a';
    EXPECT_FALSE(IsInitialSaveData(*pSaveData));

    std::memset(pSaveData, 0, sizeof(nn::nsd::SaveData));
    EXPECT_TRUE(IsInitialSaveData(*pSaveData));
    pSaveData->backboneSettings.nasServiceSettings.nasApiServiceFqdn.value[0] = 'a';
    EXPECT_FALSE(IsInitialSaveData(*pSaveData));

    std::memset(pSaveData, 0, sizeof(nn::nsd::SaveData));
    EXPECT_TRUE(IsInitialSaveData(*pSaveData));
    pSaveData->backboneSettings.nasServiceSettings.nasRequestServiceFqdn.value[0] = 'a';
    EXPECT_FALSE(IsInitialSaveData(*pSaveData));
}

TEST(NsdUnitTest_DetailApi, VerifySaveData)
{
    nn::nsd::SaveData* pSaveData = new nn::nsd::SaveData;
    ASSERT_NE(nullptr, pSaveData );
    NN_UTIL_SCOPE_EXIT
    {
        delete pSaveData;
    };

    std::memset(pSaveData, 0, sizeof(nn::nsd::SaveData));

    nn::nsd::DeviceId deviceId = {{"123456789"}};

    {
        GetDefaultSettings(pSaveData, deviceId, nn::nsd::EnvironmentIdentifierOfNotProductDevice);
        EXPECT_FALSE(IsInitialSaveData(*pSaveData));
        NNT_ASSERT_RESULT_SUCCESS( VerifySaveData(*pSaveData, deviceId, false) );
    }

    { // version 空
        GetDefaultSettings(pSaveData, deviceId, nn::nsd::EnvironmentIdentifierOfNotProductDevice);
        pSaveData->version = 0;
        NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultEmptyValueExists , VerifySaveData(*pSaveData, deviceId, false) );
    }

    { // backboneSettings 空
        GetDefaultSettings(pSaveData, deviceId, nn::nsd::EnvironmentIdentifierOfNotProductDevice);
        std::memset(&pSaveData->backboneSettings, 0, sizeof(pSaveData->backboneSettings));
        NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultEmptyValueExists , VerifySaveData(*pSaveData, deviceId, false) );
    }

    { // backboneSettings 空
        GetDefaultSettings(pSaveData, deviceId, nn::nsd::EnvironmentIdentifierOfNotProductDevice);
        std::memset(&pSaveData->environmentIdentifier, 0, sizeof(pSaveData->environmentIdentifier));
        NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultEmptyValueExists , VerifySaveData(*pSaveData, deviceId, false) );
    }

    if(NN_STATIC_CONDITION(nn::nsd::detail::config::IsDeviceIdCheckEnabled))
    { // deviceId 空
        GetDefaultSettings(pSaveData, deviceId, nn::nsd::EnvironmentIdentifierOfNotProductDevice);
        std::memset(&pSaveData->deviceId, 0, sizeof(pSaveData->deviceId));
        NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultEmptyValueExists , VerifySaveData(*pSaveData, deviceId, true) );
    }

    if(NN_STATIC_CONDITION(nn::nsd::detail::config::IsDeviceIdCheckEnabled))
    { // deviceId 不一致
        GetDefaultSettings(pSaveData, deviceId, nn::nsd::EnvironmentIdentifierOfNotProductDevice);
        pSaveData->deviceId.value[0] = 'a';
        NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultEmptyValueExists , VerifySaveData(*pSaveData, deviceId, true) );
    }
}

TEST(NsdUnitTest_DetailApi, Resolve)
{
    nn::nsd::SaveData* pSaveData = new nn::nsd::SaveData;
    ASSERT_NE(nullptr, pSaveData );
    NN_UTIL_SCOPE_EXIT
    {
        delete pSaveData;
    };

    nn::nsd::DeviceId deviceId = {{"123456789"}};

    GetDefaultSettings(pSaveData, deviceId, nn::nsd::EnvironmentIdentifierOfNotProductDevice);
    NNT_ASSERT_RESULT_SUCCESS( VerifySaveData(*pSaveData, deviceId, false) );

    NNT_ASSERT_RESULT_SUCCESS(nn::time::Initialize());
    NN_UTIL_SCOPE_EXIT
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::time::Finalize());
    };

    nn::util::Strlcpy(pSaveData->backboneSettings.fqdnEntries[0].src.value, "nx-baas.lp1.com", nn::nsd::Fqdn::Size);
    nn::util::Strlcpy(pSaveData->backboneSettings.fqdnEntries[0].dst.value, "nx-baas.dd1.com", nn::nsd::Fqdn::Size);
    {
        nn::nsd::Fqdn outFqdn = {};
        nn::nsd::Fqdn in = {{"nx-baas.lp1.com"}};
        nn::time::PosixTime current = {0};
        NNT_EXPECT_RESULT_SUCCESS(Resolve(&outFqdn, in, *pSaveData, current));
        EXPECT_STREQ("nx-baas.dd1.com", outFqdn.value);
    }

    nn::util::Strlcpy(pSaveData->backboneSettings.fqdnEntries[0].dst.value, "nx-baas.dd2.com", nn::nsd::Fqdn::Size);
    {
        nn::nsd::Fqdn outFqdn = {};
        nn::nsd::Fqdn in = {{"nx-baas.lp1.com"}};
        nn::time::PosixTime current = {0};
        NNT_EXPECT_RESULT_SUCCESS(Resolve(&outFqdn, in, *pSaveData, current));
        EXPECT_STREQ("nx-baas.dd2.com", outFqdn.value);
    }

    pSaveData->expireUnixTime = 1;
    nn::util::Strlcpy(pSaveData->applicationSettings.fqdnEntries[0].src.value, "app1.lp1.com", nn::nsd::Fqdn::Size);
    nn::util::Strlcpy(pSaveData->applicationSettings.fqdnEntries[0].dst.value, "app1.dd1.com", nn::nsd::Fqdn::Size);
    {
        nn::nsd::Fqdn outFqdn = {};
        const nn::nsd::Fqdn in = {{"app1.lp1.com"}};
        nn::time::PosixTime current = {0}; // 期限内
        NNT_EXPECT_RESULT_SUCCESS(Resolve(&outFqdn, in, *pSaveData, current));
        EXPECT_STREQ("app1.dd1.com", outFqdn.value);

        if(NN_STATIC_CONDITION(nn::nsd::detail::config::IsSettingsExpirationEnabled))
        {
            current.value = 1; // 期限切れ
            NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultSettingExpiredDetected, Resolve(&outFqdn, in, *pSaveData, current));
        }
    }

    nn::util::Strlcpy(pSaveData->applicationSettings.fqdnEntries[1].src.value, "app2.lp1.com", nn::nsd::Fqdn::Size);
    nn::util::Strlcpy(pSaveData->applicationSettings.fqdnEntries[1].dst.value, "app2.dd1.com", nn::nsd::Fqdn::Size);
    {
        nn::nsd::Fqdn outFqdn = {};
        const nn::nsd::Fqdn in = {{"app2.lp1.com"}};
        nn::time::PosixTime current = {0}; // 期限内
        NNT_EXPECT_RESULT_SUCCESS(Resolve(&outFqdn, in, *pSaveData, current));
        EXPECT_STREQ("app2.dd1.com", outFqdn.value);

        if(NN_STATIC_CONDITION(nn::nsd::detail::config::IsSettingsExpirationEnabled))
        {
            current.value = 1; // 期限切れ
            NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultSettingExpiredDetected, Resolve(&outFqdn, in, *pSaveData, current));
        }
    }

    {
        nn::nsd::Fqdn outFqdn = {};
        nn::nsd::Fqdn in = {{"app2.lp1.com"}};
        nn::time::PosixTime current = {0};
        pSaveData->version = nn::nsd::Version + 1; // 設定バージョンが大きい
        NNT_EXPECT_RESULT_SUCCESS(Resolve(&outFqdn, in, *pSaveData, current));
        EXPECT_STREQ("app2.dd1.com", outFqdn.value);
    }
    {
        nn::nsd::Fqdn outFqdn = {};
        nn::nsd::Fqdn in = {{"app2.lp1.com"}};
        nn::time::PosixTime current = {0};
        pSaveData->version = nn::nsd::Version + 100; // 設定バージョンが大きい
        NNT_EXPECT_RESULT_SUCCESS(Resolve(&outFqdn, in, *pSaveData, current));
        EXPECT_STREQ("app2.dd1.com", outFqdn.value);
    }
    {
        nn::nsd::Fqdn outFqdn = {};
        nn::nsd::Fqdn in = {{"app2.lp1.com"}};
        nn::time::PosixTime current = {0};
        pSaveData->version = 0; // 設定バージョンが小さい
        NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultInvalidSettingVersion, Resolve(&outFqdn, in, *pSaveData, current));
    }
}

TEST(NsdUnitTest_DetailApi, GetAccountServiceParameter)
{
    nn::nsd::SaveData* pSaveData = new nn::nsd::SaveData;
    ASSERT_NE(nullptr, pSaveData );
    NN_UTIL_SCOPE_EXIT
    {
        delete pSaveData;
    };

    nn::nsd::DeviceId deviceId = {{"123456789"}};
    GetDefaultSettings(pSaveData, deviceId, nn::nsd::EnvironmentIdentifierOfNotProductDevice);
    pSaveData->expireUnixTime = 1;

    // nn::nsd::NasServiceNameOfNxAccount 向け設定を入れる
    nn::util::Strlcpy(pSaveData->backboneSettings.nasServiceSettings.nasRequestServiceFqdn.value, "nas.request.com", nn::nsd::Fqdn::Size);
    nn::util::Strlcpy(pSaveData->backboneSettings.nasServiceSettings.nasApiServiceFqdn.value, "nas.api.com", nn::nsd::Fqdn::Size);
    pSaveData->backboneSettings.nasServiceSettings.entries[0].nasServiceName = nn::nsd::NasServiceNameOfNxAccount;
    pSaveData->backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.clientId = 0x0123456789ULL;
    nn::util::Strlcpy(pSaveData->backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.redirectUri.value, "https://nx.account", nn::nsd::NasServiceSetting::RedirectUri::Size);

    {
        nn::nsd::Fqdn requestFqdn;
        nn::time::PosixTime current = {0};
        NNT_EXPECT_RESULT_SUCCESS( GetNasRequestFqdn(&requestFqdn, *pSaveData, current) );
        EXPECT_STREQ("nas.request.com", requestFqdn.value);

        // dd1 の NAS FQDN は Resolve で置換される
        const nn::nsd::Fqdn Dd1NasRequestFqdn = {{"e97b8a9d672e4ce4845ec6947cd66ef6-sb.accounts.nintendo.com"}};
        NNT_EXPECT_RESULT_SUCCESS(Resolve(&requestFqdn, Dd1NasRequestFqdn,  *pSaveData, current));
        EXPECT_STREQ("nas.request.com", requestFqdn.value);

        if(NN_STATIC_CONDITION(nn::nsd::detail::config::IsSettingsExpirationEnabled))
        {
            current.value = 1; // 期限切れ
            NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultSettingExpiredDetected, GetNasRequestFqdn(&requestFqdn, *pSaveData, current) );

            // dd1 の NAS FQDN は Resolve で置換される
            NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultSettingExpiredDetected, Resolve(&requestFqdn, Dd1NasRequestFqdn,  *pSaveData, current));
        }
    }

    {
        nn::nsd::Fqdn apiFqdn;
        nn::time::PosixTime current = {0};
        NNT_EXPECT_RESULT_SUCCESS( GetNasApiFqdn(&apiFqdn, *pSaveData, current) );
        EXPECT_STREQ("nas.api.com", apiFqdn.value);

        // dd1 の NAS FQDN は Resolve で置換される
        const nn::nsd::Fqdn Dd1NasApiFqdn = {{"e97b8a9d672e4ce4845ec6947cd66ef6-sb-api.accounts.nintendo.com"}};
        NNT_EXPECT_RESULT_SUCCESS(Resolve(&apiFqdn, Dd1NasApiFqdn,  *pSaveData, current));
        EXPECT_STREQ("nas.api.com", apiFqdn.value);

        if(NN_STATIC_CONDITION(nn::nsd::detail::config::IsSettingsExpirationEnabled))
        {
            current.value = 1; // 期限切れ
            NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultSettingExpiredDetected, GetNasApiFqdn(&apiFqdn, *pSaveData, current) );

            // dd1 の NAS FQDN は Resolve で置換される
            NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultSettingExpiredDetected, Resolve(&apiFqdn, Dd1NasApiFqdn,  *pSaveData, current));
        }
    }

    {
        nn::nsd::NasServiceSetting sreviceParameter;
        nn::time::PosixTime current = {0};
        NNT_EXPECT_RESULT_SUCCESS( GetNasServiceSetting(
            &sreviceParameter,
            nn::nsd::NasServiceNameOfNxAccount,
            *pSaveData,
            current));
        EXPECT_EQ(0x0123456789ULL, sreviceParameter.clientId);
        EXPECT_STREQ("https://nx.account", sreviceParameter.redirectUri.value);

        if(NN_STATIC_CONDITION(nn::nsd::detail::config::IsSettingsExpirationEnabled))
        {
            current.value = 1; // 期限切れ
            NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultSettingExpiredDetected , GetNasServiceSetting(
                &sreviceParameter,
                nn::nsd::NasServiceNameOfNxAccount,
                *pSaveData,
                current));
        }
    }

    {
        nn::nsd::NasServiceName invalidServiceName = {{"invalid"}}; // 存在しないサービス
        nn::nsd::NasServiceSetting sreviceParameter;
        nn::time::PosixTime current = {0};
        NNT_EXPECT_RESULT_FAILURE(
            nn::nsd::ResultNotFound,
            GetNasServiceSetting(
                &sreviceParameter,
                invalidServiceName ,
                *pSaveData,
                current));
    }

    // 2つ目のサービスがある場合のテスト
    const nn::nsd::NasServiceName nasServiceNameOfNx2Account = {{"TestService"}};
    pSaveData->backboneSettings.nasServiceSettings.entries[1].nasServiceName = nasServiceNameOfNx2Account;
    pSaveData->backboneSettings.nasServiceSettings.entries[1].nasServiceSetting.clientId = 0x9876543210ULL;
    nn::util::Strlcpy(pSaveData->backboneSettings.nasServiceSettings.entries[1].nasServiceSetting.redirectUri.value, "nintendo://test.com", nn::nsd::NasServiceSetting::RedirectUri::Size);

    {
        // 1つめの NasServiceNameOfNxAccount はもちろん存在する
        nn::nsd::NasServiceSetting sreviceParameter;
        nn::time::PosixTime current = {0};
        NNT_EXPECT_RESULT_SUCCESS( GetNasServiceSetting(
            &sreviceParameter,
            nn::nsd::NasServiceNameOfNxAccount,
            *pSaveData,
            current));
        EXPECT_EQ(0x0123456789ULL, sreviceParameter.clientId);
        EXPECT_STREQ("https://nx.account", sreviceParameter.redirectUri.value);

        if(NN_STATIC_CONDITION(nn::nsd::detail::config::IsSettingsExpirationEnabled))
        {
            current.value = 1; // 期限切れ
            NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultSettingExpiredDetected, GetNasServiceSetting(
                &sreviceParameter,
                nn::nsd::NasServiceNameOfNxAccount,
                *pSaveData,
                current));
        }
    }
    {
        // 2つめの Nx2Account も存在する
        nn::nsd::NasServiceSetting sreviceParameter;
        nn::time::PosixTime current = {0};
        NNT_EXPECT_RESULT_SUCCESS( GetNasServiceSetting(
            &sreviceParameter,
            nasServiceNameOfNx2Account,
            *pSaveData,
            current));
        EXPECT_EQ(0x9876543210ULL, sreviceParameter.clientId);
        EXPECT_STREQ("nintendo://test.com", sreviceParameter.redirectUri.value);

        if(NN_STATIC_CONDITION(nn::nsd::detail::config::IsSettingsExpirationEnabled))
        {
            current.value = 1; // 期限切れ
            NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultSettingExpiredDetected, GetNasServiceSetting(
                &sreviceParameter,
                nasServiceNameOfNx2Account,
                *pSaveData,
                current));
        }
    }
} // NOLINT(impl/function_size)
