﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ns/ns_UserResourceManagementApi.h>

#include <memory>

#include <nnt/nntest.h>

#include <nn/nn_Log.h>
#include <nn/account/account_Types.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiPrivate.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SaveData.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_UserAccountSystemSaveData.h>
#include <nn/fs/fs_Utility.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/ns/ns_ApiForDfc.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/ovln/ovln_ForDevelop.h>
#include <nn/ovln/ovln_Sender.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Uuid.h>

#define NNT_NS_ASSERT_RESULT_SUCCESS(expression) \
    do { \
        nn::Result _r = (expression); \
        if (!_r.IsSuccess()) \
        { \
            NN_LOG("%s(%d): Failure: " #expression " -> %03u-%04u (0x%08lx)\n", __FILE__, __LINE__, _r.GetModule(), _r.GetDescription(), _r.GetInnerValueForDebug()); \
            ASSERT_TRUE(_r.IsSuccess()); \
        } \
    } while (NN_STATIC_CONDITION(false))

#define NNT_NS_EXPECT_RESULT_SUCCESS(expression) \
    do { \
        nn::Result _r = (expression); \
        if (!_r.IsSuccess()) \
        { \
            NN_LOG("%s(%d): Failure: " #expression " -> %03u-%04u (0x%08lx)\n", __FILE__, __LINE__, _r.GetModule(), _r.GetDescription(), _r.GetInnerValueForDebug()); \
            EXPECT_TRUE(_r.IsSuccess()); \
        } \
    } while (NN_STATIC_CONDITION(false))

namespace {

void GenerateUserId(nn::account::Uid* list, int count) NN_NOEXCEPT
{
    NN_SDK_ASSERT(count > 0);
    for (int i = 0; i < count; ++i)
    {
        auto uuid = nn::util::GenerateUuid();
        std::memcpy(list + i, &uuid, sizeof(uuid));
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
    }
}

// テスト用 Application ID
const int ApplicationCount = 0x10;
const nn::ncm::ApplicationId ApplicationId0 = {0x0100000000003120ull};

// システムセーブデータ ID
const int SystemSaveDataCount = 0x10 - 1;
const nn::fs::SystemSaveDataId SystemSaveDataId0 = 0x8000000000000001;

uint32_t SetupEnvironmentForTest(const nn::account::Uid* list, int count) NN_NOEXCEPT
{
    uint32_t created = 0;
    for (auto i = 0; i < ApplicationCount; ++ i)
    {
        nn::ncm::ApplicationId appId = {ApplicationId0.value + i};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CreateDeviceSaveData(appId, appId.value, 0x10000, 0x10000, 0));
        ++created;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CreateBcatSaveData(appId, 0x10000));
        ++created;
        for (auto j = 0; j < count; ++ j)
        {
            auto& uid = list[j];
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CreateSaveData(
                appId, nn::fs::ConvertAccountUidToFsUserId(uid), appId.value, 0x10000, 0x10000, 0));
            ++created;
        }
    }
    for (auto i = 0; i < SystemSaveDataCount; ++ i)
    {
        auto sysSaveId = SystemSaveDataId0 + i;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(sysSaveId, 0x10000, 0x10000, 0));
        ++created;
        for (auto j = 0; j < count; ++ j)
        {
            auto& uid = list[j];
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(sysSaveId, nn::fs::ConvertAccountUidToFsUserId(uid), 0x10000, 0x10000, 0));
            ++created;
        }
    }
    NN_LOG("[nnt::ns] SetupEnvironmentForTest: created=%d\n", created);
    return created;
}
void CleanupEnvirionmentForTest() NN_NOEXCEPT
{
    const nn::fs::SaveDataSpaceId TargetSpaceIds[] = {
        nn::fs::SaveDataSpaceId::User, nn::fs::SaveDataSpaceId::System};

    uint32_t removed = 0;
    uint32_t kept = 0;

    auto filter0 = [](nn::ncm::ApplicationId appId) -> bool {
        return true
            && (appId.value >= ApplicationId0.value)
            && (appId.value < ApplicationId0.value + ApplicationCount);
    };
    auto filter1 = [](nn::fs::SystemSaveDataId sysSaveId) -> bool {
        return true
            && (sysSaveId >= SystemSaveDataId0)
            && (sysSaveId < SystemSaveDataId0 + SystemSaveDataCount);
    };

    for (auto spaceId : TargetSpaceIds)
    {
        std::unique_ptr<nn::fs::SaveDataIterator> iter;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, spaceId));

        while (NN_STATIC_CONDITION(true))
        {
            int64_t count;
            nn::fs::SaveDataInfo info;
            NN_ABORT_UNLESS_RESULT_SUCCESS(iter->ReadSaveDataInfo(&count, &info, 1));
            if (count == 0)
            {
                break;
            }

            if (false
                || filter0(info.applicationId)
                || filter1(info.systemSaveDataId))
            {
                NN_LOG("[SaveData]\n");
                NN_LOG(" - SaveDataId: %016llx\n", info.saveDataId);
                NN_LOG(" - SpaceId: %d\n", info.saveDataSpaceId);
                NN_LOG(" - Type: %d\n", info.saveDataType);
                NN_LOG(" - SystemSaveDataId: %016llx\n", info.systemSaveDataId);
                NN_LOG(" - ApplicationId: %016llx\n", info.applicationId);
                NN_LOG(" - UserId: %016llx_%016llx\n", info.saveDataUserId._data[0], info.saveDataUserId._data[1]);

                auto r = nn::fs::DeleteSaveData(info.saveDataId);
                if (!(r.IsSuccess() || nn::fs::ResultTargetNotFound::Includes(r)))
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(r);
                }
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, spaceId));
                ++removed;
            }
            else
            {
                ++kept;
            }
        }
    }
    NN_LOG("[nnt::ns] CleanupEnvironmentForTest: removed=%d, kept=%d\n", removed, kept);
}

void PrintProgress(const nn::ns::ProgressMonitorForDeleteUserSaveDataAll& progress, const nn::ns::UserSaveDataStatistics& total) NN_NOEXCEPT
{
    auto current = progress.GetStatistics();
    auto elapsed = progress.GetElapsedTime();
    if (progress.IsSystemSaveData())
    {
        NN_LOG(
            "%8d sec : [SYSTEM (%u of %u)] %llu bytes of %llu bytes\n",
            elapsed.GetSeconds(), current.count, total.count, current.sizeInBytes, total.sizeInBytes);
    }
    else
    {
        nn::ApplicationId cursor;
        if (progress.TryGetCurrentApplicationId(&cursor))
        {
            NN_LOG(
                "%8d sec : [APP (%u of %u)] %llu bytes of %llu bytes, ApplicationId=%016llx\n",
                elapsed.GetSeconds(), current.count, total.count, current.sizeInBytes, total.sizeInBytes,
                cursor.value);
        }
        else
        {
            NN_LOG(
                "%8d sec : [USER (%u of %u)] %llu bytes of %llu bytes\n",
                elapsed.GetSeconds(), current.count, total.count, current.sizeInBytes, total.sizeInBytes);
        }
    }
}

} // ~namespace <anonymous>

class TestNsUrmBasic
    : public testing::Test
{
protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::ns::InitializeDependenciesForDfc();
#endif
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        CleanupEnvirionmentForTest();
#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::ns::FinalizeDependenciesForDfc();
#endif
    }

    static void SetUpTestCase() NN_NOEXCEPT
    {
        CleanupEnvirionmentForTest();

// noclean でシステムセーブデータが消えない間のワークアラウンド
#if defined( NN_BUILD_CONFIG_OS_WIN)
        nn::fs::DeleteSaveData(0x8000000000000041ull);
#endif
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
    }
};

TEST_F(TestNsUrmBasic, DeleteUserSaveDataAll)
{
    const int UserCount = 3;
    nn::account::Uid users[UserCount];
    GenerateUserId(users, UserCount);

    auto created = static_cast<int32_t>(SetupEnvironmentForTest(users, UserCount));
    NN_ABORT_UNLESS_EQUAL((UserCount + 2) * ApplicationCount + (UserCount + 1) * SystemSaveDataCount, created);

    // test
    for (auto& u : users)
    {
        nn::ns::UserSaveDataStatistics stat;
        NNT_NS_ASSERT_RESULT_SUCCESS(nn::ns::CalculateUserSaveDataStatistics(&stat, u));
        ASSERT_EQ(ApplicationCount + SystemSaveDataCount, stat.count);

        nn::ns::ProgressMonitorForDeleteUserSaveDataAll progress;
        NNT_NS_ASSERT_RESULT_SUCCESS(nn::ns::DeleteUserSaveDataAll(&progress, u));

        nn::os::MultiWaitType mw;
        nn::os::InitializeMultiWait(&mw);

        nn::os::MultiWaitHolderType holders[2];
        nn::os::TimerEvent timer(nn::os::EventClearMode_ManualClear);
        timer.StartPeriodic(nn::TimeSpan::FromMilliSeconds(500), nn::TimeSpan::FromMilliSeconds(500));
        nn::os::InitializeMultiWaitHolder(&holders[0], timer.GetBase());
        nn::os::LinkMultiWaitHolder(&mw, &holders[0]);

        nn::os::SystemEvent e;
        progress.GetSystemEvent(&e);
        nn::os::InitializeMultiWaitHolder(&holders[1], e.GetBase());
        nn::os::LinkMultiWaitHolder(&mw, &holders[1]);

        auto prev = progress.GetStatistics();
        while (NN_STATIC_CONDITION(true))
        {
            auto* pMwh = nn::os::WaitAny(&mw);
            progress.Update();

            if (pMwh == &holders[1])
            {
                // 終了した
                e.Clear();
                PrintProgress(progress, stat);
                break;
            }

            NN_ABORT_UNLESS(pMwh == &holders[0]);
            timer.Clear();
            auto current = progress.GetStatistics();
            // 前回を下回らない
            ASSERT_GE(current.count, prev.count);
            ASSERT_GE(current.sizeInBytes, prev.sizeInBytes);
            // 全体を超えない
            ASSERT_GE(stat.count, current.count);
            ASSERT_GE(stat.sizeInBytes, current.sizeInBytes);

            if (stat.sizeInBytes != prev.sizeInBytes)
            {
                PrintProgress(progress, stat);
            }
        }
        ASSERT_TRUE(progress.IsFinished());
        NNT_NS_ASSERT_RESULT_SUCCESS(progress.GetResult());

        NNT_NS_ASSERT_RESULT_SUCCESS(nn::ns::CalculateUserSaveDataStatistics(&stat, u));
        ASSERT_EQ(0, stat.count);
    }
}

TEST_F(TestNsUrmBasic, DeleteUserSystemSaveData)
{
    const int UserCount = 1;
    nn::account::Uid users[UserCount];
    GenerateUserId(users, UserCount);

    auto created = static_cast<int32_t>(SetupEnvironmentForTest(users, UserCount));
    NN_ABORT_UNLESS_EQUAL((UserCount + 2) * ApplicationCount + (UserCount + 1) * SystemSaveDataCount, created);

    // test
    for (auto& u : users)
    {
        nn::ns::UserSaveDataStatistics stat;
        NNT_NS_ASSERT_RESULT_SUCCESS(nn::ns::CalculateUserSaveDataStatistics(&stat, u));
        ASSERT_EQ(ApplicationCount + SystemSaveDataCount, stat.count);

        for (auto i = 0; i < SystemSaveDataCount; ++ i)
        {
            NNT_NS_ASSERT_RESULT_SUCCESS(nn::ns::DeleteUserSystemSaveData(u, SystemSaveDataId0 + i));

            NNT_NS_ASSERT_RESULT_SUCCESS(nn::ns::CalculateUserSaveDataStatistics(&stat, u));
            ASSERT_EQ(ApplicationCount + SystemSaveDataCount - (i + 1), stat.count);
        }

        // ユーザーデータの削除
        nn::ns::ProgressMonitorForDeleteUserSaveDataAll progress;
        NNT_NS_ASSERT_RESULT_SUCCESS(nn::ns::DeleteUserSaveDataAll(&progress, u));
        nn::os::SystemEvent e;
        progress.GetSystemEvent(&e);
        e.Wait();
        e.Clear();
        ASSERT_TRUE(progress.IsFinished());
        NNT_NS_ASSERT_RESULT_SUCCESS(progress.GetResult());

        NNT_NS_ASSERT_RESULT_SUCCESS(nn::ns::CalculateUserSaveDataStatistics(&stat, u));
        ASSERT_EQ(0, stat.count);
    }
}

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

class TestNetworkServiceAccountUnregistration
    : public testing::Test
{
private:
    nn::account::Uid m_Uid;

protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::ns::InitializeDependenciesForDfc();
#endif
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::InitializeForSystem());
        nn::account::InitializeForAdministrator();

        // ユーザーアカウント
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::BeginUserRegistration(&m_Uid));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::CompleteUserRegistration(m_Uid));

        // インターネット
        nn::nifm::InitializeSystem();
        nn::nifm::SubmitNetworkRequestAndWait();
        NN_ABORT_UNLESS(nn::nifm::IsNetworkAvailable());

        // npns 初期化
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::Suspend());
        NN_UTIL_SCOPE_EXIT
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::Resume());

        };
        char jid[nn::npns::JidLength + 1] = {};
        if (nn::npns::GetJid(jid, sizeof(jid)).IsFailure())
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::CreateJid());
        }
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        // インターネット
        nn::nifm::CancelNetworkRequest();

        // ユーザーアカウント
        nn::account::NetworkServiceAccountAdministrator admin;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, m_Uid));
        bool isRegistered;
        NN_ABORT_UNLESS_RESULT_SUCCESS(admin.IsNetworkServiceAccountRegistered(&isRegistered));
        if (isRegistered)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(admin.DeleteRegistrationInfoLocally());
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::DeleteUser(m_Uid));

        nn::npns::FinalizeForSystem();
#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::ns::FinalizeDependenciesForDfc();
#endif
    }

    static void SetUpTestCase() NN_NOEXCEPT
    {
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
    }

    nn::Result Register() NN_NOEXCEPT
    {
        nn::account::NetworkServiceAccountAdministrator admin;
        NN_RESULT_DO(nn::account::GetNetworkServiceAccountAdministrator(&admin, m_Uid));
        nn::account::AsyncContext ctx;
        NN_RESULT_DO(admin.RegisterAsync(&ctx));
        nn::os::SystemEvent e;
        NN_ABORT_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
        e.Wait();
        NN_RESULT_DO(ctx.GetResult());

        NN_RESULT_SUCCESS;
    }

    nn::account::Uid GetUid() const NN_NOEXCEPT
    {
        return m_Uid;
    }
};

template <size_t N>
bool Contains(const nn::account::Uid& uid, const nn::fs::SystemSaveDataId (&sysSaveIds)[N]) NN_NOEXCEPT
{
    auto fsUid = nn::fs::ConvertAccountUidToFsUserId(uid);

    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    NN_RESULT_DO(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::System));

    bool found[N] = {};
    while (NN_STATIC_CONDITION(true))
    {
        int64_t count;
        nn::fs::SaveDataInfo info;
        NN_RESULT_DO(iter->ReadSaveDataInfo(&count, &info, 1));
        if (count == 0)
        {
            break;
        }

        if (!(true
            && info.saveDataType == nn::fs::SaveDataType::System
            && info.saveDataUserId == fsUid))
        {
            continue;
        }

        for (auto i = 0; i < N; ++i)
        {
            if (info.systemSaveDataId == sysSaveIds[i])
            {
                NN_LOG("[ContainsNot] SystemSavedata found: %016llx\n", sysSaveIds[i]);
                found[i] = true;
            }
        }
    }

    for (auto i = 0; i < N; ++i)
    {
        if (!found[i])
        {
            NN_LOG("[ContainsNot] SystemSavedata not found: %016llx\n", sysSaveIds[i]);
            return false;
        }
    }
    return true;
}
template <size_t N>
bool ContainsNot(const nn::account::Uid& uid, const nn::fs::SystemSaveDataId (&sysSaveIds)[N]) NN_NOEXCEPT
{
    auto fsUid = nn::fs::ConvertAccountUidToFsUserId(uid);

    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    NN_RESULT_DO(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::System));

    while (NN_STATIC_CONDITION(true))
    {
        int64_t count;
        nn::fs::SaveDataInfo info;
        NN_RESULT_DO(iter->ReadSaveDataInfo(&count, &info, 1));
        if (count == 0)
        {
            break;
        }

        if (!(true
            && info.saveDataType == nn::fs::SaveDataType::System
            && info.saveDataUserId == fsUid))
        {
            continue;
        }

        for (auto i = 0; i < N; ++i)
        {
            if (info.systemSaveDataId == sysSaveIds[i])
            {
                NN_LOG("[ContainsNot] SystemSavedata still exists: %016llx\n", sysSaveIds[i]);
                return false;
            }
        }
    }
    return true;
}

TEST_F(TestNetworkServiceAccountUnregistration, Basic)
{
    auto uid = this->GetUid();

    // NSA が登録されていない場合
    auto r = nn::ns::UnregisterNetworkServiceAccount(uid);
    NN_LOG("nn::ns::UnregisterNetworkServiceAccount(uid) returned: %03d-%04d\n", r.GetModule(), r.GetDescription());
    ASSERT_TRUE(nn::account::ResultNetworkServiceAccountRegistrationRequired::Includes(r));

    // NSA と、関連する情報の登録
    NN_ABORT_UNLESS_RESULT_SUCCESS(this->Register());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::Suspend());
    NN_UTIL_SCOPE_EXIT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::Resume());

    };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::UploadTokenToBaaS(uid));

    // friends が関連ストレージを用意するまで 5 秒待つ
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    // 開始時の セーブデータ の個数を数えておく
    nn::ns::UserSaveDataStatistics stat0;
    NNT_NS_ASSERT_RESULT_SUCCESS(nn::ns::CalculateUserSaveDataStatistics(&stat0, uid));

    // システムセーブデータの作成
    static const nn::fs::SystemSaveDataId SysSaveIdShop = 0x8000000000001061ull;
    static const nn::fs::SystemSaveDataId SysSaveIdLoginShare = 0x8000000000001091ull;
    static const nn::fs::SystemSaveDataId SysSaveIdPlayerSelect = 0x80000000000010b0ull;

    nn::fs::SystemSaveDataId sysSaveIds[] = {SysSaveIdShop, SysSaveIdLoginShare, SysSaveIdPlayerSelect};
    for (const auto& id : sysSaveIds)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(id, nn::fs::ConvertAccountUidToFsUserId(uid), 0x10000, 0x10000, 0));
    }

    ASSERT_TRUE(Contains(uid, sysSaveIds));

    // NSA の登録解除
    NNT_NS_EXPECT_RESULT_SUCCESS(nn::ns::UnregisterNetworkServiceAccount(uid));

    // 必要なシステムセーブデータが削除されたことを確認
    nn::fs::SystemSaveDataId deleted[] = {SysSaveIdShop, SysSaveIdLoginShare};
    EXPECT_TRUE(ContainsNot(uid, deleted));
    nn::fs::SystemSaveDataId lasting[] = {SysSaveIdPlayerSelect};
    EXPECT_TRUE(Contains(uid, lasting));

    NNT_NS_ASSERT_RESULT_SUCCESS(nn::ns::DeleteUserSystemSaveData(uid, SysSaveIdPlayerSelect));
    ASSERT_TRUE(ContainsNot(uid, sysSaveIds));
}

#endif // NN_BUILD_CONFIG_OS_HORIZON
