﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>
#include <random>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>

#include <nn/ns/ns_Result.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/ovln/ovln_ForDevelop.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include "libraries/testNs_MountHost.h"
#include "libraries/testNs_CommandOptions.h"
#include "testNs_NcmStorageVerify.h"

#define LOG_OUT( ... )  NN_LOG( "[NsSystemUpdate] " __VA_ARGS__ )

namespace {

#if defined( NN_BUILD_CONFIG_OS_WIN )

    //!--------------------------------------------------------------------------------------
    //! @brief The implementation in C11 rules for aligned memory allocations.@n
    //!        Memory allocator with alignment for start address of rules on the windows operating system.
    //!--------------------------------------------------------------------------------------
    NN_FORCEINLINE static void* aligned_alloc( size_t beginAlignment, size_t size ) NN_NOEXCEPT
    {
        return ::_aligned_malloc( size, beginAlignment );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief The implementation for aligned memory release rules on the windows operating system.
    //!--------------------------------------------------------------------------------------
    NN_FORCEINLINE static void aligned_free( void* address ) NN_NOEXCEPT
    {
        ::_aligned_free( address );
    }

#else   // NX, Cafe, and C++11 supported platforms...

    //!--------------------------------------------------------------------------------------
    //! @brief The implementation for aligned memory release that has been imitated on the rules of the windows operating system.
    //!--------------------------------------------------------------------------------------
    NN_FORCEINLINE static void aligned_free( void* address ) NN_NOEXCEPT
    {
        ::free( address );
    }

#endif

    //!--------------------------------------------------------------------------------------
    //! 配列要素数算出.
    //!--------------------------------------------------------------------------------------
    template< typename TYPE, size_t SIZE >
    inline static unsigned CountOf( const TYPE( &)[ SIZE ] ) {
        return SIZE;
    }

    //!--------------------------------------------------------------------------------------
    //! Result出力.
    //!--------------------------------------------------------------------------------------
    static void LogoutResultValue( const nn::Result& result, const char* const pTitle )
    {
        if ( nullptr != pTitle )
        {
            LOG_OUT( "%s\n", pTitle );
        }
        LOG_OUT( "-------- Result ---------\n" );
        LOG_OUT( "module = %u\n", result.GetModule() );
        LOG_OUT( "description = %u\n", result.GetDescription() );
        LOG_OUT( "inner value = 0x%08x\n", result.GetInnerValueForDebug() );
        LOG_OUT( "-------------------------\n" );
    }

    //!--------------------------------------------------------------------------------------
    //! ワークバッファ管理
    //!--------------------------------------------------------------------------------------
    class WorkBuffer
    {
    public:
        static const size_t DefaultBufferSize = 128 * 1024 * 1024;

        explicit WorkBuffer() NN_NOEXCEPT
            : m_Size( 0 ), m_pBuffer( nullptr )
        {
        }

        ~WorkBuffer() NN_NOEXCEPT
        {
            void* pBuffer;
            if ( nullptr != ( pBuffer = m_pBuffer ) )
            {
                m_pBuffer = nullptr;
                ::aligned_free( pBuffer );
            }
            m_Size = 0;
        }

        void* Allocate( const size_t size = DefaultBufferSize ) NN_NOEXCEPT
        {
            void* pBuffer;
            size_t actualSize;
            m_Size = actualSize = nn::util::align_up( size, nn::os::MemoryPageSize );
            m_pBuffer = pBuffer = ::aligned_alloc( nn::os::MemoryPageSize, actualSize );
            return pBuffer;
        }

        NN_FORCEINLINE const size_t GetSize() const NN_NOEXCEPT
        {
            return m_Size;
        }

        NN_FORCEINLINE void* operator()() const NN_NOEXCEPT
        {
            return m_pBuffer;
        }

    private:
        size_t  m_Size;
        void*   m_pBuffer;
    };

    //!--------------------------------------------------------------------------------------
    //! Case Setup
    //!--------------------------------------------------------------------------------------
    class CupCase : public testing::Test, public nnt::nsutil::ApplicationInstaller
    {
    public:
        void RequestCardUpdate( bool* pOutSuccess, const WorkBuffer& workBuffer, const int64_t spanMillisForCancel = 0, bool cancelAfterTotalAny = false ) NN_NOEXCEPT;

    protected:
        virtual void SetUp()
        {
            nn::ncm::Initialize();

#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::ovln::PrepareSenderAndReceiverForDevelop();
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeSenderLibraryForOverlay());
#endif

            nn::ns::Initialize();

            auto& verifierInstance = nn::util::Get( m_NcmStorageVerifier );
            new ( &verifierInstance ) NcmStorageVerifier( nn::ncm::StorageId::BuildInSystem );
            verifierInstance.Initialize();
        }

        virtual void TearDown()
        {
            nn::util::Get( m_NcmStorageVerifier ).Finalize();
            nn::ns::Finalize();

#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::ovln::FinalizeSenderLibraryForOverlay();
            nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
#endif

            nn::ncm::Finalize();
        }

        static void SetUpTestCase()
        {
            testns::InitializeCommandLineOptions();
            NN_ABORT_UNLESS_RESULT_SUCCESS( testns::MountHostFileSystem( "Contents" ) );
        }

        static void TearDownTestCase()
        {
            testns::UnmountHostFileSystem();
            testns::FinalizeCommandLineOptions();
        }

    protected:
        nn::util::TypedStorage< NcmStorageVerifier, sizeof( NcmStorageVerifier ), NN_ALIGNOF( NcmStorageVerifier ) > m_NcmStorageVerifier;
    };



    // AsyncUtils
    template< typename THandle, typename TControl = nn::ns::SystemUpdateControl >
    class RequestSynchronizer
    {
    public:
        typedef TControl    ControlType;
        typedef THandle     HandleType;

        HandleType& GetHandle() NN_NOEXCEPT
        {
            return m_Handle;
        }

        const nn::TimeSpan GetElapsedTime() const NN_NOEXCEPT
        {
            return m_ElapsedTime;
        }

        nn::Result WaitResponse( const int64_t spanMillisForCancel = 0, bool cancelAfterTotalAny = false ) NN_NOEXCEPT
        {
            const auto before = nn::os::GetSystemTick();
            auto result = OnWaitResponse( m_Control, m_Handle, spanMillisForCancel, cancelAfterTotalAny );
            m_ElapsedTime = ( nn::os::GetSystemTick() - before ).ToTimeSpan();
            LOG_OUT( "Response wait : %lld ms.\n", m_ElapsedTime.GetMilliSeconds() );
            return result;
        }

    protected:
        explicit RequestSynchronizer( ControlType& control ) NN_NOEXCEPT
            : m_Control( control )
        {
        }

        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle, const int64_t spanMillisForCancel, bool cancelAfterTotalAny ) NN_NOEXCEPT = 0;

    private:
        ControlType&    m_Control;
        HandleType      m_Handle;
        nn::TimeSpan    m_ElapsedTime;
    };

    // RequestPrepareCardUpdate
    class NewlyUpdatePreparer : public RequestSynchronizer< nn::ns::AsyncResult >
    {
    public:
        typedef RequestSynchronizer< nn::ns::AsyncResult > BaseType;

        explicit NewlyUpdatePreparer( ControlType& control ) NN_NOEXCEPT
            : BaseType( control )
        {
        }

    protected:
        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle, const int64_t spanMillisForCancel, bool cancelAfterTotalAny ) NN_NOEXCEPT NN_OVERRIDE
        {
            auto resetBeginAtTotalAny = false;
            auto beginCancel = nn::os::GetSystemTick();
            while ( !handle.TryWait() )
            {
                auto progress = control.GetPrepareCardUpdateProgress();
                LOG_OUT( "loaded %lld, total %lld\n", progress.loaded, progress.total );

                if ( false == resetBeginAtTotalAny && progress.total > 0 )
                {
                    resetBeginAtTotalAny = true;
                    beginCancel = nn::os::GetSystemTick();
                }

                const auto span = ( nn::os::GetSystemTick() - beginCancel ).ToTimeSpan();
                if ( ( false == cancelAfterTotalAny || resetBeginAtTotalAny )
                    && ( spanMillisForCancel > 0 && span.GetMilliSeconds() >= spanMillisForCancel ) )
                {
                    LOG_OUT( "Request cancelled when elapsed [ %lld ] millisecond, condition[ %s ]\n",
                        spanMillisForCancel, ( resetBeginAtTotalAny ? "total > 0" : "total == 0" ) );
                    handle.Cancel();

                    // キャンセル要求後の完了までの所要時間測定.
                    const auto beforeWait = nn::os::GetSystemTick();
                    handle.Wait();
                    const auto requiredCancelSpan = ( nn::os::GetSystemTick() - beforeWait ).ToTimeSpan();
                    LOG_OUT( "Required time for cancel exection [ %lld ] millisecond.\n", requiredCancelSpan.GetMilliSeconds() );
                    break;
                }
                else
                {
                    nn::os::SleepThread( nn::TimeSpan::FromSeconds( 1 ) );
                }
            }
            auto progress = control.GetPrepareCardUpdateProgress();
            LOG_OUT( "loaded %lld, total %lld\n", progress.loaded, progress.total );
            return handle.Get();
        }
    };

    // ApplyCardUpdate
    class NewlyUpdateInstaller : public RequestSynchronizer< int >
    {
    public:
        typedef RequestSynchronizer< int > BaseType;

        explicit NewlyUpdateInstaller( ControlType& control ) NN_NOEXCEPT
            : BaseType( control )
        {
        }

    protected:
        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle, const int64_t spanMillisForCancel, bool cancelAfterTotalAny ) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED( handle );

            return control.ApplyCardUpdateForDebug();
        }
    };

    //!--------------------------------------------------------------------------------------
    //!--------------------------------------------------------------------------------------
    void CupCase::RequestCardUpdate( bool* pOutSuccess, const WorkBuffer& setupWorkBuffer, const int64_t spanMillisForCancel, bool cancelAfterTotalAny ) NN_NOEXCEPT
    {
        LOG_OUT( "===== Start CupCase::RequestCardUpdate[ cancel( %lld ) ms, afterTotal( %s ) ] =====\n",
            spanMillisForCancel, ( cancelAfterTotalAny ) ? "true" : "false"
        );

        if ( nullptr != pOutSuccess )
        {
            *pOutSuccess = false;
        }

        WorkBuffer workBuffer;
        {
            // 本体更新制御の占有開始.
            nn::ns::SystemUpdateControl control;
            NNT_ASSERT_RESULT_SUCCESS( control.Occupy() );

            // カード本体更新の事前準備.
            NNT_ASSERT_RESULT_SUCCESS( control.SetupCardUpdate( setupWorkBuffer(), setupWorkBuffer.GetSize() ) );

            bool prepared = false;
            if ( spanMillisForCancel > 0 )
            {
                {
                    // キャンセル実施。
                    // キャンセル後に AsyncResultの破棄確実にするためスコープによるデストラクタコールを実施。
                    NewlyUpdatePreparer preparer( control );
                    LOG_OUT( "Start latest update preparing with cancel... On API [RequestPrepareCardUpdate]\n" );
                    NNT_EXPECT_RESULT_SUCCESS( control.RequestPrepareCardUpdate( &preparer.GetHandle() ) );

                    const auto reqFirst = preparer.WaitResponse( spanMillisForCancel, cancelAfterTotalAny );
                    LogoutResultValue( reqFirst, "Result of a call RequestPrepareCardUpdate with cancelled =>" );
                    EXPECT_EQ( true, nn::ns::ResultCanceled::Includes( reqFirst ) );

                    // SystemUpdateControl と AsyncResult を破棄していない状態で再度要求時に AlreadyRequested になるか。
                    LOG_OUT( "Start of a call RequestPrepareCardUpdate after the preapre cancelled, with without not release the AsyncResult.\n" );
                    const auto reqPrepare = control.RequestPrepareCardUpdate( &preparer.GetHandle() );
                    LogoutResultValue( reqPrepare, "Result of a call RequestPrepareCardUpdate after the preapre cancelled, with without not release the AsyncResult =>" );
                    EXPECT_EQ( true, nn::ns::ResultPrepareCardUpdateAlreadyRequested::Includes( reqPrepare ) );
                }
                {
                    // キャンセル後の未破棄 SystemUpdateControl での再利用時のエラー検証。
                    NewlyUpdatePreparer preparer( control );

                    // 再要求用にカード本体更新の事前準備用バッファ確保.
                    // 仕様上同じバッファの再利用は出来ない => ABORT ( nn::svc::ResultInvalidCurrentMemory ) が発生する.
                    ASSERT_NE( nullptr, workBuffer.Allocate() );

                    // SystemUpdateControl を破棄していない状態で再度要求時にエラーになるか
                    LOG_OUT( "Start of a call SetupCardUpdate after the preapre cancelled.\n" );
                    const auto reqSetup = control.SetupCardUpdate( workBuffer(), workBuffer.GetSize() );
                    LogoutResultValue( reqSetup, "Result of a call SetupCardUpdate after the preapre cancelled =>" );
                    EXPECT_EQ( true, nn::ns::ResultCardUpdateAlreadySetup::Includes( reqSetup ) );

                    // SystemUpdateControl を破棄していない状態で再度要求時にエラーになるか
                    LOG_OUT( "Start of a call RequestPrepareCardUpdate after the preapre cancelled.\n" );
                    const auto reqPrepare = control.RequestPrepareCardUpdate( &preparer.GetHandle() );
                    LogoutResultValue( reqPrepare, "Result of a call RequestPrepareCardUpdate after the preapre cancelled =>" );
                    EXPECT_EQ( true, nn::ns::ResultPrepareCardUpdateAlreadyRequested::Includes( reqPrepare ) );
                }
            }
            else
            {
                // キャンセルしない正常なCUPシーケンス。
                LOG_OUT( "Start latest update preparing... On API [RequestPrepareCardUpdate]\n" );
                NewlyUpdatePreparer preparer( control );
                NNT_EXPECT_RESULT_SUCCESS( control.RequestPrepareCardUpdate( &preparer.GetHandle() ) );
                NNT_EXPECT_RESULT_SUCCESS( preparer.WaitResponse() );
                LOG_OUT( "Complete latest update preparing.\n" );
                prepared = control.HasPreparedCardUpdate();
                EXPECT_EQ( true, prepared );
            }
            if ( prepared )
            {
                // インストール. ApplyCardUpdate
                LOG_OUT( "Start install prepared update... On API [nn::ns::SystemUpdateControl::ApplyCardUpdate]\n" );
                NewlyUpdateInstaller installer( control );
                const nn::Result resInstall = installer.WaitResponse();
                if ( nullptr != pOutSuccess )
                {
                    *pOutSuccess = resInstall.IsSuccess();
                }
                NNT_ASSERT_RESULT_SUCCESS( resInstall );
                LOG_OUT( "Complete install prepared update.\n" );
            }

            // スコープアウトで control.Relieve が呼ばれます。
        }
    }
}

TEST_F( CupCase, CancelOnPrepare )
{
    // ストレージ上のインストールコンテンツ及び本体更新対象コンテンツを含むコレクション生成
    auto& verifier = nn::util::Get( m_NcmStorageVerifier );
    verifier.PrepareUnexpectedContentOnStorage();

    // カード本体更新の事前準備用バッファ確保.
    WorkBuffer workBuffer;
    ASSERT_NE( nullptr, workBuffer.Allocate() );
    LOG_OUT( "Did allocate the working buffer for CUP which have size of [ %lld ] byte.\n", workBuffer.GetSize() );

    // ランダムにキャンセル
    std::random_device seedGenerator; // 非決定的な乱数生成
    std::mt19937 randomizer( seedGenerator() );

    // 指定ミリ秒範囲のランダムを生成
    std::uniform_int_distribution<uint_fast32_t> distributorSec1( 0, 999 );
    std::uniform_int_distribution<uint_fast32_t> distributorSec5( 0, 4999 );

    //---------------------------------------------------------------------------------------------
    // progress.total <= 0 のキャンセル
    //---------------------------------------------------------------------------------------------
    // 1.0～2.0秒の間でキャンセル
    RequestCardUpdate( nullptr, workBuffer, static_cast< int64_t >( 1000 + distributorSec1( randomizer ) ) );

    // 4.0～5.0秒の間でキャンセル
    RequestCardUpdate( nullptr, workBuffer, static_cast< int64_t >( 4000 + distributorSec1( randomizer ) ) );

    //---------------------------------------------------------------------------------------------
    // progress.total > 0 以降のキャンセル
    //---------------------------------------------------------------------------------------------
    // 1.0～5.0秒の間でキャンセル
    RequestCardUpdate( nullptr, workBuffer, static_cast< int64_t >( 1000 + distributorSec5( randomizer ) ), true );

    // 5.0～10.0秒の間でキャンセル
    RequestCardUpdate( nullptr, workBuffer, static_cast< int64_t >( 5000 + distributorSec5( randomizer ) ), true );

    // 最終的に正常にCUP出来るか。
    bool couldInstalled = false;
    RequestCardUpdate( &couldInstalled, workBuffer );
    if ( couldInstalled )
    {
        // NCMデータベースベリファイ
        verifier.VerifyHasContentOnStorage();
        verifier.VerifyUnnecessaryPlaceHolderOnStorage();
        verifier.VerifyUnexpectedContentOnStorage();
    }

} // NOLINT(impl/function_size)
