﻿<#
    $ powershell -Version 2.0 -ExecutionPolicy Unrestricted MakeNupContents.ps1
#>

# main 引数
param (
    # Qualify the platforms
    [string] $Platform = "NX-NXFP2-a64",

    # Qualify the build types
    [string] $BuildType = "Develop",

    # Qualify the intermediate working folder path
    [string] $IntermediateDirectory,

    # Qualify the system update meta identifier.
    [string] $UpdateMetaId,

    # Qualify the content meta identifier of begin for collection.
    [string] $ContentMetaId,

    # Qualify the proxy configuration.
    [string] $Proxy = "http://proxy.nintendo.co.jp:8080",

    # Qualify the server environments.
    [string] $ReqEnv = "dev6"
)

# スクリプトパス関連 ( 全部文字列型らしい [string] )
$MyScriptPath          = $MyInvocation.MyCommand.Path
$MyScriptDirectoryPath = [System.IO.Path]::GetDirectoryName($MyScriptPath)
$BasicModulePath       = "${MyScriptDirectoryPath}\..\..\SystemUpdateBasic\Scripts\Modules"

Import-Module "${BasicModulePath}\PlatformUtilities"
Import-Module "${BasicModulePath}\OptionUtilities"

# SDK ROOTパス
$NintendoSdkRootPath = $(Find-NintendoSdkRootPath)

# NintendoSDK Integrate script module インポート
Import-Module "${NintendoSdkRootPath}\Integrate\Scripts\Modules\Error"
Import-Module "${NintendoSdkRootPath}\Integrate\Scripts\Modules\Path"
Import-Module "${NintendoSdkRootPath}\Integrate\Scripts\Modules\HostBridge"
Import-Module "${BasicModulePath}\FileUtilities"
Import-Module "${BasicModulePath}\SdkToolUtilities"
Import-Module "${BasicModulePath}\XmlFileUtilities"

Import-Module "${MyScriptDirectoryPath}\Modules\SdkToolUtilities"
Import-Module "${MyScriptDirectoryPath}\Modules\MetaXmlUtilities"

# -----------------------------------------------------------------------
# Upload to highway
# -----------------------------------------------------------------------
function Publish-NspRomsAsPrivate( [string]$defNspDirectoryPath, [string]$defFilePathConfig, [string]$defProxy, [string]$defEnvironment )
{
    $(Test-ExistFile ${defFilePathConfig})
    [xml] $defXmlDoc = [xml] $(Get-Content ${defFilePathConfig})
    [array] $defContentNodes = $defXmlDoc.TestApplicationSetting.TestApplication
    ${defContentNodes} | % {
        [System.Xml.XmlElement] $defContentNode = $_
        [System.Xml.XmlElement] $defMetaNode = $defContentNode.MetaSetting.SystemData
        [string] $defNspFileBaseName = $defContentNode.FileName
        [string] $defNspMetaId = $(Select-ExpectStringMetaId $defMetaNode.Id)
        [string] $defNspMetaVersion = $defMetaNode.Version

        # prepare rops upload
        [string] $defNspFile = "${defNspDirectoryPath}\${defNspFileBaseName}.nsp"
        [int] $defMetaVersion = [int] ${defNspMetaVersion}
        $(Publish-RomByRops ${defNspDirectoryPath} ${defProxy} ${defEnvironment} ${defNspMetaId} ${defNspFile} ${defMetaVersion})
    }
}

# -----------------------------------------------------------------------
# Make system update packages.
# -----------------------------------------------------------------------
function New-NspRomsAsPrivate( [string]$defIntermediateDirectory, [string]$defProxy, [string]$defReqEnv, [string]$defUpdateMetaId, [int]$defMetaVersion, [System.UInt64]$defDataMetaIdBegin )
{
    [System.UInt32] $defDataCount = 7
    [System.UInt64] $defRomSize = 64 * 1024 * 1024
    [string] $defConfigFilePath = $(Out-CreateTestApplicationConfigXml ${defIntermediateDirectory} ${defUpdateMetaId} ${defMetaVersion} "SystemData" ${defRomSize} ${defDataMetaIdBegin} ${defDataCount})
    [string] $defTitleBaseName = $(Out-CreateSystemUpdateMetaForSystemData ${defIntermediateDirectory} ${defUpdateMetaId} ${defMetaVersion} ${defDataMetaIdBegin} ${defDataCount})
    [string] $defTitleMetaFile = "${defIntermediateDirectory}/${defTitleBaseName}.nmeta"
    [string] $defNspDirectoryPath = "${defIntermediateDirectory}\Nsp_v${defMetaVersion}"
    $(Remove-ExistDirectory ${defNspDirectoryPath})

    # create SystemUpdateMeta.nsp
    [string] $defNspFile = $(Out-CreateSystemUpdateNsp ${defNspDirectoryPath} ${defTitleBaseName} ${defTitleMetaFile})

    # create SystemContents.nsp
    $(Out-CreateContentNspByConfigXml ${defNspDirectoryPath} ${defConfigFilePath})

    # publish SystemContents.nsp
    $(Publish-NspRomsAsPrivate ${defNspDirectoryPath} ${defConfigFilePath} ${defProxy} ${defReqEnv}) | Write-Host

    # publish SystemUpdateMeta.nsp
    $(Publish-RomByRops ${defNspDirectoryPath} ${defProxy} ${defReqEnv} ${defUpdateMetaId} ${defNspFile} ${defMetaVersion}) | Write-Host

    Write-Host "Out-SystemUpdateMetaNspFile: ${defNspFile}"
    return ${defNspFile}
}



# ================================
# main
# ================================
try {
    # CIジョブでのTestRunnerは、nact経由でステップ実行し、nactはUTF8で出力を受信している。
    # そのため、強制的にコンソールをUTF-8としている。
    # 但し通常のコード932のDOS窓上では本スクリプトから、Write-Hostなどに日本語指定が使えないので注意。
    $defUseEncoding = New-Object System.Text.UTF8Encoding($False)
    $defBackupConsoleEncoding = [System.Console]::OutputEncoding
    $defBackupEncode = $OutputEncoding
    [System.Console]::OutputEncoding = ${defUseEncoding}
    $OutputEncoding = ${defUseEncoding}

    # エンコード情報表示
    [string] $defConsoleEncode = [System.Console]::OutputEncoding.EncodingName
    [string] $defStartupEncode = $defBackupEncode.EncodingName
    [string] $defCurrentEncode = $OutputEncoding.EncodingName
    Write-Host "[Console]::OutputEncoding : ${defConsoleEncode}"
    Write-Host "Startup `$OutputEncoding   : ${defStartupEncode}"
    Write-Host "Current `$OutputEncoding   : ${defCurrentEncode}"

    # プラットフォームチェック
    $Platform = Resolve-PlatformConstants( ${Platform} )

    # 中間ファイル作成作業フォルダ存在確認
    [string] $IntermediateDirectory = $(Get-AbsoluteFullPath ${IntermediateDirectory})
    $(Edit-MakeDirectory ${IntermediateDirectory})

    Write-Host "Platform        : ${Platform}"
    Write-Host "SDK Root        : ${NintendoSdkRootPath}"
    Write-Host "ScriptPath      : ${MyScriptPath}"
    Write-Host "ScriptDirectory : ${MyScriptDirectoryPath}"

    # 64/32 アーキテクトシグネチャ取得
    [string] $ArchSignature = $(Get-ArchitectSignature ${Platform})
    Write-Host "Architexture    : ${ArchSignature}"

    # メタID系準備
    Write-Host "UpdateMetaId    : ${UpdateMetaId}"
    Write-Host "SystemDataId    : ${ContentMetaId}"
    [System.UInt64] $defIntContentMetaId = [System.Convert]::ToUInt64( "0x${ContentMetaId}", 16 );

    # 追加用コンテンツ生成＆アップロード ( version 0 )
    $(New-NspRomsAsPrivate ${IntermediateDirectory} ${Proxy} ${ReqEnv} ${UpdateMetaId} 0 ${defIntContentMetaId})

    # 更新用コンテンツ生成＆アップロード ( version 1 )
    #$(New-NspRomsAsPrivate ${IntermediateDirectory} ${Proxy} ${ReqEnv} ${UpdateMetaId} 1 ${defIntContentMetaId})

} catch [Exception] {

    Write-ErrorRecord $_
    $LastExitCode = 1

} finally {

    $OutputEncoding = $defBackupEncode
    [System.Console]::OutputEncoding = $defBackupConsoleEncoding

}
exit $LastExitCode

