﻿<#
    $ powershell -Version 2.0 -ExecutionPolicy Unrestricted MakeNupContents.ps1
#>

# main 引数
param (
    # Qualify the platforms
    [string] $Platform = "NX-NXFP2-a64",

    # Qualify the build types
    [string] $BuildType = "Develop",

    # Qualify the intermediate working folder path
    [string] $IntermediateDirectory,

    # Qualify the system update meta identifier.
    [string] $UpdateMetaId,

    # Qualify the content meta identifier of begin for collection.
    [string] $ContentMetaId,

    # Qualify the application identifer with the cup archives.
    [string] $ApplicationId,

    # インストール先ストレージの許容量
    [string] $StorageCapacityCount = "2048",

    [string]
    $TargetName,

    [string]
    $TargetAddress
)

# スクリプトパス関連 ( 全部文字列型らしい [string] )
$MyScriptPath          = $MyInvocation.MyCommand.Path
$MyScriptDirectoryPath = [System.IO.Path]::GetDirectoryName($MyScriptPath)
$BasicModulePath       = "${MyScriptDirectoryPath}\..\..\SystemUpdateBasic\Scripts\Modules"

Import-Module "${BasicModulePath}\PlatformUtilities"
Import-Module "${BasicModulePath}\OptionUtilities"

# SDK ROOTパス
$NintendoSdkRootPath = $(Find-NintendoSdkRootPath)

# NintendoSDK Integrate script module インポート
Import-Module "${NintendoSdkRootPath}\Integrate\Scripts\Modules\Error"
Import-Module "${NintendoSdkRootPath}\Integrate\Scripts\Modules\Path"
Import-Module "${NintendoSdkRootPath}\Integrate\Scripts\Modules\HostBridge"
Import-Module "${BasicModulePath}\FileUtilities"
Import-Module "${BasicModulePath}\SdkToolUtilities"
Import-Module "${BasicModulePath}\XmlFileUtilities"

Import-Module "${MyScriptDirectoryPath}\Modules\SdkToolUtilities"
Import-Module "${MyScriptDirectoryPath}\Modules\MetaXmlUtilities"

if ([string]::IsNullOrEmpty($TargetAddress)) {
    if ([string]::IsNullOrEmpty($TargetName)) {
        throw "TargetAddress and TargetName not specified"
    } else {
        $TargetAddress = Get-TargetAddressFromSerial $TargetName
    }
}

# -----------------------------------------------------------------------
# Make array for nsp.
# 返値: string[]: nsp ファイルパスの配列.
# $defFilePathConfig: MakeTestApplication --configXml オプションで指定したコンフィグファイル
# -----------------------------------------------------------------------
function New-NspFilePathArrayForSystemData( [string]$defNspDirectoryPath, [string]$defFilePathConfig )
{
    $(Test-ExistFile ${defFilePathConfig})

    [string[]] $defResultNspArray = @()
    [xml] $defXmlDoc = [xml] $(Get-Content ${defFilePathConfig})
    [array] $defContentNodes = $defXmlDoc.TestApplicationSetting.TestApplication
    ${defContentNodes} | % {
        [System.Xml.XmlElement] $defContentNode = $_
        [System.Xml.XmlElement] $defMetaNode = $defContentNode.MetaSetting.SystemData
        [string] $defNspFileBaseName = $defContentNode.FileName
        [string] $defNspMetaId = $(Select-ExpectStringMetaId $defMetaNode.Id)
        [string] $defNspMetaVersion = $defMetaNode.Version

        # expect nsp.
        [string] $defNspFile = "${defNspDirectoryPath}\${defNspFileBaseName}.nsp"
        $defResultNspArray += ${defNspFile}
    }
    return ${defResultNspArray}
}

# -----------------------------------------------------------------------
# Make system update packages.
# -----------------------------------------------------------------------
function New-NspRomsForSystemData( [string]$defIntermediateDirectory, [string]$defUpdateMetaId, [int]$defMetaVersion, [System.UInt64]$defDataMetaIdBegin, [System.UInt32]$defDataCount )
{
    # system update meta for verify contents on storage.
    $(Out-CreateSystemUpdateMetaForSystemData ${defIntermediateDirectory} ${defUpdateMetaId} ${defMetaVersion} ${defDataMetaIdBegin} ${defDataCount}) | Out-Null

    # nsp make by MakeTestApplication
    [string] $defConfigFilePath = $(Out-CreateTestApplicationConfigForSystemData ${defIntermediateDirectory} ${defUpdateMetaId} ${defMetaVersion} ${defDataMetaIdBegin} ${defDataCount})
    [string] $defNspDirectoryPath = "${defIntermediateDirectory}\Nsp_v${defMetaVersion}"
    $(Remove-ExistDirectory ${defNspDirectoryPath})
    $(Out-CreateContentNspByConfigXml ${defNspDirectoryPath} ${defConfigFilePath})

    # make nfa
    [string] $defCupBaseName = $(Get-ContentNspFileBaseName ${defUpdateMetaId} ${defMetaVersion} "")
    [string] $defSystemUpdateNsp = "${defNspDirectoryPath}\${defCupBaseName}.nsp"

    [string[]] $defNspContents = $(New-NspFilePathArrayForSystemData ${defNspDirectoryPath} ${defConfigFilePath})
    [string] $defContentsNfa = $(Out-MakeContentsNfa $defIntermediateDirectory ${defCupBaseName} ${defNspContents})

    # make cup archive
    [string] $defMakeCupArchive = "${NintendoSdkRootPath}\Tools\CommandLineTools\MakeCupArchive.exe"
    [string] $defKeyConfig = "${NintendoSdkRootPath}\Tools\CommandLineTools\AuthoringTool\AuthoringTool.ocean.keyconfig.xml"
    Invoke-Expression "& `"${defMakeCupArchive}`" --system-meta-id 0x${defUpdateMetaId} --system-meta-version ${defMetaVersion} -o `"${defSystemUpdateNsp}`" -i `"${defContentsNfa}`" --keyconfig `"${defKeyConfig}`"" | Out-Null
    $(Test-ExistFile ${defSystemUpdateNsp})

    return ${defSystemUpdateNsp}
}

# -----------------------------------------------------------------------
# Make cup packages.
# -----------------------------------------------------------------------
function New-CupNspAndConfig( [string]$defIntermediateDirectory, [string]$defUpdateMetaId, [int]$defMetaVersion, [System.UInt64]$defDataMetaIdBegin, [System.UInt32]$defDataCount, [string]$defApplicationNsp )
{
    [string] $defSystemUpdateNsp = $(New-NspRomsForSystemData ${IntermediateDirectory} ${UpdateMetaId} ${defMetaVersion} ${defDataMetaIdBegin} ${defDataCount})
    Write-Host "Out-SystemUpdateMetaNspFile: ${defSystemUpdateNsp}"
    [string] $defCupConfig = $(Out-MakeCupConfigFile ${IntermediateDirectory} ${defApplicationNsp} ${defSystemUpdateNsp} ${defMetaVersion})
}



# ================================
# main
# ================================
try {
    # CIジョブでのTestRunnerは、nact経由でステップ実行し、nactはUTF8で出力を受信している。
    # そのため、強制的にコンソールをUTF-8としている。
    # 但し通常のコード932のDOS窓上では本スクリプトから、Write-Hostなどに日本語指定が使えないので注意。
    $defUseEncoding = New-Object System.Text.UTF8Encoding($False)
    $defBackupConsoleEncoding = [System.Console]::OutputEncoding
    $defBackupEncode = $OutputEncoding
    [System.Console]::OutputEncoding = ${defUseEncoding}
    $OutputEncoding = ${defUseEncoding}

    # エンコード情報表示
    [string] $defConsoleEncode = [System.Console]::OutputEncoding.EncodingName
    [string] $defStartupEncode = $defBackupEncode.EncodingName
    [string] $defCurrentEncode = $OutputEncoding.EncodingName
    Write-Host "[Console]::OutputEncoding : ${defConsoleEncode}"
    Write-Host "Startup `$OutputEncoding   : ${defStartupEncode}"
    Write-Host "Current `$OutputEncoding   : ${defCurrentEncode}"

    # プラットフォームチェック
    $Platform = Resolve-PlatformConstants( ${Platform} )

    # 中間ファイル作成作業フォルダ存在確認
    [string] $IntermediateDirectory = $(Get-AbsoluteFullPath ${IntermediateDirectory})
    $(Edit-MakeDirectory ${IntermediateDirectory})

    Write-Host "Platform        : ${Platform}"
    Write-Host "SDK Root        : ${NintendoSdkRootPath}"
    Write-Host "ScriptPath      : ${MyScriptPath}"
    Write-Host "ScriptDirectory : ${MyScriptDirectoryPath}"

    # 64/32 アーキテクトシグネチャ取得
    [string] $ArchSignature = $(Get-ArchitectSignature ${Platform})
    Write-Host "Architexture    : ${ArchSignature}"

    # インストール先ストレージの許容量
    [System.UInt32] $defStorageCapacityCount = [System.Convert]::ToUInt32( ${StorageCapacityCount}, 10 )
    [System.UInt64] $defIntContentMetaId = [System.Convert]::ToUInt64( "0x${ContentMetaId}", 16 );
    Write-Host "UpdateMetaId    : ${UpdateMetaId}"
    Write-Host "SystemDataId    : ${ContentMetaId}"
    Write-Host "ApplicationId   : ${ApplicationId}"
    Write-Host "StorageCapacity : ${defStorageCapacityCount}"

    # ===== Get a count of system contents that has been installed in the target box. =====
    [System.UInt32] $defInstalledSystemContentCount = $(Get-CountOfInstalledSystemContent ${Platform} ${BuildType} ${TargetName} ${TargetAddress})

    # 論理上限値 - インストール済コンテンツ - 新規更新対象 SystemUpdateMeta x2 ( 一つは再起動までの多重化用 )
    [System.UInt32] $defBlankSpaceCount = ${defStorageCapacityCount} - ${defInstalledSystemContentCount} - [System.UInt32] 2

    # ===== Calculate a count of new system content for install at in second time. =====
    [System.UInt32] $defLogicalLimitContentCountAtSecond = [math]::Floor( ${defBlankSpaceCount} / [System.UInt32] 2 )

    # ===== Calculate a count of new system content for install at in first time. =====
    [System.UInt32] $defLogicalLimitContentCountAtFirst = [math]::Floor( ${defLogicalLimitContentCountAtSecond} + ( ${defBlankSpaceCount} % [System.UInt32] 2 ) )

    Write-Host "defInstalledSystemContentCount      : ${defInstalledSystemContentCount}"
    Write-Host "defBlankSpaceCount                  : ${defBlankSpaceCount}"
    Write-Host "defLogicalLimitSystemUpdate         : 1"
    Write-Host "defLogicalLimitContentCountAtFirst  : ${defLogicalLimitContentCountAtFirst}"
    Write-Host "defLogicalLimitContentCountAtSecond : ${defLogicalLimitContentCountAtSecond}"

    [string] $defApplicationNsp = $(New-DummyUserApplication ${IntermediateDirectory} ${ApplicationId} 0 ${ArchSignature})

    # 1回目の更新用コンテンツ生成 ( relativeID = baseId + ( 0 ～ ( defLogicalLimitContentCountAtFirst - 1  ) ) )
    $(New-CupNspAndConfig ${IntermediateDirectory} ${UpdateMetaId} 0 ${defIntContentMetaId} ${defLogicalLimitContentCountAtFirst} ${defApplicationNsp})

    # 2回目の更新用コンテンツ生成 ( relativeID = baseId + ( 0 ～ ( defLogicalLimitContentCountAtSecond - 1  ) ) )
    $(New-CupNspAndConfig ${IntermediateDirectory} ${UpdateMetaId} 1 ${defIntContentMetaId} ${defLogicalLimitContentCountAtSecond} ${defApplicationNsp})

    # 3回目の更新用コンテンツ生成 ( relativeID = baseId + ( 0 ～ defLogicalLimitContentCountAtSecond ) )
    [System.UInt32] $defLogicalLimitContentCountAtOver = ${defLogicalLimitContentCountAtSecond} + [System.UInt32] 1
    $(New-CupNspAndConfig ${IntermediateDirectory} ${UpdateMetaId} 2 ${defIntContentMetaId} ${defLogicalLimitContentCountAtOver} ${defApplicationNsp})

    # 4回目の更新用コンテンツ生成 ( relativeID = baseId + defLogicalLimitContentCountAtFirst + ( 0 ～ ( defLogicalLimitContentCountAtSecond - 1  ) ) )
    [System.UInt64] $defPaddingContentMetaId = ${defIntContentMetaId} + [System.UInt64] ${defLogicalLimitContentCountAtFirst}
    $(New-CupNspAndConfig ${IntermediateDirectory} ${UpdateMetaId} 3 ${defPaddingContentMetaId} ${defLogicalLimitContentCountAtSecond} ${defApplicationNsp})

} catch [Exception] {

    Write-ErrorRecord $_
    $LastExitCode = 1

} finally {

    $OutputEncoding = $defBackupEncode
    [System.Console]::OutputEncoding = $defBackupConsoleEncoding

}
exit $LastExitCode

