﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once
#include <string>
#include <nn/util/util_IntrusiveList.h>
#include <nn/nn_Common.h>
#include <memory>

//!---------------------------------------------------------------------------------
//! @brief Xmlパーサー( SAXスタイル )
//!---------------------------------------------------------------------------------
class XmlParser
{
    NN_DISALLOW_COPY( XmlParser );
    NN_DISALLOW_MOVE( XmlParser );

public:
    //!---------------------------------------------------------------------------------
    //! @brief 文字列コンテナ
    //!---------------------------------------------------------------------------------
    typedef std::string StringType;

    //!---------------------------------------------------------------------------------
    //! @brief 属性
    //! @details 各属性の実体文字列はXMLドキュメントストリーム内を直接参照します。
    //!---------------------------------------------------------------------------------
    class Attribute : public nn::util::IntrusiveListBaseNode< Attribute >
    {
    public:
        //!---------------------------------------------------------------------------------
        //! @brief コレクションタイプです。
        //!---------------------------------------------------------------------------------
        class CollectionType : public nn::util::IntrusiveList< Attribute, nn::util::IntrusiveListBaseNodeTraits< Attribute > >
        {
        public:
            typedef nn::util::IntrusiveList< Attribute, nn::util::IntrusiveListBaseNodeTraits< Attribute > > Impl;

            //!---------------------------------------------------------------------------------
            //! @brief コンストラクタです。
            //!---------------------------------------------------------------------------------
            explicit CollectionType() NN_NOEXCEPT
            {
            }

            //!---------------------------------------------------------------------------------
            //! @brief デストラクタです。
            //!---------------------------------------------------------------------------------
            ~CollectionType() NN_NOEXCEPT
            {
            }

            //!---------------------------------------------------------------------------------
            //! @brief 属性コンテナインスタンス生成
            //!---------------------------------------------------------------------------------
            inline Attribute* CreateNewAttribute( const StringType& name, const StringType& value ) NN_NOEXCEPT
            {
                Attribute* pResult = new Attribute( name, value );
                Impl::push_back( *pResult );
                return pResult;
            }

            //!---------------------------------------------------------------------------------
            //! @brief 要素を全てをクリアします。
            //! @details 要素の全てに対してメモリ解放を実施します。
            //!---------------------------------------------------------------------------------
            inline void Clear() NN_NOEXCEPT
            {
                for ( Impl::iterator it = Impl::begin(); it != Impl::end(); it = Impl::begin() )
                {
                    Impl::erase( it );
                    delete &( *it );
                }
                Impl::clear();
            }
        };

        //!---------------------------------------------------------------------------------
        //! @brief  コンストラクタです。
        //!---------------------------------------------------------------------------------
        explicit Attribute( const StringType& name, const StringType& value ) NN_NOEXCEPT
            : m_Name( name ), m_Value( value ) {}

        //!---------------------------------------------------------------------------------
        //! @brief  コピーコンストラクタです。
        //!---------------------------------------------------------------------------------
        NN_IMPLICIT Attribute( const Attribute& source ) NN_NOEXCEPT
            : m_Name( source.m_Name ), m_Value( source.m_Value )
        {
        }

        //!---------------------------------------------------------------------------------
        //! @brief  属性名を取得します。
        //! @return 結果を返します。
        //!---------------------------------------------------------------------------------
        const StringType& GetName() const NN_NOEXCEPT
        {
            return m_Name;
        }

        //!---------------------------------------------------------------------------------
        //! @brief  属性値を取得します。
        //! @return 結果を返します。
        //!---------------------------------------------------------------------------------
        const StringType& GetValue() const NN_NOEXCEPT
        {
            return m_Value;
        }

    private:
        const StringType    m_Name;             //!< 属性名
        const StringType    m_Value;            //!< 属性値
    };

    //!---------------------------------------------------------------------------------
    //! @brief コールバックインタフェース
    //!---------------------------------------------------------------------------------
    class CallbackInterface
    {
    public:
        //!---------------------------------------------------------------------------------
        //! @brief
        //!---------------------------------------------------------------------------------
        virtual void OnElementBegin( const StringType& name, const Attribute::CollectionType& attributes ) NN_NOEXCEPT = 0;

        //!---------------------------------------------------------------------------------
        //! @brief
        //!---------------------------------------------------------------------------------
        virtual void OnElementEnd( const StringType& name ) NN_NOEXCEPT = 0;

        //!---------------------------------------------------------------------------------
        //! @brief
        //!---------------------------------------------------------------------------------
        virtual void OnText( const StringType& text ) NN_NOEXCEPT = 0;

        //!---------------------------------------------------------------------------------
        //! @brief
        //!---------------------------------------------------------------------------------
        virtual void OnComment( const StringType& text ) NN_NOEXCEPT
        {
            NN_UNUSED( text );
        }
    };

    //!---------------------------------------------------------------------------------
    //! @brief 新しいパーサーインスタンスを生成します。
    //!---------------------------------------------------------------------------------
    static XmlParser* CreateNewParser() NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief 使用済パーサーインスタンスを破棄します。
    //!---------------------------------------------------------------------------------
    static void Finalize( XmlParser* pParser ) NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief XMLドキュメントを解析します。
    //! @param[in] callback 解析中の各タグ要素の情報確定時に呼び出されるコールバックインタフェースを指定します。
    //! @param[in] pDocument 解析する XML ドキュメントストリームの先頭アドレスを指定します。
    //! @param[in] byteSize  XMLドキュメントストリームのバイトサイズを指定します。
    //!---------------------------------------------------------------------------------
    virtual bool Parse( CallbackInterface& callback, const char* pDocument, const size_t byteSize ) NN_NOEXCEPT = 0;

protected:
    //!---------------------------------------------------------------------------------
    //! @brief コンストラクタです。
    //!---------------------------------------------------------------------------------
    XmlParser() NN_NOEXCEPT {}

    //!---------------------------------------------------------------------------------
    //! @brief デストラクタです。
    //!---------------------------------------------------------------------------------
    virtual ~XmlParser() NN_NOEXCEPT {}
};
