﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once
#include <cstdlib>
#include <cstring>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include "testNs_FsUtilities.h"
#include "testNs_MemoryAllocator.h"

//!--------------------------------------------------------------------------------------
//! @brief ファイル読み込みヘルパ。
//!--------------------------------------------------------------------------------------
class RawFileLoader
{
    NN_DISALLOW_COPY( RawFileLoader );
    NN_DISALLOW_MOVE( RawFileLoader );

public:
    //!--------------------------------------------------------------------------------------
    //! @brief デフォルトコンストラクタ
    //!--------------------------------------------------------------------------------------
    RawFileLoader() NN_NOEXCEPT;

    //!--------------------------------------------------------------------------------------
    //! @brief デストラクタ
    //!--------------------------------------------------------------------------------------
    ~RawFileLoader() NN_NOEXCEPT;

    //!--------------------------------------------------------------------------------------
    //! @brief ファイル読み込み。
    //! @return 対象ファイルを展開したメモリ領域先頭アドレス。
    //!--------------------------------------------------------------------------------------
    template< typename T >
    T* LoadFromFile( const char* pFilePath ) NN_NOEXCEPT;

    //!--------------------------------------------------------------------------------------
    //! @brief 解放
    //!--------------------------------------------------------------------------------------
    void UnloadFromMemory() NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief ロードファイルのオンメモリストリームアドレスの取得.
    //! @return ロード済であれば確保したメモリストリーム先頭へのポインタを返します.
    //!---------------------------------------------------------------------------------
    template< typename T >
    NN_FORCEINLINE T* GetStream() const NN_NOEXCEPT
    {
        return reinterpret_cast< T* >( m_pFileStream );
    }

    //!---------------------------------------------------------------------------------
    //! @brief ロードファイルのファイルパスの取得.
    //! @return ロード済であれば確保したファイルのファイルパス文字列へのポインタを返します.
    //!---------------------------------------------------------------------------------
    NN_FORCEINLINE const char* GetFilePath() const NN_NOEXCEPT
    {
        return m_pFilePath;
    }

    //!---------------------------------------------------------------------------------
    //! @brief ロードファイルのファイルサイズ
    //! @return ロード済であれば確保したファイルのファイルサイズ( byte )を返します.
    //!---------------------------------------------------------------------------------
    NN_FORCEINLINE size_t GetFileSize() const NN_NOEXCEPT
    {
        return m_FileSize;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ファイル読み込み。
    //! @return 対象ファイルを展開したメモリ領域先頭アドレス。
    //!--------------------------------------------------------------------------------------
    template< typename T >
    static T* LoadFromFile( size_t& outFileSize, const char* pFilePath, const size_t beginAlignment ) NN_NOEXCEPT;

private:
    void*       m_pFileStream;              //!< 読み込んだファイルのオンメモリストリーム
    const char* m_pFilePath;                //!< 読み込んだファイルのファイルパス
    size_t      m_FileSize;                 //!< 読み込んだファイルのファイルサイズ( byte )
};

//!--------------------------------------------------------------------------------------
//! @brief ファイル読み込み。
//! @return 対象ファイルを展開したメモリ領域先頭アドレス。
//!--------------------------------------------------------------------------------------
template< typename T >
T* RawFileLoader::LoadFromFile( size_t& outFileSize, const char* pFilePath, const size_t beginAlignment ) NN_NOEXCEPT
{
    NN_ASSERT( nullptr != pFilePath );

    size_t fileSize;
    fsutil::File file;
    T* stream = nullptr;
    nn::Result result = file.Open( pFilePath, nn::fs::OpenMode_Read );
    if ( true == result.IsSuccess() && ( fileSize = static_cast< size_t >( file.GetSize() ) ) > 0U )
    {
        void* const buffer = ::aligned_alloc( beginAlignment, fileSize );
        NN_ASSERT_NOT_NULL( buffer );
        size_t readedSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS( file.Read( &readedSize, 0, buffer, fileSize ) );
        file.Close();

        outFileSize = fileSize;
        stream = reinterpret_cast< T* >( buffer );
    }
    return stream;
}

//!--------------------------------------------------------------------------------------
//! @brief ファイル読み込み。
//! @return 対象ファイルを展開したメモリ領域先頭アドレス。
//!--------------------------------------------------------------------------------------
template< typename T >
T* RawFileLoader::LoadFromFile( const char* pFilePath ) NN_NOEXCEPT
{
    NN_ASSERT( nullptr != pFilePath );

    T* pStream = LoadFromFile< T >( m_FileSize, pFilePath, NN_ALIGNOF( T ) );
    if ( nullptr != pStream )
    {
        m_pFileStream = reinterpret_cast< void* >( pStream );
        char* pClonePath = static_cast< char* >( ::aligned_alloc( NN_ALIGNOF( char ), std::strlen( pFilePath ) + 1 ) );
        m_pFilePath = std::strcpy( pClonePath, pFilePath );
    }
    return pStream;
}
