﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
/// Notice.
/// PUX拠点から、VisualStudio のテストエクスプローラ経由テストの場合、*.runsettings ファイルの TestRunParameters に以下パラメタを宣言してください.
/// NCL/CI 及び、NN_TEST_SCRIPT_PROXY_CONFIGURATION 環境変数が設定されたコマンドライン実施の場合は宣言は不要です.
/// <Parameter name = "EnableRunTestOnIDE_FromPUX" value="true" />

using CsTestAssistants;
using System.Reflection;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using Microsoft.VisualStudio.TestTools.UnitTesting;

//using System;

namespace testNs_ShopUploader
{
    [TestClass]
    public class UnitUpload : TestClassBase
    {
        // `TestContext` プロパティを定義しておけば、UnitTestフレームワークで勝手にインスタンスが提供される.
        public TestContext TestContext { get; set; }

        private const string LogPrefix = "[UploadDataeShopBasic]";

        //!----------------------------------------------------------------------------
        /// <summary>
        /// クラスが生成され、呼び出される前に一度だけ実施。
        /// </summary>
        /// <param name="context"></param>
        //!----------------------------------------------------------------------------
        [ClassInitialize]
        public static void TestClassInitialize(TestContext context)
        {
            ClassInitialize(context, LogPrefix);
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// クラスが破棄されるタイミングで一度だけ実施。
        /// </summary>
        //!----------------------------------------------------------------------------
        [ClassCleanup]
        public static void TestClassCleanup()
        {
            ClassCleanup();
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// `TestProperty属性の "JIRA" キーの値` と `Sigloコンフィギュレーションコンテキスト` に依存した作業用ワーキングディレクトリパスを生成します.
        /// </summary>
        /// <param name="context">Sigloコンフィギュレーションコンテキスト</param>
        /// <param name="methodName">
        /// 参照対象TestProperty属性を保持するメソッド名.
        /// 指定を省略した場合、呼び出されたカレントメソッド名が自動で選択されます.
        /// 明示的な null 指定の場合、もしくは "JIRA" キーの TestPropertyが見つからない場合には、TestPropertyに依存しないベース作業用ディレクトリが返却されます。
        /// </param>
        /// <returns>
        /// TestProperty( "JIRA", * ) に依存した作業用ディレクトリパス、もしくはベース作業用ディレクトリパス。
        /// </returns>
        //!----------------------------------------------------------------------------
        private string GenerateIntermediateDirectoryAsMethod(SigloHelper.Configuration.Context context,
            [System.Runtime.CompilerServices.CallerMemberName] string methodName = null)
        {
            var runAssembly = Assembly.GetExecutingAssembly();
            var asmTitle = (AssemblyTitleAttribute)System.Attribute.GetCustomAttribute(runAssembly, typeof(AssemblyTitleAttribute));
            return GenerateIntermediateDirectory(context, asmTitle.Title, methodName);
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// 【UploadData】
        /// UnitMain.cs で使用するデータのアップロード ( app 64個, aoc i1v0 )
        /// 単体実行時で 25 ～ 30 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-69838")]
        public void UploadData_app64_aoc_i1v0()
        {
            // Application : 0x01001a500005e0b0, version 0
            //      ：     :             ：
            // Application : 0x01001a500005e0ef, version 0(計64個)
            // AoC index 1 : 0x01001a500005f0b1, version 0

            // Constants
            ID64 baseApplicationId = new ID64(0x01001a500005e0b0);
            int appVersion = 0;
            int appCount = 64;

            // Target command executor
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

            // 作業用ディレクトリ
            string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

            // Application contents upload.
            D4cHelper.NspUploader uploader = new D4cHelper.NspUploader(intermediate,
                ProxyConfiguration, ServerEnvironment);
            uploader.Publish(CliTokenAccount, baseApplicationId, appCount, appVersion, executor.TargetPlatform, 180 * 64);

            // Addon contents upload. ( 0x01001a500005f0b1 の想定,  0x01001a500005e0b0 + 0x1000 + 1 )
            var addon = new TestApplication.AddonParameter(baseApplicationId, 0, 1, 64 * 1024);
            var addonContents = TestApplication.MakeContents(intermediate, addon);
            var addonIndex1v0 = addonContents[0];
            uploader.Publish(CliTokenAccount, addonIndex1v0.NspPath,
                D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(addonIndex1v0),
                5 * 60
            );
        }


        //!----------------------------------------------------------------------------
        /// <summary>
        /// 【UploadData】
        /// UnitMain.cs で使用するデータのアップロード ( app, patch v1, aoc i1v0, aoc i2v0 )
        /// 単体実行時で 5 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-69838")]
        public void UploadData_app_patch_v1_aoc_i1v0_i2v0()
        {
            // 本来のIDは 0x01003ab001e30000 ですが、0x01003ab001e30800 の Patch の証明書エラー回避のため拡張IDにしています。
            // そのため、本テストのメタ構成は以下の通り。
            // Application : 0x01003ab001e30001, version 0
            // Patch       : 0x01003ab001e30801, version 65536
            // AoC index 1 : 0x01003ab001e31002, version 0
            // AoC index 2 : 0x01003ab001e31003, version 0
            var IdForApplication = new ID64(0x01003ab001e30001);
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

            // 作業用ディレクトリ
            string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

            // make contents.
            var genApp = new TestApplication.ApplicationParameter(IdForApplication, 0, 5 * 1024 * 1024);
            var genAddonA = new TestApplication.AddonParameter(IdForApplication, 0, 1, 64 * 1024);
            var genAddonB = new TestApplication.AddonParameter(IdForApplication, 0, 2, 64 * 1024);
            genApp.AddPatch(1);
            genApp.UseSmallCode = true;
            var request = new List<TestApplication.GenerateParameter<int>>(3) { genApp, genAddonA, genAddonB };
            var uploadContents = TestApplication.MakeContents(intermediate, request);

            // カテゴライズ
            var typedCatalog = new GeneratedContentResult.TypeCategorizedCatalog(uploadContents);
            var aocs = typedCatalog.GetTypedCatalog(ContentMeta.Type.AddOnContent);
            var apps = typedCatalog.GetTypedCatalog(ContentMeta.Type.Application);
            var patches = typedCatalog.GetTypedCatalog(ContentMeta.Type.Patch);
            if (null == aocs || null == apps || null == patches)
            {
                throw new UnexpectFailureException("MakeContents operation failed.");
            }
            GeneratedContentResult app = apps[0];
            GeneratedContentResult patch = patches[0];
            GeneratedContentResult addonA = aocs[0];
            GeneratedContentResult addonB = aocs[1];
            if (null == app || null == patch || null == addonA || null == addonB)
            {
                throw new UnexpectFailureException("MakeContents operation failed.");
            }

            // upload to td1 server by ContentsUploader.
            D4cHelper.NspUploader uploader = new D4cHelper.NspUploader(
                intermediate,
                ProxyConfiguration,
                ServerEnvironment
            );

            // アップロード
            uploader.RejectRoms(CliTokenAccount, IdForApplication);
            uploader.Publish(CliTokenAccount, app.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.WithApproved, 10 * 60);
            uploader.Publish(CliTokenAccount, patch.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.Patch, 10 * 60);
            uploader.Publish(CliTokenAccount, new List<string>(2) { addonA.NspPath, addonB.NspPath },
                D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(addonA), 20 * 60);
        }


        //!----------------------------------------------------------------------------
        /// <summary>
        /// 【UploadData】
        /// UnitMain.cs で使用するデータのアップロード ( app1, Aoc1 i1v0, i1v1, app2, Aoc2 i1v0 )
        /// 単体実行時で 5 ～ 10 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-69838")]
        public void UploadData_app1_aoc1_i1v0_i1v1_app2_aoc2_i1v0()
        {
            // Application : 0x0100394000059000, version 0
            // AoC index 1 : 0x010039400005a001, version 0
            // AoC index 1 : 0x010039400005a001, version 65536
            // Application : 0x0100db60023de000, version 0
            // AoC index 1 : 0x0100db60023df001, version 0
            ID64 applicationId1 = new ID64(0x0100394000059000);
            ID64 applicationId2 = new ID64(0x0100db60023de000);
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

            // テスト用データの生成
            string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

            var app1 = new TestApplication.ApplicationParameter(applicationId1, 0, 1 * 1024 * 1024);
            var aoc1_i1v0 = new TestApplication.AddonParameter(applicationId1, 0, 1, 1024 * 1024);
            var aoc1_i1v1 = new TestApplication.AddonParameter(applicationId1, 1, 1, 1024 * 1024);
            var app2 = new TestApplication.ApplicationParameter(applicationId2, 0, 1 * 1024 * 1024);
            var aoc2_i1v0 = new TestApplication.AddonParameter(applicationId2, 0, 1, 1024 * 1024);

            var request = new List<TestApplication.GenerateParameter<int>>() { app1, aoc1_i1v0, aoc1_i1v1, app2, aoc2_i1v0 };
            var uploadContents = TestApplication.MakeContents(intermediate, request);

            // カテゴライズ
            var typedCatalog = new GeneratedContentResult.TypeCategorizedCatalog(uploadContents);
            var app = typedCatalog.GetTypedCatalog(ContentMeta.Type.Application);
            var aocs = typedCatalog.GetTypedCatalog(ContentMeta.Type.AddOnContent);

            var generatedApp1 = app[0];
            var generatedApp2 = app[1];
            var generatedAoc1_i1v0 = aocs[0];
            var generatedAoc1_i1v1 = aocs[1];
            var generatedAoc2_i1v0 = aocs[2];

            D4cHelper.NspUploader uploader = new D4cHelper.NspUploader(
                intermediate,
                ProxyConfiguration,
                ServerEnvironment
            );

            // アップロード
            uploader.Publish(CliTokenAccount, generatedApp1.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.WithApproved, 10 * 60);
            uploader.Publish(CliTokenAccount, generatedApp2.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.WithApproved, 10 * 60);
            uploader.Publish(CliTokenAccount, generatedAoc1_i1v0.NspPath, D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(generatedAoc1_i1v0));
            uploader.Publish(CliTokenAccount, generatedAoc1_i1v1.NspPath, D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(generatedAoc1_i1v1));
            uploader.Publish(CliTokenAccount, generatedAoc2_i1v0.NspPath, D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(generatedAoc2_i1v0));
        }


        //!----------------------------------------------------------------------------
        /// <summary>
        /// 【UploadData】
        /// 製品化されたアプリ・パッチ・追加コンテンツの購入・ダウンロード検証のアップロードデータ
        /// 単体実行時で 4 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-69838")]
        public void UploadDataOfTestForPurchaseDemoTicket()
        {
            // 作業用ディレクトリ
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

            // Application   : 0x01003ab001e32000, version 0
            // Patch index 1 : 0x01003ab001e32800, version 65536
            // AoC   index 1 : 0x01003ab001e33001, version 0
            var IdForApplication = new ID64(0x01003ab001e32000);

            // 購入者アカウント
            var account = NintendoAccount;

            // make contents.
            var genApp = new TestApplication.ApplicationParameter(IdForApplication, 0, 1 * 1024 * 1024);
            var genAddonA = new TestApplication.AddonParameter(IdForApplication, 0, 1, 64 * 1024);
            genApp.AddPatch(1);
            genApp.UseSmallCode = true;               // 起動チェック用に small-code
            genApp.ChangeTicketEncryption(true);      // 製品化指定
            genAddonA.ChangeTicketEncryption(true);   // 製品化指定
            var request = new List<TestApplication.GenerateParameter<int>>(2) { genApp, genAddonA };
            var uploadContents = TestApplication.MakeContents(intermediate, request);

            // カテゴライズ
            var typedCatalog = new GeneratedContentResult.TypeCategorizedCatalog(uploadContents);
            var aocs = typedCatalog.GetTypedCatalog(ContentMeta.Type.AddOnContent);
            var apps = typedCatalog.GetTypedCatalog(ContentMeta.Type.Application);
            var patches = typedCatalog.GetTypedCatalog(ContentMeta.Type.Patch);
            if (null == aocs || null == apps || null == patches)
            {
                throw new UnexpectFailureException("MakeContents operation failed.");
            }
            GeneratedContentResult app = apps[0];
            GeneratedContentResult patch = patches[0];
            GeneratedContentResult addon = aocs[0];
            if (null == app || null == patch || null == addon)
            {
                throw new UnexpectFailureException("MakeContents operation failed.");
            }

            // upload executor to td1 server by ContentsUploader.
            D4cHelper.NspUploader uploader = new D4cHelper.NspUploader(
                intermediate,
                ProxyConfiguration,
                ServerEnvironment
            );

            // アップロード ( 製品化指定( -m ) は不要 )
            uploader.RejectRoms(CliTokenAccount, IdForApplication);
            uploader.Publish(CliTokenAccount, app.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.WithApproved, 10 * 60);
            uploader.Publish(CliTokenAccount, patch.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.Patch, 10 * 60);
            uploader.Publish(CliTokenAccount, addon.NspPath, 10 * 60);

            // PMS登録
            uploader.RegisterDemoContent(IdForApplication);   // PMS/ROM, 体験版登録 ( app )
            uploader.RegisterDemoContent(addon.NspPath);      // PMS/ROM, 体験版登録 ( aoc )
            uploader.RegisterPatch(patch.NspPath);            // PMS/ROM 登録( patch )
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// 【UploadData】
        /// TestForAddToRunningTask() のアップロードデータ
        /// 単体実行時で 3 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-69838")]
        public void UploadDataOfTestForAddToRunningTask()
        {
            // Application   : 0x010006000236c000, version 0
            // AoC   index 1 : 0x010006000236d001, version 0
            ID64 applicationId = new ID64(0x010006000236c000);

            // Target command executor
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            var account = NintendoAccount;

            // 作業用ディレクトリ
            string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

            var app = new TestApplication.ApplicationParameter(applicationId, 0, 100 * 1024 * 1024);
            var aoc = new TestApplication.AddonParameter(applicationId, 0, 1, 1024 * 1024);
            app.ChangeTicketEncryption(true);
            aoc.ChangeTicketEncryption(true);

            var request = new List<TestApplication.GenerateParameter<int>>() { app, aoc };
            var uploadContents = TestApplication.MakeContents(intermediate, request);

            D4cHelper.NspUploader uploader = new D4cHelper.NspUploader(intermediate,
                ProxyConfiguration, ServerEnvironment);

            foreach (var path in uploadContents.ToNspPaths())
            {
                uploader.Publish(CliTokenAccount, path);
                uploader.RegisterDemoContent(path);
            }
        }


        //!----------------------------------------------------------------------------
        /// <summary>
        /// 【UploadData】
        /// ダウンロード開始時点でアプリ記録更新されていることの確認のアップロードデータ
        /// 単体実行時で 3 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-69838")]
        public void UploadDataOfTestForRecordWhenCancelDownloadTask()
        {
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

            // contents
            //  Application : 0x01003ab001e32100, version 0
            //  AddOnContent: 0x01003ab001e33101, index 1
            var owner = new ID64(0x01003ab001e32100);
            var @params = new List<TestApplication.GenerateParameter<int>>();
            {
                // app
                var param = new TestApplication.ApplicationParameter(owner, 0, 1 * 1024 * 1024);
                param.UseSmallCode = true;
                param.ChangeTicketEncryption(true);
                @params.Add(param);
            }
            {
                // aoc
                var param = new TestApplication.AddonParameter(owner, 0, 1, 1 * 1024 * 1024);
                param.ChangeTicketEncryption(true);
                @params.Add(param);
            }
            var contents = TestApplication.MakeContents(intermediate, @params);
            var catalog = new GeneratedContentResult.TypeCategorizedCatalog(contents);
            var app = catalog.GetTypedCatalog(ContentMeta.Type.Application)[0];
            var aoc = catalog.GetTypedCatalog(ContentMeta.Type.AddOnContent)[0];

            // upload contents
            var uploader = new D4cHelper.NspUploader(
                intermediate,
                ProxyConfiguration,
                ServerEnvironment);

            var account = CliTokenAccount;
            uploader.RejectRoms(account, app.Identifier);
            uploader.Publish(account, app.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.WithApproved, 10 * 60);
            uploader.Publish(account, aoc.NspPath, 10 * 60);

            uploader.RegisterDemoContent(app.Identifier);
            uploader.RegisterDemoContent(aoc.NspPath);
        }


        //!----------------------------------------------------------------------------
        /// <summary>
        /// 【UploadData】
        /// 「実行中のアプリの追加コンテンツをコミットできる」検証。のアップロードデータ
        /// 単体実行時で 7 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-69838")]
        public void UploadDataOfTestForCommitAocInApplicationRunning()
        {
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            var intermediate = GenerateIntermediateDirectoryAsMethod(executor);
            var uploader = new D4cHelper.NspUploader(intermediate, ProxyConfiguration, ServerEnvironment);

            // 本テストのメタ構成は以下の通り。
            // Patch は現状使わないが、別テストで使う予定なので構成事前準備。
            // Application : 0x01003ab001e30049, version 0      : UseSmallCode( false )
            // Patch       : 0x01003ab001e30849, version 65536  : UseSmallCode( false )
            // AoC index 1 : 0x01003ab001e3104a, version 0
            // AoC index 1 : 0x01003ab001e3104a, version 65536
            // AoC index 2 : 0x01003ab001e3104b, version 0
            var IdForApplication = new ID64(0x01003ab001e30049);

            // Make parameter for test contents generating.
            var genApp = new TestApplication.ApplicationParameter(IdForApplication, 0, 64 * 1024);
            genApp.AddPatch(1);
            genApp.ChangeCodePath(System.IO.Path.Combine(executor.GetOutputTestDirectory("ShopRuntimeAssistants"), "code"));
            var genAoc_i1v0 = new TestApplication.AddonParameter(IdForApplication, 0, 1, 64 * 1024);
            var genAoc_i1v1 = new TestApplication.AddonParameter(IdForApplication, 1, 1, 64 * 1024);
            var genAoc_i2v0 = new TestApplication.AddonParameter(IdForApplication, 0, 2, 64 * 1024);
            genAoc_i2v0.ChangeRequiredVersion(1); // RequiredApplicationReleaseVersion

            // Request list of parameters.
            var request = new List<TestApplication.GenerateParameter<int>>(4) { genApp, genAoc_i1v0, genAoc_i1v1, genAoc_i2v0 };

            var uploadContents = TestApplication.MakeContents(intermediate, request);
            var app = uploadContents[0];
            var patch = uploadContents[1];
            var aoc_i1v0 = uploadContents[2];
            var aoc_i1v1 = uploadContents[3];
            var aoc_i2v0 = uploadContents[4];

            uploader.RejectRoms(CliTokenAccount, IdForApplication);
            uploader.Publish(CliTokenAccount, app.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.WithApproved, 5 * 60);
            uploader.Publish(CliTokenAccount, patch.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.Patch, 5 * 60);
            uploader.Publish(CliTokenAccount, aoc_i1v0.NspPath, D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(aoc_i1v0), 5 * 60);
            uploader.Publish(CliTokenAccount, aoc_i1v1.NspPath, D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(aoc_i1v1), 5 * 60);
            uploader.Publish(CliTokenAccount, aoc_i2v0.NspPath, D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(aoc_i2v0), 5 * 60);
        }


        //!----------------------------------------------------------------------------
        /// <summary>
        /// 【UploadData】
        /// NXBTS-11639 を踏まえた、多量のパッチ間差分適用タスクの作成/検証テストのアップロードデータ
        /// 単体実行時で 80 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-69838")]
        public void UploadDataOfTestForVerifyRunManyPatchDeltaApplyTask()
        {
            //既存テストでアップロードしていなかった為、アップロードしない。
#if false
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            var intermediate = GenerateIntermediateDirectoryAsMethod(executor);

            // 本テストのメタ構成は以下の通り。
            // 連番Index | Application ID     | Patch ID           | Version |
            //         0 | 0x01003ab001e30004 | 0x01003ab001e30804 | 0, 1, 2 |
            //         : |          :         |          :         |    :    |
            //         : |          :         |          :         |    :    |
            //        63 | 0x01003ab001e30043 | 0x01003ab001e30843 | 0, 1, 2 |

            uint appCount = 64U;
            var IdForApplication = new ID64(0x01003ab001e30004);

            // make parameter of contents.
            var gens = new List<TestApplication.GenerateParameter<int>>((int)appCount);
            for (uint index = 0; index < appCount; ++index)
            {
                var now = IdForApplication + index;
                var app = new TestApplication.ApplicationParameter(now, 0, 32);
                app.AddPatches(1, new TestApplication.PatchContexture<int>(2, AuthoringExecutor.Patch.FlagOption.UseDelta));
                app.UseSmallCode = true;
                gens.Add(app);
            }

            // make the title resource contexture without call the MakeTestApplication.
            var uploadContents = TestApplication.MakeContents(intermediate, gens);
            var titles = new GeneratedContentResult.TitleCategorizedCatalog(uploadContents);

            // upload executor to td1 server by ContentsUploader.
            D4cHelper.NspUploader uploader = new D4cHelper.NspUploader(
                intermediate,
                ProxyConfiguration,
                ServerEnvironment
            );

            // uploading. ( 5min as timeout of 1 nsp uploading. )
            uploader.Publish( CliTokenAccount, titles, 5 * 60 );
#endif
        }



        //!----------------------------------------------------------------------------
        /// <summary>
        /// 【UploadData】
        /// ネットワークインストール時の追加コンテンツ依存のテストのアップロードデータ
        /// 単体実行時で 7 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-69838")]
        public void UploadDataOfTestForSolveDependencyAocOnNIM()
        {
            // AoC index 2 : 0x01001a500005f0b2 , version 0, 65536
            // AoC index 3 : 0x01001a500005f0b3 , version 0, 65536
            // AoC index 4 : 0x01001a500005f0b4 , version 0, 65536
            ID64 IdForApplication = new ID64(0x01001a500005e0b0);

            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

            // 作業用ディレクトリ
            string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

            // make contents.
            var genIndex2v0 = new TestApplication.AddonParameter(IdForApplication, 0, 2, 4 * 1024 * 1024);
            var genIndex2v1 = new TestApplication.AddonParameter(IdForApplication, 1, 2, 3 * 1024 * 1024);
            var genIndex3v0 = new TestApplication.AddonParameter(IdForApplication, 0, 3, 1 * 1024 * 1024);
            var genIndex3v1 = new TestApplication.AddonParameter(IdForApplication, 1, 3, 4 * 1024 * 1024);
            var genIndex4v0 = new TestApplication.AddonParameter(IdForApplication, 0, 4, 1 * 1024 * 1024);
            var genIndex4v1 = new TestApplication.AddonParameter(IdForApplication, 1, 4, 5 * 1024 * 1024);

            // upload executor to td1 server by ContentsUploader.
            D4cHelper.NspUploader uploader = new D4cHelper.NspUploader(
                intermediate,
                ProxyConfiguration,
                ServerEnvironment
            );

            // make contents for version 0.
            var uploadedVersion0 = TestApplication.MakeContents(intermediate,
                new List<TestApplication.GenerateParameter<int>>(3)
                {
                    genIndex2v0, genIndex3v0, genIndex4v0
                }
            );
            // contents alias for version 0.
            var addon2v0 = uploadedVersion0[0];
            var addon3v0 = uploadedVersion0[1];
            var addon4v0 = uploadedVersion0[2];

            // upload execution for version 0.
            uploader.Publish(CliTokenAccount, uploadedVersion0.ToNspPaths(),
                D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(addon2v0), 15 * 60);

            // make contents for version 0.
            var uploadedVersion1 = TestApplication.MakeContents(intermediate,
                new List<TestApplication.GenerateParameter<int>>(3)
                {
                    genIndex2v1, genIndex3v1, genIndex4v1
                }
            );
            // contents alias for version 1.
            var addon2v1 = uploadedVersion1[0];
            var addon3v1 = uploadedVersion1[1];
            var addon4v1 = uploadedVersion1[2];

            // upload execution for version 1.
            uploader.Publish(CliTokenAccount, uploadedVersion1.ToNspPaths(),
                D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(addon2v1), 15 * 60);
        }


        //!----------------------------------------------------------------------------
        /// <summary>
        /// 【UploadData】
        /// nn::aocライブラリ経由での OnCardAoc アクセス機能の検証のアップロードデータ
        /// 単体実行時で 3 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-69838")]
        public void UploadDataOfTestForOnCardAocAccessLibrary()
        {
            var UploadTimeout = new System.TimeSpan(0, 5, 0);         // 失敗時のリトライも含めて長めに設定

            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            var intermediate = GenerateIntermediateDirectoryAsMethod(executor);

            // make contents
            //  AddOnContent: 0x01001a500005f0f6, version 0,      index 1
            //              :        〃         , version 65536,  index 1
            //              : 0x01001a500005f0f7, version 0,      index 2
            var owner = new ID64(0x01001a500005e0f5);
            var @params = new List<TestApplication.GenerateParameter<int>>();
            {
                // aoc
                var param0 = new TestApplication.AddonParameter(owner, 0, 1, 1 * 1024 * 1024);
                @params.Add(param0);
                var param1 = new TestApplication.AddonParameter(owner, 1, 1, 1 * 1024 * 1024);
                @params.Add(param1);
                var param3 = new TestApplication.AddonParameter(owner, 0, 2, 1 * 1024 * 1024);
                @params.Add(param3);
            }
            var contents = TestApplication.MakeContents(intermediate, @params);
            var catalog = new GeneratedContentResult.TypeCategorizedCatalog(contents);
            var aoc_i1_v0a = catalog.GetTypedCatalog(ContentMeta.Type.AddOnContent)[0];
            var aoc_i1_v1 = catalog.GetTypedCatalog(ContentMeta.Type.AddOnContent)[1];
            var aoc_i2_v0 = catalog.GetTypedCatalog(ContentMeta.Type.AddOnContent)[2];

            // upload contents
            var uploader = new D4cHelper.NspUploader(
                intermediate,
                ProxyConfiguration,
                ServerEnvironment);

            var account = CliTokenAccount;
            var timeout = UploadTimeout.TotalSeconds;
            uploader.Publish(account, aoc_i1_v0a.NspPath,
                D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(aoc_i1_v0a), timeout);
            uploader.Publish(account, aoc_i1_v1.NspPath,
                D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(aoc_i1_v1), timeout);
            uploader.Publish(account, aoc_i2_v0.NspPath,
                D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(aoc_i2_v0), timeout);
        }


        //!----------------------------------------------------------------------------
        /// <summary>
        /// 【UploadData】
        /// NotEnoughSpace, Suspended, SystemUpdateRequired な状態のタスクに対して、
        /// Add したら上記状態が解除され、ダウンロードが完了することの確認のアップロードデータ
        /// 単体実行時で 3 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-69838")]
        public void UploadDataOfTestAddAfterStateChanged()
        {
            // make contents
            //  Application : 0x01001a500005e100, version 0
            //  AddOnContent: 0x01001a500005f101, version 0,      index 1
            ID64 applicationId = new ID64(0x01001a500005e100);

            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

            // テスト用データの生成
            string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

            var app = new TestApplication.ApplicationParameter(applicationId, 0, 1 * 1024 * 1024);
            app.ChangeRequiredVersion(2);
            var aoc1 = new TestApplication.AddonParameter(applicationId, 0, 1, 1024 * 1024);

            var request = new List<TestApplication.GenerateParameter<int>>() { app, aoc1 };
            var uploadContents = TestApplication.MakeContents(intermediate, request);
            var typedCatalog = new GeneratedContentResult.TypeCategorizedCatalog(uploadContents);
            var aocs = typedCatalog.GetTypedCatalog(ContentMeta.Type.AddOnContent);
            var generatedAoc1 = aocs[0];

            D4cHelper.NspUploader uploader = new D4cHelper.NspUploader(intermediate,
                    ProxyConfiguration, ServerEnvironment);
            uploader.Publish(CliTokenAccount, uploadContents[0].NspPath);
            uploader.Publish(CliTokenAccount, generatedAoc1.NspPath, D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(generatedAoc1));
        }


        //!----------------------------------------------------------------------------
        /// <summary>
        /// 【UploadData】
        /// RequestUpdateApplication トリガでのダウンロードコンテンツ整合性検証テスト。のアップロードデータ
        /// 単体実行時で 7 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-69838")]
        public void UploadDataOfTestForRequestUpdateApplication()
        {
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            var intermediate = GenerateIntermediateDirectoryAsMethod(executor);

            // 本テストのメタ構成は以下の通り。
            // Type         : ID                    : old vesion        : latest  version
            // Application  : 0x01003ab001e30002    : 0                 : N/A
            // AoC index 2  : 0x01003ab001e31004    : 0                 : 65536 ( 0x10000 )
            // Patch        : 0x01003ab001e30802    : 65536 ( 0x10000 ) : 131072 ( 0x20000 )
            var IdForApplication = new ID64(0x01003ab001e30002);

            // make contents.
            var genApp = new TestApplication.ApplicationParameter(IdForApplication, 0, 1 * 1024 * 1024);
            var genAoc_i2v0 = new TestApplication.AddonParameter(IdForApplication, 0, 2, 64 * 1024);
            var genAoc_i2v1 = new TestApplication.AddonParameter(IdForApplication, 1, 2, 128 * 1024);
            genApp.AddPatches(1, 2);
            genApp.UseSmallCode = true;

            // upload executor to td1 server by ContentsUploader.
            D4cHelper.NspUploader uploader = new D4cHelper.NspUploader(
                intermediate,
                ProxyConfiguration,
                ServerEnvironment
            );

            var uploadContents = TestApplication.MakeContents(intermediate,
                new List<TestApplication.GenerateParameter<int>>(3)
                {
                    genApp, genAoc_i2v0, genAoc_i2v1
                }
            );
            // カテゴライズ
            var typedCatalog = new GeneratedContentResult.TypeCategorizedCatalog(uploadContents);
            var aocs = typedCatalog.GetTypedCatalog(ContentMeta.Type.AddOnContent);
            var apps = typedCatalog.GetTypedCatalog(ContentMeta.Type.Application);
            var patches = typedCatalog.GetTypedCatalog(ContentMeta.Type.Patch);

            // contents alias
            var app = apps[0];
            var patch_v1 = patches.FindFromVersion(0x10000);
            var patch_v2 = patches.FindFromVersion(0x20000);
            var aoc_i2v0 = aocs.FindFromVersion(0x00000);
            var aoc_i2v1 = aocs.FindFromVersion(0x10000);
            if (null == app || null == patch_v1 || null == patch_v2 || null == aoc_i2v0 || null == aoc_i2v1)
            {
                throw new UnexpectFailureException("MakeContents operation failed.");
            }

            // uploading.
            uploader.RejectRoms(CliTokenAccount, IdForApplication);
            uploader.Publish(CliTokenAccount, app.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.WithApproved, 10 * 60);
            uploader.Publish(CliTokenAccount, patch_v1.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.Patch, 10 * 60);
            uploader.Publish(CliTokenAccount, patch_v2.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.Patch, 10 * 60);
            uploader.Publish(CliTokenAccount, aoc_i2v0.NspPath, D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(aoc_i2v0), 10 * 60);
            uploader.Publish(CliTokenAccount, aoc_i2v1.NspPath, D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(aoc_i2v1), 10 * 60);
        }


        //!----------------------------------------------------------------------------
        /// <summary>
        /// 【UploadData】
        /// RequestUpdateApplication の配信停止コンテンツのダウンロード不具合検証テスト
        /// のアップロードデータ
        /// 単体実行時で 3 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-69838")]
        public void UploadDataOfTestForRequestUpdateApplicationVersionFetch()
        {
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            var intermediate = GenerateIntermediateDirectoryAsMethod(executor);

            // 本テストのメタ構成は以下の通り。
            // Type         : ID                    : client vesion         : server version
            // Application  : 0x01003ab001e30003    : N/A                   : 0
            // Patch        : 0x01003ab001e30803    : 131072 ( 0x20000 )    : 65536 ( 0x10000 )
            var IdForApplication = new ID64(0x01003ab001e30003);

            // make contents.
            var genApp0 = new TestApplication.ApplicationParameter(IdForApplication, 0, 1 * 1024 * 1024);
            genApp0.AddPatches(1, 2);
            genApp0.UseSmallCode = true;

            // upload executor to td1 server by ContentsUploader.
            D4cHelper.NspUploader uploader = new D4cHelper.NspUploader(
                intermediate,
                ProxyConfiguration,
                ServerEnvironment
            );

            var uploadContents = TestApplication.MakeContents(intermediate, genApp0);

            // カテゴライズ
            var typedCatalog = new GeneratedContentResult.TypeCategorizedCatalog(uploadContents);
            var apps = typedCatalog.GetTypedCatalog(ContentMeta.Type.Application);
            var patches = typedCatalog.GetTypedCatalog(ContentMeta.Type.Patch);

            // contents alias
            var app_v0 = apps.FindFromVersion(0x00000);
            var patch_v1 = patches.FindFromVersion(0x10000);
            var patch_v2 = patches.FindFromVersion(0x20000);
            if (null == app_v0 || null == patch_v1 || null == patch_v2)
            {
                throw new UnexpectFailureException("MakeContents operation failed.");
            }

            // uploading.
            uploader.RejectRoms(CliTokenAccount, IdForApplication);
            uploader.Publish(CliTokenAccount, app_v0.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.WithApproved, 10 * 60);
            uploader.Publish(CliTokenAccount, patch_v1.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.Patch, 10 * 60);
        }


        //!----------------------------------------------------------------------------
        /// <summary>
        /// 【UploadData】
        /// コンテンツ自動更新の検証。のアップロードデータ
        /// 単体実行時で 4 ～ 6 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-69838")]
        public void UploadDataOfTestForAutoUpdateContents()
        {
            var UploadTimeout = new System.TimeSpan(0, 5, 0);         // 失敗時のリトライも含めて長めに設定

            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            var intermediate = GenerateIntermediateDirectoryAsMethod(executor);

            // make contents
            //  AddOnContent: 0x01001a500005f0f1, version 0,      index 1
            //                                    version 65536,  index 1
            //                                    version 131072, index 1
            var owner = new ID64(0x01001a500005e0f0);
            var @params = new List<TestApplication.GenerateParameter<int>>();
            {
                // aoc
                var param0 = new TestApplication.AddonParameter(owner, 0, 1, 1 * 1024 * 1024);
                @params.Add(param0);
                var param1 = new TestApplication.AddonParameter(owner, 1, 1, 1 * 1024 * 1024);
                @params.Add(param1);
                var param2 = new TestApplication.AddonParameter(owner, 2, 1, 1 * 1024 * 1024);
                @params.Add(param2);
            }

            var contents = TestApplication.MakeContents(intermediate, @params);
            var catalog = new GeneratedContentResult.TypeCategorizedCatalog(contents);
            var aoc_v0 = catalog.GetTypedCatalog(ContentMeta.Type.AddOnContent)[0];
            var aoc_v1 = catalog.GetTypedCatalog(ContentMeta.Type.AddOnContent)[1];
            var aoc_v2 = catalog.GetTypedCatalog(ContentMeta.Type.AddOnContent)[2];

            // upload contents
            var uploader = new D4cHelper.NspUploader(
                intermediate,
                ProxyConfiguration,
                ServerEnvironment);

            var account = CliTokenAccount;
            var timeout = UploadTimeout.TotalSeconds;
            uploader.Publish(account, aoc_v0.NspPath,
                D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(aoc_v0), timeout);
            uploader.Publish(account, aoc_v1.NspPath,
                D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(aoc_v1), timeout);
            uploader.Publish(account, aoc_v2.NspPath,
                D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(aoc_v2), timeout);
        }



        //!----------------------------------------------------------------------------
        /// <summary>
        /// 【UploadData】
        /// コンテンツ配信の開始・停止テスト。のアップロードデータ
        /// 単体実行時で 3 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-69838")]
        public void UploadDataOfTestForStartAndStopDelivery()
        {
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            var intermediate = GenerateIntermediateDirectoryAsMethod(executor);

            // make contents
            //  Application : 0x01001a500005e0f1, version 0
            //  Patch       : 0x01001a500005e8f1, version 65536
            var param = new TestApplication.ApplicationParameter(
                new ID64(0x01001a500005e0f1), 0, 5 * 1024 * 1024);
            param.AddPatch(1);
            param.UseSmallCode = true;

            var contents = TestApplication.MakeContents(
                intermediate, new List<TestApplication.GenerateParameter<int>>() { param });

            var catalog = new GeneratedContentResult.TypeCategorizedCatalog(contents);
            var app = catalog.GetTypedCatalog(ContentMeta.Type.Application)[0];
            var patch = catalog.GetTypedCatalog(ContentMeta.Type.Patch)[0];

            // upload contents
            var uploader = new D4cHelper.NspUploader(
                intermediate,
                ProxyConfiguration,
                ServerEnvironment);

            uploader.RejectRoms(CliTokenAccount, app.Identifier);
            uploader.Publish(CliTokenAccount, app.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.WithApproved, 5 * 60);
            uploader.Publish(CliTokenAccount, patch.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.Patch, 5 * 60);
        }


        //!----------------------------------------------------------------------------
        /// <summary>
        /// 【UploadData】
        /// コンテンツ配信必須システムバージョンが検知されないことを確認。のアップロードデータ
        /// 単体実行時で 2 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-69838")]
        public void UploadDataOfTestForCheckContentDelivery()
        {
            var UploadTimeout = new System.TimeSpan(0, 5, 0);         // 失敗時のリトライも含めて長めに設定

            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

            // make contents
            //  Application : 0x01001a500005e0f2, version 0
            //  Patch       : 0x01001a500005e8f2, version 65536
            var param = new TestApplication.ApplicationParameter(
                new ID64(0x01001a500005e0f2), 0, 5 * 1024 * 1024);
            param.AddPatch(1);
            param.UseSmallCode = true;

            var contents = TestApplication.MakeContents(
                intermediate, new List<TestApplication.GenerateParameter<int>>() { param });

            var catalog = new GeneratedContentResult.TypeCategorizedCatalog(contents);
            var app = catalog.GetTypedCatalog(ContentMeta.Type.Application)[0];
            var patch = catalog.GetTypedCatalog(ContentMeta.Type.Patch)[0];

            // upload contents
            var uploader = new D4cHelper.NspUploader(
                intermediate,
                ProxyConfiguration,
                ServerEnvironment);

            var account = CliTokenAccount;
            var timeout = UploadTimeout.TotalSeconds;
            uploader.RejectRoms(account, app.Identifier);
            uploader.Publish(account, app.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.WithApproved, timeout);
            uploader.Publish(account, patch.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.Patch, timeout);
        }


        //!----------------------------------------------------------------------------
        /// <summary>
        /// 【UploadData】
        /// アプリがオンライン中でもタスク登録・管理データ受信できることを確認。のアップロードデータ
        /// 単体実行時で 3 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-69838")]
        public void UploadDataOfTestForVerifyWhileOnlineApplication()
        {
            var UploadTimeout = new System.TimeSpan(0, 5, 0);         // 失敗時のリトライも含めて長めに設定

            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

            // make contents
            //  Application : 0x01001a500005e0f4, version 0
            //  Patch       : 0x01001a500005e8f4, version 65536
            var param = new TestApplication.ApplicationParameter(
                new ID64(0x01001a500005e0f4), 0, 5 * 1024 * 1024);
            param.AddPatch(1);
            param.UseSmallCode = true;

            var contents = TestApplication.MakeContents(
                intermediate, new List<TestApplication.GenerateParameter<int>>() { param });

            var catalog = new GeneratedContentResult.TypeCategorizedCatalog(contents);
            var app = catalog.GetTypedCatalog(ContentMeta.Type.Application)[0];
            var patch = catalog.GetTypedCatalog(ContentMeta.Type.Patch)[0];

            // upload contents
            var uploader = new D4cHelper.NspUploader(
                intermediate,
                ProxyConfiguration,
                ServerEnvironment);

            var account = CliTokenAccount;
            var timeout = UploadTimeout.TotalSeconds;
            uploader.RejectRoms(account, app.Identifier);
            uploader.Publish(account, app.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.WithApproved, timeout);
            uploader.Publish(account, patch.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.Patch, timeout);
        }


        //!----------------------------------------------------------------------------
        /// <summary>
        /// 【UploadData】
        /// 「消えないプレースホルダ」問題の検証テストのアップロードデータ
        /// 単体実行時で 2 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-69838")]
        public void UploadDataOfTestForPlaceHolderCleanupAtTaskCancelled()
        {
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            var intermediate = GenerateIntermediateDirectoryAsMethod(executor);

            // 本テストのメタ構成は以下の通り。
            // Type         : ID                    : client vesion         : server version
            // Application  : 0x01003ab001e30003    : N/A                   : N/A
            // AoC index 2  : 0x01003ab001e31005    : N/A                   : 0
            // AoC index 3  : 0x01003ab001e31006    : N/A                   : 0
            ulong kbSizeSmall = 3 * 1024;
            ulong kbSizeLarge = 7 * 1024;
            var IdForApplication = new ID64(0x01003ab001e30003);
            var genAocSmall = new TestApplication.AddonParameter(IdForApplication, 0, 2, kbSizeSmall * 1024);
            var genAocLarge = new TestApplication.AddonParameter(IdForApplication, 0, 3, kbSizeLarge * 1024);

            // upload executor to td1 server by ContentsUploader.
            D4cHelper.NspUploader uploader = new D4cHelper.NspUploader(
                intermediate,
                ProxyConfiguration,
                ServerEnvironment
            );


            var uploadContents = TestApplication.MakeContents(intermediate,
                new List<TestApplication.GenerateParameter<int>>(2) { genAocSmall, genAocLarge }
            );
            var uploadOption = D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(0);
            uploader.Publish(CliTokenAccount, uploadContents.ToNspPaths(), uploadOption, 10 * 60);
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// 【UploadData】
        /// タスクのまとめて再開機能のテストのアップロードデータ
        /// 単体実行時で 5 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-69838")]
        public void UploadDataOfTestForResumeAllTasks()
        {
            //既存テストでアップロードしていなかった為、アップロードしない。
#if false
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            var intermediate = GenerateIntermediateDirectoryAsMethod(executor);
            var method = MethodBase.GetCurrentMethod();

            // 本テストのメタ構成は以下の通り。
            // 連番Index | Application ID     | Patch ID           | Version |
            //         0 | 0x01003ab001e30050 | 0x01003ab001e30850 | 0, 1, 2 |
            //         : |          :         |          :         |    :    |
            //         : |          :         |          :         |    :    |
            //         2 | 0x01003ab001e30052 | 0x01003ab001e30853 | 0, 1, 2 |

            uint appCount = 3U;
            var IdForApplication = new ID64(0x01003ab001e30050);

            // make parameter of contents.
            var gens = new List<TestApplication.GenerateParameter<int>>((int)appCount);
            for (uint index = 0; index < appCount; ++index)
            {
                var now = IdForApplication + index;
                var app = new ApplicationParameterWithPatchSizeMap(now, 0, 32);
                app.PatchSizeMap.Add(2, 10 * 1024 * 1024);
                app.AddPatches(1, new TestApplication.PatchContexture<int>(2, AuthoringExecutor.Patch.FlagOption.UseDelta));
                app.UseSmallCode = true;
                gens.Add(app);
            }

            // make the title resource contexture with call the MakeTestApplication.
            var uploadContents = TestApplication.MakeContents(intermediate, gens);
            var titles = new GeneratedContentResult.TitleCategorizedCatalog(uploadContents);

            // upload executor to td1 server by ContentsUploader.
            D4cHelper.NspUploader uploader = new D4cHelper.NspUploader(
                intermediate,
                ProxyConfiguration,
                ServerEnvironment
            );

            // uploading. ( 5min as timeout of 1 nsp uploading. )
            uploader.Publish(CliTokenAccount, titles, 5 * 60);
#endif
        }

    }
}
