﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
/// Notice.
/// PUX拠点から、VisualStudio のテストエクスプローラ経由テストの場合、*.runsettings ファイルの TestRunParameters に以下パラメタを宣言してください.
/// NCL/CI 及び、NN_TEST_SCRIPT_PROXY_CONFIGURATION 環境変数が設定されたコマンドライン実施の場合は宣言は不要です.
/// <Parameter name = "EnableRunTestOnIDE_FromPUX" value="true" />

using CsTestAssistants;
using System.Reflection;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace testNs_ShopDeliveryBasic
{
    using System.Linq;
    using NintendoServices = CsTestAssistants.NintendoServices;
    using ExpectEvaluator = ThrowEvaluator<UnexpectFailureException>;
    using System.Runtime.Serialization;

    /// <summary>
    /// テストクラス
    /// </summary>
    [TestClass]
    public class UnitMain : TestClassBase
    {
        // `TestContext` プロパティを定義しておけば、UnitTestフレームワークで勝手にインスタンスが提供される.
        public TestContext TestContext { get; set; }

        private const string LogPrefix = "[eShopBasic]";

        //!----------------------------------------------------------------------------
        /// <summary>
        /// クラスが生成され、呼び出される前に一度だけ実施。
        /// </summary>
        /// <param name="context"></param>
        //!----------------------------------------------------------------------------
        [ClassInitialize]
        public static void TestClassInitialize(TestContext context)
        {
            ClassInitialize(context, LogPrefix);
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// クラスが破棄されるタイミングで一度だけ実施。
        /// </summary>
        //!----------------------------------------------------------------------------
        [ClassCleanup]
        public static void TestClassCleanup()
        {
            ClassCleanup();
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// `TestProperty属性の "JIRA" キーの値` と `Sigloコンフィギュレーションコンテキスト` に依存した作業用ワーキングディレクトリパスを生成します.
        /// </summary>
        /// <param name="context">Sigloコンフィギュレーションコンテキスト</param>
        /// <param name="methodName">
        /// 参照対象TestProperty属性を保持するメソッド名.
        /// 指定を省略した場合、呼び出されたカレントメソッド名が自動で選択されます.
        /// 明示的な null 指定の場合、もしくは "JIRA" キーの TestPropertyが見つからない場合には、TestPropertyに依存しないベース作業用ディレクトリが返却されます。
        /// </param>
        /// <returns>
        /// TestProperty( "JIRA", * ) に依存した作業用ディレクトリパス、もしくはベース作業用ディレクトリパス。
        /// </returns>
        //!----------------------------------------------------------------------------
        private string GenerateIntermediateDirectoryAsMethod(SigloHelper.Configuration.Context context,
            [System.Runtime.CompilerServices.CallerMemberName] string methodName = null)
        {
            var runAssembly = Assembly.GetExecutingAssembly();
            var asmTitle = (AssemblyTitleAttribute)System.Attribute.GetCustomAttribute(runAssembly, typeof(AssemblyTitleAttribute));
            return GenerateIntermediateDirectory(context, asmTitle.Title, methodName);
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// ターゲットログのマッチング
        /// </summary>
        /// <param name="executor"></param>
        /// <param name="expected"></param>
        /// <returns></returns>
        //!----------------------------------------------------------------------------
        private bool MatchTargetLog(SigloHelper.CommodityExecutor.Context executor, string expected)
        {
            // 直前の実行結果出力から接頭辞 [target] を削除
            var output = Regex.Replace(
                executor.OutputStream.Standard.ToString(),
                @"^\[target\] ", "", RegexOptions.Multiline);

            // 期待されるパターンとのマッチング
            var match = Regex.Match(output, expected);
            Log.WriteLine("target log expected pattern : {0}", expected);
            Log.WriteLine("target log matching result : {0}", match.Success);
            return match.Success;
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// テストメソッド: SetupSequence
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        public void SetupSequence()
        {
            using (new TestMethodLog())
            {
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

                // CLIトークン事前参照.
                // 並列要求時に有効期限切れの場合、取得競合で失敗するため.
                var intermediate = GenerateIntermediateDirectoryAsMethod(executor);
                D4cHelper.Configuration.CliToken.QueryLatestValidateToken(intermediate, CliTokenAccount, ProxyConfiguration, ServerEnvironment);

                // td1( dev6 ), ネットワーク設定インポート.
                SetupSdevEnvironments(executor);

                // ゲームカード消去
                var hasInsertedGameCard = executor.RunDevMenuCommandSystem($"gamecard status", @"^Inserted$");
                if (hasInsertedGameCard) { executor.RunDevMenuCommandSystem("gamecard erase"); }
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// テストメソッド: TestForAccountDeviceRegistration
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-42738")]
        public void TestForAccountDeviceRegistration()
        {
            var method = MethodBase.GetCurrentMethod();
            Log.WriteLine(TestMethodLogBegin, method.Name);

            var account = NintendoAccount;
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

            try
            {
                // # （ローカルの）ユーザを追加する。
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("account add --register_nsa"));
                // # ニンテンドーアカウントをユーザに紐付ける
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(string.Format("account link --index 0 --id {0} --password {1}", account.ID, account.PASSWORD)));
                // # デバイスアカウントをサーバから解除して、ローカルからも削除する。クリーンアップ用。
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("shop unregister-device-account"));
                // # デバイスアカウントが未登録であることを確認する。
                ExpectEvaluator.IsFalse(executor.RunDevMenuCommandSystem("shop device-registration-info"));
                // # デバイスアカウントをサーバに登録して、ローカルにトークンを保存する。
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("shop register-device-account"));
                // # デバイスアカウントが登録済であることを確認する。
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("shop device-registration-info"));
                // # ショップのアカウント状態を確認する (ここでバーチャルアカウントが作られる)
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("shop shop-account-status 0"));
                // # 機器認証する
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("shop link-device 0"));
                // # 機器認証解除する
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("shop unlink-device 0"));
                // # アカウントを削除し、ニンテンドーアカウント連携も切る
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("account clear_all"));
                // # デバイスアカウントをサーバから解除して、ローカルからも削除する。クリーンアップ用。
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("shop unregister-device-account"));
                // # デバイスアカウントが未登録であることを確認する。
                ExpectEvaluator.IsFalse(executor.RunDevMenuCommandSystem("shop device-registration-info"));
            }
            finally
            {
                Log.WriteLineAsIs(System.Environment.NewLine);
                Log.WriteLine(TestMethodLogFinally, method.Name);

                // # 機器認証解除する
                ThrowFrameworks.SkipThrow(() => { executor.RunDevMenuCommandSystem("shop unlink-device-all"); });
                // # アカウントを削除し、ニンテンドーアカウント連携も切る
                ThrowFrameworks.SkipThrow(() => { executor.RunDevMenuCommandSystem("account clear_all"); });
                // # デバイスアカウントをサーバから解除して、ローカルからも削除する。クリーンアップ用。
                ThrowFrameworks.SkipThrow(() => { executor.RunDevMenuCommandSystem("shop unregister-device-account"); });
            }

            Log.WriteLine(TestMethodLogFinish, method.Name);
        }


        //!----------------------------------------------------------------------------
        /// <summary>
        /// テストメソッド: TestForDTLMultiCombination
        /// 複数コンテンツ種別の組み合わせと、それに対する追加コンテンツの動的追加
        /// 単体実行時で 3 分程度要します。
        /// UploadData_app_patch_v1_aoc_i1v0_i2v0 のデータ使用
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-46126")]
        public void TestForDTLMultiCombination()
        {
            var method = MethodBase.GetCurrentMethod();
            Log.WriteLine(TestMethodLogBegin, method.Name);

            // 本来のIDは 0x01003ab001e30000 ですが、0x01003ab001e30800 の Patch の証明書エラー回避のため拡張IDにしています。
            // そのため、本テストのメタ構成は以下の通り。
            // Application : 0x01003ab001e30001, version 0
            // Patch       : 0x01003ab001e30801, version 65536
            // AoC index 1 : 0x01003ab001e31002, version 0
            // AoC index 2 : 0x01003ab001e31003, version 0
            var IdForApplication = new ID64(0x01003ab001e30001);
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

            string comPush = "application push-download-task-list";
            string comVerify = $"application verify 0x{IdForApplication}";
            string comWait = $"application wait-download 0x{IdForApplication}";
            string comWaitPrepare = $"application wait-download 0x{IdForApplication} --download-progress 0/1";

            string comPushAndWait = $"--time {comPush} {{0}} AND {comWait} AND {comVerify}";
            string comPushAndPrepareAbortAndPush = $"--time {comPush} {{0}} AND {comWaitPrepare} AND {comPush} {{1}}";
            string comLastWait = $"--time {comWait} AND {comVerify}";

            string comLaunchApp = $"application launch 0x{IdForApplication}";

            string comInfoClear = "application clear-task-status-list AND application uninstall --all AND application reset-required-version --all";

            try
            {
                // 作業用ディレクトリ
                string intermediate = GenerateIntermediateDirectoryAsMethod(executor);
                FileHelper.RemoveDirectory(intermediate);

                // make contents.
                var genApp = new TestApplication.ApplicationParameter(IdForApplication, 0, 5 * 1024 * 1024);
                var genAddonA = new TestApplication.AddonParameter(IdForApplication, 0, 1, 64 * 1024);
                var genAddonB = new TestApplication.AddonParameter(IdForApplication, 0, 2, 64 * 1024);
                genApp.AddPatch(1);
                genApp.UseSmallCode = true;
                var request = new List<TestApplication.GenerateParameter<int>>(3) { genApp, genAddonA, genAddonB };
                var uploadContents = TestApplication.MakeContents(intermediate, request);

                // カテゴライズ
                var typedCatalog = new GeneratedContentResult.TypeCategorizedCatalog(uploadContents);
                var aocs = typedCatalog.GetTypedCatalog(ContentMeta.Type.AddOnContent);
                var apps = typedCatalog.GetTypedCatalog(ContentMeta.Type.Application);
                var patches = typedCatalog.GetTypedCatalog(ContentMeta.Type.Patch);
                if (null == aocs || null == apps || null == patches)
                {
                    throw new UnexpectFailureException("MakeContents operation failed.");
                }
                GeneratedContentResult app = apps[0];
                GeneratedContentResult patch = patches[0];
                GeneratedContentResult addonA = aocs[0];
                GeneratedContentResult addonB = aocs[1];
                if (null == app || null == patch || null == addonA || null == addonB)
                {
                    throw new UnexpectFailureException("MakeContents operation failed.");
                }

                // クリア
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(comInfoClear));

                // case.1 アプリ単体をダウンロードし、完了をテスト
                {
                    string json = System.IO.Path.Combine(intermediate, "case1.dtl.json");
                    var dtl = new DownloadTaskList();
                    var task = dtl.Create(IdForApplication);
                    task.AddNewTitle(app);
                    dtl.ToJson(json);

                    // DTL push and wait.
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(string.Format(comPushAndWait, json)));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommand(comLaunchApp));
                }

                // クリア
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(comInfoClear));

                // case.2 アプリ・パッチを同時ダウンロードし、完了をテスト
                {
                    string json = System.IO.Path.Combine(intermediate, "case2.dtl.json");
                    var dtl = new DownloadTaskList();
                    var task = dtl.Create(IdForApplication);
                    task.AddNewTitle(app).AddNewTitle(patch);
                    dtl.ToJson(json);

                    // DTL push and wait.
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(string.Format(comPushAndWait, json)));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommand(comLaunchApp));
                }

                // クリア
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(comInfoClear));

                // case.3 アプリ・パッチ・追加コンテンツAを同時ダウンロードし、完了をテスト
                {
                    string json = System.IO.Path.Combine(intermediate, "case3.dtl.json");
                    var dtl = new DownloadTaskList();
                    var task = dtl.Create(IdForApplication);
                    task.AddNewTitle(app).AddNewTitle(patch).AddNewTitle(addonA);
                    dtl.ToJson(json);

                    // DTL push and wait.
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(string.Format(comPushAndWait, json)));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommand(comLaunchApp));
                }

                // クリア
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(comInfoClear));

                // case.4 アプリ単体をダウンロード中、追加コンテンツBを動的追加し、全てのダウンロード完了をテスト
                {
                    var json1 = System.IO.Path.Combine(intermediate, "case4_1.dtl.json");
                    var json2 = System.IO.Path.Combine(intermediate, "case4_2.dtl.json");
                    var dtl = new DownloadTaskList();
                    dtl.Create(IdForApplication).AddNewTitle(app);
                    dtl.ToJson(json1);
                    dtl.Create(IdForApplication).AddNewTitle(addonB);
                    dtl.ToJson(json2);

                    // DTL push and wait.
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(string.Format(comPushAndPrepareAbortAndPush, json1, json2)));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(string.Format(comLastWait)));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommand(comLaunchApp));
                }

                // クリア
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(comInfoClear));

                // case.5 アプリ・パッチを同時ダウンロード中、追加コンテンツBを動的追加し、全てのダウンロード完了をテスト
                {
                    var json1 = System.IO.Path.Combine(intermediate, "case5_1.dtl.json");
                    var json2 = System.IO.Path.Combine(intermediate, "case5_2.dtl.json");
                    var dtl = new DownloadTaskList();
                    dtl.Create(IdForApplication).AddNewTitle(app).AddNewTitle(patch);
                    dtl.ToJson(json1);
                    dtl.Create(IdForApplication).AddNewTitle(addonB);
                    dtl.ToJson(json2);

                    // DTL push and wait.
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(string.Format(comPushAndPrepareAbortAndPush, json1, json2)));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(string.Format(comLastWait), 15 * 60));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommand(comLaunchApp));
                }

                // クリア
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(comInfoClear));

                // case.6 アプリ・パッチ・追加コンテンツを同時ダウンロード中、追加コンテンツBを動的追加し、全てのダウンロード完了をテスト
                {
                    var json1 = System.IO.Path.Combine(intermediate, "case6_1.dtl.json");
                    var json2 = System.IO.Path.Combine(intermediate, "case6_2.dtl.json");
                    var dtl = new DownloadTaskList();
                    dtl.Create(IdForApplication).AddNewTitle(app).AddNewTitle(patch).AddNewTitle(addonA);
                    dtl.ToJson(json1);
                    dtl.Create(IdForApplication).AddNewTitle(addonB);
                    dtl.ToJson(json2);

                    // DTL push and wait.
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(string.Format(comPushAndPrepareAbortAndPush, json1, json2)));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(string.Format(comLastWait), 20 * 60));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommand(comLaunchApp));
                }
            }
            finally
            {
                ThrowFrameworks.SkipThrow(() => { executor.RunDevMenuCommandSystem(comInfoClear); });
            }

            Log.WriteLine(TestMethodLogFinish, method.Name);
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// テストメソッド: TestForPurchaseDemoTicket
        /// 製品化されたアプリ・パッチ・追加コンテンツの購入・ダウンロード検証
        /// 単体実行時で 3 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-46243")]
        public void TestForPurchaseDemoTicket()
        {
            using (var scope = new TestMethodLog())
            {
                var method = MethodBase.GetCurrentMethod();

                // 作業用ディレクトリ
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                // オーナーアプリケーション: 0x01003ab001e32000
                var IdForApplication = new ID64(0x01003ab001e32000);

                // 購入者アカウント
                var account = NintendoAccount;

                // make contents.
                var genApp = new TestApplication.ApplicationParameter(IdForApplication, 0, 1 * 1024 * 1024);
                var genAddonA = new TestApplication.AddonParameter(IdForApplication, 0, 1, 64 * 1024);
                genApp.AddPatch(1);
                genApp.UseSmallCode = true;                 // 起動チェック用に small-code
                genApp.ChangeTicketEncryption(true);      // 製品化指定
                genAddonA.ChangeTicketEncryption(true);   // 製品化指定
                var request = new List<TestApplication.GenerateParameter<int>>(2) { genApp, genAddonA };
                var uploadContents = TestApplication.MakeContents(intermediate, request);

                // カテゴライズ
                var typedCatalog = new GeneratedContentResult.TypeCategorizedCatalog(uploadContents);
                var aocs = typedCatalog.GetTypedCatalog(ContentMeta.Type.AddOnContent);
                var apps = typedCatalog.GetTypedCatalog(ContentMeta.Type.Application);
                var patches = typedCatalog.GetTypedCatalog(ContentMeta.Type.Patch);
                if (null == aocs || null == apps || null == patches)
                {
                    throw new UnexpectFailureException("MakeContents operation failed.");
                }
                GeneratedContentResult app = apps[0];
                GeneratedContentResult patch = patches[0];
                GeneratedContentResult addon = aocs[0];
                if (null == app || null == patch || null == addon)
                {
                    throw new UnexpectFailureException("MakeContents operation failed.");
                }

                // finally: 機器認証解除する
                scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("shop unlink-device-all"); });
                // finally: アカウントを削除し、ニンテンドーアカウント連携も切る
                scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("account clear_all"); });
                // finally: デバイスアカウントをサーバから解除して、ローカルからも削除する。クリーンアップ用。
                scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("shop unregister-device-account"); });

                // # TSL＆インストール状態をクリア
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all"));
                // # （ローカルの）ユーザを追加して、ニンテンドーアカウントをユーザに紐付ける
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $"account add --register_nsa AND account link --index 0 --id {account.ID} --password {account.PASSWORD}"
                ));
                // # デバイスアカウントをサーバに登録( ローカルにトークンを保存 ), ショップのアカウント状態を確認する(バーチャルアカウント作成)
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("shop register-device-account AND shop shop-account-status 0"));
                // # DTLクリア確認
                ExpectEvaluator.IsTrue(D4cHelper.ClearDownloadTaskListOnShopServer(executor));
                // # 機器認証する
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("shop link-device 0"));

                // app / aoc 体験版購入 ( DTL が更新され、npns でプッシュ通知され、非同期にダウンロードが始まります。)
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $"shop download-demo 0 --id 0x{IdForApplication} AND shop download-demo 0 --id 0x{addon.Identifier}"
                ));

                // ダウンロード待ち
                bool resultWaitDownload = executor.RunDevMenuCommandSystem(
                    $"--time application wait-download 0x{IdForApplication} AND application verify 0x{IdForApplication}"
                );

                // 以下ログからの成否確認用
                // 受信したDTLをログに残す
                ThrowFrameworks.SkipThrow(() => { executor.RunDevMenuCommandSystem("application request-download-task-list-data"); });
                // アプリケーション通知情報をログに残す
                ThrowFrameworks.SkipThrow(() => { executor.RunDevMenuCommandSystem("application notification-info"); });

                // 情報残した後、Assertチェック
                ExpectEvaluator.IsTrue(resultWaitDownload);

                // インストールチェック ( with Ticket rights status )
                var matchStorageAny = SigloHelper.ResultMatcher.ContentValue.Storage.Any;
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application content-meta-status 0x{IdForApplication} --check-rights"));
                ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ContentMetaStatusCheckRights.Find(
                    executor.OutputStream.Standard.ToString(),
                    new SigloHelper.ResultMatcher.ContentValue(app, matchStorageAny, SigloHelper.ResultMatcher.ContentValue.RightsCheck.PersonalizedRights),
                    new SigloHelper.ResultMatcher.ContentValue(patch, matchStorageAny, SigloHelper.ResultMatcher.ContentValue.RightsCheck.CommonRights),
                    new SigloHelper.ResultMatcher.ContentValue(addon, matchStorageAny, SigloHelper.ResultMatcher.ContentValue.RightsCheck.PersonalizedRights)
                ));

                // チケット検証 ( 念押し )
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"ticket list-all"));
                ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Ticket.ListAll.Find(
                    executor.OutputStream.Standard.ToString(),
                    app.Identifier,
                    patch.Identifier,
                    addon.Identifier
                ));

                // 起動チェック
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommand($"application launch 0x{IdForApplication}"));
            }
        }


        //!----------------------------------------------------------------------------
        /// <summary>
        /// テストメソッド: TestForStandardDTL
        /// 最大数(64個)ネットワークインストールタスク登録とダウンロード完了
        /// UploadData_app64_aoc_i1v0 のデータ使用
        /// 単体実行時で 46 ～ 50 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-43133")]
        public void TestForStandardDTL()
        {
            var method = MethodBase.GetCurrentMethod();
            Log.WriteLine(TestMethodLogBegin, method.Name);

            // Constants
            ID64 baseApplicationId = new ID64(0x01001a500005e0b0);
            int appVersion = 0;
            int appCount = 64;

            const int RETRY_KIND_NONE = 0;   // リトライの種類：初期値
            const int RETRY_KIND_PARTS = 1;  // リトライの種類：FAIL VIEW リスト出力にFailデータが存在する場合
            const int RETRY_KIND_ALL = 2;    // リトライの種類：上記以外失敗
            const int RETRY_COUNT_PARTS = 2; // リトライ回数：FAIL VIEW リスト出力にFailデータが存在する場合
            const int RETRY_COUNT_ALL = 1;   // リトライ回数：上記以外失敗時

            // Target command executor
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

            try
            {
                // 作業用ディレクトリ
                string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                // テンポラリ DTL-JSON
                string jsonFilePath = System.IO.Path.Combine(intermediate, "dtl.json");
                Log.WriteLine("DTL json file : {0}", jsonFilePath);

                // Addon contents upload. ( 0x01001a500005f0b1 の想定,  0x01001a500005e0b0 + 0x1000 + 1 )
                var addon = new TestApplication.AddonParameter(baseApplicationId, 0, 1, 64 * 1024);
                var addonContents = TestApplication.MakeContents(intermediate, addon);

                // DTL生成 ( application x64 )
                var dtl = new DownloadTaskList().MakeAsApplication(baseApplicationId, appCount, appVersion);
                // DTL追加 ( addon x1 )
                var addonTask = dtl.FindTaskFrom(baseApplicationId);
                foreach (var c in addonContents)
                {
                    addonTask.AddNewTitle(c);
                }
                var stream = dtl.ToJson(jsonFilePath);

                // ファイル内容の確認用ダンプ
                var jsonStream = System.IO.File.ReadAllText(jsonFilePath, EncodeContext.UTF8_NoBOM);
                Log.WriteLine("=== DTL json stream begin ===\n{0}\n=== DTL json stream end ===\n", jsonStream);
                ExpectEvaluator.IsTrue(jsonStream.Equals(stream));

                // ------------------------------------------------------------------------
                // DL失敗時のリトライを行う。
                // ------------------------------------------------------------------------
                System.Func<bool, string, string, bool> retryDownload = (downloaded, executionLog, preRetryJsonFilePath) =>
                {
                    var retryKind = RETRY_KIND_NONE;
                    uint failedCount = (uint)appCount + 1;  // 失敗したアプリの数初期値(TimeOutの時間計算に使用)、TimeOut時は全てリトライするため 最大値を設定する必要あり
                    var retryCount = 1; // 初期値はリトライの種類決定時に変更する

                    // リトライ用の DTL-JSON
                    string retryJsonFilePath = System.IO.Path.Combine(intermediate, "retryDtl.json");

                    while (!downloaded && retryCount > 0 )
                    {
                        // FAIL VIEW リスト出力にFailデータが存在するか確認する。
                        var mc = Regex.Match(
                            executionLog,
                            @"\[target\] Failed application download requests [\s\w\r\n\[\]{]*""applicationId"": ",
                            RegexOptions.Multiline
                            );

                        // 一部のファイルが失敗している場合
                        if (mc.Success)
                        {
                            if (retryKind == RETRY_KIND_NONE)
                            {
                                retryKind = RETRY_KIND_PARTS;
                                retryCount = RETRY_COUNT_PARTS; // リトライ回数の初期値
                            }
                            if (RETRY_KIND_PARTS == retryKind)
                            {
                                Log.WriteLine("Retry some data.\n");
                            }
                            else if (RETRY_KIND_ALL == retryKind)
                            {
                                // すべてリトライの後に来た場合
                                Log.WriteLine("Partial retry after retrying all.\n");
                            }
                            else
                            {
                                throw new UnexpectFailureException("A pattern that does not exist.");
                            }

                            // 一部のファイルが失敗している場合は、アプリの数を初期化。
                            failedCount = 0;

                            // TSLをクリアする
                            ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application clear-task-status-list"));

                            // ログから ApllicationId のマッチデータを取得する
                            var regApp = SigloHelper.ResultMatcher.Application.WaitDownloadAll.FindApplicationIds(executionLog);
                            var retryDtl = new DownloadTaskList();

                            while (regApp.Success)
                            {
                                failedCount++;
                                var aplicationId = regApp.Groups[1].Value;
                                ID64 retryApplicationId = new ID64(ulong.Parse(aplicationId.ToString(), System.Globalization.NumberStyles.HexNumber));
                                Log.WriteLine("Retry applicationId :{0}\n", retryApplicationId);

                                // タスクリストに、Appを追加
                                retryDtl.MakeAsApplication(retryApplicationId, 1, appVersion);

                                // baseApplicationId だけは、addonContentsを追加する。
                                if (retryApplicationId == baseApplicationId)
                                {
                                    var addonTask2 = retryDtl.FindTaskFrom(baseApplicationId);
                                    foreach (var c in addonContents)
                                    {
                                        addonTask2.AddNewTitle(c);
                                    }
                                }
                                regApp = regApp.NextMatch();
                            }
                            // DTL状態を JSON文字列にシリアライズします。
                            retryDtl.ToJson(retryJsonFilePath);
                            Log.WriteLine("DTL RetryJson file : {0}", retryJsonFilePath);
                            var retryJson = System.IO.File.ReadAllText(retryJsonFilePath, EncodeContext.UTF8_NoBOM);
                            // ファイル内容の確認用ダンプ
                            Log.WriteLine("=== DTL RetryJson stream begin ===\n{0}\n=== DTL RetryJson stream end ===\n", retryJson);

                            // ダウンロードタスクリストを作成
                            ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application push-download-task-list " + retryJsonFilePath));
                            preRetryJsonFilePath = retryJsonFilePath;
                        }
                        else
                        {
                            if (RETRY_KIND_NONE == retryKind)
                            {
                                retryKind = RETRY_KIND_ALL;
                                retryCount = RETRY_COUNT_ALL; // リトライ回数の初期値
                            }
                            if (RETRY_KIND_PARTS == retryKind)
                            {
                                // 一部リトライの後に来た場合
                                Log.WriteLine("Retry by log without partial retry.\n");
                            }
                            else if (RETRY_KIND_ALL == retryKind)
                            {
                                Log.WriteLine("Retry all data.\n");
                            }
                            else
                            {
                                throw new UnexpectFailureException("A pattern that does not exist.");
                            }
                            // 前回実施のダウンロードタスクリストを使用
                            ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application push-download-task-list " + preRetryJsonFilePath));
                        }
                        // DL のリトライ
                        downloaded = executor.RunDevMenuCommandSystem("--time application wait-download-all", failedCount * 46);
                        executionLog = executor.OutputStream.Standard.ToString();
                        retryCount--;
                    }
                    return downloaded;
                };
                // ------------------------------------------------------------------------

                // SdCard 再初期化
                CleanupSdCard(executor);

                // NAND / SDカードの両方にインストール, SDEV初期化後のデフォルトは enable-mount-sdcard.
                // 1. if true, re-initialize the sdev after the previous download test.
                // 2. using debug mount command.
                // 3. the regex for installed record check.
                // 4. target of storage for size check.
                var mountTargets = new List<System.Tuple<bool, string, string, string>>(2) {
                    new System.Tuple<bool, string, string, string>( true, "debug disable-mount-sdcard", @"""storage"":\s?""BuiltIn\w+""", "builtin" ),
                    new System.Tuple<bool, string, string, string>( false, null, @"""storage"":\s?""SdCard""", "sdcard" ),
                };
                foreach (var mount in mountTargets)
                {
                    // mount command
                    if (false == string.IsNullOrEmpty(mount.Item2))
                    {
                        ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(mount.Item2));
                        ExpectEvaluator.IsTrue(executor.ResetTarget());
                    }

                    // SDカード / 本体ストレージ(NAND)の空き容量
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application storage-size {mount.Item4} -b k"));

                    // SDEV DTL 実施
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application push-download-task-list " + jsonFilePath));

                    // application wait-download-all による完了監視. ( timeout は 50分、1nsp = 46sec 計算 )
                    var downloaded = executor.RunDevMenuCommandSystem("--time application wait-download-all", 3000);
                    // リトライの判定と実施
                    ExpectEvaluator.IsTrue(retryDownload(downloaded, executor.OutputStream.Standard.ToString(), jsonFilePath));

                    // SDカード / 本体ストレージ(NAND)の空き容量
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application storage-size {mount.Item4} -b k"));

                    // Check installed application record.
                    int installCheckIndexCount = appCount;
                    for (int index = 0; index < installCheckIndexCount; ++index)
                    {
                        ID64 nowAppId = baseApplicationId + ((ulong)index);
                        ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application list-record 0x{nowAppId} --installed"));
                        ExpectEvaluator.IsTrue(Regex.Match(executor.OutputStream.Standard.ToString(), mount.Item3).Success);
                    }

                    // 再初期化
                    if (mount.Item1)
                    {
                        InitializeSdev(executor);
                        CleanupSdCard(executor);
                    }
                }
            }
            finally
            {
                Log.WriteLineAsIs(System.Environment.NewLine);
                Log.WriteLine(TestMethodLogFinally, method.Name);

                // マウント設定をデフォルトに。
                ThrowFrameworks.SkipThrow(() => { executor.RunDevMenuCommandSystem("debug enable-mount-sdcard"); });
                ThrowFrameworks.SkipThrow(() => { executor.ResetTarget(); });
            }

            Log.WriteLine(TestMethodLogFinish, method.Name);
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// ダウンロードタスクの実行順序が作成順序と一致するかの検証
        /// TestForStandardDTL でアップロードしたコンテンツを利用するので、
        /// TestForStandardDTL 以降に行われることが前提にしています。
        /// UploadData_app64_aoc_i1v0 のデータ使用
        /// 単体実行時で 2 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-46899")]
        public void TestForVerfiyDownloadTaskOrder()
        {
            using (var scope = new TestMethodLog())
            {
                // Constants
                var appA = new ID64(0x01001a500005e0b0);
                var appB = new ID64(0x01001a500005e0b1);
                var appC = new ID64(0x01001a500005e0b2);

                // Target command executor
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

                // 作業用ディレクトリ
                var intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                // アプリアンインストール、SD カードフォーマット、再起動
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application uninstall --all AND " +
                    "sdcard format "));
                ExpectEvaluator.IsTrue(executor.ResetTarget());

                // ダウンロードタスクを作成
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $"application create-download-task 0x{appA} AND " +     // アプリAダウンロードタスク作成
                    $"application create-download-task 0x{appB} AND " +     // アプリBダウンロードタスク作成
                    $"application uninstall 0x{appA} AND " +                // アプリAアンインストール
                    $"application create-download-task 0x{appC} "));      // アプリCダウンロードタスク作成

                // 全てのダウンロード完了まで待機
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application wait-download-all", 60));

                // 直前のログからダウンロードタスク作成順に実行されたかを確認
                var expected =
                    $"\\[" +
                    $"\\s+\\{{\\s+\"applicationId\": \"0x{appB}\",\\s+\"version\": 0\\s+\\}}," +
                    $"\\s+\\{{\\s+\"applicationId\": \"0x{appC}\",\\s+\"version\": 0\\s+\\}}\\s+" +
                    $"\\]";
                ExpectEvaluator.IsTrue(MatchTargetLog(executor, expected));

                // インストールされたアプリを検証
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $"application verify 0x{appB} AND " +
                    $"application verify 0x{appC} "));
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// インストール先の容量不足の検証
        /// TestForStandardDTL でアップロードしたコンテンツを利用するので、
        /// TestForStandardDTL 以降に行われることが前提にしています。
        /// UploadData_app64_aoc_i1v0 のデータ使用
        /// 単体実行時で 3 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-47138")]
        public void TestForCapacityShortage()
        {
            using (var scope = new TestMethodLog())
            {
                // Constants
                var appA = new ID64(0x01001a500005e0b0);
                var appB = new ID64(0x01001a500005e0b1);
                var appC = new ID64(0x01001a500005e0b2);
                var appD = new ID64(0x01001a500005e0b3);
                const int margin = 2048;

                const string expectedNotEnoughSpace =
                    "\\{{\\s+\"id\": \"0x{0}\",[\\s\\w\":,]+\"progress\": " +
                    "\\{{[\\s\\w\":,]+\"state\": \"NotEnoughSpace\"[\\s\\w\":,]+\\}}\\s+\\}}";
                const string expectedDownlaodSuccess =
                    "\\{{\\s+\"id\": \"0x{0}\",[\\s\\w\":,]+\\}}";
                const string expectedBuiltIn =
                    "\\[\\s+\\{{\\s+\"id\": \"0x{0}\",[\\s\\w\":,]+\"storage\": \"BuiltInUser\"";
                const string expectedSdCard =
                    "\\[\\s+\\{{\\s+\"id\": \"0x{0}\",[\\s\\w\":,]+\"storage\": \"SdCard\"";

                // Target command executor
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

                // 作業用ディレクトリ
                var intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                // 後始末（using ブロック終了時に実行される）
                if (!IsUnitTestOnIDE)
                {
                    scope.AddDisposer(() =>
                   {
                       executor.RunDevMenuCommandSystem(
                           "debug empty-nand-free-space AND " +
                           "application uninstall --all AND " +
                           "debug enable-mount-sdcard");
                       executor.ResetTarget();
                       executor.RunDevMenuCommandSystem(
                           "sdcard format AND " +
                           "sdcard cleanup");
                   });
                }

                // case.1 SD カード無効、NAND 埋め ⇒ ダウンロード失敗、NAND 空け ⇒ ダウンロード成功
                {
                    // テスト準備
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "debug empty-nand-free-space AND " +            // NAND の使用領域解放
                        "application uninstall --all AND " +            // アプリアンインストール
                        "sdcard format AND " +                          // SD カードフォーマット
                        "sdcard cleanup AND " +                         // SD カード初期化
                        "debug disable-mount-sdcard"));               // SD カード無効化
                    ExpectEvaluator.IsTrue(executor.ResetTarget());   // 再起動

                    // NAND を埋めてダウンロードタスク作成
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"debug fill-nand-free-space --margin {margin} AND " +
                        $"application create-download-task 0x{appA} "));

                    // ダウンロード失敗（NAND 容量なし、SD カード無効）
                    ExpectEvaluator.IsFalse(executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{appA} ", 60));

                    // アプリの状態を確認
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application list-view "));
                    ExpectEvaluator.IsTrue(
                        MatchTargetLog(executor, string.Format(expectedNotEnoughSpace, appA)));

                    // NAND の埋めた容量解放、ダウンロードタスクレジューム
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"debug empty-nand-free-space AND " +
                        $"application resume-download 0x{appA} "));

                    // ダウンロード成功（NAND 容量あり、SD カード無効）
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{appA} ", 60));

                    // アプリの状態を確認
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application list-view "));
                    ExpectEvaluator.IsTrue(
                        MatchTargetLog(executor, string.Format(expectedDownlaodSuccess, appA)));

                    // インストールされたアプリを検証
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application verify 0x{appA} AND " +
                        $"application list-record 0x{appA}  --installed "));
                    ExpectEvaluator.IsTrue(
                        MatchTargetLog(executor, string.Format(expectedBuiltIn, appA)));
                }

                // case.2 SD カード有効、SD カード埋め ⇒ ダウンロード成功（NAND に空きあり）
                {
                    // テスト準備
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "debug enable-mount-sdcard"));                // SD カード有効化
                    ExpectEvaluator.IsTrue(executor.ResetTarget());   // 再起動

                    // SD カードを埋めてダウンロードタスク作成
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"debug fill-sdcard-free-space --margin {margin} AND " +
                        $"application create-download-task 0x{appB} "));

                    // ダウンロード成功（NAND 容量あり、SD カード容量なし）
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{appB} ", 60));

                    // アプリの状態を確認
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application list-view "));
                    ExpectEvaluator.IsTrue(
                        MatchTargetLog(executor, string.Format(expectedDownlaodSuccess, appB)));

                    // インストールされたアプリを検証
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application verify 0x{appB} AND " +
                        $"application list-record 0x{appB}  --installed "));
                    ExpectEvaluator.IsTrue(
                        MatchTargetLog(executor, string.Format(expectedBuiltIn, appB)));

                    // SD カードの埋めた容量解放
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"debug empty-sdcard-free-space "));
                }

                // case.3 SD カード有効、NAND・SD カード埋め ⇒ ダウンロード失敗、NAND 空け ⇒ ダウンロード成功
                {
                    // NAND・SD カードを埋めてダウンロードタスク作成
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"debug fill-nand-free-space --margin {margin} AND " +
                        $"debug fill-sdcard-free-space --margin {margin} AND " +
                        $"application create-download-task 0x{appC} "));

                    // ダウンロード失敗（NAND 容量なし、SD カード容量なし）
                    ExpectEvaluator.IsFalse(executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{appC} ", 60));

                    // アプリの状態を確認
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application list-view "));
                    ExpectEvaluator.IsTrue(
                        MatchTargetLog(executor, string.Format(expectedNotEnoughSpace, appC)));

                    // NAND の埋めた容量解放、ダウンロードタスクレジューム
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"debug empty-nand-free-space AND " +
                        $"application resume-download 0x{appC} "));

                    // ダウンロード成功（NAND 容量あり、SD カード容量なし）
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{appC} ", 60));

                    // アプリの状態を確認
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application list-view "));
                    ExpectEvaluator.IsTrue(
                        MatchTargetLog(executor, string.Format(expectedDownlaodSuccess, appC)));

                    // インストールされたアプリを検証
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application verify 0x{appC} AND " +
                        $"application list-record 0x{appC}  --installed "));
                    ExpectEvaluator.IsTrue(
                        MatchTargetLog(executor, string.Format(expectedBuiltIn, appC)));

                    // SD カードの埋めた容量解放
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"debug empty-sdcard-free-space "));
                }

                // case.4 SD カード有効、NAND・SD カード埋め ⇒ ダウンロード失敗、SD カード空け ⇒ ダウンロード成功
                {
                    // NAND・SD カードを埋めてダウンロードタスク作成
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"debug fill-nand-free-space --margin {margin} AND " +
                        $"debug fill-sdcard-free-space --margin {margin} AND " +
                        $"application create-download-task 0x{appD} "));

                    // ダウンロード失敗（NAND 容量なし、SD カード容量なし）
                    ExpectEvaluator.IsFalse(executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{appD} ", 60));

                    // アプリの状態を確認
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application list-view "));
                    ExpectEvaluator.IsTrue(
                        MatchTargetLog(executor, string.Format(expectedNotEnoughSpace, appD)));

                    // SD カードの埋めた容量解放、ダウンロードタスクレジューム
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"debug empty-sdcard-free-space AND " +
                        $"application resume-download 0x{appD} "));

                    // ダウンロード成功（NAND 容量なし、SD カード容量あり）
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{appD} ", 60));

                    // アプリの状態を確認
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application list-view "));
                    ExpectEvaluator.IsTrue(
                        MatchTargetLog(executor, string.Format(expectedDownlaodSuccess, appD)));

                    // インストールされたアプリを検証
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application verify 0x{appD} AND " +
                        $"application list-record 0x{appD}  --installed "));
                    ExpectEvaluator.IsTrue(
                        MatchTargetLog(executor, string.Format(expectedSdCard, appD)));

                    // NAND カードの埋めた容量解放
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"debug empty-nand-free-space "));
                }
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        ///  UploadData_app1_aoc1_i1v0_i1v1_app2_aoc2_i1v0 のデータ使用
        /// 単体実行時で 1 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-47865")]
        public void TestForCleanupUnrecordedEntity()
        {
            using (var scope = new TestMethodLog())
            {
                // Constants
                ID64 baseApplicationId = new ID64(0x0100394000059000);

                // Target command executor
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

                // 作業用ディレクトリ
                string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                var aocv0 = new TestApplication.AddonParameter(baseApplicationId, 0, 1, 1024 * 1024);
                var aocv1 = new TestApplication.AddonParameter(baseApplicationId, 1, 1, 1024 * 1024);

                var request = new List<TestApplication.GenerateParameter<int>>() { aocv0, aocv1 };
                var uploadContents = TestApplication.MakeContents(intermediate, request);
                var nspAocV0 = uploadContents[0];
                var nspAocV1 = uploadContents[1];

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application uninstall --all AND " +
                    "sdcard format AND " +
                    "debug disable-mount-sdcard"));
                ExpectEvaluator.IsTrue(executor.ResetTarget());
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $"addoncontent install {nspAocV0.NspPath}"));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application list-content-meta-database -s builtin",
                    @"""\""version\"": 0"""));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "debug enable-mount-sdcard"));
                ExpectEvaluator.IsTrue(executor.ResetTarget());
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application create-download-task 0100394000059000 --type AddOnContent --id 010039400005a001 --version 65536 AND " +
                    "application wait-download 0100394000059000"));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application list-content-meta-database -s builtin",
                    @"""\[\]"""));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application list-content-meta-database -s sdcard",
                    @"""\""version\"": 65536"""));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application verify 0100394000059000"));
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// 単体実行時で 4 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-48034")]
        public void TestForAddToRunningTask()
        {
            using (var scope = new TestMethodLog())
            {
                // Constants
                ID64 applicationId = new ID64(0x010006000236c000);

                // Target command executor
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                var account = NintendoAccount;

                // finally
                scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("application uninstall --all"); });
                scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("shop unlink-device-all"); });
                scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("account clear_all"); });
                scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("shop unregister-device-account"); });

                // start test
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "shop unlink-device-all AND " +
                    "shop unregister-device-account AND " +
                    "shop register-device-account AND " +
                    "account clear_all AND " +
                    "account add"
                    ));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(string.Format("account link --index 0 --id {0} --password {1}", account.ID, account.PASSWORD)));

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "shop shop-account-status AND " +
                    "shop link-device AND " +
                    "shop delete-all-rights"
                    ));

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application uninstall --all AND " +
                    "shop download-demo 0 --id 0x010006000236c000"
                    ));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application wait-download 0x010006000236c000 --download-progress 1048576/0"
                    ));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "shop download-demo 0 --id 0x010006000236d001"
                    ));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application wait-download 0x010006000236c000 AND " +
                    "application verify 0x010006000236c000"
                    ));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application list-record 0x010006000236c000",
                    @"""\""id\"": \""0x010006000236c000\"""""
                    ));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application list-record 0x010006000236c000",
                    @"""\""id\"": \""0x010006000236d001\"""""
                    ));
            }
        }

        [TestMethod]
        [TestProperty("JIRA", "SIGLO-48486")]
        public void TestForSuppressRedownloaded()
        {
            // すでにインストールされていてレコードも存在する場合には、ダウンロードタスクを作らないことのテスト
            using (var scope = new TestMethodLog())
            {
                // ローカルで基本的に完結するので、ID は適当。ただし、サーバにはその ID で取得しにはいってしまう
                ID64 applicationId1 = new ID64(0x010000000000b148);
                ID64 applicationId2 = new ID64(0x010000000000b149);

                // Target command executor
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

                string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                var app1 = new TestApplication.ApplicationParameter(applicationId1, 0, 1 * 1024 * 1024);
                app1.AddPatch(1);
                var aoc1 = new TestApplication.AddonParameter(applicationId1, 0, 1, 1024 * 1024);

                var request1 = new List<TestApplication.GenerateParameter<int>>() { app1, aoc1 };
                var catalog1 = TestApplication.MakeContents(intermediate, request1);

                var typedCatalog1 = new GeneratedContentResult.TypeCategorizedCatalog(catalog1);
                var aocs1 = typedCatalog1.GetTypedCatalog(ContentMeta.Type.AddOnContent);
                var apps1 = typedCatalog1.GetTypedCatalog(ContentMeta.Type.Application);
                var patches1 = typedCatalog1.GetTypedCatalog(ContentMeta.Type.Patch);

                // finally
                scope.AddDisposer(() =>
               {
                   executor.RunDevMenuCommandSystem(
                       "application uninstall 0x010000000000b148;" +
                       "application uninstall 0x010000000000b149;"
                   );
               });

                // start test
                {
                    // 何もなければ普通に作れる
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application uninstall 0x010000000000b148; application create-download-task 0x010000000000b148",
                        "\"\\[DownloadTaskListManager\\] Create task: appId 0x010000000000b148 keyCount 1\""
                    ));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application uninstall 0x010000000000b148; application create-download-task 0x010000000000b148 --type Patch --id 0x010000000000b948 --version 65536",
                        "\"\\[DownloadTaskListManager\\] Create task: appId 0x010000000000b148 keyCount 1\""
                    ));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application uninstall 0x010000000000b148; application create-download-task 0x010000000000b148 --type AddOnContent --id 0x010000000000c148",
                        "\"\\[DownloadTaskListManager\\] Create task: appId 0x010000000000b148 keyCount 1\""
                    ));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application uninstall 0x010000000000b148;"
                    ));
                }
                {
                    // インストール済みなら、作られない
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        string.Join("&&", apps1.ToNspPaths().Select(p => "application install " + p)) + "&&" +
                        string.Join("&&", patches1.ToNspPaths().Select(p => "patch install " + p)) + "&&" +
                        string.Join("&&", aocs1.ToNspPaths().Select(p => "addoncontent install " + p))
                    ));

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application create-download-task 0x010000000000b148",
                        "\"\\[DownloadTaskListManager\\] Already installed, skip key 010000000000b148, 0\""
                    ));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application create-download-task 0x010000000000b148 --type Patch --id 0x010000000000b948 --version 65536",
                        "\"\\[DownloadTaskListManager\\] Already installed, skip key 010000000000b948, 65536\""
                    ));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application create-download-task 0x010000000000b148 --type AddOnContent --id 0x010000000000c149",
                        "\"\\[DownloadTaskListManager\\] Already installed, skip key 010000000000c149, 0\""
                    ));
                }
                {
                    // アプリの実体を消しても、パッチや AoC はダウンロードされない。アプリはできる
                    // インストール済みなら、作られない
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application delete-entity 0x010000000000b148 --app"
                    ));

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application create-download-task 0x010000000000b148 --type Patch --id 0x010000000000b948 --version 65536",
                        "\"\\[DownloadTaskListManager\\] Already installed, skip key 010000000000b948, 65536\""
                    ));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application create-download-task 0x010000000000b148 --type AddOnContent --id 0x010000000000c149",
                        "\"\\[DownloadTaskListManager\\] Already installed, skip key 010000000000c149, 0\""
                    ));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application uninstall 0x010000000000b148; application create-download-task 0x010000000000b148",
                        "\"\\[DownloadTaskListManager\\] Create task: appId 0x010000000000b148 keyCount 1\""
                    ));

                    // fatal タスクとなるので、application uninstall をする
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application uninstall 0x010000000000b148;"
                    ));
                }
                {
                    // 異なる application ID はきちんと登録できることを確認
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        string.Join("&&", apps1.ToNspPaths().Select(p => "application install " + p)) + "&&" +
                        string.Join("&&", patches1.ToNspPaths().Select(p => "patch install " + p)) + "&&" +
                        string.Join("&&", aocs1.ToNspPaths().Select(p => "addoncontent install " + p))
                    ));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application create-download-task 0x010000000000b149",
                        "\"\\[DownloadTaskListManager\\] Create task: appId 0x010000000000b149 keyCount 1\""
                    ));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application uninstall 0x010000000000b149; application create-download-task 0x010000000000b149 --type Patch --id 0x010000000000b949 --version 65536",
                        "\"\\[DownloadTaskListManager\\] Create task: appId 0x010000000000b149 keyCount 1\""
                    ));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application uninstall 0x010000000000b149; application create-download-task 0x010000000000b149 --type AddOnContent --id 0x010000000000c149",
                        "\"\\[DownloadTaskListManager\\] Create task: appId 0x010000000000b149 keyCount 1\""
                    ));

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application uninstall 0x010000000000b149;"
                    ));
                }
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// テストメソッド: TestForSolveDependencyAocOnNIM
        /// ネットワークインストール時の追加コンテンツ依存のテスト
        /// 単体実行時で 4 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-46447")]
        public void TestForSolveDependencyAocOnNIM()
        {
            using (var scope = new TestMethodLog())
            {
                // AoC index 2 : 0x01001a500005f0b2
                // AoC index 3 : 0x01001a500005f0b3
                // AoC index 4 : 0x01001a500005f0b4
                ID64 IdForApplication = new ID64(0x01001a500005e0b0);

                var method = MethodBase.GetCurrentMethod();
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

                // 作業用ディレクトリ
                string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                // make contents.
                var genIndex2v0 = new TestApplication.AddonParameter(IdForApplication, 0, 2, 4 * 1024 * 1024);
                var genIndex2v1 = new TestApplication.AddonParameter(IdForApplication, 1, 2, 3 * 1024 * 1024);
                var genIndex3v0 = new TestApplication.AddonParameter(IdForApplication, 0, 3, 1 * 1024 * 1024);
                var genIndex3v1 = new TestApplication.AddonParameter(IdForApplication, 1, 3, 4 * 1024 * 1024);
                var genIndex4v0 = new TestApplication.AddonParameter(IdForApplication, 0, 4, 1 * 1024 * 1024);
                var genIndex4v1 = new TestApplication.AddonParameter(IdForApplication, 1, 4, 5 * 1024 * 1024);

                // upload executor to td1 server by ContentsUploader.
                D4cHelper.NspUploader uploader = new D4cHelper.NspUploader(
                    intermediate,
                    ProxyConfiguration,
                    ServerEnvironment
                );

                // make contents for version 0.
                var uploadedVersion0 = TestApplication.MakeContents(intermediate,
                    new List<TestApplication.GenerateParameter<int>>(3)
                    {
                        genIndex2v0, genIndex3v0, genIndex4v0
                    }
                );
                // contents alias for version 0.
                var addon2v0 = uploadedVersion0[0];
                var addon3v0 = uploadedVersion0[1];
                var addon4v0 = uploadedVersion0[2];

                // make contents for version 0.
                var uploadedVersion1 = TestApplication.MakeContents(intermediate,
                    new List<TestApplication.GenerateParameter<int>>(3)
                    {
                        genIndex2v1, genIndex3v1, genIndex4v1
                    }
                );
                // contents alias for version 1.
                var addon2v1 = uploadedVersion1[0];
                var addon3v1 = uploadedVersion1[1];
                var addon4v1 = uploadedVersion1[2];


                // 最新バージョン指定
                uploader.RegisterVersion(addon2v1.NspPath);
                uploader.RegisterVersion(addon3v1.NspPath);
                uploader.RegisterVersion(addon4v1.NspPath);

                // finally
                scope.AddDisposer(() =>
               {
                   executor.RunDevMenuCommandSystem(
                       "application clear-task-status-list AND application uninstall --all AND debug enable-mount-sdcard"
                   );
                   executor.ResetTarget();
                   executor.RunDevMenuCommandSystem(
                       "sdcard format AND sdcard cleanup"
                   );
               });

                // start for builtin
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-mount-sdcard"));
                ExpectEvaluator.IsTrue(executor.ResetTarget());
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("sdcard format AND sdcard cleanup AND debug disable-mount-sdcard"));
                ExpectEvaluator.IsTrue(executor.ResetTarget());

                var matchStorage = SigloHelper.ResultMatcher.ContentValue.Storage.BuiltInAny;
                {
                    // test case 1: インストール済みの追加コンテンツの自動更新
                    // 準備:
                    // Index2 v0, Index3 v0 がローカルにインストール済み
                    // Index2 v1, Index3 v1 がサーバに登録済み
                    // サーバ上の最新バージョンを Index2 v1, Index3 v1 としておく
                    // 手順:
                    // 1. Index2 v1 のみを落とす DTL をプッシュ
                    // 期待結果:
                    // Index2 v1 はもちろんだが、依存として Index3 v1 もインストールされる。

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all"));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"addoncontent install {addon2v0.NspPath} AND addoncontent install {addon3v0.NspPath}"
                    ));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"--time " +
                        $"application create-download-task 0x{IdForApplication} --type {addon2v1.Type} --id 0x{addon2v1.Identifier} --version {addon2v1.Version} AND " +
                        $"application wait-download 0x{IdForApplication} AND " +
                        $"application verify 0x{IdForApplication}"
                    ));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application list-record 0x{IdForApplication} --installed"
                    ));
                    // インストール成否確認
                    ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                        executor.OutputStream.Standard.ToString(),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(addon2v1, matchStorage),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(addon3v1, matchStorage)
                    ));
                }
                {
                    // test case 2: インストールタスク中の追加コンテンツの更新
                    // 準備:
                    // 各index の v0, v1 の2バージョンがサーバに登録済み
                    // サーバ上の最新バージョンを v1 としておく
                    // 手順:
                    // 1. 各index の v0 を落とす DTL をプッシュ
                    // 期待結果:
                    // v0 の代わりに v1 がインストールされることを確認

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all"));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"--time " +
                        $"application create-download-task 0x{IdForApplication} --type {addon2v0.Type} --id 0x{addon2v0.Identifier} --version {addon2v0.Version} AND " +
                        $"application create-download-task 0x{IdForApplication} --type {addon3v0.Type} --id 0x{addon3v0.Identifier} --version {addon3v0.Version} AND " +
                        $"application create-download-task 0x{IdForApplication} --type {addon4v0.Type} --id 0x{addon4v0.Identifier} --version {addon4v0.Version} AND " +
                        $"application wait-download 0x{IdForApplication} AND " +
                        $"application verify 0x{IdForApplication}"
                    ));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application list-record 0x{IdForApplication} --installed"
                    ));
                    // インストール成否確認
                    ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                        executor.OutputStream.Standard.ToString(),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(addon2v1, matchStorage),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(addon3v1, matchStorage),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(addon4v1, matchStorage)
                    ));
                }
                {
                    // test case 3: テストケース2の応用
                    // 準備:
                    // 以下がサーバにアップロード / 登録済。
                    // また、サーバ上の最新バージョンを Index1 v0, Index2 v1, Index3 v1 としておく。
                    // - Index2( v0, v1 )
                    // - Index3( v0, v1 )
                    // - Index4( v0, v1 )
                    // 手順:
                    // 1. Index2 v0 を落とす DTL をプッシュ
                    // 2. インストールタスクを一時停止。
                    // 3. サーバ上の最新バージョンを Index2 v1 に更新
                    // 4. Index3 v0, Index4 v0 をタスクに追加する
                    // 5. インストールタスクを再開
                    // 期待結果:
                    // Index2 v1, Index3 v1, Index4 v1 がインストールされることを確認

                    // 最新バージョン指定
                    uploader.RegisterVersion(addon2v0.NspPath);

                    // DTL( index2 v0 )要求, prepare通過後通信遮断
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all"));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"--time " +
                        $"application create-download-task 0x{IdForApplication} --type {addon2v0.Type} --id 0x{addon2v0.Identifier} --version {addon2v0.Version} AND " +
                        $"application wait-download 0x{IdForApplication} --download-progress 0/1 AND " +
                        $"debug request-network-connection local"
                    ));

                    // 最新バージョン指定 ( 前後 15秒程挟んで、その間通信遮断を行う )
                    Log.WriteLine($"{method.Name} test3 Waiting 15 seconds...");
                    System.Threading.Thread.Sleep(15000);

                    uploader.RegisterVersion(addon2v1.NspPath);

                    Log.WriteLine($"{method.Name} test3 Waiting 15 seconds...");
                    System.Threading.Thread.Sleep(15000);

                    // ネットワーク再開, DTL( index3 v0 )要求.
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"--time debug request-network-connection AND " +
                        $"application create-download-task 0x{IdForApplication} --type {addon3v0.Type} --id 0x{addon3v0.Identifier} --version {addon3v0.Version} AND " +
                        $"application create-download-task 0x{IdForApplication} --type {addon4v0.Type} --id 0x{addon4v0.Identifier} --version {addon4v0.Version} AND " +
                        $"application wait-download 0x{IdForApplication} AND " +
                        $"application verify 0x{IdForApplication}"
                    ));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application list-record 0x{IdForApplication} --installed"
                    ));
                    // インストール成否確認
                    ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                        executor.OutputStream.Standard.ToString(),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(addon2v1, matchStorage),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(addon3v1, matchStorage),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(addon4v1, matchStorage)
                    ));
                }
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// UploadData_app1_aoc1_i1v0_i1v1_app2_aoc2_i1v0 のデータ使用
        /// 単体実行時で 2 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-48591")]
        public void TestAddAfterDownloaded()
        {
            // ダウンロードが完了したけれども commit されていないタスクに対し、content meta を add して問題ないかを試すテスト
            ID64 applicationId = new ID64(0x0100394000059000);

            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            CleanupSdCard(executor);

            try
            {
                // テスト用データの生成
                string intermediate = GenerateIntermediateDirectoryAsMethod(executor, "TestAddAfterDownloaded");

                var app = new TestApplication.ApplicationParameter(applicationId, 0, 1 * 1024 * 1024);
                var aoc1 = new TestApplication.AddonParameter(applicationId, 0, 1, 1024 * 1024);

                var request = new List<TestApplication.GenerateParameter<int>>() { app, aoc1 };
                var uploadContents = TestApplication.MakeContents(intermediate, request);
                var typedCatalog = new GeneratedContentResult.TypeCategorizedCatalog(uploadContents);
                var aocs = typedCatalog.GetTypedCatalog(ContentMeta.Type.AddOnContent);
                var generatedAoc1 = aocs[0];

                // まず devmenu での view の定期的取得を無効にする
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "debug set-boolean-fwdbg --name devmenu --key enable_application_update false;" +
                    "debug set-boolean-fwdbg --name contents_delivery --key disable_auto_commit_forcibly true;"
                ));
                ExpectEvaluator.IsTrue(executor.ResetTarget());

                // ダウンロード、コミット待ちの状態にする
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application create-download-task " + applicationId.ToString() + " AND " +
                    "application wait-download-without-commit --timeout 100000 0x" + applicationId.ToString(),
                    "\"Downloaded without commit\""
                ));

                // インストールされていないことを確認
                // devmenucommand が失敗扱いになると assert に失敗するので、|| で適当なコマンドを実行して失敗しないように
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application list-downloading-content-meta 0x" + applicationId.ToString(),
                    applicationId.ToString()
                ));

                // タスクを追加
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application create-download-task " + applicationId.ToString() + " --type AddOnContent --id 0x" + generatedAoc1.Identifier.ToString() + " AND " +
                    "application wait-download --timeout 100000 0x" + applicationId.ToString()
                ));

                // インストールされていることの確認
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application list",
                    applicationId.ToString()
                ));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "addoncontent list -s sdcard",
                    applicationId.ToString()
                ));


            }
            finally
            {
                // 定期的取得を有効にする
                executor.RunDevMenuCommandSystem(
                    "debug set-boolean-fwdbg --name devmenu --key enable_application_update true;" +
                    "debug set-boolean-fwdbg --name contents_delivery --key disable_auto_commit_forcibly false;" +
                    "application uninstall 0x" + applicationId.ToString()
                );
                executor.ResetTarget();
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// テストメソッド: TestAddAfterStateChanged
        /// NotEnoughSpace, Suspended, SystemUpdateRequired な状態のタスクに対して、
        /// Add したら上記状態が解除され、ダウンロードが完了することの確認
        /// SystemUpdateRequired は進捗が進むことを確認する
        /// 単体実行時で 4 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-65623")]
        public void TestAddAfterStateChanged()
        {
            ID64 applicationId = new ID64(0x01001a500005e100);

            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            CleanupSdCard(executor);

            System.Func<string, string, int> CountPatterns = (string s, string p) =>
            {
                return (s.Length - s.Replace(p, "").Length) / p.Length;
            };

            try
            {
                // テスト用データの生成
                string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                var app = new TestApplication.ApplicationParameter(applicationId, 0, 1 * 1024 * 1024);
                app.ChangeRequiredVersion(2);
                var aoc1 = new TestApplication.AddonParameter(applicationId, 0, 1, 1024 * 1024);

                var request = new List<TestApplication.GenerateParameter<int>>() { app, aoc1 };
                var uploadContents = TestApplication.MakeContents(intermediate, request);
                var typedCatalog = new GeneratedContentResult.TypeCategorizedCatalog(uploadContents);
                var aocs = typedCatalog.GetTypedCatalog(ContentMeta.Type.AddOnContent);
                var generatedAoc1 = aocs[0];

                // まず devmenu での view の定期的取得を無効にする
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "debug set-boolean-fwdbg --name devmenu --key enable_application_update false;" +
                    "debug set-boolean-fwdbg --name contents_delivery --key disable_auto_commit_forcibly true;"
                ));
                ExpectEvaluator.IsTrue(executor.ResetTarget());

                System.Action<string> a = (string state) =>
                {
                    // タスクの状態を変更する
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application create-download-task " + applicationId.ToString() + " AND " +
                        "application set-task-state --state " + state + " 0x" + applicationId.ToString()
                    ));

                    // 状態が期待したものになっていることの確認
                    executor.RunDevMenuCommandSystem("application list-view");
                    ExpectEvaluator.IsTrue(CountPatterns(executor.OutputStream.Standard.ToString(), "\"state\": \"" + state + "\"") == 1);

                    // タスクを追加、これによって状態が解除され、再度ダウンロードタスクが動く
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application create-download-task " + applicationId.ToString() + " --type AddOnContent --id 0x" + generatedAoc1.Identifier.ToString() + " AND " +
                        "application wait-download --timeout 100000 0x" + applicationId.ToString()
                    ));

                    // インストールされていることの確認
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application list",
                        applicationId.ToString()
                    ));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "addoncontent list -s sdcard",
                        applicationId.ToString()
                    ));

                    // 削除
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application uninstall --all"
                    ));
                };

                a("NotEnoughSpace");
                a("Suspended");

                // SystemUpdateRequired
                {
                    executor.RunDevMenuCommandSystem("systemupdate set-debug-id 1-0 ");

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application create-download-task " + applicationId.ToString() + " AND " +
                        "application wait-download-without-commit 0x" + applicationId.ToString()
                    ));

                    // 状態が期待したものになっていることの確認
                    executor.RunDevMenuCommandSystem("application list-view");
                    ExpectEvaluator.IsTrue(CountPatterns(executor.OutputStream.Standard.ToString(), "\"isSystemUpdateRequiredToCommit\": true") == 1);

                    // 本体更新付不要の状態にする
                    executor.RunDevMenuCommandSystem("systemupdate set-debug-id 0-0 ");

                    // タスクを追加、これによって状態が解除され、再度ダウンロードタスクが動く
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application create-download-task " + applicationId.ToString() + " --type AddOnContent --id 0x" + generatedAoc1.Identifier.ToString() + " AND " +
                        "application wait-download --timeout 100000 0x" + applicationId.ToString()
                    ));

                    // インストールされていることの確認
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application list",
                        applicationId.ToString()
                    ));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "addoncontent list -s sdcard",
                        applicationId.ToString()
                    ));

                    // 削除
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application uninstall --all"
                    ));
                }

            }
            finally
            {
                // 定期的取得を有効にする
                executor.RunDevMenuCommandSystem(
                    "debug set-boolean-fwdbg --name devmenu --key enable_application_update true;" +
                    "debug set-boolean-fwdbg --name contents_delivery --key disable_auto_commit_forcibly false;" +
                    "application uninstall 0x" + applicationId.ToString() + ";" +
                    "systemupdate set-debug-id 0-0"
                );
                executor.ResetTarget();
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// テストメソッド: TestForAbortInDelivery
        /// 配信中のスリープ・ドックアウト・シャットダウン(正常 / 異常)による中断検証。
        /// TestForDTLMultiCombination が実施され、生成コンテンツ( nsp, dtl )がローカルに存在する事が前提。
        /// UploadData_app_patch_v1_aoc_i1v0_i2v0 のデータ使用
        /// 単体実行時で 5 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-46446")]
        public void TestForAbortInDelivery()
        {
            using (var scope = new TestMethodLog())
            {
                // 本来のIDは 0x01003ab001e30000 ですが、0x01003ab001e30800 の Patch の証明書エラー回避のため拡張IDにしています。
                // そのため、本テストのメタ構成は以下の通り。
                // Application : 0x01003ab001e30001, version 0
                // Patch       : 0x01003ab001e30801, version 65536
                // AoC index 1 : 0x01003ab001e31002, version 0
                // AoC index 2 : 0x01003ab001e31003, version 0
                var IdForApplication = new ID64(0x01003ab001e30001);

                var b = new System.Text.StringBuilder(128);
                var method = MethodBase.GetCurrentMethod();
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                var intermediate = GenerateIntermediateDirectoryAsMethod(executor);
                var preset = GenerateIntermediateDirectoryAsMethod(executor, "TestForDTLMultiCombination");

                // TestForDTLMultiCombination で既存生成済の nsp の GeneratedContentResult 取得.
                var genApp = new TestApplication.ApplicationParameter(IdForApplication, 0);
                var genAoc1 = new TestApplication.AddonParameter(IdForApplication, 0, 1);
                var genAoc2 = new TestApplication.AddonParameter(IdForApplication, 0, 2);
                var nspApp = System.IO.Path.Combine(preset, genApp.CreateNspFileBaseName(b)) + ".nsp";
                var nspPatch = System.IO.Path.Combine(preset, genApp.CreateNspFileBaseName(1, b)) + ".nsp";
                var nspAoc1 = System.IO.Path.Combine(preset, genAoc1.CreateNspFileBaseName(b)) + ".nsp";
                var nspAoc2 = System.IO.Path.Combine(preset, genAoc2.CreateNspFileBaseName(b)) + ".nsp";
                FileHelper.TestExistsFile(nspApp);
                FileHelper.TestExistsFile(nspPatch);
                FileHelper.TestExistsFile(nspAoc1);
                FileHelper.TestExistsFile(nspAoc2);
                var authoring = new AuthoringExecutor();
                var app = authoring.ExtractContentMetaContexture(intermediate, nspApp);
                var patch = authoring.ExtractContentMetaContexture(intermediate, nspPatch);
                var aoc1 = authoring.ExtractContentMetaContexture(intermediate, nspAoc1);
                var aoc2 = authoring.ExtractContentMetaContexture(intermediate, nspAoc2);
                Log.WriteLine("download contents [");
                Log.WriteLine($"    {app}");
                Log.WriteLine($"    {patch}");
                Log.WriteLine($"    {aoc1}");
                Log.WriteLine($"    {aoc2}");
                Log.WriteLine("]");

                // 準備
                var dtl = System.IO.Path.Combine(preset, "case6_2.dtl.json");
                var comPushAndPrepareAbort = $"--time application push-download-task-list {dtl} AND application wait-download 0x{IdForApplication} --download-progress 0/1";
                var comWait = $"--time application wait-download 0x{IdForApplication} AND application verify 0x{IdForApplication}";

                // 再開後の正常インストール完了チェック共通処理 ( view による entity 確認含む )
                System.Action VerifyExpectInstalled = () =>
                {
                    var command = $"application list-view AND application list-record 0x{IdForApplication} --installed";
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(command));
                    var stream = executor.OutputStream.Standard.ToString();
                    ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(stream,
                        IdForApplication,
                        patch.Version,
                        new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                        {
                            { SigloHelper.ResultMatcher.Application.ListView.Property.isDownloading, false },
                            { SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, true},
                            { SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchRecord, true },
                            { SigloHelper.ResultMatcher.Application.ListView.Property.hasAddOnContentRecord, true },
                            { SigloHelper.ResultMatcher.Application.ListView.Property.hasMainInstallRecord, true },
                            { SigloHelper.ResultMatcher.Application.ListView.Property.hasAllEntity, true },
                            { SigloHelper.ResultMatcher.Application.ListView.Property.hasMainEntity, true },
                            { SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchEntity, true },
                            { SigloHelper.ResultMatcher.Application.ListView.Property.hasAllAddOnContentEntity, true },
                        }
                    ));
                    var matchStorageAny = SigloHelper.ResultMatcher.ContentValue.Storage.Any;
                    ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                        stream,
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app, matchStorageAny),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch, matchStorageAny),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(aoc1, matchStorageAny),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(aoc2, matchStorageAny)
                    ));
                };

                //---
                // 有線 LAN ダウンロード中ドックアウトすると、インフラ接続が切断されるため、全てのダウンロードタスクは Runnable 状態で中断されます。
                // ドックイン後にインフラ接続が確立すると自動的に再開されます。
                // 無線は無関係。
                try
                {
                    // Ethernetのみ
                    ExpectEvaluator.IsTrue(executor.RunSettingsManager("Import \"" + NetworkSettingJsonOnlyEther + "\""));
                    ExpectEvaluator.IsTrue(executor.ResetTarget());

                    // 有線LAN接続で外部接続できるか確認。
                    if (false == executor.RunDevMenuCommand("network get-global-ip-addr"))
                    {
                        Log.WriteLine("[Warning] Skip abort test on cradle, Could not found the environment of wired-LAN.");
                    }
                    else
                    {
                        ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all"));
                        ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(comPushAndPrepareAbort));
                        ExpectEvaluator.IsTrue(executor.ControlTarget("disable-cradle"));
                        Log.WriteLine($"{method.Name} waiting 10 seconds, after disable-cradle.");
                        System.Threading.Thread.Sleep(10000);
                        ExpectEvaluator.IsTrue(executor.ControlTarget("enable-cradle"));
                        ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(comWait));

                        // インストールチェック
                        VerifyExpectInstalled();
                    }
                }
                finally
                {
                    // 無線LANに戻す
                    executor.RunSettingsManager("Import \"" + NetworkSettingJsonFilePath + "\"");
                    executor.ResetTarget();
                }

                //---
                // 正常にデバイスがシャットダウンした場合、コンテンツのダウンロード、パッチ間差分の適用は全て正しく中断・永続化され、
                // 次回再起動後に同じ順序で自動的に再開されます。
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all"));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(comPushAndPrepareAbort));
                ExpectEvaluator.IsTrue(RebootTargetByPowerButton(executor));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(comWait));

                // インストールチェック
                VerifyExpectInstalled();

                //---
                // 強制電源断など、デバイスが異常シャットダウンした場合、全てのダウンロードタスク、パッチ間差分の適用タスクは削除され、
                // コンテンツの予約は全て削除されます。
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all"));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(comPushAndPrepareAbort));
                ExpectEvaluator.IsTrue(RebootTargetByPowerCommand(executor));

                // 中断済みチェック ( Expect the 'isDownloading' was 'false'. )
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application list-view"));
                ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(executor.OutputStream.Standard.ToString(),
                    IdForApplication,
                    patch.Version,
                    new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                    {
                        { SigloHelper.ResultMatcher.Application.ListView.Property.isDownloading, false },
                        { SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, true},
                        { SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchRecord, true },
                        { SigloHelper.ResultMatcher.Application.ListView.Property.hasAddOnContentRecord, true },
                        { SigloHelper.ResultMatcher.Application.ListView.Property.hasMainInstallRecord, true },
                        { SigloHelper.ResultMatcher.Application.ListView.Property.hasAllEntity, false },
                        { SigloHelper.ResultMatcher.Application.ListView.Property.hasMainEntity, false },
                        { SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchEntity, false },
                        { SigloHelper.ResultMatcher.Application.ListView.Property.hasAllAddOnContentEntity, false },
                    }
                ));

                //---
                // スリープすると、インフラ接続が切断されるため、全てのダウンロードタスクは Runnable 状態で中断されます。
                // スリープ復帰後にインフラ接続が確立すると自動的に再開されます。
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all"));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(comPushAndPrepareAbort));
                ExpectEvaluator.IsTrue(executor.ControlTarget("press-power-button"));
                Log.WriteLine($"{method.Name} waiting 30 seconds, after sleep [by press-power-button].");
                System.Threading.Thread.Sleep(30000);
                ExpectEvaluator.IsTrue(executor.ControlTarget("press-power-button"));
                System.Threading.Thread.Sleep(15000); // 起動シーケンス待ち
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(comWait));

                // インストールチェック
                VerifyExpectInstalled();
            }
        }

        //!----------------------------------------------------------------------------
        /// <remarks>
        /// *注意*
        /// SIGLO-64830, SIGLO-64668 での「アプリ実行中のAoCコミットが可能」機能は、
        /// アプリからの明示的なコミット要求があった場合 ( もしくは、DevMenuCommand の application try-commit-current-application-download-task )
        /// が対象となり、「アプリ実行中の暗黙的なAoCコミット」は仕様変更なく不可となっています。
        /// 本テストは上記暗黙コミットに対する検証を含むという観点になります。
        /// UploadData_app1_aoc1_i1v0_i1v1_app2_aoc2_i1v0 のデータ使用
        /// 単体実行時で 1 分程度要します。
        /// </remarks>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-49180")]
        public void TestForImmediateCommit()
        {
            using (var scope = new TestMethodLog())
            {
                // Constants
                ID64 commitId = new ID64(0x0100394000059000);
                ID64 runId = new ID64(0x0100db60023de000);

                // Target command executor
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

                // 作業用ディレクトリ
                string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                var commitApp = new TestApplication.ApplicationParameter(commitId, 0);
                var commitAoc = new TestApplication.AddonParameter(commitId, 0, 1);
                var runApp = new TestApplication.ApplicationParameter(runId, 0);
                var runAoc = new TestApplication.AddonParameter(runId, 0, 1);

                var request = new List<TestApplication.GenerateParameter<int>>()
                {
                    commitApp,
                    commitAoc,
                    runApp,
                    runAoc
                };
                var uploadContents = TestApplication.MakeContents(intermediate, request);
                var nspCommitApp = uploadContents[0];
                var nspCommitAoc = uploadContents[1];
                var nspRunApp = uploadContents[2];
                var nspRunAoc = uploadContents[3];

                // finally
                scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("application uninstall --all"); });

                // アプリのインストールと起動
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application uninstall --all AND " +
                    "application create-download-task 0x0100db60023de000 AND " +
                    "application wait-download 0x0100db60023de000 AND " +
                    "application launch 0x0100db60023de000"
                    ));

                // 異なるアプリはコミットできる
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystemSuppressAutoKill(
                    "application create-download-task 0x0100394000059000 AND " +
                    "application wait-download 0x0100394000059000"
                    ));

                // 異なるアプリの AoC もコミットできる
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystemSuppressAutoKill(
                    "application create-download-task 0x0100394000059000 --type AddOnContent --id 0x010039400005a001 AND " +
                    "application wait-download 0x0100394000059000"
                    ));

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystemSuppressAutoKill(
                    "application create-download-task 0x0100db60023de000 --type AddOnContent --id 0x0100db60023df001"
                    ));

                // 自アプリの AoC はコミットできない
                ExpectEvaluator.IsFalse(executor.RunDevMenuCommandSystemSuppressAutoKill(
                    "application wait-download 0x0100db60023de000"
                    ));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystemSuppressAutoKill(
                    "application wait-download 0x0100db60023de000 --wait-commit"
                    ));

                // アプリを終了したらコミットできる
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application wait-download 0x0100db60023de000"
                    ));
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// UploadData_app1_aoc1_i1v0_i1v1_app2_aoc2_i1v0 のデータ使用
        /// 単体実行時で 2 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-49339")]
        public void TestForApplicationControlRequest()
        {
            using (var scope = new TestMethodLog())
            {
                // Constants
                ID64 appId = new ID64(0x0100db60023de000);

                // Target command executor
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

                // finally
                scope.AddDisposer(() =>
                {
                    executor.RunDevMenuCommandSystem(
                        "debug enable-mount-sdcard"
                    );
                    executor.ResetTarget();
                    executor.RunDevMenuCommandSystem(
                        "application uninstall --all AND " +
                        "debug empty-nand-free-space"
                    );
                });

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application uninstall --all AND " +
                    "debug disable-mount-sdcard"
                    ));
                ExpectEvaluator.IsTrue(executor.ResetTarget());

                // NotEnoughSpace なタスクがあっても管理データが落ちてくる
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application invalidate-cache application-control AND " +
                    "debug fill-nand-free-space AND " +
                    "application create-download-task 0x0100db60023de000 AND " +
                    "application wait-download 0x0100db60023de000 --not-enough-space AND " +
                    "application wait-control 0x0100db60023de000"
                    ));

                // NotEnoughSpace なタスクを再起動しても管理データが落ちてくる
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application invalidate-cache application-control"
                    ));
                ExpectEvaluator.IsTrue(RebootTargetByPowerButton(executor));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application wait-control 0x0100db60023de000"
                    ));
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// テストメソッド: TestForRequestUpdateApplication
        /// RequestUpdateApplication トリガでのダウンロードコンテンツ整合性検証テスト。
        /// 単体実行時で 3 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-48660")]
        public void TestForRequestUpdateApplication()
        {
            using (var scope = new TestMethodLog())
            {
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                var intermediate = GenerateIntermediateDirectoryAsMethod(executor);
                var method = MethodBase.GetCurrentMethod();

                // 本テストのメタ構成は以下の通り。
                // Type         : ID                    : old vesion        : latest  version
                // Application  : 0x01003ab001e30002    : 0                 : N/A
                // AoC index 2  : 0x01003ab001e31004    : 0                 : 65536 ( 0x10000 )
                // Patch        : 0x01003ab001e30802    : 65536 ( 0x10000 ) : 131072 ( 0x20000 )
                var IdForApplication = new ID64(0x01003ab001e30002);

                // make contents.
                var genApp = new TestApplication.ApplicationParameter(IdForApplication, 0, 1 * 1024 * 1024);
                var genAoc_i2v0 = new TestApplication.AddonParameter(IdForApplication, 0, 2, 64 * 1024);
                var genAoc_i2v1 = new TestApplication.AddonParameter(IdForApplication, 1, 2, 128 * 1024);
                genApp.AddPatches(1, 2);
                genApp.UseSmallCode = true;

                // upload executor to td1 server by ContentsUploader.
                D4cHelper.NspUploader uploader = new D4cHelper.NspUploader(
                    intermediate,
                    ProxyConfiguration,
                    ServerEnvironment
                );

                var uploadContents = TestApplication.MakeContents(intermediate,
                    new List<TestApplication.GenerateParameter<int>>(3)
                    {
                        genApp, genAoc_i2v0, genAoc_i2v1
                    }
                );
                // カテゴライズ
                var typedCatalog = new GeneratedContentResult.TypeCategorizedCatalog(uploadContents);
                var aocs = typedCatalog.GetTypedCatalog(ContentMeta.Type.AddOnContent);
                var apps = typedCatalog.GetTypedCatalog(ContentMeta.Type.Application);
                var patches = typedCatalog.GetTypedCatalog(ContentMeta.Type.Patch);

                // contents alias
                var app = apps[0];
                var patch_v1 = patches.FindFromVersion(0x10000);
                var patch_v2 = patches.FindFromVersion(0x20000);
                var aoc_i2v0 = aocs.FindFromVersion(0x00000);
                var aoc_i2v1 = aocs.FindFromVersion(0x10000);
                if (null == app || null == patch_v1 || null == patch_v2 || null == aoc_i2v0 || null == aoc_i2v1)
                {
                    throw new UnexpectFailureException("MakeContents operation failed.");
                }

                // latest version settings.
                uploader.RegisterVersion(app.NspPath);
                uploader.RegisterVersion(patch_v2.NspPath);
                uploader.RegisterVersion(aoc_i2v1.NspPath);

                // 環境準備
                var hasInsertedGameCard = executor.RunDevMenuCommandSystem($"gamecard status", @"^Inserted$");
                if (hasInsertedGameCard) { ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("gamecard erase")); }

                // finally
                scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all"); });
                scope.AddDisposer(() => { if (hasInsertedGameCard) { executor.RunDevMenuCommandSystem("gamecard erase"); } });

                var comPush = "application push-download-task-list";
                var comVerify = $"application verify 0x{IdForApplication}";
                var comWait = $"application wait-download 0x{IdForApplication}";
                var comPushAndWait = $"--time {comPush} {{0}} AND {comWait} AND {comVerify}";
                var comRecord = $"application list-record 0x{IdForApplication} --installed";
                var matchStorageAny = SigloHelper.ResultMatcher.ContentValue.Storage.Any;
                var matchStorageGameCard = SigloHelper.ResultMatcher.ContentValue.Storage.GameCard;
                var ptnUpdatable = @"^\[target\] (Updatable\r|Updatable)$"; // use for multiline regex only.

                // Case1: 再ダウンロードテスト
                Log.WriteLine("=== Case1: Full entity re-downloading. ===");
                {
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all"));

                    string json = System.IO.Path.Combine(intermediate, "case1.dtl.json");
                    var dtl = new DownloadTaskList();
                    var task = dtl.Create(IdForApplication);
                    task.AddNewTitle(app).AddNewTitle(patch_v2).AddNewTitle(aoc_i2v1);
                    dtl.ToJson(json);

                    // DTL push and wait.
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(string.Format(comPushAndWait, json)));
                    // インストールチェック
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(comRecord));
                    ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                        executor.OutputStream.Standard.ToString(),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app, matchStorageAny),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch_v2, matchStorageAny),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(aoc_i2v1, matchStorageAny)
                    ));

                    // # Check case : the delete-entity all → update-info "Updatable" → update all
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application delete-entity 0x{IdForApplication} AND application update-info 0x{IdForApplication}"
                    ));
                    // "[target] Updatable" 検出, Multilineの対象改行は(LF)なので CRLFの場合( Updatable\r )で検出するようにしています。
                    ExpectEvaluator.IsTrue(Regex.Match(executor.OutputStream.Standard.ToString(), ptnUpdatable, RegexOptions.Multiline).Success);

                    // application update で全復元実施。
                    // SIGLO-49527: "application update" と分けて実行しないと "wait-download" でインフラ接続に失敗したままタイムアウトする。
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application update 0x{IdForApplication}"));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application wait-download 0x{IdForApplication} AND application list-view"));
                    ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(executor.OutputStream.Standard.ToString(),
                        IdForApplication,
                        patch_v2.Version,
                        new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                        {
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAllEntity, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainEntity, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchEntity, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAllAddOnContentEntity, true },
                        }
                    ));
                }

                // Case2: 最新バージョンへの更新テスト
                Log.WriteLine("=== Case2: Latest version updating. ===");
                {
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all"));

                    // ローカルインストール
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $@"application install ""{app.NspPath}"" AND " +
                        $@"patch install ""{patch_v1.NspPath}"" AND " +
                        $@"addoncontent install ""{aoc_i2v0.NspPath}"" AND " +
                        $"{comRecord}"
                    ));
                    ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                        executor.OutputStream.Standard.ToString(),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app, matchStorageAny),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch_v1, matchStorageAny),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(aoc_i2v0, matchStorageAny)
                    ));

                    // # Check case : update-info "Updatable" → update all
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application update-info 0x{IdForApplication}"));
                    ExpectEvaluator.IsTrue(Regex.Match(executor.OutputStream.Standard.ToString(), ptnUpdatable, RegexOptions.Multiline).Success);

                    // application update で全コンテンツの最新更新。
                    // SIGLO-49527: "application update" と分けて実行しないと "wait-download" でインフラ接続に失敗したままタイムアウトする。
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application update 0x{IdForApplication}"));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application wait-download 0x{IdForApplication} AND {comRecord}"));
                    ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                        executor.OutputStream.Standard.ToString(),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app, matchStorageAny),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch_v2, matchStorageAny),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(aoc_i2v1, matchStorageAny)
                    ));
                }

                // ゲームカードの有無確認.
                // ここで検出されなければ以降のテストは実施不可能なので Failure.
                ExpectEvaluator.IsTrue(hasInsertedGameCard);

                // ゲームカードアプリケーションに対する更新検証用共通シーケンス。
                System.Func<System.Action, SigloHelper.ResultMatcher.Application.ListView.ExpectValues, bool> UpdateVerifierOnGameCard = (prepare, expectValues) =>
                {
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all"));

                    // GameCard準備
                    prepare();

                    // # Check case : update-info "Updatable" → update all
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application delete-entity 0x{IdForApplication} AND " +
                        $"application update-info 0x{IdForApplication}"
                    ));
                    ExpectEvaluator.IsTrue(Regex.Match(executor.OutputStream.Standard.ToString(), ptnUpdatable, RegexOptions.Multiline).Success);

                    // application update でも Patch 更新できるか確認。
                    // SIGLO-49527: "application update" と分けて実行しないと "wait-download" でインフラ接続に失敗したままタイムアウトする。
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application update 0x{IdForApplication}"));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application wait-download 0x{IdForApplication} AND " +
                        $"application list-view AND " +
                        $"{comRecord}"
                    ));
                    var stream2 = executor.OutputStream.Standard.ToString();
                    ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(stream2,
                        IdForApplication,
                        patch_v2.Version,
                        expectValues
                    ));
                    // app と patch_v2 のみがレコードされている想定。
                    ExpectEvaluator.IsEqual(1, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(stream2,
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app, matchStorageGameCard),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch_v2, matchStorageAny),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(aoc_i2v1, matchStorageAny)
                    ));
                    return true;
                };


                // Case3: ゲームカードテスト
                Log.WriteLine("=== Case3: Update with the GameCard basic. ===");
                {
                    UpdateVerifierOnGameCard(() =>
                       {
                           // GameCard writing.
                           ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($@"gamecard write ""{app.NspPath}"" --verify"));
                       },
                        // 期待する list-view の一致状態
                        // ローカルレコードが無い→Entityは true扱いになる。
                        new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                        {
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.isGameCard, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasGameCardEntity, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchRecord, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAddOnContentRecord, false },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainInstallRecord, false },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAllEntity, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainEntity, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchEntity, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAllAddOnContentEntity, true },
                        }
                    );
                }

                // Case4: ゲームカード抜け殻テスト
                Log.WriteLine("=== Case4: Update with the GameCard that does not have a entity. ===");
                {
                    UpdateVerifierOnGameCard(() =>
                       {
                           // GameCard writing.
                           ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($@"gamecard write ""{app.NspPath}"" --verify"));
                           // GameCard unmount.
                           ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("gamecard erase"));
                       },
                        // 期待する list-view の一致状態
                        // ローカルレコードが無い→Entityは true扱いになる。
                        new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                        {
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.isGameCard, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasGameCardEntity, false },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchRecord, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAddOnContentRecord, false },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainInstallRecord, false },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAllEntity, false },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainEntity, false },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchEntity, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAllAddOnContentEntity, true },
                        }
                    );
                }

                // Case5: オンカードパッチテスト
                Log.WriteLine("=== Case5: Update with the GameCard that have an older patch. ===");
                {
                    UpdateVerifierOnGameCard(() =>
                       {
                           // GameCard writing ( with onCardPatch ).
                           ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($@"gamecard write ""{app.NspPath}"" --on-card-patch ""{patch_v1.NspPath}"" --verify"));
                       },
                        // 期待する list-view の一致状態
                        // ローカルレコードが無い→Entityは true扱いになる。
                        new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                        {
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.isGameCard, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasGameCardEntity, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchRecord, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAddOnContentRecord, false },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainInstallRecord, false },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAllEntity, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainEntity, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchEntity, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAllAddOnContentEntity, true },
                        }
                    );
                }
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// テストメソッド: TestForRequestUpdateApplication
        /// RequestUpdateApplication の配信停止コンテンツのダウンロード不具合検証テスト。
        /// SIGLO-49858 の不具合再現及び修正検証用。
        /// 実体がないコンテンツのバージョンが superfly のバージョンよりも新しい場合に、
        /// 実体がないコンテンツのバージョンを優先してダウンロードしてしまう不具合。
        /// 単体実行時で 1 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-49928")]
        public void TestForRequestUpdateApplicationVersionFetch()
        {
            using (var scope = new TestMethodLog())
            {
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                var intermediate = GenerateIntermediateDirectoryAsMethod(executor);
                var method = MethodBase.GetCurrentMethod();

                // 本テストのメタ構成は以下の通り。
                // Type         : ID                    : client vesion         : server version
                // Application  : 0x01003ab001e30003    : N/A                   : 0
                // Patch        : 0x01003ab001e30803    : 131072 ( 0x20000 )    : 65536 ( 0x10000 )
                var IdForApplication = new ID64(0x01003ab001e30003);

                // make contents.
                var genApp0 = new TestApplication.ApplicationParameter(IdForApplication, 0, 1 * 1024 * 1024);
                genApp0.AddPatches(1, 2);
                genApp0.UseSmallCode = true;

                // upload executor to td1 server by ContentsUploader.
                D4cHelper.NspUploader uploader = new D4cHelper.NspUploader(
                    intermediate,
                    ProxyConfiguration,
                    ServerEnvironment
                );

                var uploadContents = TestApplication.MakeContents(intermediate, genApp0);

                // カテゴライズ
                var typedCatalog = new GeneratedContentResult.TypeCategorizedCatalog(uploadContents);
                var apps = typedCatalog.GetTypedCatalog(ContentMeta.Type.Application);
                var patches = typedCatalog.GetTypedCatalog(ContentMeta.Type.Patch);

                // contents alias
                var app_v0 = apps.FindFromVersion(0x00000);
                var patch_v1 = patches.FindFromVersion(0x10000);
                var patch_v2 = patches.FindFromVersion(0x20000);
                if (null == app_v0 || null == patch_v1 || null == patch_v2)
                {
                    throw new UnexpectFailureException("MakeContents operation failed.");
                }

                // latest version settings.
                uploader.RegisterVersion(app_v0.NspPath);
                uploader.RegisterVersion(patch_v1.NspPath);

                // finally
                scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all"); });

                // テスト開始.
                var comRecord = $"application list-record 0x{IdForApplication} --installed";
                var matchStorageAny = SigloHelper.ResultMatcher.ContentValue.Storage.Any;
                var ptnUpdatable = @"^\[target\] (Updatable\r|Updatable)$"; // use for multiline regex only.

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all"));

                // ローカルインストール
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $@"application install ""{app_v0.NspPath}"" AND " +
                    $@"patch install ""{patch_v1.NspPath}"" AND " +
                    $@"patch install ""{patch_v2.NspPath}"" AND " +
                    $@"{comRecord}"
                ));
                ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                    executor.OutputStream.Standard.ToString(),
                    new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app_v0, matchStorageAny),
                    new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch_v2, matchStorageAny)
                ));

                // # Check case : update-info "Updatable" → update all
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $"application delete-entity 0x{IdForApplication} AND " +
                    $"application update-info 0x{IdForApplication}"
                ));
                ExpectEvaluator.IsTrue(Regex.Match(executor.OutputStream.Standard.ToString(), ptnUpdatable, RegexOptions.Multiline).Success);

                // SIGLO-49527: "application update" と分けて実行しないと "wait-download" でインフラ接続に失敗したままタイムアウトする。
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application update 0x{IdForApplication}"));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $"application wait-download 0x{IdForApplication} AND " +
                    $"application list-view AND " +
                    $"{comRecord}"
                ));
                var stream = executor.OutputStream.Standard.ToString();
                ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(stream,
                    IdForApplication,
                    patch_v1.Version,
                    new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                    {
                        {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, true },
                        {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchRecord, true },
                        {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAddOnContentRecord, false },
                        {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainInstallRecord, true },
                        {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAllEntity, true },
                        {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainEntity, true },
                        {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchEntity, true },
                        {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAllAddOnContentEntity, true },
                    }
                ));
                // app_v0 と patch_v1 のレコード想定
                ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(stream,
                    new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app_v0, matchStorageAny),
                    new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch_v1, matchStorageAny)
                ));
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// 「消えないプレースホルダ」問題の検証テスト
        /// ダウンロード中タスクが中断された状態でタスクを削除した場合のプレースホルダクリーン検証。
        /// 本テストでサポートするディスククラスタ範囲は 64KiB ～ 512KiB とします。
        /// 但し、512KiB の検証は実装がないので行えていません。
        /// SdCardの空き容量調節は、空き容量確保値を 512KiB クラスタ基準で 4KiB 余剰を追加したサイズにしています。
        /// これは、インストール可能判定が freeSize > requiredSize の実装のためです。
        /// さらに、debug fill-sdcard-free-space --margin で空き容量が期待値よりも 28KiB 少なかった。
        /// nn::fs のファイルエントリ管理の影響？ とりあえず下限範囲の大コンテンツサイズ ( 3416 KiB )がインストールできないサイズで大目に確保します。
        /// 単体実行時で 3 ～ 8 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-49878")]
        public void TestForPlaceHolderCleanupAtTaskCancelled()
        {
            using (var scope = new TestMethodLog())
            {
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                var intermediate = GenerateIntermediateDirectoryAsMethod(executor);
                var method = MethodBase.GetCurrentMethod();

                // 本テストのメタ構成は以下の通り。
                // Type         : ID                    : client vesion         : server version
                // Application  : 0x01003ab001e30003    : N/A                   : N/A
                // AoC index 2  : 0x01003ab001e31005    : N/A                   : 0
                // AoC index 3  : 0x01003ab001e31006    : N/A                   : 0
                ulong kbSizeSmall = 3 * 1024;
                ulong kbSizeLarge = 7 * 1024;
                var IdForApplication = new ID64(0x01003ab001e30003);
                var genAocSmall = new TestApplication.AddonParameter(IdForApplication, 0, 2, kbSizeSmall * 1024);
                var genAocLarge = new TestApplication.AddonParameter(IdForApplication, 0, 3, kbSizeLarge * 1024);


                var uploadContents = TestApplication.MakeContents(intermediate,
                    new List<TestApplication.GenerateParameter<int>>(2) { genAocSmall, genAocLarge }
                );
                var aocSmall = uploadContents[0];
                var aocLarge = uploadContents[1];

                // finally
                scope.AddDisposer(() =>
               {
                   executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all");
                   executor.RunDevMenuCommandSystem("debug enable-mount-sdcard");
                   executor.ResetTarget();
                   executor.RunDevMenuCommandSystem("sdcard format AND sdcard cleanup");
                   executor.RunDevMenuCommandSystem("debug request-network-connection");
                   executor.ResetTarget();
               });

                // Initialize the storage.
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-mount-sdcard"));
                ExpectEvaluator.IsTrue(executor.ResetTarget());
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("sdcard format AND sdcard cleanup"));
                ExpectEvaluator.IsTrue(executor.ResetTarget());

                // fill-sdcard-free-space [--margin <KiB>], maxからmarginを差し引いた残量を埋める
                // Small AoC のマージン, AoC構成 ( 詳細は * cnmt.xml にある )
                // Meta : 4096 byte
                // Data : 1134592 byte ( 1MiB + 86016 byte )
                // 各Contentごとに2クラスタの下駄を履かせるらしい。( fileエントリ+directoryエントリ )
                // 本テスト作成時は256KiB/1クラスタであるが、下限64KiB, 上限512KiB の範囲をサポートする想定で、
                // AoC 1MiB + AoC 3MiB 構成 + 512KiB/1クラスタとする。
                ulong fsControlMargin = 192;
                ulong clusterSize = 512;
                ulong metaMargin = 4 + (clusterSize * 2);
                ulong dataMargin = kbSizeSmall + (clusterSize * 2) + 84;
                ulong marginSdCard = dataMargin + metaMargin + fsControlMargin;   // ( 3,235,840 = 3.0859375 MiB ) + 196,608
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $"debug fill-sdcard-free-space --margin {marginSdCard} AND " +
                    $"application storage-size sdcard -b k"
                ));
                // 期待する空き容量確保できているか
                var freeSpaceMatched = Regex.Match(executor.OutputStream.Standard.ToString(), @"\[target\] Free space\s*[0-9]+\s*KB");
                var freeSpace = ulong.Parse(Regex.Match(freeSpaceMatched.Value, "[0-9]+").Value);
                ulong limitMargin = 4 + (64 * 2) + kbSizeLarge + (64 * 2) + 84;
                Log.WriteLine("Check the free space on sdcard, required condition that be ( {0} < {1} < {2} )",
                    dataMargin + metaMargin, freeSpace, limitMargin
                );
                ExpectEvaluator.IsTrue(freeSpace > (dataMargin + metaMargin) && freeSpace < limitMargin);

                // 共通シーケンス。
                System.Func<GeneratedContentResult, GeneratedContentResult, string, bool> FunctionForCommonSequence = (firstRequire, secondRequire, listPlaceHolderCommand) =>
                {
                    // 全部消す。
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all"));

                    // DownloadTask生成/中断
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application create-download-task 0x{IdForApplication} --type {firstRequire.Type} --id 0x{firstRequire.Identifier} --version {firstRequire.Version} AND " +
                        $"application create-download-task 0x{IdForApplication} --type {secondRequire.Type} --id 0x{secondRequire.Identifier} --version {secondRequire.Version} AND " +
                        $"application wait-download 0x{IdForApplication} --download-progress 0/1 AND " +
                        $"debug request-network-connection local"
                    ));

                    // ここからはネットワーク切断状態で作業。
                    // 正しく目的のストレージに配置されてるか？
                    // "application list-download-task-status" 呼ばないと "application list-downloading-content-meta" が正しく機能しない？
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application list-download-task-status AND application list-downloading-content-meta 0x{IdForApplication}"));
                    ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(executor.OutputStream.Standard.ToString(),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(aocSmall, SigloHelper.ResultMatcher.ContentValue.Storage.SdCard),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(aocLarge, SigloHelper.ResultMatcher.ContentValue.Storage.BuiltInUser)
                    ));

                    // 3. Check whether existed the placeholder in the storage of nand and sdcard.
                    // 生成される placeholderは、コンテンツごとなので、Data + Meta で2つ。
                    // AoC は２つなので、SdCard に2つ、Builtin に2つの、総計4個 の想定。
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(listPlaceHolderCommand));
                    var stream = SigloHelper.ResultMatcher.TrimPrefixTarget(executor.OutputStream.Standard.ToString());
                    var collection = Regex.Matches(stream, @"\[\s*""\w{8}-\w{4}-\w{4}-\w{4}-\w{12}"",\s*""\w{8}-\w{4}-\w{4}-\w{4}-\w{12}""\s*\]");
                    Log.WriteLine("Check the place-holder existed count, required condition that be ( 2 == {0} )", collection.Count);
                    ExpectEvaluator.IsEqual(2, collection.Count);

                    // 4. disable sdcard, and shutdown, and boot.
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug disable-mount-sdcard"));
                    ExpectEvaluator.IsTrue(RebootTargetByPowerButton(executor));

                    // 5. delete aborted download tasks.
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application cancel-download 0x{IdForApplication}"));

                    // 6. Check whether erased the placeholder in the storage of nand.
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application list-place-holder -s builtin"));
                    ExpectEvaluator.IsTrue(Regex.Match(executor.OutputStream.Standard.ToString(), @"\[target\] \[\]").Success);

                    // 7. enable sdcard mount, and shutdown, and boot.
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-mount-sdcard"));
                    ExpectEvaluator.IsTrue(RebootTargetByPowerButton(executor));

                    // 8. Check whether erased the placeholder in the storage of sdcard.
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application list-place-holder -s sdcard"));
                    ExpectEvaluator.IsTrue(Regex.Match(executor.OutputStream.Standard.ToString(), @"\[target\] \[\]").Success);

                    // finally for next sequences.
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug request-network-connection"));

                    return true;
                };

                // SD → NAND
                FunctionForCommonSequence(aocSmall, aocLarge, "application list-place-holder -s sdcard AND application list-place-holder -s builtin");

                // NAND → SD
                FunctionForCommonSequence(aocLarge, aocSmall, "application list-place-holder -s builtin AND application list-place-holder -s sdcard");
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// NXBTS-11639 を踏まえた、多量のパッチ間差分適用タスクの作成/検証テスト
        /// 単体実行時で 43 ～ 55 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-50507")]
        public void TestForVerifyRunManyPatchDeltaApplyTask()
        {
            using (var scope = new TestMethodLog())
            {
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                var intermediate = GenerateIntermediateDirectoryAsMethod(executor);
                var method = MethodBase.GetCurrentMethod();

                // 本テストのメタ構成は以下の通り。
                // 連番Index | Application ID     | Patch ID           | Version |
                //         0 | 0x01003ab001e30004 | 0x01003ab001e30804 | 0, 1, 2 |
                //         : |          :         |          :         |    :    |
                //         : |          :         |          :         |    :    |
                //        63 | 0x01003ab001e30043 | 0x01003ab001e30843 | 0, 1, 2 |

                uint appCount = 64U;
                var IdForApplication = new ID64(0x01003ab001e30004);

                // make parameter of contents.
                var gens = new List<TestApplication.GenerateParameter<int>>((int)appCount);
                for (uint index = 0; index < appCount; ++index)
                {
                    var now = IdForApplication + index;
                    var app = new TestApplication.ApplicationParameter(now, 0, 32);
                    app.AddPatches(1, new TestApplication.PatchContexture<int>(2, AuthoringExecutor.Patch.FlagOption.UseDelta));
                    app.UseSmallCode = true;
                    gens.Add(app);
                }

                // make the title resource contexture with call the MakeTestApplication.
                var uploadContents = TestApplication.MakeContents(intermediate, gens);
                var titles = new GeneratedContentResult.TitleCategorizedCatalog(uploadContents);

                // dtl 作成
                var dtlv1 = System.IO.Path.Combine(intermediate, "v1.dtl.json");
                var dtlv2 = System.IO.Path.Combine(intermediate, "v2.dtl.json");

                // app v0 + patch v1
                var dtl = new DownloadTaskList(titles);
                dtl.RemoveTitleAll((title) => { return title.Type == ContentMeta.Type.Patch && title.Version == 0x20000; });
                dtl.ToJson(dtlv1);

                // patch v2 追加
                foreach (var title in titles)
                {
                    dtl.Create(title.Key).AddNewTitle(title.Value.GetTypedCatalog(ContentMeta.Type.Patch).FindFromVersion(0x20000));
                }
                dtl.ToJson(dtlv2);

                // finally
                scope.AddDisposer(() =>
               {
                   executor.RunDevMenuCommandSystem(
                       "debug set-boolean-fwdbg --name devmenu --key enable_application_update true;" +
                       "debug set-boolean-fwdbg --name contents_delivery --key disable_auto_commit_forcibly false;"
                   );
                   executor.RunDevMenuCommandSystem("patch set-prefer-delta false");
                   executor.ResetTarget();
               });
                scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all"); });

                // 全部消す。
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all"));

                // 共通 list-view チェック
                System.Func<string, int, bool, bool> ExpectViewChecker = (stream, version, applyingDelta) =>
                {
                    foreach (var title in titles)
                    {
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(stream,
                            title.Key,
                            version,
                            new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                            {
                            { SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, true },
                            { SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchRecord, true },
                            { SigloHelper.ResultMatcher.Application.ListView.Property.hasMainInstallRecord, true },
                            { SigloHelper.ResultMatcher.Application.ListView.Property.hasAllEntity, !applyingDelta },
                            { SigloHelper.ResultMatcher.Application.ListView.Property.hasMainEntity, true },
                            { SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchEntity, !applyingDelta },
                            { SigloHelper.ResultMatcher.Application.ListView.Property.isDownloading, false },
                            { SigloHelper.ResultMatcher.Application.ListView.Property.isApplyingDelta, applyingDelta },
                            { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingCommit, false },
                            { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingPatchInstall, false },
                            }
                        ));
                    }
                    return true;
                };

                // v1 install 実施 && application wait-download-all による完了監視. ( timeout 1.1時間 )
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $"--time application push-download-task-list {dtlv1} AND " +
                    $"application wait-download-all AND application list-view",
                    30 * 128
                ));

                // ベースインストール完了の確認( version1 + hasAllEntity( true ) )。
                ExpectViewChecker(executor.OutputStream.Standard.ToString(), 0x10000, false);

                // devmenu での view の定期的取得を無効にする ( DevMenu が view の更新をしてしまうのを防ぐ )
                // 及び、自動コミット機能を無効化する。
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "debug set-boolean-fwdbg --name devmenu --key enable_application_update false;" +
                    "debug set-boolean-fwdbg --name contents_delivery --key disable_auto_commit_forcibly true;"
                ));
                // んで patch 適用のコンテンツを強制的に差分版を使うようにする。( デフォルトでは完全版と差分版でサイズが小さい方が使われるので変動なくすため )
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("patch set-prefer-delta true"));
                ExpectEvaluator.IsTrue(executor.ResetTarget());

                // v2 patch の without-commit ダウンロード
                // wait-download は該当アプリの GetApplicationView を呼び出す.
                // GetApplicationView は、コミット待ちのタスクに対して呼ぶと、基本的に勝手にコミットする.
                // 各タイトルごとにwait-download-without-commit で待機し、
                // 完了後 downloading-meta に対象が存在して、インストールされてない事を確認する。
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"--time application push-download-task-list {dtlv2}"));
                foreach (var title in titles)
                {
                    var patch_v2 = title.Value.GetTypedCatalog(ContentMeta.Type.Patch).FindFromVersion(0x20000);
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application wait-download-without-commit 0x{title.Key} AND " +
                        $"application list-downloading-content-meta 0x{title.Key}"
                    ));
                    ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(executor.OutputStream.Standard.ToString(),
                        new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch_v2,
                            SigloHelper.ResultMatcher.ContentValue.Storage.Any,
                            SigloHelper.ResultMatcher.ContentValue.InstallType.FragmentOnly
                        )
                    ));
                }

                // application list-view で残りを纏めてコミット( list-view 完了時にコミットは完了する )。
                // 差分タスクは非同期なので完了するまで version は v1 がリスト対象になる。
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application list-view"));
                ExpectViewChecker(executor.OutputStream.Standard.ToString(), 0x10000, true);

                // 全て v2 に更新されていることを確認する。
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application wait-download-all AND application list-view"));
                ExpectViewChecker(executor.OutputStream.Standard.ToString(), 0x20000, false);
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// UploadData_app1_aoc1_i1v0_i1v1_app2_aoc2_i1v0 のデータ使用
        /// 単体実行時で 1 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-49500")]
        public void TestForEnableAutoUpdate()
        {
            using (var scope = new TestMethodLog())
            {
                // Constants
                ID64 appId = new ID64(0x0100db60023de000);

                // Target command executor
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

                // finally
                scope.AddDisposer(() =>
                {
                    executor.RunDevMenuCommandSystem(
                        "application uninstall --all"
                    );
                });

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application uninstall --all AND " +
                    "application create-download-task 0x0100db60023de000 AND " +
                    "application wait-download 0x0100db60023de000"
                    ));

                // 新規アプリ DL 時は自動更新が有効に
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application list-record-detail",
                    "\"\\\"isAutoUpdateEnabled\\\": true\""
                    ));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application list-record-detail",
                    "\"\\\"patchId\\\": \\\"0x0100db60023de800\\\"\""
                    ));

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application disable-auto-update 0x0100db60023de000 AND" +
                    "application create-download-task 0x0100db60023de000 --type AddOnContent --id 0x0100db60023df001 AND " +
                    "application wait-download 0x0100db60023de000"
                    ));

                // 追加コンテンツ追加 DL 時は自動更新に変化なし
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application list-record-detail",
                    "\"\\\"isAutoUpdateEnabled\\\": false\""
                    ));

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application uninstall --all AND " +
                    "application create-download-task 0x0100db60023de000 --type AddOnContent --id 0x0100db60023df001 AND " +
                    "application wait-download 0x0100db60023de000"
                    ));

                // 追加コンテンツ単独 DL 時も自動更新に変化なし
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application list-record-detail",
                    "\"\\\"isAutoUpdateEnabled\\\": false\""
                    ));
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// コンテンツ自動更新の検証
        /// 自動更新されないことの確認に時間がかかります。
        /// 単体実行時で 40 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-47150")]
        public void TestForAutoUpdateContents()
        {
            var UploadTimeout = new System.TimeSpan(0, 5, 0);         // 失敗時のリトライも含めて長めに設定
            var DownloadTimeout = new System.TimeSpan(0, 3, 0);       // 通知～完了までの実績値が 1 分以内
            var VersionListInterval = new System.TimeSpan(0, 1, 0);   // supefly -> tagaya の取得間隔分を空ける

            using (var scope = new TestMethodLog())
            {
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                var intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                // make contents
                //  Application : 0x01001a500005e0f0, version 0
                //  Patch       : 0x01001a500005e8f0, version 65536
                //                                    version 131072
                //  AddOnContent: 0x01001a500005f0f1, version 0,      index 1
                //                                    version 65536,  index 1
                //                                    version 131072, index 1
                var owner = new ID64(0x01001a500005e0f0);
                var @params = new List<TestApplication.GenerateParameter<int>>();
                {
                    // app, patch
                    var param = new TestApplication.ApplicationParameter(owner, 0, 5 * 1024 * 1024);
                    param.AddPatches(1, 2);
                    param.UseSmallCode = true;
                    @params.Add(param);
                }
                {
                    // aoc
                    var param0 = new TestApplication.AddonParameter(owner, 0, 1, 1 * 1024 * 1024);
                    @params.Add(param0);
                    var param1 = new TestApplication.AddonParameter(owner, 1, 1, 1 * 1024 * 1024);
                    @params.Add(param1);
                    var param2 = new TestApplication.AddonParameter(owner, 2, 1, 1 * 1024 * 1024);
                    @params.Add(param2);
                }

                var contents = TestApplication.MakeContents(intermediate, @params);
                var catalog = new GeneratedContentResult.TypeCategorizedCatalog(contents);
                var app = catalog.GetTypedCatalog(ContentMeta.Type.Application)[0];
                var patch1 = catalog.GetTypedCatalog(ContentMeta.Type.Patch)[0];
                var patch2 = catalog.GetTypedCatalog(ContentMeta.Type.Patch)[1];
                var aoc_v0 = catalog.GetTypedCatalog(ContentMeta.Type.AddOnContent)[0];
                var aoc_v1 = catalog.GetTypedCatalog(ContentMeta.Type.AddOnContent)[1];
                var aoc_v2 = catalog.GetTypedCatalog(ContentMeta.Type.AddOnContent)[2];

                // upload contents
                var uploader = new D4cHelper.NspUploader(
                    intermediate,
                    ProxyConfiguration,
                    ServerEnvironment);

                var account = CliTokenAccount;
                var timeout = UploadTimeout.TotalSeconds;
                uploader.RejectRoms(account, app.Identifier);
                uploader.Publish(account, app.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.WithApproved, timeout);
                uploader.Publish(account, patch1.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.Patch, timeout);
                uploader.Publish(account, patch2.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.Patch, timeout);

                uploader.RegisterVersion(app.NspPath);

                // finally
                if (!IsUnitTestOnIDE)
                {
                    scope.AddDisposer(() => { CleanupContents(executor, true); });
                }

                // prepare
                CleanupContents(executor, true);

                // 検証条件：ローカルインストール後に自動更新される、Aocはインストールされていなければ更新されない。
                scope.WriteLineWithTimeStamp("auto update after local install");
                {
                    uploader.RegisterVersion(aoc_v0.NspPath);

                    // wait version list
                    uploader.DeleteVersion(patch1.NspPath);
                    ExpectEvaluator.IsTrue(D4cHelper.WaitUntilVersionListDeleted(executor, patch1, true));

                    // install
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application install {app.NspPath} "));
                    CheckAppplicationList(executor);

                    // verify
                    VerifyAppplication(executor, app.Identifier);
                    {
                        // expect: app only
                        var output = executor.OutputStream.Standard.ToString();
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            output, new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app)));
                        ExpectEvaluator.IsEqual(1, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            output, new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch1)));
                    }

                    // launch
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommand(
                        $"application launch 0x{app.Identifier} "));
                    CheckAppplicationList(executor);

                    // version-up (0 -> 1)
                    uploader.RegisterVersion(aoc_v1.NspPath);
                    uploader.RegisterVersion(patch1.NspPath);

                    var download1 = executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{app.Identifier} --version {patch1.Version} --timeout {DownloadTimeout.TotalMilliseconds} ");
                    CheckAppplicationList(executor);
                    ExpectEvaluator.IsTrue(download1);

                    // verify
                    VerifyAppplication(executor, app.Identifier);
                    {
                        // expect: app, patch1
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            executor.OutputStream.Standard.ToString(),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch1)));
                        //aocは表示されない(インストールしていないため)
                        ExpectEvaluator.IsEqual(1, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            executor.OutputStream.Standard.ToString(),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(aoc_v1)));

                    }
                }

                CleanupContentsAndWait(executor, VersionListInterval);

                // 検証条件：patch の自動更新時に aoc も更新される
                scope.WriteLineWithTimeStamp("Aoc is also updated when automatically updating patches");
                {
                    uploader.RegisterVersion(aoc_v0.NspPath);

                    // wait version list
                    uploader.DeleteVersion(patch1.NspPath);
                    ExpectEvaluator.IsTrue(D4cHelper.WaitUntilVersionListDeleted(executor, patch1, true));

                    // install
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application install {app.NspPath} AND " +
                        $"addoncontent install {aoc_v0.NspPath}"));
                    CheckAppplicationList(executor);

                    // verify
                    VerifyAppplication(executor, app.Identifier);
                    {
                        // expect: app , aoc_v0
                        var output = executor.OutputStream.Standard.ToString();
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            output, new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app)));
                        ExpectEvaluator.IsEqual(1, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            output, new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch1)));
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            output, new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(aoc_v0)));
                    }

                    // launch
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommand(
                        $"application launch 0x{app.Identifier} "));
                    CheckAppplicationList(executor);

                    // version-up (0 -> 1)
                    uploader.RegisterVersion(aoc_v1.NspPath);
                    uploader.RegisterVersion(patch1.NspPath);

                    var download1 = executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{app.Identifier} --version {patch1.Version} --timeout {DownloadTimeout.TotalMilliseconds} ");
                    CheckAppplicationList(executor);
                    ExpectEvaluator.IsTrue(download1);

                    // verify
                    VerifyAppplication(executor, app.Identifier);
                    {
                        // expect: app, patch1
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            executor.OutputStream.Standard.ToString(),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch1),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(aoc_v1)));
                    }
                }

                CleanupContentsAndWait(executor, VersionListInterval);

                // 検証条件：OnCardAoc の場合、patch/aoc の更新が行われても、aoc の更新は行われない。( sd/nand に aoc が無い場合に限る )
                scope.WriteLineWithTimeStamp("OnCardAoc is not updated by automatic updating");
                {
                    uploader.RegisterVersion(aoc_v0.NspPath);

                    // wait version list
                    uploader.DeleteVersion(patch1.NspPath);
                    ExpectEvaluator.IsTrue(D4cHelper.WaitUntilVersionListDeleted(executor, patch1, true));

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"gamecard write {app.NspPath} --on-card-aoc {aoc_v0.NspPath}"));

                    // Aoc の状態を確認( On Card Aoc )
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommand("addoncontent list"));
                    {
                        var output = executor.OutputStream.Standard.ToString();
                        ExpectEvaluator.IsTrue(SigloHelper.ResultMatcher.AddOnContent.List.Find(output, aoc_v0));
                    }

                    // launch
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommand(
                        $"application launch 0x{app.Identifier} "));
                    CheckAppplicationList(executor);

                    // version-up (0 -> 1)
                    uploader.RegisterVersion(aoc_v1.NspPath);
                    uploader.RegisterVersion(patch1.NspPath);

                    var download1 = executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{app.Identifier} --version {patch1.Version} --timeout {DownloadTimeout.TotalMilliseconds} ");
                    CheckAppplicationList(executor);
                    ExpectEvaluator.IsTrue(download1);

                    // verify
                    VerifyAppplication(executor, app.Identifier);
                    {
                        // expect: app, patch1
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            executor.OutputStream.Standard.ToString(),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch1)));
                    }

                    // Aoc の versionは 0 から変化しない
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommand("addoncontent list"));
                    {
                        var output = executor.OutputStream.Standard.ToString();
                        ExpectEvaluator.IsTrue(SigloHelper.ResultMatcher.AddOnContent.List.Find(output, aoc_v0));
                    }

                    // ------------------------------------------------------------------------------------------------
                    // 検証条件：OnCardAocの場合、sd/nand に aoc インストール後は、aocは自動更新対象になる。
                    scope.WriteLineWithTimeStamp("Manual install after auto update");

                    // 手動で Aoc を更新( 0 -> 1 )
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application install {aoc_v1.NspPath}"));
                    CheckAppplicationList(executor);

                    // verify
                    VerifyAppplication(executor, app.Identifier);
                    {
                        // expect: app, aoc_v1
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            executor.OutputStream.Standard.ToString(),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(aoc_v1)));
                    }

                    // Aoc の状態を確認 ( card:0, sd/nand:1 )
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommand("addoncontent list"));
                    {
                        var output = executor.OutputStream.Standard.ToString();
                        ExpectEvaluator.IsTrue(SigloHelper.ResultMatcher.AddOnContent.List.Find(output, aoc_v1));
                    }

                    // launch を実行しなければ patch の更新が行われないため実施
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommand(
                        $"application launch 0x{app.Identifier} "));
                    CheckAppplicationList(executor);

                    // Patch/Aoc version-up ( 1 -> 2 )
                    uploader.RegisterVersion(aoc_v2.NspPath);
                    uploader.RegisterVersion(patch2.NspPath);

                    // patch と Aoc の更新
                    var download2 = executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{app.Identifier} --version {patch2.Version} --timeout {DownloadTimeout.TotalMilliseconds} ");
                    CheckAppplicationList(executor);
                    ExpectEvaluator.IsTrue(download2);

                    // verify
                    VerifyAppplication(executor, app.Identifier);
                    {
                        // expect: app, patch2, aoc_v2
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            executor.OutputStream.Standard.ToString(),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch2),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(aoc_v2)));
                    }

                    // Aoc の状態を確認 ( card:0, sd/nand:2 )
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommand("addoncontent list"));
                    {
                        var output = executor.OutputStream.Standard.ToString();
                        ExpectEvaluator.IsTrue(SigloHelper.ResultMatcher.AddOnContent.List.Find(output, aoc_v2));
                    }
                }

                CleanupContentsAndWait(executor, VersionListInterval);

                // 検証条件：ダウンロード後に自動更新される
                scope.WriteLineWithTimeStamp("auto update after download install");
                {
                    // wait version list
                    uploader.DeleteVersion(patch1.NspPath);
                    ExpectEvaluator.IsTrue(D4cHelper.WaitUntilVersionListDeleted(executor, patch1, true));

                    // download and version-up (0 -> 1)
                    uploader.RegisterVersion(patch1.NspPath);

                    var download1 = executor.RunDevMenuCommandSystem(
                        $"--time application download 0x{app.Identifier} --version {patch1.Version} --timeout {DownloadTimeout.TotalMilliseconds} ");
                    CheckAppplicationList(executor);
                    ExpectEvaluator.IsTrue(download1);

                    // verify
                    VerifyAppplication(executor, app.Identifier);
                    {
                        // expect: app, patch1
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            executor.OutputStream.Standard.ToString(),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch1)));
                    }

                    // launch
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommand(
                        $"application launch 0x{app.Identifier} "));
                    CheckAppplicationList(executor);

                    // version-up (1 -> 2)
                    uploader.RegisterVersion(patch2.NspPath);

                    var download2 = executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{app.Identifier} --version {patch2.Version} --timeout {DownloadTimeout.TotalMilliseconds} ");
                    CheckAppplicationList(executor);
                    ExpectEvaluator.IsTrue(download2);

                    // verify
                    VerifyAppplication(executor, app.Identifier);
                    {
                        // expect: app, patch1 -> patch2
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            executor.OutputStream.Standard.ToString(),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch2)));
                    }
                }
                CleanupContentsAndWait(executor, VersionListInterval);

                // 検証条件：パッチがあたっていると自動更新されない
                // 検証条件：アプリとパッチの記録がある状態で、自動更新で新しいパッチのダウンロードが開始され、
                //           タスクをキャンセルしても、パッチの記録は上書きされない（SIGLO-51029）
                scope.WriteLineWithTimeStamp("not update if patched already");
                {
                    // wait version list
                    uploader.DeleteVersion(patch1.NspPath);
                    ExpectEvaluator.IsTrue(D4cHelper.WaitUntilVersionListDeleted(executor, patch1, true));

                    // install
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application install {app.NspPath} "));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"patch install {patch1.NspPath} "));
                    CheckAppplicationList(executor);

                    // verify
                    VerifyAppplication(executor, app.Identifier);
                    {
                        // expect: app, patch1
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            executor.OutputStream.Standard.ToString(),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch1)));
                    }

                    // launch
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommand(
                        $"application launch 0x{app.Identifier} "));
                    CheckAppplicationList(executor);
                    {
                        // expect: isAutoUpdateEnabled(true)
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordDetail.Find(
                            executor.OutputStream.Standard.ToString(),
                            app.Identifier,
                            new SigloHelper.ResultMatcher.Application.ListRecordDetail.ExpectValues()
                            {
                                {  SigloHelper.ResultMatcher.Application.ListRecordDetail.Property.isAutoUpdateEnabled, true },
                            }));

                        // expect: version(65536), hasPatchRecord(true), hasPatchEntity(true)
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(
                            executor.OutputStream.Standard.ToString(),
                            app.Identifier,
                            patch1.Version,
                            new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                            {
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.isDownloading, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchRecord, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchEntity, true },
                            }));
                    }

                    // version-up (0 -> 1)
                    uploader.RegisterVersion(patch1.NspPath);

                    var download1 = executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{app.Identifier} --version {patch2.Version} --download-progress 0/1 --timeout {DownloadTimeout.TotalMilliseconds} ");
                    CheckAppplicationList(executor);
                    ExpectEvaluator.IsTrue(!download1);
                    // wait-download のオプションは OR 条件で、指定条件のどれかを満たした場合に待機が終了する
                    // 上記の条件では ver.2 未満でダウンロード開始されるまで指定タイムアウト分待機することになる
                    // ローカルでパッチインストールしているので、新たにダウンロードされないことを確認している

                    // version-up (1 -> 2) and cancel
                    uploader.RegisterVersion(patch2.NspPath);

                    var download2 = executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{app.Identifier} --version {patch2.Version} --download-progress 0/1 --timeout {DownloadTimeout.TotalMilliseconds} AND " +
                        $"application cancel-download 0x{app.Identifier} AND " +
                        $"application list-downloading-content-meta 0x{app.Identifier} ");
                    CheckAppplicationList(executor);
                    ExpectEvaluator.IsTrue(download2);
                    {
                        // expect: isAutoUpdateEnabled(true -> false)
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordDetail.Find(
                            executor.OutputStream.Standard.ToString(),
                            app.Identifier,
                            new SigloHelper.ResultMatcher.Application.ListRecordDetail.ExpectValues()
                            {
                                {  SigloHelper.ResultMatcher.Application.ListRecordDetail.Property.isAutoUpdateEnabled, false },
                            }));

                        // expect: not changed
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(
                            executor.OutputStream.Standard.ToString(),
                            app.Identifier,
                            patch1.Version,
                            new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                            {
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.isDownloading, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchRecord, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchEntity, true },
                            }));
                        // パッチ記録は上書きされないのでバージョンは更新されない
                    }

                    // verify
                    VerifyAppplication(executor, app.Identifier);
                    {
                        // expect: not changed
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            executor.OutputStream.Standard.ToString(),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch1)));
                        // パッチ記録は上書きされないのでバージョンは更新されない
                    }
                }
                CleanupContentsAndWait(executor, VersionListInterval);

                // 検証条件：アプリ起動後に実体削除すると自動更新されない
                scope.WriteLineWithTimeStamp("not update after delete entity");
                {
                    // wait version list
                    uploader.DeleteVersion(patch1.NspPath);
                    ExpectEvaluator.IsTrue(D4cHelper.WaitUntilVersionListDeleted(executor, patch1, true));

                    // install
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application install {app.NspPath} "));
                    CheckAppplicationList(executor);

                    // verify
                    VerifyAppplication(executor, app.Identifier);
                    {
                        // expect: app only
                        var output = executor.OutputStream.Standard.ToString();
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            output, new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app)));
                        ExpectEvaluator.IsEqual(1, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            output, new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch1)));
                    }

                    // launch and delete
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommand(
                        $"application launch 0x{app.Identifier} "));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application delete-entity 0x{app.Identifier} --app "));
                    CheckAppplicationList(executor);

                    // version-up (0 -> 1)
                    uploader.RegisterVersion(patch1.NspPath);

                    var download1 = executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{app.Identifier} --version {patch1.Version} --download-progress 0/1 --timeout {DownloadTimeout.TotalMilliseconds} ");
                    CheckAppplicationList(executor);
                    ExpectEvaluator.IsTrue(!download1);
                }
                CleanupContentsAndWait(executor, VersionListInterval);

                // 検証条件：パッチタスクをダウンロード中にキャンセルすると以降自動更新されない
                // 検証条件：アプリのみの記録がある状態で、自動更新でパッチのダウンロードが開始され、
                //           タスクをキャンセルすると、パッチの記録が追加されている（SIGLO-51029）
                // 仕様変更：自動更新タスク登録時には記録を更新しない（SIGLO-60647）
                scope.WriteLineWithTimeStamp("not update after cancel download");
                {
                    // wait version list
                    uploader.DeleteVersion(patch1.NspPath);
                    ExpectEvaluator.IsTrue(D4cHelper.WaitUntilVersionListDeleted(executor, patch1, true));

                    // install
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application install {app.NspPath} "));
                    CheckAppplicationList(executor);

                    // verify
                    VerifyAppplication(executor, app.Identifier);
                    {
                        // expect: app only
                        var output = executor.OutputStream.Standard.ToString();
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            output, new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app)));
                        ExpectEvaluator.IsEqual(1, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            output, new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch1)));
                    }

                    // launch
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommand(
                        $"application launch 0x{app.Identifier} "));
                    CheckAppplicationList(executor);
                    {
                        // expect: isAutoUpdateEnabled(true)
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordDetail.Find(
                            executor.OutputStream.Standard.ToString(),
                            app.Identifier,
                            new SigloHelper.ResultMatcher.Application.ListRecordDetail.ExpectValues()
                            {
                                {  SigloHelper.ResultMatcher.Application.ListRecordDetail.Property.isAutoUpdateEnabled, true },
                            }));

                        // expect: version(0), hasPatchRecord(false), hasPatchEntity(false)
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(
                            executor.OutputStream.Standard.ToString(),
                            app.Identifier,
                            app.Version,
                            new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                            {
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchRecord, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchEntity, false },
                            }));
                    }

                    // version-up (0 -> 1) and cancel
                    uploader.RegisterVersion(patch1.NspPath);

                    var download1 = executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{app.Identifier} --version {patch1.Version} --download-progress 0/1 --timeout {DownloadTimeout.TotalMilliseconds} AND " +
                        $"application cancel-download 0x{app.Identifier} AND " +
                        $"application list-downloading-content-meta 0x{app.Identifier} ");
                    CheckAppplicationList(executor);
                    ExpectEvaluator.IsTrue(download1);
                    {
                        // expect: isAutoUpdateEnabled(true -> false)
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordDetail.Find(
                            executor.OutputStream.Standard.ToString(),
                            app.Identifier,
                            new SigloHelper.ResultMatcher.Application.ListRecordDetail.ExpectValues()
                            {
                                {  SigloHelper.ResultMatcher.Application.ListRecordDetail.Property.isAutoUpdateEnabled, false },
                            }));

                        // expect: not changed
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(
                            executor.OutputStream.Standard.ToString(),
                            app.Identifier,
                            app.Version,
                            new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                            {
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchRecord, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchEntity, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.isDownloading, false },
                            }));
                        // 仕様変更：自動更新タスク登録時には記録を更新しない（SIGLO-60647）
                    }

                    // verify
                    VerifyAppplication(executor, app.Identifier);
                    {
                        // expect: app only
                        var output = executor.OutputStream.Standard.ToString();
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            output, new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app)));
                        ExpectEvaluator.IsEqual(1, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            output, new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch1)));
                    }

                    // version-up (1 -> 2)
                    uploader.RegisterVersion(patch2.NspPath);

                    var download2 = executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{app.Identifier} --version {patch2.Version} --download-progress 0/1 --timeout {DownloadTimeout.TotalMilliseconds} ");
                    CheckAppplicationList(executor);
                    ExpectEvaluator.IsTrue(!download2);
                }

                scope.WriteLineWithTimeStamp("auto update test finished");
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// ダウンロード開始時点でアプリ記録更新されていることの確認
        /// 仕様変更：ダウンロード開始でアプリケーション記録を行うのは新規追加のみ（SIGLO-60839）
        /// TestForPurchaseDemoTicket でアップロードしたコンテンツを流用しています。
        /// 単体実行時で 9 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-51029")]
        public void TestForRecordWhenCancelDownloadTask()
        {
            using (var scope = new TestMethodLog())
            {
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                // finally
                scope.AddDisposer(() =>
               {
                   // unregister device
                   executor.RunDevMenuCommandSystem("shop unlink-device-all");
                   executor.RunDevMenuCommandSystem("account clear_all");
                   executor.RunDevMenuCommandSystem("shop unregister-device-account");
                   if (!IsUnitTestOnIDE)
                   {
                       CleanupContents(executor, true);
                   }
               });

                // prepare
                CleanupContents(executor, true);

                // 検証条件：記録がない状態で、アプリ（パッチ付き）を購入して、タスクをキャンセルすると、アプリ及びパッチの記録が追加されている
                scope.WriteLineWithTimeStamp("record app and patch when cancel download task");
                {
                    // contents (deverted contents on TestForPurchaseDemoTicket)
                    //  Application : 0x01003ab001e32000, version 0
                    //  Patch       : 0x01003ab001e32800, version 65536
                    //  AddOnContent: 0x01003ab001e33001, version 0, index 1
                    var app = new ContentMeta.Contexture.Basic(
                        new ID64(0x01003ab001e32000), 0, ContentMeta.Type.Application);
                    var patch = new ContentMeta.Contexture.Basic(
                        new ID64(0x01003ab001e32800), 65536, ContentMeta.Type.Patch);
                    //var aoc = new ContentMeta.Contexture.Basic(
                    //    new ID64( 0x01003ab001e33001 ), 0, ContentMeta.Type.AddOnContent );

                    // register device account
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"account add --register_nsa AND " +
                        $"account link --index 0 --id {NintendoAccount.ID} --password {NintendoAccount.PASSWORD} "));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"shop register-device-account AND " +
                        $"shop shop-account-status 0 "));
                    ExpectEvaluator.IsTrue(D4cHelper.ClearDownloadTaskListOnShopServer(executor));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"shop link-device 0 "));

                    // purchase and cancel download
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"shop download-demo 0 --id 0x{app.Identifier} "));
                    var download = executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{app.Identifier} --download-progress 0/1 --timeout 300000 AND " +
                        $"application cancel-download 0x{app.Identifier} AND " +
                        $"application list-downloading-content-meta 0x{app.Identifier} ");
                    CheckAppplicationList(executor);
                    ExpectEvaluator.IsTrue(download);
                    {
                        // expect: isAutoUpdateEnabled(false)
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordDetail.Find(
                            executor.OutputStream.Standard.ToString(),
                            app.Identifier,
                            new SigloHelper.ResultMatcher.Application.ListRecordDetail.ExpectValues()
                            {
                                {  SigloHelper.ResultMatcher.Application.ListRecordDetail.Property.isAutoUpdateEnabled, false },
                            }));

                        // expect: isDownloading(false), has*Record(true), has*Entity(false)
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(
                            executor.OutputStream.Standard.ToString(),
                            app.Identifier,
                            patch.Version,
                            new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                            {
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.isDownloading, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainInstallRecord, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainEntity, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchRecord, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchEntity, false },
                            }));
                        // ダウンロード中断でもアプリ及びパッチの記録が追加されている
                    }

                    VerifyAppplication(executor, app.Identifier);
                    {
                        // expect: app, patch
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            executor.OutputStream.Standard.ToString(),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app),
                            new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch)));
                        // ダウンロード中断でもアプリ及びパッチの記録が追加されている
                    }

                    // chekc ticket rights
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application content-meta-status 0x{app.Identifier} --check-rights"));
                    {
                        var storage = SigloHelper.ResultMatcher.ContentValue.Storage.None;
                        var rights = SigloHelper.ResultMatcher.ContentValue.RightsCheck.NotChecked;
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ContentMetaStatusCheckRights.Find(
                            executor.OutputStream.Standard.ToString(),
                            new SigloHelper.ResultMatcher.ContentValue(app, storage, rights),
                            new SigloHelper.ResultMatcher.ContentValue(patch, storage, rights)
                        ));
                    }

                    // unregister device account
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "shop unlink-device 0 AND " +
                        "account clear_all AND " +
                        "shop unregister-device-account "));
                }
                CleanupContents(executor);

                // 検証条件：アプリのみの記録がある状態で、追加コンテンツを購入して、タスクをキャンセルすると、追加コンテンツの記録が追加されていない
                // 仕様変更：ダウンロード開始でアプリケーション記録を行うのは新規追加のみ（SIGLO-60839）
                scope.WriteLineWithTimeStamp("not record aoc when cancel download task");
                {
                    // contents
                    //  Application : 0x01003ab001e32100, version 0
                    //  Patch       : 0x01003ab001e32900, version 65536
                    //  AddOnContent: 0x01003ab001e33101, index 1
                    var owner = new ID64(0x01003ab001e32100);
                    var @params = new List<TestApplication.GenerateParameter<int>>();
                    {
                        // app, patch(unused)
                        var param = new TestApplication.ApplicationParameter(owner, 0, 1 * 1024 * 1024);
                        //param.AddPatch( 1 );
                        param.UseSmallCode = true;
                        param.ChangeTicketEncryption(true);
                        @params.Add(param);
                    }
                    {
                        // aoc
                        var param = new TestApplication.AddonParameter(owner, 0, 1, 1 * 1024 * 1024);
                        param.ChangeTicketEncryption(true);
                        @params.Add(param);
                    }
                    var contents = TestApplication.MakeContents(intermediate, @params);
                    var catalog = new GeneratedContentResult.TypeCategorizedCatalog(contents);
                    var app = catalog.GetTypedCatalog(ContentMeta.Type.Application)[0];
                    //var patch = catalog.GetTypedCatalog( ContentMeta.Type.Patch )[ 0 ];
                    var aoc = catalog.GetTypedCatalog(ContentMeta.Type.AddOnContent)[0];

                    // register device account
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"account add --register_nsa AND " +
                        $"account link --index 0 --id {NintendoAccount.ID} --password {NintendoAccount.PASSWORD} "));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"shop register-device-account AND " +
                        $"shop shop-account-status 0 "));
                    ExpectEvaluator.IsTrue(D4cHelper.ClearDownloadTaskListOnShopServer(executor));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"shop link-device 0 "));

                    // install
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application install {app.NspPath} "));
                    CheckAppplicationList(executor);

                    // verify
                    VerifyAppplication(executor, app.Identifier);
                    {
                        // expect: app only
                        var output = executor.OutputStream.Standard.ToString();
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            output, new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app)));
                        ExpectEvaluator.IsEqual(1, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            output, new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(aoc)));
                    }

                    // launch
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommand(
                        $"application launch 0x{app.Identifier} "));
                    CheckAppplicationList(executor);
                    {
                        // expect: isAutoUpdateEnabled(true)
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordDetail.Find(
                            executor.OutputStream.Standard.ToString(),
                            app.Identifier,
                            new SigloHelper.ResultMatcher.Application.ListRecordDetail.ExpectValues()
                            {
                                {  SigloHelper.ResultMatcher.Application.ListRecordDetail.Property.isAutoUpdateEnabled, true },
                            }));

                        // expect: hasAddOnContentRecord(false), hasAllAddOnContentEntity(true)
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(
                            executor.OutputStream.Standard.ToString(),
                            app.Identifier,
                            app.Version,
                            new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                            {
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.isDownloading, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainInstallRecord, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainEntity, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAddOnContentRecord, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAllAddOnContentEntity, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAnyAddOnContentEntity, false },
                            }));
                        // 全追加コンテンツ実体は追加コンテンツ記録がない場合は true を返す
                    }

                    // purchase and cancel download
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"shop download-demo 0 --id 0x{aoc.Identifier} "));
                    var download = executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{app.Identifier} --version 65536 --download-progress 0/1 --timeout 300000 AND " +
                        $"application cancel-download 0x{app.Identifier} AND " +
                        $"application list-downloading-content-meta 0x{app.Identifier} ");
                    CheckAppplicationList(executor);
                    ExpectEvaluator.IsTrue(download);
                    // wait-download のオプションは OR 条件で、指定条件のどれかを満たした場合に待機が終了する
                    // 上記の条件では ver.1 未満でダウンロード開始されるまで指定タイムアウト分待機することになる
                    // 既にローカルでインストールしているので、新たなダウンロードが開始されるまで待機している
                    {
                        // expect: isAutoUpdateEnabled(true -> false)
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordDetail.Find(
                            executor.OutputStream.Standard.ToString(),
                            app.Identifier,
                            new SigloHelper.ResultMatcher.Application.ListRecordDetail.ExpectValues()
                            {
                                {  SigloHelper.ResultMatcher.Application.ListRecordDetail.Property.isAutoUpdateEnabled, false },
                            }));

                        // expect: not changed
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(
                            executor.OutputStream.Standard.ToString(),
                            app.Identifier,
                            app.Version,
                            new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                            {
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.isDownloading, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainInstallRecord, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainEntity, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAddOnContentRecord, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAllAddOnContentEntity, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAnyAddOnContentEntity, false },
                            }));
                        // 仕様変更：ダウンロード開始でアプリケーション記録を行うのは新規追加のみ（SIGLO-60839）
                    }

                    VerifyAppplication(executor, app.Identifier);
                    {
                        // expect: app only
                        var output = executor.OutputStream.Standard.ToString();
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            output, new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app)));
                        ExpectEvaluator.IsEqual(1, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            output, new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(aoc)));
                    }

                    // chekc ticket rights
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application content-meta-status 0x{app.Identifier} --check-rights"));
                    {
                        // expect: app only
                        var output = executor.OutputStream.Standard.ToString();
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ContentMetaStatusCheckRights.Find(
                            output, new SigloHelper.ResultMatcher.ContentValue(app,
                                SigloHelper.ResultMatcher.ContentValue.Storage.Any,
                                SigloHelper.ResultMatcher.ContentValue.RightsCheck.CommonRights)));
                        ExpectEvaluator.IsEqual(1, SigloHelper.ResultMatcher.Application.ContentMetaStatusCheckRights.Find(
                            output, new SigloHelper.ResultMatcher.ContentValue(aoc,
                                SigloHelper.ResultMatcher.ContentValue.Storage.None,
                                SigloHelper.ResultMatcher.ContentValue.RightsCheck.NotChecked)
                        ));
                    }

                    // unregister device account
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "shop unlink-device 0 AND " +
                        "account clear_all AND " +
                        "shop unregister-device-account "));
                }
                CleanupContents(executor);

                // 検証条件：アプリの記録がない状態で、追加コンテンツを購入して、タスクをキャンセルすると、追加コンテンツの記録が追加されている
                scope.WriteLineWithTimeStamp("record aoc when cancel download task");
                {
                    // contents (deverted contents on TestForPurchaseDemoTicket)
                    //  Application : 0x01003ab001e32000, version 0
                    //  Patch       : 0x01003ab001e32800, version 65536
                    //  AddOnContent: 0x01003ab001e33001, version 0, index 1
                    var app = new ContentMeta.Contexture.Basic(
                        new ID64(0x01003ab001e32000), 0, ContentMeta.Type.Application);
                    var patch = new ContentMeta.Contexture.Basic(
                        new ID64(0x01003ab001e32800), 65536, ContentMeta.Type.Patch);
                    var aoc = new ContentMeta.Contexture.Basic(
                        new ID64(0x01003ab001e33001), 0, ContentMeta.Type.AddOnContent);

                    // register device account
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"account add --register_nsa AND " +
                        $"account link --index 0 --id {NintendoAccount.ID} --password {NintendoAccount.PASSWORD} "));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"shop register-device-account AND " +
                        $"shop shop-account-status 0 "));
                    ExpectEvaluator.IsTrue(D4cHelper.ClearDownloadTaskListOnShopServer(executor));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"shop link-device 0 "));

                    // purchase and cancel download
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"shop download-demo 0 --id 0x{aoc.Identifier} "));
                    var download = executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{app.Identifier} --version 65536 --download-progress 0/1 --timeout 300000 AND " +
                        $"application cancel-download 0x{app.Identifier} AND " +
                        $"application list-downloading-content-meta 0x{app.Identifier} ");
                    CheckAppplicationList(executor);
                    ExpectEvaluator.IsTrue(download);
                    // wait-download のオプションは OR 条件で、指定条件のどれかを満たした場合に待機が終了する
                    // 上記の条件では ver.1 未満でダウンロード開始されるまで指定タイムアウト分待機することになる
                    {
                        // expect: isAutoUpdateEnabled(false)
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordDetail.Find(
                            executor.OutputStream.Standard.ToString(),
                            app.Identifier,
                            new SigloHelper.ResultMatcher.Application.ListRecordDetail.ExpectValues()
                            {
                                {  SigloHelper.ResultMatcher.Application.ListRecordDetail.Property.isAutoUpdateEnabled, false },
                            }));

                        // expect: hasAddOnContentRecord(true)
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(
                            executor.OutputStream.Standard.ToString(),
                            app.Identifier,
                            app.Version,
                            new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                            {
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.isDownloading, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainInstallRecord, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainEntity, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAddOnContentRecord, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAllAddOnContentEntity, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAnyAddOnContentEntity, false },
                            }));
                    }

                    VerifyAppplication(executor, app.Identifier);
                    {
                        // expect: aoc only
                        var output = executor.OutputStream.Standard.ToString();
                        ExpectEvaluator.IsEqual(1, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            output, new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app)));
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            output, new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(aoc)));
                    }

                    // chekc ticket rights
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application content-meta-status 0x{app.Identifier} --check-rights"));
                    {
                        // expect: aoc only
                        ExpectEvaluator.IsEqual(1, SigloHelper.ResultMatcher.Application.ContentMetaStatusCheckRights.Find(
                            executor.OutputStream.Standard.ToString(),
                            new SigloHelper.ResultMatcher.ContentValue(app,
                                SigloHelper.ResultMatcher.ContentValue.Storage.Any,
                                SigloHelper.ResultMatcher.ContentValue.RightsCheck.Any)));
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ContentMetaStatusCheckRights.Find(
                            executor.OutputStream.Standard.ToString(),
                            new SigloHelper.ResultMatcher.ContentValue(aoc,
                                SigloHelper.ResultMatcher.ContentValue.Storage.None,
                                SigloHelper.ResultMatcher.ContentValue.RightsCheck.NotChecked)));
                    }

                    // retry downlaod
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application update 0x{app.Identifier} "));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{app.Identifier} --timeout 300000 "));

                    CheckAppplicationList(executor);
                    {
                        // expect: isAutoUpdateEnabled(false)
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordDetail.Find(
                            executor.OutputStream.Standard.ToString(),
                            app.Identifier,
                            new SigloHelper.ResultMatcher.Application.ListRecordDetail.ExpectValues()
                            {
                                {  SigloHelper.ResultMatcher.Application.ListRecordDetail.Property.isAutoUpdateEnabled, false },
                            }));

                        // expect: hasAllAddOnContentEntity(false -> true) hasAnyAddOnContentEntity(false -> true)
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(
                            executor.OutputStream.Standard.ToString(),
                            app.Identifier,
                            patch.Version,
                            new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                            {
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.isDownloading, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainInstallRecord, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainEntity, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAddOnContentRecord, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAllAddOnContentEntity, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasAnyAddOnContentEntity, true },
                            }));
                    }

                    VerifyAppplication(executor, app.Identifier);
                    {
                        // expect: aoc only
                        var output = executor.OutputStream.Standard.ToString();
                        ExpectEvaluator.IsEqual(1, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            output, new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app)));
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                            output, new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(aoc)));
                    }

                    // unregister device account
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "shop unlink-device 0 AND " +
                        "account clear_all AND " +
                        "shop unregister-device-account "));
                }
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// コンテンツ配信の開始・停止テスト
        /// 配信設定によりアプリケーション管理情報の取得内容が異なる
        /// 単体実行時で 3 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-50352")]
        public void TestForStartAndStopDelivery()
        {
            using (var scope = new TestMethodLog())
            {
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                var intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                // make contents
                //  Application : 0x01001a500005e0f1, version 0
                //  Patch       : 0x01001a500005e8f1, version 65536
                var param = new TestApplication.ApplicationParameter(
                    new ID64(0x01001a500005e0f1), 0, 5 * 1024 * 1024);
                param.AddPatch(1);
                param.UseSmallCode = true;

                var contents = TestApplication.MakeContents(
                    intermediate, new List<TestApplication.GenerateParameter<int>>() { param });

                var catalog = new GeneratedContentResult.TypeCategorizedCatalog(contents);
                var app = catalog.GetTypedCatalog(ContentMeta.Type.Application)[0];
                var patch = catalog.GetTypedCatalog(ContentMeta.Type.Patch)[0];

                // upload contents
                var uploader = new D4cHelper.NspUploader(
                    intermediate,
                    ProxyConfiguration,
                    ServerEnvironment);

                uploader.RegisterVersion(app.NspPath);
                uploader.DeleteVersion(patch.NspPath);
                ExpectEvaluator.IsTrue(D4cHelper.WaitUntilVersionListDeleted(executor, patch, true));

                // prepare
                CleanupContents(executor, true);

                // install for create application control data
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $"application install {app.NspPath} "));

                // show application control data
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $"application invalidate-cache application-control AND " +
                    $"application download-control 0x{app.Identifier} AND " +
                    $"application show-control 0x{app.Identifier} "));
                ExpectEvaluator.IsTrue(
                    MatchTargetLog(executor, $"Name: 0x{app.Identifier}_v{app.ReleaseVersion}_AmericanEnglish"));
                // インストールした状態だとインストール中の管理情報が返される。

                // cleanup contents
                CleanupContents(executor);

                // show application control data
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $"application invalidate-cache application-control AND " +
                    $"application download-control 0x{app.Identifier} AND " +
                    $"application show-control 0x{app.Identifier} "));
                ExpectEvaluator.IsTrue(
                    MatchTargetLog(executor, $"Name: 0x{app.Identifier}_v{patch.ReleaseVersion}_AmericanEnglish"));
                // アンインストールした状態だと atum 上の管理情報が返される。
                // atum では、配信状態の最新バージョンを返すので、
                // supefly の登録バージョンに関わらず、v1 になってしまう。

                // stop delivery
                var rops = new D4cHelper.Rops(
                    intermediate,
                    CliTokenAccount,
                    ProxyConfiguration,
                    ServerEnvironment);

                rops.ListTitles(patch);
                rops.StopDelivery(patch);
                rops.ApproveTitle(patch);
                rops.ListTitles(patch);

                // show application control data
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $"application invalidate-cache application-control AND " +
                    $"application download-control 0x{app.Identifier} AND " +
                    $"application show-control 0x{app.Identifier} "));
                ExpectEvaluator.IsTrue(
                    MatchTargetLog(executor, $"Name: 0x{app.Identifier}_v{app.ReleaseVersion}_AmericanEnglish"));
                // osiris, atum で patch の配信を停止したら v0 になる。

                // start delivery
                rops.StartDelivery(patch);
                rops.ApproveTitle(patch);
                rops.ListTitles(patch);
                // このテストの単体実行用に配信設定を元に戻しておく。
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// コンテンツ配信必須システムバージョンが検知されないことを確認
        /// TestForPurchaseDemoTicket でアップロードしたコンテンツを流用しています。
        /// 通知待機するため時間がかかります。
        /// 単体実行時で 20 ～ 25 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-51779")]
        public void TestForCheckContentDelivery()
        {
            var UploadTimeout = new System.TimeSpan(0, 5, 0);         // 失敗時のリトライも含めて長めに設定
            var DownloadTimeout = new System.TimeSpan(0, 3, 0);       // 通知～完了までの実績値が 1 分以内
            var VersionListInterval = new System.TimeSpan(0, 1, 0);   // supefly -> tagaya の取得間隔分を空ける

            var ResponseTrue = @"[\r\n]true[\r\n]";
            var ResponseFalse = @"[\r\n]false[\r\n]";

            using (var scope = new TestMethodLog())
            {
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                // finally
                scope.AddDisposer(() =>
               {
                   // change to not required system update
                   executor.RunDevMenuCommandSystem("systemupdate set-debug-id-for-content-delivery 0-0 ");
                   ExpectEvaluator.IsTrue(executor.ResetTarget());

                   // unregister device
                   executor.RunDevMenuCommandSystem("shop unlink-device-all");
                   executor.RunDevMenuCommandSystem("account clear_all");
                   executor.RunDevMenuCommandSystem("shop unregister-device-account");
                   if (!IsUnitTestOnIDE)
                   {
                       CleanupContents(executor, true);
                   }
               });

                // prepare
                CleanupContents(executor, true);

                // 検証条件：サーバ上のバージョンリストを更新し、当該バージョンリストの通知が来ても、
                //           コンテンツ配信必須システムバージョンが検知されないことを確認する
                scope.WriteLineWithTimeStamp("check content delivery by version list");
                {
                    // make contents
                    //  Application : 0x01001a500005e0f2, version 0
                    //  Patch       : 0x01001a500005e8f2, version 65536
                    var param = new TestApplication.ApplicationParameter(
                        new ID64(0x01001a500005e0f2), 0, 5 * 1024 * 1024);
                    param.AddPatch(1);
                    param.UseSmallCode = true;

                    var contents = TestApplication.MakeContents(
                        intermediate, new List<TestApplication.GenerateParameter<int>>() { param });

                    var catalog = new GeneratedContentResult.TypeCategorizedCatalog(contents);
                    var app = catalog.GetTypedCatalog(ContentMeta.Type.Application)[0];
                    var patch = catalog.GetTypedCatalog(ContentMeta.Type.Patch)[0];

                    // upload contents
                    var uploader = new D4cHelper.NspUploader(
                        intermediate,
                        ProxyConfiguration,
                        ServerEnvironment);

                    uploader.RegisterVersion(app.NspPath);
                    uploader.DeleteVersion(patch.NspPath);
                    ExpectEvaluator.IsTrue(D4cHelper.WaitUntilVersionListDeleted(executor, patch, true));

                    // change to required system update
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "systemupdate set-debug-id-for-content-delivery 1-0 "));
                    ExpectEvaluator.IsTrue(executor.ResetTarget());

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "systemupdate check-content-delivery "));
                    ExpectEvaluator.IsTrue(MatchTargetLog(executor, ResponseFalse));

                    // install and launch
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application install {app.NspPath} AND " +
                        $"application launch 0x{app.Identifier} "));
                    CheckAppplicationList(executor);

                    // version-up (0 -> 1)
                    System.Threading.Thread.Sleep(VersionListInterval);
                    uploader.RegisterVersion(patch.NspPath);

                    var download = executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{app.Identifier} --version {patch.Version} --download-progress 0/1 --timeout {DownloadTimeout.TotalMilliseconds} ");
                    CheckAppplicationList(executor);
                    ExpectEvaluator.IsTrue(!download);
                    // バージョンリストを受信するがコンテンツ配信必須システムバージョンを満たさないため自動更新されない

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application notification-info "));

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "systemupdate check-content-delivery "));
                    ExpectEvaluator.IsTrue(MatchTargetLog(executor, ResponseFalse));
                    // バージョンリスト受信では、コンテンツ必須配信システムバージョンが高いことを検知しない

                    // cold boot
                    ExpectEvaluator.IsTrue(RebootTargetByPowerButton(executor));
                    System.Threading.Thread.Sleep(VersionListInterval);

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "systemupdate check-content-delivery "));
                    ExpectEvaluator.IsTrue(MatchTargetLog(executor, ResponseFalse));
                    // コールドブート後はコンテンツ配信必須システムバージョンは検知されない

                    // change to not required system update
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "systemupdate set-debug-id-for-content-delivery 0-0 "));
                    ExpectEvaluator.IsTrue(RebootTargetByPowerButton(executor));

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "systemupdate check-content-delivery "));
                    ExpectEvaluator.IsTrue(MatchTargetLog(executor, ResponseFalse));
                    // 無効な値に変更したのでコンテンツ配信必須システムバージョンは検知されない

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{app.Identifier} --version {patch.Version} --timeout {DownloadTimeout.TotalMilliseconds} "));
                    CheckAppplicationList(executor);
                    // コンテンツ配信必須システムバージョン解除後は自動更新が成功する
                }
                CleanupContents(executor);

                // 検証条件：サーバ上の DTL を更新し、DTL の通知が来たらコンテンツ配信必須システムバージョンが検知されることを確認する
                scope.WriteLineWithTimeStamp("check content delivery by download task");
                {
                    // contents (deverted contents on TestForPurchaseDemoTicket)
                    //  Application : 0x01003ab001e32000, version 0
                    var app = new ContentMeta.Contexture.Basic(
                        new ID64(0x01003ab001e32000), 0, ContentMeta.Type.Application);

                    // register device account
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"account add --register_nsa AND " +
                        $"account link --index 0 --id {NintendoAccount.ID} --password {NintendoAccount.PASSWORD} "));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"shop register-device-account AND " +
                        $"shop shop-account-status 0 "));
                    ExpectEvaluator.IsTrue(D4cHelper.ClearDownloadTaskListOnShopServer(executor));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"shop link-device 0 "));

                    // change to required system update
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "systemupdate set-debug-id-for-content-delivery 1-0 "));
                    ExpectEvaluator.IsTrue(executor.ResetTarget());

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "systemupdate check-content-delivery "));
                    ExpectEvaluator.IsTrue(MatchTargetLog(executor, ResponseFalse));

                    // purchase and wait
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"shop download-demo 0 --id 0x{app.Identifier} "));
                    var download = executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{app.Identifier} --download-progress 0/1 --timeout {DownloadTimeout.TotalMilliseconds} ");
                    CheckAppplicationList(executor);
                    ExpectEvaluator.IsTrue(!download);
                    // DTL を受信するがコンテンツ配信必須システムバージョンを満たさないためダウンロードされない

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application notification-info "));

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "systemupdate check-content-delivery "));
                    ExpectEvaluator.IsTrue(MatchTargetLog(executor, ResponseTrue));
                    // DTL 受信では、従来通りコンテンツ必須配信システムバージョンが高いことを検知する

                    // cold boot
                    ExpectEvaluator.IsTrue(RebootTargetByPowerButton(executor));
                    System.Threading.Thread.Sleep(VersionListInterval);

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "systemupdate check-content-delivery "));
                    ExpectEvaluator.IsTrue(MatchTargetLog(executor, ResponseFalse));
                    // コールドブート後はコンテンツ配信必須システムバージョンは検知されない

                    // change to not required system update
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "systemupdate set-debug-id-for-content-delivery 0-0 "));
                    ExpectEvaluator.IsTrue(RebootTargetByPowerButton(executor));

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "systemupdate check-content-delivery "));
                    ExpectEvaluator.IsTrue(MatchTargetLog(executor, ResponseFalse));
                    // 無効な値に変更したのでコンテンツ配信必須システムバージョンは検知されない

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{app.Identifier} --version {app.Version} --timeout {DownloadTimeout.TotalMilliseconds} "));
                    CheckAppplicationList(executor);
                    // コンテンツ配信必須システムバージョンが解除されたのでダウンロードが成功する

                    // unregister device account
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "shop unlink-device 0 AND " +
                        "account clear_all AND " +
                        "shop unregister-device-account "));
                }
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// アプリがオンライン中でもタスク登録・管理データ受信できることを確認
        /// TestForPurchaseDemoTicket でアップロードしたコンテンツを流用しています。
        /// 通知待機するため時間がかかります。
        /// 単体実行時で 13 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-52356")]
        public void TestForVerifyWhileOnlineApplication()
        {
            var UploadTimeout = new System.TimeSpan(0, 5, 0);         // 失敗時のリトライも含めて長めに設定
            var DownloadTimeout = new System.TimeSpan(0, 3, 0);       // 通知～完了までの実績値が 1 分以内
            var VersionListInterval = new System.TimeSpan(0, 1, 0);   // supefly -> tagaya の取得間隔分を空ける

            using (var scope = new TestMethodLog())
            {
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                // finally
                scope.AddDisposer(() =>
               {
                   // change to not required system update
                   executor.RunDevMenuCommandSystem("systemupdate set-debug-id-for-content-delivery 0-0 ");
                   ExpectEvaluator.IsTrue(executor.ResetTarget());

                   // unregister device
                   executor.RunDevMenuCommandSystem("shop unlink-device-all");
                   executor.RunDevMenuCommandSystem("account clear_all");
                   executor.RunDevMenuCommandSystem("shop unregister-device-account");
                   if (!IsUnitTestOnIDE)
                   {
                       CleanupContents(executor, true);
                   }
               });

                // prepare
                CleanupContents(executor, true);

                // online application
                GeneratedContentResult onlineApp;
                {
                    // make contents (local only)
                    //  Application : 0x01001a500005e0f3, version 0
                    var param = new TestApplication.ApplicationParameter(
                        new ID64(0x01001a500005e0f3), 0, 5 * 1024 * 1024);
                    param.RequestNetwork = true;

                    var contents = TestApplication.MakeContents(
                        intermediate, new List<TestApplication.GenerateParameter<int>>() { param });

                    var catalog = new GeneratedContentResult.TypeCategorizedCatalog(contents);
                    onlineApp = catalog.GetTypedCatalog(ContentMeta.Type.Application)[0];
                }

                // 検証条件：アプリのオンライン中にサーバ上のバージョンリストを更新して以下を確認する
                //  ・自動更新によるタスクの登録は可能
                //  ・登録されたタスクについて管理データの受信は可能
                //  ・オンライン中はコンテンツはダウンロードされない
                scope.WriteLineWithTimeStamp("notify via version list while application online");
                {
                    // make contents
                    //  Application : 0x01001a500005e0f4, version 0
                    //  Patch       : 0x01001a500005e8f4, version 65536
                    var param = new TestApplication.ApplicationParameter(
                        new ID64(0x01001a500005e0f4), 0, 5 * 1024 * 1024);
                    param.AddPatch(1);
                    param.UseSmallCode = true;

                    var contents = TestApplication.MakeContents(
                        intermediate, new List<TestApplication.GenerateParameter<int>>() { param });

                    var catalog = new GeneratedContentResult.TypeCategorizedCatalog(contents);
                    var app = catalog.GetTypedCatalog(ContentMeta.Type.Application)[0];
                    var patch = catalog.GetTypedCatalog(ContentMeta.Type.Patch)[0];

                    // upload contents
                    var uploader = new D4cHelper.NspUploader(
                        intermediate,
                        ProxyConfiguration,
                        ServerEnvironment);

                    uploader.RegisterVersion(app.NspPath);
                    uploader.DeleteVersion(patch.NspPath);
                    ExpectEvaluator.IsTrue(D4cHelper.WaitUntilVersionListDeleted(executor, patch, true));

                    // install and launch for enable auto-update
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application install {app.NspPath} AND " +
                        $"application launch 0x{app.Identifier} "));
                    CheckAppplicationList(executor);

                    // check application control data
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application invalidate-cache application-control AND " +
                        $"application download-control 0x{app.Identifier} AND " +
                        $"application show-control 0x{app.Identifier} "));
                    ExpectEvaluator.IsTrue(
                        MatchTargetLog(executor, $"Name: 0x{app.Identifier}_v{app.ReleaseVersion}_AmericanEnglish"));

                    // keep running online application
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application install {onlineApp.NspPath} AND " +
                        $"application launch 0x{onlineApp.Identifier} "));

                    // version-up (0 -> 1)
                    System.Threading.Thread.Sleep(VersionListInterval);
                    uploader.RegisterVersion(patch.NspPath);

                    var download = executor.RunDevMenuCommandSystemSuppressAutoKill(
                        $"--time application wait-download 0x{app.Identifier} --version {patch.Version} --download-progress 0/1 --timeout {DownloadTimeout.TotalMilliseconds} ");
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystemSuppressAutoKill(
                        $"application list --detail AND " +
                        $"application list-record-detail AND " +
                        $"application list-view "));
                    ExpectEvaluator.IsTrue(!download);
                    {
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(
                            executor.OutputStream.Standard.ToString(),
                            app.Identifier, app.Version,
                            new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                            {
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchRecord, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainEntity, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchEntity, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.isDownloading, true },
                            }));
                    }
                    // バージョンリストを受信するがアプリがオンライン中なので自動更新されない
                    // 仕様変更：自動更新タスク登録時には記録を更新しない（SIGLO-60647）

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystemSuppressAutoKill(
                        $"application notification-info AND " +
                        $"application list-download-task-status "));

                    // check application control data
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystemSuppressAutoKill(
                        $"application invalidate-cache application-control AND " +
                        $"application download-control 0x{app.Identifier} AND " +
                        $"application show-control 0x{app.Identifier} "));
                    ExpectEvaluator.IsTrue(
                        MatchTargetLog(executor, $"Name: 0x{app.Identifier}_v{app.ReleaseVersion}_AmericanEnglish"));

                    // terminate online application and wait download
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{app.Identifier} --version {patch.Version} --timeout {DownloadTimeout.TotalMilliseconds} "));
                    CheckAppplicationList(executor);
                    {
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(
                            executor.OutputStream.Standard.ToString(),
                            app.Identifier, patch.Version,
                            new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                            {
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchRecord, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainEntity, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchEntity, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.isDownloading, false },
                            }));
                    }
                    // オンライン中のアプリが終了したのでダウンロードが成功する

                    // check application control data
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystemSuppressAutoKill(
                        $"application invalidate-cache application-control AND " +
                        $"application download-control 0x{app.Identifier} AND " +
                        $"application show-control 0x{app.Identifier} "));
                    ExpectEvaluator.IsTrue(
                        MatchTargetLog(executor, $"Name: 0x{app.Identifier}_v{patch.ReleaseVersion}_AmericanEnglish"));
                }
                CleanupContents(executor);

                // 検証条件：アプリのオンライン中にサーバ上の DTL を更新して以下を確認する
                //  ・DTL によるタスクの登録は可能
                //  ・登録されたタスクについて管理データの受信は可能
                //  ・オンライン中はコンテンツはダウンロードされない
                scope.WriteLineWithTimeStamp("notify via download task while application online");
                {
                    // contents (deverted contents on TestForPurchaseDemoTicket)
                    //  Application : 0x01003ab001e32000, version 0
                    //  Patch       : 0x01003ab001e32800, version 65536
                    var app = new ContentMeta.Contexture.Basic(
                        new ID64(0x01003ab001e32000), 0, ContentMeta.Type.Application);
                    var patch = new ContentMeta.Contexture.Basic(
                        new ID64(0x01003ab001e32800), 65536, ContentMeta.Type.Patch);

                    // register device account
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"account add --register_nsa AND " +
                        $"account link --index 0 --id {NintendoAccount.ID} --password {NintendoAccount.PASSWORD} "));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"shop register-device-account AND " +
                        $"shop shop-account-status 0 "));
                    ExpectEvaluator.IsTrue(D4cHelper.ClearDownloadTaskListOnShopServer(executor));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"shop link-device 0 "));

                    // check application control data
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application invalidate-cache application-control AND " +
                        $"application download-control 0x{app.Identifier} AND " +
                        $"application show-control 0x{app.Identifier} "));
                    ExpectEvaluator.IsTrue(
                        MatchTargetLog(executor, $"Name: 0x{app.Identifier}_v{patch.ReleaseVersion}_AmericanEnglish"));

                    // keep running online application
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application install {onlineApp.NspPath} AND " +
                        $"application launch 0x{onlineApp.Identifier} "));

                    // purchase and wait
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystemSuppressAutoKill(
                        $"shop download-demo 0 --id 0x{app.Identifier} "));
                    var download = executor.RunDevMenuCommandSystemSuppressAutoKill(
                        $"--time application wait-download 0x{app.Identifier} --download-progress 0/1 --timeout {DownloadTimeout.TotalMilliseconds} ");
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystemSuppressAutoKill(
                        $"application list --detail AND " +
                        $"application list-record-detail AND " +
                        $"application list-view "));
                    ExpectEvaluator.IsTrue(!download);
                    {
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(
                            executor.OutputStream.Standard.ToString(),
                            app.Identifier, patch.Version,
                            new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                            {
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchRecord, true },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainEntity, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchEntity, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.isDownloading, true },
                            }));
                    }
                    // DTL を受信するがアプリがオンライン中なのでダウンロードされない

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystemSuppressAutoKill(
                        $"application notification-info AND " +
                        $"application list-download-task-status "));

                    // check application control data
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystemSuppressAutoKill(
                        $"application invalidate-cache application-control AND " +
                        $"application download-control 0x{app.Identifier} AND " +
                        $"application show-control 0x{app.Identifier} "));
                    ExpectEvaluator.IsTrue(
                        MatchTargetLog(executor, $"Name: 0x{app.Identifier}_v{patch.ReleaseVersion}_AmericanEnglish"));

                    // terminate online application and wait download
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"--time application wait-download 0x{app.Identifier} --version {patch.Version} --timeout {DownloadTimeout.TotalMilliseconds} "));
                    CheckAppplicationList(executor);
                    // オンライン中のアプリが終了したのでダウンロードが成功する

                    // unregister device account
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"shop unlink-device 0 AND " +
                        $"account clear_all AND " +
                        $"shop unregister-device-account "));
                }
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// ほんとは PatchContexture で指定できるようにするのが綺麗。
        /// これは UnitMain.cs 側だけで回避するためのやり方です。
        /// </summary>
        //!----------------------------------------------------------------------------
        private class ApplicationParameterWithPatchSizeMap : TestApplication.ApplicationParameter
        {
            public Dictionary<int, ulong> PatchSizeMap { get; }

            public ApplicationParameterWithPatchSizeMap(ID64 id, int version, ulong size)
                : base(id, version, size)
            {
                PatchSizeMap = new Dictionary<int, ulong>(8);
            }

            public override TestApplication.GenerateParameter<int> CreatePatchParameter(int newPatchProperty)
            {
                ulong outValue;
                var newParam = (TestApplication.ApplicationParameter)base.CreatePatchParameter(newPatchProperty);
                newParam.Size = (PatchSizeMap.TryGetValue(newPatchProperty, out outValue)) ? outValue : Size;
                return newParam;
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// タスクのまとめて再開機能のテスト
        /// 単体実行時で 3 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-52035")]
        public void TestForResumeAllTasks()
        {
            System.Func<string, string, int> CountPatterns = (string s, string p) =>
            {
                return (s.Length - s.Replace(p, "").Length) / p.Length;
            };

            using (var scope = new TestMethodLog())
            {
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                var intermediate = GenerateIntermediateDirectoryAsMethod(executor);
                var method = MethodBase.GetCurrentMethod();

                // 本テストのメタ構成は以下の通り。
                // 連番Index | Application ID     | Patch ID           | Version |
                //         0 | 0x01003ab001e30050 | 0x01003ab001e30850 | 0, 1, 2 |
                //         : |          :         |          :         |    :    |
                //         : |          :         |          :         |    :    |
                //         2 | 0x01003ab001e30052 | 0x01003ab001e30853 | 0, 1, 2 |

                uint appCount = 3U;
                var IdForApplication = new ID64(0x01003ab001e30050);

                // make parameter of contents.
                var gens = new List<TestApplication.GenerateParameter<int>>((int)appCount);
                for (uint index = 0; index < appCount; ++index)
                {
                    var now = IdForApplication + index;
                    var app = new ApplicationParameterWithPatchSizeMap(now, 0, 32);
                    app.PatchSizeMap.Add(2, 10 * 1024 * 1024);
                    app.AddPatches(1, new TestApplication.PatchContexture<int>(2, AuthoringExecutor.Patch.FlagOption.UseDelta));
                    app.UseSmallCode = true;
                    gens.Add(app);
                }

                executor.RunDevMenuCommandSystem("patch set-prefer-delta true");
                executor.ResetTarget();
                scope.AddDisposer(() =>
                {
                    executor.RunDevMenuCommandSystem(
                        string.Format("application uninstall 0x{0}", IdForApplication) + " THEN " +
                        string.Format("application uninstall 0x{0}", IdForApplication + 1) + " THEN " +
                        string.Format("application uninstall 0x{0}", IdForApplication + 2) + " THEN " +
                        "debug empty-nand-free-space THEN " +
                        "debug empty-sdcard-free-space THEN " +
                        "patch set-prefer-delta false");
                    executor.ResetTarget();
                });

                executor.RunDevMenuCommandSystem("debug fill-sdcard-free-space AND debug fill-nand-free-space");

                // 1.ダウンロードタスクの容量不足でまとめて再開するか
                executor.RunDevMenuCommandSystem(
                    string.Format("application create-download-task 0x{0}", IdForApplication) + " AND " +
                    string.Format("application create-download-task 0x{0}", IdForApplication + 1) + " AND " +
                    string.Format("application create-download-task 0x{0}", IdForApplication + 2) + " AND " +
                    string.Format("application wait-download 0x{0}", IdForApplication + 2));

                // list-view で NotEnoughSpace を確認
                executor.RunDevMenuCommandSystem("application list-view");
                ExpectEvaluator.IsTrue(CountPatterns(executor.OutputStream.Standard.ToString(), "\"state\": \"NotEnoughSpace\"") == 3);

                // 容量を開けずに Resume --all
                executor.RunDevMenuCommandSystem(
                    string.Format("application resume-download --all AND application wait-download 0x{0}", IdForApplication + 2));

                // list-view で NotEnoughSpace が残る
                executor.RunDevMenuCommandSystem("application list-view");
                ExpectEvaluator.IsTrue(CountPatterns(executor.OutputStream.Standard.ToString(), "\"state\": \"NotEnoughSpace\"") == 3);

                // 容量を開けて　Resume --all
                executor.RunDevMenuCommandSystem(
                    "debug empty-nand-free-space AND " +
                    string.Format("application resume-download --all AND application wait-download 0x{0}", IdForApplication + 2));

                // list-view で NotEnoughSpace が 0
                executor.RunDevMenuCommandSystem("application list-view");
                ExpectEvaluator.IsTrue(CountPatterns(executor.OutputStream.Standard.ToString(), "\"state\": \"NotEnoughSpace\"") == 0);

                // 2.ダウンロードタスクが Suspended で再開するか
                // パッチ v1 のダウンロードを行い、まとめて再開するかを見る
                executor.RunDevMenuCommandSystem(
                    string.Format("application create-download-task 0x{0} --type Patch --id {1} --version 65536", IdForApplication, IdForApplication + 0x800) + " AND " +
                    string.Format("application create-download-task 0x{0} --type Patch --id {1} --version 65536", IdForApplication + 1, IdForApplication + 0x801) + " AND " +
                    string.Format("application set-task-state 0x{0} --state Suspended", IdForApplication) + " AND " +
                    string.Format("application set-task-state 0x{0} --state Suspended", IdForApplication + 1));

                // list-view で Suspended が 2
                executor.RunDevMenuCommandSystem("application list-view");
                ExpectEvaluator.IsTrue(CountPatterns(executor.OutputStream.Standard.ToString(), "\"state\": \"Suspended\"") == 2);

                // Resume --all でタスクが成功する
                executor.RunDevMenuCommandSystem(
                    string.Format("application resume-download --all AND application wait-download 0x{0}", IdForApplication + 1));

                // list-view で Suspended が 0
                executor.RunDevMenuCommandSystem("application list-view");
                ExpectEvaluator.IsTrue(CountPatterns(executor.OutputStream.Standard.ToString(), "\"state\": \"Suspended\"") == 0);

                // 3.NotEnoughSpace の適用タスクが混ざっていても再開するか

                // 差分を一個ダウンロードできる程度の容量だけ開けて埋める。差分一個 10MB なので、15MB 開けておく
                // 差分の適用一個、差分のダウンロード一個、直接パッチダウンロード一個ができて、それぞれ NotEnoughSpace になる
                executor.RunDevMenuCommandSystem(string.Format("debug fill-nand-free-space --margin {0}", 15 * 1024));
                executor.RunDevMenuCommandSystem(
                    string.Format("application create-download-task 0x{0} --type Patch --id {1} --version 131072", IdForApplication, IdForApplication + 0x800) + " AND " +
                    string.Format("application create-download-task 0x{0} --type Patch --id {1} --version 131072", IdForApplication + 1, IdForApplication + 0x801) + " AND " +
                    string.Format("application create-download-task 0x{0} --type Patch --id {1} --version 131072", IdForApplication + 2, IdForApplication + 0x802) + " AND " +
                    string.Format("application wait-download 0x{0}", IdForApplication + 2));

                // list-view で NotEnoughSpace を確認
                executor.RunDevMenuCommandSystem("application list-view");
                ExpectEvaluator.IsTrue(CountPatterns(executor.OutputStream.Standard.ToString(), "\"state\": \"NotEnoughSpace\"") == 3);

                // 容量を開けずに Resume --all
                executor.RunDevMenuCommandSystem(
                    string.Format("application resume-download --all AND application wait-download 0x{0}", IdForApplication + 2));

                // list-view で NotEnoughSpace が残る
                executor.RunDevMenuCommandSystem("application list-view");
                ExpectEvaluator.IsTrue(CountPatterns(executor.OutputStream.Standard.ToString(), "\"state\": \"NotEnoughSpace\"") == 3);

                // 容量を開けて　Resume --all
                executor.RunDevMenuCommandSystem(
                    "debug empty-nand-free-space AND " +
                    string.Format("application resume-download --all AND application wait-download 0x{0}", IdForApplication + 2));

                // list-view で NotEnoughSpace が 0
                executor.RunDevMenuCommandSystem("application list-view");
                ExpectEvaluator.IsTrue(CountPatterns(executor.OutputStream.Standard.ToString(), "\"state\": \"NotEnoughSpace\"") == 0);
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// ダウンロード中アプリの容量とストレージの空き容量の整合性検証。
        /// TestForDTLMultiCombination が実施され、生成コンテンツ( nsp, dtl )がローカルに存在する事が前提。
        /// UploadData_app_patch_v1_aoc_i1v0_i2v0 のデータ使用
        /// 単体実行時で 4 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-51948")]
        public void TestForVerifyConsistencyOfStorageCapacity()
        {
            using (var scope = new TestMethodLog())
            {
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                var preset = GenerateIntermediateDirectoryAsMethod(executor, "TestForDTLMultiCombination");
                var intermediate = GenerateIntermediateDirectoryAsMethod(executor);
                var method = MethodBase.GetCurrentMethod();

                // 本テストのメタ構成は以下の通り。
                // Application : 0x01003ab001e30001, version 0
                // Patch       : 0x01003ab001e30801, version 65536
                // AoC index 1 : 0x01003ab001e31002, version 0
                // AoC index 2 : 0x01003ab001e31003, version 0
                var IdForApplication = new ID64(0x01003ab001e30001);

                // TestForDTLMultiCombination で既存生成済の nsp の GeneratedContentResult 取得.
                // make the title resource contexture without call the MakeTestApplication.
                var genApp = new TestApplication.ApplicationParameter(IdForApplication, 0);
                var genAddonA = new TestApplication.AddonParameter(IdForApplication, 0, 1);
                var genAddonB = new TestApplication.AddonParameter(IdForApplication, 0, 2);
                genApp.AddPatch(1);
                var title = TestApplication.MakeFromExistsContents(new AuthoringExecutor(), preset,
                    new List<TestApplication.GenerateParameter<int>>(3)
                    {
                        genApp, genAddonA, genAddonB
                    }
                );

                // コンテンツエンティティ数
                int nContentEntity = 0;
                foreach (var content in title)
                {
                    nContentEntity += content.ContentProperties.Count;
                    Log.WriteLine($"Content meta [ {content.Identifier}, {content.Type}, VER={content.Version} ] {{{{");
                    foreach (var p in content.ContentProperties)
                    {
                        Log.WriteLine($"    entity => {p.ToString()}");
                    }
                    Log.WriteLine("}}");
                }
                Log.WriteLine($"Detected total content entity => {nContentEntity}");

                // finally
                scope.AddDisposer(() =>
               {
                   executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all");
                   executor.RunDevMenuCommandSystem("debug enable-mount-sdcard");
                   executor.ResetTarget();
                   executor.RunDevMenuCommandSystem("sdcard format AND sdcard cleanup");
                   executor.ResetTarget();
               });

                // アプリケーション容量検出
                System.Func<string, string, string, double> ExtractApplicationOccupiedSizeValue = (lStream, lMatchStorage, lUnit) =>
                {
                    var result = SigloHelper.ResultMatcher.Application.OccupiedSize.ExtractBlock(lStream, lMatchStorage, lUnit);
                    if (null == result)
                    {
                        throw new UnexpectFailureException("Could not find the \"application occupied-size\" command logs.");
                    }
                    var lSizeApp = result.Application;
                    var lSizePatch = result.Patch;
                    var lSizeAoc = result.AddOnContent;
                    var lSizeTotal = lSizeApp + lSizePatch + lSizeAoc;
                    Log.WriteLine($"Application occupied size : {lSizeTotal} {lUnit} => [ {lSizeApp} + {lSizePatch} + {lSizeAoc} ].");
                    return lSizeTotal;
                };

                // アプリケーション消費容量取得要求 ( コマンド付き )
                System.Func<ID64, string, double> RequestApplicationOccupiedSize = (ID64 lApplicationId, string lMatchStorage) =>
                {
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystemUseOnlyFilter(
                        $"application occupied-size 0x{lApplicationId} -b k"
                    ));
                    return ExtractApplicationOccupiedSizeValue(executor.OutputStream.Standard.ToString(), lMatchStorage, "KB");
                };

                // ダウンロード時の要求サイズ検証
                System.Func<bool, bool> VerifyOnlineDownloadCase = (lOnBuiltin) =>
                {
                    if (lOnBuiltin)
                    {
                        ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug disable-mount-sdcard"));
                        ExpectEvaluator.IsTrue(executor.ResetTarget());
                    }

                    // Got storage-size symbol.
                    var lStorage = lOnBuiltin ? "builtin" : "sdcard";
                    var matchStorage = lOnBuiltin
                        ? SigloHelper.ResultMatcher.Application.OccupiedSize.Storage.BuiltInUser
                        : SigloHelper.ResultMatcher.Application.OccupiedSize.Storage.SdCard;

                    // 初期空き容量取得
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application clear-task-status-list THEN " +
                        $"application uninstall --all THEN " +
                        $"application storage-size {lStorage} -b k"
                    ));
                    var beforeFreeSpace = SigloHelper.ResultMatcher.Application.OccupiedSize.ExtractSizeValue(executor.OutputStream.Standard.ToString(), "Free space", "KB");

                    // ダウンロード開始
                    var dtl = System.IO.Path.Combine(preset, "case6_2.dtl.json");
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application push-download-task-list {dtl} AND " +
                        $"application wait-download 0x{IdForApplication} --download-progress 0/1"
                    ));

                    // ダウンロード中のアプリケーション要求サイズ情報取得
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystemUseOnlyFilter(
                        $"application occupied-size 0x{IdForApplication} -b k AND " +
                        $"application storage-size {lStorage} -b k"
                    ));
                    var stream1 = executor.OutputStream.Standard.ToString();
                    var totalSize = ExtractApplicationOccupiedSizeValue(stream1, matchStorage, "KB");
                    var afterFreeSpace = SigloHelper.ResultMatcher.Application.OccupiedSize.ExtractSizeValue(stream1, "Free space", "KB");
                    Log.WriteLine($"Storage[ {lStorage} ], Check the application occupied size, required condition that be ( {totalSize} > 0 ).");
                    ExpectEvaluator.IsGreater(totalSize, 0.0);

                    var diffFreeSpace = beforeFreeSpace - afterFreeSpace;
                    var diffThreshold = System.Math.Abs(diffFreeSpace - totalSize);
                    Log.WriteLine($"Storage[ {lStorage} ], Free space [ {beforeFreeSpace} ] KiB -> [ {afterFreeSpace} ] KiB, difference [ {diffFreeSpace} ] KiB.");
                    Log.WriteLine($"Storage[ {lStorage} ], Detected difference threshold ( {diffFreeSpace} - {totalSize} = {diffThreshold} ).");

                    // 差分有効範囲条件: ダウンロードコンテンツ * 「ストレージクラスタ * 4」
                    // ※ここでのコンテンツは、コンテンツメタではなく、*.cnmt.xml の <Content> 要素単位を示します。
                    // ※アプリnspの場合、Meta, Control, Program のコンテンツで構成されるので 3 * 4 = 12 になります。
                    // x4 の内訳は以下の通り。( SIGLO-60698 に詳細あり。)
                    // 各コンテンツごとに
                    // - registered/ 以下に作られるエントリ ( directory( hash, nca ), file )
                    // - placeholder 用に作られるエントリ ( directory )
                    // ストレージクラスタは本テスト構築時は 256KiB / 1クラスタ。(将来的にAPIから取得できる)
                    // 512/1クラスタだと、1コンテンツあたり 2MiBの下駄。
                    // 256/1クラスタだと、1コンテンツあたり 1MiBの下駄。
                    var expectThreshold = nContentEntity * (256.0 * 4.0); // app,patch,aoc(1,2) の4コンテンツメタ構成。
                    Log.WriteLine($"Storage[ {lStorage} ], Check the free space, required condition that be ( {diffThreshold} <= {expectThreshold} ).");
                    ExpectEvaluator.IsLesserThan(diffThreshold, expectThreshold);

                    // ダウンロード完了後もアプリケーション要求サイズの変動がないかチェック。
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application wait-download 0x{IdForApplication}"
                    ));
                    var donwloadedTotalSize = RequestApplicationOccupiedSize(IdForApplication, matchStorage);
                    Log.WriteLine($"Storage[ {lStorage} ], Check the application occupied size at downloaded, required condition that be ( {totalSize} == {donwloadedTotalSize} ).");
                    ExpectEvaluator.IsEqual(totalSize, donwloadedTotalSize);

                    if (lOnBuiltin)
                    {
                        executor.RunDevMenuCommandSystem("debug enable-mount-sdcard");
                        executor.ResetTarget();
                    }
                    return true;
                };

                // Initialize sdcard storage.
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-mount-sdcard"));
                ExpectEvaluator.IsTrue(executor.ResetTarget());
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("sdcard format AND sdcard cleanup"));
                ExpectEvaluator.IsTrue(executor.ResetTarget());

                //===============
                // ダウンロードは、app, patch, aoc(index1), aoc(index2) の4コンテンツ構成を nand/sdcard に対して検証
                //===============
                // case: download required on builtin.
                VerifyOnlineDownloadCase(true);

                // case: download required on sdcard.
                VerifyOnlineDownloadCase(false);

                //===============
                // GameCard は App 単体ケースの検証とする。
                //===============
                var hasInsertedGameCard = executor.RunDevMenuCommandSystem($"gamecard status", @"^Inserted$");
                if (hasInsertedGameCard)
                {
                    // finally
                    scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("gamecard erase"); });

                    // インストール対象 Applicatoin nsp情報取得.
                    var application = title[0];

                    // Clean initialize and game card write.
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application clear-task-status-list THEN " +
                        $"application uninstall --all THEN " +
                        $@"gamecard erase AND gamecard write ""{application.NspPath}"" --verify"
                    ));

                    var cardReadySize = RequestApplicationOccupiedSize(application.Identifier, SigloHelper.ResultMatcher.Application.OccupiedSize.Storage.BuiltInUser);
                    Log.WriteLine($"Storage[ Gamecard ], Check the application occupied size, required condition that be ( {cardReadySize} == 0 ).");
                    ExpectEvaluator.IsEqual(cardReadySize, 0.0);

                    // Local install to the storage of built-in.
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystemUseOnlyFilter(
                        $"application install {application.NspPath} -s builtin"
                    ));

                    var installedSize = RequestApplicationOccupiedSize(application.Identifier, SigloHelper.ResultMatcher.Application.OccupiedSize.Storage.BuiltInUser);
                    Log.WriteLine($"Storage[ Gamecard ], Check the application occupied size at builtin, required condition that be ( {installedSize} > 0 ).");
                    ExpectEvaluator.IsGreater(installedSize, 0.0);
                }
                else
                {
                    Log.WriteLine("Could not find the game card, Skip game card verification.");
                }
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// DTL で追加した際のダウンロードタスク状態を確認する
        /// ・ダウンロード中の追加
        /// ・ダウンロード後の追加
        /// ・コンテンツをアンインストールした後
        /// ・TLS をクリアした後
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-60965")]
        public void TestForVerifyTaskStatusWhenPushDTL()
        {
            using (var scope = new TestMethodLog())
            {
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                // finally
                if (!IsUnitTestOnIDE)
                {
                    scope.AddDisposer(() => { CleanupContents(executor); });
                }

                // prepare
                CleanupContents(executor, true);

                // contents (deverted contents on TestForDTLMultiCombination)
                //  Application : 0x01003ab001e30001, version 0
                //  Patch       : 0x01003ab001e30801, version 65536
                var app = new ContentMeta.Contexture.Basic(
                    new ID64(0x01003ab001e30001), 0, ContentMeta.Type.Application);
                var patch = new ContentMeta.Contexture.Basic(
                    new ID64(0x01003ab001e30801), 65536, ContentMeta.Type.Patch);

                scope.WriteLineWithTimeStamp("case1");
                {
                    // create dtl
                    var files = new List<DownloadTaskListFile>();
                    for (int i = 0; i < 3; i++)
                    {
                        var dtl = new DownloadTaskList();
                        var task = dtl.Create(app.Identifier);
                        task.AddNewTitle(app);

                        files.Add(DownloadTaskListFile.Create(System.IO.Path.Combine(intermediate, $"dtl-{i}.json"), dtl));
                    }

                    // push dtl and wait for download
                    {
                        CheckAppplicationList(executor);
                        {
                            // expect: not exist app
                            ExpectEvaluator.IsEqual(1, SigloHelper.ResultMatcher.Application.ListView.Find(
                                executor.OutputStream.Standard.ToString(),
                                app.Identifier, app.Version,
                                new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                                {
                                    {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, true },
                                }));
                        }

                        ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                            $"application push-download-task-list {files[0].Path} AND " +
                            $"application list-download-task-status AND " +
                            $"application wait-download {app.Identifier} --download-progress 0/1 "));
                        var output0 = executor.OutputStream.Standard.ToString();

                        ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                            $"application push-download-task-list {files[1].Path} AND " +
                            $"application list-download-task-status AND " +
                            $"application wait-download {app.Identifier} "));
                        var output1 = executor.OutputStream.Standard.ToString();

                        ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                            $"application push-download-task-list {files[2].Path} AND " +
                            $"application list-download-task-status "));
                        var output2 = executor.OutputStream.Standard.ToString();

                        // verify
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListDownloadTaskStatus.Find(
                            output0, files[0].DTL, SigloHelper.ResultMatcher.Application.ListDownloadTaskStatus.Status.Created));
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListDownloadTaskStatus.Find(
                            output1, files[1].DTL, SigloHelper.ResultMatcher.Application.ListDownloadTaskStatus.Status.Added));
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListDownloadTaskStatus.Find(
                            output2, files[2].DTL, SigloHelper.ResultMatcher.Application.ListDownloadTaskStatus.Status.AlreadyExists));

                        CheckAppplicationList(executor);
                        {
                            // expect: exist app
                            ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(
                                executor.OutputStream.Standard.ToString(),
                                app.Identifier, app.Version,
                                new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                                {
                                    {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, true },
                                    {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainEntity, true },
                                    {  SigloHelper.ResultMatcher.Application.ListView.Property.isDownloading, false },
                                }));
                        }
                        VerifyAppplication(executor, app.Identifier);
                        {
                            // expect: exist app
                            ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                                executor.OutputStream.Standard.ToString(),
                                new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app)));
                        }
                    }

                    // uninstall
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application uninstall --all "));

                    // push dtl and wait for download
                    {
                        CheckAppplicationList(executor);
                        {
                            // expect: not exist app
                            ExpectEvaluator.IsEqual(1, SigloHelper.ResultMatcher.Application.ListView.Find(
                                executor.OutputStream.Standard.ToString(),
                                app.Identifier, app.Version,
                                new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                                {
                                    {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, true },
                                }));
                        }

                        ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                            $"application push-download-task-list {files[2].Path} AND " +
                            $"application list-download-task-status "));
                        var output2 = executor.OutputStream.Standard.ToString();

                        ExpectEvaluator.IsFalse(executor.RunDevMenuCommandSystem(
                            $"application wait-download {app.Identifier} --version {patch.Version} --download-progress 0/1 --timeout 180000 "));

                        // verify
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListDownloadTaskStatus.Find(
                            output2, files[2].DTL, SigloHelper.ResultMatcher.Application.ListDownloadTaskStatus.Status.AlreadyExists));

                        CheckAppplicationList(executor);
                        {
                            // expect: not exist app
                            ExpectEvaluator.IsEqual(1, SigloHelper.ResultMatcher.Application.ListView.Find(
                                executor.OutputStream.Standard.ToString(),
                                app.Identifier, app.Version,
                                new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                                {
                                    {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, true },
                                }));
                        }
                    }
                }
                CleanupContents(executor);

                scope.WriteLineWithTimeStamp("case2");
                {
                    // create dtl
                    DownloadTaskListFile dtlPatchOnly = null;
                    {
                        var dtl = new DownloadTaskList();
                        var task = dtl.Create(app.Identifier);
                        task.AddNewTitle(patch);
                        dtlPatchOnly = DownloadTaskListFile.Create(System.IO.Path.Combine(intermediate, $"dtl-patch.json"), dtl);
                    }
                    DownloadTaskListFile dtlAppAndPatch = null;
                    {
                        var dtl = new DownloadTaskList();
                        var task = dtl.Create(app.Identifier);
                        task.AddNewTitle(app);
                        task.AddNewTitle(patch);
                        dtlAppAndPatch = DownloadTaskListFile.Create(System.IO.Path.Combine(intermediate, $"dtl-app_patch.json"), dtl);
                    }

                    // push DTL and wait for download
                    {
                        CheckAppplicationList(executor);
                        {
                            // expect: not exist app
                            ExpectEvaluator.IsEqual(1, SigloHelper.ResultMatcher.Application.ListView.Find(
                                executor.OutputStream.Standard.ToString(),
                                app.Identifier, patch.Version,
                                new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                                {
                                    {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, true },
                                }));
                        }

                        ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                            $"application push-download-task-list {dtlPatchOnly.Path} AND " +
                            $"application list-download-task-status AND " +
                            $"application wait-download {app.Identifier} --download-progress 0/1 "));
                        var output0 = executor.OutputStream.Standard.ToString();

                        ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                            $"application push-download-task-list {dtlAppAndPatch.Path} AND " +
                            $"application list-download-task-status AND " +
                            $"application wait-download {app.Identifier} "));
                        var output1 = executor.OutputStream.Standard.ToString();

                        // verify task status
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListDownloadTaskStatus.Find(
                            output0, dtlPatchOnly.DTL, SigloHelper.ResultMatcher.Application.ListDownloadTaskStatus.Status.Created));
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListDownloadTaskStatus.Find(
                            output1, dtlAppAndPatch.DTL, SigloHelper.ResultMatcher.Application.ListDownloadTaskStatus.Status.Added));

                        CheckAppplicationList(executor);
                        {
                            // expect: exist app and patch
                            ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(
                                executor.OutputStream.Standard.ToString(),
                                app.Identifier, patch.Version,
                                new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                                {
                                    {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, true },
                                    {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainEntity, true },
                                    {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchRecord, true },
                                    {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchEntity, true },
                                    {  SigloHelper.ResultMatcher.Application.ListView.Property.isDownloading, false },
                                }));
                        }
                        VerifyAppplication(executor, app.Identifier);
                        {
                            // expect: exist app and patch
                            ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                                executor.OutputStream.Standard.ToString(),
                                new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app),
                                new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch)));
                        }
                    }

                    // cleanup
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application uninstall --all AND " +
                        $"application clear-task-status-list "));

                    // push DTL and wait for download
                    {
                        CheckAppplicationList(executor);
                        {
                            // expect: not exist app
                            ExpectEvaluator.IsEqual(1, SigloHelper.ResultMatcher.Application.ListView.Find(
                                executor.OutputStream.Standard.ToString(),
                                app.Identifier, patch.Version,
                                new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                                {
                                    {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, true },
                                }));
                        }

                        ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                            $"application push-download-task-list {dtlAppAndPatch.Path} AND " +
                            $"application list-download-task-status AND " +
                            $"application wait-download {app.Identifier} "));

                        // verify task status
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListDownloadTaskStatus.Find(
                            executor.OutputStream.Standard.ToString(),
                            dtlAppAndPatch.DTL, SigloHelper.ResultMatcher.Application.ListDownloadTaskStatus.Status.Created));

                        CheckAppplicationList(executor);
                        {
                            // expect: exist app and patch
                            ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(
                                executor.OutputStream.Standard.ToString(),
                                app.Identifier, patch.Version,
                                new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                                {
                                    {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainRecord, true },
                                    {  SigloHelper.ResultMatcher.Application.ListView.Property.hasMainEntity, true },
                                    {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchRecord, true },
                                    {  SigloHelper.ResultMatcher.Application.ListView.Property.hasPatchEntity, true },
                                    {  SigloHelper.ResultMatcher.Application.ListView.Property.isDownloading, false },
                                }));
                        }
                        VerifyAppplication(executor, app.Identifier);
                        {
                            // expect: exist app and patch
                            ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListRecordInstalled.Find(
                                executor.OutputStream.Standard.ToString(),
                                new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(app),
                                new SigloHelper.ResultMatcher.Application.ListRecordInstalled.MatchingValue(patch)));
                        }
                    }
                }
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// 「実行中のアプリの追加コンテンツをコミットできる」検証。
        /// 単体実行時で 4 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-64729")]
        public void TestForCommitAocInApplicationRunning()
        {
            using (var scope = new TestMethodLog())
            {
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                var intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                // 本テストのメタ構成は以下の通り。
                // Patch は現状使わないが、別テストで使う予定なので構成事前準備。
                // Application : 0x01003ab001e30049, version 0      : UseSmallCode( false )
                // Patch       : 0x01003ab001e30849, version 65536  : UseSmallCode( false )
                // AoC index 1 : 0x01003ab001e3104a, version 0
                // AoC index 1 : 0x01003ab001e3104a, version 65536
                // AoC index 2 : 0x01003ab001e3104b, version 0
                var IdForApplication = new ID64(0x01003ab001e30049);

                // Make parameter for test contents generating.
                var genApp = new TestApplication.ApplicationParameter(IdForApplication, 0, 64 * 1024);
                genApp.AddPatch(1);
                genApp.ChangeCodePath(System.IO.Path.Combine(executor.GetOutputTestDirectory("ShopRuntimeAssistants"), "code"));
                var genAoc_i1v0 = new TestApplication.AddonParameter(IdForApplication, 0, 1, 64 * 1024);
                var genAoc_i1v1 = new TestApplication.AddonParameter(IdForApplication, 1, 1, 64 * 1024);
                var genAoc_i2v0 = new TestApplication.AddonParameter(IdForApplication, 0, 2, 64 * 1024);
                genAoc_i2v0.ChangeRequiredVersion(1); // RequiredApplicationReleaseVersion

                // Request list of parameters.
                var request = new List<TestApplication.GenerateParameter<int>>(4) { genApp, genAoc_i1v0, genAoc_i1v1, genAoc_i2v0 };

                var uploadContents = TestApplication.MakeContents(intermediate, request);
                var app = uploadContents[0];
                var patch = uploadContents[1];
                var aoc_i1v0 = uploadContents[2];
                var aoc_i1v1 = uploadContents[3];
                var aoc_i2v0 = uploadContents[4];

                /// --------------------------------------------------------------------------
                /// finally
                /// --------------------------------------------------------------------------
                scope.AddDisposer(() =>
               {
                   executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all");
               });

                /// --------------------------------------------------------------------------
                /// wait-commit からの dynamic-commit の一連検証シーケンス
                ///   - テスト対象アプリケーションをインストール＆起動
                ///   - DTL json ファイルに基づいて DL が行われる ( SuppressAutoKill, wait-commit )
                ///   - dynamic commit 実施 ( SuppressAutoKill, try-commit-current-application-download-task )
                ///   - 結果を判定
                /// --------------------------------------------------------------------------
                System.Action<string, System.Func<bool, string, bool>, System.Func<string, bool>, System.Func<string, bool>, string, GeneratedContentResult>
                    AocDynamicCommitVerifier = (lDtlFilePath, lCheckDynamicCommit, lCheckBefore, lCheckAfter, lExpectAocMountResult, lExpectMountableAoc) =>
                {
                    // 起動中 AoC ダウンロード ( wait-commit )
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystemSuppressAutoKill(
                        $"application push-download-task-list {lDtlFilePath} AND " +
                        $"--time application wait-download {app.Identifier} --wait-commit AND " +
                        $"addoncontent list;application list-view"
                    ));
                    /// コミット前 ApplicationView 期待値検証式
                    ExpectEvaluator.IsTrue(lCheckBefore(executor.OutputStream.Standard.ToString()));

                    // dynamic commit
                    var resultOfDynamicCommit = executor.RunDevMenuCommandSystemSuppressAutoKill(
                        $"application try-commit-current-application-download-task"
                    );
                    /// dynamic commit 要求結果検証式
                    ExpectEvaluator.IsTrue(lCheckDynamicCommit(resultOfDynamicCommit, executor.OutputStream.Standard.ToString()));

                    // after condition verification for add on content.
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystemSuppressAutoKill(
                        $"addoncontent list;application list-view"
                    ));
                    /// コミット後 ApplicationView 期待値検証式
                    ExpectEvaluator.IsTrue(lCheckAfter(executor.OutputStream.Standard.ToString()));

                    // RunOnTarget での後述 DevMenuCommand sleep でのマウントチェック受信待ちと並列通知させるためにタスク化
                    ThrowFrameworks.DelayRun(5000, () =>
                       {
                           var prop = lExpectMountableAoc.Properties;
                           var aocVersion = lExpectMountableAoc.Version;
                           var aocIndex = prop.GetValueAsUI32(ContentMeta.Contexture.Advance.PropertyKey.Index.ToString());
                           /// AocMountRequest.xml 作成
                           var xml = new System.Xml.XmlDocument();
                           xml.AppendChild(xml.CreateXmlDeclaration(@"1.0", @"UTF-8", @"yes"));
                           var eXmlRoot = xml.CreateElement("Root");
                           xml.AppendChild(eXmlRoot);
                           var eXmlAocIndex = xml.CreateElement("AocIndex");
                           var eXmlAocVersion = xml.CreateElement("AocVersion");
                           eXmlRoot.AppendChild(eXmlAocIndex);
                           eXmlRoot.AppendChild(eXmlAocVersion);
                           eXmlAocIndex.InnerText = $"{aocIndex}";
                           eXmlAocVersion.InnerText = $"{aocVersion}";
                           xml.Save(System.IO.Path.Combine(intermediate, "AocMountRequest.xml"));
                       }
                    );

                    /// --suppress-polling-process 使って、起動中アプリからのログ検出を想定。
                    /// ※--suppress-polling-process は指定nsp起動完了後もタイムアウトまでプロセス監視を維持する。
                    /// ※ただ、何故か RunOnTargetPrivate --suppress-auto-kill が効かないので util sleep で代用。
                    uint timeoutSecondsUntilAocMount = 15;
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystemCustomOption($"util sleep {timeoutSecondsUntilAocMount}",
                        new List<string>(2)
                        {
                            "--suppress-auto-kill",
                            $"--pattern-success-exit {lExpectAocMountResult}",
                        },
                        timeoutSecondsUntilAocMount + 1U
                    ));
                };

                /// --------------------------------------------------------------------------
                /// 検証シーケンス ( AoC提供ケース : 新規, 更新, 要求システムバージョン制約 )
                /// --------------------------------------------------------------------------
                System.Action<GeneratedContentResult, string,
                    SigloHelper.ResultMatcher.Application.ListView.ExpectValues,
                    SigloHelper.ResultMatcher.Application.ListView.ExpectValues>
                    MainVerifier = (lInclusionPatch, lDtlFilePath, lExistWaitCommit, lNotExistWaitCommit) =>
                {
                    /// --------------------------------------------------------------------------
                    /// partial or full 差分補正
                    /// partial commit
                    ///     DLタスクが残る。
                    ///     一部コンテンツがコミットされる。記録も更新される。
                    ///     コミット済みのものもタスクが抱える Content Meta ID として残る。
                    /// full commit
                    ///     DLタスクは消える。
                    ///     記録も更新される。
                    ///     従来の View 取得時のコミットと同じ。全コンテンツがコミットされる。
                    ///
                    /// タスクが残っていると、nim 仕様において以下制約により、任意コンテンツの新Versionのdtl受理がされない。
                    ///     "同じmeta ID, 異なるVersion のdtl 追加は無視される"
                    /// --------------------------------------------------------------------------
                    var lDtlGenerator = new DownloadTaskList();
                    var lLatestAppVersion = 0;
                    var lDynamicCommitStyle = "full";
                    var lWaitCommitOnAocUpdated = lExistWaitCommit;

                    if (null != lInclusionPatch)
                    {
                        lDynamicCommitStyle = "partial";
                        lLatestAppVersion = lInclusionPatch.Version;
                        lWaitCommitOnAocUpdated = lNotExistWaitCommit;
                        // patch ダウンロード時にも Aoc の最新バージョンが落ちるため、明示的に "config:0" を指定し Aoc の最新バージョンが落ちない様にしています。
                        lDtlGenerator.Create(IdForApplication, DownloadTaskList.Task.InstallConfig.ForceNone).AddNewTitle(lInclusionPatch);
                    }

                    /// --------------------------------------------------------------------------
                    /// 追加 AoC ダウンロード
                    ///  - AoC コミット成功
                    ///  - AoC マウント可能
                    ///  - Patch インストール待ち
                    /// --------------------------------------------------------------------------
                    lDtlGenerator.Create(IdForApplication, DownloadTaskList.Task.InstallConfig.ForceNone).AddNewTitle(aoc_i1v0);
                    lDtlGenerator.ToJson(lDtlFilePath);
                    AocDynamicCommitVerifier(lDtlFilePath, (lResultDynamicCommit, lStream) =>
                       {
                           var expectMatchPattern = $@"\[CommitManager\] DynamicCommit: Do {lDynamicCommitStyle} commit";
                           var matchFound = Regex.Match(lStream, expectMatchPattern);
                           return true == lResultDynamicCommit && matchFound.Success;
                       },
                        (lStream) =>
                        {
                            var lFoundAoc = SigloHelper.ResultMatcher.AddOnContent.List.Find(lStream, aoc_i1v0);
                            var lMatchCount = SigloHelper.ResultMatcher.Application.ListView.Find(
                                lStream, app.Identifier, app.Version, lExistWaitCommit
                            );
                            return 0 == lMatchCount && false == lFoundAoc;
                        },
                        (lStream) =>
                        {
                            var lFoundAoc = SigloHelper.ResultMatcher.AddOnContent.List.Find(lStream, aoc_i1v0);
                            var lMatchCount = SigloHelper.ResultMatcher.Application.ListView.Find(
                                lStream, app.Identifier, app.Version, lNotExistWaitCommit
                            );
                            return 0 == lMatchCount && lFoundAoc;
                        },
                        @"""\[CheckAocMountable\] Success\.""", aoc_i1v0
                    );

                    /// --------------------------------------------------------------------------
                    // 更新 AoC ダウンロード
                    ///  - コミット失敗
                    ///  - AoC マウント不可
                    ///  - Patch インストール待ち
                    /// --------------------------------------------------------------------------
                    lDtlGenerator.Create(IdForApplication, DownloadTaskList.Task.InstallConfig.ForceNone).AddNewTitle(aoc_i1v1);
                    lDtlGenerator.ToJson(lDtlFilePath);
                    System.Func<string, bool> aocVerifierIndex1v1 = (lStream) =>
                    {
                        var lFoundAoc = SigloHelper.ResultMatcher.AddOnContent.List.Find(lStream, aoc_i1v1);
                        var lMatchCount = SigloHelper.ResultMatcher.Application.ListView.Find(
                            lStream, app.Identifier, app.Version, lWaitCommitOnAocUpdated
                        );
                        return 0 == lMatchCount && false == lFoundAoc;
                    };
                    AocDynamicCommitVerifier(lDtlFilePath, (lResultDynamicCommit, lStream) =>
                       {
                           var expectMatchPattern = @"\[CommitManager\] DynamicCommit:";
                           var matchFound = Regex.Match(lStream, expectMatchPattern);
                           return true == lResultDynamicCommit && false == matchFound.Success;
                       },
                        aocVerifierIndex1v1,
                        aocVerifierIndex1v1,
                        @"""\[CheckAocMountable\] Failure\.""", aoc_i1v1
                    );

                    /// --------------------------------------------------------------------------
                    /// 実行中のアプリよりも RequiredApplicationVersion が高い AoC ダウンロード
                    ///  - コミット失敗
                    ///  - AoC マウント不可
                    ///  - Patch インストール待ち
                    /// --------------------------------------------------------------------------
                    lDtlGenerator.Create(IdForApplication).AddNewTitle(aoc_i2v0);
                    lDtlGenerator.ToJson(lDtlFilePath);
                    System.Func<string, bool> aocVerifierIndex2v0 = (lStream) =>
                    {
                        var lFoundAoc1 = SigloHelper.ResultMatcher.AddOnContent.List.Find(lStream, aoc_i1v1);
                        var lFoundAoc2 = SigloHelper.ResultMatcher.AddOnContent.List.Find(lStream, aoc_i2v0);
                        var lMatchCount = SigloHelper.ResultMatcher.Application.ListView.Find(
                            lStream, app.Identifier, app.Version, lExistWaitCommit
                        );
                        return 0 == lMatchCount && false == lFoundAoc1 && false == lFoundAoc2;
                    };
                    AocDynamicCommitVerifier(lDtlFilePath, (lResultDynamicCommit, lStream) =>
                       {
                           var expectMatchPattern = @"\[CommitManager\] DynamicCommit:";
                           var matchFound = Regex.Match(lStream, expectMatchPattern);
                           return true == lResultDynamicCommit && false == matchFound.Success;
                       },
                        aocVerifierIndex2v0,
                        aocVerifierIndex2v0,
                        @"""\[CheckAocMountable\] Failure\.""", aoc_i2v0
                    );

                    /// --------------------------------------------------------------------------
                    /// アプリ実行終了後であればコミットされる。
                    /// --------------------------------------------------------------------------
                    // partial の場合、タスク残存かつ 新Versionの未受理のため、index1v1 はインストール対象外。
                    // ダウンロード済なので wait-download は不要。list-view から呼び出してコミット確定。
                    var lExpectFinalConditionAoc1 = (null != lInclusionPatch) ? aoc_i1v0 : aoc_i1v1;
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application list-view; addoncontent list"));
                    {
                        var stream = executor.OutputStream.Standard.ToString();
                        var lFoundAoc1 = SigloHelper.ResultMatcher.AddOnContent.List.Find(stream, lExpectFinalConditionAoc1);
                        var lFoundAoc2 = SigloHelper.ResultMatcher.AddOnContent.List.Find(stream, aoc_i2v0);
                        var lMatchCount = SigloHelper.ResultMatcher.Application.ListView.Find(
                            stream, app.Identifier, lLatestAppVersion,
                            new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                            {
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingCommit, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingAocCommit, false },
                                {  SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingPatchInstall, false },
                            }
                        );
                        ExpectEvaluator.IsTrue(lFoundAoc1 && lFoundAoc2 && 0 == lMatchCount);
                    }
                };

                /// --------------------------------------------------------------------------
                /// ケース1 fully ( AoC のみ: 新規, 更新, 要求システムバージョン制約 )
                /// --------------------------------------------------------------------------
                scope.WriteLineWithTimeStamp("Case1: Verification to dynamic commit of package that inclusion the aoc only.");
                {
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application clear-task-status-list AND " +
                        $"application uninstall --all AND " +
                        $"application install \"{app.NspPath}\""
                    ));
                    ExpectEvaluator.IsTrue(executor.RunOnTarget($"0x{app.Identifier}",
                        @"--pattern-failure-exit ""\[FAILURE\]"" --no-wait",
                        $"--testns_MountHost={intermediate} --Command=CheckAocMountable"
                    ));

                    System.Threading.Thread.Sleep(2 * 1000); // アプリ起動 2秒待ち

                    MainVerifier(null, System.IO.Path.Combine(intermediate, "aoc_dynamic_commit.dtl"),
                        new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                        {
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingCommit, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingAocCommit, true },
                        },
                        new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                        {
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingCommit, false },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingAocCommit, false },
                        }
                    );
                }

                /// --------------------------------------------------------------------------
                /// ケース2 partial ( AoC と Patch: 新規, 更新, 要求システムバージョン制約 )
                /// --------------------------------------------------------------------------
                scope.WriteLineWithTimeStamp("Case2: Verification to partial commit of package that inclusion the aoc and patch.");
                {
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application clear-task-status-list AND " +
                        $"application uninstall --all AND " +
                        $"application install \"{app.NspPath}\""
                    ));
                    ExpectEvaluator.IsTrue(executor.RunOnTarget($"0x{app.Identifier}",
                        @"--pattern-failure-exit ""\[FAILURE\]"" --no-wait",
                        $"--testns_MountHost={intermediate} --Command=CheckAocMountable"
                    ));

                    System.Threading.Thread.Sleep(2 * 1000); // アプリ起動 2秒待ち

                    MainVerifier(patch, System.IO.Path.Combine(intermediate, "aoc_partial_commit.dtl"),
                        new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                        {
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingCommit, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingAocCommit, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingPatchInstall, true },
                        },
                        new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                        {
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingCommit, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingAocCommit, false },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingPatchInstall, true },
                        }
                    );
                }

                /// --------------------------------------------------------------------------
                /// ケース3 partial ( Gamecard 起動, AoC, Patch: 新規, 更新, 要求システムバージョン制約 )
                /// --------------------------------------------------------------------------
                scope.WriteLineWithTimeStamp("Case3: [GameCard] Verification to partial commit of package that inclusion the aoc and patch.");
                var hasInsertedGameCard = executor.RunDevMenuCommandSystem($"gamecard status", @"^Inserted$");
                if (hasInsertedGameCard)
                {
                    // finally
                    scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("gamecard erase"); });

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"gamecard erase AND " +
                        $"application clear-task-status-list AND " +
                        $"application uninstall --all AND " +
                        $@"gamecard erase AND gamecard write ""{app.NspPath}"" --verify"
                    ));
                    ExpectEvaluator.IsTrue(executor.RunOnTarget($"0x{app.Identifier}",
                        @"--pattern-failure-exit ""\[FAILURE\]"" --no-wait",
                        $"--testns_MountHost={intermediate} --Command=CheckAocMountable"
                    ));

                    System.Threading.Thread.Sleep(2 * 1000); // アプリ起動 2秒待ち

                    MainVerifier(patch, System.IO.Path.Combine(intermediate, "aoc_partial_commit.withgc.dtl"),
                        new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                        {
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingCommit, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingAocCommit, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingPatchInstall, true },
                        },
                        new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                        {
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingCommit, true },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingAocCommit, false },
                            {  SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingPatchInstall, true },
                        }
                    );
                }
                else
                {
                    Log.WriteLine("Skip Case3 because by not find the gamecard inserted.");
                }
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// 「アプリ・パッチが RequiredSystemVersion を満たさない場合はコミット待ちする」検証。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-65713")]
        public void TestForCommitWaitedWhenTaskBeSystemUpdateRequired()
        {
            using (var scope = new TestMethodLog())
            {
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                var intermediate = GenerateIntermediateDirectoryAsMethod(executor);
                var method = MethodBase.GetCurrentMethod();

                /// --------------------------------------------------------------------------
                // 本テストのメタ構成は以下の通り。
                // Application : 0x01003ab001e30045, version 0
                // Patch       : 0x01003ab001e30845, version 65536
                // AoC index 1 : 0x01003ab001e31046, version 0
                // Application : 0x01003ab001e30046, version 0
                // Patch       : 0x01003ab001e30846, version 65536
                // AoC index 1 : 0x01003ab001e31047, version 0
                // Application : 0x01003ab001e30047, version 0
                // Patch       : 0x01003ab001e30847, version 65536
                // AoC index 1 : 0x01003ab001e31048, version 0
                // Application : 0x01003ab001e30048, version 0
                // Patch       : 0x01003ab001e30848, version 65536
                // AoC index 1 : 0x01003ab001e31049, version 0
                /// --------------------------------------------------------------------------

                /// --------------------------------------------------------------------------
                /// Contents upload assistans.
                /// --------------------------------------------------------------------------
                D4cHelper.NspUploader uploader = new D4cHelper.NspUploader(
                    intermediate,
                    ProxyConfiguration,
                    ServerEnvironment
                );

                /// --------------------------------------------------------------------------
                /// クライアントの SystemUpdateMeta の version 取得
                /// --------------------------------------------------------------------------
                var nupId = new ID64(0x0100000000000816);
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"systemprogram list-details 0x{nupId}"));
                var lMatchedClientSystemUpdate = Regex.Match(executor.OutputStream.Standard.ToString(), $@"\[target\] 0x{nupId}, ver\.[0-9]+");
                var lClientNupVersion = uint.Parse(Regex.Match(lMatchedClientSystemUpdate.Value, "[0-9]+$").Value);
                Log.WriteLine($"Detected client systemupdate meta version : {nupId}-{lClientNupVersion}");
                var lNextNupVersion = lClientNupVersion + 1;

                /// --------------------------------------------------------------------------
                /// コミット成功時の view フラグ期待値。
                /// --------------------------------------------------------------------------
                var expectCommitSuccess = new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                {
                    { SigloHelper.ResultMatcher.Application.ListView.Property.hasRecord, true },
                    { SigloHelper.ResultMatcher.Application.ListView.Property.hasAllEntity, true },
                    { SigloHelper.ResultMatcher.Application.ListView.Property.isDownloading, false },
                    { SigloHelper.ResultMatcher.Application.ListView.Property.isApplyingDelta, false },
                    { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingCommit, false },
                    { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingApplicationCommit, false },
                    { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingPatchInstall, false },
                    { SigloHelper.ResultMatcher.Application.ListView.Property.isSystemUpdateRequiredToCommit, false },
                };

                /// --------------------------------------------------------------------------
                /// Application / Patch 同時コミット待ち状態期待値
                /// DLタスクは ApplicationID 単位で管理され、DTL上のタスクが分散していても1タスクで処理されるため Commit状態は同じになる。
                /// --------------------------------------------------------------------------
                var expectCommitWaiting = new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                {
                    { SigloHelper.ResultMatcher.Application.ListView.Property.hasRecord, true },
                    { SigloHelper.ResultMatcher.Application.ListView.Property.hasAllEntity, false },
                    { SigloHelper.ResultMatcher.Application.ListView.Property.isDownloading, false },
                    { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingCommit, true },
                    { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingApplicationCommit, true },
                    { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingPatchInstall, true },
                    { SigloHelper.ResultMatcher.Application.ListView.Property.isSystemUpdateRequiredToCommit, true },
                };

                /// --------------------------------------------------------------------------
                /// Declare the shared function, for make and upload the test contents.
                /// --------------------------------------------------------------------------
                System.Func<ID64, uint, uint, GeneratedContentResult.Catalog> fCreateAndUploadContents = (lApplicationId, lReqVersionApp, lReqVersionPatch) =>
                {
                    var lGenAoc = new TestApplication.AddonParameter(lApplicationId, 0, 1, 64 * 1024);
                    var lGenApp = new TestApplication.ApplicationParameter(lApplicationId, 0, 64);
                    lGenApp.ChangeRequiredVersion(lReqVersionApp);
                    lGenApp.UseSmallCode = true;
                    lGenApp.AddPatch(1);
                    lGenApp.OnChangePatchParameter = (lOutParam, lSourceParam) =>
                    {
                        lOutParam.RequiredVersion = lReqVersionPatch;
                    };
                    var request = new List<TestApplication.GenerateParameter<int>>(2) { lGenApp, lGenAoc };

#if true    // コンテンツの生成及び、アップロードを行う場合はこちらのコードを有効にする。

                    var lUploadContents = TestApplication.MakeContents(intermediate, request);
                    var app = lUploadContents[0];
                    var patch = lUploadContents[1];
                    var aoc = lUploadContents[2];

                    uploader.RejectRoms(CliTokenAccount, lApplicationId);
                    uploader.Publish(CliTokenAccount, app.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.WithApproved, 5 * 60);
                    uploader.Publish(CliTokenAccount, patch.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.Patch, 5 * 60);
                    uploader.Publish(CliTokenAccount, aoc.NspPath, D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(aoc), 5 * 60);

#else   // 既にアップロード済でアップロード時に生成したnspファイルがOutput/に残されていた場合アップロードスキップで利用可能。

                    var lUploadContents = TestApplication.MakeFromExistsContents( new AuthoringExecutor(), intermediate, request );
                    var app = lUploadContents[ 0 ];
                    var patch = lUploadContents[ 1 ];

#endif

                    Log.WriteLine($"Created test contents [0x{lApplicationId}] information=>" +
                        $"{System.Environment.NewLine}RequiredSystemVersion[ Application ]={app.Properties.GetValueAsUI32("RequiredSystemVersion", uint.MaxValue)}" +
                        $"{System.Environment.NewLine}RequiredSystemVersion[ Patch ]={patch.Properties.GetValueAsUI32("RequiredSystemVersion", uint.MaxValue)}"
                    );
                    return lUploadContents;
                };

                /// --------------------------------------------------------------------------
                /// Declare the shared function, for list-view log matching.
                /// --------------------------------------------------------------------------
                System.Action<string, ID64, int, SigloHelper.ResultMatcher.Application.ListView.ExpectValues>
                fCommonViewCheck = (lCommand, lExpectViewId, lExpectViewVersion, lExpectViewFlags) =>
                {
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(lCommand));
                    ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(
                        executor.OutputStream.Standard.ToString(),
                        lExpectViewId,
                        lExpectViewVersion,
                        lExpectViewFlags
                    ));
                };

                /// --------------------------------------------------------------------------
                /// Declare the shared verification function, for combination check frameworks.
                /// --------------------------------------------------------------------------
                System.Action<ID64.Holder<GeneratedContentResult.Catalog>, bool, bool, bool, string, SigloHelper.ResultMatcher.Application.ListView.ExpectValues>
                fVerifierCore = (lContents, lUseApp, lUsePatch, lWithAoc, lDtlFilePath, lExpectViewFlags) =>
                {
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "systemupdate set-debug-id 0x0000000000000000-0; " +
                        "application clear-task-status-list; " +
                        "application uninstall --all;"
                    ));

                    int lRecordVersion;
                    var app = lContents.Value[0];
                    var patch = lContents.Value[1];
                    var aoc = lContents.Value[2];
                    var lDtlGenerator = new DownloadTaskList();
                    var isWaitingCommit = lExpectViewFlags[SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingCommit];
                    var isSystemUpdateRequired = lExpectViewFlags[SigloHelper.ResultMatcher.Application.ListView.Property.isSystemUpdateRequiredToCommit];
                    if (lUsePatch)
                    {
                        if (lUseApp)
                        {
                            lRecordVersion = (isWaitingCommit) ? app.Version : patch.Version;
                            lDtlGenerator.Create(lContents.Identifier).AddNewTitle(app);
                            lDtlGenerator.Create(lContents.Identifier).AddNewTitle(patch);
                            if (lWithAoc)
                            {
                                lDtlGenerator.Create(lContents.Identifier).AddNewTitle(aoc);
                            }
                        }
                        else
                        {
                            // patch のみの場合は app をローカルインストール
                            lRecordVersion = app.Version;
                            lDtlGenerator.Create(lContents.Identifier).AddNewTitle(patch);
                            if (lWithAoc)
                            {
                                lDtlGenerator.Create(lContents.Identifier).AddNewTitle(aoc);
                            }
                            ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application install \"{app.NspPath}\""));
                        }
                    }
                    else
                    {
                        lRecordVersion = app.Version;
                        lDtlGenerator.Create(lContents.Identifier).AddNewTitle(app);
                        if (lWithAoc)
                        {
                            lDtlGenerator.Create(lContents.Identifier).AddNewTitle(aoc);
                        }
                    }
                    lDtlGenerator.ToJson(lDtlFilePath);

                    // ダウンロードトリガ及び、コミット出来ずに失敗する検証.
                    ExpectEvaluator.IsEqual(false == isWaitingCommit, executor.RunDevMenuCommandSystem(
                        $"application push-download-task-list {lDtlFilePath} AND " +
                        $"application wait-download 0x{lContents.Identifier}"
                    ));
                    // コミット待ち状態の確認及び、list-view での isSystemUpdateRequiredToCommit 発生検証.
                    fCommonViewCheck((isWaitingCommit)
                        ? $"application wait-download 0x{lContents.Identifier} --wait-commit AND application list-view"
                        : $"application list-view",
                        lContents.Identifier,
                        lRecordVersion,
                        lExpectViewFlags
                    );

                    if (isSystemUpdateRequired)
                    {
                        // 正常終了による再起動.
                        ExpectEvaluator.IsTrue(RebootTargetByPowerButton(executor));

                        // 再起動後の isSystemUpdateRequiredToCommit 永続化確認。
                        fCommonViewCheck(
                            "application list --detail AND application list-view",
                            lContents.Identifier,
                            lRecordVersion,
                            lExpectViewFlags
                        );

                        // Resume task 後の isSystemUpdateRequiredToCommit 継続確認。
                        fCommonViewCheck(
                            $"application resume-download 0x{lContents.Identifier} AND " +
                            $"application wait-download 0x{lContents.Identifier} --wait-commit AND " +
                            $"application list-view",
                            lContents.Identifier,
                            lRecordVersion,
                            lExpectViewFlags
                        );

                        // 本体更新後のコミット成功確認。
                        fCommonViewCheck(
                            $"systemupdate set-debug-id 0x{nupId}-{lNextNupVersion} AND " +
                            $"application resume-download 0x{lContents.Identifier} AND " +
                            $"application wait-download 0x{lContents.Identifier} AND " +
                            $"application list-view",
                            lContents.Identifier,
                            (lUsePatch) ? patch.Version : app.Version,
                            expectCommitSuccess
                        );
                    }
                };

                /// --------------------------------------------------------------------------
                /// Declare the shared verification function, for combination check frameworks with aoc.
                /// --------------------------------------------------------------------------
                System.Action<ID64.Holder<GeneratedContentResult.Catalog>, bool, bool, string, SigloHelper.ResultMatcher.Application.ListView.ExpectValues>
                fVerifier = (lContents, lUseApp, lUsePatch, lDtlFilePath, lExpectViewFlags) =>
                {
                    // Aoc を DTL に含めないケース
                    fVerifierCore(lContents, lUseApp, lUsePatch, false, lDtlFilePath, lExpectViewFlags);
                    // Aoc を DTL に含めるケース
                    fVerifierCore(lContents, lUseApp, lUsePatch, true, $"{lDtlFilePath}.aoc", lExpectViewFlags);
                };

                /// --------------------------------------------------------------------------
                /// finally, disoser commands for target cleanup at test finished.
                /// --------------------------------------------------------------------------
                scope.AddDisposer(() =>
               {
                   // devmenu での view の定期的取得、自動コミット機能、NUP通知受信ハンドラを復帰する.
                   executor.RunDevMenuCommandSystem(
                      "debug set-boolean-fwdbg --name devmenu --key enable_application_update true;" +
                      "debug set-boolean-fwdbg --name contents_delivery --key disable_auto_commit_forcibly false;" +
                      "debug set-boolean-fwdbg --name \"ns.notification\" --key \"enable_network_update\" true;" +
                      "systemupdate set-debug-id 0x0000000000000000-0"
                  );
                   executor.ResetTarget();
               });
                scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all"); });

                /// ==========================================================================
                /// メインシーケンスエントリポイント.
                /// ==========================================================================
                // devmenu での view の定期的取得、自動コミット機能、NUP通知受信ハンドラを無効にする.
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "debug set-boolean-fwdbg --name devmenu --key enable_application_update false;" +
                    "debug set-boolean-fwdbg --name contents_delivery --key disable_auto_commit_forcibly true;" +
                    "debug set-boolean-fwdbg --name \"ns.notification\" --key \"enable_network_update\" false;"
                ));
                ExpectEvaluator.IsTrue(executor.ResetTarget());

                // Application / Patch の RequiredSystemVersion 組み合わせ構成ごとにID割り振り.
                var dataOkOk = new ID64.Holder<GeneratedContentResult.Catalog>(new ID64(0x01003ab001e30045), null);
                var dataNgOk = new ID64.Holder<GeneratedContentResult.Catalog>(new ID64(0x01003ab001e30046), null);
                var dataOkNg = new ID64.Holder<GeneratedContentResult.Catalog>(new ID64(0x01003ab001e30047), null);
                var dataNgNg = new ID64.Holder<GeneratedContentResult.Catalog>(new ID64(0x01003ab001e30048), null);

                // --------------------------------------------------------------------------
                // アプリOK, パッチOK ( 境界値判定==RequiredSystemVersion が一致 )
                Log.WriteLineAsIs(string.Empty);
                scope.WriteLineWithTimeStamp($"{System.Environment.NewLine}=> Case: Application[ OK ], Patch[ OK ], with equal on the border.");
                dataOkOk.Value = fCreateAndUploadContents(dataOkOk.Identifier, lClientNupVersion, lClientNupVersion);
                var taskNgOk = ThrowFrameworks.DelayRun(0, () =>
               {
                   // 実機テスト実行中に次コンテンツの生成・アップロードを並列実施
                   dataNgOk.Value = fCreateAndUploadContents(dataNgOk.Identifier, lNextNupVersion, lClientNupVersion);
               });
                fVerifier(dataOkOk, true, true, System.IO.Path.Combine(intermediate, "dtl.ok.ok"),
                    expectCommitSuccess
                );
                taskNgOk.Wait();

                // --------------------------------------------------------------------------
                // アプリ単品NG
                Log.WriteLineAsIs(string.Empty);
                scope.WriteLineWithTimeStamp($"{System.Environment.NewLine}=> Case: Application[ NG ], Patch[ NONE ].");
                var taskOkNg = ThrowFrameworks.DelayRun(0, () =>
               {
                   // 実機テスト実行中に次コンテンツの生成・アップロードを並列実施
                   dataOkNg.Value = fCreateAndUploadContents(dataOkNg.Identifier, lClientNupVersion, lNextNupVersion);
               });
                fVerifier(dataNgOk, true, false, System.IO.Path.Combine(intermediate, "dtl.ng.none"),
                    new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                    {
                        { SigloHelper.ResultMatcher.Application.ListView.Property.hasRecord, true },
                        { SigloHelper.ResultMatcher.Application.ListView.Property.hasAllEntity, false },
                        { SigloHelper.ResultMatcher.Application.ListView.Property.isDownloading, false },
                        { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingCommit, true },
                        { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingApplicationCommit, true },
                        { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingPatchInstall, false },
                        { SigloHelper.ResultMatcher.Application.ListView.Property.isSystemUpdateRequiredToCommit, true },
                    }
                );
                taskOkNg.Wait();

                // --------------------------------------------------------------------------
                // パッチ単品NG
                Log.WriteLineAsIs(string.Empty);
                scope.WriteLineWithTimeStamp($"{System.Environment.NewLine}=> Case: Application[ NONE ], Patch[ NG ].");
                var taskNgNg = ThrowFrameworks.DelayRun(0, () =>
               {
                   // 実機テスト実行中に次コンテンツの生成・アップロードを並列実施
                   dataNgNg.Value = fCreateAndUploadContents(dataNgNg.Identifier, lNextNupVersion, lNextNupVersion);
               });
                fVerifier(dataOkNg, false, true, System.IO.Path.Combine(intermediate, "dtl.none.ng"),
                    new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                    {
                        { SigloHelper.ResultMatcher.Application.ListView.Property.hasRecord, true },
                        { SigloHelper.ResultMatcher.Application.ListView.Property.hasAllEntity, true },
                        { SigloHelper.ResultMatcher.Application.ListView.Property.isDownloading, false },
                        { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingCommit, true },
                        { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingApplicationCommit, false },
                        { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingPatchInstall, true },
                        { SigloHelper.ResultMatcher.Application.ListView.Property.isSystemUpdateRequiredToCommit, true },
                    }
                );
                taskNgNg.Wait();

                // --------------------------------------------------------------------------
                // アプリNG, パッチNG
                Log.WriteLineAsIs(string.Empty);
                scope.WriteLineWithTimeStamp($"{System.Environment.NewLine}=> Case: Application[ NG ], Patch[ NG ].");
                fVerifier(dataNgNg, true, true, System.IO.Path.Combine(intermediate, "dtl.ng.ng"),
                    expectCommitWaiting
                );

                // --------------------------------------------------------------------------
                // アプリOK, パッチNG
                Log.WriteLineAsIs(string.Empty);
                scope.WriteLineWithTimeStamp($"{System.Environment.NewLine}=> Case: Application[ OK ], Patch[ NG ].");
                fVerifier(dataOkNg, true, true, System.IO.Path.Combine(intermediate, "dtl.ok.ng"),
                    expectCommitWaiting
                );

                // --------------------------------------------------------------------------
                // アプリNG, パッチOK
                Log.WriteLineAsIs(string.Empty);
                scope.WriteLineWithTimeStamp($"{System.Environment.NewLine}=> Case: Application[ NG ], Patch[ OK ].");
                fVerifier(dataNgOk, true, true, System.IO.Path.Combine(intermediate, "dtl.ng.ok"),
                    expectCommitWaiting
                );

                /// ==========================================================================
                /// SystemUpdateRequired の状態でも、Dynamic Commit からの Partial commit が動作するか。
                /// マウント成否の詳細検証は TestForCommitAocInApplicationRunning で行う前提。
                /// app(ng),patch(ng),aoc 構成DTLで以下の二種を検証。
                /// - NGを含んだDTLでコミット待ち中の Aoc partial commit.
                /// - 事前に SystemUpdateRequired にしてからの Aoc partial commit.
                /// ==========================================================================

                /// --------------------------------------------------------------------------
                /// Declare the shared verification function, for dynamic commit frameworks.
                ///
                /// dtl が app(ng), patch(ng), aoc 構成を想定した固定判定です。
                /// DTL構成に依存した可変判定にする場合は以下を留意ください。
                /// DTLに App を含まないなら true == lDownloadAtLaunchBefore 時は、list-view version が patch の version になります。
                /// これは、DTL中の patch が初レコードになるため。
                /// また逆に false == lDownloadAtLaunchBefore 時は、GameCard app v0 が初レコードなので app の version になります。
                /// --------------------------------------------------------------------------
                System.Action<bool, GeneratedContentResult, GeneratedContentResult, string>
                fDynamicCommitLiteVerifier = (lDownloadAtLaunchBefore, lApplication, lAddOnContent, lDtlFilePath) =>
                {
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application clear-task-status-list; application uninstall --all;gamecard erase"
                    ));
                    if (lDownloadAtLaunchBefore)
                    {
                        // 起動前ダウンロード( SystemUpdateRequired 状態確認 )
                        fCommonViewCheck(
                            $"application push-download-task-list {lDtlFilePath} AND " +
                            $"application wait-download {lApplication.Identifier} --wait-commit AND " +
                            $"application list-view",
                            lApplication.Identifier,
                            lApplication.Version,
                            new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                            {
                                { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingCommit, true },
                                { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingAocCommit, true },
                                { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingPatchInstall, true },
                                { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingApplicationCommit, true },
                                { SigloHelper.ResultMatcher.Application.ListView.Property.isSystemUpdateRequiredToCommit, true },
                            }
                        );
                    }
                    // GameCard 経由で起動.
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $@"gamecard write ""{lApplication.NspPath}"" --verify"
                    ));
                    ExpectEvaluator.IsTrue(executor.RunOnTarget(
                        $"0x{lApplication.Identifier}",
                        @"--pattern-failure-exit ""\[FAILURE\]"" --no-wait"
                    ));
                    System.Threading.Thread.Sleep(5 * 1000); // アプリ起動待ち

                    if (false == lDownloadAtLaunchBefore)
                    {
                        // 起動中ダウンロード
                        ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystemSuppressAutoKill(
                            $"application push-download-task-list {lDtlFilePath} AND " +
                            $"application wait-download {lApplication.Identifier} --wait-commit AND " +
                            $"application list-view"
                        ));
                        ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ListView.Find(
                            executor.OutputStream.Standard.ToString(),
                            lApplication.Identifier,
                            lApplication.Version,
                            new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                            {
                                { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingCommit, true },
                                { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingAocCommit, true },
                                { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingPatchInstall, true },
                                { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingApplicationCommit, true },
                                { SigloHelper.ResultMatcher.Application.ListView.Property.isSystemUpdateRequiredToCommit, true },
                            }
                        ));
                    }

                    // dynamic commit
                    var lResultOfDynamicCommit = executor.RunDevMenuCommandSystemSuppressAutoKill(
                        $"application try-commit-current-application-download-task"
                    );
                    // dynamic commit 要求結果検証式
                    var lEexpectMatchPattern = $@"\[CommitManager\] DynamicCommit: Do partial commit";
                    var lMatchFound = Regex.Match(executor.OutputStream.Standard.ToString(), lEexpectMatchPattern);
                    ExpectEvaluator.IsTrue(true == lResultOfDynamicCommit && lMatchFound.Success);

                    // after condition verification for add on content.
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystemSuppressAutoKill(
                        $"addoncontent list;application list-view"
                    ));
                    // コミット後 ApplicationView 期待値検証式
                    var lAfterStream = executor.OutputStream.Standard.ToString();
                    var lFoundAoc = SigloHelper.ResultMatcher.AddOnContent.List.Find(lAfterStream, lAddOnContent);
                    var lMatchCount = SigloHelper.ResultMatcher.Application.ListView.Find(
                        lAfterStream, lApplication.Identifier,
                        lApplication.Version,
                        new SigloHelper.ResultMatcher.Application.ListView.ExpectValues()
                        {
                            { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingCommit, true },
                            { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingAocCommit, false },
                            { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingPatchInstall, true },
                            { SigloHelper.ResultMatcher.Application.ListView.Property.isWaitingApplicationCommit, true },
                            { SigloHelper.ResultMatcher.Application.ListView.Property.isSystemUpdateRequiredToCommit, true },
                        }
                    );
                    ExpectEvaluator.IsTrue(0 == lMatchCount && lFoundAoc);
                };

                /// ==========================================================================
                /// Dynamic commit 検証エントリポイント.
                /// ==========================================================================
                if (false == executor.RunDevMenuCommandSystem($"systemupdate set-debug-id 0x0000000000000000-0;gamecard status", @"^Inserted$"))
                {
                    Log.WriteLine("Skip the case of `Dynamic commit verification` because by not find the gamecard inserted.");
                }
                else
                {
                    scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("gamecard erase"); });
                    var contents = dataNgNg;

                    // ローカルインストール想定で app v0 は large-code で作り直し.
                    var localApp = new TestApplication.ApplicationParameter(contents.Identifier, 0);
                    localApp.ChangeRequiredVersion(lClientNupVersion);
                    var app = TestApplication.MakeContents(intermediate, localApp)[0];
                    var patch = contents.Value[1];
                    var aoc = contents.Value[2];

                    // application(ng) + patch(ng) + addoncontent
                    var dtl = System.IO.Path.Combine(intermediate, "dtl.dynamic.commit");
                    var dtlGenerator = new DownloadTaskList();
                    dtlGenerator.Create(contents.Identifier).AddNewTitle(app);
                    dtlGenerator.Create(contents.Identifier).AddNewTitle(patch);
                    dtlGenerator.Create(contents.Identifier).AddNewTitle(aoc);
                    dtlGenerator.ToJson(dtl);

                    // --------------------------------------------------------------------------
                    // NGを含んだDTLでコミット待ち中の Aoc partial commit.
                    Log.WriteLineAsIs(string.Empty);
                    scope.WriteLineWithTimeStamp($"{System.Environment.NewLine}=> Case: Dynamic commit verification, download at launch after.");
                    fDynamicCommitLiteVerifier(false, app, aoc, dtl);

                    // --------------------------------------------------------------------------
                    // 事前に SystemUpdateRequired にしてからの Aoc partial commit.
                    Log.WriteLineAsIs(string.Empty);
                    scope.WriteLineWithTimeStamp($"{System.Environment.NewLine}=> Case: Dynamic commit verification, download at launch before.");
                    fDynamicCommitLiteVerifier(true, app, aoc, dtl);
                }
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// 未来の鍵世代のデータをダウンロードして、システムが fatal しないかを確認
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-67948")]
        public void TestFutureKeyGeneration()
        {
            using (var scope = new TestMethodLog())
            {
                // あらかじめ、0x0100f6900049a010 を鍵世代 128 としてアップロードしてある
                var id = new ID64(0x0100f6900049a010);
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

                scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("application uninstall --all"); });

                // コントロールの取得は、FG で待つので即時失敗
                ExpectEvaluator.IsFalse(executor.RunDevMenuCommandSystem(
                            $"application download-control {id}"
                ));

                // ダウンロードタスクの登録は成功する
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                            $"application create-download-task {id}"
                ));

                // タスクを待つとそれは失敗になる
                ExpectEvaluator.IsFalse(executor.RunDevMenuCommandSystem(
                            $"application wait-download {id}"
                ));

                // システムが fatal していないかの確認のため、列挙だけする
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                            $"application list"
                ));
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// nn::aocライブラリ経由での OnCardAoc アクセス機能の検証
        /// 単体実行時で 10 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-66487")]
        public void TestForOnCardAocAccessLibrary()
        {
            var UploadTimeout = new System.TimeSpan(0, 5, 0);         // 失敗時のリトライも含めて長めに設定
            var DownloadTimeout = new System.TimeSpan(0, 3, 0);       // 通知～完了までの実績値が 1 分以内
            var VersionListInterval = new System.TimeSpan(0, 1, 0);   // supefly -> tagaya の取得間隔分を空ける

            using (var scope = new TestMethodLog())
            {
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                var intermediate = GenerateIntermediateDirectoryAsMethod(executor);
                string aoc_i1_v0a_hash; // Card に格納する Aoc の hash 値
                string aoc_i1_v0b_hash; // sd に格納する Aoc の hash 値

                const int LOCALINSTAL = 0;
                const int DOWNLOAD = 1;

                // make contents
                //  Application : 0x01001a500005e0f5, version 0
                //  AddOnContent: 0x01001a500005f0f6, version 0,      index 1
                //              :        〃         , version 65536,  index 1
                //              : 0x01001a500005f0f7, version 0,      index 2
                var owner = new ID64(0x01001a500005e0f5);
                var @params = new List<TestApplication.GenerateParameter<int>>();
                {
                    // app
                    var param = new TestApplication.ApplicationParameter(owner, 0, 5 * 1024 * 1024);
                    param.ChangeCodePath(System.IO.Path.Combine(executor.GetOutputTestDirectory("ShopRuntimeAssistants"), "code"));
                    param.UseSmallCode = true;
                    @params.Add(param);
                }
                {
                    // aoc
                    var param0 = new TestApplication.AddonParameter(owner, 0, 1, 1 * 1024 * 1024);
                    @params.Add(param0);
                    var param1 = new TestApplication.AddonParameter(owner, 1, 1, 1 * 1024 * 1024);
                    @params.Add(param1);
                    var param3 = new TestApplication.AddonParameter(owner, 0, 2, 1 * 1024 * 1024);
                    @params.Add(param3);
                }
                var contents = TestApplication.MakeContents(intermediate, @params);
                var catalog = new GeneratedContentResult.TypeCategorizedCatalog(contents);
                var app = catalog.GetTypedCatalog(ContentMeta.Type.Application)[0];
                var aoc_i1_v0a = catalog.GetTypedCatalog(ContentMeta.Type.AddOnContent)[0];
                var aoc_i1_v1 = catalog.GetTypedCatalog(ContentMeta.Type.AddOnContent)[1];
                var aoc_i2_v0 = catalog.GetTypedCatalog(ContentMeta.Type.AddOnContent)[2];

                // app だけは、本テスト内でアップロードして上書きするため、ここでも行う。
                // upload contents
                var uploader = new D4cHelper.NspUploader(
                    intermediate,
                    ProxyConfiguration,
                    ServerEnvironment);

                var account = CliTokenAccount;
                var timeout = UploadTimeout.TotalSeconds;
                uploader.RejectRoms(account, app.Identifier);
                uploader.Publish(account, app.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.WithApproved, timeout);

                // finally
                if (!IsUnitTestOnIDE)
                {
                    scope.AddDisposer(() =>
                    {
                        CleanupContents(executor, true);
                    });
                }

                // ShopRuntimeAssistants の CheckAocContents 機能のログをチェックする。
                // Aoc の数、appに対するAoc index の数、マウントした Aoc の version のチェック
                System.Func<int, int, int, bool> checkAocContents = (totalIndexCount, indexCount, mountVersion) =>
                {
                    var expected = $"CountAddOnContent: {totalIndexCount}";
                    ExpectEvaluator.IsTrue(MatchTargetLog(executor, expected));
                    expected = $"ListAddOnContent:totalIndex= {indexCount}";
                    ExpectEvaluator.IsTrue(MatchTargetLog(executor, expected));
                    expected = string.Format(@"installVersion= \[ 0x{0:x} \]", mountVersion);
                    ExpectEvaluator.IsTrue(MatchTargetLog(executor, expected));
                    return true;
                };

                // ShopRuntimeAssistants の CheckAocHash 機能から hash 値を取得する。
                System.Func<SigloHelper.CommodityExecutor.Context, string> getAocHash = (localExecutor) =>
                {
                    Regex regHash = new Regex(@"AddOnContent mount check for `hash \[ (.*?) \]`");
                    System.Text.RegularExpressions.Match mc = regHash.Match(localExecutor.OutputStream.Standard.ToString());
                    return mc.Groups[1].Value;
                };

                CleanupContents(executor, true);

                // 検証条件：OnCardAoc の場合
                scope.WriteLineWithTimeStamp("Check OnCardAoc");
                {
                    // OnCard Aoc ( index 1, version 0 ) をインストール
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"gamecard write {app.NspPath} --on-card-aoc {aoc_i1_v0a.NspPath}"
                    ));
                    // aoc のハッシュ値を出力
                    ExpectEvaluator.IsTrue(executor.RunOnTarget($"0x{app.Identifier}",
                        @"--pattern-failure-exit ""\[FAILURE\]"" ",
                        $"--Command=CheckAocHash%CheckIndexNo:1"
                    ));
                    // ログから aoc_i1_v0a のハッシュ値を取得する。( card にしか aoc がないので、card のハッシュ値 )
                    aoc_i1_v0a_hash = getAocHash(executor);
                    Log.WriteLine(string.Format(" aoc_i1_v0a hash: {0}", aoc_i1_v0a_hash));

                    // Aocの中身を確認。(aoc::CountAddOnContent / aoc::ListAddOnContent / マウントの実施 )
                    // index 1 の内容を確認。
                    ExpectEvaluator.IsTrue(executor.RunOnTarget($"0x{app.Identifier}",
                        @"--pattern-failure-exit ""\[FAILURE\]"" ",
                        $"--Command=CheckAocContents%CheckIndexNo:1"
                    ));

                    // Aoc( 1[card]: index 1, version 0 )
                    int totalIndexCount = 1; // Aoc の数
                    int indexCount = 1;      // appに対する Aoc の数
                    int mountVersion = aoc_i1_v0a.Version; // マウント先の Aoc version
                    checkAocContents(totalIndexCount, indexCount, mountVersion);

                    for (int route = 0; route < 2; ++route)
                    {
                        if (route == LOCALINSTAL)
                        {
                            //Aoc ( index 1, version 1 ) をインストール
                            ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application install {aoc_i1_v1.NspPath}"));
                        }
                        else if (route == DOWNLOAD)
                        {
                            // sd / nand のデータを削除
                            ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                                $"application delete-entity {app.Identifier} --aoc -s builtin AND " +
                                $"application delete-entity {app.Identifier} --aoc -s sdcard "
                            ));

                            ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                                $"application create-download-task 0x{app.Identifier} --type AddOnContent --id 0x{aoc_i1_v1.Identifier} --version {aoc_i1_v1.Version} AND " +
                                $"application wait-download 0x{app.Identifier}  --timeout {DownloadTimeout.TotalMilliseconds}"
                            ));
                        }
                        else
                        {
                            // 通らない
                            throw new UnexpectFailureException("A pattern that does not exist.");
                        }

                        ExpectEvaluator.IsTrue(executor.RunOnTarget($"0x{app.Identifier}",
                            @"--pattern-failure-exit ""\[FAILURE\]"" ",
                            $"--Command=CheckAocContents%CheckIndexNo:1"
                        ));

                        // Aoc( 1[card]: index 1, version 0    2[sd/nand]: index 1, version 1 )
                        // 同じindex がある場合、 version が高い aoc にマウントされる
                        totalIndexCount = 1;
                        indexCount = 1;
                        mountVersion = aoc_i1_v1.Version;
                        checkAocContents(totalIndexCount, indexCount, mountVersion);

                        if (route == LOCALINSTAL)
                        {
                            //Aoc ( index 2, version 0 ) をインストール
                            ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application install {aoc_i2_v0.NspPath}"));
                        }
                        else if (route == DOWNLOAD)
                        {
                            ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                                $"application create-download-task 0x{app.Identifier} --type AddOnContent --id 0x{aoc_i2_v0.Identifier} --version {aoc_i2_v0.Version} AND " +
                                $"application wait-download 0x{app.Identifier} --timeout {DownloadTimeout.TotalMilliseconds}"
                            ));
                        }
                        else
                        {
                            // 通らない
                            throw new UnexpectFailureException("A pattern that does not exist.");
                        }

                        // index 1 をマウント
                        ExpectEvaluator.IsTrue(executor.RunOnTarget($"0x{app.Identifier}",
                            @"--pattern-failure-exit ""\[FAILURE\]"" ",
                            $"--Command=CheckAocContents%CheckIndexNo:1"
                        ));

                        // Aoc( 1[card]: index 1, version 0    2[sd]: index 1, version 1    3[sd]: index 2, version 0 )
                        // index 1 にマウント
                        totalIndexCount = 2;
                        indexCount = 2;
                        mountVersion = aoc_i1_v1.Version;
                        checkAocContents(totalIndexCount, indexCount, mountVersion);

                        // index 2 をマウント
                        ExpectEvaluator.IsTrue(executor.RunOnTarget($"0x{app.Identifier}",
                            @"--pattern-failure-exit ""\[FAILURE\]"" ",
                            $"--Command=CheckAocContents%CheckIndexNo:2"
                        ));

                        // Aoc( 1[card]: index 1, version 0    2[sd]: index 1, version 1    3[sd]: index 2, version 0 )
                        // index 2 にマウント
                        totalIndexCount = 2;
                        indexCount = 2;
                        mountVersion = aoc_i1_v0a.Version;
                        checkAocContents(totalIndexCount, indexCount, mountVersion);
                    }
                }

                CleanupContents(executor, true);

                // 検証条件： OnStorage( sdcard/builtin )
                scope.WriteLineWithTimeStamp("OnStorage( sdcard/builtin ) instal");
                {
                    //Aoc ( index 1, version 0 ) をインストール
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application install {app.NspPath}"));
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application install {aoc_i1_v0a.NspPath}"));
                    ExpectEvaluator.IsTrue(executor.RunOnTarget($"0x{app.Identifier}",
                        @"--pattern-failure-exit ""\[FAILURE\]"" ",
                        $"--Command=CheckAocContents%CheckIndexNo:1"
                    ));

                    int totalIndexCount = 1;
                    int indexCount = 1;
                    int mountVersion = aoc_i1_v0a.Version;
                    checkAocContents(totalIndexCount, indexCount, mountVersion);

                    //Aoc ( index 1, version 1 ) をインストール
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application install {aoc_i1_v1.NspPath}"));
                    ExpectEvaluator.IsTrue(executor.RunOnTarget($"0x{app.Identifier}",
                        @"--pattern-failure-exit ""\[FAILURE\]"" ",
                        $"--Command=CheckAocContents%CheckIndexNo:1"
                    ));

                    // Aoc( 1: index 1, version 0    2: index 1, version 1 )
                    totalIndexCount = 1;
                    indexCount = 1;
                    mountVersion = aoc_i1_v1.Version;
                    checkAocContents(totalIndexCount, indexCount, mountVersion);
                }

                CleanupContents(executor, true);

                // 検証条件： OnStorage( sdcard/builtin ) ダウンロード
                scope.WriteLineWithTimeStamp("OnStorage( sdcard/builtin ) download");
                {
                    // アップロードコンテンツの最新バージョンを superfly に設定する。
                    uploader.RegisterVersion(aoc_i1_v0a.NspPath);

                    //Aoc ( index 1, version 0 ) をダウンロード
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application create-download-task 0x{app.Identifier} AND " +
                        $"application create-download-task 0x{app.Identifier} --type AddOnContent --id 0x{aoc_i1_v0a.Identifier} --version {aoc_i1_v0a.Version} AND " +
                        $"application wait-download 0x{app.Identifier}  --timeout {DownloadTimeout.TotalMilliseconds}"
                    ));

                    ExpectEvaluator.IsTrue(executor.RunOnTarget($"0x{app.Identifier}",
                        @"--pattern-failure-exit ""\[FAILURE\]"" ",
                        $"--Command=CheckAocContents%CheckIndexNo:1"
                    ));

                    int totalIndexCount = 1;
                    int indexCount = 1;
                    int mountVersion = aoc_i1_v0a.Version;
                    checkAocContents(totalIndexCount, indexCount, mountVersion);

                    // アップロードコンテンツの最新バージョンを superfly に設定する。
                    uploader.RegisterVersion(aoc_i1_v1.NspPath);

                    //Aoc ( index 1, version 1 ) をダウンロード
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application create-download-task 0x{app.Identifier} --type AddOnContent --id 0x{aoc_i1_v1.Identifier} --version {aoc_i1_v1.Version} AND " +
                        $"application wait-download 0x{app.Identifier}  --timeout {DownloadTimeout.TotalMilliseconds}"
                    ));

                    ExpectEvaluator.IsTrue(executor.RunOnTarget($"0x{app.Identifier}",
                        @"--pattern-failure-exit ""\[FAILURE\]"" ",
                        $"--Command=CheckAocContents%CheckIndexNo:1"
                    ));

                    // Aoc( 1: index 1, version 0    2: index 1, version 1 )
                    totalIndexCount = 1;
                    indexCount = 1;
                    mountVersion = aoc_i1_v1.Version;
                    checkAocContents(totalIndexCount, indexCount, mountVersion);
                }

                CleanupContents(executor, true);

                // 検証条件： 重複 ( ローカルインストール検証のみ )
                scope.WriteLineWithTimeStamp("Aoc Overlap");
                {
                    // card Aoc ( index 0, version 0)
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"gamecard write {app.NspPath} --on-card-aoc {aoc_i1_v0a.NspPath}"
                    ));

                    // sd Aoc ( index 0, version 0)
                    // 同index 同version は1つしか作成できない為、１つずつ MakeContens を実行する。
                    var @params2 = new List<TestApplication.GenerateParameter<int>>();
                    {
                        // app
                        var param = new TestApplication.ApplicationParameter(owner, 0, 5 * 1024 * 1024);
                        param.ChangeCodePath(System.IO.Path.Combine(executor.GetOutputTestDirectory(
                            "ShopRuntimeAssistants"), "code"
                        ));
                        param.UseSmallCode = true;
                        @params2.Add(param);
                    }
                    {
                        // aoc
                        var param4 = new TestApplication.AddonParameter(owner, 0, 1, 1 * 1024 * 512);
                        @params2.Add(param4);
                    }

                    // Request list of parameters.
                    var contents2 = TestApplication.MakeContents(intermediate, @params2);
                    var catalog2 = new GeneratedContentResult.TypeCategorizedCatalog(contents2);
                    var aoc_i1_v0b = catalog2.GetTypedCatalog(ContentMeta.Type.AddOnContent)[0];

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application install { aoc_i1_v0b.NspPath } -s sdcard --force"
                    ));

                    // nand Aoc ( index 0, version 0)
                    // 同index 同version は1つしか作成できない為、１つずつ MakeContens を実行する。
                    var @params3 = new List<TestApplication.GenerateParameter<int>>();
                    {
                        // app
                        var param = new TestApplication.ApplicationParameter(owner, 0, 5 * 1024 * 1024);
                        param.ChangeCodePath(System.IO.Path.Combine(executor.GetOutputTestDirectory(
                            "ShopRuntimeAssistants"), "code"
                        ));
                        param.UseSmallCode = true;
                        @params3.Add(param);
                    }
                    {
                        // aoc
                        var param5 = new TestApplication.AddonParameter(owner, 0, 1, 1 * 1024 * 256);
                        @params3.Add(param5);
                    }

                    // Request list of parameters.
                    var contents3 = TestApplication.MakeContents(intermediate, @params3);
                    var catalog3 = new GeneratedContentResult.TypeCategorizedCatalog(contents3);
                    var aoc_i1_v0c = catalog3.GetTypedCatalog(ContentMeta.Type.AddOnContent)[0];

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application install { aoc_i1_v0c.NspPath } -s builtin --force"
                    ));

                    // Aoc の状態を確認
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommand("addoncontent list"));

                    // Aoc のハッシュ値をログ出力
                    ExpectEvaluator.IsTrue(executor.RunOnTarget($"0x{app.Identifier}",
                        @"--pattern-failure-exit ""\[FAILURE\]"" ",
                        $"--Command=CheckAocHash%CheckIndexNo:1"
                    ));

                    {
                        // ハッシュ値のチェック ( card と一致すること )
                        var expected = @"AddOnContent mount check for `hash \[ " + aoc_i1_v0a_hash + @" \]";
                        ExpectEvaluator.IsTrue(MatchTargetLog(executor, expected));
                    }

                    ExpectEvaluator.IsTrue(executor.RunOnTarget($"0x{app.Identifier}",
                        @"--pattern-failure-exit ""\[FAILURE\]"" ",
                        $"--Command=CheckAocContents%CheckIndexNo:1"
                    ));

                    // Aoc( 1[card]: index 1 , version 0    2[sd]: index 1, version 0   3[nand]: index 1, version 0 )
                    int totalIndexCount = 1;
                    int indexCount = 1;
                    int mountVersion = aoc_i1_v0a.Version;
                    checkAocContents(totalIndexCount, indexCount, mountVersion);

                    // gameCard を削除
                    var isInsertedCard = executor.RunDevMenuCommandSystem($"gamecard status", @"^Inserted$");
                    if (isInsertedCard)
                    {
                        ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("gamecard erase"));
                    }

                    // Appも消えているため、Appをインストール。( Appが無いとマウントできない )
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"gamecard write {app.NspPath}"));

                    // Aoc の状態を確認 ( sd, nand )
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommand("addoncontent list"));

                    // Aoc のハッシュ値を取得
                    ExpectEvaluator.IsTrue(executor.RunOnTarget($"0x{app.Identifier}",
                        @"--pattern-failure-exit ""\[FAILURE\]"" ",
                        $"--Command=CheckAocHash%CheckIndexNo:1"
                    ));

                    // ログから aoc のハッシュ値を取得する。( sd のハッシュ値が取得される想定 )
                    aoc_i1_v0b_hash = getAocHash(executor);
                    Log.WriteLine(string.Format(" aoc_i1_v0b hash: {0}", aoc_i1_v0b_hash));

                    ExpectEvaluator.IsTrue(executor.RunOnTarget($"0x{app.Identifier}",
                        @"--pattern-failure-exit ""\[FAILURE\]"" ",
                        $"--Command=CheckAocContents%CheckIndexNo:1"
                    ));

                    // Aoc( 1[sd]: index 1, version 0   2[nand]: index 1, version 0 )
                    totalIndexCount = 1;
                    indexCount = 1;
                    mountVersion = aoc_i1_v0b.Version;
                    checkAocContents(totalIndexCount, indexCount, mountVersion);


                    // nand のデータを削除し、sd のみの時のハッシュ値を取得し、上記のハッシュ値が sdであることを確認
                    // nand のaoc削除
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application delete-entity {app.Identifier} --aoc -s builtin"
                    ));

                    // Aoc のハッシュ値をログ出力
                    ExpectEvaluator.IsTrue(executor.RunOnTarget($"0x{app.Identifier}",
                        @"--pattern-failure-exit ""\[FAILURE\]"" ",
                        $"--Command=CheckAocHash%CheckIndexNo:1"
                    ));

                    {
                        // ハッシュ値のチェック ( 上で出力したハッシュ値と一致すること
                        // 一致すれば 上の出力は sd のハッシュ値で正しい
                        var expected = @"AddOnContent mount check for `hash \[ " + aoc_i1_v0b_hash + @" \]";
                        ExpectEvaluator.IsTrue(MatchTargetLog(executor, expected));
                    }
                }
                CleanupContents(executor, true);

            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// DynamicCommitイベントを正常に取得できるかを確認
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-68176")]
        public void TestDynamicCommitEvent()
        {
            using (var scope = new TestMethodLog())
            {
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                var intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                // 本テストのメタ構成は以下の通り。
                // Application  : 0x01001a500005e0f6, version 0      : UseSmallCode( false )
                // AoC index 1  : 0x01001a500005f0f7, version 0
                // AoC index 1  : 0x01001a500005f0f7, version 65536
                // AoC index 99 : 0x01001a500005f159, version 0
                var IdForApplication = new ID64(0x01001a500005e0f6);

                // Make parameter for test contents generating.
                var genApp = new TestApplication.ApplicationParameter(IdForApplication, 0);
                // --rutime-install-aoc オプションを付ける
                genApp.RuntimeInstallAoc = true;
                genApp.ChangeCodePath(System.IO.Path.Combine(executor.GetOutputTestDirectory("ShopRuntimeAssistants"), "code"));

                var genAoc_i1v0 = new TestApplication.AddonParameter(IdForApplication, 0, 1, 64 * 1024);
                var genAoc_i1v1 = new TestApplication.AddonParameter(IdForApplication, 1, 1, 64 * 1024);
                var genAoc_i99v0 = new TestApplication.AddonParameter(IdForApplication, 0, 99, 64 * 1024);

                // Request list of parameters.
                var request = new List<TestApplication.GenerateParameter<int>>(4) { genApp, genAoc_i1v0, genAoc_i1v1, genAoc_i99v0 };

                var contents = TestApplication.MakeContents(intermediate, request);
                var app = contents[0];
                var aoc_i1v0 = contents[1];
                var aoc_i1v1 = contents[2];
                var aoc_i99v0 = contents[3];

                // 念のため動作環境をきれいにしておく
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $"application clear-task-status-list AND " +
                    $"application uninstall --all AND " +
                    $"application install \"{app.NspPath}\""
                ));

                /// --------------------------------------------------------------------------
                /// finally
                /// --------------------------------------------------------------------------
                scope.AddDisposer(() =>
                {
                    // テスト実行後も環境をクリアにする
                    executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all");
                });

                System.Action<GeneratedContentResult> CreateAocMountRequestXml = (lExpectMountableAoc) =>
                {
                    var xmlFilePath = System.IO.Path.Combine(intermediate, "AocMountRequest.xml");
                    if (System.IO.File.Exists(xmlFilePath))
                    {
                        // 存在すれば消しておく
                        System.IO.File.Delete(xmlFilePath);
                    }

                    var prop = lExpectMountableAoc.Properties;
                    var aocVersion = lExpectMountableAoc.Version;
                    var aocIndex = prop.GetValueAsUI32(ContentMeta.Contexture.Advance.PropertyKey.Index.ToString());
                    /// AocMountRequest.xml 作成
                    var xml = new System.Xml.XmlDocument();
                    xml.AppendChild(xml.CreateXmlDeclaration(@"1.0", @"UTF-8", @"yes"));
                    var eXmlRoot = xml.CreateElement("Root");
                    xml.AppendChild(eXmlRoot);
                    var eXmlAocIndex = xml.CreateElement("AocIndex");
                    var eXmlAocVersion = xml.CreateElement("AocVersion");
                    eXmlRoot.AppendChild(eXmlAocIndex);
                    eXmlRoot.AppendChild(eXmlAocVersion);
                    eXmlAocIndex.InnerText = $"{aocIndex}";
                    eXmlAocVersion.InnerText = $"{aocVersion}";
                    xml.Save(xmlFilePath);
                };

                System.Action<GeneratedContentResult, string> DynamicCommitAocVerifier = (lExpectMountableAoc, lExpectAocMountResult) =>
                {
                    // 期待値となる XML ファイルを作成しておく
                    CreateAocMountRequestXml(lExpectMountableAoc);

                    // アプリを起動
                    // 本テスト用の CheckDynamicCommitAoc で動作させる
                    ExpectEvaluator.IsTrue(executor.RunOnTarget($"0x{app.Identifier}",
                        @"--pattern-failure-exit ""\[FAILURE\]"" --no-wait",
                        $"--testns_MountHost={intermediate} --Command=CheckDynamicCommitAoc"
                    ));

                    System.Threading.Thread.Sleep(2 * 1000); // アプリ起動 2秒待ち

                    // Aoc を --dynamic 付きでインストール( AddOnContentListChangedEvent がアプリ内に通知されるはず)
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystemSuppressAutoKill(
                        $"addoncontent install \"{lExpectMountableAoc.NspPath}\" --dynamic"
                    ));

                    // アプリが正常に終了したかどうかの監視
                    // AddOnContentListChangedEvent が発生しない場合などの異常パターンを検知するため
                    uint timeoutSecondsUntilAocMount = 15;
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystemCustomOption($"util sleep {timeoutSecondsUntilAocMount}",
                        new List<string>(2)
                        {
                            "--suppress-auto-kill",
                            $"--pattern-success-exit {lExpectAocMountResult}",
                        },
                        timeoutSecondsUntilAocMount + 1U
                    ));
                };

                var successLog = @"""[CheckDynamicCommitAoc] Success.""";

                // インデックス 1, バージョン 0 の Aoc インストール実行、検証
                DynamicCommitAocVerifier(aoc_i1v0, successLog);
                // インデックス 99, バージョン 0 の Aoc インストール実行、検証
                DynamicCommitAocVerifier(aoc_i99v0, successLog);
                // インデックス 1, バージョン 1 の Aoc インストール実行、検証
                // ただし、アプリが v0 のコンテンツをマウントしているときには正しく動作しないため、動作サポート対象とはならない
                DynamicCommitAocVerifier(aoc_i1v1, successLog);
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// テストメソッド: TestForPrepurchase
        /// あらかじめダウンロードの購入・ダウンロード・チケット取得検証
        /// 予約購入でコンテンツを購入してコンテンツの状態が予約状態でダウンロードされるか
        /// 予約購入状態で機器認証を解除すると予約状態が解除されるか、また再度機器認証をすると予約状態となるか
        /// 予約購入状態からコンテンツの配信日時となるとチケットを取得でき、コンテンツを利用できるか
        /// 予約購入状態でコンテンツ利用可能通知(ETicketAvailable)を受け取るとチケットを取得でき、コンテンツを利用できるか
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-68706")]
        public void TestForPrepurchase()
        {
            using (var scope = new TestMethodLog())
            {
                var method = MethodBase.GetCurrentMethod();

                // 作業用ディレクトリ
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                // オーナーアプリケーション: 0x01001e30074dc000
                // アプリケーションはアップロード済みを前提とする
                var IdForApplication = new ID64(0x01001e30074dc000);
                var InitialCode = "1DJAA";

                // 購入者アカウント
                var account = NintendoAccount;

                // upload executor to td1 server by ContentsUploader.
                D4cHelper.NspUploader uploader = new D4cHelper.NspUploader(
                    intermediate,
                    ProxyConfiguration,
                    ServerEnvironment
                );

                // td1 環境のサーバ設定
                const int ComeBackAfter = 300;            // 権利問い合わせの後、再度問い合わせが可能になる時間 ( s )
                const int NotificationAfterRelease = 300; // 配信開始後、コンテンツ利用可能通知(ETicketAvailable)が発行される時間 ( s )

                // 配信日時を設定する ( 配信開始日時は現在時刻から 15 分後以降で一番近い 5 の倍数の時間とする )
                var releaseDate = System.DateTime.Now;
                releaseDate = releaseDate.AddMinutes(15 + (5 - releaseDate.Minute % 5));
                releaseDate = releaseDate.AddSeconds(-releaseDate.Second);
                var releaseDateString = "\"" + releaseDate.ToShortDateString() + " " + releaseDate.ToShortTimeString() + "\"";

                // 時刻をログ出力
                System.Console.WriteLine($"[TimeInfo] TestStartDate: {System.DateTime.Now.ToString()}");
                System.Console.WriteLine($"[TimeInfo] ReleaseDate: {releaseDate.ToString()}");

                // PMS登録
                uploader.RegisterTitleContent(CliTokenAccount, IdForApplication, InitialCode, "auto", releaseDateString, "JP");   // PMS/ROM, 製品登録 + 予約販売設定 ( app )

                // ログから ns-uid を取得する
                var nsuid = new Regex("ApplicationNsUid:  (?<nsuid>\\d+?) ").Match(uploader.LastLog).Groups["nsuid"].Value;

                // finally: 機器認証解除する
                // finally: アカウントを削除し、ニンテンドーアカウント連携も切る
                // finally: デバイスアカウントをサーバから解除して、ローカルからも削除する。クリーンアップ用。
                scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("shop unlink-device-all THEN account clear_all THEN shop unregister-device-account"); });

                // # TSL＆インストール状態をクリア
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application clear-task-status-list AND application uninstall --all"));
                // # （ローカルの）ユーザを追加して、ニンテンドーアカウントをユーザに紐付ける
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $"account add --register_nsa AND account link --index 0 --id {account.ID} --password {account.PASSWORD}"
                ));
                // # デバイスアカウントをサーバに登録( ローカルにトークンを保存 ), ショップのアカウント状態を確認する(バーチャルアカウント作成)
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("shop register-device-account AND shop shop-account-status 0"));
                // # DTLクリア確認
                ExpectEvaluator.IsTrue(D4cHelper.ClearDownloadTaskListOnShopServer(executor));
                // # 機器認証する
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("shop link-device 0 AND shop delete-all-rights AND ticket delete-all"));

                // SMT 上の販売設定が切り替わるまで待つ
                System.Threading.Thread.Sleep(System.TimeSpan.FromMinutes(10));

                // 予約販売 app 購入 ( DTL が更新され、npns でプッシュ通知され、非同期にダウンロードが始まります。)
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $"shop purchase 0 --ns-uid {nsuid} --auto"
                ));

                // ダウンロード待ち
                bool resultWaitDownload = executor.RunDevMenuCommandSystem(
                    $"--time application wait-download 0x{IdForApplication} AND application verify 0x{IdForApplication}"
                );

                // 以下ログからの成否確認用
                // 受信したDTLをログに残す
                ThrowFrameworks.SkipThrow(() => { executor.RunDevMenuCommandSystem("application request-download-task-list-data"); });
                // アプリケーション通知情報をログに残す
                ThrowFrameworks.SkipThrow(() => { executor.RunDevMenuCommandSystem("application notification-info"); });

                // 情報残した後、Assertチェック
                ExpectEvaluator.IsTrue(resultWaitDownload);

                // 購入完了が配信日時後になっていたらテスト失敗にする
                ExpectEvaluator.IsTrue(System.DateTime.Now < releaseDate);

                // 時刻をログ出力
                System.Console.WriteLine($"[TimeInfo] PurchaseCompletedDate: {System.DateTime.Now.ToString()}");

                // インストールチェック ( with Ticket rights status )
                var matchStorageAny = SigloHelper.ResultMatcher.ContentValue.Storage.Any;
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application content-meta-status 0x{IdForApplication} --check-rights"));
                ExpectEvaluator.IsEqual(0, SigloHelper.ResultMatcher.Application.ContentMetaStatusCheckRights.Find(
                    executor.OutputStream.Standard.ToString(),
                    new SigloHelper.ResultMatcher.ContentValue(IdForApplication, 0, ContentMeta.Type.Application, matchStorageAny, SigloHelper.ResultMatcher.ContentValue.RightsCheck.NoRights)
                ));

                // [コンテンツの状態が予約購入状態になっているか確認する。]
                // 権利チェック ( コンテンツが予約購入状態なっているか確認 )
                ExpectEvaluator.IsFalse(executor.RunDevMenuCommandSystem($"application check-launch-rights 0x{IdForApplication}"));
                ExpectEvaluator.IsEqual(true, SigloHelper.ResultMatcher.Application.CheckLaunchRights.Find(
                    executor.OutputStream.Standard.ToString(),
                    SigloHelper.ResultMatcher.Application.CheckLaunchRights.Status.ContentPrepurchased
                ).Success);

                // チケット検証 ( 予約購入情報を取得しているか確認 )
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"ticket list-all --ticket-type prepurchased"));
                ExpectEvaluator.IsEqual(true, SigloHelper.ResultMatcher.Ticket.ListAll.Find(
                    executor.OutputStream.Standard.ToString(),
                    IdForApplication
                ).Success);

                // チケット検証 ( パーソナライズドチケットを取得していないか確認 )
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"ticket list-all --ticket-type personalized"));
                ExpectEvaluator.IsEqual(false, SigloHelper.ResultMatcher.Ticket.ListAll.Find(
                    executor.OutputStream.Standard.ToString(),
                    IdForApplication
                ).Success);

                // 時刻をログ出力
                System.Console.WriteLine($"[TimeInfo] CheckRightsBeforeReleaseDate: {System.DateTime.Now.ToString()}");

                // 予約購入したコンテンツが利用可能になっているか問い合わせ ( 1回目: 配信前 )
                // この段階ではまだ利用できないので、ContentStillUnavailable となるか確認
                ExpectEvaluator.IsFalse(executor.RunDevMenuCommandSystem($"application download-application-prepurchased-rights 0x{IdForApplication}"));
                ExpectEvaluator.IsEqual(true, SigloHelper.ResultMatcher.Application.DownloadApplicationPrepurchasedRights.Find(
                    executor.OutputStream.Standard.ToString(),
                    SigloHelper.ResultMatcher.Application.DownloadApplicationPrepurchasedRights.Status.ContentStillUnavailable
                ).Success);
                var lastCheckRightsDate = System.DateTime.Now;

                // [機器認証解除するとコンテンツの予約状態が解除されることを確認する。また再度機器認証するとコンテンツが予約状態になることを確認する。]
                // 機器認証解除する
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("shop unlink-device 0"));

                // 権利チェック ( 機器認証解除されたので、コンテンツが権利なしとなっているか確認 )
                ExpectEvaluator.IsFalse(executor.RunDevMenuCommandSystem($"application check-launch-rights 0x{IdForApplication}"));
                ExpectEvaluator.IsEqual(true, SigloHelper.ResultMatcher.Application.CheckLaunchRights.Find(
                    executor.OutputStream.Standard.ToString(),
                    SigloHelper.ResultMatcher.Application.CheckLaunchRights.Status.RightsNotFound
                ).Success);

                // 機器認証する
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("shop link-device 0"));

                // チケット同期する
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("shop sync-ticket"));

                // 権利チェック ( 機器認証されたので、コンテンツが再度予約購入状態なっているか確認 )
                ExpectEvaluator.IsFalse(executor.RunDevMenuCommandSystem($"application check-launch-rights 0x{IdForApplication}"));
                ExpectEvaluator.IsEqual(true, SigloHelper.ResultMatcher.Application.CheckLaunchRights.Find(
                    executor.OutputStream.Standard.ToString(),
                    SigloHelper.ResultMatcher.Application.CheckLaunchRights.Status.ContentPrepurchased
                ).Success);

                // 配信時刻か次に問い合わせが可能になる時刻の遅い方まで待つ
                var waitTime = releaseDate - System.DateTime.Now > lastCheckRightsDate + System.TimeSpan.FromSeconds(ComeBackAfter) - System.DateTime.Now
                    ? releaseDate - System.DateTime.Now
                    : lastCheckRightsDate + System.TimeSpan.FromSeconds(ComeBackAfter) - System.DateTime.Now;
                System.Threading.Thread.Sleep(waitTime + System.TimeSpan.FromSeconds(30));

                // [配信開始時刻になるとチケットがダウンロードでき、コンテンツが利用可能になることを確認する。]
                // 時刻をログ出力
                System.Console.WriteLine($"[TimeInfo] CheckRightsAfterReleaseDate: {System.DateTime.Now.ToString()}");

                // 予約購入したコンテンツが利用可能になっているか問い合わせ ( 2回目:配信後 )
                // 配信後なので、Available となるかを確認する
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application download-application-prepurchased-rights 0x{IdForApplication}"));
                ExpectEvaluator.IsEqual(true, SigloHelper.ResultMatcher.Application.DownloadApplicationPrepurchasedRights.Find(
                    executor.OutputStream.Standard.ToString(),
                    SigloHelper.ResultMatcher.Application.DownloadApplicationPrepurchasedRights.Status.Available
                ).Success);

                // 権利チェック ( コンテンツが利用できるようになっているか確認 )
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"application check-launch-rights 0x{IdForApplication}"));
                ExpectEvaluator.IsEqual(true, SigloHelper.ResultMatcher.Application.CheckLaunchRights.Find(
                    executor.OutputStream.Standard.ToString(),
                    SigloHelper.ResultMatcher.Application.CheckLaunchRights.Status.Available
                ).Success);

                // 起動チェック
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommand($"application launch 0x{IdForApplication}"));

                // [チケット利用可能通知が届いてチケットがダウンロード出来ることを確認する。]
                // 手動で取得したパーソナライズドチケットを削除する
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("ticket delete-all --ticket-type personalized"));

                // 権利チェック ( パーソナライズドチケットを削除したので、コンテンツが再度予約購入状態なっているか確認 )
                ExpectEvaluator.IsFalse(executor.RunDevMenuCommandSystem($"application check-launch-rights 0x{IdForApplication}"));
                ExpectEvaluator.IsEqual(true, SigloHelper.ResultMatcher.Application.CheckLaunchRights.Find(
                    executor.OutputStream.Standard.ToString(),
                    SigloHelper.ResultMatcher.Application.CheckLaunchRights.Status.ContentPrepurchased
                ).Success);

                // 通知が発行される日時を計算する
                var notificationDate = releaseDate + System.TimeSpan.FromSeconds(NotificationAfterRelease);

                // 通知が発行される日時後になっていたらテスト失敗にする
                ExpectEvaluator.IsTrue(System.DateTime.Now < notificationDate);

                // 通知が届くまで待つ
                System.Threading.Thread.Sleep(notificationDate - System.DateTime.Now + System.TimeSpan.FromSeconds(30));

                // 権利チェック ( コンテンツが利用できるようになっているか確認 )
                bool isContentAvailable = false;
                for (int i = 0; i < 6; i++) // 通知は最大 5 分遅れる可能性があるので、発行時刻の 6 分後程度までリトライする
                {
                    isContentAvailable = executor.RunDevMenuCommandSystem($"application check-launch-rights 0x{IdForApplication}");

                    if (isContentAvailable)
                    {
                        ExpectEvaluator.IsEqual(true, SigloHelper.ResultMatcher.Application.CheckLaunchRights.Find(
                            executor.OutputStream.Standard.ToString(),
                            SigloHelper.ResultMatcher.Application.CheckLaunchRights.Status.Available
                        ).Success);

                        break;
                    }
                    else
                    {
                        System.Threading.Thread.Sleep(System.TimeSpan.FromMinutes(1));
                    }
                }

                // 通知によりチケットが取得されたか確認
                ExpectEvaluator.IsTrue(isContentAvailable);

                // 起動チェック
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommand($"application launch 0x{IdForApplication}"));
            }
        }

        [DataContract]
        class ErrorContext
        {
            [DataMember(Order = 0)]
            public string result { get; set; }
            [DataMember(Order = 1)]
            public string fqdn { get; set; }
            [DataMember(Order = 2)]
            public string ip { get; set; }
        }
        //!----------------------------------------------------------------------------
        /// <summary>
        /// テストメソッド: TestForErrorContext
        /// ErrorContext を正しく取得できるかどうか
        /// エラーシミュレートと ShopRuntimeAssistants を駆使して ErrorContext が取れることをテストする
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-70287")]
        public void TestForErrorContext()
        {
            using (var scope = new TestMethodLog())
            {
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                System.Func<ErrorContext> getErrorContext = () =>
                {
                    var outputLines = executor.OutputStream.Standard.ToString().Split('\n');
                    var errorContextString = outputLines.FirstOrDefault((string s) => s.Contains("[CheckErrorContext] ErrorContextJson"));
                    if (errorContextString == null)
                    {
                        return null;
                    }
                    var json = errorContextString.Split(new System.String[] { "ErrorContextJson: " }, System.StringSplitOptions.None)[1].Trim();
                    return json.DeserializeFromJson<ErrorContext>();
                };

                // 動作確認のために、外部鍵で、パッチテストで使っている ID と同じアプリを作る
                var intermediate = GenerateIntermediateDirectoryAsMethod(executor);
                var IdForApplication = new ID64(0x0100f6900049a000);
                var genApp = new TestApplication.ApplicationParameter(IdForApplication, 0);
                genApp.TicketEncryption = true;
                var request = new List<TestApplication.GenerateParameter<int>>() { genApp };
                var contents = TestApplication.MakeContents(intermediate, request);
                var app = contents[0];

                var intermediateCard = System.IO.Path.Combine(intermediate, "card");
                var genAppCard = new TestApplication.ApplicationParameter(IdForApplication, 0);
                var requestCard = new List<TestApplication.GenerateParameter<int>>() { genAppCard };
                var contentsCard = TestApplication.MakeContents(intermediateCard, requestCard);
                var appCard = contentsCard[0];

                // ゲームカードが挿さっているか確認
                var hasInsertedGameCard = executor.RunDevMenuCommandSystem($"gamecard status", @"^Inserted$");

                scope.AddDisposer(() =>
                {
                    // ゲームカードがある場合はゲームカード内を削除
                    if (hasInsertedGameCard) { ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("gamecard erase")); }

                    executor.RunDevMenuCommandSystem(
                        "application uninstall --all;" +
                        "debug disable-nim-error-simulate;" +
                        "shop unlink-device-all;" +
                        "account clear_all;" +
                        "shop unregister-device-account;"
                    );
                });

                {
                    var account = NintendoAccount;

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "account add --register_nsa AND " +
                        string.Format("account link --index 0 --id {0} --password {1}", account.ID, account.PASSWORD) + " AND " +
                        "shop unregister-device-account AND " +
                        "shop register-device-account AND " +
                        "application install " + app.NspPath
                    ));
                    ExpectEvaluator.IsTrue(RebootTargetByPowerCommand(executor));
                }
                var assist = System.IO.Path.Combine(executor.GetOutputTestDirectory("ShopRuntimeAssistants"), "ShopRuntimeAssistants.nsp");
                {
                    // ApplicationControlData
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-nim-error-simulate --url https://atum.hac.td1.d4c.nintendo.net/a/d --result 0x12345678"));
                    ExpectEvaluator.IsTrue(executor.RunOnTarget(assist, argsOfNsp: "--Command=CheckErrorContext --AsyncName=ApplicationControlData"));
                    var errorContext = getErrorContext();
                    ExpectEvaluator.IsTrue(errorContext != null);
                    ExpectEvaluator.IsTrue(errorContext.fqdn == "atum.hac.td1.d4c.nintendo.net");
                    ExpectEvaluator.IsTrue(errorContext.result == "0x12345678");
                }
                {
                    // EnsureDownloadTask
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-nim-error-simulate --url https://pushmo.hac.td1.eshop.nintendo.net/pushmo --result 0x12345678"));
                    ExpectEvaluator.IsTrue(executor.RunOnTarget(assist, argsOfNsp: "--Command=CheckErrorContext --AsyncName=EnsureDownloadTask"));
                    var errorContext = getErrorContext();
                    ExpectEvaluator.IsTrue(errorContext != null);
                    ExpectEvaluator.IsTrue(errorContext.fqdn == "pushmo.hac.td1.eshop.nintendo.net");
                    ExpectEvaluator.IsTrue(errorContext.result == "0x12345678");
                }
                {
                    // DownloadTaskListData
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-nim-error-simulate --url https://pushmo.hac.td1.eshop.nintendo.net/pushmo --result 0x12345678"));
                    ExpectEvaluator.IsTrue(executor.RunOnTarget(assist, argsOfNsp: "--Command=CheckErrorContext --AsyncName=DownloadTaskListData"));
                    var errorContext = getErrorContext();
                    ExpectEvaluator.IsTrue(errorContext != null);
                    ExpectEvaluator.IsTrue(errorContext.fqdn == "pushmo.hac.td1.eshop.nintendo.net");
                    ExpectEvaluator.IsTrue(errorContext.result == "0x12345678");
                }
                {
                    // CheckLatestUpdate
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-nim-error-simulate --url https://sun.hac.td1.d4c.nintendo.net/v1/system_update_meta --result 0x12345678"));
                    ExpectEvaluator.IsTrue(executor.RunOnTarget(assist, argsOfNsp: "--Command=CheckErrorContext --AsyncName=CheckLatestUpdate"));
                    var errorContext = getErrorContext();
                    ExpectEvaluator.IsTrue(errorContext != null);
                    ExpectEvaluator.IsTrue(errorContext.fqdn == "sun.hac.td1.d4c.nintendo.net");
                    ExpectEvaluator.IsTrue(errorContext.result == "0x12345678");
                }
                {
                    // DownloadLatestSystemUpdate
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-nim-error-simulate --url https://sun.hac.td1.d4c.nintendo.net/v1/system_update_meta --result 0x12345678"));
                    ExpectEvaluator.IsTrue(executor.RunOnTarget(assist, argsOfNsp: "--Command=CheckErrorContext --AsyncName=DownloadLatestUpdate"));
                    var errorContext = getErrorContext();
                    ExpectEvaluator.IsTrue(errorContext != null);
                    ExpectEvaluator.IsTrue(errorContext.fqdn == "sun.hac.td1.d4c.nintendo.net");
                    ExpectEvaluator.IsTrue(errorContext.result == "0x12345678");
                }
                {
                    // ApplicationUpdateInfo
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-nim-error-simulate --url https://superfly.hac.td1.d4c.nintendo.net/v1/a --result 0x12345678"));
                    ExpectEvaluator.IsTrue(executor.RunOnTarget(assist, argsOfNsp: "--Command=CheckErrorContext --AsyncName=ApplicationUpdateInfo"));
                    var errorContext = getErrorContext();
                    ExpectEvaluator.IsTrue(errorContext != null);
                    ExpectEvaluator.IsTrue(errorContext.fqdn == "superfly.hac.td1.d4c.nintendo.net");
                    ExpectEvaluator.IsTrue(errorContext.result == "0x12345678");
                }
                {
                    // UpdateApplication
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-nim-error-simulate --url https://superfly.hac.td1.d4c.nintendo.net/v1/a --result 0x12345678"));
                    ExpectEvaluator.IsTrue(executor.RunOnTarget(assist, argsOfNsp: "--Command=CheckErrorContext --AsyncName=UpdateApplication"));
                    var errorContext = getErrorContext();
                    ExpectEvaluator.IsTrue(errorContext != null);
                    ExpectEvaluator.IsTrue(errorContext.fqdn == "superfly.hac.td1.d4c.nintendo.net");
                    ExpectEvaluator.IsTrue(errorContext.result == "0x12345678");
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application cancel-download 0x0100f6900049a000"));
                }
                {
                    // LinkDevice
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-nim-error-simulate --url https://beach.hac.td1.eshop.nintendo.net/v1/my/account --result 0x12345678"));
                    ExpectEvaluator.IsTrue(executor.RunOnTarget(assist, argsOfNsp: "--Command=CheckErrorContext --AsyncName=LinkDevice --UserIndex=0"));
                    var errorContext = getErrorContext();
                    ExpectEvaluator.IsTrue(errorContext != null);
                    ExpectEvaluator.IsTrue(errorContext.fqdn == "beach.hac.td1.eshop.nintendo.net");
                    ExpectEvaluator.IsTrue(errorContext.result == "0x12345678");

                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-nim-error-simulate --url https://beach.hac.td1.eshop.nintendo.net/v1/my/devices/hac/link --result 0x12345678"));
                    ExpectEvaluator.IsTrue(executor.RunOnTarget(assist, argsOfNsp: "--Command=CheckErrorContext --AsyncName=LinkDevice --UserIndex=0"));
                    errorContext = getErrorContext();
                    ExpectEvaluator.IsTrue(errorContext != null);
                    ExpectEvaluator.IsTrue(errorContext.fqdn == "beach.hac.td1.eshop.nintendo.net");
                    ExpectEvaluator.IsTrue(errorContext.result == "0x12345678");
                }

                Log.WriteLine("Test of GameCard");
                // ゲームカードにデータ書き込み
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($@"gamecard write ""{appCard.NspPath}"""));
                {
                    // CheckGameCardRegistration
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-nim-error-simulate --url https://beach.hac.td1.eshop.nintendo.net/v1/rom_cards --result 0x12345678"));
                    ExpectEvaluator.IsTrue(executor.RunOnTarget(assist, argsOfNsp: "--Command=CheckErrorContext --AsyncName=CheckGameCardRegistration"));
                    var errorContext = getErrorContext();
                    ExpectEvaluator.IsTrue(errorContext != null);
                    ExpectEvaluator.IsTrue(errorContext.fqdn == "beach.hac.td1.eshop.nintendo.net");
                    ExpectEvaluator.IsTrue(errorContext.result == "0x12345678");
                }
                {
                    // GameCardRegistrationGoldPoint
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-nim-error-simulate --url https://beach.hac.td1.eshop.nintendo.net/v1/rom_cards --result 0x12345678"));
                    ExpectEvaluator.IsTrue(executor.RunOnTarget(assist, argsOfNsp: "--Command=CheckErrorContext --AsyncName=GameCardRegistrationGoldPoint --UserIndex=0"));
                    var errorContext = getErrorContext();
                    ExpectEvaluator.IsTrue(errorContext != null);
                    ExpectEvaluator.IsTrue(errorContext.fqdn == "beach.hac.td1.eshop.nintendo.net");
                    ExpectEvaluator.IsTrue(errorContext.result == "0x12345678");
                }
                {
                    // RegisterGameCard
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-nim-error-simulate --url https://beach.hac.td1.eshop.nintendo.net/v1/rom_cards --result 0x12345678"));
                    ExpectEvaluator.IsTrue(executor.RunOnTarget(assist, argsOfNsp: "--Command=CheckErrorContext --AsyncName=RegisterGameCard --UserIndex=0"));
                    var errorContext = getErrorContext();
                    ExpectEvaluator.IsTrue(errorContext != null);
                    ExpectEvaluator.IsTrue(errorContext.fqdn == "beach.hac.td1.eshop.nintendo.net");
                    ExpectEvaluator.IsTrue(errorContext.result == "0x12345678");
                }
                // ゲームカードがある場合はゲームカード内を削除
                if (hasInsertedGameCard) { ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("gamecard erase")); }

                // DownloadApplicationPrepurchasedRights のテストを行うためにアプリのチケットを削除する
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("ticket delete-all --ticket-type common"));
                {
                    // DownloadApplicationPrepurchasedRights
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-nim-error-simulate --url https://ecs-td1.hac.shop.nintendo.net/ecs/services/rest/AccountListETicketIdsByRightsId --result 0x12345678"));
                    ExpectEvaluator.IsTrue(executor.RunOnTarget(assist, argsOfNsp: "--Command=RegisterPrepurchaseInfo"));
                    ExpectEvaluator.IsTrue(executor.RunOnTarget(assist, argsOfNsp: "--Command=CheckErrorContext --AsyncName=DownloadApplicationPrepurchasedRights"));
                    var errorContext = getErrorContext();
                    ExpectEvaluator.IsTrue(errorContext != null);
                    ExpectEvaluator.IsTrue(errorContext.fqdn == "ecs-td1.hac.shop.nintendo.net");
                    ExpectEvaluator.IsTrue(errorContext.result == "0x12345678");
                }
            }
        }


        //!----------------------------------------------------------------------------
        /// <summary>
        /// NT をショップに登録する際のリトライ間隔のテスト
        /// NT をショップに登録時、最初の失敗からリトライ間隔を倍々にしていき、
        /// 最終的には指定したリトライ間隔の上限値を超えないことを確認する。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-75932")]
        public void TestForNtRegistrationRetryInterval()
        {
            using (var scope = new TestMethodLog())
            {
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

                scope.AddDisposer(() =>
                {
                    // 変更した設定を元に戻す。
                    executor.RunDevMenuCommandSystem(
                        "debug disable-nim-error-simulate;" +
                        "debug set-integer-fwdbg --name ns.notification --key retry_interval_min 60;" +
                        "debug set-integer-fwdbg --name ns.notification --key retry_interval_max 900;"
                    );
                });

                // 初回リトライ時の待ち秒と、上限の秒を設定
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug set-integer-fwdbg --name ns.notification --key retry_interval_min 5"));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug set-integer-fwdbg --name ns.notification --key retry_interval_max 60"));

                // デバイスID の取得
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("systemupdate get-device-id"));

                // ログからデバイスID を取得する
                Regex regDeviceId = new Regex(@"\[target\] [0-9a-f]{16}");
                System.Text.RegularExpressions.Match mc = regDeviceId.Match(executor.OutputStream.Standard.ToString());
                // デバイスID が存在したかチェック
                ExpectEvaluator.IsTrue(mc.Success);
                var matchString = mc.Groups[0].Value;
                // デバイスIDの数字部分だけ取り出す
                var deviceId = matchString.Substring(9);

                Log.WriteLine("deviceId {0}", deviceId);

                // NT登録をエラーにする設定
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"debug enable-nim-error-simulate --url https://beach.hac.td1.eshop.nintendo.net/v1/devices/{deviceId}/notification_token/register --result 0x01234567"));

                // 再起動(システム起動時に ns が npns から NotificationToken を取得する為)
                ExpectEvaluator.IsTrue(executor.ResetTarget());

                // 待ち時間(215) = 5 + 10 + 20 + 40 + 60 + 60 +20(余時間)
                uint timeoutSeconds = 215;
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($"util sleep {timeoutSeconds}",
                    timeoutSeconds + 1U
                ));

                // Interval が最小値から倍に増えることと、最大値以上増えないことの確認用の配列。
                // 基本的に出力するログは、…40, 60, 60 まで、ラグにより …40, 60, 60, 60 まで出力されることを考慮。
                string[] judgeInterval = { "5", "10", "20", "40", "60", "60", "60" };

                // ログから一番最初に出力した Interval のログを取得
                Regex IntervalTime = new Regex(@"\[PushNotificationDispatcher\] Interval (.*?) sec.");
                mc = IntervalTime.Match(executor.OutputStream.Standard.ToString());
                ExpectEvaluator.IsTrue(mc.Success);

                // ログに出力した Interval の値をチェックし、次の Interval ログを取得する。
                int i = 0;
                while (mc.Success)
                {
                    // Interval の値確認。
                    Log.WriteLine("Log Interval {0} sec. Expectation Interval {1} sec.", mc.Groups[1].Value, judgeInterval[i]);
                    ExpectEvaluator.IsTrue(judgeInterval[i] == mc.Groups[1].Value );
                    // 次の Intervalログ 検索結果を取得
                    mc = mc.NextMatch();
                    i++;
                    if (i > judgeInterval.Length)
                    {
                        // sleep 時間の設定から 8個以上の Interval はでない。
                        throw new UnexpectFailureException("A pattern that does not exist.");
                    }
                }
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// 検証内容：ListAddOnContentの offset を指定し、正しくindexが返ってくるかを確認する。
        /// 単体実行時で 1 分程度要します。
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-80092")]
        public void TestForOffsetOfListAddOnContent()
        {
            using (var scope = new TestMethodLog())
            {
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                var intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                // make contents
                //  Application : 0x01001a500005e0f5, version 0
                //  AddOnContent: 0x01001a500005f0f6, version 0,      index 1
                //              : 0x01001a500005f0f7, version 0,      index 2
                var owner  = new ID64(0x01001a500005e0f5); //(TestForOnCardAocAccessLibrary のデータ流用 )
                var @params = new List<TestApplication.GenerateParameter<int>>();
                {
                    // app
                    var param = new TestApplication.ApplicationParameter(owner, 0);
                    param.UseSmallCode = true;
                    param.ChangeCodePath(System.IO.Path.Combine(executor.GetOutputTestDirectory("ShopRuntimeAssistants"), "code"));
                    @params.Add(param);
                }
                {
                    // aoc
                    var param0 = new TestApplication.AddonParameter(owner, 0, 1);
                    @params.Add(param0);
                    var param1 = new TestApplication.AddonParameter(owner, 0, 2);
                    @params.Add(param1);
                }

                var contents = TestApplication.MakeContents(intermediate, @params);
                var catalog = new GeneratedContentResult.TypeCategorizedCatalog(contents);
                var app = catalog.GetTypedCatalog(ContentMeta.Type.Application)[0];
                var aoc_i1 = catalog.GetTypedCatalog(ContentMeta.Type.AddOnContent)[0];
                var aoc_i2 = catalog.GetTypedCatalog(ContentMeta.Type.AddOnContent)[1];

                // ゲームカードが挿さっているか確認
                var hasInsertedGameCard = executor.RunDevMenuCommandSystem($"gamecard status", @"^Inserted$");

                // finally
                if (!IsUnitTestOnIDE)
                {
                    scope.AddDisposer(() => {
                        // ゲームカードがある場合はゲームカード内を削除
                        if (hasInsertedGameCard) { ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("gamecard erase")); }
                        ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application uninstall --all"));
                    });
                }

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application uninstall --all"));

                // 検証内容：app,aoc,aocの順にインストールし ListAddOnContentのoffsetを指定し、正しくindexが返ってくるかを確認する。
                scope.WriteLineWithTimeStamp("app, aoc, aoc");
                {
                    // App, Aoc を install
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"application install {app.NspPath} AND " +
                        $"addoncontent install {aoc_i1.NspPath} AND " +
                        $"addoncontent install {aoc_i2.NspPath}"));

                    // aoc の ListAddOnContent 内容をログ出力 (ShopRuntimeAssistants.cpp の処理が実行される)
                    ExpectEvaluator.IsTrue(executor.RunOnTarget($"0x{app.Identifier}",
                        @"--pattern-failure-exit ""\[FAILURE\]"" ",
                        $"--Command=CheckListAddOnContent"
                    ));

                    // ListAddOnContent のログ内容チェック
                    ExpectEvaluator.IsTrue(MatchTargetLog(executor, "ListAddOnContent:indexCount= 1 offset= 0 index= 1"));
                    ExpectEvaluator.IsTrue(MatchTargetLog(executor, "ListAddOnContent:indexCount= 1 offset= 1 index= 2"));
                }

                // ゲームカードが挿さっている場合のみテストする。
                if (hasInsertedGameCard)
                {
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application uninstall --all"));

                    // 検証内容：GameCardApp,aoc,aocの順にインストールし ListAddOnContentのoffsetを指定し、正しくindexが返ってくるかを確認する。
                    scope.WriteLineWithTimeStamp("GameCardApp, aoc, aoc");
                    {
                        // App, Aoc を install
                        ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                            $"gamecard write {app.NspPath}"
                        ));
                        ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                            $"addoncontent install {aoc_i1.NspPath} AND " +
                            $"addoncontent install {aoc_i2.NspPath}"));
                        // aoc の ListAddOnContent 内容をログ出力 (ShopRuntimeAssistants.cpp の処理が実行される)
                        ExpectEvaluator.IsTrue(executor.RunOnTarget($"0x{app.Identifier}",
                            @"--pattern-failure-exit ""\[FAILURE\]"" ",
                            $"--Command=CheckListAddOnContent"
                        ));

                        // ListAddOnContent のログ内容チェック
                        ExpectEvaluator.IsTrue(MatchTargetLog(executor, "ListAddOnContent:indexCount= 1 offset= 0 index= 1"));
                        ExpectEvaluator.IsTrue(MatchTargetLog(executor, "ListAddOnContent:indexCount= 1 offset= 1 index= 2"));
                    }
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("gamecard erase"));
                }

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("application uninstall --all"));

                // 検証内容：aoc,aoc,app(appが最後)の順にインストールし ListAddOnContentのoffsetを指定し、正しくindexが返ってくるかを確認する。
                scope.WriteLineWithTimeStamp("aoc, aoc, app");
                {
                    // App, Aoc を install
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        $"addoncontent install {aoc_i1.NspPath} AND " +
                        $"addoncontent install {aoc_i2.NspPath} AND " +
                        $"application install {app.NspPath}"));

                    // aoc の ListAddOnContent 内容をログ出力 (ShopRuntimeAssistants.cpp の処理が実行される)
                    ExpectEvaluator.IsTrue(executor.RunOnTarget($"0x{app.Identifier}",
                        @"--pattern-failure-exit ""\[FAILURE\]"" ",
                        $"--Command=CheckListAddOnContent"
                    ));

                    // ListAddOnContent のログ内容チェック
                    ExpectEvaluator.IsTrue(MatchTargetLog(executor, "ListAddOnContent:indexCount= 1 offset= 0 index= 1"));
                    ExpectEvaluator.IsTrue(MatchTargetLog(executor, "ListAddOnContent:indexCount= 1 offset= 1 index= 2"));
                }
            }
        }
    }
}
