﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_StringUtil.h>
#include "testNs_PropertyCollection.h"
#include "libraries/testNs_MountHost.h"
#include "libraries/testNs_RawFileLoader.h"

#include <algorithm>

namespace testns {

    //!---------------------------------------------------------------------------------
    //! @brief コンストラクタ
    //!---------------------------------------------------------------------------------
    KeyValueCollection::KeyValueCollection( const unsigned reservedCapacity ) NN_NOEXCEPT
        : m_pXmlParser( nullptr ),
        m_DetectRootElement( false )
    {
        CollectionType::reserve( reservedCapacity );
    }

    //!---------------------------------------------------------------------------------
    //! @brief デストラクタ
    //!---------------------------------------------------------------------------------
    KeyValueCollection::~KeyValueCollection() NN_NOEXCEPT
    {
        Finalize();
    }

    //!---------------------------------------------------------------------------------
    //! @brief ファイナライザ
    //!---------------------------------------------------------------------------------
    void KeyValueCollection::Finalize() NN_NOEXCEPT
    {
        XmlParser* pParser;
        if ( nullptr != ( pParser = m_pXmlParser ) )
        {
            m_pXmlParser = nullptr;
            XmlParser::Finalize( pParser );
        }
        m_DetectRootElement = false;
        m_DetectedValue.clear();
        m_DetectedKey.clear();
        CollectionType::clear();
    }

    //!---------------------------------------------------------------------------------
    //! @brief コレクションローダー ( from File )
    //!---------------------------------------------------------------------------------
    bool KeyValueCollection::LoadFromFile( const char* pFilePath ) NN_NOEXCEPT
    {
        // XML リソースファイルを読み込み
        XmlParser* pParser;
        if ( nullptr == ( pParser = m_pXmlParser ) )
        {
            m_pXmlParser = pParser = XmlParser::CreateNewParser();
        }
        const char* pStream;
        RawFileLoader loader;
        if ( nullptr != ( pStream = loader.LoadFromFile< char >( pFilePath ) ) )
        {
            CollectionType::clear();
            m_DetectedKey.clear();
            m_DetectedValue.clear();
            m_DetectRootElement = false;
            return pParser->Parse( *this, pStream, loader.GetFileSize() );
        }
        return false;
    }

    //!---------------------------------------------------------------------------------
    //! @brief
    //!---------------------------------------------------------------------------------
    void KeyValueCollection::OnElementBegin( const XmlParser::StringType& name, const XmlParser::Attribute::CollectionType& attributes ) NN_NOEXCEPT
    {
        if ( false == m_DetectRootElement )
        {
            // 最初に見つかるのはルート要素
            m_DetectedKey.clear();
            m_DetectedValue.clear();
            m_DetectRootElement = true;
        }
        else if ( false == name.empty() )
        {
            // キー検出
            m_DetectedKey = name;
            m_DetectedValue.clear();
        }
        NN_UNUSED( attributes );
    }

    //!---------------------------------------------------------------------------------
    //! @brief
    //!---------------------------------------------------------------------------------
    void KeyValueCollection::OnText( const XmlParser::StringType& text ) NN_NOEXCEPT
    {
        if ( false == m_DetectedKey.empty() )
        {
            m_DetectedValue = text;
        }
    }

    //!---------------------------------------------------------------------------------
    //! @brief
    //!---------------------------------------------------------------------------------
    void KeyValueCollection::OnElementEnd( const XmlParser::StringType& name ) NN_NOEXCEPT
    {
        if ( false == m_DetectedKey.empty() && false == m_DetectedValue.empty() )
        {
            ( *this )[ m_DetectedKey ] = m_DetectedValue;
            m_DetectedKey.clear();
            m_DetectedValue.clear();
        }
        NN_UNUSED( name );
    }
}
