﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Net;
using System.Text;
using System.Text.RegularExpressions;

namespace CsTestAssistants.NintendoServices
{
    using Configuration = D4cHelper.Configuration;

    /// <summary>
    /// pms サーバアクセス支援
    /// </summary>
    public class PmsEditor
    {
        public enum Method : byte
        {
            POST,
            DELETE,
            PATCH,
            GET,
            PUT
        }

        public static class Constants
        {
            public static readonly BasicAccount StandardUser = new BasicAccount( "userallall", "R3_userallall" );

            public const string UriApiBase = "https://pms.wc.{0}.eshop.nintendo.net/pms/rest/v2/";
            public const string UriRomsBase = UriApiBase + "roms";
            public const string UriRomsTitle = UriRomsBase + "/{1}";
            public const string UriRomsChangeStatus = UriRomsTitle + "/status";
        }

        public Configuration.ServerEnvironment Server { get; }
        public BasicAccount Account { get; }
        public WebProxy Proxy { get; }

        public PmsEditor( WebProxy proxy = null, Configuration.ServerEnvironment server = null, BasicAccount account = null )
        {
            Server = ( null != server ) ? server : Configuration.ServerEnvironment.DEV6;
            Account = ( null != account ) ? account : Constants.StandardUser;
            Proxy = proxy;
        }

        protected HttpWebRequest CreateBasicRequest( Method method, string uri, string formData = null )
        {
            HttpWebRequest request = WebRequest.CreateHttp( uri );
            request.Accept = "application/json";
            request.ContentType = "application/json";
            request.Headers[ "Authorization" ] = "Basic " + Account.ToAuthorizationBase64();
            request.Proxy = Proxy;
            request.Method = method.ToString();
            request.ServerCertificateValidationCallback = ( sender, cert, chain, sslPolicyErros ) =>
            {
                return true;    // --insecure, -k, 自己署名SSL許可。
            };
            return ( WebExceptionStatus.Success == request.WriteToStream( formData ) ) ? request : null;
        }

        protected HttpExtension.ConnectionResult ExecuteCommunication( HttpWebRequest request )
        {
            if ( null == request )
            {
                throw new System.ArgumentNullException( $"PmsEditor: request is null." );
            }
            HttpExtension.ConnectionResult result;
            WebExceptionStatus rc = request.ConnectToServer( out result );
            if ( WebExceptionStatus.Success != rc || null == result || HttpStatusCode.OK != result.StatusCode )
            {
                throw new UnexpectFailureException( $"PmsEditor: [ Method ={ request.Method }, URI ={ request.Address} ] => [ WebException={rc}, result={result} ]" );
            }
            return result;
        }

        public void RegisterRoms( ID64 id, ContentMeta.Type type )
        {
            // ROM登録
            {
                var request = CreateBasicRequest( Method.POST, string.Format( Constants.UriRomsBase, Server.Name ),
                    $"{{\"title_id\":\"{id}\", \"type\":\"{type.ToString()}\"}}"
                );
                if ( null == request )
                {
                    throw new UnexpectFailureException( $"RegisterRoms: Request could not be created." );
                }
                HttpExtension.ConnectionResult result;
                WebExceptionStatus rc = request.ConnectToServer( out result );
                if ( null != result && !string.IsNullOrEmpty( result.Text ) && result.Text.Contains( "1008" ) )
                {
                    // 1008 レスポンスの時は注釈を書くだけでプロセス続行 ( 旧 5011 )
                    Log.WriteLine( $"1008 Responce: The rom of {id} is already registrated." );
                }
                else if ( WebExceptionStatus.Success != rc || null == result || HttpStatusCode.OK != result.StatusCode )
                {
                    throw new UnexpectFailureException( $"RegisterRoms: [ Method ={ request.Method }, URI ={ request.Address} ] => [ WebException={rc}, result={result} ]" );
                }
            }

            // ROM状態更新
            {
                // rom properties を取得.
                var result = ExecuteCommunication( CreateBasicRequest( Method.GET, string.Format( Constants.UriRomsTitle, Server.Name, id ) ) );
                // property, status の 内容を reject する.
                ExecuteCommunication( CreateBasicRequest( Method.PUT, string.Format( Constants.UriRomsTitle, Server.Name, id ),
                    Regex.Replace( result.Text, "\"status\":\"[A-Za-z0-9]*\",", string.Empty )
                ) );
            }

            // ROM承認依頼
            {
                ExecuteCommunication( CreateBasicRequest( Method.PUT, string.Format( Constants.UriRomsChangeStatus, Server.Name, id ),
                    "{\"status\":\"RequestApproval\"}"
                ) );
            }

            // ROM承認
            {
                ExecuteCommunication( CreateBasicRequest( Method.PUT, string.Format( Constants.UriRomsChangeStatus, Server.Name, id ),
                    "{\"status\":\"Approved\"}"
                ) );
            }
        }
    }
}
