﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Net;
using System.Runtime.Serialization;
using System.Text;

namespace CsTestAssistants.NintendoServices
{
    using Configuration = D4cHelper.Configuration;

    /// <summary>
    /// NPNSサーバーアクセス支援
    /// </summary>
    public static class Npns
    {
        public enum Method : byte
        {
            POST,
        }

        public class Notifier
        {
            /// <summary>
            /// NPNS通知要求コンテナ
            /// </summary>
            [DataContract]
            public class Requests
            {
                /// <summary>
                /// デバイス単位要求
                /// </summary>
                [DataContract]
                public class Request
                {
                    /// <summary>
                    /// デバイストークン
                    /// </summary>
                    [DataMember( Order = 0 )]
                    public string to;
                    /// <summary>
                    /// 要求通知内容データフォーム
                    /// </summary>
                    [DataMember( Order = 1 )]
                    public string data;
                    /// <summary>
                    /// ？
                    /// </summary>
                    [DataMember( Order = 2 )]
                    public bool store_offline;

                    /// <summary>
                    /// コンストラクタ
                    /// </summary>
                    /// <param name="deviceToken"></param>
                    /// <param name="data"></param>
                    /// <param name="store_offline"></param>
                    public Request( string deviceToken, string data, bool store_offline = false )
                    {
                        this.to = deviceToken;
                        this.data = data;
                        this.store_offline = store_offline;
                    }
                }

                /// <summary>
                /// 要求配列
                /// </summary>
                [DataMember( Order = 0 )]
                public Request[] requests;

                /// <summary>
                /// ルートコンストラクタ
                /// </summary>
                /// <param name="requests"></param>
                public Requests( Request[] requests )
                {
                    this.requests = requests;
                }
            }

            // "jd1" or "td1"
            private const string UriApiBase = "https://provider-{0}.npns.srv.nintendo.net/api/v1/notifications";

            public WebProxy Proxy { get; }
            public Configuration.ServerEnvironment Server { get; }
            public string NdidAccessToken { get; }

            public Notifier( string ndidAccessToken, WebProxy proxy = null, Configuration.ServerEnvironment server = null )
            {
                if ( string.IsNullOrEmpty( ndidAccessToken ) )
                {
                    throw new System.ArgumentException();
                }
                Server = ( null != server ) ? server : Configuration.ServerEnvironment.DEV6;
                NdidAccessToken = ndidAccessToken;
                Proxy = proxy;
            }

            public Notifier( Configuration.CliToken ndidAccessToken, WebProxy proxy = null, Configuration.ServerEnvironment server = null )
                : this( ( null != ndidAccessToken ) ? ndidAccessToken.Deserialize().access_token : null, proxy, server )
            {
            }

            public Notifier( string cliTokenOutPath, BasicAccount cliAccount, WebProxy proxy = null, Configuration.ServerEnvironment server = null )
                : this( Configuration.CliToken.QueryLatestValidateToken( cliTokenOutPath, cliAccount, proxy, server ), proxy, server )
            {
            }

            public HttpWebRequest Create( Method method, string uri, string formData = null )
            {
                HttpWebRequest request = WebRequest.CreateHttp( uri );
                request.ContentType = "application/json";
                request.Headers[ "X-Ndid-AccessToken" ] = NdidAccessToken;
                request.Proxy = Proxy;
                request.Method = method.ToString();
                request.ServerCertificateValidationCallback = ( sender, cert, chain, sslPolicyErros ) =>
                {
                    return true;    // --insecure, -k, 自己署名SSL許可。
                };
                return ( WebExceptionStatus.Success == request.WriteToStream( formData ) ) ? request : null;
            }

            public HttpExtension.ConnectionResult ExecuteCommunication( HttpWebRequest request )
            {
                HttpExtension.ConnectionResult result;
                WebExceptionStatus rc = request.ConnectToServer( out result );
                if ( WebExceptionStatus.Success != rc || null == result || HttpStatusCode.OK != result.StatusCode )
                {
                    throw new UnexpectFailureException( $"Npns.Notifier: [ Method ={ request.Method }, URI ={ request.Address} ] => [ WebException={rc}, result={result} ]" );
                }
                return result;
            }

            /// <summary>
            /// 文字列化可能データ宣言
            /// </summary>
            public interface ISerializableData
            {
                string Serialize();
            }

            /// <summary>
            /// NUP通知要求構成
            /// </summary>
            [DataContract]
            public class Nup : ISerializableData
            {
                [DataMember( Order = 0 )]
                public string type;
                [DataMember( Order = 1 )]
                public string title_id;
                [DataMember( Order = 2 )]
                public uint title_version;
                [DataMember( Order = 3 )]
                public string required_title_id;
                [DataMember( Order = 4 )]
                public uint required_title_version;

                public Nup( ID64 title_id, uint title_version, ID64 required_title_id, uint required_title_version )
                {
                    this.type = "Nup";
                    this.title_id = title_id.ToString();
                    this.title_version = title_version;
                    this.required_title_id = required_title_id.ToString();
                    this.required_title_version = required_title_version;
                }

                public string Serialize()
                {
                    return this.SerializeToJson();
                }
            }

            /// <summary>
            /// VersionList更新通知要求構成
            /// </summary>
            [DataContract]
            public class VersionList : ISerializableData
            {
                [DataMember( Order = 0 )]
                public string type;
                [DataMember( Order = 1 )]
                public string e_tag;

                public VersionList( string etag )
                {
                    type = "VersionList";
                    e_tag = etag;
                }

                public string Serialize()
                {
                    return this.SerializeToJson();
                }
            }

            /// <summary>
            /// NPNS通知要求
            /// </summary>
            /// <param name="deviceTokenForNotification"></param>
            /// <param name="requestData"></param>
            /// <param name="retryCount"></param>
            /// <returns></returns>
            public HttpExtension.ConnectionResult RequestExecution( string deviceTokenForNotification, ISerializableData requestData, uint retryCount = 5 )
            {
                var requests = new Requests( new Requests.Request[]
                {
                    new Requests.Request( deviceTokenForNotification, requestData.Serialize() )
                } );
                var postdata = requests.SerializeToJson();
                Log.WriteLine( $"Npns.Notification : Create request post data ={System.Environment.NewLine}{{0}}", JsonExtension.FormatParagraphIndent( postdata ) );
                var request = Create( Method.POST, string.Format( UriApiBase, Server.Alias ), postdata );
                if ( null == request )
                {
                    return null;
                }

                HttpExtension.ConnectionResult result = null;
                do
                {
                    try
                    {
                        if ( HttpStatusCode.OK == ( result = ExecuteCommunication( request ) ).StatusCode )
                        {
                            break;
                        }
                        Log.WriteLine( $"Npns.Notification unexpect response => [ code={result.StatusCode}, text={result.Text}]" );
                    }
                    catch ( System.Exception e )
                    {
                        Log.WriteLine( "Npns.Notification unexpect exception => {0}", e.Message );
                    }
                    Log.WriteLine( "Npns.Notification failed, Retry after 10 seconds." );
                    System.Threading.Thread.Sleep( 10000 ); // 10秒待機
                } while ( ( --retryCount ) > 0 );
                return result;
            }
        }
    }
}
