﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Text;

namespace CsTestAssistants
{
    /// <summary>
    /// ファイル/ディレクトリ操作支援。
    /// </summary>
    /// <remarks>
    /// 異常状態の発生時は例外でのエラー通知を基本コンセプトとしたインタフェースです。
    /// 例外が発生しない場合は、期待する動作が成功したことを意味します。
    /// </remarks>
    public static class FileHelper
    {
        public static DirectoryInfo MakeDirectory( string path )
        {
            if ( string.IsNullOrEmpty( path ) )
            {
                throw new System.ArgumentNullException( "Could not create the directory." );
            }
            return ( false == Directory.Exists( path ) ) ? Directory.CreateDirectory( path ) : new DirectoryInfo( path );
        }

        public static void RemoveDirectory( string path, bool recursive = true )
        {
            if ( false == string.IsNullOrEmpty( path ) && Directory.Exists( path ) )
            {
                Directory.Delete( path, recursive );
            }
        }

        public static void RemoveDirectory( DirectoryInfo directory, bool recursive = true )
        {
            if ( null != directory && directory.Exists )
            {
                directory.Delete( recursive );
            }
        }

        public static void RemoveFile( string path )
        {
            if ( false == string.IsNullOrEmpty( path ) && File.Exists( path ) )
            {
                File.Delete( path );
            }
        }

        public static void RemoveFile( FileInfo file )
        {
            if ( null != file && file.Exists )
            {
                file.Delete();
            }
        }

        /// <summary>
        /// 対象ディレクトリ内の全ファイルを削除します。
        /// </summary>
        /// <param name="directory"></param>
        public static void RemoveFiles( DirectoryInfo directory )
        {
            if ( null != directory && directory.Exists )
            {
                var files = directory.GetFiles( "*" );
                if ( null != files )
                {
                    foreach ( var file in files )
                    {
                        File.Delete( file.FullName );
                    }
                }
            }
        }

        public static void RenameFile( string oldPath, string newPath )
        {
            if ( false == string.IsNullOrEmpty( oldPath ) && File.Exists( oldPath ) )
            {
                RemoveFile( newPath );
                File.Move( oldPath, newPath );
            }
        }

        public static void TestExistsFile( string path )
        {
            if ( false == File.Exists( path ) )
            {
                throw new FileNotFoundException( "Not found the " + path );
            }
        }

        /// <summary>
        /// 引数で指定された文字列を '_' で結合した一時ディレクトリの名前を生成する。
        /// 一時ディレクトリの名前は args[0...n] + 日時 + ランダム文字列から構成される。
        ///
        /// 生成する名前の例： arg0_arg1_2015-06-08-15-03-07_jyafvb2z.xoh
        /// </summary>
        public static string MakeTemporaryName( params object[] args )
        {
            StringBuilder b = new StringBuilder( 128 );
            if ( null != args )
            {
                for ( int i = 0; i < args.Length; ++i )
                {
                    b.Append( args[ i ] ).Append( '_' );
                }
            }
            b.Append( DateTime.Now.ToString( "yyyy-MM-dd-HH-mm-ss" ) );
            b.Append( '_' );
            b.Append( Path.GetRandomFileName() );
            return b.ToString();
        }

        /// <summary>
        /// MakeTemporaryName を利用したテンポラリディレクトリ作成支援メソッド。
        /// </summary>
        /// <param name="basePath"></param>
        /// <param name="args"></param>
        /// <returns>作成されたディレクトリのDirectoryInfo</returns>
        public static DirectoryInfo MakeTemporaryDirectory( string basePath, params object[] args )
        {
            return MakeDirectory( Path.Combine( basePath, MakeTemporaryName( args ) ) );
        }

        /// <summary>
        /// 指定ファイルリストの内容をテンポラリフォルダにコピーします。
        /// </summary>
        /// <param name="sourceFiles">コピー元ファイルリスト</param>
        /// <param name="parentDirectory">テンポラリフォルダが生成されるベースフォルダパスを指定します。
        /// null の場合、Path.GetTempPath() からパスルートを取得します。</param>
        /// <returns>コピーした先のテンポラリフォルダ。使用後の削除責務はユーザに帰属します。</returns>
        public static DirectoryInfo CopyToTemporary( List<string> sourceFiles, string parentDirectory = null )
        {
            if ( string.IsNullOrEmpty( parentDirectory ) )
            {
                parentDirectory = Path.GetTempPath();
            }
            var info = MakeTemporaryDirectory( parentDirectory );
            RemoveFiles( info );
            foreach( var source in sourceFiles )
            {
                TestExistsFile( source );
                var filename = Path.GetFileName( source );
                var destpath = Path.Combine( info.FullName, filename );
                File.Copy( source, destpath, true );
            }
            return info;
        }
    }

    /// <summary>
    /// Collectable temporary files helper.
    /// </summary>
    public class TemporaryFileHolder : IDisposable
    {
        /// <summary>
        /// 今使用するべき一時ディレクトリのパスを取得します。
        /// </summary>
        /// <returns></returns>
        public static string GetTemporaryRoot()
        {
            if ( GlobalTemporaryRoot != null )
            {
                return GlobalTemporaryRoot;
            }
            return Path.GetTempPath();
        }

        /// <summary>
        /// 一時ファイルの配置場所を指定して作成します。
        /// </summary>
        public TemporaryFileHolder( string toolName, string temporaryRoot = null, bool afterDelete = true )
        {
            ToolName = toolName;
            AfterDelete = afterDelete;
            TemporaryRoot = ( null != temporaryRoot ) ? temporaryRoot : GetTemporaryRoot();
        }

        /// <summary>
        /// 一時ファイルのパスを作成します。
        /// ファイルは作成しません。
        /// 作成したファイルはこのインスタンスにより管理され、Dispose 時に削除されます。
        /// </summary>
        public FileInfo CreateTemporaryFilePath(string name, string extension = "")
        {
            var tempFile = new FileInfo(Path.Combine(TemporaryRoot, FileHelper.MakeTemporaryName(ToolName, name) + extension));

            if (tempFile.Exists)
            {
                throw new System.IO.IOException(string.Format("Temporary file already exists: {0}", tempFile.FullName));
            }

            AddTemporaryFile(tempFile);
            return tempFile;
        }

        /// <summary>
        /// 一時ファイルを追加します。
        /// 別の場所で作成した一時ファイルを削除したい場合に使用します。
        /// </summary>
        public void AddTemporaryFile(FileInfo fileInfo)
        {
            files.Add(fileInfo);
        }

        /// <summary>
        /// 一時ディレクトリのパスを作成します。
        /// ディレクトリは作成しません。
        /// 作成したディレクトリはこのインスタンスにより管理され、Dispose 時に削除されます。
        /// </summary>
        public DirectoryInfo CreateTemporaryDirectoryPath(string name)
        {
            var tempDirectory = new DirectoryInfo(Path.Combine(TemporaryRoot, FileHelper.MakeTemporaryName(ToolName, name)));

            if (tempDirectory.Exists)
            {
                throw new System.IO.IOException(string.Format("Temporary directory already exists: {0}", tempDirectory.FullName));
            }

            AddTemporaryDirectory(tempDirectory);
            return tempDirectory;
        }

        /// <summary>
        /// 一時ディレクトリを作成します。
        /// 作成したディレクトリはこのインスタンスにより管理され、Dispose 時に削除されます。
        /// </summary>
        public DirectoryInfo CreateTemporaryDirectory(string name)
        {
            var temporaryDirectory = CreateTemporaryDirectoryPath(name);
            if ( false == temporaryDirectory.Exists )
            {
                temporaryDirectory.Create();
            }
            return temporaryDirectory;
        }

        /// <summary>
        /// 一時ディレクトリを追加します。
        /// 別の場所で作成した一時ディレクトリを削除したい場合に使用できます。
        /// </summary>
        public void AddTemporaryDirectory(DirectoryInfo directoryInfo)
        {
            directories.Add(directoryInfo);
        }

        /// <summary>
        /// 一時ファイルや一時ディレクトリを全て削除します。
        /// Dispose 時に呼ばれるものなので、通常は明示的に呼ぶ必要はありません。
        /// </summary>
        public void DeleteAll()
        {
            foreach (var file in files)
            {
                if (File.Exists(file.FullName))
                {
                    File.Delete(file.FullName);
                }
            }

            foreach (var directory in directories)
            {
                if (Directory.Exists(directory.FullName))
                {
                    Directory.Delete(directory.FullName, true);
                }
            }
        }

        public void Dispose()
        {
            if (AfterDelete)
            {
                DeleteAll();
            }
        }

        public static string GlobalTemporaryRoot { get; set; }

        public string ToolName { get; private set; }
        public string TemporaryRoot { get; private set; }
        public bool AfterDelete { get; private set; }
        private List<FileInfo> files = new List<FileInfo>();
        private List<DirectoryInfo> directories = new List<DirectoryInfo>();
    }
}
