﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        指定されたテストスクリプトを繰り返し実行します。

    .DESCRIPTION
        -ScriptPath で指定されるテストスクリプトを -Target で指定される開発機上で繰り返し実行
        します。あらかじめ、Setup-TargetEnvironment.ps1 などを実行して開発機でテストが行える
        環境を構築しておく必要があります。

        テスト試行の失敗は、テストスクリプトのエラーストリームへの出力により検出します。
        テストの失敗を検出すると、Start-AgingTest.ps1 はその試行の終了後に以下の操作を行います。
            - Invoke-TargetReset
            - Teardown-TargetEnvironment.ps1
            - Setup-TargetEnvironment.ps1
        これらの操作が失敗した場合、開発機が予期せぬ状態に陥っていると判断し、エージングテストは
        その時点で中断されます。

        1 回のテスト試行のタイムアウト時間を -Timeout で指定します。この時間以内にテストスクリプト
        の実行が終了しない場合、テスト試行は失敗であると見なされ、実行中のテストスクリプトは中断されます。

        繰り返し回数は -MaximumTrial, 繰り返し時間は -MaximumDuration で指定できます。いずれも
        無指定の場合は永久にテストを実行し続けます。

        -ReportDirectory を指定すると、エージングテスト全体の結果概要、およびエージングテスト中の
        コンソール出力をログファイルとして保存することができます。結果概要は html ファイルで出力
        されます。ログファイルは PowerShell のトランスクリプト機能を使用して、テスト試行ごとに生成されます。

    .PARAMETER ScriptPath
        実行するテストスクリプト

    .PARAMETER Target
        ControlTarget, RunOnTarget の -t オプションに渡す文字列

    .PARAMETER Timeout
        1 回あたりのテスト試行のタイムアウト秒数。-ScriptPath で指定されるスクリプトがこの時間以内に
        終了しない場合をテスト失敗と見なします。

    .PARAMETER NetworkAccountId
        テストに使用する NNID のユーザ ID。無指定の場合は、CI テスト用の NNID が使用されます。

    .PARAMETER NetworkAccountPassword
        テストに使用する NNID のパスワード。無指定の場合は、CI テスト用の NNID が使用されます。

    .PARAMETER NetworkSettingsPath
        SettingsManager にインポートさせるネットワーク接続設定を記述した json ファイルのパス。
        無指定の場合、スクリプトと同じディレクトリに NetworkSettings.local.json というファイルが
        あればそれを使用します。無ければ CI テスト用の json ファイルを使用します。

    .PARAMETER MaximumTrial
        最大試行回数。無指定の場合、試行回数によってエージングテストが終了することはありません。

    .PARAMETER MaximumDuration
        最大試行時間 (秒)。無指定の場合、試行時間によってエージングテストが終了することはありません。

    .PARAMETER ReportDirectory
        エージングテストの結果に関するファイルの保存先ディレクトリ

    .PARAMETER StopTestingOnFailure
        テストが失敗した時、その時点でエージングテストを中断するようにします。

    .PARAMETER NoKillingTargetManager
        開発機との通信が正常に行えず、かつ Nintendo Target Manager の異常が疑われる場合は
        エージングテストを中断するようにします。既定では、Nintendo Target Manager を再起動
        することによってテストを続行しようとします。複数台の開発機を同時に使用している場合、
        Nintendo Target Manager の再起動によって他の開発機の動作に影響を及ぼす可能性があります。
#>

param
(
    [Parameter(Mandatory = $true)]
    [string]
    $ScriptPath,

    [Parameter(Mandatory = $true)]
    [string]
    $Target,

    [string]
    $NetworkAccountId,

    [string]
    $NetworkAccountPassword,

    [string]
    $NetworkSettingsPath,

    [int]
    $Timeout = 600,

    [int]
    $MaximumTrial,

    [int]
    $MaximumDuration,

    [string]
    $ReportDirectory,

    [switch]
    $StopTestingOnFailure,

    [switch]
    $NoKillingTargetManager
)

$ErrorActionPreference = 'Stop'

Set-StrictMode -Version Latest

$MyScriptPath      = $MyInvocation.MyCommand.Path
$MyScriptDirectory = Split-Path $MyScriptPath


$ScriptPath = Resolve-Path $ScriptPath

# --------------------------------------------------------------------------
# 共通モジュール読み込み & 定数定義など
# --------------------------------------------------------------------------
Import-Module "$MyScriptDirectory\Common"
. "$MyScriptDirectory\Constants.ps1"
. "$MyScriptDirectory\Functions.ps1"

function Try-TargetReset
{
    param
    (
        [int]
        $Timeout = 30
    )

    $ResetJob = Start-Job -ArgumentList $script:Target,$script:MyScriptDirectory -ScriptBlock {
        param($Target, $MyScriptDirectory)
        Import-Module "$MyScriptDirectory\Common"
        . "$MyScriptDirectory\Constants.ps1"
        . "$MyScriptDirectory\Functions.ps1"
        Invoke-TargetReset $Target
    }

    $ResetCompleted = Wait-Job -Job $ResetJob -Timeout $Timeout
    try
    {
        Receive-Job -Job $ResetJob | Write-Host
        Remove-Job -Job $ResetJob -Force

        return $ResetCompleted
    }
    catch
    {
        Write-FailureLog $_
        return $false
    }
}

function Get-JsonDateTime
{
    param
    (
        [System.DateTime]
        $DateTime
    )

    if (-not $DateTime)
    {
        $DateTime = Get-Date
    }

    $DateTime.ToString('s')
}

function New-TestSummary
{
    param
    (
    )

    return @{ScriptName = ''; TargetName = ''; Trial = 0; Success = 0; Failure = 0; Results = @(); Start = ''; End = ''}
}

function Add-SuccessResult
{
    param
    (
        [Parameter(Mandatory = $true)]
        [hashtable]
        $TestSummary,

        [Parameter(Mandatory = $true)]
        [System.DateTime]
        $StartDate,

        [System.DateTime]
        $EndDate
    )

    if (-not $EndDate)
    {
        $EndDate = Get-Date
    }

    ++$TestSummary.Trial
    ++$TestSummary.Success
    $TestSummary.Results += @{Trial = $TestSummary.Trial; Result = 'Success'; Message = ''; Start = (Get-JsonDateTime $StartDate); End = (Get-JsonDateTime $EndDate)}
}

function Add-FailureResult
{
    param
    (
        [Parameter(Mandatory = $true)]
        [hashtable]
        $TestSummary,

        [Parameter(Mandatory = $true)]
        [System.DateTime]
        $StartDate,

        [System.DateTime]
        $EndDate,

        [string]
        $ErrorMessage
    )

    if (-not $EndDate)
    {
        $EndDate = Get-Date
    }

    ++$TestSummary.Trial
    ++$TestSummary.Failure
    $TestSummary.Results += @{Trial = $TestSummary.Trial; Result = 'Failure'; Message = $ErrorMessage; Start = (Get-JsonDateTime $StartDate); End = (Get-JsonDateTime $EndDate)}
}

function Set-TestSummary
{
    param
    (
        [Parameter(Mandatory = $true)]
        [hashtable]
        $TestSummary,

        [Parameter(Mandatory = $true)]
        [string]
        $TestSummaryFilePath
    )

    if (-not (Test-Path (Split-Path $TestSummaryFilePath)))
    {
        New-Item -ItemType Directory (Split-Path $TestSummaryFilePath) >$null
    }

    ConvertTo-Json20 $TestSummary | Set-Content $TestSummaryFilePath
}

function Get-TestSummaryJsonFilePath
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $ReportDirectory
    )

    "$ReportDirectory\Summary.json"
}

function Start-TrialTranscript
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $TranscriptFilePath
    )

    $TranscriptFileDirectory = Split-Path $TranscriptFilePath
    if (-not (Test-Path $TranscriptFileDirectory))
    {
        New-Item -ItemType Directory $TranscriptFileDirectory >$null
    }

    Start-Transcript -Path $TranscriptFilePath
}

function Get-TrialTranscriptFilePath
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $ReportDirectory,

        [Parameter(Mandatory = $true)]
        [int]
        $Trial
    )

    "$ReportDirectory\Logs\Trial{0:D5}.log" -f $Trial
}

function Stop-TrialTranscript
{
    param
    (
    )

    try
    {
        Stop-Transcript
    }
    catch
    {
        # 既にトランスクリプトが停止していると例外が発生する ---> 無視する
    }
}


# --------------------------------------------------------------------------
# テスト
# --------------------------------------------------------------------------
$TestSummary = New-TestSummary
$TestSummary.ScriptName = Get-Item $ScriptPath | % { $_.Name }
$TestSummary.TargetName = $Target

try
{
    $AgingStartDate = Get-Date
    $TestSummary.Start = Get-JsonDateTime $AgingStartDate

    if ($ReportDirectory)
    {
        if (Test-Path $ReportDirectory)
        {
            Remove-Item $ReportDirectory -Recurse
        }

        Set-TestSummary $TestSummary (Get-TestSummaryJsonFilePath $ReportDirectory)

        $TestReportTemplateFilePath = "$MyScriptDirectory\AgingTestReportTemplate.html"
        $TestReportFilePath = "$ReportDirectory\$(Split-Path -Leaf $ScriptPath).report.html"
        Copy-Item $TestReportTemplateFilePath $TestReportFilePath
    }

    for ($Trial = 1; ; ++$Trial)
    {
        if ($ReportDirectory)
        {
            Start-TrialTranscript (Get-TrialTranscriptFilePath $ReportDirectory $Trial)
        }

        Write-InfoLog "######################## Trial $Trial ########################"

        $TestJob = Start-Job -ArgumentList $ScriptPath,$Target -ScriptBlock {
            param($ScriptPath, $Target)
            & $ScriptPath $Target -AgingTest
        }

        try
        {
            $TrialStartDate = Get-Date
            while ($true)
            {
                $Completed = Wait-Job -Job $TestJob -Timeout 1

                # テストが失敗した場合はここから例外が投げられる
                Receive-Job -Job $TestJob

                if ($Completed)
                {
                    break
                }

                $Diff = (Get-Date) - $TrialStartDate
                if ($Diff.TotalSeconds -gt $Timeout)
                {
                    break
                }
            }
            if ($Completed)
            {
                # 成功
                Write-SuccessLog "*** Trial $Trial Finished ***"
                Add-SuccessResult $TestSummary -StartDate $TrialStartDate

                Remove-Job -Job $TestJob
            }
            else
            {
                throw "Test script timeout ($Timeout sec.)"
            }
        }
        catch
        {
            # 失敗
            Write-FailureLog $_
            Write-FailureLog "*** Trial $Trial Failed ***"
            Add-FailureResult $TestSummary -StartDate $TrialStartDate -ErrorMessage $_

            Remove-Job -Job $TestJob -Force

            if ($StopTestingOnFailure)
            {
                throw
            }

            # ターゲットがどういう状態になっているか不明なので、セットアップをやり直す

            $KilledTargetManager = $false
            $ResetCompleted = Try-TargetReset

            if (-not $ResetCompleted -and -not $NoKillingTargetManager)
            {
                Write-InfoLog "Target reset timeout. Will kill NintendoTargetManager and try again."
                Get-Process | ? { $_.Name -eq "NintendoTargetManager" } | Stop-Process -Force
                $KilledTargetManager = $true

                $ResetCompleted = Try-TargetReset
            }
            if (-not $ResetCompleted)
            {
                throw "Target reset timeout. Cannot continue testing!"
            }

            # TargetManager を殺して復帰した場合、テストは TargetManager の影響で失敗した可能性が高い
            if ($KilledTargetManager)
            {
                $Result = $TestSummary.Results | ? { $_.Trial -eq $Trial }
                $Result.Message = "Cannot connect to target because Target Manager hangs."
            }

            & "$MyScriptDirectory\Teardown-TargetEnvironment.ps1" $Target -TrialCount 200
            & "$MyScriptDirectory\Setup-TargetEnvironment.ps1" $Target -TrialCount 200 -NetworkAccountId $NetworkAccountId -NetworkAccountPassword $NetworkAccountPassword -NetworkSettingsPath $NetworkSettingsPath
        }

        Stop-TrialTranscript

        if ($ReportDirectory)
        {
            Set-TestSummary $TestSummary (Get-TestSummaryJsonFilePath $ReportDirectory)
        }

        Write-InfoLog "Trial done = $($TestSummary.Trial), Success = $($TestSummary.Success), Failure = $($TestSummary.Failure)"

        if ($MaximumTrial -and $Trial -ge $MaximumTrial)
        {
            Write-InfoLog "Trial count reached the designated maximum count."
            break
        }
        if ($MaximumDuration -and ((Get-Date) - $AgingStartDate).TotalSeconds -gt $MaximumDuration)
        {
            Write-InfoLog "Test time elapsed the designated maximum duration."
            break
        }
    }
}
finally
{
    $TestSummary.End = Get-JsonDateTime

    # TODO: 重複コードの整理
    Stop-TrialTranscript

    if ($ReportDirectory)
    {
        Set-TestSummary $TestSummary (Get-TestSummaryJsonFilePath $ReportDirectory)
    }

    Write-InfoLog "Trial done = $($TestSummary.Trial), Success = $($TestSummary.Success), Failure = $($TestSummary.Failure)"

    Write-InfoLog "Removing running jobs..."
    Get-Job | Remove-Job -Force
}
