﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <future>

#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_InstallApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_ApplicationEntitySystemApi.h>

#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/util/util_Optional.h>
#include <nn/ncm/ncm_InstallTaskBase.h>

#include "ApplicationManagerTestTool_AppOperation.h"

#include "ApplicationManagerTestTool_Common.h"
#include "ApplicationManagerTestTool_FsUtilities.h"
#include "ApplicationManagerTestTool_Stopwatch.h"

nn::util::optional<nn::ncm::StorageId> ToStorageId(const std::string& storage)
{
    if (storage == "host") return nn::ncm::StorageId::Host;
    if (storage == "builtin") return nn::ncm::StorageId::BuildInUser;
    if (storage == "sdcard") return nn::ncm::StorageId::SdCard;

    return nullptr;
}

nn::util::optional<nn::ncm::StorageId> ToInstallStorage(const std::string& storage)
{
    if (storage == "builtin") return nn::ncm::StorageId::BuildInUser;
    if (storage == "sdcard") return nn::ncm::StorageId::SdCard;
    if (storage == "auto") return nn::ncm::StorageId::Any;

    return nullptr;
}

const size_t InstallBufferSize = 16 * 1024 * 1024;

std::vector<nn::ncm::StorageId> AppOperation::m_TestStorageIdList;

// 自分自身のアプリケーション ID 定義 ( ApplicationManagerTestTool.nmeta で設定)
const nn::ncm::ApplicationId SelfApplicationId = { 0x0005000C10000000 };

AppOperation::AppOperation() NN_NOEXCEPT
{
}

AppOperation::~AppOperation() NN_NOEXCEPT
{
}

void AppOperation::Initialize() NN_NOEXCEPT
{
#if 0
    // なぜか落ちる？
    // ひとまずこの処理は保留
    if (m_TestStorageIdList.empty() == true)
    {
        m_TestStorageIdList.push_back(nn::ncm::StorageId::BuildInSystem);
        m_TestStorageIdList.push_back(nn::ncm::StorageId::BuildInUser);
        m_TestStorageIdList.push_back(nn::ncm::StorageId::SdCard);
    }
#endif
}

nn::Result WaitInstallTaskExecute(nn::ns::ApplicationInstallTask* task)
{
    auto result = std::async(std::launch::async, [&task] { return task->Execute(); });

    while (NN_STATIC_CONDITION(true))
    {
        auto status = result.wait_for(std::chrono::milliseconds(250));

        //auto progress = task->GetProgress();
        //NN_LOG("%lld / %lld\n", progress.installedSize, progress.totalSize);

        if (status == std::future_status::ready)
        {
            break;
        }
    }
    return result.get();
}

nn::Result AppOperation::Install(const std::string& inNspFilePath, const std::string& inInstallStorage) NN_NOEXCEPT
{
    nn::Result result;
    const char* filePath = inNspFilePath.c_str();
    if (!fsutil::IsAbsolutePath(filePath))
    {
        NN_LOG("'%s' is not an absolute path.\n", filePath);
        NN_RESULT_SUCCESS;
    }
    if (!fsutil::IsNspFile(filePath))
    {
        NN_LOG("Warning: The file extension is not .nsp\n");
    }

    fsutil::File file;
    NNT_EXPECT_RESULT_SUCCESS(file.Open(filePath, nn::fs::OpenMode_Read));

    std::unique_ptr<char[]> buffer(new char[InstallBufferSize]);
    nn::ns::ApplicationInstallTask task;

    auto storage = ToInstallStorage(inInstallStorage);
    result = task.Initialize(file.GetHandle(), *storage, buffer.get(), InstallBufferSize);
    NNT_EXPECT_RESULT_SUCCESS(result);

    //NN_LOG("Preparing to install %s...\n", filePath);
    result = task.Prepare();
    NNT_EXPECT_RESULT_SUCCESS(result);
    if (result.IsFailure())
    {
        return result;
    }
    // インストール前に application コンテンツが入っているかをチェックする
    {
        const int Count = 1;
        const int Offset = 0;
        nn::ncm::ApplicationContentMetaKey key;
        int outCount;

        NNT_EXPECT_RESULT_SUCCESS(task.ListApplicationContentMetaKey(&outCount, &key, Count, Offset));

        // key が 1 つも返って来ない == application が含まれていないシステムコンテンツの可能性あり
        if (outCount == 0)
        {
            NN_LOG("This file doesn't contain application contents.\n");
            NNT_EXPECT_RESULT_SUCCESS(task.Cleanup());
            NN_RESULT_SUCCESS;
        }
    }
    NNT_EXPECT_RESULT_SUCCESS(WaitInstallTaskExecute(&task));
    NNT_EXPECT_RESULT_SUCCESS(task.Commit());

    NN_RESULT_SUCCESS;
}

nn::Result AppOperation::GetViewList(std::vector<nn::ns::ApplicationView>& outList) NN_NOEXCEPT
{
    outList.clear();

    const size_t bufferSize = 1024;
    std::unique_ptr<nn::ns::ApplicationRecord[]> recordBuffer(new nn::ns::ApplicationRecord[bufferSize]);
    auto recordList = recordBuffer.get();

    int offset = 0;
    int count = 1;
    while (count > 0)
    {
        count = nn::ns::ListApplicationRecord(recordList, bufferSize, offset);
        for (int i = 0; i < count; ++i)
        {
            auto& record = recordList[i];
            nn::ns::ApplicationView view;
            auto result = nn::ns::GetApplicationView(&view, &record.id, 1);
            NNT_EXPECT_RESULT_SUCCESS(result);
            if (result.IsFailure())
            {
                NN_LOG(" [Error] GetApplicationView Failed : 0x%016llx\n", record.id);
                return result;
            }

            if (view.id.value == SelfApplicationId.value)
            {
                // (SIGLO-72215) 自分自身のアプリケーション ID は対象外とする
                continue;
            }

            outList.push_back(view);
        }

        offset += count;
    }

    NN_RESULT_SUCCESS;
}

nn::Result AppOperation::GetTestPropertyList(std::vector<TestProperty>& outList) NN_NOEXCEPT
{
    outList.clear();

    std::vector<nn::ns::ApplicationView> viewList;
    auto result = AppOperation::GetViewList(viewList);
    NNT_EXPECT_RESULT_SUCCESS(result);
    if (result.IsFailure())
    {
        return result;
    }

    for (auto& v : viewList)
    {
        TestProperty testProperty;
        memcpy(&testProperty.view, &v, sizeof(testProperty.view));

        result = nn::ns::CalculateApplicationOccupiedSize(&testProperty.size, v.id);
        NNT_EXPECT_RESULT_SUCCESS(result);
        if (result.IsFailure())
        {
            NN_LOG(" [Error] CalculateApplicationOccupiedSize Failed : 0x%016llx\n", v.id);
            return result;
        }

        for (auto& id : m_TestStorageIdList)
        {
            const bool isMovable = nn::ns::IsApplicationEntityMovable(v.id, id);
            testProperty.movableList.push_back(std::make_pair(id, isMovable));
        }

        {
            const size_t bufferSize = 1024 * 1024;
            std::unique_ptr<char[]> buffer(new char[bufferSize]);
            size_t outSize;
            result = nn::ns::GetApplicationControlData(
                &outSize, buffer.get(), bufferSize, nn::ns::ApplicationControlSource::Storage, v.id);
            // Aocのみインストールされている場合を考慮
            if (result.IsSuccess())
            {
                nn::ns::ApplicationControlDataAccessor accessor(buffer.get(), outSize);
                memcpy(&testProperty.controlProperty, &accessor.GetProperty(), sizeof(testProperty.controlProperty));
            }
        }

        outList.push_back(testProperty);
    }

    NN_RESULT_SUCCESS;
}

nn::Result AppOperation::GetInstalledIdList(std::set<nn::Bit64>& outList) NN_NOEXCEPT
{
    outList.clear();

    std::vector<nn::ns::ApplicationView> viewList;
    auto result = AppOperation::GetViewList(viewList);
    NNT_EXPECT_RESULT_SUCCESS(result);
    if (result.IsFailure())
    {
        return result;
    }

    for (auto& v : viewList)
    {
        outList.insert(v.id.value);
    }

    NN_RESULT_SUCCESS;
}

nn::Result AppOperation::Uninstall(nn::ncm::ApplicationId inAppId) NN_NOEXCEPT
{
    return nn::ns::DeleteApplicationCompletely(inAppId);
}

nn::Result AppOperation::UninstallAll() NN_NOEXCEPT
{
    NN_LOG(" [Trace] Uninstall All Application ... ");
    Stopwatch watch;
    watch.Start();

    auto result = nn::ns::DeleteRedundantApplicationEntity();
    NNT_EXPECT_RESULT_SUCCESS(result);
    if (result.IsFailure())
    {
        return result;
    }

    std::vector<nn::ns::ApplicationView> viewList;
    result = AppOperation::GetViewList(viewList);
    NNT_EXPECT_RESULT_SUCCESS(result);
    if (result.IsFailure())
    {
        return result;
    }

    const auto flags = nn::ns::ApplicationEntityFlag_All | nn::ns::ApplicationEntityFlag_SkipGameCardCheck::Mask | nn::ns::ApplicationEntityFlag_SkipRunningCheck::Mask;
    for (auto& v : viewList)
    {
        // (SIGLO-77484) UninstallAll で実行する削除処理は DeleteApplicationCompletelyForDebug を利用する
        result = nn::ns::DeleteApplicationCompletelyForDebug(v.id, flags);
        NNT_EXPECT_RESULT_SUCCESS(result);
        if (result.IsFailure())
        {
            NN_LOG(" [Error] DeleteApplicationCompletelyForDebug Failed : 0x%016llx\n", v.id.value);
            return result;
        }
    }

    watch.Stop();
    NN_LOG("Done : (time=%lld [ms]) \n", watch.GetMilliSecond());

    NN_RESULT_SUCCESS;
}

void AppOperation::ClearCache() NN_NOEXCEPT
{
    nn::ns::InvalidateAllApplicationControlCache();
}

nn::Result AppOperation::GetHostStorageSize(int64_t& outFreeSize, int64_t& outTotalSize) NN_NOEXCEPT
{
    const std::string storageStr = "host";
    auto result = nn::ns::GetTotalSpaceSize(&outTotalSize, *(ToStorageId(storageStr)));
    if (result.IsFailure() == true)
    {
        // 取得できない場合がありえるため、以降の処理は実施しない
        return result;
    }
    result = nn::ns::GetFreeSpaceSize(&outFreeSize, *(ToStorageId(storageStr)));
    NNT_EXPECT_RESULT_SUCCESS(result);

    NN_RESULT_SUCCESS;
}

nn::Result AppOperation::GetBuiltInStorageSize(int64_t& outFreeSize, int64_t& outTotalSize) NN_NOEXCEPT
{
    const std::string storageStr = "builtin";
    auto result = nn::ns::GetTotalSpaceSize(&outTotalSize, *(ToStorageId(storageStr)));
    if (result.IsFailure() == true)
    {
        // 取得できない場合がありえるため、以降の処理は実施しない
        return result;
    }
    result = nn::ns::GetFreeSpaceSize(&outFreeSize, *(ToStorageId(storageStr)));
    NNT_EXPECT_RESULT_SUCCESS(result);

    NN_RESULT_SUCCESS;
}

nn::Result AppOperation::GetSdCardStorageSize(int64_t& outFreeSize, int64_t& outTotalSize) NN_NOEXCEPT
{
    const std::string storageStr = "sdcard";
    auto result = nn::ns::GetTotalSpaceSize(&outTotalSize, *(ToStorageId(storageStr)));
    if (result.IsFailure() == true)
    {
        // 取得できない場合がありえるため、以降の処理は実施しない
        return result;
    }
    result = nn::ns::GetFreeSpaceSize(&outFreeSize, *(ToStorageId(storageStr)));
    NNT_EXPECT_RESULT_SUCCESS(result);

    NN_RESULT_SUCCESS;
}

bool AppOperation::IsMovable(nn::ncm::ApplicationId inAppId, const std::string& inDstStorage) NN_NOEXCEPT
{
    auto storage = ToStorageId(inDstStorage);
    if (storage == nn::util::nullopt)
    {
        NN_LOG("Invalid Argument : inDstStorage=%s\n", inDstStorage.c_str());
        return false;
    }

    return nn::ns::IsApplicationEntityMovable(inAppId, *storage);
}

nn::Result AppOperation::MoveApplication(nn::ncm::ApplicationId inAppId, const std::string& inDstStorage) NN_NOEXCEPT
{
    auto storage = ToStorageId(inDstStorage);
    if (storage == nn::util::nullopt)
    {
        NN_LOG("Invalid Argument : inDstStorage=%s\n", inDstStorage.c_str());
        NN_RESULT_SUCCESS;
    }

    return nn::ns::MoveApplicationEntity(inAppId, *storage);
}

int AppOperation::CountContentMeta(nn::ncm::ApplicationId inAppId) NN_NOEXCEPT
{
    int retCount = 0;
    auto result = nn::ns::CountApplicationContentMeta(&retCount, inAppId);
    NNT_EXPECT_RESULT_SUCCESS(result);

    return retCount;
}

nn::Result AppOperation::GetContentMetaStatus(nn::ncm::ApplicationId inAppId, ContentMetaStatusList& outMetaStatusList) NN_NOEXCEPT
{
    outMetaStatusList.clear();

    const auto count = AppOperation::CountContentMeta(inAppId);
    if (count == 0)
    {
        NN_RESULT_SUCCESS;
    }

    std::unique_ptr<nn::ns::ApplicationContentMetaStatus[]> buffer(new nn::ns::ApplicationContentMetaStatus[count]);
    auto listPtr = buffer.get();

    int outCount = 0;
    auto result = nn::ns::ListApplicationContentMetaStatus(&outCount, listPtr, count, inAppId, 0);
    NNT_EXPECT_RESULT_SUCCESS(result);
    if (result.IsFailure())
    {
        NN_LOG(" [Error] ListApplicationContentMetaStatus Failed : 0x%016llx\n", inAppId.value);
        return result;
    }

    for (int i = 0; i < count; ++i)
    {
        outMetaStatusList.push_back(listPtr[i]);
    }

    return result;
}

nn::Result AppOperation::CleanupSdCard() NN_NOEXCEPT
{
    auto result = nn::ns::CleanupSdCard();
    NNT_EXPECT_RESULT_SUCCESS(result);
    return result;
}

nn::Result AppOperation::FormatSdCard() NN_NOEXCEPT
{
    auto result = nn::ns::FormatSdCard();
    NNT_EXPECT_RESULT_SUCCESS(result);
    // Unexpected error が返ってきた場合を考慮
    if (result.IsFailure() && nn::ns::ResultSdCardFormatUnexpected::Includes(result))
    {
        auto lastResult = nn::ns::GetLastSdCardFormatUnexpectedResult();
        NN_LOG(" [Error] FormatSdCard() Unexpected error: 0x%08x\n", lastResult.GetInnerValueForDebug());
    }
    return result;
}

nn::Result AppOperation::MeasureApplicationManagerTime() NN_NOEXCEPT
{
    const size_t bufferSize = 8200;
    std::unique_ptr<nn::ns::ApplicationRecord[]> recordBuffer(new nn::ns::ApplicationRecord[bufferSize]);
    auto recordList = recordBuffer.get();

    nn::Result result;
    Stopwatch watch;

    NN_LOG("\n");

    watch.Start();
    auto count = nn::ns::ListApplicationRecord(recordList, bufferSize, 0);
    watch.Stop();
    NN_LOG("  ListApplicationRecord() Time: %28lld [ms]\n", watch.GetMilliSecond());

    for (int i = 0; i < count; ++i)
    {
        auto& record = recordList[i];
        NN_LOG("  Application Id:                         0x%016llx\n", record.id);

        TestProperty testProperty;
        watch.Start();
        result = nn::ns::GetApplicationView(&testProperty.view, &record.id, 1);
        watch.Stop();
        NNT_EXPECT_RESULT_SUCCESS(result);
        if (result.IsFailure())
        {
            NN_LOG(" [Error] GetApplicationView() Failed: 0x%016llx\n", record.id);
            return result;
        }
        NN_LOG("   GetApplicationView() Time: %30lld [ms]\n", watch.GetMilliSecond());

        watch.Start();
        result = nn::ns::CalculateApplicationOccupiedSize(&testProperty.size, record.id);
        watch.Stop();
        NNT_EXPECT_RESULT_SUCCESS(result);
        if (result.IsFailure())
        {
            NN_LOG(" [Error] CalculateApplicationOccupiedSize() Failed: 0x%016llx\n", record.id);
            return result;
        }
        NN_LOG("   CalculateApplicationOccupiedSize() Time: %16lld [ms]\n", watch.GetMilliSecond());
    }

    watch.Start();
    bool isRedundantApplicationExisted = nn::ns::IsAnyApplicationEntityRedundant();
    watch.Stop();
    if (!isRedundantApplicationExisted)
    {
        NN_LOG("  RedundantApplication does not exist.\n");
    }
    NN_LOG("  IsAnyApplicationEntityRedundant() Time: %18lld [ms]\n", watch.GetMilliSecond());

    watch.Start();
    result = nn::ns::DeleteRedundantApplicationEntity();
    watch.Stop();
    NNT_EXPECT_RESULT_SUCCESS(result);
    if (result.IsFailure())
    {
        NN_LOG(" [Error] DeleteRedundantApplicationEntity() Failed\n");
        return result;
    }
    NN_LOG("  DeleteRedundantApplicationEntity() Time: %17lld [ms]\n", watch.GetMilliSecond());

    NN_LOG("\n");

    NN_RESULT_SUCCESS;
}

