﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <curl/curl.h>
#include <nn/os.h>
#include <nn/socket.h>

#include <nn/ns/ns_Result.h>
#include <nn/nifm.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>

namespace {

    class NotificationTokenTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            nn::nim::InitializeForNetworkInstallManager();
        }

        virtual void TearDown()
        {
            nn::nim::FinalizeForNetworkInstallManager();
        }

        static void SetUpTestCase()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
            nn::nifm::SubmitNetworkRequestAndWait();
        }

        static void TearDownTestCase()
        {
            nn::nifm::CancelNetworkRequest();
        }

        nn::Result InvokeRequestRegisterNotificationToken(nn::nim::AsyncResult* out, const nn::npns::NotificationToken& token)
        {
            const int RetryCount = 3;
            const int RetryInterval = 10;

            nn::Result lastResult;
            for (int i = 0; i < RetryCount; ++i)
            {
                NN_LOG("Try: %d\n", i);
                lastResult = nn::nim::RequestRegisterNotificationToken(out, token);
                NN_RESULT_TRY(lastResult)
                    // タイミングによってはシステム起動時に ns がリクエストするトークン送信とかぶってしまう。
                    // その場合は StaticOneFactory::CreateSharedEmplaced<nim::detail::IAsyncResult, AsyncRegisterNotificationTokenImpl>(); が ResultAllocationMemoryFailed で失敗する。
                    // リトライで対処する。
                    NN_RESULT_CATCH(nn::nim::ResultAllocationMemoryFailed)
                    {
                        nn::os::SleepThread(nn::TimeSpan::FromSeconds(RetryInterval));
                        continue;
                    }
                NN_RESULT_END_TRY;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_THROW(lastResult);
        }
    };

}


TEST_F(NotificationTokenTest, Register)
{
    nn::npns::NotificationToken token = {};
    nn::util::TSNPrintf(token.data, sizeof(token.data) - 1, "TestTokenData");
    {
        nn::nim::AsyncResult asyncResult;
        NNT_EXPECT_RESULT_SUCCESS(InvokeRequestRegisterNotificationToken(&asyncResult, token));
        asyncResult.Cancel();
        while (!asyncResult.TryWait())
        {
            NN_LOG("Waiting register notification token canceled...\n");

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
        NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultHttpConnectionCanceled, asyncResult.Get());
        asyncResult.Finalize();
    }

    {
        nn::nim::AsyncResult asyncResult;
        NNT_EXPECT_RESULT_SUCCESS(InvokeRequestRegisterNotificationToken(&asyncResult, token));
        while (!asyncResult.TryWait())
        {
            NN_LOG("Waiting register notification token ...\n");

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
        NNT_EXPECT_RESULT_SUCCESS(asyncResult.Get());
        asyncResult.Finalize();
    }
}
